/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     Singleton.h
 *\brief
 *
 *\author   CM/ESA2
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_SINGLETON_H
#define ASF_CORE_SINGLETON_H

#include "asf/core/Internal.h"
#include "asf/core/Types.h"
#include "asf/threading/Once.h"

namespace asf {
namespace core {

/*!
 *\class Singleton
 *\brief Class template for Singletons
 *
 * Use get() to retrieve a reference to the singleton instance.
 * The singleton instance is registered automatically for destruction at application termination.
 * destroy() can be used to destruct the singleton instance explicitly.
 * This requires that previously returned references to the singleton instance
 * are not in use anymore.
 * Instantiating the singleton is thread-safe, its destruction is not.
 */
template < typename T >
class Singleton {
public:
    // Thread-safe
    static T& get() {
        ::asf::threading::callOnce(_once, &Singleton< T >::init);
        return *_instance;
    }

    // Not thread-safe, as this would require a mutex in get() and destroy() for the _instance
    // variable bearing higher runtime costs instead of a more efficient lock-free (platform
    // specific) implementation in get() realized within ::asf::threading::callOnce.
    static void destroy() {
        delete _instance;
        _instance = 0;
        _once = ASF_ONCE_INIT;
    }

private:
    // prevent (copy-)construction, destruction and assignment
    Singleton();
    ~Singleton();
    Singleton(Singleton const&);
    Singleton& operator=(Singleton const&);

    static void init() {
        _instance = new T();
        ::asf::core::registerShutdownHook(Singleton< T >::destroy);
    }

    static ::asf::threading::once_t _once;

    static T* _instance;
};

template < typename T >
::asf::threading::once_t Singleton< T >::_once = ASF_ONCE_INIT;

template < typename T >
T* Singleton< T >::_instance = 0;

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_SINGLETON_H
