/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     LocalMessage.h
 *\brief
 *
 *\author   CM/ESA2
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2016-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_LOCALMESSAGE_H
#define ASF_CORE_LOCALMESSAGE_H

#include <boost/shared_ptr.hpp>
#include <sstream>
#include "asf/core/ComponentMessage.h"
#include "asf/core/LocalMessageToken.h"
#include "asf/core/Logger.h"

namespace asf {
namespace core {

extern ::asf::core::Logger localMessageLogger;

/**
 * A LocalMessage is a message which is send inside an application to a component
 * It can't be serialized and therefore it can not be used in a remote context.
 * A local message can carry any C++ data as its payload.
 *
 * Template parameters:
 *
 * CB is the callback and needs to have a "onLocalMessage(shared_ptr<P> payload)" method
 *
 * P is the payload and needs to have a "setAct(act_t act)" method and an implementation of the
 * << operator.
 */
template < typename CB, typename P >
class LocalMessage : public ::asf::core::ComponentMessage {
public:
    LocalMessage(CB& callback, boost::shared_ptr< P > payload)
        : ComponentMessage(),
          _act(LocalMessageToken::getAct()),
          _callback(callback),
          _payload(payload) {
        _payload->setAct(_act);
    }

    virtual ~LocalMessage() { LocalMessageToken::returnAct(_act); }

    virtual void processMessage() {
        logMessageProcessing();
        _callback.onLocalMessage(_payload);
    }

    void logMessageSending() {
        ::asf::core::Logger& _logger = localMessageLogger;
        LOG_DEBUG("-> sendLocalMessage, act=%" PRIuPTR ", cb=%p", getAct(), &_callback);
        logPayload();
    }

    virtual ::asf::core::MessageSharedPtr clone() const {
        return ::asf::core::MessageSharedPtr(new LocalMessage(*this));
    }

    virtual bool isRemote() const { return false; }

    virtual bool sendRemote() { return false; }

    act_t getAct() const { return _act; }

private:
    void logMessageProcessing() {
        ::asf::core::Logger& _logger = localMessageLogger;
        LOG_DEBUG("<- onLocalMessage, act=%" PRIuPTR ", cb=%p", getAct(), &_callback);
        logPayload();
    }

    void logPayload() {
        ::asf::core::Logger& _logger = localMessageLogger;
        // prevent the streaminng of payload if logging is not enabled
        if (IS_LOG_DEBUG_ENABLED()) {
            std::ostringstream stream;
            stream << "payload: " << *_payload;
            LOG_DEBUG("%s", stream.str().c_str());
        }
    }

    act_t _act;

    CB& _callback;

    boost::shared_ptr< P > _payload;
};
}  // namespace core
}  // namespace asf

#endif
