/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     ComponentFactory.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_COMPONENTFACTORY_H
#define ASF_CORE_COMPONENTFACTORY_H

#include "asf/core/ContextData.h"
#include "asf/core/Exceptions.h"
#include "asf/core/Logger.h"
#include "asf/core/Types.h"
#include "asf/threading/Guard.h"
#include "asf/threading/Mutex.h"

namespace asf {
namespace core {

class BaseComponent;

class ComponentFactoryIF {
public:
    virtual void setComponentName(const ::std::string& componentName) = 0;

    virtual BaseComponent* getComponent() = 0;

    virtual void setComponent(BaseComponent* component) = 0;

    virtual bool isComponentInstantiated() = 0;

    virtual void deleteComponent() = 0;

    virtual ~ComponentFactoryIF() {}
};

extern ::asf::core::Logger componentFactoryLogger;

template < class T >
class ComponentFactory : public ComponentFactoryIF {
public:
    ComponentFactory() : _component(0) {}

    virtual ~ComponentFactory() { deleteComponent(); }

    virtual void setComponentName(const ::std::string& componentName) {
        _componentName = componentName;
    }

    virtual BaseComponent* getComponent() {
        ::asf::core::Logger& _logger = componentFactoryLogger;
        LOG_ASSERT_FATAL_MSG(ContextData::isAvailable(),
                             "called getComponent in wrong thread context");
        if (!isComponentInstantiated()) {
            LOG_DEBUG("Creating component '%s'", _componentName.c_str());
            __try {
                T* component = new T();
                CHECK_ALLOCATION(component);

                // There are cases (using AsyncTask in component constructor) where getComponent()
                // is called while it is still active. This is prevented by checking whether
                // _component is already set. These multiple invocations are triggered while T is
                // constructed. In order to prevent multiple constructions of T it must be checked
                // whether T is constructed before T's constructor returns.
                // This is not possible from within BaseComponent, therefore this information must
                // be provided externally. The only instance to know whether T will be constructed
                // when getComponent() is activated a second time while the construction of T is not
                // complete is the base class of T, namely BaseComponent. BaseComponent retrieves in
                // its constructor via the ComponentDescription the ComponentFactory and injects the
                // address of the T being in construction. When during T's construction
                // ComponentFactory::getComponent() is activated again, its address is already known
                // and by this information a second construction prevented. The equality of object
                // T's address returned by its constructor and injected by T' base class
                // BaseComponent is an invariant that is checked here by a fatal assert.

                LOG_ASSERT_FATAL(_component == component);

                _component = component;  // get rid of lint "Custodial pointer 'component' (line 67)
                                         // has not been freed or returned"
                LOG_INFO(
                    "Component '%s' (%p) successful created", _componentName.c_str(), _component);
            }
            __catch(...) {
                _component = 0;
                LOG_FATAL("Component '%s' couldn't be created, caught an exception",
                          _componentName.c_str());
            }
        }
        return _component;
    }

    virtual void setComponent(BaseComponent* component) {
        _component = static_cast< T* >(component);
    }

    virtual bool isComponentInstantiated() { return (_component != 0); }

    virtual void deleteComponent() {
        ::asf::core::Logger& _logger = componentFactoryLogger;
        LOG_ASSERT_FATAL_MSG(ContextData::isAvailable(),
                             "called deleteComponent in wrong thread context");
        __try {
            LOG_DEBUG("deleting component '%s'", _componentName.c_str());
            delete _component;
            _component = 0;
        }
        __catch(...) {
            _component = 0;
            LOG_FATAL("Caught an exception while destroying component '%s'",
                      _componentName.c_str());
        }
    }

private:
    ComponentFactory(const ComponentFactory&);
    ComponentFactory& operator=(const ComponentFactory&);

    BaseComponent* _component;

    ::std::string _componentName;
};

/**
 * LockingComponentFactory - A thread safe ComponentFactory
 *
 * The getComponent() is usually only called from the components thread. Thus no lock
 * is required. When having a component which makes use of a thread bridge, the getComponent()
 * method can also be called from the bridged thread, see the User Guide for the concept
 * of a thread bridge. Therefore a lock is needed to prevent from instantiating the component
 * twice (when calling getComponent() form the bridged thread and the component thread
 * simultaneously).
 */
template < class T >
class LockingComponentFactory : public ComponentFactory< T > {
public:
    virtual ~LockingComponentFactory() {}

    virtual BaseComponent* getComponent() {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(lock);
        return ComponentFactory< T >::getComponent();
    }

private:
    ::asf::threading::Mutex lock;
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_COMPONENTFACTORY_H
