/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012-2016
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     ComponentContainer.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.kulla@de.bosch.com
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2016 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_BASE_COMPONENT_CONTAINER_HPP
#define ASF_BASE_COMPONENT_CONTAINER_HPP

#include "asf/core/ComponentDescription.h"
#include "asf/core/ConnectionMessage.h"
#include "asf/core/Logger.h"
#include "asf/core/Message.h"
#include "asf/core/StartStopWaitIF.h"
#include "asf/core/ThreadConfiguration.h"
#include "asf/core/Types.h"

#include "asf/threading/Mutex.h"
#include "asf/threading/Thread.h"

#include <vector>

namespace asf {
namespace core {

class Application;
template < typename T >
class MessageQueue;

/**
 * A ComponentContainer represents a collection of components. The
 * component container receives messages and puts them in a queue.
 * A number of reader threads is reading those messages from the
 * queue and processes the messages.
 */
class ComponentContainer : public StartStopWaitIF {
public:
    ComponentContainer(const std::string& name, const ThreadConfiguration& threadConfiguration);

    ~ComponentContainer();

    void addComponentDescription(ComponentDescriptionSharedPtr component);

    const std::string& getName() const;

    void setApplication(Application* application);

    Application* getApplication() const;

    ProvidedPort* findProvidedPort(const std::string& serviceName, const std::string& portName);

    ComponentDescriptionSharedPtr findComponentDescription(const std::string& componentName);

    void onConnectionMessage(ConnectionMessageSharedPtr& msg);

    void createImmediateComponents();

    /**
     * Send a message to the component container. The message will be
     * added to the queue and processed later.
     */
    virtual void sendMessage(MessageSharedPtr message);

    void start();

    void stop();

    void waitForCompletion();

    void enableExceptionHandler(bool enable) { _exceptionHandlerEnabled = enable; };

private:
    ComponentContainer(const ComponentContainer&);

    ComponentContainer& operator=(const ComponentContainer&);

    void deleteComponents();

    static void* threadFunction(void* arg);

    void receiverLoop();

    void handleException() const;

    DECLARE_CLASS_LOGGER();

    std::string _name;

    ThreadConfiguration _threadConfiguration;

    Application* _application;

    MessageQueue< Message >* _messageQueue;

    ::asf::threading::Mutex _lockComponentDescriptions;

    std::vector< ComponentDescriptionSharedPtr > _componentDescriptions;

    std::vector< ::asf::threading::Thread* > _threads;

    bool _exceptionHandlerEnabled;
};

typedef ::boost::shared_ptr< ComponentContainer > ComponentContainerSharedPtr;

}  // namespace core
}  // namespace asf

#endif
