/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     BaseComponent.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

// !!!!!!!! When changing this file please update the ASF User Guide !!!!!!!!!!!

#ifndef ASF_CORE_BASECOMPONENT_H
#define ASF_CORE_BASECOMPONENT_H

#include <boost/shared_ptr.hpp>
#include "asf/core/ApplicationIF.h"
#include "asf/core/AsyncTask.h"
#include "asf/core/LocalMessage.h"
#include "asf/core/Logger.h"
#include "asf/core/PayloadTuple.h"
#include "asf/core/Types.h"
#include "asf/threading/RunnableIF.h"

namespace asf {
namespace core {

class Logger;

class ComponentDescription;

// These are the callback interfaces for BaseComponent::sendLocalMessage(). Each callback interface
// corresponds to a payload tuple class (see PayloadTuple.h). The number at the end of
// LocalMessageCallbackIF corresponds to the number of elements in PayloadTuple and its number of
// type arguments.

class LocalMessageCallbackIF0 {
public:
    virtual ~LocalMessageCallbackIF0(){};

    /**
     * Called when a local message, sent by BaseComponent::sendLocalMessage(), is processed.
     * The act can be retrieved by calling payload->getAct().
     */
    virtual void onLocalMessage(::boost::shared_ptr< PayloadTuple0 >& payload) = 0;
};

template < typename T0 >
class LocalMessageCallbackIF1 {
public:
    virtual ~LocalMessageCallbackIF1(){};

    /**
     * Called when a local message, sent by BaseComponent::sendLocalMessage(), is processed.
     * The act can be retrieved by calling payload->getAct().
     */
    virtual void onLocalMessage(::boost::shared_ptr< PayloadTuple1< T0 > >& payload) = 0;
};

template < typename T0, typename T1 >
class LocalMessageCallbackIF2 {
public:
    virtual ~LocalMessageCallbackIF2(){};

    /**
     * Called when a local message, sent by BaseComponent::sendLocalMessage(), is processed.
     * The act can be retrieved by calling payload->getAct().
     */
    virtual void onLocalMessage(::boost::shared_ptr< PayloadTuple2< T0, T1 > >& payload) = 0;
};

template < typename T0, typename T1, typename T2 >
class LocalMessageCallbackIF3 {
public:
    virtual ~LocalMessageCallbackIF3(){};

    /**
     * Called when a local message, sent by BaseComponent::sendLocalMessage(), is processed.
     * The act can be retrieved by calling payload->getAct().
     */
    virtual void onLocalMessage(::boost::shared_ptr< PayloadTuple3< T0, T1, T2 > >& payload) = 0;
};

/**
 * BaseComponent is the base class of all ASF components.
 *
 * The user has to derive components from this class.
 */
class BaseComponent {
public:
    BaseComponent();

    virtual ~BaseComponent();

    /**
     * Returns the instance name of the component as defined in the application manifest (cma).
     */
    const std::string& getName() const;

    /**
     * Returns the application in which the component runs.
     *
     * This method is virtual for mocking purposes.
     */
    virtual ::asf::core::ApplicationIF* getApplication();

    /**
     * Returns the component which is currently being executed. May return NULL if no component
     * is running.
     */
    static BaseComponent* getCurrentComponent();

    /**
     * This function moves all fragments (required and provided ports) from the source component to
     * the destination component. All ports from the source component are part of the destination
     * component after the call. Be sure what you are doing, because this call changes the model on
     * run-time.
     */
    void importComponentFragment(const std::string& sourceComponentName) const;

    /**
     * Sends a local message to the component with an empty payload. The message will be processed
     * in FIFO order on the component thread. The callback will be executed when the message is
     * processed.
     *
     * @return act This act identifies the local message. The corresponding callback invocation
     * provides the same act with help of payload->getAct().
     */
    act_t sendLocalMessage(LocalMessageCallbackIF0& cb);

    /**
     * Sends a local message to the component with a tuple of size 1. The message will be processed
     * in FIFO order on the component thread. The callback will be executed when the message is
     * processed.
     *
     * @return act This act identifies the local message. The corresponding callback invocation
     * provides the same act with help of payload->getAct().
     */
    template < typename T0 >
    act_t sendLocalMessage(LocalMessageCallbackIF1< T0 >& cb, const T0& t0);

    /**
     * Sends a local message to the component with a tuple of size 2. The message will be processed
     * in FIFO order on the component thread. The callback will be executed when the message is
     * processed.
     *
     * @return act This act identifies the local message. The corresponding callback invocation
     * provides the same act with help of payload->getAct().
     */
    template < typename T0, typename T1 >
    act_t sendLocalMessage(LocalMessageCallbackIF2< T0, T1 >& cb, const T0& t0, const T1& t1);

    /**
     * Sends a local message to the component with a tuple of size 3. The message will be processed
     * in FIFO order on the component thread. The callback will be executed when the message is
     * processed.
     *
     * @return act This act identifies the local message. The corresponding callback invocation
     * provides the same act with help of payload->getAct().
     */
    template < typename T0, typename T1, typename T2 >
    act_t sendLocalMessage(LocalMessageCallbackIF3< T0, T1, T2 >& cb,
                           const T0& t0,
                           const T1& t1,
                           const T2& t2);

    /**
     * Executes an asynchronous task without any callback (fire and forget). The asyncClass function
     * returns immediately. The passed function object (derived from AsyncTask) will be executed in
     * another thread which is provided by the ASF core.
     *
     * @param task A shared pointer to a function object which must be derived from "AsyncTask"
     */
    void asyncClass(const ::boost::shared_ptr< AsyncTask >& task) const;

    /**
     * Executes an asynchronous task. The asyncClass function returns immediately. The passed
     * function object (derived from AsyncTask) will be executed in another thread. After the
     * execution of the "AsyncTask" the callback "AsyncCompletedCallbackIF::onAsyncCompleted(act_t)"
     * will be called in the thread context of the ASF component. The callback function has no
     * return value, it includes only the related asynchronous completion token (act_t).
     *
     * @param cb The callback which must be derived from "AsyncCompletedCallbackIF"
     * @param task A shared pointer to a function object which must be derived from "AsyncTask"
     *
     * @return act This act identifies the asynchronous task. The corresponding callback invocation
     * provides the same act.
     */
    act_t asyncClass(AsyncCompletedCallbackIF& cb,
                     const ::boost::shared_ptr< AsyncTask >& task) const;

    /**
     * Executes an asynchronous task. The asyncClass function returns immediately. The passed
     * function object (derived from AsyncErrorTask<ERROR>) will be executed in another thread.
     * The template argument <ERROR> defines the type of the ERROR_ value. After the execution of
     * the "AsyncErrorTask" the callback interface "AsyncErrorCallbackIF<ERROR>" will be called in
     * the thread context of the ASF component. If the functor is executed successfully the callback
     * "AsyncErrorCallbackIF::onAsyncCompleted(act_t)" will be called. In case of a ERROR_ the
     * callback "AsyncErrorCallbackIF::onAsyncError(..)" will be called.
     *
     * @param cb The callback which must be derived from "AsyncErrorCallbackIF<ERROR>"
     * @param task A shared pointer to a function object which must be derived from
     * "AsyncErrorTask<ERROR>"
     *
     * @return act This act identifies the asynchronous task. The corresponding callback invocation
     * provides the same act directly or with help of payload->getAct().
     */
    template < typename ERROR_ >
    act_t asyncClass(AsyncErrorCallbackIF< ERROR_ >& cb,
                     const ::boost::shared_ptr< AsyncErrorTask< ERROR_ > >& task) const;

    /**
     * Executes an asynchronous task. The asyncClass function returns immediately. The passed
     * function
     * object (derived from AsyncResultTask<RESULT>) will be executed in another thread. The
     * template argument <RESULT> defines the type of the result value. After the execution of the
     * "AsyncResultTask" the callback interface "AsyncResultCallbackIF<RESULT>" will be called in
     * the thread context of the ASF component. If the functor is executed successfully the callback
     * "AsyncErrorCallbackIF::onAsyncResult(..)" will be called.
     *
     * @param cb The callback which must be derived from "AsyncResultCallbackIF<RESULT>"
     * @param task A shared pointer to a function object which must be derived from
     * "AsyncResultTask<RESULT>"
     *
     * @return act This act identifies the asynchronous task. The corresponding callback invocation
     * provides the same act with help of payload->getAct().
     */
    template < typename RESULT >
    act_t asyncClass(AsyncResultCallbackIF< RESULT >& cb,
                     const ::boost::shared_ptr< AsyncResultTask< RESULT > >& task) const;

    /**
     * Executes an asynchronous task. The asyncClass function returns immediately. The passed
     * function object (derived from AsyncResultErrorTask<RESULT>) will be executed in another
     * thread. The template arguments <RESULT, ERROR> defines the types of the result and ERROR_
     * values. After the execution of the "AsyncResultErrorTask" the callback interface
     * "AsyncResultErrorCallbackIF<RESULT, ERROR>" will be called in the thread context of the ASF
     * component. If the functor is executed successfully the callback
     * "AsyncErrorCallbackIF::onAsyncResult(..)" will be called. In case of a ERROR_ the
     * callback "AsyncErrorCallbackIF::onAsyncError(..)" will be called.
     *
     * @param cb The callback which must be derived from "AsyncResultErrorCallbackIF<RESULT,ERROR>"
     * @param task A shared pointer to a function object which must be derived from
     * "AsyncResultErrorTask<RESULT,ERROR>"
     *
     * @return act This act identifies the asynchronous task. The corresponding callback invocation
     * provides the same act with help of payload->getAct().
     */
    template < typename RESULT, typename ERROR_ >
    act_t asyncClass(
        AsyncResultErrorCallbackIF< RESULT, ERROR_ >& cb,
        const ::boost::shared_ptr< AsyncResultErrorTask< RESULT, ERROR_ > >& task) const;

    /**
     * Executes a function asynchronously without any callback (fire and forget). The asyncFunction
     * returns immediately. The passed function will be executed in another thread. The parameter
     * AsyncTask& of the passed function is mandatory.
     *
     * @param task A function, void(AsyncTask&)
     */
    void asyncFunction(::boost::function< void(AsyncTask&) > runner) const;

    /**
     * Executes a function asynchronously. The asyncFunction returns immediately. The passed
     * function will be executed in another thread. The parameter AsyncTask& of the passed function
     * is mandatory. If you want to bind additional parameters to the function you have to use the
     * macro ASYNC_FUNCTION.
     * The callback interface  AsyncCompletedCallbackIF will be called if the execution of the
     * async function is finished.
     *
     * @param task Async function from type "void(AsyncTask&)"
     *
     * @return act This act identifies the asynchronous function. The corresponding callback
     * invocation provides the same act with help of payload->getAct().
     */
    act_t asyncFunction(AsyncCompletedCallbackIF& cb,
                        ::boost::function< void(AsyncTask&) > runner) const;

    /**
     * Executes a function asynchronously. The asyncFunction returns immediately. The passed
     * function will be executed in another thread. The parameter AsyncErrorTask<ERROR_>& of the
     * passed function is mandatory. The parameter is used to set the error (return value) out of
     * the async function. If you want to bind additional parameters to the function
     * you have to use the macro ASYNC_FUNCTION.
     * The callback interface AsyncErrorCallbackIF< ERROR_ > will be called if the execution of the
     * async function is finished.
     *
     * @param task Async function from type "void(AsyncErrorTask<ERROR>&)"
     *
     * @return act This act identifies the asynchronous function. The corresponding callback
     * invocation provides the same act with help of payload->getAct().
     */
    template < typename ERROR_ >
    act_t asyncFunction(AsyncErrorCallbackIF< ERROR_ >& cb,
                        ::boost::function< void(AsyncErrorTask< ERROR_ >&) > runner) const;

    /**
     * Executes a function asynchronously. The asyncFunction returns immediately. The passed
     * function will be executed in another thread. The parameter AsyncResultTask<RESULT>& of the
     * passed function is mandatory. The parameter is used to set the result (return value) out of
     * the async function. If you want to bind additional parameters to the function
     * you have to use the macro ASYNC_FUNCTION.
     * The callback interface AsyncResultCallbackIF<RESULT> will be called if the execution of the
     * async function is finished.
     *
     * @param task Async function from type "void(AsyncResultTask< RESULT >&)"
     *
     * @return act This act identifies the asynchronous function. The corresponding callback
     * invocation provides the same act with help of payload->getAct().
     */
    template < typename RESULT >
    act_t asyncFunction(AsyncResultCallbackIF< RESULT >& cb,
                        ::boost::function< void(AsyncResultTask< RESULT >&) > runner) const;

    /**
     * Executes a function asynchronously. The asyncFunction returns immediately. The passed
     * function will be executed in another thread. The parameter
     * AsyncResultErrorTask<RESULT,ERROR>& of the passed function is mandatory. The parameter is
     * used to set the result or error (return value) out of the async function. If you want to bind
     * additional parameters to the function you have to use the macro ASYNC_FUNCTION.
     * The callback interface AsyncResultErrorCallbackIF<RESULT,ERROR> will be called if the
     * execution of the async function returns a result or an error.
     *
     * @param task Async function from type "void(AsyncResultErrorTask<RESULT,ERROR>&)"
     *
     * @return act This act identifies the asynchronous function. The corresponding callback
     * invocation provides the same act with help of payload->getAct().
     */
    template < typename RESULT, typename ERROR_ >
    act_t asyncFunction(
        AsyncResultErrorCallbackIF< RESULT, ERROR_ >& cb,
        ::boost::function< void(AsyncResultErrorTask< RESULT, ERROR_ >&) > runner) const;

private:
    friend class Logger;

    void sendComponentMessage(boost::shared_ptr< ComponentMessage > msg);

    ::asf::identifier_t getId() const;

    bool isCurrentComponent() const;

    template < typename CB, typename P >
    act_t sendLocalMessage(CB& cb, boost::shared_ptr< P > payload);

    void executeAsyncTask(const ::boost::shared_ptr< ::asf::threading::RunnableIF >& run) const;

    template < typename CB, typename TASK >
    act_t asyncGeneric(CB& cb, TASK& task) const;

    ComponentDescription* _componentDescription;

    DECLARE_CLASS_LOGGER();
};

// The method "act_t sendLocalMessage(LocalMessageCallbackIF0& cb)" is implemented
// in the BaseComponent.cpp file as it doesn't have template arguments.

template < typename T0 >
act_t BaseComponent::sendLocalMessage(LocalMessageCallbackIF1< T0 >& cb, const T0& t0) {
    return sendLocalMessage< LocalMessageCallbackIF1< T0 >, PayloadTuple1< T0 > >(
        cb, boost::shared_ptr< PayloadTuple1< T0 > >(new PayloadTuple1< T0 >(t0)));
}

template < typename T0, typename T1 >
act_t BaseComponent::sendLocalMessage(LocalMessageCallbackIF2< T0, T1 >& cb,
                                      const T0& t0,
                                      const T1& t1) {
    return sendLocalMessage< LocalMessageCallbackIF2< T0, T1 >, PayloadTuple2< T0, T1 > >(
        cb, boost::shared_ptr< PayloadTuple2< T0, T1 > >(new PayloadTuple2< T0, T1 >(t0, t1)));
}

template < typename T0, typename T1, typename T2 >
act_t BaseComponent::sendLocalMessage(LocalMessageCallbackIF3< T0, T1, T2 >& cb,
                                      const T0& t0,
                                      const T1& t1,
                                      const T2& t2) {
    return sendLocalMessage< LocalMessageCallbackIF3< T0, T1, T2 >, PayloadTuple3< T0, T1, T2 > >(
        cb,
        boost::shared_ptr< PayloadTuple3< T0, T1, T2 > >(
            new PayloadTuple3< T0, T1, T2 >(t0, t1, t2)));
}

template < typename CB, typename P >
act_t BaseComponent::sendLocalMessage(CB& cb, boost::shared_ptr< P > payload) {
    LOG_ASSERT_FATAL_MSG(isCurrentComponent(),
                         "sending local message to foreign component not allowed");
    ::boost::shared_ptr< LocalMessage< CB, P > > msg(new LocalMessage< CB, P >(cb, payload));
    msg->logMessageSending();
    sendComponentMessage(msg);
    return msg->getAct();
}

template < typename ERROR_ >
act_t BaseComponent::asyncClass(AsyncErrorCallbackIF< ERROR_ >& cb,
                                const ::boost::shared_ptr< AsyncErrorTask< ERROR_ > >& task) const {
    return asyncGeneric< AsyncErrorCallbackIF< ERROR_ >,
                         const ::boost::shared_ptr< AsyncErrorTask< ERROR_ > > >(cb, task);
}

template < typename RESULT >
act_t BaseComponent::asyncClass(
    AsyncResultCallbackIF< RESULT >& cb,
    const ::boost::shared_ptr< AsyncResultTask< RESULT > >& task) const {
    return asyncGeneric< AsyncResultCallbackIF< RESULT >,
                         const ::boost::shared_ptr< AsyncResultTask< RESULT > > >(cb, task);
}

template < typename RESULT, typename ERROR_ >
act_t BaseComponent::asyncClass(
    AsyncResultErrorCallbackIF< RESULT, ERROR_ >& cb,
    const ::boost::shared_ptr< AsyncResultErrorTask< RESULT, ERROR_ > >& task) const {
    return asyncGeneric< AsyncResultErrorCallbackIF< RESULT, ERROR_ >,
                         const ::boost::shared_ptr< AsyncResultErrorTask< RESULT, ERROR_ > > >(
        cb, task);
}

template < typename CB, typename TASK >
act_t BaseComponent::asyncGeneric(CB& cb, TASK& task) const {
    AsyncTaskRunnable< CB, TASK >* runner =
        new AsyncTaskRunnable< CB, TASK >(cb, task, _componentDescription->getContainer());
    CHECK_ALLOCATION(runner);
    ::boost::shared_ptr< AsyncTaskRunnable< CB, TASK > > runnerShared(runner);
    act_t act = runnerShared->getAct();
    LOG_ASYNC_LOGGER();
    LOG_INFO_STATIC("-> async, act=%" PRIuPTR ", cb=%p", act, &cb);
    executeAsyncTask(runnerShared);
    return act;
}

template < typename ERROR_ >
act_t BaseComponent::asyncFunction(
    AsyncErrorCallbackIF< ERROR_ >& cb,
    ::boost::function< void(AsyncErrorTask< ERROR_ >&) > runner) const {
    ::boost::shared_ptr< AsyncErrorTaskRunner< ERROR_ > > task(
        new AsyncErrorTaskRunner< ERROR_ >());
    task->setRunner(runner);
    return asyncGeneric< AsyncErrorCallbackIF< ERROR_ >,
                         ::boost::shared_ptr< AsyncErrorTaskRunner< ERROR_ > > >(cb, task);
}

template < typename RESULT >
act_t BaseComponent::asyncFunction(
    AsyncResultCallbackIF< RESULT >& cb,
    ::boost::function< void(AsyncResultTask< RESULT >&) > runner) const {
    ::boost::shared_ptr< AsyncResultTaskRunner< RESULT > > task(
        new AsyncResultTaskRunner< RESULT >());
    task->setRunner(runner);
    return asyncGeneric< AsyncResultCallbackIF< RESULT >,
                         ::boost::shared_ptr< AsyncResultTaskRunner< RESULT > > >(cb, task);
}

template < typename RESULT, typename ERROR_ >
act_t BaseComponent::asyncFunction(
    AsyncResultErrorCallbackIF< RESULT, ERROR_ >& cb,
    ::boost::function< void(AsyncResultErrorTask< RESULT, ERROR_ >&) > runner) const {
    ::boost::shared_ptr< AsyncResultErrorTaskRunner< RESULT, ERROR_ > > task(
        new AsyncResultErrorTaskRunner< RESULT, ERROR_ >());
    task->setRunner(runner);
    return asyncGeneric< AsyncResultErrorCallbackIF< RESULT, ERROR_ >,
                         ::boost::shared_ptr< AsyncResultErrorTaskRunner< RESULT, ERROR_ > > >(
        cb, task);
}

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_BASECOMPONENT_H
