/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     Application.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_APPLICATION_H
#define ASF_CORE_APPLICATION_H

#include <boost/shared_ptr.hpp>
#include <vector>
#include "asf/core/ApplicationIF.h"
#include "asf/core/CommunicationStack.h"
#include "asf/core/ComponentContainer.h"
#include "asf/core/ComponentDescription.h"
#include "asf/core/ComponentMessage.h"
#include "asf/core/ComponentMessageAdapter.h"
#include "asf/core/MessageQueue.h"
#include "asf/core/Types.h"
#include "asf/core/Uri.h"
#include "asf/threading/Mutex.h"
#include "asf/threading/Signal.h"
#include "asf/threading/TaskPool.h"

namespace asf {
namespace core {

/**
 * Internal representation of an ASF application.
 */
class Application : public ApplicationIF, public ApplicationBinderIF {
public:
    Application(const std::string& packageName,
                const std::string& name,
                const Version& version = ::asf::core::Version(),
                const std::vector< std::string >& arguments = std::vector< std::string >(),
                uint32 asyncTaskCount = 4);

    virtual ~Application();

    // implementation of ApplicationIF

    virtual const std::string& getName() const;

    virtual const std::string& getPackageName() const;

    virtual const Version& getVersion() const;

    virtual int getTerminationExitCode() const;

    virtual void setTerminationExitCode(int terminationExitCode);

    virtual void shutdown(int exitCode = 0);

    virtual const std::vector< std::string >& getArguments() const;

    virtual bool hasArguments() const;

    static Application* getApplication();

    static Application* createApplication(
        const std::string& packageName,
        const std::string& name,
        const Version& version = ::asf::core::Version(),
        const std::vector< std::string >& arguments = std::vector< std::string >(),
        uint32 asyncTaskCount = 4);

    static void deleteApplication();

    void enableExceptionHandler(bool enable);

    // Used by the communication stacks to transport incoming messages and
    // events

    virtual void onReceivedMessage(ComponentMessage& msg);

    virtual void onDisconnected(ConnectionMessageSharedPtr msg);

    virtual void onConnected(ConnectionMessageSharedPtr msg);

    // Used by the generated main() function to assemble a complete
    // Application.

    virtual void start();

    virtual int waitForCompletion();

    virtual void sendMessage(ComponentMessage& msg);

    virtual void addContainer(ComponentContainerSharedPtr container);

    virtual void addStubComStack(CommunicationStack* stack);

    virtual size_t getStubComStackCount();

    virtual void addProxyComStack(CommunicationStack* stack);

    virtual size_t getProxyComStackCount();

    virtual CommunicationStack* getProxyComStack(const std::string& scheme);

    virtual void addStarter(StartStopWaitIFSharedPtr starter);

    virtual bool isShutdownActive() const;

    static bool staticIsShutdownActive();

    virtual bool bindPorts(ProvidedPort& providedPort, ExportedPort& exportedPort);

    virtual bool bindPorts(RequiredPort& requiredPort, ImportedPort& importedPort);

    virtual bool bindPorts(RequiredPort& requiredPort, ProvidedPort& providedPort);

    virtual ProvidedPort* findProvidedPort(const std::string& serviceName,
                                           const std::string& portName);

    virtual ComponentDescriptionSharedPtr findComponentDescription(
        const std::string& componentName);

    /**
     * This function moves all fragments (required and provided ports)
     * from the source component to the destination component. All ports
     * from the source component are part of the destination component after
     * the call. Be sure what you are doing, because this call changes the
     * model on run time.
     */
    virtual void importComponentFragment(const std::string& srcComponentName,
                                         const std::string& destinationComponentName);

    virtual void addShutdownHook(::asf::core::HookContainer::VoidFunctionPtr, int priority = 0);

    /**
     * Shutdown the current application.
     *
     * @return true if an application exists and the shutdown was initiated.
     */
    static bool shutdownCurrentApplication(int exitCode = 0);

    /**
     * Terminates the current application by initiating the application shutdown. The application
     * will
     * exit with an exit code as set via setTerminateExitCode() (default value of 255). This method
     * is used by ASF when receiving a SIGTERM signal.
     *
     * @return true if an application exists and the shutdown was initiated.
     */
    static bool terminateCurrentApplication();

    void executeAsyncTask(const ::boost::shared_ptr< ::asf::threading::RunnableIF >& run) {
        _asyncTaskPool.execute(run);
    }

private:
    void createImmediateComponents();

    void onConnectionMessage(ConnectionMessageSharedPtr& msg);

    void doShutdown(int exitCode);

    static bool doShutdownCurrent(int exitCode, bool termination = false);

    std::string _packageName;

    std::string _name;

    bool _exceptionHandlerEnabled;

    ::asf::threading::Mutex _lockStarter;

    std::vector< StartStopWaitIFSharedPtr > _starter;

    std::vector< CommunicationStack* > _stubStacks;

    std::vector< CommunicationStack* > _proxyStacks;

    ::asf::threading::Mutex _lockContainers;

    std::vector< ComponentContainerSharedPtr > _containers;

    int _exitCode;

    int _terminationExitCode;

    std::vector< std::string > _arguments;

    Version _version;

    static bool _shutdownActive;

    ::asf::core::MessageQueue< bool > _shutdownQueue;

    ::asf::core::HookContainer _shutdownHooks;

    static Application* _instance;

    // A global lock to safely access the _instance field,
    // protect against simultaneously destructor call
    static ::asf::threading::Mutex _lock;

    ::asf::threading::TaskPool _asyncTaskPool;

protected:
    DECLARE_CLASS_LOGGER();
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_APPLICATION_H
