/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     SocketServer.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CONNECTION_SOCKETSERVER_H
#define ASF_CONNECTION_SOCKETSERVER_H

#include "asf/core/IncomingOutgoingIF.h"
#include "asf/core/Logger.h"
#include "asf/core/TokenPool.h"
#include "asf/core/Types.h"
#include "asf/threading/Mutex.h"
#include "asf/threading/ThreadRunnableExecutor.h"

#ifdef WIN32
#include <winsock2.h>
#else
#include <netinet/in.h>
#endif

#include <set>

#include <boost/enable_shared_from_this.hpp>
#include <boost/shared_ptr.hpp>

namespace asf {
namespace connection {

#ifdef WIN32
typedef SOCKET socket_t;
#else
typedef int socket_t;
#endif

class SocketClientHandler : public ::asf::core::ConnectionIF,
                            public ::asf::threading::RunnableIF,
                            public ::boost::enable_shared_from_this< SocketClientHandler > {
public:
    SocketClientHandler(socket_t socketToClient,
                        uint16 parentPort,
                        size_t bufferLength,
                        ::asf::core::IncomingIF* incomingEvent);

    virtual ~SocketClientHandler();

    ssize_t sendData(size_t size, const uint8* data);

    ::asf::identifier_t getConnectionId() { return _connectionId; }

    socket_t getSocket() const { return _socketToClient; }

    bool isConnected() const { return _isConnected; }

    ::boost::shared_ptr< SocketClientHandler > selfSharedPtr() { return shared_from_this(); }

    void initialize();

    void stop();

    void waitForCompletion();

private:
    SocketClientHandler(const SocketClientHandler&);

    SocketClientHandler& operator=(const SocketClientHandler&);

    RunnableIF::tenRun enRun();

    void closeSocket();

    ::asf::threading::ThreadRunnableExecutor* _clientThread;

    ::asf::core::IncomingIF* _incomingEvent;

    bool _isConnected;

    socket_t _socketToClient;

    ::asf::identifier_t _connectionId;

    uint16 _parentPort;

    size_t _bufferLength;

    uint8* _pBuffer;

    ::asf::threading::Mutex _lockSocketFd;

    static ::asf::core::TokenPool< ::asf::identifier_t > _tokenPool;

    DECLARE_CLASS_LOGGER();
};

class SocketServer : public ::asf::core::OutgoingIF,
                     public ::asf::threading::RunnableIF,
                     public ::asf::core::StartStopWaitIF {
public:
    SocketServer(::asf::core::IncomingIF* poIncomingEvent,
                 uint16 port,
                 size_t bufferLength,
                 const std::string& bindToAddress = "",
                 const ::std::string& bindToInterface = "");

    virtual ~SocketServer();

    // implement StartStopWaitIF
    void start();

    void stop();

    void waitForCompletion();

    // implement OutgoingIF
    virtual ::asf::core::ConnectionIFSharedPtr connectServer(const std::string& address);

    virtual ssize_t sendData(::asf::core::ConnectionIFSharedPtr connection,
                             size_t size,
                             const uint8* data);

private:
    SocketServer(const SocketServer&);

    SocketServer& operator=(const SocketServer&);

    void createAndAddClient(socket_t socketToClient);

    void removeDisconnectedClients();

    void clearAllClients();

    void initializeServer();

    void closeListenSocket();

    RunnableIF::tenRun enRun();

    ::asf::threading::ThreadRunnableExecutor* _mainThread;

    ::asf::core::IncomingIF* _incomingEvent;

    uint16 _port;

    size_t _bufferLength;

    ::std::string _bindToAddress;

    ::std::string _bindToInterface;

    bool _isStopped;

    socket_t _listenSocket;

    ::asf::threading::Mutex _lock;

    std::set< ::boost::shared_ptr< SocketClientHandler > > _clients;

    DECLARE_CLASS_LOGGER();
};

}  // namespace connection
}  // namespace asf

#endif  // ASF_CONNECTION_SOCKETSERVER_H
