/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CCaProxyRegistry.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_CCA_CCAPROXYREGISTRY_H
#define ASF_CCA_CCAPROXYREGISTRY_H

#include "asf/core/IncomingOutgoingIF.h"
#include "asf/core/Logger.h"
#include "asf/core/Proxy.h"
#include "asf/core/Types.h"

#include "asf/threading/Mutex.h"

#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>
#include <map>
#include <string>

namespace asf {

namespace core {
class RequiredPort;
}

namespace cca {

class CcaProxyDelegate;
class CcaConnector;

/**
 * The CcaProxyRegistry is used to determine the proxy and with it the
 * component which belongs to a received CCA message. Whenever a CCA
 * proxy message (a message from server to client) is received a lookup
 * in the registry happens.
 *
 * The CcaProxyRegistry stores all available proxies and provides finder
 * methods to lookup a certain proxy by the serviceId and subId of a
 * received message.
 *
 * There is a single proxy registry in the system. Whenever a CCA proxy
 * instance is created it adds itself to the registry. On destruction it
 * removes itself from the registry.
 *
 * The CcaProxyRegistry is thread safe and can be used concurrently from
 * multiple threads.
 */
class CcaProxyRegistry {
public:
    bool registerProxy(const ::boost::shared_ptr< CcaProxyDelegate >& proxyDelegate,
                       const ::boost::shared_ptr< ::asf::core::Proxy >& proxy);

    bool registerProxy(const ::boost::shared_ptr< CcaProxyDelegate >& proxyDelegate,
                       const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
                       uint16 appId,
                       uint16 majorVersion,
                       uint16 minorVersion,
                       uint32 flags);

    bool deregisterProxy(CcaProxyDelegate* proxy);

    ::boost::weak_ptr< CcaProxyDelegate > findProxy(uint16 serviceId, uint16 subId);

    ::boost::shared_ptr< ::asf::core::Proxy > getProxy(
        const std::string& portName, const ::asf::core::ServiceAvailableIF& serviceAvailable) const;

    void onDisconnect(::asf::core::ConnectionIFSharedPtr connection);

    static CcaProxyRegistry& getInstance();

    void addConnector(CcaConnector* connector);

    CcaConnector* getConnector();

    void resetConnector();

    virtual ~CcaProxyRegistry() { resetConnector(); };

private:
    CcaProxyRegistry();

    static CcaProxyRegistry* createCcaProxyRegistry();

    static CcaProxyRegistry& getCcaProxyRegistry();

    static void deleteCcaProxyRegistry();

    bool registerProxy(const ::boost::shared_ptr< CcaProxyDelegate >& proxyDelegate);

    struct ServiceAndSubId {
        uint16 _serviceId;
        uint16 _subId;

        bool operator<(const ServiceAndSubId& a) const;
    };

    typedef std::map< ServiceAndSubId, ::boost::weak_ptr< CcaProxyDelegate > >
        ServiceAndSubIdToProxyMap;

    ServiceAndSubIdToProxyMap _proxyMap;

    CcaConnector* _connector;

    ::asf::threading::Mutex _lock;

    static CcaProxyRegistry* _instance;

    DECLARE_CLASS_LOGGER();
};
}  // namespace cca
}  // namespace asf

#endif
