/**
* @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_ROUTEINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_ROUTEINFOS_H_

#include <string>
#include <sstream>
#include <vector>
#include <utility>
#include "RouteSettings.h"
#include "InfoTypes.h"


namespace navmiddleware
{

enum RouteListConfiguration
{
   ROUTE_LIST_CONFIGURATION__DEFAULT = 0,
   ROUTE_LIST_CONFIGURATION__WITH_FREEWAY_EXITS
};

enum WaypointEditMode
{
   WAYPOINT_EDIT_MODE__WITH_RECALCULATION = 0,
   WAYPOINT_EDIT_MODE__WITHOUT_RECALCULATION
};

enum WaypointListMode
{
   WAYPOINT_LIST_MODE__VIEW = 0,
   WAYPOINT_LIST_MODE__EDIT_WITH_RECALCULATION,
   WAYPOINT_LIST_MODE__EDIT_WITHOUT_RECALCULATION
};

enum RouteCalculationStatus
{
   ROUTE_CALCULATION_STATUS__UNDEFINED     = 0,
   ROUTE_CALCULATION_STATUS__IN_PROGRESS,
   ROUTE_CALCULATION_STATUS__COMPLETED,
   ROUTE_CALCULATION_STATUS__ERROR

};

enum POIOpeningHoursStatus
{
   POI_OPENING_HOURS_STATUS__OPEN = 0,
   POI_OPENING_HOURS_STATUS__ALREADY_CLOSED = 1,
   POI_OPENING_HOURS_STATUS__CLOSED_AT_ARRIVAL = 2,
   POI_OPENING_HOURS_STATUS__CLOSED_SOON_AFTER_ARRIVAL = 3
};

enum ConsumptionUnit
{
   CONSUMPTION_UNIT__MICROLITER,
   CONSUMPTION_UNIT__WATTSECOND,
   CONSUMPTION_UNIT__WATTMINUTE
};

enum Reachability
{
   REACHABILITY__STANDARD = 0,
   REACHABILITY__ECO,
   REACHABILITY__ECO_STANDARD_MODE,
   REACHABILITY__ECO_ECO_MODE,
   REACHABILITY__NOT
};

enum RangePredictionStatus
{
   RANGE_PREDICTION_STATUS__INACTIVE = 0,
   RANGE_PREDICTION_STATUS__ACTIVE,
   RANGE_PREDICTION_STATUS__ERROR
};

enum RouteCalculationMode
{
   ROUTE_CALCULATION_MODE__DEFAULT = 0,
   ROUTE_CALCULATION_MODE__STEALTH
};

enum AutomaticChargingStationDetectionStatus
{
   AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__INACTIVE = 0,
   AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__IN_PROGRESS,
   AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__COMPLETED,
   AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__ERROR
};

enum AutomaticChargingStationDetectionReason
{
   AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNKNOWN = 0,
   AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNREACHABLE_DESTINATION,
   AUTOMATIC_CHARGING_STATION_DETECTION_REASON__OPTIMIZATION
};

enum ExpirationStatus
{
   EXPIRATION_STATUS__NOT_EXPIRED = 0,
   EXPIRATION_STATUS__EXPIRED
};

inline ::std::string toString(POIOpeningHoursStatus openingHoursStatus)
{
   switch (openingHoursStatus)
   {
   case POI_OPENING_HOURS_STATUS__OPEN:
      return "POI_OPENING_HOURS_STATUS__OPEN";
   case POI_OPENING_HOURS_STATUS__ALREADY_CLOSED:
      return "POI_OPENING_HOURS_STATUS__ALREADY_CLOSED";
   case POI_OPENING_HOURS_STATUS__CLOSED_AT_ARRIVAL:
      return "POI_OPENING_HOURS_STATUS__CLOSED_AT_ARRIVAL";
   case POI_OPENING_HOURS_STATUS__CLOSED_SOON_AFTER_ARRIVAL:
      return "POI_OPENING_HOURS_STATUS__CLOSED_SOON_AFTER_ARRIVAL";
   default:
      ::std::stringstream stream;
      stream << "OPENING_HOURS_STATUS__<" << static_cast<unsigned int>(openingHoursStatus) << ">";
      return stream.str();
   }
}

inline ::std::string toString(Reachability reachability)
{
   switch (reachability)
   {
   case REACHABILITY__STANDARD:
      return "REACHABILITY__STANDARD";
   case REACHABILITY__ECO:
      return "REACHABILITY__ECO";
   case REACHABILITY__ECO_STANDARD_MODE:
      return "REACHABILITY__ECO_STANDARD_MODE";
   case REACHABILITY__ECO_ECO_MODE:
      return "REACHABILITY__ECO_ECO_MODE";
   case REACHABILITY__NOT:
      return "REACHABILITY__NOT";
   default:
      ::std::stringstream stream;
      stream << "REACHABILITY__<" << static_cast<unsigned int>(reachability) << ">";
      return stream.str();
   }
}

inline ::std::string toString(RouteCalculationStatus routeCalculationStatus)
{
   switch (routeCalculationStatus)
   {
   case ROUTE_CALCULATION_STATUS__UNDEFINED:
      return "ROUTE_CALCULATION_STATUS__UNDEFINED";
   case ROUTE_CALCULATION_STATUS__IN_PROGRESS:
      return "ROUTE_CALCULATION_STATUS__IN_PROGRESS";
   case ROUTE_CALCULATION_STATUS__COMPLETED:
      return "ROUTE_CALCULATION_STATUS__COMPLETED";
   case ROUTE_CALCULATION_STATUS__ERROR:
      return "ROUTE_CALCULATION_STATUS__ERROR";
   default:
      ::std::stringstream stream;
      stream << "ROUTE_CALCULATION_STATUS__<" << static_cast<unsigned int>(routeCalculationStatus) << ">";
      return stream.str();
   }
}

inline ::std::string toString(RangePredictionStatus rangePredictionStatus)
{
   switch (rangePredictionStatus)
   {
   case RANGE_PREDICTION_STATUS__INACTIVE:
      return "RANGE_PREDICTION_STATUS__INACTIVE";
   case RANGE_PREDICTION_STATUS__ACTIVE:
      return "RANGE_PREDICTION_STATUS__ACTIVE";
   case RANGE_PREDICTION_STATUS__ERROR:
      return "RANGE_PREDICTION_STATUS__ERROR";
   default:
      ::std::stringstream stream;
      stream << "RANGE_PREDICTION_STATUS__<" << static_cast<unsigned int>(rangePredictionStatus) << ">";
      return stream.str();
   }
}

inline ::std::string toString(AutomaticChargingStationDetectionStatus automaticChargingStationDetectionStatus)
{
   switch (automaticChargingStationDetectionStatus)
   {
   case AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__INACTIVE:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__INACTIVE";
   case AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__IN_PROGRESS:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__IN_PROGRESS";
   case AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__COMPLETED:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__COMPLETED";
   case AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__ERROR:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__ERROR";
   default:
      ::std::stringstream stream;
      stream << "AutomaticChargingStationDetectionStatus__<" << static_cast<unsigned int>(automaticChargingStationDetectionStatus) << ">";
      return stream.str();
   }
}

inline ::std::string toString(AutomaticChargingStationDetectionReason automaticChargingStationDetectionReason)
{
   switch (automaticChargingStationDetectionReason)
   {
   case AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNKNOWN:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNKNOWN";
   case AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNREACHABLE_DESTINATION:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNREACHABLE_DESTINATION";
   case AUTOMATIC_CHARGING_STATION_DETECTION_REASON__OPTIMIZATION:
      return "AUTOMATIC_CHARGING_STATION_DETECTION_REASON__OPTIMIZATION";
   default:
      ::std::stringstream stream;
      stream << "AutomaticChargingStationDetectionReason__<" << static_cast<unsigned int>(automaticChargingStationDetectionReason) << ">";
      return stream.str();
   }
}

inline ::std::string toString(RouteCalculationMode routeCalculationMode)
{
   switch (routeCalculationMode)
   {
   case ROUTE_CALCULATION_MODE__DEFAULT:
      return "ROUTE_CALCULATION_MODE__DEFAULT";
   case ROUTE_CALCULATION_MODE__STEALTH:
      return "ROUTE_CALCULATION_MODE__STEALTH";
   default:
      ::std::stringstream stream;
      stream << "RouteCalculationMode__<" << static_cast<unsigned int>(routeCalculationMode) << ">";
      return stream.str();
   }
}

inline ::std::string toString(ExpirationStatus expirationStatus)
{
   switch (expirationStatus)
   {
   case EXPIRATION_STATUS__NOT_EXPIRED:
      return "EXPIRATION_STATUS__NOT_EXPIRED";
   case EXPIRATION_STATUS__EXPIRED:
      return "EXPIRATION_STATUS__EXPIRED";
   default:
      ::std::stringstream stream;
      stream << "ExpirationStatus__<" << static_cast<unsigned int>(expirationStatus) << ">";
      return stream.str();
   }
}

/**
 * Structure contains the road type information present in a route or routeListElement.
 *
 * Road type info common for both route and routeListElement - TollRoad, Motorway, Freeway, Ferry, AutoTrain,
 * TimeRestrictedRoad, SubscriptionRoad, UnpavedRoad, HOVRoad, SeasonalClosed.
 * The corresponding members will provide info regarding whether a particular road type is present or not.
 * Incase if a particular route option is violated in the calculated route, that info will also be provided.
 *
 * Road type info specific to routeListElement - Tunnel, OffRoad, ProhibitionOnFollowingTurn,
 * InConstruction, CurrentlyClosed, VehicleDimensionExceeded, BorderCrossing, ServiceAccess,
 * ControlledAccessExit, MotorwayIntersection, Waypoint, FinalDestination.
 * The corresponding members will only provide info regarding whether a particular road type is present or not.
 */

struct RouteInfoProperties
{
   enum Status
   {
      STATUS__NOT_PRESENT = 0,
      STATUS__PRESENT,
      STATUS__PRESENT_DESPITE_AVOIDING
   };

   RouteInfoProperties():
      m_tollRoad(STATUS__NOT_PRESENT),
      m_motorway(STATUS__NOT_PRESENT),
      m_freeway(STATUS__NOT_PRESENT),
      m_ferry(STATUS__NOT_PRESENT),
      m_autoTrain(STATUS__NOT_PRESENT),
      m_timeRestrictedRoad(STATUS__NOT_PRESENT),
      m_subscriptionRoad(STATUS__NOT_PRESENT),
      m_unpavedRoad(STATUS__NOT_PRESENT),
      m_hovRoad(STATUS__NOT_PRESENT),
      m_tunnel(STATUS__NOT_PRESENT),
      m_seasonalClosed(STATUS__NOT_PRESENT),
      m_offRoad(STATUS__NOT_PRESENT),
      m_prohibitionOnFollowingTurn(STATUS__NOT_PRESENT),
      m_inConstruction(STATUS__NOT_PRESENT),
      m_currentlyClosed(STATUS__NOT_PRESENT),
      m_vehicleDimensionExceeded(STATUS__NOT_PRESENT),
      m_borderCrossing(STATUS__NOT_PRESENT),
      m_serviceAccess(STATUS__NOT_PRESENT),
      m_controlledAccessExit(STATUS__NOT_PRESENT),
      m_motorwayIntersection(STATUS__NOT_PRESENT),
      m_waypoint(STATUS__NOT_PRESENT),
      m_finalDestination(STATUS__NOT_PRESENT),
      m_restrictedAccess(STATUS__NOT_PRESENT) {}

   RouteInfoProperties(const RouteInfoProperties& routeInfoProperties):
      m_tollRoad(routeInfoProperties.m_tollRoad),
      m_motorway(routeInfoProperties.m_motorway),
      m_freeway(routeInfoProperties.m_freeway),
      m_ferry(routeInfoProperties.m_ferry),
      m_autoTrain(routeInfoProperties.m_autoTrain),
      m_timeRestrictedRoad(routeInfoProperties.m_timeRestrictedRoad),
      m_subscriptionRoad(routeInfoProperties.m_subscriptionRoad),
      m_unpavedRoad(routeInfoProperties.m_unpavedRoad),
      m_hovRoad(routeInfoProperties.m_hovRoad),
      m_tunnel(routeInfoProperties.m_tunnel),
      m_seasonalClosed(routeInfoProperties.m_seasonalClosed),
      m_offRoad(routeInfoProperties.m_offRoad),
      m_prohibitionOnFollowingTurn(routeInfoProperties.m_prohibitionOnFollowingTurn),
      m_inConstruction(routeInfoProperties.m_inConstruction),
      m_currentlyClosed(routeInfoProperties.m_currentlyClosed),
      m_vehicleDimensionExceeded(routeInfoProperties.m_vehicleDimensionExceeded),
      m_borderCrossing(routeInfoProperties.m_borderCrossing),
      m_serviceAccess(routeInfoProperties.m_serviceAccess),
      m_controlledAccessExit(routeInfoProperties.m_controlledAccessExit),
      m_motorwayIntersection(routeInfoProperties.m_motorwayIntersection),
      m_waypoint(routeInfoProperties.m_waypoint),
      m_finalDestination(routeInfoProperties.m_finalDestination),
      m_restrictedAccess(routeInfoProperties.m_restrictedAccess) {}

   bool operator == (const RouteInfoProperties& rhs) const
   {
      return((m_tollRoad == rhs.m_tollRoad)&&
             (m_motorway == rhs.m_motorway)&&
             (m_freeway == rhs.m_freeway)&&
             (m_ferry == rhs.m_ferry)&&
             (m_autoTrain == rhs.m_autoTrain)&&
             (m_timeRestrictedRoad == rhs.m_timeRestrictedRoad)&&
             (m_subscriptionRoad == rhs.m_subscriptionRoad)&&
             (m_unpavedRoad == rhs.m_unpavedRoad)&&
             (m_hovRoad == rhs.m_hovRoad)&&
             (m_tunnel == rhs.m_tunnel)&&
             (m_seasonalClosed == rhs.m_seasonalClosed)&&
             (m_offRoad == rhs.m_offRoad)&&
             (m_prohibitionOnFollowingTurn == rhs.m_prohibitionOnFollowingTurn)&&
             (m_inConstruction == rhs.m_inConstruction)&&
             (m_currentlyClosed == rhs.m_currentlyClosed)&&
             (m_vehicleDimensionExceeded == rhs.m_vehicleDimensionExceeded)&&
             (m_borderCrossing == rhs.m_borderCrossing)&&
             (m_serviceAccess == rhs.m_serviceAccess)&&
             (m_controlledAccessExit == rhs.m_controlledAccessExit)&&
             (m_motorwayIntersection == rhs.m_motorwayIntersection)&&
             (m_waypoint == rhs.m_waypoint)&&
             (m_finalDestination == rhs.m_finalDestination)&&
             (m_restrictedAccess == rhs.m_restrictedAccess));
   }

   RouteInfoProperties& operator = (const RouteInfoProperties& rhs)
   {
      if (&rhs != this)
      {
         m_tollRoad = rhs.m_tollRoad;
         m_motorway = rhs.m_motorway;
         m_freeway = rhs.m_freeway;
         m_ferry = rhs.m_ferry;
         m_autoTrain = rhs.m_autoTrain;
         m_timeRestrictedRoad = rhs.m_timeRestrictedRoad;
         m_subscriptionRoad = rhs.m_subscriptionRoad;
         m_unpavedRoad = rhs.m_unpavedRoad;
         m_hovRoad = rhs.m_hovRoad;
         m_tunnel = rhs.m_tunnel;
         m_seasonalClosed = rhs.m_seasonalClosed;
         m_offRoad = rhs.m_offRoad;
         m_prohibitionOnFollowingTurn = rhs.m_prohibitionOnFollowingTurn;
         m_inConstruction = rhs.m_inConstruction;
         m_currentlyClosed = rhs.m_currentlyClosed;
         m_vehicleDimensionExceeded = rhs.m_vehicleDimensionExceeded;
         m_borderCrossing = rhs.m_borderCrossing;
         m_serviceAccess = rhs.m_serviceAccess;
         m_controlledAccessExit = rhs.m_controlledAccessExit;
         m_motorwayIntersection = rhs.m_motorwayIntersection;
         m_waypoint = rhs.m_waypoint;
         m_finalDestination = rhs.m_finalDestination;
         m_restrictedAccess = rhs.m_restrictedAccess;
      }
      return *this;
   }

   bool operator != (const RouteInfoProperties& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      bool hasRoadTypeInfo = false;

      if(m_tollRoad != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_tollRoad == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Toll Road present despite avoiding" << ::std::endl) :
         (stream << "Toll Road" << ::std::endl) ;
      }

      if(m_motorway != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_motorway == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Motorway present despite avoiding" << ::std::endl) :
         (stream << "Motorway" << ::std::endl) ;
      }

      if(m_freeway != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_freeway == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Freeway present despite avoiding" << ::std::endl) :
         (stream << "Freeway" << ::std::endl) ;
      }

      if(m_ferry != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_ferry == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Ferry present despite avoiding" << ::std::endl) :
         (stream << "Ferry" << ::std::endl) ;
      }

      if(m_autoTrain != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_autoTrain == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Auto Train present despite avoiding" << ::std::endl) :
         (stream << "Auto Train" << ::std::endl) ;
      }

      if(m_timeRestrictedRoad != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_timeRestrictedRoad == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Time Restricted Road present despite avoiding" << ::std::endl) :
         (stream << "Time Restricted Road" << ::std::endl) ;
      }

      if(m_subscriptionRoad != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_subscriptionRoad == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Subscription Road present despite avoiding" << ::std::endl) :
         (stream << "Subscription Road" << ::std::endl) ;
      }

      if(m_unpavedRoad != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_unpavedRoad == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "Unpaved Road present despite avoiding" << ::std::endl) :
         (stream << "Unpaved Road" << ::std::endl) ;
      }

      if(m_hovRoad != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_hovRoad == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "HOV present despite avoiding" << ::std::endl) :
         (stream << "HOV present" << ::std::endl) ;
      }

      if(m_tunnel != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Tunnel" << ::std::endl;
      }
      if(m_seasonalClosed != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         (m_seasonalClosed == STATUS__PRESENT_DESPITE_AVOIDING) ? (stream << "SeasonalClosed road present despite avoiding" << ::std::endl) :
         (stream << "SeasonalClosed road" << ::std::endl) ;
      }
      if(m_offRoad != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Off-Road" << ::std::endl;
      }
      if(m_prohibitionOnFollowingTurn != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Prohibition On Following Turn" << ::std::endl;
      }
      if(m_inConstruction != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "In Construction" << ::std::endl;
      }
      if(m_currentlyClosed != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Currently Closed" << ::std::endl;
      }
      if(m_vehicleDimensionExceeded != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Vehicle Dimension Exceeded" << ::std::endl;
      }
      if(m_borderCrossing != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Border Crossing" << ::std::endl;
      }
      if(m_serviceAccess != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Service Access" << ::std::endl;
      }
      if(m_controlledAccessExit != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Controlled Access Exit" << ::std::endl;
      }
      if(m_motorwayIntersection != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Motorway Intersection" << ::std::endl;
      }
      if(m_waypoint != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Waypoint" << ::std::endl;
      }
      if(m_finalDestination != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Final Destination" << ::std::endl;
      }
      if(m_restrictedAccess != STATUS__NOT_PRESENT)
      {
         hasRoadTypeInfo = true;
         stream << "Restricted access" << ::std::endl;
      }

      if(!hasRoadTypeInfo)
      {
         stream << "Road Type Info not present" << ::std::endl;
      }
      else
      {
         stream << "Road Type Info present" << ::std::endl;
      }

      return stream.str();
   }

   Status m_tollRoad;
   Status m_motorway;
   Status m_freeway;
   Status m_ferry;
   Status m_autoTrain;
   Status m_timeRestrictedRoad;
   Status m_subscriptionRoad;
   Status m_unpavedRoad;
   Status m_hovRoad;
   Status m_tunnel;
   Status m_seasonalClosed;
   Status m_offRoad;
   Status m_prohibitionOnFollowingTurn;
   Status m_inConstruction;
   Status m_currentlyClosed;
   Status m_vehicleDimensionExceeded;
   Status m_borderCrossing;
   Status m_serviceAccess;
   Status m_controlledAccessExit;
   Status m_motorwayIntersection;
   Status m_waypoint;
   Status m_finalDestination;
   Status m_restrictedAccess;
};

class StateOfCharge
{
public:
   StateOfCharge()
      : m_totalNumberOfSegments(0),
        m_numberOfChargedSegments(0),
        m_stateOfChargeAsValue(0),
        m_stateOfChargeAsPercentage(0)
   {}

   StateOfCharge(unsigned int totalNumberOfSegments,
                 unsigned numberOfChargedSegments,
                 unsigned int stateOfChargeAsValue,
                 const ::std::string& stateOfChargeAsPercentageString,
                 unsigned int stateOfChargeAsPercentage)
      : m_totalNumberOfSegments(totalNumberOfSegments)
      , m_numberOfChargedSegments(numberOfChargedSegments)
      , m_stateOfChargeAsValue(stateOfChargeAsValue)
      , m_stateOfChargeAsPercentageString(stateOfChargeAsPercentageString)
      , m_stateOfChargeAsPercentage(stateOfChargeAsPercentage)
   {}

   StateOfCharge& operator = (const StateOfCharge& rhs)
   {
      if (&rhs != this)
      {
         m_totalNumberOfSegments = rhs.m_totalNumberOfSegments;
         m_numberOfChargedSegments = rhs.m_numberOfChargedSegments;
         m_stateOfChargeAsValue = rhs.m_stateOfChargeAsValue;
         m_stateOfChargeAsPercentageString = rhs.m_stateOfChargeAsPercentageString;
         m_stateOfChargeAsPercentage = rhs.m_stateOfChargeAsPercentage;
      }
      return *this;
   }

   bool operator == (const StateOfCharge& rhs) const
   {
      return (m_totalNumberOfSegments == rhs.m_totalNumberOfSegments)&&
             (m_numberOfChargedSegments == rhs.m_numberOfChargedSegments)&&
             (m_stateOfChargeAsValue == rhs.m_stateOfChargeAsValue)&&
             (m_stateOfChargeAsPercentageString == rhs.m_stateOfChargeAsPercentageString)&&
             (m_stateOfChargeAsPercentage == rhs.m_stateOfChargeAsPercentage);
   }

   bool operator != (const StateOfCharge& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("StateOfCharge payload:\n");

      stream << ::std::endl
             << "Total number of segments = " << m_totalNumberOfSegments  << ::std::endl
             << "number of charged segments = " << m_numberOfChargedSegments  << ::std::endl
             << "State of charge as value = " << m_stateOfChargeAsValue << ::std::endl
             << "State of charge as percentage value string = "<< m_stateOfChargeAsPercentageString << ::std::endl
             << "State of charge as percentage value = " << m_stateOfChargeAsPercentage << ::std::endl;
      return stream.str();
   }

   unsigned int getTotalNumberOfBatterySegments()
   {
      return m_totalNumberOfSegments;
   }

   void setTotalNumberOfBatterySegments(unsigned int totalNumberOfSegments)
   {
      m_totalNumberOfSegments = totalNumberOfSegments;
   }

   unsigned int getNumberOfChargedSegments()
   {
      return m_numberOfChargedSegments;
   }

   void setNumberOfChargedSegments(unsigned int numberOfChargedSegments)
   {
      m_numberOfChargedSegments = numberOfChargedSegments;
   }

   unsigned int getStateOfChargeAsValue()
   {
      return m_stateOfChargeAsValue;
   }

   void setStateOfChargeAsValue(unsigned int stateOfChargeAsValue)
   {
      m_stateOfChargeAsValue = stateOfChargeAsValue;
   }

   ::std::string getStateOfChargeAsPercentageString()
   {
      return m_stateOfChargeAsPercentageString;
   }

   void setStateOfChargeAsPercentageString(const ::std::string& stateOfChargeAsPercentageString)
   {
      m_stateOfChargeAsPercentageString = stateOfChargeAsPercentageString;
   }

   unsigned int getStateOfChargeAsPercentage()
   {
      return m_stateOfChargeAsPercentage;
   }

   void setStateOfChargeAsPercentage(unsigned int stateOfChargeAsPercentage)
   {
      m_stateOfChargeAsPercentage = stateOfChargeAsPercentage;
   }

private:
   // total number of battery segments
   unsigned int m_totalNumberOfSegments;
   // number of charged battery segments
   unsigned int m_numberOfChargedSegments;
   // state of charge value in the unit 'Ws' - Watt second
   unsigned int m_stateOfChargeAsValue;
   // state of charge as percentage in string
   ::std::string m_stateOfChargeAsPercentageString;
   // state of charge as percentage value
   unsigned int m_stateOfChargeAsPercentage;
};

class ReachabilityInfo
{
public:
   ReachabilityInfo()
   {}

   ReachabilityInfo(ValidValue< Reachability > reachabilityMode,
                    ValidValue< unsigned int > distanceToEmptyHighProbability,
                    ValidValue< unsigned int > distanceToEmptyLowProbability)
      : m_reachabilityMode(reachabilityMode)
      , m_distanceToEmptyHighProbability(distanceToEmptyHighProbability)
      , m_distanceToEmptyLowProbability(distanceToEmptyLowProbability)
   {}

   ReachabilityInfo& operator = (const ReachabilityInfo& rhs)
   {
      if (&rhs != this)
      {
         m_reachabilityMode = rhs.m_reachabilityMode;
         m_distanceToEmptyHighProbability = rhs.m_distanceToEmptyHighProbability;
         m_distanceToEmptyLowProbability = rhs.m_distanceToEmptyLowProbability;
      }
      return *this;
   }

   bool operator == (const ReachabilityInfo& rhs) const
   {
      return (m_reachabilityMode == rhs.m_reachabilityMode)&&
             (m_distanceToEmptyHighProbability == rhs.m_distanceToEmptyHighProbability)&&
             (m_distanceToEmptyLowProbability == rhs.m_distanceToEmptyLowProbability);
   }

   bool operator != (const ReachabilityInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setReachabilityMode(const ValidValue<Reachability>& reachabilityMode)
   {
      m_reachabilityMode = reachabilityMode;
   }

   const ValidValue<Reachability>& getReachabilityMode() const
   {
      return m_reachabilityMode;
   }

   void setDistanceToEmptyHighProbability(const ValidValue<unsigned int>& distanceToEmptyHighProbability)
   {
      m_distanceToEmptyHighProbability = distanceToEmptyHighProbability;
   }

   const ValidValue<unsigned int>& getDistanceToEmptyHighProbability() const
   {
      return m_distanceToEmptyHighProbability;
   }

   void setDistanceToEmptyLowProbability(const ValidValue<unsigned int>& distanceToEmptyLowProbability)
   {
      m_distanceToEmptyLowProbability = distanceToEmptyLowProbability;
   }

   const ValidValue<unsigned int>& getDistanceToEmptyLowProbability() const
   {
      return m_distanceToEmptyLowProbability;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ReachabilityInfo payload:\n");

      stream << ::std::endl
             << "Reachability Mode =" << (m_reachabilityMode.isValid()? ::navmiddleware::toString(m_reachabilityMode.getValue()) : "invalid") << ::std::endl
             << "Distance to Empty High Probability =" << m_distanceToEmptyHighProbability.toString() << ::std::endl
             << "Distance to Empty Low Probability =" << m_distanceToEmptyLowProbability.toString() << ::std::endl;
      return stream.str();
   }

private:
   ValidValue< Reachability > m_reachabilityMode;
   ValidValue< unsigned int > m_distanceToEmptyHighProbability;
   ValidValue< unsigned int > m_distanceToEmptyLowProbability;
};

class DestinationInfos
{

public:
   /**
    * Class contains latitude and longitude as well as the country and sub country code of the destination
    * The field m_countryCode: Letter string representing ISO 3166-1 ALPHA 3 country code
    * The field m_subCountryCode: Letter string representing ISO 3166-2 country subdivision code (contains only the subdivision part with up to 3 characters)
    * The field m_locationString: includes a generated name for the location, depending on the market and type of destination
    *           descriptive representation of address or special name with parts of address
    *           e.g. address NAR (street name first) has other order than address EU
    *                poi's are also named differently than addresses
    * The object m_locationAttributeInfos: includes all available details for the destination
    * The object m_destinationRouteInfoProperties: Includes the destinationRouteInfoProperties including the information about presence of different route element types
    */
   class DestinationInfo
   {
   public:
      DestinationInfo()
         : m_latitude(0.0)
         , m_longitude(0.0) {}

      DestinationInfo(double latitude, double longitude, const ::std::string& countryCode, const ::std::string& subCountryCode,
                      const ::std::string& locationString, const ValidValue<RouteInfoProperties>& destinationRouteInfoProperties,   //<-- INF4CV
                      const ValidValue<ExpirationStatus>& expirationStatus)
         : m_latitude(latitude)
         , m_longitude(longitude)
         , m_countryCode(countryCode)
         , m_subCountryCode(subCountryCode)
         , m_locationString(locationString)
         , m_destinationRouteInfoProperties(destinationRouteInfoProperties)
         , m_expirationStatus(expirationStatus)
      {}

      DestinationInfo& operator = (const DestinationInfo& rhs)
      {
         if (&rhs != this)
         {
            m_latitude = rhs.m_latitude;
            m_longitude = rhs.m_longitude;
            m_countryCode = rhs.m_countryCode;
            m_subCountryCode = rhs.m_subCountryCode;
            m_locationString = rhs.m_locationString;
            m_locationAttributeInfos = rhs.m_locationAttributeInfos;
            m_destinationRouteInfoProperties = rhs.m_destinationRouteInfoProperties;
            m_expirationStatus = rhs.m_expirationStatus;
         }
         return *this;
      }

      bool operator == (const DestinationInfo& rhs) const
      {
         return (m_latitude == rhs.m_latitude)&&
                (m_longitude == rhs.m_longitude)&&
                (m_countryCode == rhs.m_countryCode)&&
                (m_subCountryCode == rhs.m_subCountryCode)&&
                (m_locationString == rhs.m_locationString)&&
                (m_locationAttributeInfos == rhs.m_locationAttributeInfos) &&
                (m_destinationRouteInfoProperties == rhs.m_destinationRouteInfoProperties) &&
                (m_expirationStatus == rhs.m_expirationStatus);
      }

      bool operator != (const DestinationInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("DestinationInfo payload:\n");

         stream << ::std::endl
                << " Destination Longitude = " << m_longitude << ::std::endl
                << " Destination Latitude = " << m_latitude << ::std::endl
                << " Location string = " << m_locationString << ::std::endl
                << " Location attributes = " << m_locationAttributeInfos.toString() << ::std::endl
                //<-- INF4CV
                << " Destination properties = " << (m_destinationRouteInfoProperties.isValid() ?
                                                    m_destinationRouteInfoProperties.getValue().toString() :
                                                    "invalid") << ::std::endl
                //->
                << " Expiration status = "<< (m_expirationStatus.isValid() ?
                                              ::navmiddleware::toString(m_expirationStatus.getValue()) :
                                              "invalid") << ::std::endl;

         return stream.str();
      }

      double m_latitude;
      double m_longitude;
      ::std::string m_countryCode;
      ::std::string m_subCountryCode;
      ::std::string m_locationString;
      ::navmiddleware::LocationAttributeInfos m_locationAttributeInfos;
      ValidValue<RouteInfoProperties> m_destinationRouteInfoProperties;   //<-- INF4CV
      /* Expiration Status will be valid only for Google destinations*/
      ValidValue<ExpirationStatus> m_expirationStatus;
   };

   const ::std::vector<DestinationInfo>& getDestinationInfos() const
   {
      return m_destinationInfos;
   }
   ::std::vector<DestinationInfo>& getDestinationInfos()
   {
      return m_destinationInfos;
   }

   bool operator == (const DestinationInfos& rhs) const
   {
      return (m_destinationInfos == rhs.m_destinationInfos);
   }

   bool operator != (const DestinationInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationInfos payload:\n");
      stream << ::std::endl;

      for(unsigned int destinationListIndex = 0; destinationListIndex < getDestinationInfos().size(); destinationListIndex++)
      {
         stream << "Destination list index = " << destinationListIndex << ::std::endl
                << getDestinationInfos().at(destinationListIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<DestinationInfo> m_destinationInfos;
};

class ArrivalInfos
{
public:

   ArrivalInfos()
      : m_currentDestinationIndex(0)
      , m_isApproachingCurrentDest(false)
      , m_isRouteCalcCompletedForAllDestinations(false)
   {}

   class ArrivalInfo
   {
   public:
      ArrivalInfo()
         : m_distanceUnit(DISTANCEUNIT_MILES)
         , m_timeToDest(0)
         , m_distanceToDest(0)
         , m_estimatedTimeOfArrivalHour(0)
         , m_estimatedTimeOfArrivalMinute(0)
         , m_consumptionUnit(CONSUMPTION_UNIT__MICROLITER)
         , m_consumptionToDestAsInt(0)
         , m_trafficDelayToDestAsInt(0)
         , m_relativeDirectionToDestination(0)
         , m_chargingTimeAtDestinationAsInt(0)
         , m_additionalDurationOfStayAsInt(0)
         , m_stateOfChargeAfterChargingAsInt(0)
      {}

      ArrivalInfo(const ::std::string& timeToDestination, const ::std::string& distanceToDestination,
                  const ::std::string& distanceToDestinationNoUnit, DistanceUnit distanceUnit,
                  const ::std::string& estimatedTimeOfArrival,
                  ValidValue< POIOpeningHoursStatus > poiOpeningHoursStatus,
                  int timeToDest, int distanceToDest,
                  int estimatedTimeOfArrivalHour, int estimatedTimeOfArrivalMinute,
                  ConsumptionUnit consumptionUnit,
                  unsigned int consumptionToDestAsInt, const ::std::string& consumptionToDestAsString,
                  int trafficDelayToDest, const ::std::string& trafficDelayToDestination,
                  const RouteInfoProperties& routeInfoProperties,
                  unsigned int relativeDirectionToDestination,
                  const ReachabilityInfo& reachabilityInfo,
                  const ValidValue<StateOfCharge>& stateOfChargeOnArrival,
                  const ValidValue<StateOfCharge>& stateOfChargeOnDeparture,
                  const StateOfCharge& stateOfChargeAfterChargingWithBatterySegments,
                  const ::std::string& chargingTimeAtDestination,
                  const ::std::string& additionalDurationOfStayAtNextDest,
                  int chargingTimeAtDestinationAsInt,
                  int additionalDurationOfStayAsInt,
                  const ::std::string& stateOfChargeAfterCharging,
                  int stateOfChargeAfterChargingAsInt,
                  ValidValue< ::navmiddleware::settings::RouteCriterion > routeCriterion)
         : m_timeToDestination(timeToDestination)
         , m_distanceToDestination(distanceToDestination)
         , m_distanceToDestinationNoUnit(distanceToDestinationNoUnit)
         , m_distanceUnit(distanceUnit)
         , m_estimatedTimeOfArrival(estimatedTimeOfArrival)
         , m_poiOpeningHoursStatus(poiOpeningHoursStatus)
         , m_timeToDest(timeToDest)
         , m_distanceToDest(distanceToDest)
         , m_estimatedTimeOfArrivalHour(estimatedTimeOfArrivalHour)
         , m_estimatedTimeOfArrivalMinute(estimatedTimeOfArrivalMinute)
         , m_consumptionUnit(consumptionUnit)
         , m_consumptionToDestAsInt(consumptionToDestAsInt)
         , m_consumptionToDestinationAsString(consumptionToDestAsString)
         , m_trafficDelayToDestAsInt(trafficDelayToDest)
         , m_trafficDelayToDestinationAsString(trafficDelayToDestination)
         , m_routeInfoProperties(routeInfoProperties)
         , m_relativeDirectionToDestination(relativeDirectionToDestination)
         , m_reachabilityInfo(reachabilityInfo)
         , m_stateOfChargeOnArrival(stateOfChargeOnArrival)
         , m_stateOfChargeOnDeparture(stateOfChargeOnDeparture)
         , m_stateOfChargeAfterChargingWithBatterySegments(stateOfChargeAfterChargingWithBatterySegments)
         , m_chargingTimeAtDestination(chargingTimeAtDestination)
         , m_additionalDurationOfStayAtNextDest(additionalDurationOfStayAtNextDest)
         , m_chargingTimeAtDestinationAsInt(chargingTimeAtDestinationAsInt)
         , m_additionalDurationOfStayAsInt(additionalDurationOfStayAsInt)
         , m_stateOfChargeAfterCharging(stateOfChargeAfterCharging)
         , m_stateOfChargeAfterChargingAsInt(stateOfChargeAfterChargingAsInt)
         , m_routeCriterion(routeCriterion)
      {}

      ArrivalInfo& operator = (const ArrivalInfo& rhs)
      {
         if (&rhs != this)
         {
            m_timeToDestination = rhs.m_timeToDestination;
            m_distanceToDestination = rhs.m_distanceToDestination;
            m_distanceToDestinationNoUnit = rhs.m_distanceToDestinationNoUnit;
            m_distanceUnit = rhs.m_distanceUnit;
            m_estimatedTimeOfArrival = rhs.m_estimatedTimeOfArrival;
            m_poiOpeningHoursStatus = rhs.m_poiOpeningHoursStatus;
            m_timeToDest = rhs.m_timeToDest;
            m_distanceToDest =rhs.m_distanceToDest;
            m_estimatedTimeOfArrivalHour = rhs.m_estimatedTimeOfArrivalHour;
            m_estimatedTimeOfArrivalMinute = rhs.m_estimatedTimeOfArrivalMinute;
            m_consumptionUnit = rhs.m_consumptionUnit;
            m_consumptionToDestAsInt = rhs.m_consumptionToDestAsInt;
            m_consumptionToDestinationAsString = rhs.m_consumptionToDestinationAsString;
            m_trafficDelayToDestAsInt = rhs.m_trafficDelayToDestAsInt;
            m_trafficDelayToDestinationAsString = rhs.m_trafficDelayToDestinationAsString;
            m_routeInfoProperties = rhs.m_routeInfoProperties;
            m_relativeDirectionToDestination = rhs.m_relativeDirectionToDestination;
            m_reachabilityInfo = rhs.m_reachabilityInfo;
            m_stateOfChargeOnArrival = rhs.m_stateOfChargeOnArrival;
            m_stateOfChargeOnDeparture = rhs.m_stateOfChargeOnDeparture;
            m_stateOfChargeAfterChargingWithBatterySegments = rhs.m_stateOfChargeAfterChargingWithBatterySegments;
            m_chargingTimeAtDestination = rhs.m_chargingTimeAtDestination;
            m_additionalDurationOfStayAtNextDest = rhs.m_additionalDurationOfStayAtNextDest;
            m_chargingTimeAtDestinationAsInt = rhs.m_chargingTimeAtDestinationAsInt;
            m_additionalDurationOfStayAsInt = rhs.m_additionalDurationOfStayAsInt;
            m_stateOfChargeAfterCharging = rhs.m_stateOfChargeAfterCharging;
            m_stateOfChargeAfterChargingAsInt = rhs.m_stateOfChargeAfterChargingAsInt;
            m_routeCriterion = rhs.m_routeCriterion;
         }
         return *this;
      }

      bool operator == (const ArrivalInfo& rhs) const
      {
         return (m_timeToDestination == rhs.m_timeToDestination)&&
                (m_distanceToDestination == rhs.m_distanceToDestination)&&
                (m_distanceToDestinationNoUnit == rhs.m_distanceToDestinationNoUnit)&&
                (m_distanceUnit == rhs.m_distanceUnit)&&
                (m_estimatedTimeOfArrival == rhs.m_estimatedTimeOfArrival)&&
                (m_poiOpeningHoursStatus == rhs.m_poiOpeningHoursStatus)&&
                (m_timeToDest == rhs.m_timeToDest)&&
                (m_distanceToDest == rhs.m_distanceToDest)&&
                (m_estimatedTimeOfArrivalHour ==rhs.m_estimatedTimeOfArrivalHour)&&
                (m_estimatedTimeOfArrivalMinute == rhs.m_estimatedTimeOfArrivalMinute)&&
                (m_consumptionUnit == rhs.m_consumptionUnit)&&
                (m_consumptionToDestAsInt == rhs.m_consumptionToDestAsInt)&&
                (m_consumptionToDestinationAsString == rhs.m_consumptionToDestinationAsString)&&
                (m_trafficDelayToDestAsInt == rhs.m_trafficDelayToDestAsInt)&&
                (m_trafficDelayToDestinationAsString == rhs.m_trafficDelayToDestinationAsString)&&
                (m_routeInfoProperties == rhs.m_routeInfoProperties)&&
                (m_relativeDirectionToDestination == rhs.m_relativeDirectionToDestination)&&
                (m_reachabilityInfo == rhs.m_reachabilityInfo)&&
                (m_stateOfChargeOnArrival == rhs.m_stateOfChargeOnArrival)&&
                (m_stateOfChargeOnDeparture == rhs.m_stateOfChargeOnDeparture)&&
                (m_stateOfChargeAfterChargingWithBatterySegments == rhs.m_stateOfChargeAfterChargingWithBatterySegments)&&
                (m_chargingTimeAtDestination == rhs.m_chargingTimeAtDestination)&&
                (m_additionalDurationOfStayAtNextDest == rhs.m_additionalDurationOfStayAtNextDest)&&
                (m_chargingTimeAtDestinationAsInt == rhs.m_chargingTimeAtDestinationAsInt)&&
                (m_additionalDurationOfStayAsInt == rhs.m_additionalDurationOfStayAsInt)&&
                (m_stateOfChargeAfterCharging == rhs.m_stateOfChargeAfterCharging)&&
                (m_stateOfChargeAfterChargingAsInt == rhs.m_stateOfChargeAfterChargingAsInt)&&
                (m_routeCriterion == rhs.m_routeCriterion);
      }

      bool operator != (const ArrivalInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("ArrivalInfo payload:\n");

         stream << ::std::endl << "Distance to destination =" << m_distanceToDest  << ", " << ::std::endl
                << "Distance Unit =" << m_distanceUnit  << ", " << ::std::endl
                << "Distance to destination =" << m_distanceToDestination << ::std::endl
                << "Distance to destination without Unit=" << m_distanceToDestinationNoUnit << ::std::endl
                << "Estimated Time of Arrival =" << m_estimatedTimeOfArrival  << ", " << ::std::endl
                << "POI reachability status =" << m_poiOpeningHoursStatus.toString() << ::std::endl
                << "Time to destination =" << m_timeToDest  << ", " << std::endl
                << "Time to destination as string =" << m_timeToDestination << ::std::endl
                << "Estimated time of arrival hour =" << m_estimatedTimeOfArrivalHour << ", " << ::std::endl
                << "Estimated time of arrival minute =" << m_estimatedTimeOfArrivalMinute << ::std::endl
                << "Consumption Unit =" << m_consumptionUnit << ::std::endl
                << "Consumption to Destination =" << m_consumptionToDestAsInt << ::std::endl
                << "Consumption to Destination as string =" << m_consumptionToDestinationAsString << ::std::endl
                << "TrafficDelay to Destination =" << m_trafficDelayToDestAsInt << ::std::endl
                << "TrafficDelay to Destination as string =" << m_trafficDelayToDestinationAsString << ::std::endl
                << "Road type Info for current destination:" << ::std::endl << m_routeInfoProperties.toString() << ::std::endl
                << "Direction to destination =" << m_relativeDirectionToDestination << ::std::endl
                << "Reachability Info : " << m_reachabilityInfo.toString() << ::std::endl
                << "State of charge on arrival at an epoi : " << (m_stateOfChargeOnArrival.isValid()? m_stateOfChargeOnArrival.getValue().toString() : "invalid") << ::std::endl
                << "State of charge on departure from an epoi : " << (m_stateOfChargeOnDeparture.isValid()? m_stateOfChargeOnDeparture.getValue().toString() : "invalid") << ::std::endl
                << "State of charge after charging with battery segments : " << m_stateOfChargeAfterChargingWithBatterySegments.toString() << ::std::endl
                << "Charging time to destination = "<< m_chargingTimeAtDestination << ::std::endl
                << "Additional duration of stay at next destination= " << m_additionalDurationOfStayAtNextDest << ::std::endl
                << "Charging time to destination as integer =" << m_chargingTimeAtDestinationAsInt << ::std::endl
                << "Additional duration as integer =" << m_additionalDurationOfStayAsInt << ::std::endl
                << "State of charge after charging as string(%) = " << m_stateOfChargeAfterCharging << ::std::endl
                << "State of charge after charging as integer(%) = " << m_stateOfChargeAfterChargingAsInt << ::std::endl
                << "Route Criterion type=" << m_routeCriterion.toString() << ::std::endl;
         return stream.str();
      }

      ::std::string m_timeToDestination;
      ::std::string m_distanceToDestination;
      ::std::string m_distanceToDestinationNoUnit;
      DistanceUnit m_distanceUnit;
      ::std::string m_estimatedTimeOfArrival;
      ValidValue< POIOpeningHoursStatus > m_poiOpeningHoursStatus;
      int m_timeToDest; /* [s] */
      int m_distanceToDest; /* [m] */
      int m_estimatedTimeOfArrivalHour;
      int m_estimatedTimeOfArrivalMinute;
      ConsumptionUnit m_consumptionUnit;
      unsigned int m_consumptionToDestAsInt;
      ::std::string m_consumptionToDestinationAsString;
      int m_trafficDelayToDestAsInt; /* [s] */
      ::std::string m_trafficDelayToDestinationAsString;
      // contains the road type information for the current destination/waypoint
      RouteInfoProperties m_routeInfoProperties;
      /* direction to destination relative to heading of car in degree, 0 = straight ahead, clockwise */
      unsigned int m_relativeDirectionToDestination;
      ReachabilityInfo m_reachabilityInfo;
      ValidValue<StateOfCharge> m_stateOfChargeOnArrival;
      ValidValue<StateOfCharge> m_stateOfChargeOnDeparture;
      StateOfCharge m_stateOfChargeAfterChargingWithBatterySegments; /*Deprecated. Not supported for Japan*/
      ::std::string m_chargingTimeAtDestination;
      ::std::string m_additionalDurationOfStayAtNextDest;
      int m_chargingTimeAtDestinationAsInt; /* [s] */
      int m_additionalDurationOfStayAsInt; /* [s] */
      ::std::string m_stateOfChargeAfterCharging; /* [%] deprecated */
      int m_stateOfChargeAfterChargingAsInt; /* [%] deprecated */
      ValidValue< ::navmiddleware::settings::RouteCriterion > m_routeCriterion;
   };

   const ::std::vector<ArrivalInfo>& getArrivalInfos() const
   {
      return m_arrivalInfos;
   }
   ::std::vector<ArrivalInfo>& getArrivalInfos()
   {
      return m_arrivalInfos;
   }

   unsigned int getCurrentDestinationIndex() const
   {
      return m_currentDestinationIndex;
   }

   void setCurrentDestinationIndex(unsigned int currentDestinationIndex)
   {
      m_currentDestinationIndex = currentDestinationIndex;
   }

   void setApproachingCurrentDest(bool isApproachingDest)
   {
      m_isApproachingCurrentDest = isApproachingDest;
   }

   bool isApproachingCurrentDest() const
   {
      return m_isApproachingCurrentDest;
   }

   const ValidValue< ::std::string>& getTollToDestination() const
   {
      return m_tollToDestination;
   }

   void setTollToDestination(const ValidValue< ::std::string>& tollToDestination)
   {
      m_tollToDestination = tollToDestination;
   }

   void setRouteCalcCompletedForAllDestinations(bool isRouteCalcCompletedForAllDestinations)
   {
      m_isRouteCalcCompletedForAllDestinations = isRouteCalcCompletedForAllDestinations;
   }

   bool isRouteCalcCompletedForAllDestinations() const
   {
      return m_isRouteCalcCompletedForAllDestinations;
   }

   const ValidValue<bool>& isRouteGuided() const
   {
      return m_isRouteGuided;
   }

   void setIsRouteGuided(const ValidValue<bool>& isRouteGuided)
   {
      m_isRouteGuided = isRouteGuided;
   }

   ArrivalInfos& operator = (const ArrivalInfos& rhs)
   {
      if (&rhs != this)
      {
         m_arrivalInfos = rhs.m_arrivalInfos;
         m_currentDestinationIndex = rhs.m_currentDestinationIndex;
         m_isApproachingCurrentDest = rhs.m_isApproachingCurrentDest;
         m_tollToDestination = rhs.m_tollToDestination;
         m_isRouteCalcCompletedForAllDestinations = rhs.m_isRouteCalcCompletedForAllDestinations;
         m_isRouteGuided = rhs.m_isRouteGuided;
      }

      return *this;
   }

   bool operator == (const ArrivalInfos& rhs) const
   {
      return (m_arrivalInfos == rhs.m_arrivalInfos)&&
             (m_currentDestinationIndex == rhs.m_currentDestinationIndex)&&
             (m_isApproachingCurrentDest == rhs.m_isApproachingCurrentDest)&&
             (m_tollToDestination == rhs.m_tollToDestination)&&
             (m_isRouteCalcCompletedForAllDestinations == rhs.m_isRouteCalcCompletedForAllDestinations) &&
             (m_isRouteGuided == rhs.m_isRouteGuided);
   }

   bool operator != (const ArrivalInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ArrivalInfos payload:\n");
      stream << ::std::endl
             << "Current destination index = " << m_currentDestinationIndex << ::std::endl
             << "Is approaching current destination = " << m_isApproachingCurrentDest << ::std::endl
             << "Toll to destination = " << m_tollToDestination.toString() << ::std::endl
             << "Is Route Calculation Completed for all destinations = " << m_isRouteCalcCompletedForAllDestinations << ::std::endl
             << "Is Route Guided = " << m_isRouteGuided.toString() << ::std::endl;

      for(unsigned int arrivalInfoListIndex = 0; arrivalInfoListIndex < getArrivalInfos().size(); arrivalInfoListIndex++)
      {
         stream << "Arrival Infos list index = " << arrivalInfoListIndex << ::std::endl
                << "Arrival info details  :" << getArrivalInfos().at(arrivalInfoListIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<ArrivalInfo> m_arrivalInfos;
   unsigned int m_currentDestinationIndex;
   // This flag will be set only when approaching the final destination and not the waypoints
   bool m_isApproachingCurrentDest;
   // [JP]: Tolls to destination, in locale specific currency format (without currency symbol)
   ValidValue< ::std::string> m_tollToDestination;
   bool m_isRouteCalcCompletedForAllDestinations;
   ValidValue<bool> m_isRouteGuided;
};

class ArrivalInfosAlternatives
{
public:

   ArrivalInfosAlternatives()
      : m_indexActiveAlternative(0),
        m_alternativeCalculationStatusNextDest(ROUTE_CALCULATION_STATUS__UNDEFINED),
        m_alternativeCalculationStatusFinalDest(ROUTE_CALCULATION_STATUS__UNDEFINED)
   {}

   enum EcoRating
   {
      ECO_RATING_1,
      ECO_RATING_2,
      ECO_RATING_3,
      ECO_RATING_4,
      ECO_RATING_5
   };

   class ArrivalInfoAlternative
   {
   public:
      ArrivalInfoAlternative()
         : m_consumptionToNextDestAsInt(0)
         , m_trafficDelayToNextDestAsInt(0)
         , m_distanceUnit(DISTANCEUNIT_MILES)
         , m_consumptionUnit(CONSUMPTION_UNIT__MICROLITER)
         , m_consumptionToFinalDestAsInt(0)
         , m_trafficDelayToFinalDestAsInt(0)
         , m_routeCriterion(::navmiddleware::settings::ROUTE_CRITERION_ECONOMIC)
         , m_indexOfReferencedAlternative(0)
         , m_stateOfChargeAfterChargingAtNextDestAsInt(0)
         , m_chargingTimeAtNextDestAsInt(0)
         , m_additionalDurationOfStayAsInt(0)
      {}

      ArrivalInfoAlternative(
         const ::std::string& timeToNextDestination,
         const ::std::string& distanceToNextDestination,
         const ::std::string& distanceToNextDestinationNoUnit,
         const ::std::string& etaToNextDestination,
         unsigned int consumptionToNextDestAsInt,
         const ::std::string& consumptionToNextDestAsString,
         int trafficDelayToNextDestAsInt,
         const ::std::string& trafficDelayToNextDestAsString,
         const ::std::string& timeToFinalDestination,
         const ::std::string& distanceToFinalDestination,
         const ::std::string& distanceToFinalDestinationNoUnit,
         const ::std::string& etaToFinalDestination,
         DistanceUnit distanceUnit,
         ConsumptionUnit consumptionUnit,
         unsigned int consumptionToFinalDestAsInt,
         const ::std::string& consumptionToFinalDestAsString,
         int trafficDelayToFinalDestAsInt, const ::std::string& trafficDelayToFinalDestAsString,
         ::navmiddleware::settings::RouteCriterion routeCriterion,
         unsigned int indexOfReferencedAlternative,
         ::navmiddleware::Color routeColor,
         const RouteInfoProperties& routeInfoPropertiesNextDestination,
         const RouteInfoProperties& routeInfoPropertiesFinalDestination,
         const ValidValue< POIOpeningHoursStatus >& poiOpeningHoursReachabilityStatusNextDest,
         const ValidValue< POIOpeningHoursStatus >& poiOpeningHoursReachabilityStatusFinalDest,
         const ReachabilityInfo& reachabilityInfoNextDest,
         const ReachabilityInfo& reachabilityInfoFinalDest,
         const ::std::string& chargingTimeAtNextDestination,
         const ::std::string& additionalDurationOfStayAtNextDest,
         const ValidValue< ::std::string>& tollToFinalDestination,
         const ::std::string& stateOfChargeAfterChargingAtNextDest,
         int stateOfChargeAfterChargingAtNextDestAsInt,
         const ValidValue< StateOfCharge >& stateOfChargeOnArrivalNextDest,
         const ValidValue< StateOfCharge >& stateOfChargeOnDepartureNextDest,
         const ValidValue< StateOfCharge >& stateOfChargeOnArrivalFinalDest,
         int chargingTimeAtNextDestAsInt,
         int additionalDurationOfStayAsInt)
         : m_timeToNextDestination(timeToNextDestination)
         , m_distanceToNextDestination(distanceToNextDestination)
         , m_distanceToNextDestinationNoUnit(distanceToNextDestinationNoUnit)
         , m_etaToNextDestination(etaToNextDestination)
         , m_consumptionToNextDestAsInt(consumptionToNextDestAsInt)
         , m_consumptionToNextDestAsString(consumptionToNextDestAsString)
         , m_trafficDelayToNextDestAsInt(trafficDelayToNextDestAsInt)
         , m_trafficDelayToNextDestAsString(trafficDelayToNextDestAsString)
         , m_timeToFinalDestination(timeToFinalDestination)
         , m_distanceToFinalDestination(distanceToFinalDestination)
         , m_distanceToFinalDestinationNoUnit(distanceToFinalDestinationNoUnit)
         , m_etaToFinalDestination(etaToFinalDestination)
         , m_distanceUnit(distanceUnit)
         , m_consumptionUnit(consumptionUnit)
         , m_consumptionToFinalDestAsInt(consumptionToFinalDestAsInt)
         , m_consumptionToFinalDestAsString(consumptionToFinalDestAsString)
         , m_trafficDelayToFinalDestAsInt(trafficDelayToFinalDestAsInt)
         , m_trafficDelayToFinalDestAsString(trafficDelayToFinalDestAsString)
         , m_routeCriterion(routeCriterion)
         , m_indexOfReferencedAlternative(indexOfReferencedAlternative)
         , m_routeColor(routeColor)
         , m_routeInfoPropertiesNextDestination(routeInfoPropertiesNextDestination)
         , m_routeInfoPropertiesFinalDestination(routeInfoPropertiesFinalDestination)
         , m_poiOpeningHoursReachabilityStatusNextDest(poiOpeningHoursReachabilityStatusNextDest)
         , m_poiOpeningHoursReachabilityStatusFinalDest(poiOpeningHoursReachabilityStatusFinalDest)
         , m_reachabilityInfoNextDest(reachabilityInfoNextDest)
         , m_reachabilityInfoFinalDest(reachabilityInfoFinalDest)
         , m_chargingTimeAtNextDestination(chargingTimeAtNextDestination)
         , m_additionalDurationOfStayAtNextDest(additionalDurationOfStayAtNextDest)
         , m_tollToFinalDestination(tollToFinalDestination)
         , m_stateOfChargeAfterChargingAtNextDest(stateOfChargeAfterChargingAtNextDest)
         , m_stateOfChargeAfterChargingAtNextDestAsInt(stateOfChargeAfterChargingAtNextDestAsInt)
         , m_stateOfChargeOnArrivalNextDest(stateOfChargeOnArrivalNextDest)
         , m_stateOfChargeOnDepartureNextDest(stateOfChargeOnDepartureNextDest)
         , m_stateOfChargeOnArrivalFinalDest(stateOfChargeOnArrivalFinalDest)
         , m_chargingTimeAtNextDestAsInt(chargingTimeAtNextDestAsInt)
         , m_additionalDurationOfStayAsInt(additionalDurationOfStayAsInt)
      {}

      ArrivalInfoAlternative& operator = (const ArrivalInfoAlternative& rhs)
      {
         if (&rhs != this)
         {
            m_timeToNextDestination = rhs.m_timeToNextDestination;
            m_distanceToNextDestination = rhs.m_distanceToNextDestination;
            m_distanceToNextDestinationNoUnit = rhs.m_distanceToNextDestinationNoUnit;
            m_etaToNextDestination = rhs.m_etaToNextDestination;
            m_consumptionToNextDestAsInt = rhs.m_consumptionToNextDestAsInt;
            m_consumptionToNextDestAsString = rhs.m_consumptionToNextDestAsString;
            m_trafficDelayToNextDestAsInt = rhs.m_trafficDelayToNextDestAsInt;
            m_trafficDelayToNextDestAsString = rhs.m_trafficDelayToNextDestAsString;
            m_timeToFinalDestination = rhs.m_timeToFinalDestination;
            m_distanceToFinalDestination = rhs.m_distanceToFinalDestination;
            m_distanceToFinalDestinationNoUnit = rhs.m_distanceToFinalDestinationNoUnit;
            m_etaToFinalDestination = rhs.m_etaToFinalDestination;
            m_distanceUnit = rhs.m_distanceUnit;
            m_consumptionUnit = rhs.m_consumptionUnit;
            m_consumptionToFinalDestAsInt = rhs.m_consumptionToFinalDestAsInt;
            m_consumptionToFinalDestAsString = rhs.m_consumptionToFinalDestAsString;
            m_trafficDelayToFinalDestAsInt = rhs.m_trafficDelayToFinalDestAsInt;
            m_trafficDelayToFinalDestAsString = rhs.m_trafficDelayToFinalDestAsString;
            m_routeColor = rhs.m_routeColor;
            m_routeCriterion = rhs.m_routeCriterion;
            m_indexOfReferencedAlternative = rhs.m_indexOfReferencedAlternative;
            m_routeInfoPropertiesNextDestination = rhs.m_routeInfoPropertiesNextDestination;
            m_routeInfoPropertiesFinalDestination = rhs.m_routeInfoPropertiesFinalDestination;
            m_poiOpeningHoursReachabilityStatusNextDest = rhs.m_poiOpeningHoursReachabilityStatusNextDest;
            m_poiOpeningHoursReachabilityStatusFinalDest = rhs.m_poiOpeningHoursReachabilityStatusFinalDest;
            m_reachabilityInfoNextDest = rhs.m_reachabilityInfoNextDest;
            m_reachabilityInfoFinalDest = rhs.m_reachabilityInfoFinalDest;
            m_chargingTimeAtNextDestination = rhs.m_chargingTimeAtNextDestination;
            m_additionalDurationOfStayAtNextDest = rhs.m_additionalDurationOfStayAtNextDest;
            m_tollToFinalDestination = rhs.m_tollToFinalDestination;
            m_stateOfChargeAfterChargingAtNextDest = rhs.m_stateOfChargeAfterChargingAtNextDest;
            m_stateOfChargeAfterChargingAtNextDestAsInt = rhs.m_stateOfChargeAfterChargingAtNextDestAsInt;
            m_stateOfChargeOnArrivalNextDest = rhs.m_stateOfChargeOnArrivalNextDest;
            m_stateOfChargeOnDepartureNextDest = rhs.m_stateOfChargeOnDepartureNextDest;
            m_stateOfChargeOnArrivalFinalDest = rhs.m_stateOfChargeOnArrivalFinalDest;
            m_chargingTimeAtNextDestAsInt = rhs.m_chargingTimeAtNextDestAsInt;
            m_additionalDurationOfStayAsInt = rhs.m_additionalDurationOfStayAsInt;
         }
         return *this;
      }

      bool operator == (const ArrivalInfoAlternative& rhs) const
      {
         return (m_timeToNextDestination == rhs.m_timeToNextDestination)&&
                (m_distanceToNextDestination == rhs.m_distanceToNextDestination)&&
                (m_distanceToNextDestinationNoUnit == rhs.m_distanceToNextDestinationNoUnit)&&
                (m_etaToNextDestination == rhs.m_etaToNextDestination) &&
                (m_consumptionToNextDestAsInt == rhs.m_consumptionToNextDestAsInt) &&
                (m_consumptionToNextDestAsString == rhs.m_consumptionToNextDestAsString) &&
                (m_trafficDelayToNextDestAsInt == rhs.m_trafficDelayToNextDestAsInt) &&
                (m_trafficDelayToNextDestAsString == rhs.m_trafficDelayToNextDestAsString) &&
                (m_timeToFinalDestination == rhs.m_timeToFinalDestination)&&
                (m_distanceToFinalDestination == rhs.m_distanceToFinalDestination)&&
                (m_distanceToFinalDestinationNoUnit == rhs.m_distanceToFinalDestinationNoUnit)&&
                (m_etaToFinalDestination == rhs.m_etaToFinalDestination)&&
                (m_distanceUnit == rhs.m_distanceUnit)&&
                (m_consumptionUnit == rhs.m_consumptionUnit)&&
                (m_consumptionToFinalDestAsInt == rhs.m_consumptionToFinalDestAsInt) &&
                (m_consumptionToFinalDestAsString == rhs.m_consumptionToFinalDestAsString) &&
                (m_trafficDelayToFinalDestAsInt == rhs.m_trafficDelayToFinalDestAsInt) &&
                (m_trafficDelayToFinalDestAsString == rhs.m_trafficDelayToFinalDestAsString) &&
                (m_routeColor == rhs.m_routeColor)&&
                (m_routeCriterion == rhs.m_routeCriterion)&&
                (m_indexOfReferencedAlternative == rhs.m_indexOfReferencedAlternative)&&
                (m_routeInfoPropertiesNextDestination == rhs.m_routeInfoPropertiesNextDestination)&&
                (m_routeInfoPropertiesFinalDestination == rhs.m_routeInfoPropertiesFinalDestination)&&
                (m_poiOpeningHoursReachabilityStatusNextDest == rhs.m_poiOpeningHoursReachabilityStatusNextDest)&&
                (m_poiOpeningHoursReachabilityStatusFinalDest == rhs.m_poiOpeningHoursReachabilityStatusFinalDest)&&
                (m_reachabilityInfoNextDest == rhs.m_reachabilityInfoNextDest)&&
                (m_reachabilityInfoFinalDest == rhs.m_reachabilityInfoFinalDest)  &&
                (m_chargingTimeAtNextDestination == rhs.m_chargingTimeAtNextDestination) &&
                (m_additionalDurationOfStayAtNextDest == rhs.m_additionalDurationOfStayAtNextDest) &&
                (m_tollToFinalDestination == rhs.m_tollToFinalDestination)&&
                (m_stateOfChargeAfterChargingAtNextDest == rhs.m_stateOfChargeAfterChargingAtNextDest)&&
                (m_stateOfChargeAfterChargingAtNextDestAsInt == rhs.m_stateOfChargeAfterChargingAtNextDestAsInt)&&
                (m_stateOfChargeOnArrivalNextDest == rhs.m_stateOfChargeOnArrivalNextDest)&&
                (m_stateOfChargeOnDepartureNextDest == rhs.m_stateOfChargeOnDepartureNextDest)&&
                (m_stateOfChargeOnArrivalFinalDest == rhs.m_stateOfChargeOnArrivalFinalDest)&&
                (m_chargingTimeAtNextDestAsInt == rhs.m_chargingTimeAtNextDestAsInt)&&
                (m_additionalDurationOfStayAsInt == rhs.m_additionalDurationOfStayAsInt);
      }

      bool operator!=(const ArrivalInfoAlternative& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("ArrivalInfoAlternative payload:\n");

         stream << ::std::endl
                << "Distance to next destination =" << m_distanceToNextDestination << ", "
                << "Distance to final destination =" << m_distanceToFinalDestination << ::std::endl
                << "Distance to next destination without unit =" << m_distanceToNextDestinationNoUnit << ", "
                << "Distance to final destination without unit =" << m_distanceToFinalDestinationNoUnit << ::std::endl
                << "Distance unit =" << m_distanceUnit << ::std::endl
                << "ETA to next destination =" << m_etaToNextDestination << ::std::endl
                << "Consumption to next destination =" << m_consumptionToNextDestAsInt <<  ", "
                << "Consumption to next destination as string =" << m_consumptionToNextDestAsString << ::std::endl
                << "Traffic Delay to next destination =" << m_trafficDelayToNextDestAsInt << ", "
                << "Traffic Delay  to next destination as string =" << m_trafficDelayToNextDestAsString << ::std::endl
                << "ETA to final destination =" << m_etaToFinalDestination << ::std::endl
                << "Time to next destination =" << m_timeToNextDestination << ", "
                << "Time to final destination =" << m_timeToFinalDestination << ::std::endl
                << "Consumption unit =" << m_consumptionUnit << ::std::endl
                << "Consumption to next destination =" << m_consumptionToFinalDestAsInt << ", "
                << "Consumption to next destination as string =" << m_consumptionToFinalDestAsString << ::std::endl
                << "Traffic Delay to next destination =" << m_trafficDelayToFinalDestAsInt << ", "
                << "Traffic Delay  to next destination as string =" << m_trafficDelayToFinalDestAsString << ::std::endl
                << "Route Criterion type=" << m_routeCriterion << ::std::endl
                << "Index of Referenced Alternative=" << m_indexOfReferencedAlternative << ::std::endl   //<-- INF4CV
                << "Route color(R,G,B,Alpha): " << m_routeColor.toString() << ::std::endl
                << "Road type Info for next destination: " << ::std::endl << m_routeInfoPropertiesNextDestination.toString() << ::std::endl
                << "Road type Info for Final destination: " << ::std::endl << m_routeInfoPropertiesFinalDestination.toString() << ::std::endl
                << "Poi opening hours status next destination: " << ::std::endl << m_poiOpeningHoursReachabilityStatusNextDest.toString() << ::std::endl
                << "Poi opening hours status final destination: " << ::std::endl << m_poiOpeningHoursReachabilityStatusFinalDest.toString() << ::std::endl
                << "Reachability Info for next destination: " << ::std::endl << m_reachabilityInfoNextDest.toString() << ::std::endl
                << "Reachability Info for final destination: " << ::std::endl << m_reachabilityInfoFinalDest.toString() << ::std::endl
                << "Charging time at Next Destination = " << m_chargingTimeAtNextDestination << ::std::endl
                << "Additional duration of stay at next destination= " << m_additionalDurationOfStayAtNextDest << ::std::endl
                << "Toll to final destination = " << m_tollToFinalDestination.toString() << ::std::endl
                << "State of charge after charging at next destination as string = " << m_stateOfChargeAfterChargingAtNextDest << ::std::endl
                << "State of charge after charging at next destination as integer = " << m_stateOfChargeAfterChargingAtNextDestAsInt << ::std::endl
                << "State of charge on arrival at next destination = " << (m_stateOfChargeOnArrivalNextDest.isValid()? m_stateOfChargeOnArrivalNextDest.getValue().toString() : "invalid") << ::std::endl
                << "State of charge on departure from next destination = " << (m_stateOfChargeOnDepartureNextDest.isValid()? m_stateOfChargeOnDepartureNextDest.getValue().toString() : "invalid") << ::std::endl
                << "State of charge on arrival at final destination = " << (m_stateOfChargeOnArrivalFinalDest.isValid()? m_stateOfChargeOnArrivalFinalDest.getValue().toString() : "invalid") << ::std::endl
                << "Charging time at next destination as integer = " << m_chargingTimeAtNextDestAsInt << ::std::endl
                << "Additional duration of stay as integer (%) = " << m_additionalDurationOfStayAsInt << ::std::endl;

         return stream.str();
      }

      ::std::string m_timeToNextDestination;
      ::std::string m_distanceToNextDestination;
      ::std::string m_distanceToNextDestinationNoUnit;
      ::std::string m_etaToNextDestination;
      unsigned int m_consumptionToNextDestAsInt;
      ::std::string m_consumptionToNextDestAsString;
      int m_trafficDelayToNextDestAsInt;
      ::std::string m_trafficDelayToNextDestAsString;
      ::std::string m_timeToFinalDestination;
      ::std::string m_distanceToFinalDestination;
      ::std::string m_distanceToFinalDestinationNoUnit;
      ::std::string m_etaToFinalDestination;
      DistanceUnit m_distanceUnit;
      ConsumptionUnit m_consumptionUnit;
      unsigned int m_consumptionToFinalDestAsInt;
      ::std::string m_consumptionToFinalDestAsString;
      int m_trafficDelayToFinalDestAsInt;
      ::std::string m_trafficDelayToFinalDestAsString;
      ::navmiddleware::settings::RouteCriterion m_routeCriterion;
      /* index of referenced route alternative */
      unsigned int m_indexOfReferencedAlternative;
      Color m_routeColor;
      RouteInfoProperties m_routeInfoPropertiesNextDestination;
      RouteInfoProperties m_routeInfoPropertiesFinalDestination;
      ValidValue< POIOpeningHoursStatus > m_poiOpeningHoursReachabilityStatusNextDest;
      ValidValue< POIOpeningHoursStatus > m_poiOpeningHoursReachabilityStatusFinalDest;
      ReachabilityInfo m_reachabilityInfoNextDest;
      ReachabilityInfo m_reachabilityInfoFinalDest;
      ::std::string m_chargingTimeAtNextDestination;
      ::std::string m_additionalDurationOfStayAtNextDest;
      // [JP]: Tolls to destination, in locale specific currency format (without currency symbol)
      ValidValue< ::std::string> m_tollToFinalDestination;
      // If the next way-point is an ePOI, based on the duration of charging time the following 2 variables
      // shall calculate and store the SOC after charging at the next destination.
      ::std::string m_stateOfChargeAfterChargingAtNextDest; /* [%] deprecated */
      int m_stateOfChargeAfterChargingAtNextDestAsInt; /* [%] deprecated */
      ValidValue<StateOfCharge> m_stateOfChargeOnArrivalNextDest;
      ValidValue<StateOfCharge> m_stateOfChargeOnDepartureNextDest;
      ValidValue< StateOfCharge > m_stateOfChargeOnArrivalFinalDest;
      int m_chargingTimeAtNextDestAsInt; /* [s] */
      int m_additionalDurationOfStayAsInt; /* [s] */
   };

   const ::std::vector<ArrivalInfoAlternative>& getArrivalInfosAlternatives() const
   {
      return m_arrivalInfosAlternatives;
   }
   ::std::vector<ArrivalInfoAlternative>& getArrivalInfosAlternatives()
   {
      return m_arrivalInfosAlternatives;
   }
   unsigned int getIndexActiveAlternative() const
   {
      return m_indexActiveAlternative;
   }
   void setIndexActiveAlternative(unsigned int indexActiveAlternative)
   {
      m_indexActiveAlternative = indexActiveAlternative;
   }

   RouteCalculationStatus getAlternativeCalculationStatusNextDest() const
   {
      return m_alternativeCalculationStatusNextDest;
   }

   void setAlternativeCalculationStatusNextDest(RouteCalculationStatus alternativeCalculationStatusNextDest)
   {
      m_alternativeCalculationStatusNextDest = alternativeCalculationStatusNextDest;
   }

   RouteCalculationStatus getAlternativeCalculationStatusFinalDest() const
   {
      return m_alternativeCalculationStatusFinalDest;
   }

   void setAlternativeCalculationStatusFinalDest(RouteCalculationStatus alternativeCalculationStatusFinalDest)
   {
      m_alternativeCalculationStatusFinalDest = alternativeCalculationStatusFinalDest;
   }

   ArrivalInfosAlternatives& operator = (const ArrivalInfosAlternatives& rhs)
   {
      if (&rhs != this)
      {
         m_arrivalInfosAlternatives = rhs.m_arrivalInfosAlternatives;
         m_indexActiveAlternative = rhs.m_indexActiveAlternative;
         m_alternativeCalculationStatusNextDest = rhs.m_alternativeCalculationStatusNextDest;
         m_alternativeCalculationStatusFinalDest = rhs.m_alternativeCalculationStatusFinalDest;
         m_ecoRatingToNextDestination = rhs.m_ecoRatingToNextDestination;
         m_ecoRatingToFinalDestination = rhs.m_ecoRatingToFinalDestination;
      }

      return *this;
   }

   bool operator == (const ArrivalInfosAlternatives& rhs) const
   {
      return (m_arrivalInfosAlternatives == rhs.m_arrivalInfosAlternatives)&&
             (m_indexActiveAlternative == rhs.m_indexActiveAlternative)&&
             (m_alternativeCalculationStatusNextDest == rhs.m_alternativeCalculationStatusNextDest) &&
             (m_alternativeCalculationStatusFinalDest == rhs.m_alternativeCalculationStatusFinalDest) &&
             (m_ecoRatingToNextDestination == rhs.m_ecoRatingToNextDestination)&&
             (m_ecoRatingToFinalDestination == rhs.m_ecoRatingToFinalDestination);
   }

   bool operator != (const ArrivalInfosAlternatives& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ArrivalInfosAlternatives payload:\n");
      stream << ::std::endl;
      stream << "Index active alternative = " << m_indexActiveAlternative << ::std::endl;
      stream << "Alternative calculation status Next destination = " << ::navmiddleware::toString(m_alternativeCalculationStatusNextDest) << ::std::endl;
      stream << "Alternative calculation status Final destination = " << ::navmiddleware::toString(m_alternativeCalculationStatusFinalDest) << ::std::endl;
      stream << "Eco rating next destination = " << m_ecoRatingToNextDestination.toString() << ::std::endl;
      stream << "Eco rating final destination = " << m_ecoRatingToFinalDestination.toString() << ::std::endl;

      for(unsigned int arrivalInfosAlternativesListIndex = 0; arrivalInfosAlternativesListIndex < getArrivalInfosAlternatives().size(); arrivalInfosAlternativesListIndex++)
      {
         stream << "Arrival Infos alternatives list index = " << arrivalInfosAlternativesListIndex << ::std::endl
                << "Arrival info alternavive =" << getArrivalInfosAlternatives().at(arrivalInfosAlternativesListIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

   void setEcoRatingToNextDestination(const ValidValue<EcoRating>& ecoRating)
   {
      m_ecoRatingToNextDestination = ecoRating;
   }

   const ValidValue<EcoRating>& getEcoRatingToNextDestination() const
   {
      return m_ecoRatingToNextDestination;
   }

   ValidValue<EcoRating>& getEcoRatingToNextDestination()
   {
      return m_ecoRatingToNextDestination;
   }

   void setEcoRatingToFinalDestination(const ValidValue<EcoRating>& ecoRating)
   {
      m_ecoRatingToFinalDestination = ecoRating;
   }

   const ValidValue<EcoRating>& getEcoRatingToFinalDestination() const
   {
      return m_ecoRatingToFinalDestination;
   }

   ValidValue<EcoRating>& getEcoRatingToFinalDestination()
   {
      return m_ecoRatingToFinalDestination;
   }

private:
   ::std::vector<ArrivalInfoAlternative> m_arrivalInfosAlternatives;
   unsigned int m_indexActiveAlternative;
   RouteCalculationStatus m_alternativeCalculationStatusNextDest;
   RouteCalculationStatus m_alternativeCalculationStatusFinalDest;
   /* Eco rating level is only available when arrival infos with criteria ECONOMIC and FAST are present */
   ValidValue<EcoRating> m_ecoRatingToNextDestination;
   ValidValue<EcoRating> m_ecoRatingToFinalDestination;
};

class WaypointOptimizationDetails
{
public:
   WaypointOptimizationDetails()
      : m_estimatedDurationOfStay(0),
        m_finalDestination(false)
   {
   }

   const ValidValue<int64_t>& getEarliestArrivalTime() const
   {
      return m_earliestArrivalTime;
   }

   void setEarliestArrivalTime(const ValidValue<int64_t>& earliestArrivalTime)
   {
      m_earliestArrivalTime = earliestArrivalTime;
   }

   const ValidValue<int64_t>& getLatestArrivalTime() const
   {
      return m_latestArrivalTime;
   }

   void setLatestArrivalTime(const ValidValue<int64_t>& latestArrivalTime)
   {
      m_latestArrivalTime = latestArrivalTime;
   }

   int32_t getEstimatedDurationOfStay() const
   {
      return m_estimatedDurationOfStay;
   }

   void setEstimatedDurationOfStay(int32_t estimatedDurationOfStay)
   {
      m_estimatedDurationOfStay = estimatedDurationOfStay;
   }

   bool isFinalDestination() const
   {
      return m_finalDestination;
   }

   void setFinalDestination(bool finalDestination)
   {
      m_finalDestination = finalDestination;
   }

   WaypointOptimizationDetails& operator = (const WaypointOptimizationDetails& rhs)
   {
      if (&rhs != this)
      {
         m_earliestArrivalTime = rhs.m_earliestArrivalTime;
         m_latestArrivalTime = rhs.m_latestArrivalTime;
         m_estimatedDurationOfStay = rhs.m_estimatedDurationOfStay;
         m_finalDestination = rhs.m_finalDestination;
      }
      return *this;
   }

   bool operator == (const WaypointOptimizationDetails& wos) const
   {
      return (m_earliestArrivalTime.getValue() == wos.m_earliestArrivalTime.getValue())&&
             (m_latestArrivalTime.getValue() == wos.m_latestArrivalTime.getValue())&&
             (m_estimatedDurationOfStay == wos.m_estimatedDurationOfStay)&&
             (m_finalDestination == wos.m_finalDestination);
   }

   bool operator != (const WaypointOptimizationDetails& wos) const
   {
      return !(*this == wos);
   }

private:
   ValidValue<int64_t> m_earliestArrivalTime;
   ValidValue<int64_t> m_latestArrivalTime;
   int32_t m_estimatedDurationOfStay;
   bool m_finalDestination;
};

class StartPositionInfo
{
   /**
    * Class contains details of "Current Position" on EditRoute screen which is other than the CVP
    * The field m_locationString: includes a generated name for the location
    * The field m_startPositionCoordinate: contains the geocoordinate of the start position
    */
public:
   StartPositionInfo()
   {}

   StartPositionInfo(const ::std::string& locationString,
                     const GeoCoordinateDegree& startPosition)
      : m_locationString(locationString)
      , m_startPositionCoordinate(startPosition)
   {
   }

   StartPositionInfo& operator = (const StartPositionInfo& rhs)
   {
      if (&rhs != this)
      {
         m_locationString = rhs.m_locationString;
         m_startPositionCoordinate = rhs.m_startPositionCoordinate;
      }
      return *this;
   }

   bool operator == (const StartPositionInfo& rhs) const
   {
      return (m_locationString == rhs.m_locationString) &&
             (m_startPositionCoordinate == rhs.m_startPositionCoordinate);
   }

   bool operator != (const StartPositionInfo& rhs) const
   {
      return !(*this == rhs);
   }

   const ::std::string& getLocationDetails() const
   {
      return m_locationString;
   }

   void setLocationDetails(const ::std::string& locationString)
   {
      m_locationString = locationString;
   }

   const GeoCoordinateDegree& getStartPositionCoordinate() const
   {
      return m_startPositionCoordinate;
   }

   void setStartPositionCoordinate(const GeoCoordinateDegree& startPositionCoordinate)
   {
      m_startPositionCoordinate = startPositionCoordinate;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("StartPositionInfo payload:\n");

      stream << ::std::endl
             << "Location detail = " << m_locationString << ::std::endl
             << "Coordinate detail = " << m_startPositionCoordinate.toString() << ::std::endl;

      return stream.str();
   }

private:
   ::std::string m_locationString;
   /* start position set by the user which is independent of CVP */
   GeoCoordinateDegree m_startPositionCoordinate;
};

enum WaypointType
{
   /* Normal waypoint */
   WAYPOINT_TYPE__NORMAL = 0,
   /* [JP] Waypoint from move motorway entry */
   WAYPOINT_TYPE__IC_IN,
   /* [JP] Waypoint from move motorway exit */
   WAYPOINT_TYPE__IC_OUT,
   /* Charging station as waypoint */
   WAYPOINT_TYPE_CHARGING_STATION
};

inline ::std::string toString(const WaypointType& waypointType)
{
   switch ( waypointType )
   {
   case WAYPOINT_TYPE__NORMAL:
      return "WAYPOINT_TYPE__NORMAL";
   case WAYPOINT_TYPE__IC_IN:
      return "WAYPOINT_TYPE__IC_IN";
   case WAYPOINT_TYPE__IC_OUT:
      return "WAYPOINT_TYPE__IC_OUT";
   case WAYPOINT_TYPE_CHARGING_STATION:
      return "WAYPOINT_TYPE_CHARGING_STATION";
   default:
      {
         ::std::stringstream stream;
         stream << "WAYPOINT_TYPE__<" << waypointType << ">";
         return stream.str();
      }
   }
}

enum ChargingStationAdditionType
{
   CHARGING_STATION_ADDITION_TYPE__AUTOMATIC = 0,
   CHARGING_STATION_ADDITION_TYPE__MANUAL
};

inline ::std::string toString(const ChargingStationAdditionType& chargingStationAdditionType)
{
   switch ( chargingStationAdditionType )
   {
   case CHARGING_STATION_ADDITION_TYPE__AUTOMATIC:
      return "CHARGING_STATION_ADDITION_TYPE__AUTOMATIC";
   case CHARGING_STATION_ADDITION_TYPE__MANUAL:
      return "CHARGING_STATION_ADDITION_TYPE__MANUAL";
   default:
      {
         ::std::stringstream stream;
         stream << "CHARGING_STATION_ADDITION_TYPE__<" << chargingStationAdditionType << ">";
         return stream.str();
      }
   }
}

enum WaypointSource
{
   WAYPOINT_SOURCE__INTERNAL = 0,
   WAYPOINT_SOURCE__EXTERNAL
};

inline ::std::string toString(const WaypointSource& waypointSource)
{
   switch ( waypointSource )
   {
   case WAYPOINT_SOURCE__INTERNAL:
      return "WAYPOINT_SOURCE__INTERNAL";
   case WAYPOINT_SOURCE__EXTERNAL:
      return "WAYPOINT_SOURCE__EXTERNAL";
   default:
      {
         ::std::stringstream stream;
         stream << "WAYPOINT_SOURCE__<" << waypointSource << ">";
         return stream.str();
      }
   }
}

class WaypointInfos
{
public:

   enum OptimizationStatus
   {
      /* Reserved value */
      OPTIMIZATION_STATUS__UNKNOWN = 0,
      /* No optimization was requested */
      OPTIMIZATION_STATUS__NO_OPTIMIZATION_REQUESTED,
      /* Waypoint optimization is currently in progress */
      OPTIMIZATION_STATUS__IN_PROGRESS,
      /* Route successfully optimized. The destination order has changed */
      OPTIMIZATION_STATUS__COMPLETED_WITH_CHANGED_DEST_ORDER,
      /* Route successfully optimized. The destination order is unchanged */
      OPTIMIZATION_STATUS__COMPLETED_WITH_UNCHANGED_DEST_ORDER,
      /* Optimization failed because the defined time windows can't be regarded in an optimized route */
      OPTIMIZATION_STATUS__FAILED_TIME_WINDOWS_NOT_APPLICABLE,
      /* Optimization failed but the reason is unclear */
      OPTIMIZATION_STATUS__FAILED_NO_OPTIMIZATION_POSSIBLE
   };

   WaypointInfos()
      : m_optimizationStatus(OPTIMIZATION_STATUS__NO_OPTIMIZATION_REQUESTED)
      , m_currentDestinationIndex(0)
      , m_skipCurrentWaypointOnDeviations(false)
   {}

   class WaypointInfo
   {
   public:
      WaypointInfo()
         : m_routeCriterion(::navmiddleware::settings::ROUTE_CRITERION_ECONOMIC)
         , m_waypointReachedInGuidedRoute(false)
         , m_waypointType(WAYPOINT_TYPE__NORMAL)
         , m_waypointSource (WAYPOINT_SOURCE__INTERNAL)
      {}

      WaypointInfo(const ::std::string& locationString,
                   ::navmiddleware::settings::RouteCriterion routeCriterion,
                   WaypointOptimizationDetails waypointOptimizationDetails,
                   const GeoCoordinateDegree& coordinate,
                   bool waypointReachedInGuidedRoute,
                   WaypointType waypointType,
                   const ValidValue<ChargingStationAdditionType>& chargingStationAdditionType,
                   WaypointSource waypointSource,
                   const LocationAttributeInfos& locationAttributeInfos)
         : m_locationString(locationString)
         , m_routeCriterion(routeCriterion)
         , m_waypointOptimizationDetails(waypointOptimizationDetails)
         , m_coordinate(coordinate)
         , m_waypointReachedInGuidedRoute(waypointReachedInGuidedRoute)
         , m_waypointType(waypointType)
         , m_chargingStationAdditionType(chargingStationAdditionType)
         , m_waypointSource(waypointSource)
         , m_locationAttributeInfos(locationAttributeInfos)
      {}

      WaypointInfo& operator = (const WaypointInfo& rhs)
      {
         if (&rhs != this)
         {
            m_locationString = rhs.m_locationString;
            m_routeCriterion = rhs.m_routeCriterion;
            m_waypointOptimizationDetails = rhs.m_waypointOptimizationDetails;
            m_coordinate = rhs.m_coordinate;
            m_waypointReachedInGuidedRoute = rhs.m_waypointReachedInGuidedRoute;
            m_waypointType = rhs.m_waypointType;
            m_chargingStationAdditionType = rhs.m_chargingStationAdditionType;
            m_waypointSource = rhs.m_waypointSource;
            m_locationAttributeInfos = rhs.m_locationAttributeInfos;
         }
         return *this;
      }

      bool operator == (const WaypointInfo& rhs) const
      {
         return (m_locationString == rhs.m_locationString)&&
                (m_routeCriterion == rhs.m_routeCriterion)&&
                (m_waypointOptimizationDetails == rhs.m_waypointOptimizationDetails)&&
                (m_coordinate == rhs.m_coordinate)&&
                (m_waypointReachedInGuidedRoute == rhs.m_waypointReachedInGuidedRoute)&&
                (m_waypointType == rhs.m_waypointType)&&
                (m_chargingStationAdditionType == rhs.m_chargingStationAdditionType)&&
                (m_waypointSource == rhs.m_waypointSource)&&
                (m_locationAttributeInfos == rhs.m_locationAttributeInfos);
      }

      bool operator != (const WaypointInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("WaypointInfo payload:\n");

         stream << ::std::endl
                << "Location detail =" << m_locationString << ::std::endl
                << "Route Criterion type =" << m_routeCriterion << ::std::endl
                << "WaypointOptimizationSettings: " << ::std::endl
                << "   m_earliestArrivalTime = " << m_waypointOptimizationDetails.getEarliestArrivalTime().getValue() << ::std::endl
                << "   m_latestArrivalTime = " << m_waypointOptimizationDetails.getLatestArrivalTime().getValue() << ::std::endl
                << "   m_estimatedDurationOfStay = " << m_waypointOptimizationDetails.getEstimatedDurationOfStay() << ::std::endl
                << "   m_finalDestination = " << m_waypointOptimizationDetails.isFinalDestination() << ::std::endl
                << "coordinate info of waypoint=" << m_coordinate.toString() << ::std::endl
                << "Is waypoint in Guided Route reached =" << m_waypointReachedInGuidedRoute <<::std::endl
                << "Waypoint Type =" << m_waypointType << ::std::endl
                << "Charging Station AdditionType =" << (m_chargingStationAdditionType.isValid() ? ::navmiddleware::toString(m_chargingStationAdditionType.getValue()) : "InValid") << ::std::endl
                << "Waypoint Source =" << ::navmiddleware::toString(m_waypointSource) << ::std::endl
                << "Location Attribute Infos =" << m_locationAttributeInfos.toString() << ::std::endl;

         return stream.str();
      }

      ::std::string m_locationString;
      ::navmiddleware::settings::RouteCriterion m_routeCriterion;
      WaypointOptimizationDetails m_waypointOptimizationDetails;
      GeoCoordinateDegree m_coordinate;
      bool m_waypointReachedInGuidedRoute;
      WaypointType m_waypointType;
      ValidValue <ChargingStationAdditionType> m_chargingStationAdditionType;
      WaypointSource m_waypointSource;
      LocationAttributeInfos m_locationAttributeInfos;
   };

   const ::std::vector< WaypointInfo >& getWaypointInfos() const
   {
      return m_waypointInfos;
   }

   ::std::vector< WaypointInfo >& getWaypointInfos()
   {
      return m_waypointInfos;
   }

   unsigned int getCurrentDestinationIndex() const
   {
      return m_currentDestinationIndex;
   }

   void setCurrentDestinationIndex(unsigned int currentDestinationIndex)
   {
      m_currentDestinationIndex = currentDestinationIndex;
   }

   const ValidValue<unsigned int>& getOptimizeFromSegment() const
   {
      return m_optimizeFromSegment;
   }

   const ValidValue<unsigned int>& getOptimizeToSegment() const
   {
      return m_optimizeToSegment;
   }

   void setOptimizationRange(const ValidValue<unsigned int>& optimizeFromSegment, const ValidValue<unsigned int>& optimizeToSegment)
   {
      m_optimizeFromSegment = optimizeFromSegment;
      m_optimizeToSegment   = optimizeToSegment;
   }

   OptimizationStatus getOptimizationStatus() const
   {
      return m_optimizationStatus;
   }

   void setOptimizationStatus(OptimizationStatus optimizationStatus)
   {
      m_optimizationStatus = optimizationStatus;
   }

   bool isSkipCurrentWaypoint() const
   {
      return m_skipCurrentWaypointOnDeviations;
   }

   const ValidValue< StartPositionInfo >& getStartPositionInfo() const
   {
      return m_startPositionInfo;
   }

   void setStartPositionInfo(const ValidValue< StartPositionInfo >& startPositionInfo)
   {
      m_startPositionInfo = startPositionInfo;
   }

   void setSkipCurrentWaypoint(bool skipCurrentWaypointOnDeviations)
   {
      m_skipCurrentWaypointOnDeviations = skipCurrentWaypointOnDeviations;
   }

   WaypointInfos& operator = (const WaypointInfos& rhs)
   {
      m_waypointInfos = rhs.m_waypointInfos;
      m_optimizeFromSegment = rhs.m_optimizeFromSegment;
      m_optimizeToSegment = rhs.m_optimizeToSegment;
      m_optimizationStatus = rhs.m_optimizationStatus;
      m_currentDestinationIndex = rhs.m_currentDestinationIndex;
      m_skipCurrentWaypointOnDeviations = rhs.m_skipCurrentWaypointOnDeviations;
      m_startPositionInfo = rhs.m_startPositionInfo;

      return *this;
   }

   bool operator == (const WaypointInfos& rhs) const
   {
      return  (m_waypointInfos == rhs.m_waypointInfos)&&
              (m_optimizeFromSegment == rhs.m_optimizeFromSegment)&&
              (m_optimizeToSegment == rhs.m_optimizeToSegment)&&
              (m_optimizationStatus == rhs.m_optimizationStatus) &&
              (m_currentDestinationIndex == rhs.m_currentDestinationIndex)&&
              (m_skipCurrentWaypointOnDeviations == rhs.m_skipCurrentWaypointOnDeviations)&&
              (m_startPositionInfo == rhs.m_startPositionInfo);
   }

   bool operator != (const WaypointInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("WaypointInfos payload:\n");
      stream << ::std::endl;
      stream << "OptimizeFromSegment = " << m_optimizeFromSegment.getValueOr(0) << ::std::endl;
      stream << "OptimizeToSegment = " << m_optimizeToSegment.getValueOr(0) << ::std::endl;
      stream << "OptimizationStatus = " << m_optimizationStatus << ::std::endl;
      stream << "CurrentDestinationIndex = " << m_currentDestinationIndex << ::std::endl;
      stream << "SkipCurrentWaypointOnDeviations = " << m_skipCurrentWaypointOnDeviations << ::std::endl;
      stream << "Start Position Info = " << (m_startPositionInfo.isValid()? m_startPositionInfo.getValue().toString() : "invalid") << ::std::endl;

      for(unsigned int waypointListIndex = 0; waypointListIndex < getWaypointInfos().size(); waypointListIndex++)
      {
         stream << "waypoint index = " << waypointListIndex << ::std::endl
                << "Location detail = " << getWaypointInfos().at(waypointListIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector< WaypointInfo > m_waypointInfos;
   ValidValue<unsigned int> m_optimizeFromSegment;
   ValidValue<unsigned int> m_optimizeToSegment;
   OptimizationStatus m_optimizationStatus;
   unsigned int m_currentDestinationIndex;
   bool m_skipCurrentWaypointOnDeviations;
   ValidValue< StartPositionInfo > m_startPositionInfo;
};

class TrafficImpactRouteAlternativeInfos
{
public:
   typedef ::std::pair<const unsigned char*, unsigned int> T_IconDataPair;/**<@DEPRECATED*/

   enum TrafficImpactCause
   {
      /* Currently there is a blocking on route */
      TRAFFIC_IMPACT_CAUSE_BLOCKING_ON_ROUTE,
      /* Currently there is a user defined blocking on route */
      TRAFFIC_IMPACT_CAUSE_USER_DEFINED_BLOCKING_ON_ROUTE,
      /* Improvement of relevant travel values (time, distance, consumption) on current route. Often caused by traffic event deletion */
      TRAFFIC_IMPACT_CAUSE_IMPROVEMENT,
      /* Worsening of relevant travel values (time, distance, consumption) on current route. Caused by new incoming traffic events */
      TRAFFIC_IMPACT_CAUSE_WORSENING,
      /* default value */
      TRAFFIC_IMPACT_CAUSE_INVALID
   };

   TrafficImpactRouteAlternativeInfos()
      :m_valid(false)
      ,m_trafficImpactCause(TRAFFIC_IMPACT_CAUSE_INVALID)
      ,m_isDetourPossible(true)
   {}

   bool isValid() const
   {
      return m_valid;
   }

   void setValid(bool isValid)
   {
      m_valid = isValid;
   }

   TrafficImpactCause getTrafficImpactCause() const
   {
      return m_trafficImpactCause;
   }

   void setTrafficImpactCause(TrafficImpactCause trafficImpactCause)
   {
      m_trafficImpactCause = trafficImpactCause;
   }

   bool isDetourPossible() const
   {
      return m_isDetourPossible;
   }

   void setDetourPossible(bool isDetourPossible)
   {
      m_isDetourPossible = isDetourPossible;
   }

   // TimeDifference
   void setTimeDifference(const ValidValue<int>& timeDifference)
   {
      m_timeDifference = timeDifference;
   }

   const ValidValue<int>& getTimeDifference()  const
   {
      return m_timeDifference;
   }

   // Distance Difference
   void setDistanceDifference(const ValidValue<int>& distanceDifference)
   {
      m_distanceDifference = distanceDifference;
   }

   const ValidValue<int>& getDistanceDifference() const
   {
      return m_distanceDifference;
   }

   // Consumption Difference
   void setConsumptionDifference(const ValidValue<int>& consumptionDifference)
   {
      m_consumptionDifference = consumptionDifference;
   }

   const ValidValue<int>& getConsumptionDifference() const
   {
      return m_consumptionDifference;
   }

   // TimeAdvantage
   void setTimeAdvantage(const ValidValue<unsigned int>& timeAdvantage)
   {
      m_timeAdvantage= timeAdvantage;
   }

   const ValidValue<unsigned int>& getTimeAdvantage() const
   {
      return m_timeAdvantage;
   }

   // Consumption Advantage
   void setConsumptionAdvantage(const ValidValue<unsigned int>& consumptionAdvantage)
   {
      m_consumptionAdvantage = consumptionAdvantage;
   }

   const ValidValue<unsigned int>& getConsumptionAdvantage() const
   {
      return m_consumptionAdvantage;
   }

   void setDistanceDifference(const ::std::string& distanceDifference)
   {
      m_stringDistanceDifference = distanceDifference;
   }

   const ::std::string& getDistanceDifferenceAsString() const
   {
      return m_stringDistanceDifference;
   }

   void setTimeDifference(const ::std::string& timeDifference)
   {
      m_stringTimeDifference = timeDifference;
   }

   const ::std::string& getTimeDifferenceAsString() const
   {
      return m_stringTimeDifference;
   }

   void setTimeAdvantage(const ::std::string& timeAdvantage)
   {
      m_stringTimeAdvantage = timeAdvantage;
   }

   const ::std::string& getTimeAdvantageAsString() const
   {
      return m_stringTimeAdvantage;
   }

   const ::std::string& getTrafficMessageDescription() const
   {
      return m_trafficMessageDescription;
   }

   void setTrafficMessageDescription(const ::std::string& trafficMessageDescription)
   {
      m_trafficMessageDescription = trafficMessageDescription;
   }

   const ::std::string& getTrafficMessageHeadline() const
   {
      return m_trafficMessageHeadline;
   }

   void setTrafficMessageHeadline(const ::std::string& trafficMessageHeadline)
   {
      m_trafficMessageHeadline = trafficMessageHeadline;
   }

   const ::std::string& getDistanceToTrafficMessageWithUnit() const
   {
      return m_distanceToTrafficMessageWithUnit;
   }

   void setDistanceToTrafficMessageWithUnit(const ::std::string& distanceToTrafficMessageWithUnit)
   {
      m_distanceToTrafficMessageWithUnit = distanceToTrafficMessageWithUnit;
   }

   const ValidValue<T_IconDataPair>& getTrafficImageBlob() const /**<@DEPRECATED*/
   {
      return m_trafficImageBlob;
   }

   void setTrafficImageBlob(const ValidValue<T_IconDataPair>& trafficImageBlob) /**<@DEPRECATED*/
   {
      m_trafficImageBlob = trafficImageBlob;
   }

   const ValidValue<Image>& getTrafficImage() const
   {
      return m_trafficImage;
   }

   void setTrafficImage(const ValidValue<Image>& trafficImage)
   {
      m_trafficImage = trafficImage;
   }

   const ::std::vector<ArrivalInfos::ArrivalInfo>& getArrivalInfos() const
   {
      return m_arrivalInfos;
   }

   ::std::vector<ArrivalInfos::ArrivalInfo>& getArrivalInfos()
   {
      return m_arrivalInfos;
   }

   TrafficImpactRouteAlternativeInfos& operator = (const TrafficImpactRouteAlternativeInfos& rhs)
   {
      if (&rhs != this)
      {
         m_valid = rhs.m_valid;
         m_trafficImpactCause = rhs.m_trafficImpactCause;
         m_isDetourPossible = rhs.m_isDetourPossible;
         m_timeDifference = rhs.m_timeDifference;
         m_distanceDifference = rhs.m_distanceDifference;
         m_consumptionDifference = rhs.m_consumptionDifference;
         m_timeAdvantage = rhs.m_timeAdvantage;
         m_consumptionAdvantage = rhs.m_consumptionAdvantage;
         m_stringDistanceDifference = rhs.m_stringDistanceDifference;
         m_stringTimeDifference = rhs.m_stringTimeDifference;
         m_stringTimeAdvantage = rhs.m_stringTimeAdvantage;
         m_trafficMessageHeadline = rhs.m_trafficMessageHeadline;
         m_trafficMessageDescription = rhs.m_trafficMessageDescription;
         m_trafficImageBlob = rhs.m_trafficImageBlob;
         m_trafficImage = rhs.m_trafficImage;
         m_distanceToTrafficMessageWithUnit = rhs.m_distanceToTrafficMessageWithUnit;
         m_arrivalInfos = rhs.m_arrivalInfos;
      }
      return *this;
   }

   bool operator == (const TrafficImpactRouteAlternativeInfos& rhs) const
   {
      return (m_valid == rhs.m_valid)&&
             (m_trafficImpactCause == rhs.m_trafficImpactCause)&&
             (m_isDetourPossible == rhs.m_isDetourPossible)&&
             (m_timeDifference == rhs.m_timeDifference)&&
             (m_distanceDifference == rhs.m_distanceDifference)&&
             (m_consumptionDifference == rhs.m_consumptionDifference)&&
             (m_timeAdvantage == rhs.m_timeAdvantage)&&
             (m_consumptionAdvantage ==  rhs.m_consumptionAdvantage)&&
             (m_stringDistanceDifference == rhs.m_stringDistanceDifference)&&
             (m_stringTimeDifference == rhs.m_stringTimeDifference)&&
             (m_stringTimeAdvantage == rhs.m_stringTimeAdvantage)&&
             (m_trafficMessageHeadline == rhs.m_trafficMessageHeadline)&&
             (m_trafficMessageDescription == rhs.m_trafficMessageDescription)&&
             (m_trafficImageBlob == rhs.m_trafficImageBlob)&&
             (m_trafficImage == rhs.m_trafficImage)&&
             (m_distanceToTrafficMessageWithUnit == rhs.m_distanceToTrafficMessageWithUnit)&&
             (m_arrivalInfos == rhs.m_arrivalInfos);
   }

   bool operator != (const TrafficImpactRouteAlternativeInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TrafficImpactRouteAlternativeInfos payload:\n");
      stream << ::std::endl;

      if(m_valid)
      {
         stream << "Is traffic impact data valid  =" << m_valid << ::std::endl
                << "Impact cause  =" << m_trafficImpactCause << ::std::endl
                << "Is Detour possible =" << m_isDetourPossible << ::std::endl
                << "Time difference =" << m_timeDifference.getValue() << ","
                << "Distance difference =" << m_distanceDifference.getValue() << ","
                << "Consumption difference =" << m_consumptionDifference.getValue() << ::std::endl
                << "Time difference as string =" << m_stringTimeDifference << ","
                << "Distance difference as string =" << m_stringDistanceDifference << ::std::endl
                << "Time advantage =" << m_timeAdvantage.getValue() << ", "
                << "Consumption advantage =" << m_consumptionAdvantage.getValue() << ", "
                << "Time advantage as string=" << m_stringTimeAdvantage << ", "
                << "Traffic message header =" << m_trafficMessageHeadline << ", "
                << "Traffic message description =" << m_trafficMessageDescription << ", "
                << "Distance to traffic message =" << m_distanceToTrafficMessageWithUnit << ", "
                << "Traffic message icon present =" << m_trafficImageBlob.isValid() << ","
                << "Traffic message icon image present =" << m_trafficImage.isValid() << ::std::endl;
         for(unsigned int arrivalInfoListIndex = 0; arrivalInfoListIndex < getArrivalInfos().size(); arrivalInfoListIndex++)
         {
            stream << "Arrival Infos list index = " << arrivalInfoListIndex << ::std::endl
                   << "Arrival info details  :" << getArrivalInfos().at(arrivalInfoListIndex).toString() << ::std::endl;
         }
      }
      else
      {
         stream << "No valid traffic impact data to display" << ::std::endl;
      }
      return stream.str();
   }

private:
   /* Indicates, if traffic impact data is valid. At any time the traffic impact data may become invalid,
    * because of new incoming traffic events, which may lead to a route recalculation. */
   bool m_valid;
   /* Holds cause of recalculation triggered by incoming traffic events. */
   TrafficImpactCause m_trafficImpactCause;
   /* Indicates if detour(alternative route) is possible due to traffic impact.
    * Default value kept as true since only in very few cases an alternative route will not be available
    * and this value will be set to false. For all other cases, this value will be true. */
   bool m_isDetourPossible;
   /* Expected travel time (in seconds) on proposed (not yet active) route alternative (taking new traffic scenario
    * into account) compared to previously expected travel time (old traffic scenario) on current route.
    * A positive value indicates, that the proposed route alternative is faster than the current one. */
   ValidValue<int> m_timeDifference;
   /* Expected travel distance (in meters) on proposed (not yet active) route alternative (taking new traffic scenario
    * into account) compared to previously expected travel distance (old traffic scenario) on current route.
    * A positive value indicates, that the proposed route alternative is shorter than the current one. */
   ValidValue<int> m_distanceDifference;
   /* Expected travel consumption on proposed (not yet active) route alternative (taking new traffic scenario
    * into account) compared to previously expected travel consumption (old traffic scenario) on current route.
    * A positive value indicates, that the proposed route alternative is more economical than the current one. */
   ValidValue<int> m_consumptionDifference;
   /* Time advantage (in seconds) of proposed (not yet active) route alternative compared to current route
    * (here for both routes the new traffic scenario is taken into account).
    * A positive value indicates, that the proposed route alternative is faster than the current one. */
   ValidValue<unsigned int> m_timeAdvantage;
   /* Consumption advantage of proposed (not yet active) route alternative compared to current route
    * (here for both routes the new traffic scenario is taken into account).
    * A positive value indicates, that the proposed route alternative is more economical than the current one. */
   ValidValue<unsigned int> m_consumptionAdvantage;
   /* see description of m_distanceDifference */
   ::std::string m_stringDistanceDifference;
   /* see description of m_timeDifference */
   ::std::string m_stringTimeDifference;
   /* see description of m_timeAdvantage */
   ::std::string m_stringTimeAdvantage;
   /* traffic message header */
   ::std::string m_trafficMessageHeadline;
   /* traffic message description */
   ::std::string m_trafficMessageDescription;
   /* traffic message icon */
   ValidValue<T_IconDataPair> m_trafficImageBlob; /**<@DEPRECATED*/
   ValidValue<Image> m_trafficImage;
   /* distance from current vehicle position to the traffic message including system unit */
   ::std::string m_distanceToTrafficMessageWithUnit;
   /* ArrivalInfos for detour */
   ::std::vector<ArrivalInfos::ArrivalInfo> m_arrivalInfos;
};

/* Represents traffic icon of type HMI resource and of type traffic event icons(as provided by TIP_FI component) */
enum TrafficIconType
{
   TRAFFIC_ICON_TYPE_HMI_RESOURCE,
   TRAFFIC_ICON_TYPE_BLOB
};

class RouteListInfos
{
public:

   enum TurnListIconType
   {
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_STRAIGHT,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_HALF_RIGHT,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_RIGHT,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_SHARP_RIGHT,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_BACK,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_SHARP_LEFT,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_LEFT,
      TURN_LIST_ICON_TYPE_ROUNDABOUT_EXIT_HALF_LEFT,
      TURN_LIST_ICON_TYPE_KEEP_LEFT,
      TURN_LIST_ICON_TYPE_KEEP_RIGHT,
      TURN_LIST_ICON_TYPE_TURN_STRAIGHT,
      TURN_LIST_ICON_TYPE_TURN_HALF_RIGHT,
      TURN_LIST_ICON_TYPE_TURN_RIGHT,
      TURN_LIST_ICON_TYPE_TURN_SHARP_RIGHT,
      TURN_LIST_ICON_TYPE_UTURN_LEFT,
      TURN_LIST_ICON_TYPE_TURN_SHARP_LEFT,
      TURN_LIST_ICON_TYPE_TURN_LEFT,
      TURN_LIST_ICON_TYPE_TURN_HALF_LEFT,
      TURN_LIST_ICON_TYPE_EXIT_LEFT,
      TURN_LIST_ICON_TYPE_EXIT_RIGHT,
      TURN_LIST_ICON_TYPE_UTURN_RIGHT,
      TURN_LIST_ICON_TYPE_MOTORWAY_INTERSECTION,
      TURN_LIST_ICON_TYPE_BORDER_CROSSING,
      TURN_LIST_ICON_TYPE_WAYPOINT_FLAG,
      TURN_LIST_ICON_TYPE_DESTINATION_FLAG,
      TURN_LIST_ICON_TYPE_SERVICE_ACCESS,
      TURN_LIST_ICON_TYPE_INVALID
   };

   enum AvoidElementsStatus
   {
      AVOID_ELEMENTS_STATUS_INACTIVE,
      AVOID_ELEMENTS_STATUS_ACTIVE,
      AVOID_ELEMENTS_STATUS_NO_DETOUR_POSSIBLE
   };

   enum RouteListError
   {
      ROUTE_LIST_ERROR__NO_ERROR,
      ROUTE_LIST_ERROR__BUSY,
      ROUTE_LIST_ERROR__INTERNAL_ERROR
   };

   RouteListInfos()
      : m_indexAlternative(0)
      , m_totalNumberOfElements(0)
      , m_currentDestinationIndex(0)
      , m_routeListError(ROUTE_LIST_ERROR__NO_ERROR)
      , m_avoidElementsStatus(AVOID_ELEMENTS_STATUS_INACTIVE)
      , m_hasRoadNumberImages(false)   //<-- INF4CV
   {}

   class RouteListElementInfo
   {
   public:
      typedef ::std::pair<const unsigned char*, unsigned int> T_IconDataPair;/**<@DEPRECATED*/
      typedef ::std::vector< T_IconDataPair > T_RoadNumberIconList;/**<@DEPRECATED*/
      typedef ::std::vector<Image> T_RoadNumberImageList;

      struct TrafficInfo
      {
         TrafficIconType m_trafficIconType;
         ::std::vector< T_IconDataPair > m_trafficImageBlobs; /**<@DEPRECATED*/
         ::std::vector<Image> m_trafficImages;
         ValidValue < ::navmiddleware::Color > m_trafficFlowColor;

         TrafficInfo()
            :m_trafficIconType(TRAFFIC_ICON_TYPE_HMI_RESOURCE)
         {}
         bool operator == (const TrafficInfo& rhs) const
         {
            return (m_trafficIconType == rhs.m_trafficIconType)&&
                   (m_trafficFlowColor == rhs.m_trafficFlowColor);
            // remark: member m_trafficImageBlobs and m_trafficImages are intentionally not used for comparison
         }

         bool operator != (const TrafficInfo& rhs) const
         {
            return !(*this == rhs);
         }
      };

      struct ExitPOIInfo
      {
         PoiData m_poiData; //--internal data; type used for transport purposes
         ::std::vector< ExitPOICategory > m_availablePOICategories;

         bool operator == (const ExitPOIInfo& rhs) const
         {
            return (m_poiData == rhs.m_poiData)&&
                   (m_availablePOICategories == rhs.m_availablePOICategories);
         }

         bool operator != (const ExitPOIInfo& rhs) const
         {
            return !(*this == rhs);
         }
      };

      struct SignPostInfo
      {
         ::std::string m_exitNumber;
         ::std::string m_exitName;
         ::std::vector<RoadNumberInfo> m_signpostRoadNumberInfos;
         ::std::vector< ::std::string> m_textLines;

         bool operator == (const SignPostInfo& rhs) const
         {
            return (m_exitNumber == rhs.m_exitNumber)&&
                   (m_exitName == rhs.m_exitName)&&
                   (m_signpostRoadNumberInfos == rhs.m_signpostRoadNumberInfos)&&
                   (m_textLines == rhs.m_textLines);
         }

         bool operator != (const SignPostInfo& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream("SignPostInfo payload:\n");
            stream << ::std::endl
                   << "exitName = " <<m_exitName << ::std::endl
                   << "exitNumber = " << m_exitNumber << ::std::endl
                   << "total number of Road Number icons " << m_signpostRoadNumberInfos.size() << ::std::endl;
            for(unsigned int roadNumberIndex = 0; roadNumberIndex < m_signpostRoadNumberInfos.size(); roadNumberIndex++)
            {
               stream <<::std::endl<< "road Number info at list index: " << roadNumberIndex << ::std::endl
                      << m_signpostRoadNumberInfos.at(roadNumberIndex).toString() << ::std::endl;
            }

            for(unsigned int textLineIndexIndex = 0; textLineIndexIndex < m_textLines.size(); textLineIndexIndex++)
            {
               stream <<::std::endl<< "text line index" << textLineIndexIndex << ": "
                      << m_textLines.at(textLineIndexIndex)<< ::std::endl;
            }
            return stream.str();
         }
      };

      struct TimeValuesIncludingChargingTime
      {
         TimeValuesIncludingChargingTime()
         {}

         TimeValuesIncludingChargingTime(
            const ::std::string& timeToStartOfElement,
            const ::std::string& etaAtStartOfElement)
            : m_timeToStartOfElement(timeToStartOfElement)
            , m_etaAtStartOfElement(etaAtStartOfElement)
         {}

         bool operator == (const TimeValuesIncludingChargingTime& rhs) const
         {
            return (m_timeToStartOfElement == rhs.m_timeToStartOfElement)&&
                   (m_etaAtStartOfElement == rhs.m_etaAtStartOfElement);
         }

         TimeValuesIncludingChargingTime& operator = (const TimeValuesIncludingChargingTime& rhs)
         {
            if (&rhs != this)
            {
               m_timeToStartOfElement = rhs.m_timeToStartOfElement;
               m_etaAtStartOfElement = rhs.m_etaAtStartOfElement;
            }
            return *this;
         }

         bool operator != (const TimeValuesIncludingChargingTime& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream("TimeValuesIncludingChargingTime payload:");
            stream << ::std::endl
                   << "Time To Start Of Element = " <<  m_timeToStartOfElement << ::std::endl
                   << "Eta At Start Of Element = "  <<  m_etaAtStartOfElement << ::std::endl;
            return stream.str();
         }

         /* formatted time from car position to start of route list element including charging time for all ePOIs
          * between car position and route list element */
         ::std::string m_timeToStartOfElement;
         // m_timeToStartOfElement converted based on timezone of route list element
         ::std::string m_etaAtStartOfElement;
      };


      RouteListElementInfo()
         : m_indexInRouteList(0)
         , m_lengthOfElementAsValue(0)
         , m_distanceToStartOfElementAsValue(0)
         , m_lengthAndDistanceAsValueUnitSystem(DISTANCEUNITSYSTEM_METRIC)
         , m_timeOfElementInSeconds(0)
         , m_turnListIconType(TURN_LIST_ICON_TYPE_INVALID)
         , m_trafficSense(TRAFFIC_SENSE__RIGHTHAND_TRAFFIC)
         , m_destinationOfSegment(0)
      {}

      void addRoadNumberIcon(const unsigned char* roadNumImageBlob, unsigned int roadNumImageBlobLength) /**<@DEPRECATED*/
      {
         if(roadNumImageBlobLength > 0)
         {
            m_roadNumberImageBlobs.push_back(::std::make_pair(roadNumImageBlob, roadNumImageBlobLength));
         }
      }

      void addRoadNumberIcon(const Image& roadNumberImage)
      {
         m_roadNumberImages.push_back(roadNumberImage);
      }

      const ValidValue< TrafficInfo >& getTrafficInfo() const
      {
         return m_trafficInfo;
      }

      void setTrafficInfo(const ValidValue< TrafficInfo >& trafficInfo)
      {
         m_trafficInfo = trafficInfo;
      }

      const ValidValue< ExitPOIInfo >& getExitPOIInfo() const
      {
         return m_exitPOIInfo;
      }

      void setPOIInfo(const ValidValue< ExitPOIInfo >& exitPOIInfo)
      {
         m_exitPOIInfo = exitPOIInfo;
      }

      const ::std::vector<SignPostInfo>& getSignPostInfoList() const
      {
         return m_signPostInfoList;
      }

      ::std::vector<SignPostInfo>& getSignPostInfoList()
      {
         return m_signPostInfoList;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("RouteListElementInfo payload:\n");

         stream << ::std::endl
                << "Index in complete route list = " << m_indexInRouteList << ::std::endl
                << "Length of the element = " << m_lengthOfElement << ::std::endl
                << "Length of the element as a number = " << m_lengthOfElementAsValue << ::std::endl
                << "Distance from car to start of the element = " << m_distanceToStartOfElement << ::std::endl
                << "Distance from car to start of the element as a number = " << m_distanceToStartOfElementAsValue << ::std::endl
                << "Unit system of length and distances in the element = " << m_lengthAndDistanceAsValueUnitSystem << ::std::endl
                << "Road info = " << m_roadInfo << ::std::endl
                << "Time of element = " << m_timeOfElement << ::std::endl
                << "Time of element in seconds = " << m_timeOfElementInSeconds << ::std::endl
                << "Time from car position to start of element = " << m_timeToStartOfElement << ::std::endl
                << "ETA at start of element = " << m_etaAtStartOfElement << ::std::endl
                << "Time values including charging time: " << m_timeValuesIncludingChargingTime.toString() << ::std::endl
                << "Formatted charging time at element = " << m_formattedChargingTimeAtElement.toString() << ::std::endl
                << "Charging time at element in seconds = " << m_chargingTimeAtElementInSeconds.toString() << ::std::endl
                << "Turn list icon type = " << m_turnListIconType << ::std::endl
                << "Traffic sense = " << m_trafficSense << ::std::endl
                << "Road Type Info: " << ::std::endl << m_routeInfoProperties.toString() << ::std::endl
                << m_coordinate.toString() << ::std::endl;

         if(m_trafficInfo.isValid())
         {
            stream << "traffic info is present and icon type is"
                   << m_trafficInfo.getValue().m_trafficIconType << ::std::endl;

            if( m_trafficInfo.getValue().m_trafficIconType == TRAFFIC_ICON_TYPE_BLOB )
            {
               for(unsigned int idx = 0; idx < m_trafficInfo.getValue().m_trafficImageBlobs.size(); ++idx) /**<deprecated*/
               {
                  stream << "Traffic event category icon len is " <<
                         m_trafficInfo.getValue().m_trafficImageBlobs[idx].second << ::std::endl;
               }
               for(unsigned int idx = 0; idx < m_trafficInfo.getValue().m_trafficImages.size(); ++idx)
               {
                  stream << "Traffic event category icon len is " <<
                         m_trafficInfo.getValue().m_trafficImages[idx].getBlob() << ::std::endl;
               }
            }

            if(m_trafficInfo.getValue().m_trafficFlowColor.isValid())
            {
               stream << "traffic flow color(R,G,B,Alpha) : " <<  m_trafficInfo.getValue().m_trafficFlowColor.getValue().toString() << ::std::endl;
            }
            else
            {
               stream << "Traffic flow color not available " << ::std::endl;
            }
         }
         else
         {
            stream << "no traffic info for icon type contained in this route list element" << ::std::endl;
         }

         if( m_exitPOIInfo.isValid())
         {
            stream << "m_exitPOIInfo is present"
                   << m_exitPOIInfo.getValue().m_poiData.m_poiId << ::std::endl;

            for(unsigned int idx = 0; idx < m_exitPOIInfo.getValue().m_availablePOICategories.size(); ++idx)
            {
               stream << "Available POI category is " <<
                      m_exitPOIInfo.getValue().m_availablePOICategories[idx] << ::std::endl;
            }
         }
         else
         {
            stream << "no exit POI info for this route list element" << ::std::endl;
         }

         for(unsigned int signPostIndex = 0; signPostIndex < getSignPostInfoList().size(); ++signPostIndex)
         {
            stream << "Sign Post Index: " << signPostIndex << ::std::endl
                   << getSignPostInfoList().at(signPostIndex).toString() << ::std::endl;
         }

         return stream.str();
      }

      bool operator == (const RouteListElementInfo& rhs) const
      {
         return (m_indexInRouteList == rhs.m_indexInRouteList)&&
                (m_roadInfo == rhs.m_roadInfo)&&
                (m_lengthOfElement == rhs.m_lengthOfElement)&&
                (m_lengthOfElementAsValue == rhs.m_lengthOfElementAsValue)&&
                (m_distanceToStartOfElement == rhs.m_distanceToStartOfElement)&&
                (m_distanceToStartOfElementAsValue == rhs.m_distanceToStartOfElementAsValue)&&
                (m_lengthAndDistanceAsValueUnitSystem == rhs.m_lengthAndDistanceAsValueUnitSystem)&&
                (m_timeOfElement == rhs.m_timeOfElement)&&
                (m_timeOfElementInSeconds == rhs.m_timeOfElementInSeconds)&&
                (m_timeToStartOfElement == rhs.m_timeToStartOfElement)&&
                (m_etaAtStartOfElement == rhs.m_etaAtStartOfElement)&&
                (m_timeValuesIncludingChargingTime == rhs.m_timeValuesIncludingChargingTime)&&
                (m_formattedChargingTimeAtElement == rhs.m_formattedChargingTimeAtElement)&&
                (m_chargingTimeAtElementInSeconds == rhs.m_chargingTimeAtElementInSeconds)&&
                (m_turnListIconType == rhs.m_turnListIconType)&&
                (m_trafficSense == rhs.m_trafficSense)&&
                (m_routeInfoProperties == rhs.m_routeInfoProperties)&&
                (m_coordinate == rhs.m_coordinate)&&
                (m_signPostInfoList == rhs.m_signPostInfoList)&&
                (m_destinationOfSegment == rhs.m_destinationOfSegment)&&
                (m_trafficInfo == rhs.m_trafficInfo);
         // remark: member m_roadNumberImageBlobs and m_roadNumberImages are intentionally not used for comparison
      }

      bool operator != (const RouteListElementInfo& rhs) const
      {
         return !(*this == rhs);
      }

      /* Index of the route info element in complete route list (HMI usually only displays a part of it) */
      unsigned int        m_indexInRouteList;
      /* road info string (created by evaluating road names and road numbers) */
      ::std::string       m_roadInfo;
      /* formatted length of route list element as string */
      ::std::string       m_lengthOfElement;
      /* length of route list element as a number */
      double              m_lengthOfElementAsValue;
      /* formatted length  from car position to start of route list element as string */
      ::std::string       m_distanceToStartOfElement;
      /* length  from car position to start of route list element as a number */
      double              m_distanceToStartOfElementAsValue;
      /* Unit System of length of element as a number */
      DistanceUnitSystem  m_lengthAndDistanceAsValueUnitSystem;
      /* formatted driving time of route list element */
      ::std::string        m_timeOfElement;
      // driving time of element in seconds
      uint32_t               m_timeOfElementInSeconds;
      /* formatted time from car position to start of route list element */
      ::std::string        m_timeToStartOfElement;
      /* formatted estimated time of arrival at start of route list element */
      ::std::string        m_etaAtStartOfElement;
      TimeValuesIncludingChargingTime  m_timeValuesIncludingChargingTime;
      // formatted charging time at element. Valid if element is an ePOI
      ValidValue < ::std::string>   m_formattedChargingTimeAtElement;
      // charging time at element. Valid if element is an ePOI
      ValidValue <uint32_t>   m_chargingTimeAtElementInSeconds;
      /* turn list icon identifier (simple guidance symbol) */
      TurnListIconType     m_turnListIconType;
      /* traffic sense */
      TrafficSense         m_trafficSense;
      /* list of pairs that hold road number icon byte arrays and array sizes */
      T_RoadNumberIconList m_roadNumberImageBlobs; //decrepated
      /* list of pairs that hold road number images */
      T_RoadNumberImageList m_roadNumberImages;
      /* Contains the road type information of route list element */
      RouteInfoProperties m_routeInfoProperties;
      /* geo coordinate */
      GeoCoordinateDegree m_coordinate;
      /* Holds the traffic info present in route list which could be of type HMI resource or blob data(contains traffic icons).
       * A positive value indicates, that there is traffic event located on the route list element.*/
      ValidValue<TrafficInfo> m_trafficInfo;
      /* Contains the POI information for exit elements */
      ValidValue<ExitPOIInfo> m_exitPOIInfo;
      /* Holds the value of destination of current segment if the particular route list element is either a waypoint or a destination*/
      unsigned int        m_destinationOfSegment;
      /* List of SignPostInfo */
      ::std::vector< SignPostInfo > m_signPostInfoList;
   };

   class RouteListElementIndexMapEntry
   {
   public:
      RouteListElementIndexMapEntry()
         : m_currentIndex(0)
         , m_previousIndex(0)
         , m_isExtendedElement(false)
      {}

      ::std::string toString() const
      {
         ::std::stringstream stream("RouteListElementIndexMapEntry payload:\n");

         stream << ::std::endl
                << "currentIndex = " <<m_currentIndex << ::std::endl
                << "previousIndex = " << m_previousIndex << ::std::endl
                << "isExtendedElement = " << m_isExtendedElement << ::std::endl;

         return stream.str();
      }

      bool operator == (const RouteListElementIndexMapEntry& rhs) const
      {
         return (m_currentIndex == rhs.m_currentIndex)&&
                (m_previousIndex == rhs.m_previousIndex)&&
                (m_isExtendedElement == rhs.m_isExtendedElement);
      }

      bool operator != (const RouteListElementIndexMapEntry& rhs) const
      {
         return !(*this == rhs);
      }

      //--index in currently valid route list
      unsigned int m_currentIndex;
      //--index in last requested route list
      unsigned int m_previousIndex;
      //--element is start of an extended element
      bool m_isExtendedElement;
   };


   const ::std::vector<RouteListElementInfo>& getRouteListElementInfos() const
   {
      return m_routeListElementInfos;
   }
   ::std::vector<RouteListElementInfo>& getRouteListElementInfos()
   {
      return m_routeListElementInfos;
   }

   const ::std::vector<RouteListElementIndexMapEntry>& getRouteListElementIndexMap() const
   {
      return m_routeListElementIndexMap;
   }
   ::std::vector<RouteListElementIndexMapEntry>& getRouteListElementIndexMap()
   {
      return m_routeListElementIndexMap;
   }

   const ::std::vector<unsigned int>& getAllIndexOfRouteListElementInfo() const
   {
      return m_allIndexOfRouteListElementInfo;
   }

   ::std::vector<unsigned int>& getAllIndexOfRouteListElementInfo()
   {
      return m_allIndexOfRouteListElementInfo;
   }

   unsigned int getCurrentDestinationIndex() const
   {
      return m_currentDestinationIndex;
   }

   void setCurrentDestinationIndex(unsigned int currentDestinationIndex)
   {
      m_currentDestinationIndex = currentDestinationIndex;
   }

   unsigned int getIndexAlternative() const
   {
      return m_indexAlternative;
   }

   void setIndexAlternative(unsigned int indexAlternative)
   {
      m_indexAlternative = indexAlternative;
   }

   unsigned int getTotalNumberOfElements() const
   {
      return m_totalNumberOfElements;
   }

   void setTotalNumberOfElements(unsigned int totalNumberOfElements)
   {
      m_totalNumberOfElements = totalNumberOfElements;
   }

   void setAvoidElementsStatus(AvoidElementsStatus status)
   {
      m_avoidElementsStatus = status;
   }

   AvoidElementsStatus getAvoidElementsStatus() const
   {
      return m_avoidElementsStatus;
   }

   void setRouteListError(RouteListError error)
   {
      m_routeListError = error;
   }

   RouteListError getRouteListError() const
   {
      return m_routeListError;
   }

   void setRequestId(const ValidValue<RequestId>& requestId)
   {
      m_requestId = requestId;
   }

   const ValidValue<RequestId>& getRequestId() const
   {
      return m_requestId;
   }

   ::std::string toString(AvoidElementsStatus avoidElementsStatus) const
   {
      switch (avoidElementsStatus)
      {
      case AVOID_ELEMENTS_STATUS_INACTIVE:
         return "AVOID_ELEMENTS_STATUS_INACTIVE";
      case AVOID_ELEMENTS_STATUS_ACTIVE:
         return "AVOID_ELEMENTS_STATUS_ACTIVE";
      case AVOID_ELEMENTS_STATUS_NO_DETOUR_POSSIBLE:
         return "AVOID_ELEMENTS_STATUS_NO_DETOUR_POSSIBLE";
      default:
         ::std::stringstream stream;
         stream << "avoidElementsStatus<" << static_cast<unsigned int>(avoidElementsStatus) << ">";
         return stream.str();
      }
   }

   //<-- INF4CV
   void setHasRoadNumberImages(bool hasRoadNumberImages)
   {
      m_hasRoadNumberImages = hasRoadNumberImages;
   }

   bool hasRoadNumberImages() const
   {
      return m_hasRoadNumberImages;
   }
   //->

   ::std::string toString(RouteListError routeListError) const
   {
      switch (routeListError)
      {
      case ROUTE_LIST_ERROR__NO_ERROR:
         return "ROUTE_LIST_ERROR__NO_ERROR";
      case ROUTE_LIST_ERROR__BUSY:
         return "ROUTE_LIST_ERROR__BUSY";
      case ROUTE_LIST_ERROR__INTERNAL_ERROR:
         return "ROUTE_LIST_ERROR__INTERNAL_ERROR";
      default:
         ::std::stringstream stream;
         stream << "routeListError<" << static_cast<unsigned int>(routeListError) << ">";
         return stream.str();
      }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RouteListInfos payload:\n");
      stream << ::std::endl;

      stream << "Total route list elements =" << m_totalNumberOfElements << ::std::endl
             << "Current destination index =" << m_currentDestinationIndex << ::std::endl
             << "RouteListError =" << toString(m_routeListError) << ::std::endl
             << "RequestId =" << m_requestId.toString() << ::std::endl
             << "AvoidElementStatus =" << toString( m_avoidElementsStatus) << ::std::endl;

      for(unsigned int routeListElementIndex = 0; routeListElementIndex < getRouteListElementInfos().size(); ++routeListElementIndex)
      {
         stream << "RouteListInfos: Index of requested element = " << routeListElementIndex << ::std::endl
                << "Route list element info =" << getRouteListElementInfos().at(routeListElementIndex).toString() << ::std::endl;
      }
      for(unsigned int routeElementIndex = 0; routeElementIndex < getRouteListElementIndexMap().size(); ++routeElementIndex)
      {
         stream << "RouteListElementIndexMapEntry: Index of requested element = " << routeElementIndex << ::std::endl
                << "Route list element index map entry =" << getRouteListElementIndexMap().at(routeElementIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

   bool operator == (const RouteListInfos& rhs) const
   {
      bool isEqual = m_indexAlternative == rhs.m_indexAlternative
                     && m_totalNumberOfElements == rhs.m_totalNumberOfElements
                     && m_currentDestinationIndex == rhs.m_currentDestinationIndex
                     && m_avoidElementsStatus == rhs.m_avoidElementsStatus
                     && m_hasRoadNumberImages == rhs.m_hasRoadNumberImages   //<-- INF4CV
                     && m_routeListElementInfos == rhs.m_routeListElementInfos
                     && m_routeListElementIndexMap == rhs.m_routeListElementIndexMap
                     && m_allIndexOfRouteListElementInfo == rhs.m_allIndexOfRouteListElementInfo
                     && m_routeListError == rhs.m_routeListError
                     && m_requestId == rhs.m_requestId;
      return isEqual;
   }

   bool operator != (const RouteListInfos& rhs) const
   {
      return !(*this == rhs);
   }

private:
   /* index of route alternative */
   unsigned int m_indexAlternative;
   /* Contains the total number of route list elements. */
   unsigned int m_totalNumberOfElements;
   /* index of current destination (in list of destinations) */
   unsigned int m_currentDestinationIndex;
   RouteListError m_routeListError;
   /* this will be same as the id returned by navmiddleware::requestRoutelist */
   ValidValue<RequestId> m_requestId;
   //<-- INF4CV
   /* Set to true if road number images are available in RouteListElementInfo */
   bool m_hasRoadNumberImages;
   //->

   /* ACTIVE if some route element(s) were avoided from a route, INACTIVE if not,
    * NO_DETOUR_POSSIBLE when calculation does not find an alternative*/
   AvoidElementsStatus m_avoidElementsStatus;

   ::std::vector<RouteListElementInfo> m_routeListElementInfos;
   ::std::vector<RouteListElementIndexMapEntry> m_routeListElementIndexMap;

   /* [JP] specific, NOT TO BE USED IN ROW  */
   /*Hold every RouteListElementInfo's m_indexInRouteList in a vector*/
   ::std::vector<unsigned int> m_allIndexOfRouteListElementInfo;
};

class JamAheadListInfo
{
public:
   enum JamAheadActivityStatus
   {
      JAM_AHEAD_ACTIVITY_STATUS_INACTIVE,
      JAM_AHEAD_ACTIVITY_STATUS_ACTIVE,
      JAM_AHEAD_ACTIVITY_STATUS_NO_DETOUR_POSSIBLE
   };

   class JamAheadListElement
   {
   public:
      JamAheadListElement():
         m_distanceUnit(DISTANCEUNIT_METERS),
         m_enabled(false)
      {
      }

      JamAheadListElement& operator = (const JamAheadListElement& rhs)
      {
         if (&rhs != this)
         {
            m_distance = rhs.m_distance;
            m_distanceWithoutUnit = rhs.m_distanceWithoutUnit;
            m_distanceUnit = rhs.m_distanceUnit;
            m_enabled = rhs.m_enabled;
         }
         return *this;
      }

      bool operator == (const JamAheadListElement& rhs) const
      {
         return (m_distance == rhs.m_distance)&&
                (m_distanceWithoutUnit == rhs.m_distanceWithoutUnit)&&
                (m_distanceUnit == rhs.m_distanceUnit)&&
                (m_enabled == rhs.m_enabled);
      }

      bool operator != (const JamAheadListElement& rhs) const
      {
         return !(*this == rhs);

      }

      ::std::string toString() const
      {
         ::std::stringstream stream("JamAheadListElement payload:\n");

         stream << ::std::endl
                << "distance of the jam =" << m_distance << ::std::endl
                << "distance without unit =" << m_distanceWithoutUnit <<::std::endl
                << "distance unit" << m_distanceUnit << ::std::endl
                << "Can the element be selected =" << m_enabled << ::std::endl;

         return stream.str();
      }

      /* distance with unit. can be mi or km depending on metric system*/
      ::std::string m_distance;
      ::std::string m_distanceWithoutUnit;
      DistanceUnit m_distanceUnit;
      /* is the element in the list selectable or not  */
      bool m_enabled;
   };

   JamAheadListInfo():
      m_jamAheadActivityStatus(JAM_AHEAD_ACTIVITY_STATUS_INACTIVE)
   {}

   void setJamAheadActivityStatus(JamAheadActivityStatus jamAheadStatus)
   {
      m_jamAheadActivityStatus = jamAheadStatus;
   }

   JamAheadActivityStatus getJamAheadActivityStatus() const
   {
      return m_jamAheadActivityStatus;
   }

   const ::std::vector<JamAheadListElement>& getJamAheadList() const
   {
      return m_jamAheadList;
   }

   ::std::vector<JamAheadListElement>& getJamAheadList()
   {
      return m_jamAheadList;
   }

   JamAheadListInfo& operator = (const JamAheadListInfo& rhs)
   {
      m_jamAheadList = rhs.m_jamAheadList;
      m_jamAheadActivityStatus = rhs.m_jamAheadActivityStatus;

      return *this;
   }

   bool operator == (const JamAheadListInfo& rhs) const
   {
      return (m_jamAheadList == rhs.m_jamAheadList)&&
             (m_jamAheadActivityStatus == rhs.m_jamAheadActivityStatus);
   }

   bool operator != (const JamAheadListInfo& rhs)
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("JamAheadListInfo payload:\n");
      stream << ::std::endl;
      stream << "Jam ahead activity status =";

      switch(m_jamAheadActivityStatus)
      {
      case JAM_AHEAD_ACTIVITY_STATUS_INACTIVE:
         stream << " INACTIVE" << ::std::endl;
         break;
      case JAM_AHEAD_ACTIVITY_STATUS_ACTIVE:
         stream << " ACTIVE" << ::std::endl;
         break;
      case JAM_AHEAD_ACTIVITY_STATUS_NO_DETOUR_POSSIBLE:
         stream << " NO_DETOUR_POSSIBLE" << ::std::endl;
         break;
      default:
         stream << m_jamAheadActivityStatus << ::std::endl;
         break;
      }

      for(unsigned int jamAheadListIndex = 0; jamAheadListIndex < getJamAheadList().size(); jamAheadListIndex++)
      {
         stream << "RouteListInfos: Route Alternative index = " << jamAheadListIndex << ::std::endl
                << "Jam Ahead List Element =" << getJamAheadList().at(jamAheadListIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<JamAheadListElement> m_jamAheadList;
   JamAheadActivityStatus m_jamAheadActivityStatus;
};

class RouteCalculationDurationInfos
{
public:
   RouteCalculationDurationInfos()
   {}

   ::navmiddleware::ValidValue<unsigned int> getRouteCalDurationFastToFinalDest() const
   {
      return m_routeCalDurationFastToFinalDest;
   }

   ::navmiddleware::ValidValue<unsigned int> getRouteCalDurationEcoToFinalDest() const
   {
      return m_routeCalDurationEcoToFinalDest;
   }

   ::navmiddleware::ValidValue<unsigned int> getRouteCalDurationShortToFinalDest() const
   {
      return m_routeCalDurationShortToFinalDest;
   }

   ::navmiddleware::ValidValue<unsigned int> getRouteCalDurationFastToNextDest() const
   {
      return m_routeCalDurationFastToNextDest;
   }

   ::navmiddleware::ValidValue<unsigned int> getRouteCalDurationEcoToNextDest() const
   {
      return m_routeCalDurationEcoToNextDest;
   }

   ::navmiddleware::ValidValue<unsigned int> getRouteCalDurationShortToNextDest() const
   {
      return m_routeCalDurationShortToNextDest;
   }

   void setRouteCalDurationFastToFinalDest(unsigned int routeCalcDurationFastToFinalDest)
   {
      m_routeCalDurationFastToFinalDest.setValue(routeCalcDurationFastToFinalDest);
   }

   void setRouteCalDurationEcoToFinalDest(unsigned int routeCalDurationEcoToFinalDest)
   {
      m_routeCalDurationEcoToFinalDest.setValue(routeCalDurationEcoToFinalDest);
   }

   void setRouteCalDurationShortToFinalDest(unsigned int routeCalDurationShortToFinalDest)
   {
      m_routeCalDurationShortToFinalDest.setValue(routeCalDurationShortToFinalDest);
   }

   void setRouteCalDurationFastToNextDest(unsigned int routeCalDurationFastToNextDest)
   {
      m_routeCalDurationFastToNextDest.setValue(routeCalDurationFastToNextDest);
   }

   void setRouteCalDurationEcoToNextDest(unsigned int routeCalDurationEcoToNextDest)
   {
      m_routeCalDurationEcoToNextDest.setValue(routeCalDurationEcoToNextDest);
   }

   void setRouteCalDurationShortToNextDest(unsigned int routeCalDurationShortToNextDest)
   {
      m_routeCalDurationShortToNextDest.setValue(routeCalDurationShortToNextDest);
   }

   void clear()
   {
      m_routeCalDurationFastToFinalDest.invalidate();
      m_routeCalDurationEcoToFinalDest.invalidate();
      m_routeCalDurationShortToFinalDest.invalidate();
      m_routeCalDurationFastToNextDest.invalidate();
      m_routeCalDurationEcoToNextDest.invalidate();
      m_routeCalDurationShortToNextDest.invalidate();
   }

   RouteCalculationDurationInfos& operator = (const RouteCalculationDurationInfos& rhs)
   {
      m_routeCalDurationFastToFinalDest = rhs.m_routeCalDurationFastToFinalDest;
      m_routeCalDurationEcoToFinalDest = rhs.m_routeCalDurationEcoToFinalDest;
      m_routeCalDurationShortToFinalDest = rhs.m_routeCalDurationShortToFinalDest;
      m_routeCalDurationFastToNextDest = rhs.m_routeCalDurationFastToNextDest;
      m_routeCalDurationEcoToNextDest = rhs.m_routeCalDurationEcoToNextDest;
      m_routeCalDurationShortToNextDest = rhs.m_routeCalDurationShortToNextDest;

      return *this;
   }

   bool operator == (const RouteCalculationDurationInfos& rhs) const
   {
      return (m_routeCalDurationFastToFinalDest == rhs.m_routeCalDurationFastToFinalDest)&&
             (m_routeCalDurationEcoToFinalDest == rhs.m_routeCalDurationEcoToFinalDest)&&
             (m_routeCalDurationShortToFinalDest == rhs.m_routeCalDurationShortToFinalDest)&&
             (m_routeCalDurationFastToNextDest == rhs.m_routeCalDurationFastToNextDest)&&
             (m_routeCalDurationEcoToNextDest == rhs.m_routeCalDurationEcoToNextDest)&&
             (m_routeCalDurationShortToNextDest == rhs.m_routeCalDurationShortToNextDest);
   }

   bool operator != (const RouteCalculationDurationInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RouteCalculationDurationInfos payload:\n");

      stream << ::std::endl
             << "Route Calculation duration(FAST) to final destination = " << m_routeCalDurationFastToFinalDest.getValue() << ""
             << "Route Calculation duration(ECO) to final destination = " << m_routeCalDurationEcoToFinalDest.getValue() << ""
             << "Route Calculation duration(SHORT) to final destination  = " << m_routeCalDurationShortToFinalDest.getValue() << ::std::endl
             << "Route Calculation duration(FAST) to next destination = " << m_routeCalDurationFastToNextDest.getValue() << ""
             << "Route Calculation duration(ECO) to next destination = " << m_routeCalDurationEcoToNextDest.getValue() << ""
             << "Route Calculation duration(SHORT) to next destination  = " << m_routeCalDurationShortToNextDest.getValue() << ::std::endl;

      return stream.str();
   }

private:
   ::navmiddleware::ValidValue<unsigned int> m_routeCalDurationFastToFinalDest;
   ::navmiddleware::ValidValue<unsigned int> m_routeCalDurationEcoToFinalDest;
   ::navmiddleware::ValidValue<unsigned int> m_routeCalDurationShortToFinalDest;
   ::navmiddleware::ValidValue<unsigned int> m_routeCalDurationFastToNextDest;
   ::navmiddleware::ValidValue<unsigned int> m_routeCalDurationEcoToNextDest;
   ::navmiddleware::ValidValue<unsigned int> m_routeCalDurationShortToNextDest;
};

class RouteCalculationInfo
{
public:
   enum Status
   {
      STATUS__NOT_STARTED = 0,
      STATUS__ROUTE_AVAILABLE,
      STATUS__ERROR
   };

   enum Error
   {
      ERROR__UNKNOWN = 0,
      ERROR__INVALID_LOCATION,
      ERROR__CALCULATION_FAILED,
      ERROR__ROUTE_IMPOSSIBLE,
      ERROR__ABORTED_BY_CLIENT
   };

   RouteCalculationInfo()
      : m_status(STATUS__NOT_STARTED)
   {}

   class RouteCalculationInfoAlternative
   {
   public:
      enum Reason
      {
         REASON__UNKNOWN = 0,
         REASON__NO_CALCULATION,
         REASON__FIRST_CALCULATION,
         REASON__OPTIMIZE_ROUTE,
         REASON__REGARDING_TRAFFIC_MESSAGE,
         REASON__DISREGARD_ROUTE,
         REASON__REGARDING_USER_TRAFFIC_MESSAGE,
         REASON__ROUTE_OPTION_CHANGED,
         REASON__ROUTE_CRITERION_CHANGED,
         REASON__ROUTE_OPTION_CRITERION_CHANGED,
         REASON__ENTRY_POINT_EVALUATION
      };

      RouteCalculationInfoAlternative()
         : m_reason(REASON__NO_CALCULATION)
         , m_routeCriterion(settings::ROUTE_CRITERION_ECONOMIC)   //<-- INF4CV
      {}

      class RouteRestrictionStatus
      {
      public:
         RouteRestrictionStatus()
            : m_isSeasonallyClosedRoadAvoided(false)
            , m_isTimeRestrictedRoadAvoided(false)
            , m_isLcvRestrictionViolated(false)
            //<-- INF4CV
            , m_isHardPhysicalRestrictionViolated(false)
            , m_isHardGeneralRestrictionViolated(false)
            , m_isSoftPhysicalRestrictionViolated(false)
            //->
         {}

         RouteRestrictionStatus& operator = (const RouteRestrictionStatus& rhs)
         {
            if (&rhs != this)
            {
               m_isSeasonallyClosedRoadAvoided = rhs.m_isSeasonallyClosedRoadAvoided;
               m_isTimeRestrictedRoadAvoided = rhs.m_isTimeRestrictedRoadAvoided;
               m_isLcvRestrictionViolated = rhs.m_isLcvRestrictionViolated;
               //<-- INF4CV
               m_isHardPhysicalRestrictionViolated = rhs.m_isHardPhysicalRestrictionViolated;
               m_isHardGeneralRestrictionViolated = rhs.m_isHardGeneralRestrictionViolated;
               m_isSoftPhysicalRestrictionViolated = rhs.m_isSoftPhysicalRestrictionViolated;
               //->
            }
            return *this;
         }

         bool operator == (const RouteRestrictionStatus& rhs) const
         {
            return ((m_isSeasonallyClosedRoadAvoided == rhs.m_isSeasonallyClosedRoadAvoided) &&
                    (m_isTimeRestrictedRoadAvoided == rhs.m_isTimeRestrictedRoadAvoided) &&
                    (m_isLcvRestrictionViolated == rhs.m_isLcvRestrictionViolated) &&
                    //<-- INF4CV
                    (m_isHardPhysicalRestrictionViolated == rhs.m_isHardPhysicalRestrictionViolated) &&
                    (m_isHardGeneralRestrictionViolated == rhs.m_isHardGeneralRestrictionViolated) &&
                    (m_isSoftPhysicalRestrictionViolated == rhs.m_isSoftPhysicalRestrictionViolated));
                    //->
         }

         bool operator != (const RouteRestrictionStatus& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream("RouteRestrictionStatus payload:\n");

            stream << ::std::endl
                   << "Is Seasonally Closed Road Avoided = " << m_isSeasonallyClosedRoadAvoided << ::std::endl
                   << "Is Time Restricted Road Avoided = " << m_isTimeRestrictedRoadAvoided << ::std::endl
                   << "Is LCV restriction violated = " << m_isLcvRestrictionViolated << ::std::endl
                   //<-- INF4CV
                   << "Is Hard Physical restriction violated = " << m_isHardPhysicalRestrictionViolated << ::std::endl
                   << "Is Hard General restriction violated = " << m_isHardGeneralRestrictionViolated << ::std::endl
                   << "Is Soft Physical restriction violated = " << m_isSoftPhysicalRestrictionViolated << ::std::endl;
                   //->

            return stream.str();
         }

         bool m_isSeasonallyClosedRoadAvoided;
         bool m_isTimeRestrictedRoadAvoided;
         bool m_isLcvRestrictionViolated;
         //<-- INF4CV
         bool m_isHardPhysicalRestrictionViolated;
         bool m_isHardGeneralRestrictionViolated;
         bool m_isSoftPhysicalRestrictionViolated;
         //->
      };

      RouteCalculationInfoAlternative& operator = (const RouteCalculationInfoAlternative& rhs)
      {
         if (&rhs != this)
         {
            m_reason = rhs.m_reason;
            m_routeRestrictionStatus = rhs.m_routeRestrictionStatus;
         }
         return *this;
      }

      bool operator == (const RouteCalculationInfoAlternative& rhs) const
      {
         return ( (m_reason == rhs.m_reason) &&
                  (m_routeRestrictionStatus == rhs.m_routeRestrictionStatus) );
      }

      bool operator != (const RouteCalculationInfoAlternative& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("RouteCalculationInfoAlternative payload:\n");
         stream << ::std::endl;

         stream << "Reason = ";

         switch(m_reason)
         {
         case REASON__UNKNOWN:
            stream << "UNKNOWN" << ::std::endl;
            break;
         case REASON__NO_CALCULATION:
            stream << "NO CALCULATION" << ::std::endl;
            break;
         case REASON__FIRST_CALCULATION:
            stream << "FIRST CALCULATION" << ::std::endl;
            break;
         case REASON__OPTIMIZE_ROUTE:
            stream << "OPTIMIZE ROUTE" << ::std::endl;
            break;
         case REASON__REGARDING_TRAFFIC_MESSAGE:
            stream << "REGARDING TRAFFIC MESSAGE" << ::std::endl;
            break;
         case REASON__DISREGARD_ROUTE:
            stream << "DISREGARD ROUTE" << ::std::endl;
            break;
         case REASON__REGARDING_USER_TRAFFIC_MESSAGE:
            stream << "REGARDING USER TRAFFIC MESSAGE" << ::std::endl;
            break;
         case REASON__ROUTE_OPTION_CHANGED:
            stream << "ROUTE OPTION CHANGED" << ::std::endl;
            break;
         case REASON__ROUTE_CRITERION_CHANGED:
            stream << "ROUTE CRITERION CHANGED" << ::std::endl;
            break;
         case REASON__ROUTE_OPTION_CRITERION_CHANGED:
            stream << "ROUTE OPTION CRITERION CHANGED" << ::std::endl;
            break;
         case REASON__ENTRY_POINT_EVALUATION:
            stream << "ENTRY POINT EVALUATION" << ::std::endl;
            break;

         default:
            stream << m_reason << ::std::endl;
            break;
         }

         stream << "Route Restriction Status Info:" << m_routeRestrictionStatus.toString() << ::std::endl;

         //<-- INF4CV
         stream << "RouteCriterion = ";

         switch(m_routeCriterion)
         {
         case settings::ROUTE_CRITERION_ECONOMIC:
            stream << "ROUTE_CRITERION_ECONOMIC" << ::std::endl;
            break;
         case settings::ROUTE_CRITERION_FAST:
            stream << "ROUTE_CRITERION_FAST" << ::std::endl;
            break;
         case settings::ROUTE_CRITERION_SHORT:
            stream << "ROUTE_CRITERION_SHORT"<< ::std::endl;
            break;

         default:
            stream << m_routeCriterion << ::std::endl;
            break;
         }
         //->

         return stream.str();
      }
      // indicates the reason of route calculation of the particular alternative route
      Reason m_reason;
      RouteRestrictionStatus m_routeRestrictionStatus;
      settings::RouteCriterion m_routeCriterion;   //<-- INF4CV
   };

   Status getStatus() const
   {
      return m_status;
   }

   void setStatus(Status status)
   {
      m_status = status;
   }

   const ValidValue<bool>& isRouteGuided() const
   {
      return m_isRouteGuided;
   }

   void setIsRouteGuided(const ValidValue<bool>& isRouteGuided)
   {
      m_isRouteGuided = isRouteGuided;
   }

   const ValidValue<Error>& getError() const
   {
      return m_error;
   }

   void setError(const ValidValue<Error>& error)
   {
      m_error = error;
   }

   const ValidValue<unsigned int>& getIndexActiveAlternative() const
   {
      return m_indexActiveAlternative;
   }

   void setIndexActiveAlternative(const ValidValue<unsigned int>& indexActiveAlternative)
   {
      m_indexActiveAlternative = indexActiveAlternative;
   }

   const ::std::vector<RouteCalculationInfoAlternative>& getRouteCalculationInfoAlternatives() const
   {
      return m_routeCalculationInfoAlternatives;
   }

   ::std::vector<RouteCalculationInfoAlternative>& getRouteCalculationInfoAlternatives()
   {
      return m_routeCalculationInfoAlternatives;
   }

   RouteCalculationInfo& operator = (const RouteCalculationInfo& rhs)
   {
      if (&rhs != this)
      {
         m_status = rhs.m_status;
         m_isRouteGuided = rhs.m_isRouteGuided;
         m_error = rhs.m_error;
         m_indexActiveAlternative = rhs.m_indexActiveAlternative;
         m_routeCalculationInfoAlternatives = rhs.m_routeCalculationInfoAlternatives;
      }
      return *this;
   }

   bool operator == (const RouteCalculationInfo& rhs) const
   {
      return (m_status == rhs.m_status)&&
             (m_isRouteGuided == rhs.m_isRouteGuided)&&
             (m_error == rhs.m_error)&&
             (m_indexActiveAlternative == rhs.m_indexActiveAlternative)&&
             (m_routeCalculationInfoAlternatives == rhs.m_routeCalculationInfoAlternatives);
   }

   bool operator != (const RouteCalculationInfo& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RouteCalculationInfo payload:\n");
      stream << ::std::endl;
      stream << "Status =";

      switch(m_status)
      {
      case STATUS__NOT_STARTED:
         stream << " NOT STARTED" << ::std::endl;
         break;
      case STATUS__ROUTE_AVAILABLE:
         stream << " ROUTE AVAILABLE" << ::std::endl;
         break;
      case STATUS__ERROR:
         stream << " ERROR" << ::std::endl;
         break;

      default:
         stream << m_status << ::std::endl;
         break;
      }

      stream << "Is Route Guided = " << m_isRouteGuided.toString() << ::std::endl;

      if(m_error.isValid())
      {
         stream << "Error =";
         unsigned int errorEnumValue = m_error.getValue();

         switch(errorEnumValue)
         {
         case ERROR__UNKNOWN:
            stream << " UNKNOWN" << ::std::endl;
            break;
         case ERROR__INVALID_LOCATION:
            stream << " INVALID LOCATION" << ::std::endl;
            break;
         case ERROR__CALCULATION_FAILED:
            stream << " CALCULATION FAILED" << ::std::endl;
            break;
         case ERROR__ROUTE_IMPOSSIBLE:
            stream << " ROUTE IMPOSSIBLE" << ::std::endl;
            break;
         case ERROR__ABORTED_BY_CLIENT:
            stream << " ABORTED BY CLIENT" << ::std::endl;
            break;

         default:
            stream << errorEnumValue << ::std::endl;
            break;
         }
      }

      stream << "Index active alternative = " << m_indexActiveAlternative.toString() << ::std::endl;

      for(unsigned int routeCalculationInfoAlternativeListIndex = 0;
            routeCalculationInfoAlternativeListIndex < getRouteCalculationInfoAlternatives().size();
            routeCalculationInfoAlternativeListIndex++)
      {
         stream << "Route Calculation Info Alternative List Index = " << routeCalculationInfoAlternativeListIndex << ::std::endl
                << "Route Calculation Info Alternative ="
                << getRouteCalculationInfoAlternatives().at(routeCalculationInfoAlternativeListIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   /*   m_status - indicates the current status of Route Calculation
      * m_status                | Calculation Status | Valid members                                               | Invalid members
      * -------------------------------------------- -----------------------------------------------------------------------------------------------------------------------------------------
      * STATUS__NOT_STARTED     | Not started yet    |                                                             | m_error, m_routeCalculationInfoAlternatives, m_indexActiveAlternative
      * STATUS__ROUTE_AVAILABLE | Successful         | m_routeCalculationInfoAlternatives, m_indexActiveAlternative | m_error
      * STATUS__ERROR           | Not successful     | m_error                                                     | m_routeCalculationInfoAlternatives, m_indexActiveAlternative
      * -------------------------------------------- ----------------------------------------------------------------------------------------------------------------------------------------
   */
   Status m_status;
   // Indicates whether the RouteCalculationInfo corresponds to a guided route or a planned route
   ValidValue<bool> m_isRouteGuided;
   ValidValue<Error> m_error;
   ValidValue<unsigned int> m_indexActiveAlternative;
   ::std::vector<RouteCalculationInfoAlternative> m_routeCalculationInfoAlternatives;
};

class PlannedRouteHistoryInfos
{
   /**
    * PlannedRouteHistoryInfos class saves all waypoint operations performed on planned route
    * The field m_waypointOperations: contains a list of waypoint operations performed on planned route
    */
public:
   enum WaypointOperation
   {
      WAYPOINT_OPERATION__INSERTED,
      WAYPOINT_OPERATION__DELETED,
      WAYPOINT_OPERATION__MOVED,
      WAYPOINT_OPERATION__REPLACED
   };

   PlannedRouteHistoryInfos() {}

   PlannedRouteHistoryInfos& operator = (const PlannedRouteHistoryInfos& rhs)
   {
      if (&rhs != this)
      {
         m_waypointOperations = rhs.m_waypointOperations;
      }
      return *this;
   }

   bool operator == (const PlannedRouteHistoryInfos& rhs) const
   {
      return (m_waypointOperations == rhs.m_waypointOperations);
   }

   bool operator != (const PlannedRouteHistoryInfos& rhs) const
   {
      return !(*this == rhs);
   }

   const ::std::vector<WaypointOperation>& getWaypointOperations() const
   {
      return m_waypointOperations;
   }

   ::std::vector<WaypointOperation>& getWaypointOperations()
   {
      return m_waypointOperations;
   }

   ::std::string toString(WaypointOperation operation) const
   {
      switch (operation)
      {
      case WAYPOINT_OPERATION__INSERTED:
         return "WAYPOINT_OPERATION__INSERTED";
      case WAYPOINT_OPERATION__DELETED:
         return "WAYPOINT_OPERATION__DELETED";
      case WAYPOINT_OPERATION__MOVED:
         return "WAYPOINT_OPERATION__MOVED";
      case WAYPOINT_OPERATION__REPLACED:
         return "WAYPOINT_OPERATION__REPLACED";
      default:
         ::std::stringstream stream;
         stream << "WAYPOINT_OPERATION__<" << static_cast<unsigned int>(operation) << ">";
         return stream.str();
      }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PlannedRouteHistoryInfos payload:\n");

      stream << ::std::endl
             << "Number of WaypointOperations =" << m_waypointOperations.size() << ::std::endl;
      for(unsigned int waypointOperationIndex = 0; waypointOperationIndex < m_waypointOperations.size(); waypointOperationIndex++)
      {
         stream << " Waypoint operation at index " << waypointOperationIndex <<" "<<
                toString(m_waypointOperations.at(waypointOperationIndex)) << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector< WaypointOperation > m_waypointOperations;

};

enum ChargerType
{
   CHARGER_TYPE__NORMAL = 0,
   CHARGER_TYPE__QUICK
};

inline ::std::string toString(const ChargerType& chargerType)
{
   switch ( chargerType )
   {
   case CHARGER_TYPE__NORMAL:
      return "CHARGER_TYPE__NORMAL";
   case CHARGER_TYPE__QUICK:
      return "CHARGER_TYPE__QUICK";
   default:
      {
         ::std::stringstream stream;
         stream << "CHARGER_TYPE__<" << chargerType << ">";
         return stream.str();
      }
   }
}

class Tour
{
public:
   class TourElement
   {
   public:
      class ChargingDetails
      {
      public:
         ChargingDetails()
            : m_chargerType(CHARGER_TYPE__NORMAL)
            , m_chargingTimeMinutes(0)
            , m_stateOfChargeIncrementPercent(0)
         {}

         ChargingDetails(const ChargerType& chargerType, uint16_t chargingTimeMinutes, uint16_t stateOfChargeIncrementPercent)
            : m_chargerType(chargerType)
            , m_chargingTimeMinutes(chargingTimeMinutes)
            , m_stateOfChargeIncrementPercent(stateOfChargeIncrementPercent)
         {}

         void setChargerType(const ChargerType& chargerType)
         {
            m_chargerType = chargerType;
         }

         ChargerType getChargerType() const
         {
            return m_chargerType;
         }

         void setChargingTimeMinutes(uint16_t chargingTimeMinutes)
         {
            m_chargingTimeMinutes = chargingTimeMinutes;
         }

         uint16_t getChargingTimeMinutes() const
         {
            return m_chargingTimeMinutes;
         }

         void setStateOfChargeIncrementPercent(uint16_t stateOfChargeIncrementPercent)
         {
            m_stateOfChargeIncrementPercent = std::min(stateOfChargeIncrementPercent, static_cast<uint16_t>(100));
         }

         uint16_t getStateOfChargeIncrementPercent() const
         {
            return m_stateOfChargeIncrementPercent;
         }

         bool operator==(const ChargingDetails& rhs) const
         {
            return (m_chargerType == rhs.m_chargerType) &&
                     (m_chargingTimeMinutes == rhs.m_chargingTimeMinutes) &&
                     (m_stateOfChargeIncrementPercent == rhs.m_stateOfChargeIncrementPercent);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "ChargingDetails: "
               << " ChargerType = " << navmiddleware::toString(m_chargerType)
               << ", Time = " << m_chargingTimeMinutes << "min"
               << ", StateOfCharge increment = " << m_stateOfChargeIncrementPercent << "%";
            return stream.str();
         }

      private:
         ChargerType m_chargerType;
         uint16_t m_chargingTimeMinutes;
         uint16_t m_stateOfChargeIncrementPercent;
      };

      TourElement() {}

      TourElement(const ExternalLocation& location, 
                  const ValidValue< ::navmiddleware::settings::RouteCriterion >& routeCriterion,
                  const ValidValue< WaypointType >& waypointType,
                  const ValidValue< ChargingStationAdditionType >& chargingStationAdditionType,
                  const ValidValue< ChargingDetails>& chargingDetails)
         : m_location(location)
         , m_routeCriterion(routeCriterion)
         , m_waypointType(waypointType)
         , m_chargingStationAdditionType(chargingStationAdditionType)
         , m_chargingDetails(chargingDetails)
      {}

      bool operator == (const TourElement& rhs) const
      {
         return (m_location == rhs.m_location)&&
                (m_routeCriterion == rhs.m_routeCriterion)&&
                (m_waypointType == rhs.m_waypointType)&&
                (m_chargingStationAdditionType == rhs.m_chargingStationAdditionType)&&
                (m_chargingDetails == rhs.m_chargingDetails);
      }

      bool operator != (const TourElement& rhs) const
      {
         return !(*this == rhs);
      }

      const ExternalLocation& getLocation() const
      {
         return m_location;
      }

      void setLocation(const ExternalLocation& location)
      {
         m_location = location;
      }

      const ValidValue< ::navmiddleware::settings::RouteCriterion >& getRouteCriterion() const
      {
         return m_routeCriterion;
      }

      void setRouteCriterion(const ValidValue< ::navmiddleware::settings::RouteCriterion >& routeCriterion)
      {
         m_routeCriterion = routeCriterion;
      }

      const ValidValue<WaypointType>& getWaypointType() const
      {
         return m_waypointType;
      }

      void setWaypointType(const ValidValue<WaypointType>& waypointType)
      {
         m_waypointType = waypointType;
      }

      const ValidValue<ChargingStationAdditionType>& getChargingStationAdditionType() const
      {
         return m_chargingStationAdditionType;
      }

      void setChargingStationAdditionType(const ValidValue<ChargingStationAdditionType>& chargingStationAdditionType)
      {
         m_chargingStationAdditionType = chargingStationAdditionType;
      }

      const ValidValue<ChargingDetails>& getChargingDetails() const
      {
         return m_chargingDetails;
      }

      void setChargingDetails(const ValidValue<ChargingDetails>& chargingDetails)
      {
         m_chargingDetails = chargingDetails;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "TourElement: ";

         stream << m_location.toString() << ::std::endl
                << "RouteCriterion = " << m_routeCriterion.toString() << ::std::endl
                << "Waypoint Type =" << (m_waypointType.isValid() ? ::navmiddleware::toString(m_waypointType.getValue()) : "InValid") << ::std::endl
                << "Charging Station AdditionType =" << (m_chargingStationAdditionType.isValid() ? ::navmiddleware::toString(m_chargingStationAdditionType.getValue()) : "InValid") << ::std::endl
                << "ChargingDetails = " << ::navmiddleware::toString< ::navmiddleware::Tour::TourElement::ChargingDetails>(m_chargingDetails) << ::std::endl;

         return stream.str();
      }

   private:
      ExternalLocation m_location;
      // Remark: default route criterion (as defined in route settings) will be used, if route criterion is not set / invalid
      ValidValue< ::navmiddleware::settings::RouteCriterion > m_routeCriterion;
      ValidValue<WaypointType> m_waypointType;
      ValidValue<ChargingStationAdditionType> m_chargingStationAdditionType;
      ValidValue<ChargingDetails> m_chargingDetails;
   };

   const ::std::vector<TourElement>& getTourElements() const
   {
      return m_tourElements;
   }

   ::std::vector<TourElement>& getTourElements()
   {
      return m_tourElements;
   }

   bool operator == (const Tour& rhs) const
   {
      return m_tourElements == rhs.m_tourElements;
   }

   bool operator != (const Tour& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("Tour:");
      stream << ::std::endl;
      for(unsigned int tourElementIndex = 0; tourElementIndex < getTourElements().size(); tourElementIndex++)
      {
         stream << "Tour element index = " << tourElementIndex << ::std::endl
                << getTourElements().at(tourElementIndex).toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector< TourElement > m_tourElements;
};

class RangePredictionInfos
{
public:

   class WaypointRangePredictionInfo
   {
   public:
      WaypointRangePredictionInfo()
         : m_rangePredictionStatus(RANGE_PREDICTION_STATUS__INACTIVE)
         , m_waypointIndex(0)
      {}

      WaypointRangePredictionInfo& operator = (const WaypointRangePredictionInfo& rhs)
      {
         if (&rhs != this)
         {
            m_rangePredictionStatus = rhs.m_rangePredictionStatus;
            m_waypointIndex = rhs.m_waypointIndex;
         }
         return *this;
      }

      bool operator == (const WaypointRangePredictionInfo& rhs) const
      {
         return (m_rangePredictionStatus == rhs.m_rangePredictionStatus)&&
                (m_waypointIndex == rhs.m_waypointIndex);
      }

      bool operator != (const WaypointRangePredictionInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("WaypointRangePredictionInfo payload:\n");

         stream << ::std::endl
                << "Range prediction status = " << ::navmiddleware::toString(m_rangePredictionStatus) << ::std::endl
                << "waypoint index : " << m_waypointIndex << ::std::endl;
         return stream.str();
      }

      RangePredictionStatus m_rangePredictionStatus;
      unsigned int m_waypointIndex;
   };

   RangePredictionInfos()
      : m_cvpRangePredictionStatus(RANGE_PREDICTION_STATUS__INACTIVE)
   {}

   RangePredictionInfos(RangePredictionStatus cvpRangePredictionStatus,
                        const ValidValue< WaypointRangePredictionInfo >& waypointRangePredictionInfo)
      : m_cvpRangePredictionStatus(cvpRangePredictionStatus)
      , m_waypointRangePredictionInfo(waypointRangePredictionInfo)
   {}

   bool operator == (const RangePredictionInfos& rhs) const
   {
      return (m_cvpRangePredictionStatus == rhs.m_cvpRangePredictionStatus)&&
             (m_waypointRangePredictionInfo == rhs.m_waypointRangePredictionInfo);
   }

   bool operator != (const RangePredictionInfos& rhs) const
   {
      return !(*this == rhs);
   }

   void setCvpRangePredictionStatus(RangePredictionStatus cvpRangePredictionStatus)
   {
      m_cvpRangePredictionStatus = cvpRangePredictionStatus;
   }

   RangePredictionStatus getCvpRangePredictionStatus() const
   {
      return m_cvpRangePredictionStatus;
   }

   void setWaypointRangePredictionInfo(const ValidValue< WaypointRangePredictionInfo >& waypointRangePredictionInfo)
   {
      m_waypointRangePredictionInfo = waypointRangePredictionInfo;
   }

   const ValidValue< WaypointRangePredictionInfo >& getWaypointRangePredictionInfo() const
   {
      return m_waypointRangePredictionInfo;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RangePredictionInfos payload:\n");

      stream << ::std::endl
             << "cvp range prediction status =" << ::navmiddleware::toString(m_cvpRangePredictionStatus) << ::std::endl
             << "Waypoint range prediction Info : " << (m_waypointRangePredictionInfo.isValid() ?
                   m_waypointRangePredictionInfo.getValue().toString() : "invalid") << ::std::endl;
      return stream.str();
   }

private:
   RangePredictionStatus m_cvpRangePredictionStatus;
   ValidValue < WaypointRangePredictionInfo > m_waypointRangePredictionInfo;
};

class RouteCountryAndStateInfos
{
public:
   class CountryAndStateInfo
   {
   public:
      CountryAndStateInfo() {}

      CountryAndStateInfo( const ValidValue< ::std::string>& countryName, const ValidValue< ::std::string>& stateName,
                           const ValidValue< ::std::string>& countryCode, const ValidValue< ::std::string>& stateCode )
         : m_countryName(countryName)
         , m_stateName(stateName)
         , m_countryCode(countryCode)
         , m_stateCode(stateCode)
      {}

      CountryAndStateInfo& operator = (const CountryAndStateInfo& rhs)
      {
         if (&rhs != this)
         {
            m_countryName = rhs.m_countryName;
            m_stateName = rhs.m_stateName;
            m_countryCode = rhs.m_countryCode;
            m_stateCode = rhs.m_stateCode;
            m_countryFlag = rhs.m_countryFlag;
            m_stateFlag = rhs.m_stateFlag;
         }
         return *this;
      }

      bool operator == (const CountryAndStateInfo& rhs) const
      {
         return ((m_countryName == rhs.m_countryName) &&
                 (m_stateName == rhs.m_stateName) &&
                 (m_countryCode == rhs.m_countryCode) &&
                 (m_stateCode == rhs.m_stateCode) &&
                 (m_countryFlag == rhs.m_countryFlag) &&
                 (m_stateFlag == rhs.m_stateFlag));
      }

      bool operator != (const CountryAndStateInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("CountryAndStateInfo payload:\n");

         stream << ::std::endl
                << " Country name = " << m_countryName.toString() << ::std::endl
                << " State name = " << m_stateName.toString() << ::std::endl
                << " Country code = " << m_countryCode.toString() << ::std::endl
                << " State code = " << m_stateCode.toString() << ::std::endl
                << " Country flag data = " << ((m_countryFlag.isValid() == true) ? m_countryFlag.getValue().toString()
                                               : "Country flag not available") << ::std::endl
                << " State flag data = " << ((m_stateFlag.isValid() == true) ? m_stateFlag.getValue().toString()
                                             : "State flag not available") << ::std::endl;

         return stream.str();
      }

      ValidValue< ::std::string> m_countryName;
      ValidValue< ::std::string> m_stateName;
      // country code returned as ISO 3166-1-alpha-3.
      ValidValue< ::std::string> m_countryCode;
      // subCountry code returned as ISO 3166-2. This could be an empty string eg: For Germany
      ValidValue< ::std::string> m_stateCode;
      ValidValue< Image > m_countryFlag;
      ValidValue< Image > m_stateFlag;
   };

   const ::std::vector< CountryAndStateInfo >& getRouteCountryAndStateInfos() const
   {
      return m_countryAndStateNameListInfos;
   }
   ::std::vector< CountryAndStateInfo >& getRouteCountryAndStateInfos()
   {
      return m_countryAndStateNameListInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RouteCountryAndStateInfos payload:\n");
      stream << ::std::endl
             << ::navmiddleware::toString<CountryAndStateInfo>(m_countryAndStateNameListInfos);

      return stream.str();
   }

   bool operator == (const RouteCountryAndStateInfos& rhs) const
   {
      return m_countryAndStateNameListInfos == rhs.m_countryAndStateNameListInfos;
   }

   bool operator != (const RouteCountryAndStateInfos& rhs) const
   {
      return !(*this == rhs);
   }

private:
   ::std::vector< CountryAndStateInfo > m_countryAndStateNameListInfos;
};

class AutomaticChargingStationDetectionInfo
{
public:
   AutomaticChargingStationDetectionInfo()
      : m_totalNumberOfDetectedChargingStations(0)
      , m_automaticChargingStationDetectionInfoStatus(AUTOMATIC_CHARGING_STATION_DETECTION_STATUS__INACTIVE)
      , m_automaticChargingStationDetectionReason(AUTOMATIC_CHARGING_STATION_DETECTION_REASON__UNKNOWN)
      , m_routeCalculationStatus(ROUTE_CALCULATION_STATUS__UNDEFINED)
   {}


   AutomaticChargingStationDetectionInfo(
      const ValidValue< unsigned int> totalNumberOfDetectedChargingStations,
      const ValidValue< ::std::string>& totalExtraTime,
      const ValidValue< ::std::string>& totalExtraDistance,
      AutomaticChargingStationDetectionStatus automaticChargingStationDetectionInfoStatus,
      AutomaticChargingStationDetectionReason automaticChargingStationDetectionReason,
      RouteCalculationStatus routeCalculationStatus)
      : m_totalNumberOfDetectedChargingStations(totalNumberOfDetectedChargingStations)
      , m_totalExtaTime(totalExtraTime)
      , m_totalExtraDistance(totalExtraDistance)
      , m_automaticChargingStationDetectionInfoStatus(automaticChargingStationDetectionInfoStatus)
      , m_automaticChargingStationDetectionReason(automaticChargingStationDetectionReason)
      , m_routeCalculationStatus(routeCalculationStatus)
   {}

   AutomaticChargingStationDetectionInfo& operator = (const AutomaticChargingStationDetectionInfo& rhs)
   {
      if (&rhs != this)
      {
         m_totalNumberOfDetectedChargingStations = rhs.m_totalNumberOfDetectedChargingStations;
         m_totalExtaTime = rhs.m_totalExtaTime;
         m_totalExtraDistance = rhs.m_totalExtraDistance;
         m_automaticChargingStationDetectionInfoStatus = rhs.m_automaticChargingStationDetectionInfoStatus;
         m_automaticChargingStationDetectionReason = rhs.m_automaticChargingStationDetectionReason;
         m_routeCalculationStatus = rhs.m_routeCalculationStatus;

      }
      return *this;
   }

   bool operator == (const AutomaticChargingStationDetectionInfo& rhs) const
   {
      return (m_totalNumberOfDetectedChargingStations == rhs.m_totalNumberOfDetectedChargingStations) &&
             (m_totalExtaTime == rhs.m_totalExtaTime) &&
             (m_totalExtraDistance == rhs.m_totalExtraDistance) &&
             (m_automaticChargingStationDetectionInfoStatus == rhs.m_automaticChargingStationDetectionInfoStatus) &&
             (m_automaticChargingStationDetectionReason == rhs.m_automaticChargingStationDetectionReason) &&
             (m_routeCalculationStatus == rhs.m_routeCalculationStatus);

   }

   bool operator != (const AutomaticChargingStationDetectionInfo& rhs) const
   {
      return !(*this == rhs);
   }

   const ValidValue< unsigned int>& getTotalNumberOfDetectedChargingStations() const
   {
      return m_totalNumberOfDetectedChargingStations;
   }

   void setTotalNumberOfDetectedChargingStations(const ValidValue<unsigned int>& totalNumberOfDetectedChargingStations)
   {
      m_totalNumberOfDetectedChargingStations = totalNumberOfDetectedChargingStations;
   }

   const ValidValue< ::std::string>& getTotalExtraTime() const
   {
      return m_totalExtaTime;
   }

   void setTotalExtraTime(const ValidValue< ::std::string>& totalExtraTime)
   {
      m_totalExtaTime = totalExtraTime;
   }

   const ValidValue< ::std::string>& getTotalExtraDistance() const
   {
      return m_totalExtraDistance;
   }

   void setTotalExtraDistance(const ValidValue< ::std::string>& totalExtraDistance)
   {
      m_totalExtraDistance = totalExtraDistance;
   }

   AutomaticChargingStationDetectionStatus getAutomaticChargingStationDetectionInfoStatus() const
   {
      return m_automaticChargingStationDetectionInfoStatus;
   }

   void setAutomaticChargingStationDetectionInfoStatus(AutomaticChargingStationDetectionStatus automaticChargingStationDetectionInfoStatus)
   {
      m_automaticChargingStationDetectionInfoStatus = automaticChargingStationDetectionInfoStatus;
   }

   AutomaticChargingStationDetectionReason getAutomaticChargingStationDetectionReason() const
   {
      return m_automaticChargingStationDetectionReason;
   }

   void setAutomaticChargingStationDetectionReason(AutomaticChargingStationDetectionReason automaticChargingStationDetectionReason)
   {
      m_automaticChargingStationDetectionReason = automaticChargingStationDetectionReason;
   }

   RouteCalculationStatus getRouteCalculationStatus() const
   {
      return m_routeCalculationStatus;
   }

   void setRouteCalculationStatus(RouteCalculationStatus routeCalculationStatus)
   {
      m_routeCalculationStatus = routeCalculationStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("AutomaticChargingStationDetectionInfos payload:\n");

      stream << ::std::endl
             << "Total num of charging stations detected = " << m_totalNumberOfDetectedChargingStations.toString() << ::std::endl
             << "Total extra time as string = " << (m_totalExtaTime.isValid()? m_totalExtaTime.getValue() : "invalid") << ::std::endl
             << "Total extra distance as string = " << (m_totalExtraDistance.isValid()? m_totalExtraDistance.getValue() : "invalid") << ::std::endl
             << "Automatic charging station detection status = " << ::navmiddleware::toString(m_automaticChargingStationDetectionInfoStatus) << ::std::endl
             << "Automatic charging station detection reason = " << ::navmiddleware::toString(m_automaticChargingStationDetectionReason) << ::std::endl
             << "Route Calculation Status = " << ::navmiddleware::toString(m_routeCalculationStatus) << ::std::endl;
      return stream.str();
   }

private:
   ValidValue< unsigned int> m_totalNumberOfDetectedChargingStations;
   /* Total driving time, including time to reach first waypoint and
    * all intermediate POIs, plus charging time and AverageTimeOfStay. */
   ValidValue< ::std::string> m_totalExtaTime;
   /* Total driving distance, including distance to all intermediate POIs. */
   ValidValue< ::std::string> m_totalExtraDistance;
   AutomaticChargingStationDetectionStatus m_automaticChargingStationDetectionInfoStatus;
   AutomaticChargingStationDetectionReason m_automaticChargingStationDetectionReason;
   RouteCalculationStatus m_routeCalculationStatus;

};

class ShapePointsAvailabilityInfo
{
public:
   ShapePointsAvailabilityInfo()
      : m_shapePointsAvailability(false)
      , m_revisionNumber(0)
   {
   }

   ShapePointsAvailabilityInfo(
      bool shapePointsAvailability
      , uint64_t revisionNumber
      , ValidValue < uint32_t > shapePointIndexBasedOnCVP)
      : m_shapePointsAvailability(shapePointsAvailability)
      , m_revisionNumber(revisionNumber)
      , m_shapePointIndexBasedOnCVP(shapePointIndexBasedOnCVP)
   {
   }

   bool getShapePointsAvailability() const
   {
      return m_shapePointsAvailability;
   }

   void setShapePointsAvailability(bool shapePointsAvailability)
   {
      m_shapePointsAvailability = shapePointsAvailability;
   }

   uint64_t getRevisionNumber() const
   {
      return m_revisionNumber;
   }

   void setRevisionNumber(uint64_t revisionNumber)
   {
      m_revisionNumber = revisionNumber;
   }

   const ValidValue < uint32_t >& getShapePointIndexBasedOnCVP() const
   {
      return m_shapePointIndexBasedOnCVP;
   }

   void setShapePointIndexBasedOnCVP(const ValidValue < uint32_t >& shapePointIndexBasedOnCVP)
   {
      m_shapePointIndexBasedOnCVP = shapePointIndexBasedOnCVP;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ShapePointsAvailabilityInfo payload:\n");

      stream << ::std::endl
             << "ShapePointsAvailability = " << m_shapePointsAvailability << ::std::endl
             << "RevisionNumber = " << m_revisionNumber << ::std::endl
             << "ShapePointIndexBasedOnCVP = " << m_shapePointIndexBasedOnCVP.toString() << ::std::endl;
      return stream.str();
   }

   bool operator == (const ShapePointsAvailabilityInfo& rhs) const
   {
      return (m_shapePointsAvailability == rhs.m_shapePointsAvailability) &&
             (m_revisionNumber == rhs.m_revisionNumber) &&
             (m_shapePointIndexBasedOnCVP == rhs.m_shapePointIndexBasedOnCVP);
   }

   bool operator != (const ShapePointsAvailabilityInfo& rhs) const
   {
      return !(*this == rhs);
   }

private:
   bool m_shapePointsAvailability;
   /*
    * Whenever shape points are available revision number will be set along with shapePointsAvailability.
    * And the same revision number will be in the ShapePointsInfo on further request of ShapePointsInfo and client can
    * make sure for which revision number the shape points provided.
    * This revision number will be incremented if there is any route recalculation happens.
    * If it mismatches with revision number in the ShapePointsInfo, client can re-request for ShapePointsInfo.
    */
   uint64_t m_revisionNumber;
   /*
    * The shapePointIndexBasedOnCVP shall not be used if ShapePointsAvailabilityInfo::revisionNumber does not match
    * with the ShapePointsInfo::revisionNumber.
    * This will be valid when client request for latest ShapePointsInfo and shapePointIndexBasedOnCVP with respect to
    * ShapePointsInfo::shapePoints will be provided.
    * At start this will be 0 and this will be incremented when vechicle is moving and the air distance between the
    * current vechicle position and next shape point is below 100 meters.
    */
   ValidValue < uint32_t > m_shapePointIndexBasedOnCVP;
};

class ShapePointsInfo
{
public:
   ShapePointsInfo()
      : m_revisionNumber(0)
      , m_areShapePointsProvidedForCompleteRoute(false)
   {
   }

   ShapePointsInfo(
      uint64_t revisionNumber,
      const ::std::vector< GeoCoordinateDegree > shapePoints,
      bool areShapePointsProvidedForCompleteRoute)
      : m_revisionNumber(revisionNumber)
      , m_shapePoints(shapePoints)
      , m_areShapePointsProvidedForCompleteRoute(areShapePointsProvidedForCompleteRoute)
   {
   }

   uint64_t getRevisionNumber() const
   {
      return m_revisionNumber;
   }

   void setRevisionNumber(uint64_t revisionNumber)
   {
      m_revisionNumber = revisionNumber;
   }

   const ::std::vector< GeoCoordinateDegree >& getShapePoints() const
   {
      return m_shapePoints;
   }

   void setShapePoints(const ::std::vector< GeoCoordinateDegree >& shapePoints)
   {
      m_shapePoints = shapePoints;
   }

   bool areShapePointsProvidedForCompleteRoute() const
   {
      return m_areShapePointsProvidedForCompleteRoute;
   }

   void setAreShapePointsProvidedForCompleteRoute(bool areShapePointsProvidedForCompleteRoute)
   {
      m_areShapePointsProvidedForCompleteRoute = areShapePointsProvidedForCompleteRoute;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ShapePointsInfos payload:\n");

      stream << ::std::endl << "RevisionNumber = " << m_revisionNumber << ::std::endl;

      for(unsigned int shapePointIndex = 0; shapePointIndex < getShapePoints().size(); shapePointIndex++)
      {
         stream << "ShapePoint: Index = " << shapePointIndex << ::std::endl
                << getShapePoints().at(shapePointIndex).toString();
      }
      stream << "AreShapePointsProvidedForCompleteRoute = " << m_areShapePointsProvidedForCompleteRoute << ::std::endl;

      return stream.str();
   }

   bool operator == (const ShapePointsInfo& rhs) const
   {
      return (m_revisionNumber == rhs.m_revisionNumber) &&
             (m_shapePoints == rhs.m_shapePoints) &&
             (m_areShapePointsProvidedForCompleteRoute == rhs.m_areShapePointsProvidedForCompleteRoute);
   }

   bool operator != (const ShapePointsInfo& rhs) const
   {
      return !(*this == rhs);
   }

private:
   /*
    * This is the revision number same as the latest revision number sent in ShapePointsAvailabilityInfo previously.
    * Client can make sure after receiving the ShapePointsInfo whether the same ShapePointsInfo received for the
    * previously received ShapePointsAvailabilityInfo.
    */
   uint64_t m_revisionNumber;
   ::std::vector< GeoCoordinateDegree > m_shapePoints;
   /*
    * AreShapePointsProvidedForCompleteRoute represents whether the m_shapePoints contains the shape points till
    * final destination or not.
    * The shape points are not provided for complete route in case of longer routes, in this case
    * AreShapePointsProvidedForCompleteRoute flag is false initially, when vechicle is moving and re-requests of shape
    * points info if shape points are given till final destination this flag will be set to true.
    */
   bool m_areShapePointsProvidedForCompleteRoute;
};

}
#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_ROUTEINFOS_H_
