/**
* @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_NAVDATAUPDATEINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_NAVDATAUPDATEINFOS_H_

#include <stdint.h>
#include <sstream>
#include <string>
#include <vector>
#include <set>
#include <map>
#include <utility>
#include "InfoTypes.h"

namespace navmiddleware
{

enum AvailableUpdatesRequestType
{
   AVAILABLE_UPDATES_REQUEST_TYPE__INITIAL_REQUEST,
   AVAILABLE_UPDATES_REQUEST_TYPE__INITIAL_REQUEST_BY_DIAGNOSTIC,
   AVAILABLE_UPDATES_REQUEST_TYPE__MODIFY_SELECTION_REQUEST,
   AVAILABLE_UPDATES_REQUEST_TYPE__CANCEL_SELECTION_REQUEST
};

enum ScomoComponent
{
   SCOMO_COMPONENT__UNKNOWN,
   SCOMO_COMPONENT__MAP_CATALOGUE,
   SCOMO_COMPONENT__MAP_CONTEXT,
   SCOMO_COMPONENT__MAP_REGION,
   SCOMO_COMPONENT__MAP_COMPATIBILITY_MATRIX
};

inline ::std::string ScomoComponentToString(ScomoComponent scomoComponent)
{
   switch (scomoComponent)
   {
   case SCOMO_COMPONENT__MAP_CATALOGUE:
      return("SCOMO_COMPONENT__MAP_CATALOGUE");
   case SCOMO_COMPONENT__MAP_CONTEXT:
      return("SCOMO_COMPONENT__MAP_CONTEXT");
   case SCOMO_COMPONENT__MAP_REGION:
      return("SCOMO_COMPONENT__MAP_REGION");
   case SCOMO_COMPONENT__MAP_COMPATIBILITY_MATRIX:
      return("SCOMO_COMPONENT__MAP_COMPATIBILITY_MATRIX");
   default:
      return("SCOMO_COMPONENT__UNKNOWN");
   }
}

enum ScomoComponentRegistrationStatus
{
   SCOMO_COMPONENT_REGISTRATION_STATUS__UNKNOWN,
   SCOMO_COMPONENT_REGISTRATION_STATUS__REGISTERED,
   SCOMO_COMPONENT_REGISTRATION_STATUS__FAILURE
};

inline ::std::string toString(ScomoComponentRegistrationStatus registrationStatus)
{
   switch (registrationStatus)
   {
   case SCOMO_COMPONENT_REGISTRATION_STATUS__UNKNOWN:
      return("SCOMO_COMPONENT_REGISTRATION_STATUS__UNKNOWN");
   case SCOMO_COMPONENT_REGISTRATION_STATUS__REGISTERED:
      return("SCOMO_COMPONENT_REGISTRATION_STATUS__REGISTERED");
   case SCOMO_COMPONENT_REGISTRATION_STATUS__FAILURE:
      return("SCOMO_COMPONENT_REGISTRATION_STATUS__FAILURE");
   default:
      return("SCOMO_COMPONENT_REGISTRATION_STATUS__UNKNOWN");
   }
}

enum CatalogueStatus
{
   CATALOGUE_STATUS__UNKNOWN,
   CATALOGUE_STATUS__AVAILABLE,
   CATALOGUE_STATUS__NOT_AVAILABLE,
   CATALOGUE_STATUS__OUTDATED,
   CATALOGUE_STATUS__UPDATING
};

inline ::std::string CatalogueStatusToString(CatalogueStatus catalogueStatus)
{
   switch (catalogueStatus)
   {
   case CATALOGUE_STATUS__AVAILABLE:
      return("CATALOGUE_STATUS__AVAILABLE");
   case CATALOGUE_STATUS__NOT_AVAILABLE:
      return("CATALOGUE_STATUS__NOT_AVAILABLE");
   case CATALOGUE_STATUS__OUTDATED:
      return("CATALOGUE_STATUS__OUTDATED");
   case CATALOGUE_STATUS__UPDATING:
      return("CATALOGUE_STATUS__UPDATING");
   default:
      return("CATALOGUE_STATUS__UNKNOWN");
   }
}

enum NavDataUpdateUSBDeviceStatus
{
   NAVDATAUPDATE_USB_DEVICE_STATUS__SUCCESS,
   NAVDATAUPDATE_USB_DEVICE_STATUS__FAILURE,
   NAVDATAUPDATE_USB_DEVICE_STATUS__OUT_OF_SPACE_ERROR,
   NAVDATAUPDATE_USB_DEVICE_STATUS__REMOVED_ERROR,
   NAVDATAUPDATE_USB_DEVICE_STATUS__SUBSCRIPTION_EXPIRED
};

enum NavDataUpdateInstallStage
{
   NAVDATAUPDATE_INSTALL_STAGE__UNKNOWN,
   NAVDATAUPDATE_INSTALL_STAGE__EXECUTE_INSTALLATION,
   NAVDATAUPDATE_INSTALL_STAGE__ACTIVATE_INSTALLATION,
   NAVDATAUPDATE_INSTALL_STAGE__COMPLETED
};

enum NavDataStatus
{
   NAVDATA_STATUS__OK,
   NAVDATA_STATUS__INCONSISTENT,
   NAVDATA_STATUS__UPDATE_IN_PROGRESS,
   NAVDATA_STATUS__FULL_UPDATE_IN_PROGRESS,
   NAVDATA_STATUS__UNKNOWN
};

inline ::std::string NavDataStatusToString(NavDataStatus navDataStatus)
{
   switch (navDataStatus)
   {
   case NAVDATA_STATUS__OK:
      return("NAVDATA_STATUS__OK");
   case NAVDATA_STATUS__INCONSISTENT:
      return("NAVDATA_STATUS__INCONSISTENT");
   case NAVDATA_STATUS__UPDATE_IN_PROGRESS:
      return("NAVDATA_STATUS__UPDATE_IN_PROGRESS");
   case NAVDATA_STATUS__FULL_UPDATE_IN_PROGRESS:
      return("NAVDATA_STATUS__FULL_UPDATE_IN_PROGRESS");
   default:
      return("NAVDATA_STATUS__UNKNOWN");
   }
}

enum DeinstallResult
{
   DEINSTALL_RESULT__SUCCESS,
   DEINSTALL_RESULT__DEINSTALL_PACKAGE_MISSING,
   DEINSTALL_RESULT__BUSY,
   DEINSTALL_RESULT__REGION_NOT_INSTALLED,
   DEINSTALL_RESULT__SUBSCRIPTION_EXPIRED,
   DEINSTALL_RESULT__FAILED,
   DEINSTALL_RESULT__UNSUPPORTED_UPDATE_DEVICE,
   DEINSTALL_RESULT__UNKNOWN
};

/** The product types are directly matching to the NDS specification (ProductTypeMaskValues)*/
enum ProductType
{
   PRODUCT_TYPE__NAVIGATION     = 1,
   PRODUCT_TYPE__POI            = 2,
   PRODUCT_TYPE__DTM            = 4,
   PRODUCT_TYPE__OBJECTS3D      = 8,
   PRODUCT_TYPE__CONTOUR        = 16,
   PRODUCT_TYPE__AERIAL         = 32,
   PRODUCT_TYPE__SATELLITE      = 64,
   PRODUCT_TYPE__WORLD_OVERVIEW = 128,
   PRODUCT_TYPE__ADAS           = 256,
   PRODUCT_TYPE__DISPLAY_ONLY   = 512
};

enum UpdateFavoriteUpdateRegionsResult
{
   SET_FAVORITE_UPDATE_REGION_RESULT__UNKNOWN,
   SET_FAVORITE_UPDATE_REGION_RESULT__SUCCESS,
   SET_FAVORITE_UPDATE_REGION_RESULT__FAILED,
   SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_UPDATE_REGION_NOT_AVAILABLE,
   SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_NOT_ALLOWED
};

inline ::std::string toString(UpdateFavoriteUpdateRegionsResult result)
{
   switch (result)
   {
   case SET_FAVORITE_UPDATE_REGION_RESULT__SUCCESS:
      return("SET_FAVORITE_UPDATE_REGION_RESULT__SUCCESS");

   case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED:
      return("SET_FAVORITE_UPDATE_REGION_RESULT__FAILED");

   case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_UPDATE_REGION_NOT_AVAILABLE:
      return("SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_UPDATE_REGION_NOT_AVAILABLE");

   case SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_NOT_ALLOWED:
      return("SET_FAVORITE_UPDATE_REGION_RESULT__FAILED_NOT_ALLOWED");

   default:
      return("SET_FAVORITE_UPDATE_REGION_RESULT__UNKNOWN");
   }
}

enum AutomaticMapUpdateStatusResult
{
   AUTOMATIC_MAP_UPDATE_STATUS_RESULT__SUCCESS,
   AUTOMATIC_MAP_UPDATE_STATUS_RESULT__FAILURE
};

inline ::std::string toString(AutomaticMapUpdateStatusResult automaticMapUpdateStatusResult)
{
   switch (automaticMapUpdateStatusResult)
   {
   case AUTOMATIC_MAP_UPDATE_STATUS_RESULT__SUCCESS:
      return("AUTOMATIC_MAP_UPDATE_STATUS_RESULT__SUCCESS");

   case AUTOMATIC_MAP_UPDATE_STATUS_RESULT__FAILURE:
      return("AUTOMATIC_MAP_UPDATE_STATUS_RESULT__FAILURE");

   default:
      return("AUTOMATIC_MAP_UPDATE_STATUS_RESULT__NOT_DEFINED");
   }
}

enum DeviceType
{
   DEVICETYPE__USB,
   DEVICETYPE__SMARTPHONE,
   DEVICETYPE__TCP,
   DEVICETYPE__SOTA,
   DEVICETYPE__SOTA_HMI,
   DEVICETYPE__LOCAL_CATALOGUE,
   DEVICETYPE__UNKNOWN
};

inline ::std::string toString(DeviceType deviceType)
{
   switch (deviceType)
   {
   case DEVICETYPE__USB:
      return("DEVICETYPE__USB");
   case DEVICETYPE__SMARTPHONE:
      return("DEVICETYPE__SMARTPHONE");
   case DEVICETYPE__TCP:
      return("DEVICETYPE__TCP");
   case DEVICETYPE__SOTA:
      return("DEVICETYPE__SOTA");
   case DEVICETYPE__SOTA_HMI:
      return("DEVICETYPE__SOTA_HMI");
   case DEVICETYPE__LOCAL_CATALOGUE:
      return("DEVICETYPE__LOCAL_CATALOGUE");
   case DEVICETYPE__UNKNOWN:
      return("DEVICETYPE__UNKNOWN");
   default:
      return("DEVICETYPE__UNKNOWN");
   }
}

enum DeviceStatus
{
   DEVICESTATUS__AVAILABLE,
   DEVICESTATUS__INC_UPDATE,
   DEVICESTATUS__FULL_UPDATE,
   DEVICESTATUS__AUTOMATIC_FULL_UPDATE,
   DEVICESTATUS__UNKNOWN
};

inline ::std::string toString(DeviceStatus deviceStatus)
{
   switch (deviceStatus)
   {
   case DEVICESTATUS__AVAILABLE:
      return("DEVICESTATUS__AVAILABLE");
   case DEVICESTATUS__INC_UPDATE:
      return("DEVICESTATUS__INC_UPDATE");
   case DEVICESTATUS__FULL_UPDATE:
      return("DEVICESTATUS__FULL_UPDATE");
   case DEVICESTATUS__AUTOMATIC_FULL_UPDATE:
      return("DEVICESTATUS__AUTOMATIC_FULL_UPDATE");
   case DEVICESTATUS__UNKNOWN:
      return("DEVICESTATUS__UNKNOWN");
   default:
      return("DEVICESTATUS__UNKNOWN");
   }
}

enum UpdateVariant
{
   UPDATE_VARIANT__FULL_UPDATE,
   UPDATE_VARIANT__UPDATE,
   UPDATE_VARIANT__UNKNOWN
};

inline ::std::string toString(UpdateVariant updateVariant)
{
   switch (updateVariant)
   {
   case UPDATE_VARIANT__FULL_UPDATE:
      return("UPDATE_VARIANT__FULL_UPDATE");
   case UPDATE_VARIANT__UPDATE:
      return("UPDATE_VARIANT__UPDATE");
   case UPDATE_VARIANT__UNKNOWN:
      return("UPDATE_VARIANT__UNKNOWN");
   default:
      return("UPDATE_VARIANT__UNKNOWN");
   }
}

enum UpdateType
{
   UPDATE_TYPE__NORMAL,
   UPDATE_TYPE__LIGHTWEIGHT,
   UPDATE_TYPE__VIRGIN,
   UPDATE_TYPE__RECOVERY,
   UPDATE_TYPE__UNKNOWN
};

inline ::std::string toString(UpdateType updateType)
{
   switch (updateType)
   {
   case UPDATE_TYPE__NORMAL:
      return("UPDATE_TYPE__NORMAL");
   case UPDATE_TYPE__LIGHTWEIGHT:
      return("UPDATE_TYPE__LIGHTWEIGHT");
   case UPDATE_TYPE__VIRGIN:
      return("UPDATE_TYPE__VIRGIN");
   case UPDATE_TYPE__RECOVERY:
      return("UPDATE_TYPE__RECOVERY");
   case UPDATE_TYPE__UNKNOWN:
      return("UPDATE_TYPE__UNKNOWN");
   default:
      return("UPDATE_TYPE__UNKNOWN");
   }
}

enum UpdateResult
{
   UPDATE_RESULT__SUCCEEDED,
   UPDATE_RESULT__FAILED,
   UPDATE_RESULT__UNKNOWN
};

inline ::std::string toString(UpdateResult updateResult)
{
   switch (updateResult)
   {
   case UPDATE_RESULT__SUCCEEDED:
      return("UPDATE_RESULT__SUCCEEDED");
   case UPDATE_RESULT__FAILED:
      return("UPDATE_RESULT__FAILED");
   case UPDATE_RESULT__UNKNOWN:
      return("UPDATE_RESULT__UNKNOWN");
   default:
      return("UPDATE_RESULT__UNKNOWN");
   }
}

enum UpdateDeclineReason
{
   UPDATE_DECLINE_REASON__NO,
   UPDATE_DECLINE_REASON__LATER,
   UPDATE_DECLINE_REASON__UNKNOWN
};

inline ::std::string toString(UpdateDeclineReason updateDeclineReason)
{
   switch (updateDeclineReason)
   {
   case UPDATE_DECLINE_REASON__NO:
      return("UPDATE_DECLINE_REASON__NO");
   case UPDATE_DECLINE_REASON__LATER:
      return("UPDATE_DECLINE_REASON__LATER");
   case UPDATE_DECLINE_REASON__UNKNOWN:
      return("UPDATE_DECLINE_REASON__UNKNOWN");
   default:
      ::std::stringstream stream;
      stream << " UnMapped UpdateDeclineReason - " << updateDeclineReason << ::std::endl;
      return stream.str();
   }
}

class VersionDescription
{
public:
   VersionDescription() : m_versionId(0), m_versionDate(0), m_versionName()
   {
   }

   VersionDescription(uint32_t versionId, uint64_t versionDate, ::std::string versionName)
      : m_versionId(versionId), m_versionDate(versionDate), m_versionName(versionName)
   {
   }

   uint64_t getVersionDate() const
   {
      return m_versionDate;
   }

   void setVersionDate(uint64_t versionDate)
   {
      m_versionDate = versionDate;
   }

   uint32_t getVersionId() const
   {
      return m_versionId;
   }

   void setVersionId(uint32_t versionId)
   {
      m_versionId = versionId;
   }

   const ::std::string& getVersionName() const
   {
      return m_versionName;
   }

   void setVersionName(const ::std::string& versionName)
   {
      m_versionName = versionName;
   }
   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "VersionDescription:" << ::std::endl;
      stream << "VersionId   " << m_versionId   << ", "
             << "VersionDate " << m_versionDate << ", "
             << "VersionName " << m_versionName << ::std::endl;
      return stream.str();
   }

private:
   uint32_t      m_versionId;
   uint64_t      m_versionDate; //unix timestamp (January 1st, 1970 at UTC)
   ::std::string m_versionName;
};

class NavDataUpdateOverviewInfo
{
public:
   NavDataUpdateOverviewInfo()
      : m_estimatedTime(0)
      , m_dataSize(0)
      , m_totalNumberOfUpdates(0)
      , m_selectedNumberOfUpdates(0)
   {
   }

   NavDataUpdateOverviewInfo(const ::std::string& currentVersion,
                             const ::std::string& newVersion,
                             const ::std::string& updateableAreaName,
                             uint32_t estimatedTime,
                             uint64_t dataSize,
                             uint32_t totalUpdates,
                             uint32_t selectedUpdates,
                             const ValidValue<bool>& requiresFullDataUpdate)
      : m_currentVersion(currentVersion)
      , m_newVersion(newVersion)
      , m_updateableAreaName(updateableAreaName)
      , m_estimatedTime(estimatedTime)
      , m_dataSize(dataSize)
      , m_totalNumberOfUpdates(totalUpdates)
      , m_selectedNumberOfUpdates(selectedUpdates)
      , m_requiresFullDataUpdate(requiresFullDataUpdate)
   {
   }

   void setCurrentVersion(const ::std::string& currentVersion)
   {
      m_currentVersion = currentVersion;
   }

   const ::std::string& getCurrentVersion() const
   {
      return m_currentVersion;
   }

   void setNewVersion(const ::std::string& newVersion)
   {
      m_newVersion = newVersion;
   }

   const ::std::string& getNewVersion() const
   {
      return m_newVersion;
   }

   void setUpdateableAreaName(const ::std::string& updateableAreaName)
   {
      m_updateableAreaName = updateableAreaName;
   }

   const ::std::string& getUpdateableAreaName() const
   {
      return m_updateableAreaName;
   }

   void setEstimatedTime(uint32_t estimatedTime)
   {
      m_estimatedTime = estimatedTime;
   }

   uint32_t getEstimatedTime() const
   {
      return m_estimatedTime;
   }

   void setDataSize(uint64_t dataSize)
   {
      m_dataSize = dataSize;
   }

   uint64_t getDataSize() const
   {
      return m_dataSize;
   }

   void setTotalNumberOfUpdates(uint32_t totalUpdates)
   {
      m_totalNumberOfUpdates = totalUpdates;
   }

   uint32_t getTotalNumberOfUpdates() const
   {
      return m_totalNumberOfUpdates;
   }

   void setSelectedNumberOfUpdates(uint32_t selectedUpdates)
   {
      m_selectedNumberOfUpdates = selectedUpdates;
   }

   uint32_t getSelectedNumberOfUpdates() const
   {
      return m_selectedNumberOfUpdates;
   }
   
   void setRequiresFullDataUpdate(const ValidValue<bool>& requiresFullDataUpdate)
   {
      m_requiresFullDataUpdate = requiresFullDataUpdate;
   }

   const ValidValue<bool>& getRequiresFullDataUpdate() const
   {
      return m_requiresFullDataUpdate;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataUpdateOverviewInfo payload:" << ::std::endl;
      stream << "Current version            = " << m_currentVersion          << ::std::endl
             << "New version                = " << m_newVersion              << ::std::endl
             << "Updateable Area Name       = " << m_updateableAreaName      << ::std::endl
             << "Estimated time             = " << m_estimatedTime           << ::std::endl
             << "Data size                  = " << m_dataSize                << ::std::endl
             << "Total number of updates    = " << m_totalNumberOfUpdates    << ::std::endl
             << "Selected number of updates = " << m_selectedNumberOfUpdates << ::std::endl
             << "Requires full data update  = " << (m_requiresFullDataUpdate.isValid() ? 
                   (m_requiresFullDataUpdate.getValue() ? "true" : "false") : "unset") << std::endl;
      return stream.str();
   }

private:
   ::std::string m_currentVersion;
   ::std::string m_newVersion;
   ::std::string m_updateableAreaName;
   uint32_t      m_estimatedTime; // Measured in seconds
   uint64_t      m_dataSize;      // Measured in bytes
   uint32_t      m_totalNumberOfUpdates;
   uint32_t      m_selectedNumberOfUpdates;
   ValidValue<bool> m_requiresFullDataUpdate; // Supported only in [JP]
};

class NavDataUpdateInfos
{
public:
   /** Inner class representing one single element of the list of available updates. */
   class DataUpdateInfo
   {
   public:
      DataUpdateInfo()
         : m_updateInfoId(0)
         , m_selected(false)
         , m_downloadSize(0)
         , m_installSize(0)
         , m_isDownloaded(false)
      {}

      DataUpdateInfo(uint32_t updateInfoId, const ::std::string& productName, const ::std::string& regionName,
                     const ::std::string& versionName, const ::std::string& versionDateTime,
                     const ::std::string& description, bool selected,
                     uint64_t downloadSize, uint64_t installSize, bool isDownloaded,
                     const ValidValue<bool>& requiresFullDataUpdate)
         : m_updateInfoId(updateInfoId)
         , m_productName(productName)
         , m_regionName(regionName)
         , m_versionName(versionName)
         , m_versionDateTime(versionDateTime)
         , m_description(description)
         , m_selected(selected)
         , m_downloadSize(downloadSize)
         , m_installSize(installSize)
         , m_isDownloaded(isDownloaded)
         , m_requiresFullDataUpdate(requiresFullDataUpdate)
      {}
      virtual ~DataUpdateInfo() {}

      void setSelection(bool selected)
      {
         m_selected = selected;
      }

      bool getSelection() const
      {
         return m_selected;
      }

      uint32_t getId() const
      {
         return m_updateInfoId;
      }

      const ::std::string& getDescription() const
      {
         return m_description;
      }

      uint64_t getDownloadSize() const
      {
         return m_downloadSize;
      }

      uint64_t getInstallSize() const
      {
         return m_installSize;
      }

      bool getIsDownloaded() const
      {
         return m_isDownloaded;
      }
      
      void setRequiresFullDataUpdate(const ValidValue<bool>& requiresFullDataUpdate)
      {
         m_requiresFullDataUpdate = requiresFullDataUpdate;
      }

      const ValidValue<bool>& getRequiresFullDataUpdate() const
      {
         return m_requiresFullDataUpdate;
      }

      virtual ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "DataUpdateInfo payload:" << ::std::endl
                << "Update Info ID        = " << m_updateInfoId    << ::std::endl
                << "Product name          = " << m_productName     << ::std::endl
                << "Region name           = " << m_regionName      << ::std::endl
                << "Version name          = " << m_versionName     << ::std::endl
                << "Version date and time = " << m_versionDateTime << ::std::endl
                << "Display name          = " << m_description     << ::std::endl
                << "Download size         = " << m_downloadSize    << ::std::endl
                << "Install size          = " << m_installSize     << ::std::endl
                << "Already downloaded    = " << m_isDownloaded    << ::std::endl
                << "Selected              = " << m_selected        << ::std::endl
                << "Requires full update  = " << (m_requiresFullDataUpdate.isValid() ? 
                        (m_requiresFullDataUpdate.getValue() ? "true" : "false") : "unset") << std::endl;
         return stream.str();
      }

   private:
      uint32_t      m_updateInfoId;
      ::std::string m_productName;
      ::std::string m_regionName;
      ::std::string m_versionName;
      ::std::string m_versionDateTime;
      ::std::string m_description;
      bool          m_selected;
      uint64_t      m_downloadSize;
      uint64_t      m_installSize;
      bool          m_isDownloaded;
      ValidValue<bool> m_requiresFullDataUpdate; // supported only in [JP]
   };

   NavDataUpdateInfos()
      : m_updateInfoList()
      , m_numberOfAvailableUpdates(0)
   {}

   NavDataUpdateInfos(const ::std::vector<DataUpdateInfo>& updateInfoList,
                      uint32_t numberOfAvailableUpdates)
      : m_updateInfoList(updateInfoList)
      , m_numberOfAvailableUpdates(numberOfAvailableUpdates)
   {}

   /** Returns the total number of stored elements in the list of update informations */
   uint32_t getNbrOfAllAvailableUpdateInfos() const
   {
      return m_numberOfAvailableUpdates ;
   }

   /** Returns the list of stored elements in range. */
   const ::std::vector<DataUpdateInfo>& getUpdateInfoList() const
   {
      return m_updateInfoList;
   }

   void setNumberOfAvailableUpdates(const uint32_t numberOfAvailableUpdates)
   {
      m_numberOfAvailableUpdates = numberOfAvailableUpdates;
   }

   void setUpdateInfoList(const ::std::vector< ::navmiddleware::NavDataUpdateInfos::DataUpdateInfo >& updateInfoList)
   {
      m_updateInfoList = updateInfoList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataUpdateInfos payload:" << ::std::endl;
      for(unsigned int dataUpdateInfoListIndex = 0; dataUpdateInfoListIndex < m_updateInfoList.size(); dataUpdateInfoListIndex++)
      {
         stream << "NavDataUpdateInfos : " << m_updateInfoList.at(dataUpdateInfoListIndex).toString() << ::std::endl;
      }

      return stream.str();
   }

private:
   ::std::vector<DataUpdateInfo>             m_updateInfoList;
   uint32_t                                  m_numberOfAvailableUpdates;
};


class NavDataDeviceInfo
{
public:

   NavDataDeviceInfo()
      : m_deviceId(0)
      , m_deviceType(DEVICETYPE__UNKNOWN)
      , m_deviceStatus(DEVICESTATUS__UNKNOWN)
   {
   }

   NavDataDeviceInfo(uint32_t deviceId, DeviceType deviceType, DeviceStatus deviceStatus):
      m_deviceId(deviceId),
      m_deviceType(deviceType),
      m_deviceStatus(deviceStatus)
   {
   }

   uint32_t getDeviceId() const
   {
      return m_deviceId;
   }

   DeviceType getDeviceType() const
   {
      return m_deviceType;
   }

   DeviceStatus getDeviceStatus() const
   {
      return m_deviceStatus;
   }

   void setDeviceId( uint32_t deviceId)
   {
      m_deviceId = deviceId;
   }

   void setDeviceType(DeviceType deviceType)
   {
      m_deviceType = deviceType;
   }

   void setDeviceStatus(DeviceStatus deviceStatus)
   {
      m_deviceStatus = deviceStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataDeviceInfo payload:" << ::std::endl;
      stream << "Device Id     = " << m_deviceId     << ::std::endl
             << "Device Type   = " << ::navmiddleware::toString(m_deviceType)   << ::std::endl
             << "Device Status = " << ::navmiddleware::toString(m_deviceStatus) << ::std::endl;
      return stream.str();
   }

private:
   uint32_t     m_deviceId;
   DeviceType   m_deviceType;
   DeviceStatus m_deviceStatus;
};

class NavDataDeviceInfos
{
public:
   NavDataDeviceInfos()
   {}

   explicit NavDataDeviceInfos(const ::std::vector<NavDataDeviceInfo>& deviceStatusInfoList)
      : m_deviceStatusInfoList(deviceStatusInfoList)
   {}
   const ::std::vector<NavDataDeviceInfo>& getNavDataDeviceInfos() const
   {
      return m_deviceStatusInfoList;
   }

   ::std::vector<NavDataDeviceInfo>& getNavDataDeviceInfosMutable()
   {
      return m_deviceStatusInfoList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataDeviceInfos payload:" << ::std::endl;
      for(unsigned int index = 0; index < m_deviceStatusInfoList.size(); index++)
      {
         stream << "NavDataDeviceInfo list Index = " << index << ::std::endl
                << "NavDataDeviceInfo Info =" << m_deviceStatusInfoList[index].toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<NavDataDeviceInfo> m_deviceStatusInfoList;
};

/** Informs about the status of the currently active update process. This includes getting information about available
 * updates, downloading and installing selected update(s).
 * In case of status  DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS m_progress contains the progress of getting the
 * available updates in percent.
 * In case of status DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS, DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS m_progress
 * contains the percentual progress and m_remainingSeconds contains the estimated number of seconds remaining until 100%
 * is completed. m_remainingSeconds is set only if available/applicable.
 * m_productName and m_regionName contains the related names of currently downloaded or installed updates. In case name
 * is not available or applicable it is set to empty string.
 */
class NavDataUpdateStatus
{
public:
   enum DataUpdateStatus
   {
      DATAUPDATESTATUS__IDLE,
      DATAUPDATESTATUS__AVAILABLE_UPDATES_STARTED,
      DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS,
      DATAUPDATESTATUS__AVAILABLE_UPDATES_FINISHED,
      DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR,
      DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER,
      DATAUPDATESTATUS__DOWNLOAD_STARTED,
      DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS,
      DATAUPDATESTATUS__DOWNLOAD_STOPPED,
      DATAUPDATESTATUS__DOWNLOAD_FINISHED,
      DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR,
      DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_USER,
      DATAUPDATESTATUS__INSTALLATION_STARTED,
      DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS,
      DATAUPDATESTATUS__INSTALLATION_STOPPED,
      DATAUPDATESTATUS__INSTALLATION_FINISHED,
      DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER,
      DATAUPDATESTATUS__INSTALLATION_BACKGROUND_STARTED,
      DATAUPDATESTATUS__INSTALLATION_BACKGROUND_FINISHED,
      DATAUPDATESTATUS__INSTALLATION_FOREGROUND_STARTED,
      DATAUPDATESTATUS__INSTALLATION_FOREGROUND_FINISHED,
      DATAUPDATESTATUS__UPDATE_ABORTED_BY_USER,
      DATAUPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED,
      DATAUPDATESTATUS__USB_REMOVED,
      DATAUPDATESTATUS__UPDATE_SOURCE_ERROR,
      DATAUPDATESTATUS__FULL_UPDATE_REQUIRED,
      DATAUPDATESTATUS__OUT_OF_SPACE,
      DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE,
      DATAUPDATESTATUS__SUBSCRIPTION_EXPIRED, /** date is expired */
      DATAUPDATESTATUS__SUBSCRIPTION_VALIDATION_IMPOSSIBLE, /** date is not available / missing connection */
      DATAUPDATESTATUS__ABNORMAL_TERMINATION, /** Any abnormal failure in case of AvailableUpdates, Download, Installation */
      DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE, /** The update package does not match the compatibility with current navigation software */
      DATAUPDATESTATUS__COMPATIBILITY_MATRIX_NOT_FOUND, /** Compatibility matrix is not available to check data compatibility with current navigation SW */
      DATAUPDATESTATUS__COMPATIBILITY_MATRIX_OUTDATED, /** Compatibility matrix is not applicable, compatibility can't be evaluated */
      DATAUPDATESTATUS__MAP_DATA_NOT_COMPATIBLE, /** New Map data is not compatible with installed Map */
      DATAUPDATESTATUS__UNKNOWN
   };

   ::std::string dataUpdateStatusToString(NavDataUpdateStatus::DataUpdateStatus dataUpdateStatus) const
   {
      switch(dataUpdateStatus)
      {
      case NavDataUpdateStatus::DATAUPDATESTATUS__IDLE:
         return "DATAUPDATESTATUS__IDLE";
      case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_STARTED:
         return "DATAUPDATESTATUS__AVAILABLE_UPDATES_STARTED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS:
         return "DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS";
      case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_FINISHED:
         return "DATAUPDATESTATUS__AVAILABLE_UPDATES_FINISHED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR:
         return "DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR";
      case NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER:
         return "DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STARTED:
         return "DATAUPDATESTATUS__DOWNLOAD_STARTED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS:
         return "DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STOPPED:
         return "DATAUPDATESTATUS__DOWNLOAD_STOPPED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_FINISHED:
         return "DATAUPDATESTATUS__DOWNLOAD_FINISHED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR:
         return "DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_USER:
         return "DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_USER";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STARTED:
         return "DATAUPDATESTATUS__INSTALLATION_STARTED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS:
         return "DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STOPPED:
         return "DATAUPDATESTATUS__INSTALLATION_STOPPED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FINISHED:
         return "DATAUPDATESTATUS__INSTALLATION_FINISHED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER:
         return "DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_BACKGROUND_STARTED:
         return "DATAUPDATESTATUS__INSTALLATION_BACKGROUND_STARTED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_BACKGROUND_FINISHED:
         return "DATAUPDATESTATUS__INSTALLATION_BACKGROUND_FINISHED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FOREGROUND_STARTED:
         return "DATAUPDATESTATUS__INSTALLATION_FOREGROUND_STARTED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FOREGROUND_FINISHED:
         return "DATAUPDATESTATUS__INSTALLATION_FOREGROUND_FINISHED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_ABORTED_BY_USER:
         return "DATAUPDATESTATUS__UPDATE_ABORTED_BY_USER";
      case NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED:
         return "DATAUPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__USB_REMOVED:
         return "DATAUPDATESTATUS__USB_REMOVED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_SOURCE_ERROR:
         return "DATAUPDATESTATUS__UPDATE_SOURCE_ERROR";
      case NavDataUpdateStatus::DATAUPDATESTATUS__FULL_UPDATE_REQUIRED:
         return "DATAUPDATESTATUS__FULL_UPDATE_REQUIRED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__OUT_OF_SPACE:
         return "DATAUPDATESTATUS__OUT_OF_SPACE";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE:
         return "DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE";
      case NavDataUpdateStatus::DATAUPDATESTATUS__SUBSCRIPTION_EXPIRED:
         return "DATAUPDATESTATUS__SUBSCRIPTION_EXPIRED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__SUBSCRIPTION_VALIDATION_IMPOSSIBLE:
         return "DATAUPDATESTATUS__SUBSCRIPTION_VALIDATION_IMPOSSIBLE";
      case NavDataUpdateStatus::DATAUPDATESTATUS__ABNORMAL_TERMINATION:
         return "DATAUPDATESTATUS__ABNORMAL_TERMINATION";
      case NavDataUpdateStatus::DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE:
         return "DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE";
      case NavDataUpdateStatus::DATAUPDATESTATUS__COMPATIBILITY_MATRIX_NOT_FOUND:
         return "DATAUPDATESTATUS__COMPATIBILITY_MATRIX_NOT_FOUND";
      case NavDataUpdateStatus::DATAUPDATESTATUS__COMPATIBILITY_MATRIX_OUTDATED:
         return "DATAUPDATESTATUS__COMPATIBILITY_MATRIX_OUTDATED";
      case NavDataUpdateStatus::DATAUPDATESTATUS__MAP_DATA_NOT_COMPATIBLE:
         return "DATAUPDATESTATUS__MAP_DATA_NOT_COMPATIBLE";
      case NavDataUpdateStatus::DATAUPDATESTATUS__UNKNOWN:
         return "DATAUPDATESTATUS__UNKNOWN";
      default:
         ::std::stringstream stream;
         stream << "Unmapped DATAUPDATESTATUS = " << dataUpdateStatus << ::std::endl;
         return stream.str();
      }
   }

   NavDataUpdateStatus()
      : m_dataUpdateStatus(DATAUPDATESTATUS__IDLE)
      , m_deviceId(0)
      , m_progress(0)
      , m_installPhaseProgress(0)
      , m_remainingSeconds()
      , m_productName()
      , m_regionName()
      , m_productId(0)
      , m_supplierId(0)
      , m_regionId(0)
      , m_isProductUpdate(false)
      , m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
   {}

   NavDataUpdateStatus(DataUpdateStatus dataUpdateStatus, unsigned char percentualDownloadProgress,
                       const ::navmiddleware::ValidValue<uint32_t>& remainingSeconds,
                       const ::std::string& productName, const ::std::string& regionName )
      : m_dataUpdateStatus(dataUpdateStatus)
      , m_deviceId(0)
      , m_progress(percentualDownloadProgress)
      , m_installPhaseProgress(0)
      , m_remainingSeconds(remainingSeconds)
      , m_productName(productName)
      , m_regionName(regionName)
      , m_productId(0)
      , m_supplierId(0)
      , m_regionId(0)
      , m_isProductUpdate(false)
      , m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
   {}

   NavDataUpdateStatus(DataUpdateStatus dataUpdateStatus, unsigned char percentualDownloadProgress,
                       const ::navmiddleware::ValidValue<uint32_t>& remainingSeconds,
                       const ::std::string& productName, const ::std::string& regionName,
                       uint32_t productId, uint32_t supplierId, uint32_t regionId, bool isProduct)
      : m_dataUpdateStatus(dataUpdateStatus)
      , m_deviceId(0)
      , m_progress(percentualDownloadProgress)
      , m_installPhaseProgress(0)
      , m_remainingSeconds(remainingSeconds)
      , m_productName(productName)
      , m_regionName(regionName)
      , m_productId(productId)
      , m_supplierId(supplierId)
      , m_regionId(regionId)
      , m_isProductUpdate(isProduct)
      , m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
   {}

   NavDataUpdateStatus(DataUpdateStatus dataUpdateStatus, unsigned char percentualProgress,
                       unsigned char percentualInstallPhaseProgress,
                       const ::navmiddleware::ValidValue<uint32_t>& estimatedInstallPhaseTime,
                       const ::navmiddleware::ValidValue<uint32_t>& estimatedTotalInstallTime,
                       const ::navmiddleware::ValidValue<uint32_t>& remainingSeconds,
                       const ::std::string& productName, const ::std::string& regionName,
                       uint32_t productId, uint32_t supplierId, uint32_t regionId, bool isProduct)
      : m_dataUpdateStatus(dataUpdateStatus)
      , m_deviceId(0)
      , m_progress(percentualProgress)
      , m_installPhaseProgress(percentualInstallPhaseProgress)
      , m_estimatedTotalInstallTime(estimatedTotalInstallTime)
      , m_estimatedInstallPhaseTime(estimatedInstallPhaseTime)
      , m_remainingSeconds(remainingSeconds)
      , m_productName(productName)
      , m_regionName(regionName)
      , m_productId(productId)
      , m_supplierId(supplierId)
      , m_regionId(regionId)
      , m_isProductUpdate(isProduct)
      , m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
   {}

   NavDataUpdateStatus(DataUpdateStatus dataUpdateStatus, uint32_t deviceId, unsigned char percentualProgress,
                       unsigned char percentualInstallPhaseProgress,
                       const ::navmiddleware::ValidValue<uint32_t>& estimatedInstallPhaseTime,
                       const ::navmiddleware::ValidValue<uint32_t>& estimatedTotalInstallTime,
                       const ::navmiddleware::ValidValue<uint32_t>& remainingSeconds,
                       const ::std::string& productName, const ::std::string& regionName,
                       uint32_t productId, uint32_t supplierId, uint32_t regionId, bool isProduct)
      : m_dataUpdateStatus(dataUpdateStatus)
      , m_deviceId(deviceId)
      , m_progress(percentualProgress)
      , m_installPhaseProgress(percentualInstallPhaseProgress)
      , m_estimatedTotalInstallTime(estimatedTotalInstallTime)
      , m_estimatedInstallPhaseTime(estimatedInstallPhaseTime)
      , m_remainingSeconds(remainingSeconds)
      , m_productName(productName)
      , m_regionName(regionName)
      , m_productId(productId)
      , m_supplierId(supplierId)
      , m_regionId(regionId)
      , m_isProductUpdate(isProduct)
      , m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
   {}

   DataUpdateStatus getDataUpdateStatus() const
   {
      return( m_dataUpdateStatus );
   }

   uint32_t getDeviceId() const
   {
      return( m_deviceId );
   }

   void setDeviceId(uint32_t deviceId)
   {
      m_deviceId = deviceId;
   }

   uint8_t getProgress() const
   {
      return( m_progress );
   }

   const ValidValue<uint32_t>& getEstimatedInstallPhaseTime() const
   {
      return m_estimatedInstallPhaseTime;
   }

   void setEstimatedInstallPhaseTime(const ValidValue<uint32_t>& estimatedInstallPhaseTime)
   {
      m_estimatedInstallPhaseTime = estimatedInstallPhaseTime;
   }

   const ValidValue<uint32_t>& getEstimatedTotalInstallTime() const
   {
      return m_estimatedTotalInstallTime;
   }

   void setEstimatedTotalInstallTime(const ValidValue<uint32_t>& estimatedTotalInstallTime)
   {
      m_estimatedTotalInstallTime = estimatedTotalInstallTime;
   }

   uint8_t getInstallPhaseProgress() const
   {
      return m_installPhaseProgress;
   }

   void setInstallPhaseProgress(uint8_t installPhaseProgress)
   {
      m_installPhaseProgress = installPhaseProgress;
   }

   const ValidValue<uint32_t>& getRemainingTimeInSeconds() const
   {
      return( m_remainingSeconds );
   }

   const ::std::string& getUpdateProductName() const
   {
      return( m_productName );
   }

   const ::std::string& getUpdateRegionName() const
   {
      return( m_regionName );
   }

   bool isProductUpdate() const
   {
      return m_isProductUpdate;
   }

   uint32_t getProductId() const
   {
      return m_productId;
   }

   uint32_t getRegionId() const
   {
      return m_regionId;
   }

   uint32_t getSupplierId() const
   {
      return m_supplierId;
   }

   ScomoComponent getScomoComponent() const
   {
      return m_scomoComponent;
   }

   void setScomoComponent(ScomoComponent scomoComponent)
   {
      m_scomoComponent = scomoComponent;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataUpdateStatus payload:" << ::std::endl;
      stream << "Data update status              = " << dataUpdateStatusToString(m_dataUpdateStatus) << ::std::endl
             << "DeviceId                        = " << m_deviceId         << ::std::endl
             << "Download/Install progress value = " << m_progress         << ::std::endl
             << "Install phase progress value    = " << m_installPhaseProgress  << ::std::endl
             << "Estimated total install time    = " << m_estimatedTotalInstallTime.toString() << ::std::endl
             << "Estimated phase install time    = " << m_estimatedInstallPhaseTime.toString() << ::std::endl
             << "Remaining time in seconds       = " << m_remainingSeconds.toString() << ::std::endl
             << "ProductName                     = " << m_productName      << ::std::endl
             << "RegionName                      = " << m_regionName       << ::std::endl
             << "ProductId                       = " << m_productId        << ::std::endl
             << "SupplierId                      = " << m_supplierId       << ::std::endl
             << "RegionId                        = " << m_regionId         << ::std::endl
             << "IsProductUpdate                 = " << (m_isProductUpdate? "true":"false") << ::std::endl
             << "Scomo component                 = " << ScomoComponentToString(m_scomoComponent) << ::std::endl;

      return stream.str();
   }

private:
   DataUpdateStatus      m_dataUpdateStatus;
   uint32_t              m_deviceId;
   uint8_t               m_progress;
   uint8_t               m_installPhaseProgress;
   ValidValue<uint32_t>  m_estimatedTotalInstallTime;
   ValidValue<uint32_t>  m_estimatedInstallPhaseTime;
   ValidValue<uint32_t>  m_remainingSeconds;
   ::std::string         m_productName;
   ::std::string         m_regionName;
   uint32_t              m_productId;
   uint32_t              m_supplierId;
   uint32_t              m_regionId;
   bool                  m_isProductUpdate;
   ScomoComponent        m_scomoComponent;
};

class NavDataCompatibilityInfo
{
public:
   enum CompatibilityState
   {
      COMPATIBILITYSTATE__COMPATIBLE =1,
      COMPATIBILITYSTATE__INCOMPATIBLE,
      COMPATIBILITYSTATE__UNKNOWN
   };

   NavDataCompatibilityInfo() : m_compatibilityState(COMPATIBILITYSTATE__UNKNOWN)
   {}

   explicit NavDataCompatibilityInfo(CompatibilityState compatibilitystate) : m_compatibilityState(compatibilitystate)
   {}

   CompatibilityState getCompatibilityState() const
   {
      return( m_compatibilityState );
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataCompatibilityInfo payload:" << ::std::endl;
      stream << "compatibility is = ";
      switch(m_compatibilityState)
      {
      case COMPATIBILITYSTATE__COMPATIBLE:
         stream << "COMPATIBILITYSTATE__COMPATIBLE" << ::std::endl;
         break;
      case COMPATIBILITYSTATE__INCOMPATIBLE:
         stream << "COMPATIBILITYSTATE__INCOMPATIBLE" << ::std::endl;
         break;
      default:
         stream << "COMPATIBILITYSTATE__UNKNOWN" << ::std::endl;
         break;
      }

      return stream.str();
   }

private:
   CompatibilityState m_compatibilityState;
};

class NavDataVersionInfo
{
public:
   NavDataVersionInfo()
   {}

   explicit NavDataVersionInfo(const ::std::string& currentVersion)
      : m_currentVersion(currentVersion)
   {}

   virtual ~NavDataVersionInfo() {}

   const ::std::string& getCurrentVersion() const
   {
      return m_currentVersion;
   }

   void setCurrentVersion(const ::std::string& currentVersion)
   {
      m_currentVersion = currentVersion;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataVersionInfo payload:" << ::std::endl;
      stream << "current version = " << m_currentVersion << ::std::endl;
      return stream.str();
   }

private:
   ::std::string m_currentVersion;
};

class NavDataUpdateSelection
{
public:
   NavDataUpdateSelection()
      : m_totalNumberOfUpdates(0)
      , m_selectedNumberOfUpdates(0)
   {}

   NavDataUpdateSelection(uint32_t totalNumberOfUpdates,
                          uint32_t selectedNumberOfUpdates)
      : m_totalNumberOfUpdates(totalNumberOfUpdates)
      , m_selectedNumberOfUpdates(selectedNumberOfUpdates)
   {}

   virtual ~NavDataUpdateSelection() {}

   uint32_t getTotalNumberOfUpdates() const
   {
      return m_totalNumberOfUpdates;
   }

   uint32_t getSelectedNumberOfUpdates() const
   {
      return m_selectedNumberOfUpdates;
   }

   void setTotalNumberOfUpdates(uint32_t totalNumberOfUpdates)
   {
      m_totalNumberOfUpdates = totalNumberOfUpdates;
   }

   void setSelectedNumberOfUpdates(uint32_t selectedNumberOfUpdates)
   {
      m_selectedNumberOfUpdates = selectedNumberOfUpdates;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataUpdateSelection payload:" << ::std::endl;
      stream << "totalNumberOfUpdates = " << m_totalNumberOfUpdates << ::std::endl;
      stream << "selectedNumberOfUpdates = " << m_selectedNumberOfUpdates << ::std::endl;
      return stream.str();
   }

private:
   uint32_t m_totalNumberOfUpdates;
   uint32_t m_selectedNumberOfUpdates;
};

// SOTA related definitions

class ScomoInstallComponentResult
{
public:
   enum ScomoInstallResult
   {
      SCOMO_INSTALL_RESULT__SUCCESS,
      SCOMO_INSTALL_RESULT__COMPONENT_UNKNOWN,
      SCOMO_INSTALL_RESULT__PATH_INVALID,
      SCOMO_INSTALL_RESULT__UNSUPPORTED_UPDATE_DEVICE,
      SCOMO_INSTALL_RESULT__FAILED_BY_BUSY_CONDITION,
      SCOMO_INSTALL_RESULT__REQUIRED_RESOURCES_NOT_AVAILABLE,
      SCOMO_INSTALL_RESULT__ABORTED_BY_REQUEST,
      SCOMO_INSTALL_RESULT__NO_UPDATES_FOUND,
      SCOMO_INSTALL_RESULT__UPDATE_PACKAGE_INCONSISTENT,
      SCOMO_INSTALL_RESULT__CHECKSUM_VALIDATION_ERROR,
      SCOMO_INSTALL_RESULT__CATALOG_NOT_FOUND,
      SCOMO_INSTALL_RESULT__OUT_OF_SPACE,
      SCOMO_INSTALL_RESULT__INSTALLATION_NOT_EXECUTED,
      SCOMO_INSTALL_RESULT__SUBSCRIPTION_EXPIRED,
      SCOMO_INSTALL_RESULT__INCOMPATIBLE_UPDATE_PACKAGE,
      SCOMO_INSTALL_RESULT__REBOOT_REQUIRED,
      SCOMO_INSTALL_RESULT__COMPATIBILITY_MATRIX_NOT_FOUND,
      SCOMO_INSTALL_RESULT__INSTALLATION_RESULT_NOT_AVAILABLE,
      SCOMO_INSTALL_RESULT__ACTIVATION_ON_SHUTDOWN_COMPLETED,
      SCOMO_INSTALL_RESULT__UNKNOWN
   };

   ScomoInstallComponentResult()
      : m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
      , m_scomoComponentVersion()
      , m_result(SCOMO_INSTALL_RESULT__UNKNOWN)
      , m_installStage(NAVDATAUPDATE_INSTALL_STAGE__UNKNOWN)
      , m_updateRegionName()
   {}

   ~ScomoInstallComponentResult()
   {}

   NavDataUpdateInstallStage getInstallStage() const
   {
      return m_installStage;
   }

   ScomoInstallResult getResult() const
   {
      return m_result;
   }

   ScomoComponent getScomoComponent() const
   {
      return m_scomoComponent;
   }

   const ::std::string& getScomoComponentVersion() const
   {
      return m_scomoComponentVersion;
   }

   const ::std::string& getUpdateRegionName() const
   {
      return m_updateRegionName;
   }

   void setInstallStage(NavDataUpdateInstallStage installStage)
   {
      m_installStage = installStage;
   }

   void setResult(ScomoInstallResult result)
   {
      m_result = result;
   }

   void setScomoComponent(ScomoComponent scomoComponent)
   {
      m_scomoComponent = scomoComponent;
   }

   void setScomoComponentVersion(const ::std::string& scomoComponentVersion)
   {
      m_scomoComponentVersion = scomoComponentVersion;
   }

   void setUpdateRegionName(const ::std::string& updateRegionName)
   {
      m_updateRegionName = updateRegionName;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ScomoInstallComponentResult payload:" << ::std::endl;
      stream << "m_scomoComponent: " << ScomoComponentToString(m_scomoComponent) << ::std::endl;
      stream << "m_scomoComponentVersion: " << m_scomoComponentVersion << ::std::endl;
      stream << "m_result = " << m_result << ", ";
      stream << "m_installStage = " << m_installStage << ::std::endl;
      stream << "m_updateRegionName: " << m_updateRegionName << ::std::endl;
      return stream.str();
   }

private:
   ScomoComponent            m_scomoComponent;
   ::std::string             m_scomoComponentVersion;
   ScomoInstallResult        m_result;
   NavDataUpdateInstallStage m_installStage;
   ::std::string             m_updateRegionName;
};

class ScomoComponentStatus
{
public:
   struct UpdateComponentStatus
   {
      ::std::string version;
   };

   ScomoComponentStatus()
      : m_scomoComponent(SCOMO_COMPONENT__UNKNOWN)
   {}
   ~ScomoComponentStatus()
   {}

   ScomoComponent getScomoComponent() const
   {
      return m_scomoComponent;
   }

   void setScomoComponent(ScomoComponent scomoComponent)
   {
      m_scomoComponent = scomoComponent;
   }

   const ::std::map< ::std::string, UpdateComponentStatus>& getUpdateComponentStatus() const
   {
      return(m_updateComponentStatus);
   }

   ::std::map< ::std::string, UpdateComponentStatus>& getUpdateComponentStatusMutable()
   {
      return(m_updateComponentStatus);
   }

   void setUpdateComponentStatus(const ::std::map< ::std::string, UpdateComponentStatus>& updateComponentStatus)
   {
      m_updateComponentStatus = updateComponentStatus;
   }

   void insertUpdateComponentStatus(const ::std::string& componentId, const UpdateComponentStatus& updateComponentStatus)
   {
      m_updateComponentStatus.insert(::std::pair< ::std::string, UpdateComponentStatus>(componentId,updateComponentStatus));
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ScomoComponentStatus payload:" << ::std::endl;
      stream << "m_scomoComponent: " << ScomoComponentToString(m_scomoComponent) << ::std::endl;
      for(::std::map< ::std::string, UpdateComponentStatus>::const_iterator it = m_updateComponentStatus.begin();
            it != m_updateComponentStatus.end(); ++it)
      {
         stream << "Component = " << it->first << ", "
                << "VersionInfo =" << it->second.version << ::std::endl;
      }
      return stream.str();
   }

private:
   ScomoComponent m_scomoComponent;
   ::std::map< ::std::string, UpdateComponentStatus> m_updateComponentStatus;
};

class NavDataContext
{
public:
   //inner classes begin
   class StorageSpace
   {
   public:
      StorageSpace()
         : m_spaceTotalInBytes(0)
         , m_spaceLeftInBytes(0)
      {}
      StorageSpace(uint64_t spaceTotalInBytes, uint64_t spaceLeftInBytes)
         : m_spaceTotalInBytes(spaceTotalInBytes)
         , m_spaceLeftInBytes(spaceLeftInBytes)
      {}
      ~StorageSpace()
      {}

      uint64_t getSpaceLeftInBytes() const
      {
         return m_spaceLeftInBytes;
      }

      void setSpaceLeftInBytes(uint64_t spaceLeftInBytes)
      {
         m_spaceLeftInBytes = spaceLeftInBytes;
      }

      uint64_t getSpaceTotalInBytes() const
      {
         return m_spaceTotalInBytes;
      }

      void setSpaceTotalInBytes(uint64_t spaceTotalInBytes)
      {
         m_spaceTotalInBytes = spaceTotalInBytes;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "StorageSpace payload:" << ::std::endl;
         stream << "m_spaceTotalInBytes " << m_spaceTotalInBytes << ::std::endl
                << "m_spaceLeftInBytes  " << m_spaceLeftInBytes << ::std::endl;
         return stream.str();
      }

   private:
      uint64_t m_spaceTotalInBytes;
      uint64_t m_spaceLeftInBytes;
   };

   class UpdateRegionInfo
   {
   public:
      struct AreaInfo
      {
         ::std::string       m_areaName;
         ValidValue<Image>   m_areaFlag;

         ::std::string toString() const
         {
            ::std::stringstream    stream;
            stream << "m_areaName : " << m_areaName << ::std::endl;
            if(m_areaFlag.isValid())
            {
               stream << "m_areaFlag : " << m_areaFlag.getValue().toString() << ::std::endl;
            }
            else
            {
               stream << "Flag not valid" << ::std::endl;
            }
            return stream.str();
         }
      };
      struct CountryAndSubCountryInfo
      {
         AreaInfo                   m_countryInfo;
         ::std::vector<AreaInfo>    m_subCountryInfos;

         ::std::string toString() const
         {
            ::std::stringstream    stream;
            stream << "m_countryInfo : " << m_countryInfo.toString() << ::std::endl
                   << "m_subCountryInfos : " << ::navmiddleware::toString<AreaInfo>(m_subCountryInfos) << ::std::endl;
            return stream.str();
         }
      };
      UpdateRegionInfo()
         : m_componentId()
         , m_regionId(0)
         , m_regionVersion()
         , m_regionVersionIsDirty(false)
         , m_regionName()
         , m_currentSizeInBytes(0)
         , m_isFavorite(false)
         , m_isSubscriptionActive(false)
         , m_isRemovable(false)
         , m_isInstalled(false)
      {}
      ~UpdateRegionInfo() {}

      const ::std::string& getComponentId() const
      {
         return m_componentId;
      }

      void setComponentId(const ::std::string& componentId)
      {
         m_componentId = componentId;
      }

      uint64_t getCurrentSizeInBytes() const
      {
         return m_currentSizeInBytes;
      }

      void setCurrentSizeInBytes(uint64_t currentSizeInBytes)
      {
         m_currentSizeInBytes = currentSizeInBytes;
      }

      bool isFavorite() const
      {
         return m_isFavorite;
      }

      void setIsFavorite(bool isFavorite)
      {
         m_isFavorite = isFavorite;
      }

      bool isInstalled() const
      {
         return m_isInstalled;
      }

      void setIsInstalled(bool isInstalled)
      {
         m_isInstalled = isInstalled;
      }

      bool isSubscriptionActive() const
      {
         return m_isSubscriptionActive;
      }

      void setIsSubscriptionActive(bool isSubscriptionActive)
      {
         m_isSubscriptionActive = isSubscriptionActive;
      }

      bool isRemovable() const
      {
         return m_isRemovable;
      }

      void setIsRemovable(bool isRemovable)
      {
         m_isRemovable = isRemovable;
      }

      uint32_t getRegionId() const
      {
         return m_regionId;
      }

      void setRegionId(uint32_t regionId)
      {
         m_regionId = regionId;
      }

      const VersionDescription& getRegionVersion() const
      {
         return m_regionVersion;
      }

      void setRegionVersion(const VersionDescription& regionVersion)
      {
         m_regionVersion = regionVersion;
      }

      bool isRegionVersionIsDirty() const
      {
         return m_regionVersionIsDirty;
      }

      void setRegionVersionIsDirty(bool regionVersionIsDirty)
      {
         m_regionVersionIsDirty = regionVersionIsDirty;
      }

      const ::std::string& getRegionName() const
      {
         return m_regionName;
      }

      void setRegionName(const ::std::string& regionName)
      {
         m_regionName = regionName;
      }

      void setCountryAndSubCountryInfo(const ::std::vector<CountryAndSubCountryInfo>& data)
      {
         m_countryAndSubCountryInfos = data;
      }

      const ::std::vector<CountryAndSubCountryInfo>& getCountryAndSubCountryInfo() const
      {
         return m_countryAndSubCountryInfos;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "UpdateRegionInfo payload:" << ::std::endl;
         stream << "m_regionName " << m_regionName << ::std::endl
                << "m_componentId " << m_componentId << ::std::endl
                << "m_regionId " << m_regionId << ::std::endl
                << "m_regionVersion " << m_regionVersion.toString() << ::std::endl
                << "m_regionVersionIsDirty " << (m_regionVersionIsDirty? "true":"false") << ::std::endl
                << "m_currentSizeInBytes   " << m_currentSizeInBytes << ::std::endl
                << "m_isFavorite    " << (m_isFavorite? "true": "false") << ::std::endl
                << "m_isSubscriptionActive " << (m_isSubscriptionActive? "true":"false") << ::std::endl
                << "m_isRemovable   " << (m_isRemovable? "true":"false") << ::std::endl
                << "m_isInstalled   " << (m_isInstalled? "true":"false") << ::std::endl
                << "m_countryAndSubCountryInfos   " << ::navmiddleware::toString<CountryAndSubCountryInfo>(m_countryAndSubCountryInfos) << std::endl;
         return stream.str();
      }
      bool operator<(const UpdateRegionInfo& other) const
      {
         return m_regionId < other.m_regionId;
      }

   private:
      ::std::string                                            m_componentId;
      uint32_t                                                 m_regionId;
      VersionDescription                                       m_regionVersion;
      bool                                                     m_regionVersionIsDirty;
      ::std::string                                            m_regionName;
      uint64_t                                                 m_currentSizeInBytes;
      bool                                                     m_isFavorite;
      bool                                                     m_isSubscriptionActive;
      bool                                                     m_isRemovable;
      bool                                                     m_isInstalled;
      ::std::vector<CountryAndSubCountryInfo>                  m_countryAndSubCountryInfos;
   };

   class RegionSetInfo
   {
   public:
      RegionSetInfo()
         : m_regionSetId(0)
         , m_regionSetName()
      {}
      ~RegionSetInfo()
      {}

      uint32_t getRegionSetId() const
      {
         return m_regionSetId;
      }

      void setRegionSetId(uint32_t regionSetId)
      {
         m_regionSetId = regionSetId;
      }

      const ::std::string& getRegionSetName() const
      {
         return m_regionSetName;
      }

      void setRegionSetName(const ::std::string& regionSetName)
      {
         m_regionSetName = regionSetName;
      }

      const ::std::set<uint32_t>& getUpdateRegionIds() const
      {
         return m_updateRegionIds;
      }

      void setUpdateRegionIds(const ::std::set<uint32_t>& updateRegionIds)
      {
         m_updateRegionIds = updateRegionIds;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "RegionSetInfo payload:" << ::std::endl;
         stream << "m_regionSetId   " << m_regionSetId << ::std::endl
                << "m_regionSetName " << m_regionSetName << ::std::endl
                << "UpdateRegionIds: ";
         uint32_t count=0;
         for (::std::set<uint32_t>::const_iterator it = m_updateRegionIds.begin(); it != m_updateRegionIds.end(); ++it)
         {
            stream << *it << ", ";
            if ( ++count % 10 == 0 )
            {
               stream << ::std::endl;
            }
         }
         return stream.str();
      }

      bool operator<(const RegionSetInfo& other) const
      {
         return m_regionSetId < other.m_regionSetId;
      }

   private:
      uint32_t             m_regionSetId;
      ::std::string        m_regionSetName;
      ::std::set<uint32_t> m_updateRegionIds;
   };

   class ProductInfo
   {
   public:
      ProductInfo()
         : m_componentId()
         , m_productId(0)
         , m_supplierId(0)
         , m_baselineMapId(0)
         , m_productGroupId(0)
         , m_productTypeMask(0)
         , m_productVersion()
         , m_productVersionIsDirty(false)
         , m_productName()
      {}
      ~ProductInfo()
      {}

      uint32_t getBaselineMapId() const
      {
         return m_baselineMapId;
      }

      void setBaselineMapId(uint32_t baselineMapId)
      {
         m_baselineMapId = baselineMapId;
      }

      const ::std::string& getComponentId() const
      {
         return m_componentId;
      }

      void setComponentId(const ::std::string& componentId)
      {
         m_componentId = componentId;
      }

      uint32_t getProductId() const
      {
         return m_productId;
      }

      void setProductId(uint32_t productId)
      {
         m_productId = productId;
      }

      uint16_t getProductGroupId() const
      {
         return m_productGroupId;
      }

      void setProductGroupId(uint16_t productGroupId)
      {
         m_productGroupId = productGroupId;
      }

      /** The product type is a bitwise AND of one or more elements of enum PRoductType. */
      uint32_t getProductTypeMask() const
      {
         return m_productTypeMask;
      }

      void setProductTypeMask(uint32_t productTypeMask)
      {
         m_productTypeMask = productTypeMask;
      }

      const ::std::string& getProductName() const
      {
         return m_productName;
      }

      void setProductName(const ::std::string& productName)
      {
         m_productName = productName;
      }

      const VersionDescription& getProductVersion() const
      {
         return m_productVersion;
      }

      void setProductVersion(const VersionDescription& productVersion)
      {
         m_productVersion = productVersion;
      }

      bool isProductVersionIsDirty() const
      {
         return m_productVersionIsDirty;
      }

      void setProductVersionIsDirty(bool productVersionIsDirty)
      {
         m_productVersionIsDirty = productVersionIsDirty;
      }

      const ::std::vector<RegionSetInfo>& getRegionSets() const
      {
         return m_regionSets;
      }

      void setRegionSets(const ::std::vector<RegionSetInfo>& regionSets)
      {
         m_regionSets = regionSets;
      }

      uint32_t getSupplierId() const
      {
         return m_supplierId;
      }

      void setSupplierId(uint32_t supplierId)
      {
         m_supplierId = supplierId;
      }

      const ::std::vector<UpdateRegionInfo>& getUpdateRegions() const
      {
         return m_updateRegions;
      }

      void setUpdateRegions(const ::std::vector<UpdateRegionInfo>& updateRegions)
      {
         m_updateRegions = updateRegions;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "ProductInfo payload:" << ::std::endl;
         stream << "m_productName " << m_productName << ::std::endl
                << "m_productId " << m_productId << ", "
                << "m_supplierId " << m_supplierId << ", "
                << "m_baselineMapId " << m_baselineMapId << ", "
                << "m_productGroupId " << m_productGroupId << ::std::endl
                << "m_productTypeMask " << m_productTypeMask << ::std::endl
                << "m_productVersion " << m_productVersion.toString()
                << "m_productVersionIsDirty " << (m_productVersionIsDirty? "true" : "false") << ::std::endl
                << "m_componentId " << m_componentId << ::std::endl;
         for (::std::vector<RegionSetInfo>::const_iterator it = m_regionSets.begin(); it != m_regionSets.end(); ++it)
         {
            stream << it->toString() << ::std::endl;
         }
         for (::std::vector<UpdateRegionInfo>::const_iterator it = m_updateRegions.begin(); it != m_updateRegions.end(); ++it)
         {
            stream << it->toString();
         }
         return stream.str();
      }

   private:
      ::std::string                   m_componentId;
      uint32_t                        m_productId;
      uint32_t                        m_supplierId;
      uint32_t                        m_baselineMapId;
      uint16_t                        m_productGroupId;
      uint32_t                        m_productTypeMask; /** The productTypeMask is a bitwise AND of one or more elements of enum ProductType. */
      VersionDescription              m_productVersion;
      bool                            m_productVersionIsDirty;
      ::std::string                   m_productName;
      ::std::vector<RegionSetInfo>    m_regionSets;
      ::std::vector<UpdateRegionInfo> m_updateRegions;
   };
   //inner classes end

   NavDataContext()
   {}
   ~NavDataContext()
   {}

   const ::std::string& getNaviVersionString() const
   {
      return( m_naviVersionString );
   }

   void setNaviVersionString(const ::std::string& naviVersionString)
   {
      m_naviVersionString = naviVersionString;
   }

   const ::std::string& getPackageId() const
   {
      return( m_packageId );
   }

   void setPackageId(const ::std::string& packageId)
   {
      m_packageId = packageId;
   }

   const ::std::vector<ProductInfo>& getProducts() const
   {
      return( m_products );
   }

   void setProducts(const ::std::vector<ProductInfo>& products)
   {
      m_products = products;
   }

   void insertProducts(const ProductInfo& product)
   {
      m_products.push_back(product);
   }

   void setStorageSpace(const StorageSpace& storageSpace)
   {
      m_storageSpace = storageSpace;
   }

   const StorageSpace& getStorageSpace() const
   {
      return( m_storageSpace);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataContext payload:" << ::std::endl;
      stream << "Navi Version " << m_naviVersionString << ::std::endl
             << "Package Id " << m_packageId << ::std::endl
             << m_storageSpace.toString();
      for (::std::vector<ProductInfo>::const_iterator it = m_products.begin();
            it != m_products.end(); ++it)
      {
         stream << it->toString();
      }
      return stream.str();
   }

private:
   ::std::vector<ProductInfo> m_products;
   StorageSpace               m_storageSpace;
   ::std::string              m_naviVersionString;
   ::std::string              m_packageId;
};

class NavDataPartNumber
{
public:
   enum PartNumberResult
   {
      PART_NUMBER_RESULT__SUCCESS = 0,
      PART_NUMBER_RESULT__NO_PART_NUMBER_SET = 1, //there is no part number available
      PART_NUMBER_RESULT__NO_PART_NUMBER_FOR_DEVICE_SET = 2, //there is no part number available for the given device type
      PART_NUMBER_RESULT__DATA_UNAVAILABLE = 3, //no navigation data is available
      PART_NUMBER_RESULT__META_DATA_UNAVAILABLE = 4, //navigation data is available, but no meta data found for part number
      PART_NUMBER_RESULT__DEVICE_UNKNOWN = 5, //the device type is not set
      PART_NUMBER_RESULT__FAILURE = 6, //miscellaneous errors
      PART_NUMBER_RESULT__UNKNOWN = 0xFF
   };

   NavDataPartNumber()
      : m_result(PART_NUMBER_RESULT__UNKNOWN)
      , m_partNumber()
   {}

   ~NavDataPartNumber()
   {}

   const ::std::string& getPartNumber() const
   {
      return m_partNumber;
   }

   void setPartNumber(const ::std::string& partNumber)
   {
      m_partNumber = partNumber;
   }

   PartNumberResult getResult() const
   {
      return m_result;
   }

   void setResult(PartNumberResult result)
   {
      m_result = result;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataPartNumber payload:" << ::std::endl;
      stream << "Result: " << toString(m_result) << ", Part number: " << m_partNumber << ::std::endl;
      return stream.str();
   }

private:
   inline ::std::string toString(PartNumberResult result) const
   {
      switch(result)
      {
      case PART_NUMBER_RESULT__SUCCESS:
         return( "PART_NUMBER_RESULT__SUCCESS" );
      case PART_NUMBER_RESULT__NO_PART_NUMBER_SET:
         return( "PART_NUMBER_RESULT__NO_PART_NUMBER_SET" );
      case PART_NUMBER_RESULT__NO_PART_NUMBER_FOR_DEVICE_SET:
         return( "PART_NUMBER_RESULT__NO_PART_NUMBER_FOR_DEVICE_SET" );
      case PART_NUMBER_RESULT__DATA_UNAVAILABLE:
         return( "PART_NUMBER_RESULT__DATA_UNAVAILABLE" );
      case PART_NUMBER_RESULT__META_DATA_UNAVAILABLE:
         return( "PART_NUMBER_RESULT__META_DATA_UNAVAILABLE" );
      case PART_NUMBER_RESULT__UNKNOWN:
         return( "PART_NUMBER_RESULT__UNKNOWN" );
      default:
         return( "PART_NUMBER_RESULT__UNKNOWN" );
      }
   }

   PartNumberResult m_result;
   ::std::string    m_partNumber;
};

class ScomoAvailableUpdates
{
public:
   enum  AvailableUpdatesResult
   {
      AVAILABLE_UPDATES_RESULT__SUCCESS = 0,
      AVAILABLE_UPDATES_RESULT__INSTALLER_BUSY = 1,
      AVAILABLE_UPDATES_RESULT__CATALOGUE_OUTDATED = 2,
      AVAILABLE_UPDATES_RESULT__CATALOGUE_NOT_AVAILABLE = 3,
      AVAILABLE_UPDATES_RESULT__SUBSCRIPTION_EXPIRED = 4,
      AVAILABLE_UPDATES_RESULT__FAILED = 5
   };

   enum UpdateState
   {
      UPDATE_STATE__UP_TO_DATE = 0,
      UPDATE_STATE__UPDATE_AVAILABLE = 1,
      UPDATE_STATE__UPDATING = 2,

   };
   //inner class begin
   class UpdateDescription
   {
   public:
      UpdateDescription()
         : m_updateVersion()
         , m_downloadSizeInBytes(0)
         , m_estimatedInstallTimeInSeconds(0)
      {}
      ~UpdateDescription()
      {}

      uint64_t getDownloadSizeInBytes() const
      {
         return m_downloadSizeInBytes;
      }

      void setDownloadSizeInBytes(uint64_t downloadSizeInBytes)
      {
         m_downloadSizeInBytes = downloadSizeInBytes;
      }

      uint64_t getEstimatedInstallTimeInSeconds() const
      {
         return m_estimatedInstallTimeInSeconds;
      }

      void setEstimatedInstallTimeInSeconds(uint64_t estimatedInstallTimeInSeconds)
      {
         m_estimatedInstallTimeInSeconds = estimatedInstallTimeInSeconds;
      }

      const VersionDescription& getUpdateVersion() const
      {
         return m_updateVersion;
      }

      void setUpdateVersion(const VersionDescription& updateVersion)
      {
         m_updateVersion = updateVersion;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "UpdateDescription payload:" << ::std::endl;
         stream << "m_updateVersion " << m_updateVersion.toString() << ::std::endl
                << "m_downloadSizeInBytes " << m_downloadSizeInBytes << ::std::endl
                << "m_estimatedInstallTimeInSeconds " << m_estimatedInstallTimeInSeconds << ::std::endl;
         return stream.str();
      }
   private:
      VersionDescription m_updateVersion;
      uint64_t           m_downloadSizeInBytes;
      uint64_t           m_estimatedInstallTimeInSeconds;
   };

   class UpdateRegionVersionInfo
   {
   public:
      UpdateRegionVersionInfo()
         : m_updateRegionId(0)
         , m_updateStatus(UPDATE_STATE__UP_TO_DATE)
         , m_updateDescription()
      {}
      ~UpdateRegionVersionInfo()
      {}

      const ::std::vector<UpdateDescription>& getUpdateDescription() const
      {
         return m_updateDescription;
      }

      void setUpdateDescription(const ::std::vector<UpdateDescription>& updateDescription)
      {
         m_updateDescription = updateDescription;
      }

      uint32_t getUpdateRegionId() const
      {
         return m_updateRegionId;
      }

      void setUpdateRegionId(uint32_t updateRegionId)
      {
         m_updateRegionId = updateRegionId;
      }

      UpdateState getUpdateStatus() const
      {
         return m_updateStatus;
      }

      void setUpdateStatus(UpdateState updateStatus)
      {
         m_updateStatus = updateStatus;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "UpdateRegionVersionInfo payload:" << ::std::endl;
         stream << "m_updateRegionId " << m_updateRegionId << ::std::endl
                << "m_updateStatus " << m_updateStatus << ::std::endl;
         for (::std::vector<UpdateDescription>::const_iterator it = m_updateDescription.begin();
               it != m_updateDescription.end(); ++it)
         {
            stream << it->toString() << ::std::endl;
         }
         return stream.str();
      }
   private:
      uint32_t    m_updateRegionId;
      UpdateState m_updateStatus;
      ::std::vector<UpdateDescription> m_updateDescription;
   };

   class ProductVersionInfo
   {
   public:
      ProductVersionInfo()
         : m_productId(0)
         , m_supplierId(0)
         , m_baselineMapId(0)
         , m_productTypeMask(0)
         , m_updateRegionVersionInfos()
      {}
      ~ProductVersionInfo()
      {}

      uint32_t getBaselineMapId() const
      {
         return m_baselineMapId;
      }

      void setBaselineMapId(uint32_t baselineMapId)
      {
         m_baselineMapId = baselineMapId;
      }

      uint32_t getProductId() const
      {
         return m_productId;
      }

      void setProductId(uint32_t productId)
      {
         m_productId = productId;
      }

      uint32_t getSupplierId() const
      {
         return m_supplierId;
      }

      void setSupplierId(uint32_t supplierId)
      {
         m_supplierId = supplierId;
      }

      uint32_t getProductTypeMask() const
      {
         return m_productTypeMask;
      }

      void setProductTypeMask(uint32_t productTypeMask)
      {
         m_productTypeMask = productTypeMask;
      }

      const ::std::vector<UpdateRegionVersionInfo>& getUpdateRegionVersionInfos() const
      {
         return m_updateRegionVersionInfos;
      }

      void setUpdateRegionVersionInfos(const ::std::vector<UpdateRegionVersionInfo>& updateRegionVersionInfos)
      {
         m_updateRegionVersionInfos = updateRegionVersionInfos;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "ProductVersionInfo payload:" << ::std::endl;
         stream << "m_productId " << m_productId << ::std::endl
                << "m_supplierId " << m_supplierId << ::std::endl
                << "m_baselineMapId " << m_baselineMapId << ::std::endl
                << "m_productTypeMask " << m_productTypeMask << ::std::endl;
         for (::std::vector<UpdateRegionVersionInfo>::const_iterator it = m_updateRegionVersionInfos.begin();
               it != m_updateRegionVersionInfos.end(); ++it)
         {
            stream << it->toString() << ::std::endl;
         }
         return stream.str();
      }

   private:
      uint32_t m_productId;
      uint32_t m_supplierId;
      uint32_t m_baselineMapId;
      uint32_t m_productTypeMask;
      ::std::vector<UpdateRegionVersionInfo> m_updateRegionVersionInfos;
   };
   //inner class end
   ScomoAvailableUpdates()
      : m_productVersionInfos()
      , m_availableUpdatesResult(AVAILABLE_UPDATES_RESULT__FAILED)
   {}
   ~ScomoAvailableUpdates()
   {}

   AvailableUpdatesResult getAvailableUpdatesResult() const
   {
      return(m_availableUpdatesResult);
   }

   void setAvailableUpdatesResult(AvailableUpdatesResult availableUpdatesResult)
   {
      m_availableUpdatesResult = availableUpdatesResult;
   }

   const ::std::vector<ProductVersionInfo>& getProductVersionInfos() const
   {
      return(m_productVersionInfos);
   }

   void addProductVersionInfos(const ProductVersionInfo& productVersionInfo)
   {
      m_productVersionInfos.push_back(productVersionInfo);
   }

   void setProductVersionInfos(const ::std::vector<ProductVersionInfo>& productVersionInfos)
   {
      m_productVersionInfos = productVersionInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ScomoAvailableUpdates payload:" << ::std::endl;
      stream << "Result: " << m_availableUpdatesResult << ::std::endl;
      for (::std::vector<ProductVersionInfo>::const_iterator it = m_productVersionInfos.begin();
            it != m_productVersionInfos.end(); ++it)
      {
         stream << it->toString() << ::std::endl;
      }
      return stream.str();
   }
private:
   ::std::vector<ProductVersionInfo> m_productVersionInfos;
   AvailableUpdatesResult            m_availableUpdatesResult;
};

class DeinstallScomoComponentResult
{
public:
   explicit DeinstallScomoComponentResult(DeinstallResult result)
      : m_result(result)
   {}
   ~DeinstallScomoComponentResult()
   {}

   DeinstallResult getResult() const
   {
      return m_result;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "DeinstallScomoComponentResult payload:" << ::std::endl;
      stream << "result " << m_result << ::std::endl;
      return stream.str();
   }
private:
   DeinstallResult m_result;
};

//NavDatasetInfo
//Enum SharedDataAccess used in SharedDataAccessAttributes (part of NavDatasetInfo)
enum SharedDataAccess
{
   SHARED_DATA_ACCESS__NOT_REQUIRED,
   SHARED_DATA_ACCESS__REQUIRED,
   SHARED_DATA_ACCESS__ENABLE,
   SHARED_DATA_ACCESS__UNKNOWN
};

inline ::std::string toString(SharedDataAccess sharedDataAccess)
{
   switch (sharedDataAccess)
   {
   case SHARED_DATA_ACCESS__NOT_REQUIRED:
      return("SHARED_DATA_ACCESS__NOT_REQUIRED");
   case SHARED_DATA_ACCESS__REQUIRED:
      return("SHARED_DATA_ACCESS__REQUIRED");
   case SHARED_DATA_ACCESS__ENABLE:
      return("SHARED_DATA_ACCESS__ENABLE");
   case SHARED_DATA_ACCESS__UNKNOWN:
      return("SHARED_DATA_ACCESS__UNKNOWN");
   default:
      return("SHARED_DATA_ACCESS__UNKNOWN");
   }
}
//Enum JournalingMode used in SharedDataAccessAttributes (part of NavDatasetInfo)
enum JournalingMode
{
   JOURNALING_MODE__NONE,
   JOURNALING_MODE__WAL,
   JOURNALING_MODE__UNKNOWN
};

inline ::std::string toString(JournalingMode journalingMode)
{
   switch (journalingMode)
   {
   case JOURNALING_MODE__NONE:
      return("JOURNALING_MODE__NONE");
   case JOURNALING_MODE__WAL:
      return("JOURNALING_MODE__WAL");
   case JOURNALING_MODE__UNKNOWN:
      return("JOURNALING_MODE__UNKNOWN");
   default:
      return("JOURNALING_MODE__UNKNOWN");
   }
}
//Enum LockType used in SharedDataAccessAttributes (part of NavDatasetInfo)
enum LockType
{
   LOCK_TYPE__NONE,
   LOCK_TYPE__NORMAL,
   LOCK_TYPE__UNKNOWN
};

inline ::std::string toString(LockType lockType)
{
   switch (lockType)
   {
   case LOCK_TYPE__NONE:
      return("LOCK_TYPE__NONE");
   case LOCK_TYPE__NORMAL:
      return("LOCK_TYPE__NORMAL");
   case LOCK_TYPE__UNKNOWN:
      return("LOCK_TYPE__UNKNOWN");
   default:
      return("LOCK_TYPE__UNKNOWN");
   }
}

class NavDatasetInfo
{
public:
   //inner struct definition
   struct SharedDataAccessAttributes
   {
      SharedDataAccessAttributes()
         : m_sharedDataAccess(SHARED_DATA_ACCESS__UNKNOWN)
         , m_journalingMode(JOURNALING_MODE__UNKNOWN)
         , m_lockType(LOCK_TYPE__UNKNOWN)
      {}

      bool operator==(const SharedDataAccessAttributes& other) const
      {
         return (this->m_sharedDataAccess == other.m_sharedDataAccess)
                && (this->m_journalingMode == other.m_journalingMode)
                && (this->m_lockType == other.m_lockType);
      }

      bool operator!=(const SharedDataAccessAttributes& other) const
      {
         return !(*this == other);
      }

      SharedDataAccess m_sharedDataAccess;
      JournalingMode   m_journalingMode;
      LockType         m_lockType;
   };

   NavDatasetInfo()
      : m_datasetId(0)
   {}

   bool operator==(const NavDatasetInfo& other) const
   {
      return (this->m_datasetId == other.m_datasetId)
             && (this->m_persistentDatasetId == other.m_persistentDatasetId)
             && (this->m_sharedDataAccessAttributes == other.m_sharedDataAccessAttributes);
   }

   bool operator!=(const NavDatasetInfo& other) const
   {
      return !(*this == other);
   }

   uint32_t getDatasetId() const
   {
      return m_datasetId;
   }

   void setDatasetId(uint32_t datasetId)
   {
      m_datasetId = datasetId;
   }

   const ::std::string& getPersistentDatasetId() const
   {
      return m_persistentDatasetId;
   }

   void setPersistentDatasetId(const ::std::string& persistentDatasetId)
   {
      m_persistentDatasetId = persistentDatasetId;
   }

   const SharedDataAccessAttributes& getSharedDataAccessAttributes() const
   {
      return m_sharedDataAccessAttributes;
   }

   void setSharedDataAccessAttributes(const SharedDataAccessAttributes& sharedDataAccessAttributes)
   {
      m_sharedDataAccessAttributes = sharedDataAccessAttributes;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDatasetInfo payload:" << ::std::endl;
      stream << "m_datasetId " << m_datasetId << ::std::endl;
      stream << "m_persistentDatasetId " << m_persistentDatasetId << ::std::endl;
      stream << "m_sharedDataAccessAttributes.m_sharedDataAccess " << ::navmiddleware::toString(m_sharedDataAccessAttributes.m_sharedDataAccess) << ::std::endl;
      stream << "m_sharedDataAccessAttributes.m_journalingMode " << ::navmiddleware::toString(m_sharedDataAccessAttributes.m_journalingMode) << ::std::endl;
      stream << "m_sharedDataAccessAttributes.m_lockType " << ::navmiddleware::toString(m_sharedDataAccessAttributes.m_lockType) << ::std::endl;
      return stream.str();
   }

private:
   uint32_t                   m_datasetId;
   ::std::string              m_persistentDatasetId;
   SharedDataAccessAttributes m_sharedDataAccessAttributes;
};
//RegisterDatasetUserResult
//Enum NdsDatasetUser used in RegisterDatasetUserResult
enum NdsDatasetUser
{
   NDS_DATASET_USER__SDS,
   NDS_DATASET_USER__UNKNOWN
};

inline ::std::string toString(NdsDatasetUser ndsDatasetUser)
{
   switch (ndsDatasetUser)
   {
   case NDS_DATASET_USER__SDS:
      return("NDS_DATASET_USER__SDS");
   case NDS_DATASET_USER__UNKNOWN:
      return("NDS_DATASET_USER__UNKNOWN");
   default:
      return("NDS_DATASET_USER__UNKNOWN");
   }
}
//Enum RegisterDatasetUserStatus used in RegisterDatasetUserResult
enum RegisterDatasetUserStatus
{
   REGISTER_DATASET_USER_STATUS__REG_OK,
   REGISTER_DATASET_USER_STATUS__REG_FAILED,
   REGISTER_DATASET_USER_STATUS__DEREG_OK,
   REGISTER_DATASET_USER_STATUS__DEREG_FAILED,
   REGISTER_DATASET_USER_STATUS__UNKNOWN
};
inline ::std::string toString(RegisterDatasetUserStatus registerDatasetUserStatus)
{
   switch (registerDatasetUserStatus)
   {
   case REGISTER_DATASET_USER_STATUS__REG_OK:
      return("REGISTER_DATASET_USER_STATUS__REG_OK");
   case REGISTER_DATASET_USER_STATUS__REG_FAILED:
      return("REGISTER_DATASET_USER_STATUS__REG_FAILED");
   case REGISTER_DATASET_USER_STATUS__DEREG_OK:
      return("REGISTER_DATASET_USER_STATUS__DEREG_OK");
   case REGISTER_DATASET_USER_STATUS__DEREG_FAILED:
      return("REGISTER_DATASET_USER_STATUS__DEREG_FAILED");
   case REGISTER_DATASET_USER_STATUS__UNKNOWN:
      return("REGISTER_DATASET_USER_STATUS__UNKNOWN");
   default:
      return("REGISTER_DATASET_USER_STATUS__UNKNOWN");
   }
}

class RegisterDatasetUserResult
{
public:
   RegisterDatasetUserResult()
      : m_ndsDatasetUser(NDS_DATASET_USER__UNKNOWN)
      , m_datasetId(0)
      , m_registerDatasetUserStatus(REGISTER_DATASET_USER_STATUS__UNKNOWN)
   {}

   uint32_t getDatasetId() const
   {
      return m_datasetId;
   }

   void setDatasetId(uint32_t datasetId)
   {
      m_datasetId = datasetId;
   }

   NdsDatasetUser getNdsDatasetUser() const
   {
      return m_ndsDatasetUser;
   }

   void setNdsDatasetUser(NdsDatasetUser ndsDatasetUser)
   {
      m_ndsDatasetUser = ndsDatasetUser;
   }

   RegisterDatasetUserStatus getRegisterDatasetUserStatus() const
   {
      return m_registerDatasetUserStatus;
   }

   void setRegisterDatasetUserStatus(RegisterDatasetUserStatus registerDatasetUserStatus)
   {
      m_registerDatasetUserStatus = registerDatasetUserStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "RegisterDatasetUserResult payload:" << ::std::endl;
      stream << "m_ndsDatasetUser " << ::navmiddleware::toString(m_ndsDatasetUser) << ::std::endl;
      stream << "m_datasetId " << m_datasetId << ::std::endl;
      stream << "m_registerDatasetUserStatus " << ::navmiddleware::toString(m_registerDatasetUserStatus) << ::std::endl;
      return stream.str();
   }

private:
   NdsDatasetUser            m_ndsDatasetUser;
   uint32_t                  m_datasetId;
   RegisterDatasetUserStatus m_registerDatasetUserStatus;
};

//Enum NdsDataAccessType used in request datasetUserAccess
enum NdsDataAccessType
{
   NDS_DATA_ACCESS_TYPE__NOT_SHARED,
   NDS_DATA_ACCESS_TYPE__SHARED,
   NDS_DATA_ACCESS_TYPE__UNKNOWN
};

inline ::std::string toString(NdsDataAccessType ndsDataAccessType)
{
   switch (ndsDataAccessType)
   {
   case NDS_DATA_ACCESS_TYPE__NOT_SHARED:
      return("NDS_DATA_ACCESS_TYPE__NOT_SHARED");
   case NDS_DATA_ACCESS_TYPE__SHARED:
      return("NDS_DATA_ACCESS_TYPE__SHARED");
   case NDS_DATA_ACCESS_TYPE__UNKNOWN:
      return("NDS_DATA_ACCESS_TYPE__UNKNOWN");
   default:
      return("NDS_DATA_ACCESS_TYPE__UNKNOWN");
   }
}

/** Describes a navigation data release by package id, region, release date and supplier as available from navigation
 * data meta description.
 */
class NavDataReleaseInfo
{
public:
   NavDataReleaseInfo()
   {}

   const ::std::string& getPackageId() const
   {
      return m_packageId;
   }

   void setPackageId(const ::std::string& packageId)
   {
      m_packageId = packageId;
   }

   const ::std::string& getRegionName() const
   {
      return m_regionName;
   }

   void setRegionName(const ::std::string& regionName)
   {
      m_regionName = regionName;
   }

   const ::std::string& getReleaseDate() const
   {
      return m_releaseDate;
   }

   void setReleaseDate(const ::std::string& releaseDate)
   {
      m_releaseDate = releaseDate;
   }

   const ::std::string& getSupplierName() const
   {
      return m_supplierName;
   }

   void setSupplierName(const ::std::string& supplierName)
   {
      m_supplierName = supplierName;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataReleaseInfo:" << ::std::endl;
      stream << "m_packageId: " << m_packageId
             << ", m_releaseDate: " << m_releaseDate
             << ", m_regionName: " << m_regionName
             << ", m_supplierName: " << m_supplierName << ::std::endl;
      return stream.str();
   }

private:
   ::std::string m_packageId;     /* unique identifier for each package */
   ::std::string m_releaseDate;   /* release date of the package */
   ::std::string m_regionName;    /* region name of the map for which the package - do not mix it with update region. */
   ::std::string m_supplierName;  /* supplier name from where the package */

};

/** Describes an updateRegion by offering the NDS related unique identifier of the product and the updateRegion contained
 * in this product. Additional the names of product and updateRegion are set if they were available. If possible it
 * is shown in current display language and stored in the navigation data itself.
 */
class UpdateRegionDescription
{
public:
   UpdateRegionDescription()
      : m_productId(0)
      , m_supplierId(0)
      , m_baselineMapId(0)
      , m_updateRegionId(0)
   {}

   uint32_t getBaselineMapId() const
   {
      return m_baselineMapId;
   }

   void setBaselineMapId(uint32_t baselineMapId)
   {
      m_baselineMapId = baselineMapId;
   }

   uint32_t getProductId() const
   {
      return m_productId;
   }

   void setProductId(uint32_t productId)
   {
      m_productId = productId;
   }

   const ::std::string& getProductName() const
   {
      return m_productName;
   }

   void setProductName(const ::std::string& productName)
   {
      m_productName = productName;
   }

   const ::std::string& getRegionName() const
   {
      return m_regionName;
   }

   void setRegionName(const ::std::string& regionName)
   {
      m_regionName = regionName;
   }

   uint32_t getSupplierId() const
   {
      return m_supplierId;
   }

   void setSupplierId(uint32_t supplierId)
   {
      m_supplierId = supplierId;
   }

   uint32_t getUpdateRegionId() const
   {
      return m_updateRegionId;
   }

   void setUpdateRegionId(uint32_t updateRegionId)
   {
      m_updateRegionId = updateRegionId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "UpdateRegionDescription: "
             << "m_productId: " << m_productId
             << ", m_supplierId: " << m_supplierId
             << ", m_baselineMapId: " << m_baselineMapId
             << ", m_updateRegionId: " << m_updateRegionId
             << ", m_productName: " << m_productName
             << ", m_regionName: " << m_regionName
             << ::std::endl;
      return stream.str();
   }

private:
   uint32_t m_productId;
   uint32_t m_supplierId;
   uint32_t m_baselineMapId;
   uint32_t m_updateRegionId;
   ::std::string m_productName;
   ::std::string m_regionName;
};

/** Describes one history log entry of a update installation. Depending on the type of update the updateRegion being updated
 * is described in detail - for full updates it is not available.
 */
class NavDataUpdateLogEntry
{
public:
   NavDataUpdateLogEntry()
      : m_updateCount(0)
      , m_dateOfUpdate(0)
      , m_timeTakenForUpdate(0)
      , m_updateDeviceType(DEVICETYPE__UNKNOWN)
      , m_updateVariant(UPDATE_VARIANT__UNKNOWN)
      , m_updateType(UPDATE_TYPE__UNKNOWN)
      , m_updateResult(UPDATE_RESULT__UNKNOWN)
   {}

   uint64_t getDateOfUpdate() const
   {
      return m_dateOfUpdate;
   }

   void setDateOfUpdate(uint64_t dateOfUpdate)
   {
      m_dateOfUpdate = dateOfUpdate;
   }

   const ::std::string& getPackageIdBeforeUpdate() const
   {
      return m_packageIdBeforeUpdate;
   }

   void setPackageIdBeforeUpdate(const ::std::string& packageIdBeforeUpdate)
   {
      m_packageIdBeforeUpdate = packageIdBeforeUpdate;
   }

   const NavDataReleaseInfo& getPackageReleaseInfo() const
   {
      return m_packageReleaseInfo;
   }

   void setPackageReleaseInfo(const NavDataReleaseInfo& packageReleaseInfo)
   {
      m_packageReleaseInfo = packageReleaseInfo;
   }

   uint64_t getTimeTakenForUpdate() const
   {
      return m_timeTakenForUpdate;
   }

   void setTimeTakenForUpdate(uint64_t timeTakenForUpdate)
   {
      m_timeTakenForUpdate = timeTakenForUpdate;
   }

   uint32_t getUpdateCount() const
   {
      return m_updateCount;
   }

   void setUpdateCount(uint32_t updateCount)
   {
      m_updateCount = updateCount;
   }

   DeviceType getUpdateDeviceType() const
   {
      return m_updateDeviceType;
   }

   void setUpdateDeviceType(DeviceType updateDeviceType)
   {
      m_updateDeviceType = updateDeviceType;
   }

   const ValidValue< UpdateRegionDescription >& getUpdateRegionDescription() const
   {
      return m_updateRegionDescription;
   }

   void setUpdateRegionDescription(const ValidValue< UpdateRegionDescription >& updateRegionDescription)
   {
      m_updateRegionDescription = updateRegionDescription;
   }

   UpdateResult getUpdateResult() const
   {
      return m_updateResult;
   }

   void setUpdateResult(UpdateResult updateResult)
   {
      m_updateResult = updateResult;
   }

   UpdateType getUpdateType() const
   {
      return m_updateType;
   }

   void setUpdateType(UpdateType updateType)
   {
      m_updateType = updateType;
   }

   UpdateVariant getUpdateVariant() const
   {
      return m_updateVariant;
   }

   void setUpdateVariant(UpdateVariant updateVariant)
   {
      m_updateVariant = updateVariant;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataUpdateLogEntry:" << ::std::endl;
      stream << "m_updateCount:" << m_updateCount
             << ", m_dateOfUpdate: " << m_dateOfUpdate
             << ", m_timeTakenForUpdate: " << m_timeTakenForUpdate << ::std::endl;
      stream << "m_updateDeviceType: " << ::navmiddleware::toString(m_updateDeviceType)
             << ", m_updateVariant: " << ::navmiddleware::toString(m_updateVariant)
             << ", m_updateType: " << ::navmiddleware::toString(m_updateType)
             << ", m_updateResult: " << ::navmiddleware::toString(m_updateResult) << ::std::endl;
      if (m_updateRegionDescription.isValid())
      {
         m_updateRegionDescription.getValue().toString();
      }
      else
      {
         stream << "No updateRegionDescription" << ::std::endl;
      }
      stream << "m_packageIdBeforeUpdate: " << m_packageIdBeforeUpdate << ::std::endl;
      stream << "m_packageReleaseInfo: " << m_packageReleaseInfo.toString();
      return stream.str();
   }

private:
   uint32_t m_updateCount; /* count of consecutive number of updates */
   uint64_t m_dateOfUpdate; /* date when the update was started (POSIX time stamp) */
   uint64_t m_timeTakenForUpdate; /* time taken for update in seconds*/
   DeviceType m_updateDeviceType; /* describes the type of update source the update was received from */
   UpdateVariant m_updateVariant; /* either update/fullUpdate */
   UpdateType m_updateType; /* more details about the type of update */
   UpdateResult m_updateResult; /* update installation result */
   ValidValue< UpdateRegionDescription > m_updateRegionDescription; /* region being updated if it was no of type FULL_UPDATE */
   ::std::string m_packageIdBeforeUpdate; /* the package identifier from release info before the update was applied */
   NavDataReleaseInfo m_packageReleaseInfo; /* new releaseInfo after update */

};

/** Class describes the history of all past update installations. It contains the information about the initial
 * navigation data package installed on production and a list of applied/rejected update installations.
 * Depending on project requirements the maximum number of entries can be limited. In case after maximum is reached
 * the oldest entry is removed to add a new update.
 */
class NavDataUpdateHistory
{
public:
   NavDataUpdateHistory()
   {}

   const NavDataReleaseInfo& getInitialNavDataReleaseInfo() const
   {
      return m_initialNavDataReleaseInfo;
   }

   void setInitialNavDataReleaseInfo(const NavDataReleaseInfo& initialNavDataReleaseInfo)
   {
      m_initialNavDataReleaseInfo = initialNavDataReleaseInfo;
   }

   const ::std::vector< NavDataUpdateLogEntry >& getNavDataUpdateLogs() const
   {
      return m_navDataUpdateLogs;
   }

   void setNavDataUpdateLogs(const ::std::vector<NavDataUpdateLogEntry>& navDataUpdateLogs)
   {
      m_navDataUpdateLogs = navDataUpdateLogs;
   }

   ::std::vector< NavDataUpdateLogEntry >& getNavDataUpdateLogsMutable()
   {
      return m_navDataUpdateLogs;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "NavDataUpdateHistory payload:" << ::std::endl;
      stream << "m_initialNavDataReleaseInfo " << m_initialNavDataReleaseInfo.toString() << ::std::endl;
      stream << "m_navDataUpdateLogs: " << ::std::endl;
      for (::std::vector< NavDataUpdateLogEntry >::const_iterator iterator = m_navDataUpdateLogs.begin();
            iterator != m_navDataUpdateLogs.end(); ++iterator)
      {
         stream << iterator->toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   NavDataReleaseInfo m_initialNavDataReleaseInfo;
   ::std::vector< NavDataUpdateLogEntry > m_navDataUpdateLogs;
};

enum AutomaticFullUpdateStatus
{
   AUTOMATIC_FULL_UPDATE_STATUS__UNKNOWN,
   AUTOMATIC_FULL_UPDATE_STATUS__COMPLETED_WITH_SUCCESS,
   AUTOMATIC_FULL_UPDATE_STATUS__COMPLETED_WITH_FAILURE
};

inline ::std::string toString(AutomaticFullUpdateStatus status)
{
   switch (status)
   {
   case AUTOMATIC_FULL_UPDATE_STATUS__COMPLETED_WITH_SUCCESS:
      return("AUTOMATIC_FULL_UPDATE_STATUS__COMPLETED_WITH_SUCCESS");
   case AUTOMATIC_FULL_UPDATE_STATUS__COMPLETED_WITH_FAILURE:
      return("AUTOMATIC_FULL_UPDATE_STATUS__COMPLETED_WITH_FAILURE");
   case AUTOMATIC_FULL_UPDATE_STATUS__UNKNOWN:
   default:
      return("AUTOMATIC_FULL_UPDATE_STATUS__UNKNOWN");
   }
}

enum AutomaticFullUpdateFailureReason
{
   AUTOMATIC_FULL_UPDATE_FAILURE_REASON__UNKNOWN,
   AUTOMATIC_FULL_UPDATE_FAILURE_REASON__LOCK_NOT_ACQUIRED,
   AUTOMATIC_FULL_UPDATE_FAILURE_REASON__CACHE_DIRECTORY_NOT_ACQUIRED
};

inline ::std::string toString(AutomaticFullUpdateFailureReason reason)
{
   switch(reason)
   {
   case AUTOMATIC_FULL_UPDATE_FAILURE_REASON__LOCK_NOT_ACQUIRED:
      return("AUTOMATIC_FULL_UPDATE_FAILURE_REASON__LOCK_NOT_ACQUIRED");
   case AUTOMATIC_FULL_UPDATE_FAILURE_REASON__CACHE_DIRECTORY_NOT_ACQUIRED:
      return("AUTOMATIC_FULL_UPDATE_FAILURE_REASON__CACHE_DIRECTORY_NOT_ACQUIRED");
   case AUTOMATIC_FULL_UPDATE_FAILURE_REASON__UNKNOWN:
   default:
      return("AUTOMATIC_FULL_UPDATE_FAILURE_REASON__UNKNOWN");
   }
}

} /* namespace navmiddleware */

#endif // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_NAVDATAUPDATEINFOS_H_
