/**
* @copyright (c) 2015-2017 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_MAPPICKINGRESULTS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_MAPPICKINGRESULTS_H_

#include <stdint.h>
#include <cstring>
#include <string>
#include <sstream>
#include <vector>
#include <utility>
#include "InfoTypes.h"

namespace navmiddleware
{

/** Enumerates all possible Road Classes*/
enum PickingResultRoadClass
{
   PICKING_RESULT_ROAD_CLASS__UNKNOWN,             /**< Unknown road class.*/
   PICKING_RESULT_ROAD_CLASS__MOST_IMPORTANT,      /**< The most important road class.*/
   PICKING_RESULT_ROAD_CLASS__SECOND_IMPORTANT,    /**< Second important road class.*/
   PICKING_RESULT_ROAD_CLASS__THIRD_IMPORTANT,     /**< Third important road class*/
   PICKING_RESULT_ROAD_CLASS__FOURTH_IMPORTANT,    /**< Fourth important road class*/
   PICKING_RESULT_ROAD_CLASS__FIFTH_IMPORTANT,     /**< Fifth important road class*/
   PICKING_RESULT_ROAD_CLASS__SIXTH_IMPORTANT,     /**< Sixth important road class*/
   PICKING_RESULT_ROAD_CLASS__LEAST_IMPORTANT      /**< The least important road class.*/
};

inline ::std::string toString(PickingResultRoadClass pickingResultRoadClass)
{
   switch (pickingResultRoadClass)
   {
   case PICKING_RESULT_ROAD_CLASS__UNKNOWN:
      return "PICKING_RESULT_ROAD_CLASS__UNKNOWN";
   case PICKING_RESULT_ROAD_CLASS__MOST_IMPORTANT:
      return "PICKING_RESULT_ROAD_CLASS__MOST_IMPORTANT";
   case PICKING_RESULT_ROAD_CLASS__SECOND_IMPORTANT:
      return "PICKING_RESULT_ROAD_CLASS__SECOND_IMPORTANT";
   case PICKING_RESULT_ROAD_CLASS__THIRD_IMPORTANT:
      return "PICKING_RESULT_ROAD_CLASS__THIRD_IMPORTANT";
   case PICKING_RESULT_ROAD_CLASS__FOURTH_IMPORTANT:
      return "PICKING_RESULT_ROAD_CLASS__FOURTH_IMPORTANT";
   case PICKING_RESULT_ROAD_CLASS__FIFTH_IMPORTANT:
      return "PICKING_RESULT_ROAD_CLASS__FIFTH_IMPORTANT";
   case PICKING_RESULT_ROAD_CLASS__SIXTH_IMPORTANT:
      return "PICKING_RESULT_ROAD_CLASS__SIXTH_IMPORTANT";
   case PICKING_RESULT_ROAD_CLASS__LEAST_IMPORTANT:
      return "PICKING_RESULT_ROAD__CLASS_LEAST_IMPORTANT";
   default:
      ::std::stringstream stream;
      stream << "PICKING_RESULT_ROAD_CLASS__<" << static_cast<unsigned int>(pickingResultRoadClass) << ">";
      return stream.str();
   }
}

/** Enumerates all possible Road Sub-Types*/
enum PickingResultRoadSubType
{
   PICKING_RESULT_ROAD_SUB_TYPE__INVALID, /**< Invalid road sub type.*/
   PICKING_RESULT_ROAD_SUB_TYPE__NORMAL,  /**< Normal road sub type.*/
   PICKING_RESULT_ROAD_SUB_TYPE__FERRY,   /**< Ferry road sub type.*/
   PICKING_RESULT_ROAD_SUB_TYPE__TUNNEL,  /**< Tunnel road sub type.*/
   PICKING_RESULT_ROAD_SUB_TYPE__BRIDGE   /**< Bridge road sub type.*/
};

inline ::std::string toString(PickingResultRoadSubType pickingResultRoadSubType)
{
   switch (pickingResultRoadSubType)
   {
   case PICKING_RESULT_ROAD_SUB_TYPE__INVALID:
      return "PICKING_RESULT_ROAD_SUB_TYPE__INVALID";
   case PICKING_RESULT_ROAD_SUB_TYPE__NORMAL:
      return "PICKING_RESULT_ROAD_SUB_TYPE__NORMAL";
   case PICKING_RESULT_ROAD_SUB_TYPE__FERRY:
      return "PICKING_RESULT_ROAD_SUB_TYPE__FERRY";
   case PICKING_RESULT_ROAD_SUB_TYPE__TUNNEL:
      return "PICKING_RESULT_ROAD_SUB_TYPE__TUNNEL";
   case PICKING_RESULT_ROAD_SUB_TYPE__BRIDGE:
      return "PICKING_RESULT_ROAD_SUB_TYPE__BRIDGE";
   default:
      ::std::stringstream stream;
      stream << "PICKING_RESULT_ROAD_SUB_TYPE_<" << static_cast<unsigned int>(pickingResultRoadSubType) << ">";
      return stream.str();
   }
}

/** Enumerates all possible POI Display Types*/
enum PickingResultPOIDisplayType
{
   PICKING_RESULT_POI_DISPLAY_TYPE__REGULAR,       /**< A regular POI*/
   PICKING_RESULT_POI_DISPLAY_TYPE__LANDMARK,      /**< A landmark POI*/
   PICKING_RESULT_POI_DISPLAY_TYPE__LANDMARK_3D    /**< A landmark POI as real 3D model*/
};

inline ::std::string toString(PickingResultPOIDisplayType pickingResultPOIDisplayType)
{
   switch (pickingResultPOIDisplayType)
   {
   case PICKING_RESULT_POI_DISPLAY_TYPE__REGULAR:
      return "PICKING_RESULT_POI_DISPLAY_TYPE__REGULAR";
   case PICKING_RESULT_POI_DISPLAY_TYPE__LANDMARK:
      return "PICKING_RESULT_POI_DISPLAY_TYPE__LANDMARK";
   case PICKING_RESULT_POI_DISPLAY_TYPE__LANDMARK_3D:
      return "PICKING_RESULT_POI_DISPLAY_TYPE__LANDMARK_3D";
   default:
      ::std::stringstream stream;
      stream << "PICKING_RESULT_POI_DISPLAY_TYPE_<" << static_cast<unsigned int>(pickingResultPOIDisplayType) << ">";
      return stream.str();
   }
}

enum PickingResultExtendedDetailPOIType
{
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__UNDEFINED,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__SHORT_DESCRIPTION,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__LONG_DESCRIPTION,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__ADDRESS,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__HOUSE_NUMBER,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__STREET_NAME,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__TOWN_NAME,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__TOWN_ADD_ON,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__ZIP_CODE,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FLOOR_NUMBER,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__PHONE_NO,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FAX_NO,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__EMAIL,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__WWW,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FILE,
   PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__NAME
};

inline ::std::string toString(PickingResultExtendedDetailPOIType pickingResultExtendedDetailPOIType)
{
   switch (pickingResultExtendedDetailPOIType)
   {
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__UNDEFINED:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__UNDEFINED";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__SHORT_DESCRIPTION:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__SHORT_DESCRIPTION";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__LONG_DESCRIPTION:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__LONG_DESCRIPTION";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__ADDRESS:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__ADDRESS";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__HOUSE_NUMBER:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__HOUSE_NUMBER";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__STREET_NAME:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__STREET_NAME";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__TOWN_NAME:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__TOWN_NAME";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__TOWN_ADD_ON:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__TOWN_ADD_ON";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__ZIP_CODE:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__ZIP_CODE";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FLOOR_NUMBER:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FLOOR_NUMBER";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__PHONE_NO:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__PHONE_NO";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FAX_NO:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FAX_NO";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__EMAIL:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__EMAIL";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__WWW:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__WWW";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FILE:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__FILE";
   case PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__NAME:
      return "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__NAME";
   default:
      ::std::stringstream stream;
      stream << "PICKING_RESULT_EXTENDED_DETAIL_POI_TYPE__<" << static_cast<unsigned int>(pickingResultExtendedDetailPOIType) << ">";
      return stream.str();
   }
}

/** Enumerates different types of Elements that can be picked from map*/
enum PickingResultType
{
   PICKING_RESULT_TYPE__INVALID,                /**< Not valid feature.*/
   PICKING_RESULT_TYPE__DESTINATION,            /**< Route Destination (e.g. shown as flag).*/
   PICKING_RESULT_TYPE__ROAD,                   /**< Road element on the map.*/
   PICKING_RESULT_TYPE__POI,                    /**< A point of interest (such as restaurant, fuel station).*/
   PICKING_RESULT_TYPE__TRAFFIC_ICON,           /**< An icon representing a traffic message.*/
   PICKING_RESULT_TYPE__ADDRESS_BOOK,           /**< Address book element displayed on the map.*/
   PICKING_RESULT_TYPE__CARSOR,                 /**< Carsor element*/
   PICKING_RESULT_TYPE__STORM_ATTRIBUTES,       /**< Storm attribute element displayed on the map.*/
   PICKING_RESULT_TYPE__STORM_TRACK_POSITION,   /**< The storm tracker shows the strength and the path of a particular storm.*/
   PICKING_RESULT_TYPE__WEATHER_ALERT,          /**< Areas with specific weather conditions are highlighted.*/
   PICKING_RESULT_TYPE__WEATHER_ICON,           /**< Icons showing the weather.*/
   PICKING_RESULT_TYPE__AREA,                   /**< [JP] Area element on the map (e.g. city), used if no other element is found. */
   PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON,     /**< Icons showing safety camera. */
   PICKING_RESULT_TYPE__VICS_PARKING,           /**< [JP] VICS or SA/PA parking displayed on the map. */
   PICKING_RESULT_TYPE__DYNAMIC_ICON            /**< Dynamic icon on the map*/
};

inline ::std::string toString(PickingResultType pickingResultType)
{
   switch (pickingResultType)
   {
   case PICKING_RESULT_TYPE__INVALID:
      return "PICKING_RESULT_TYPE__INVALID";
   case PICKING_RESULT_TYPE__DESTINATION:
      return "PICKING_RESULT_TYPE__DESTINATION";
   case PICKING_RESULT_TYPE__ROAD:
      return "PICKING_RESULT_TYPE__ROAD";
   case PICKING_RESULT_TYPE__POI:
      return "PICKING_RESULT_TYPE__POI";
   case PICKING_RESULT_TYPE__TRAFFIC_ICON:
      return "PICKING_RESULT_TYPE__TRAFFIC_ICON";
   case PICKING_RESULT_TYPE__ADDRESS_BOOK:
      return "PICKING_RESULT_TYPE__ADDRESS_BOOK";
   case PICKING_RESULT_TYPE__CARSOR:
      return "PICKING_RESULT_TYPE__CARSOR";
   case PICKING_RESULT_TYPE__STORM_ATTRIBUTES:
      return "PICKING_RESULT_TYPE__STORM_ATTRIBUTES";
   case PICKING_RESULT_TYPE__STORM_TRACK_POSITION:
      return "PICKING_RESULT_TYPE__STORM_TRACK_POSITION";
   case PICKING_RESULT_TYPE__WEATHER_ALERT:
      return "PICKING_RESULT_TYPE__WEATHER_ALERT";
   case PICKING_RESULT_TYPE__WEATHER_ICON:
      return "PICKING_RESULT_TYPE__WEATHER_ICON";
   case PICKING_RESULT_TYPE__AREA:
      return "PICKING_RESULT_TYPE__AREA";
   case PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON:
      return "PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON";
   case PICKING_RESULT_TYPE__VICS_PARKING:
      return "PICKING_RESULT_TYPE__VICS_PARKING";
   case PICKING_RESULT_TYPE__DYNAMIC_ICON:
      return "PICKING_RESULT_TYPE__DYNAMIC_ICON";
   default:
      ::std::stringstream stream;
      stream << "PICKING_RESULT_TYPE__<" << static_cast<unsigned int>(pickingResultType) << ">";
      return stream.str();
   }
}

/** Enumerates Error occurred while Picking */
enum PickingError
{
   PICKING_ERROR__INTERNAL = 0,    /**< Internal failure*/
   PICKING_ERROR__REQUEST_ABORTED, /**< Request Aborted*/
};

inline ::std::string toString(PickingError pickingError)
{
   switch (pickingError)
   {
   case PICKING_ERROR__INTERNAL:
      return "PICKING_ERROR__INTERNAL";
   case PICKING_ERROR__REQUEST_ABORTED:
      return "PICKING_ERROR__REQUEST_ABORTED";
   default:
      ::std::stringstream stream;
      stream << "PICKING_ERROR__<" << static_cast<unsigned int>(pickingError) << ">";
      return stream.str();
   }
}

class PickingResults
{
public:

   PickingResults()
      : m_mapViewId(MAP_VIEW_ID__PRIMARY)
   {}

   virtual ~PickingResults()
   {}

   class PickingResultDetail
   {
   public:
      PickingResultDetail()
      {}

      PickingResultDetail(const PickingResultDetail& pickingResultDetail)
         : m_iconData(pickingResultDetail.m_iconData)
      {}

      virtual ~PickingResultDetail() {}

      virtual PickingResultDetail* clone() = 0;

      PickingResultDetail& operator=(const PickingResultDetail& pickingResultDetail)
      {
         if (this != &pickingResultDetail)
         {
            m_iconData = pickingResultDetail.m_iconData;
         }
         return *this;
      }

      bool operator==(const PickingResultDetail& rhs) const
      {
         return m_iconData == rhs.m_iconData;
      }

      bool operator!=(const PickingResultDetail& rhs) const
      {
         return !(*this == rhs);
      }

      void addIconData(const Image& iconData)
      {
         m_iconData = iconData;
      }

      Image m_iconData;
   };

   /** Holds Road type element's info*/
   class PickingResultDetailRoad : public PickingResultDetail
   {
   public:
      PickingResultDetailRoad()
         : m_hasRestrictions(false)
      {}

      virtual ~PickingResultDetailRoad() {}

      PickingResultDetailRoad(const PickingResultDetailRoad& pickingResultDetailRoad)
         : PickingResultDetail(pickingResultDetailRoad)
         , m_lineId(pickingResultDetailRoad.m_lineId)
         , m_linkId(pickingResultDetailRoad.m_linkId)
         , m_namedObjectId(pickingResultDetailRoad.m_namedObjectId)
         , m_roadNumberInfoObject(pickingResultDetailRoad.m_roadNumberInfoObject)
         , m_hasRestrictions(pickingResultDetailRoad.m_hasRestrictions)
         , m_vehicleRestrictionsInfo(pickingResultDetailRoad.m_vehicleRestrictionsInfo)
      {
      }

      PickingResultDetailRoad& operator=(const PickingResultDetailRoad& pickingResultDetailRoad)
      {
         if(this != &pickingResultDetailRoad)
         {
            PickingResultDetail::operator=(pickingResultDetailRoad);
            m_lineId = pickingResultDetailRoad.m_lineId;
            m_linkId = pickingResultDetailRoad.m_linkId;
            m_namedObjectId = pickingResultDetailRoad.m_namedObjectId;
            m_roadNumberInfoObject = pickingResultDetailRoad.m_roadNumberInfoObject;
            m_hasRestrictions = pickingResultDetailRoad.m_hasRestrictions;
            m_vehicleRestrictionsInfo = pickingResultDetailRoad.m_vehicleRestrictionsInfo;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailRoad& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_linkId == rhs.m_linkId) &&
                (m_namedObjectId == rhs.m_namedObjectId) &&
                (m_hasRestrictions == rhs.m_hasRestrictions) &&
                (m_lineId == rhs.m_lineId) &&
                (m_vehicleRestrictionsInfo == rhs.m_vehicleRestrictionsInfo);

      }

      bool operator!=(const PickingResultDetailRoad& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailRoad payload:\n");

         stream << ::std::endl
                << "PickingResultDetailRoad LineID = " << m_lineId.getValue().toString()<< ::std::endl
                << "PickingResultDetailRoad LinkID = " << m_linkId.toString() << ::std::endl
                << "PickingResultDetailRoad NamedObjectId = " << m_namedObjectId.toString() << ::std::endl
                << "PickingResultDetailRoad Has Restrictions = " << m_hasRestrictions << ::std::endl
                << "PickingResultDetailRoad VehicleRestrictionsInfo = " << m_vehicleRestrictionsInfo.getValue().toString() << ::std::endl
                << "PickingResultDetailRoad RoadNumberInfo= [" << ::std::endl;

         if(!m_roadNumberInfoObject.empty())
         {
            ::std::vector< ::std::string >::const_iterator roadNumberInfoIterator;
            for(roadNumberInfoIterator = m_roadNumberInfoObject.begin();
                  roadNumberInfoIterator !=  m_roadNumberInfoObject.end();
                  ++roadNumberInfoIterator)
            {
               stream << " ' " << *roadNumberInfoIterator << " ',"<< ::std::endl;
            }
         }
         else
         {
            stream << "No RoadNumberInfo found !!! "<< ::std::endl;
         }

         stream << "]" << ::std::endl;
         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailRoad(*this);
      }


      ValidValue<LineId>                                   m_lineId;
      ValidValue<uint64_t>                                 m_linkId;
      ValidValue<uint64_t>                                 m_namedObjectId;
      ::std::vector< ::std::string >                       m_roadNumberInfoObject;
      bool                                                 m_hasRestrictions;
      ValidValue<VehicleRestrictionsInfo>                  m_vehicleRestrictionsInfo;
   };

   class PickingResultDetailPoi : public PickingResultDetail
   {
   public:
      PickingResultDetailPoi()
         : m_selectionCriterion(0)
      {}
      virtual ~PickingResultDetailPoi() {}

      bool operator==(const PickingResultDetailPoi& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_poiId == rhs.m_poiId) &&
                (m_selectionCriterion == rhs.m_selectionCriterion) &&
                (m_locationAttributeInfos == rhs.m_locationAttributeInfos);
      }

      bool operator!=(const PickingResultDetailPoi& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailPoi payload:\n");

         stream << "POI Id = "<< m_poiId << ", "
                << "SelectionCriterion = "<< m_selectionCriterion << ", "
                << "Icon = " << m_iconData.toString() << ", "
                << "Location Attribute Infos = "<< m_locationAttributeInfos.toString() << ::std::endl;

         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailPoi(*this);
      }

      PoiId                        m_poiId;
      uint32_t                     m_selectionCriterion;

      /**
       * The helper function getLocationNameInfo() should be used to get either POI or address name
       * and function getAddressDetailInfos().getAddressValue() should be used to get the required details.
       * This function takes the AddressInfoType enum as a parameter and returns the content as a string.
       */
      LocationAttributeInfos       m_locationAttributeInfos;
   };

   class PickingResultDetailAddressBookPoi : public PickingResultDetail
   {
   public:
      PickingResultDetailAddressBookPoi()
         : m_category( ::navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK) {}
      virtual ~PickingResultDetailAddressBookPoi() {}

      bool operator==(const PickingResultDetailAddressBookPoi& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_locationAttributeInfos == rhs.m_locationAttributeInfos) &&
                (m_addressBookPoiId == rhs.m_addressBookPoiId) &&
                (m_phoneNumber == rhs.m_phoneNumber) &&
                (m_userIconId == rhs.m_userIconId) &&
                (m_addressBookGroup == rhs.m_addressBookGroup) &&
                (m_category == rhs.m_category);
      }

      bool operator!=(const PickingResultDetailAddressBookPoi& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailAddressBookPoi payload:\n");

         stream << "Address Book POI Id = "<< m_addressBookPoiId.toString() << ", "
                << "Icon = " << m_iconData.toString() << ", "
                << "Phone number = "<< m_phoneNumber.getValueOr("")
                << "User Icon ID = " << m_userIconId.getValueOr(0) << ", "
                << "Address Book Group ID = " << m_addressBookGroup.getValueOr(::navmiddleware::DESTINATION_MEMORY_ATTRIBUTES_GROUP__NONE) << ", "
                << "Location Category = " << ::navmiddleware::toString(m_category);

         stream << "LocationAttributeInfos : "
                << m_locationAttributeInfos.toString();
         stream << ::std::endl;

         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailAddressBookPoi(*this);;
      }

      /**
       * The helper function getLocationNameInfo() should be used to get either POI or address name
       * and function getAddressDetailInfos().getAddressValue() should be used to get the required details.
       * This function takes the AddressInfoType enum as a parameter and returns the content as a string.
       */
      LocationAttributeInfos                        m_locationAttributeInfos;
      ValidValue<PoiId>                             m_addressBookPoiId;
      ValidValue< ::std::string >                   m_phoneNumber;
      ValidValue<uint64_t>                          m_userIconId;
      ValidValue<DestinationMemoryAttributesGroup>  m_addressBookGroup;
      DestinationMemoryEntryCategory                m_category;
   };

   class PickingResultDetailTm : public PickingResultDetail
   {
   public:
      PickingResultDetailTm()
         : m_isUserDefinedTrafficMessage(false)
      {
      }
      virtual ~PickingResultDetailTm() {}

      PickingResultDetailTm(const PickingResultDetailTm& pickingResultDetailTm)
         : PickingResultDetail(pickingResultDetailTm)
         , m_tmId(pickingResultDetailTm.m_tmId)
         , m_headline(pickingResultDetailTm.m_headline)
         , m_description(pickingResultDetailTm.m_description)
         , m_isUserDefinedTrafficMessage(pickingResultDetailTm.m_isUserDefinedTrafficMessage)
      {
      }

      PickingResultDetailTm& operator=(const PickingResultDetailTm& pickingResultDetailTm)
      {
         if (this != &pickingResultDetailTm)
         {
            PickingResultDetail::operator=(pickingResultDetailTm);
            m_tmId = pickingResultDetailTm.m_tmId;
            m_description = pickingResultDetailTm.m_description;
            m_headline = pickingResultDetailTm.m_headline;
            m_isUserDefinedTrafficMessage = pickingResultDetailTm.m_isUserDefinedTrafficMessage;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailTm& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_tmId == rhs.m_tmId) &&
                (m_headline == rhs.m_headline) &&
                (m_description == rhs.m_description) &&
                (m_isUserDefinedTrafficMessage == rhs.m_isUserDefinedTrafficMessage);
      }

      bool operator!=(const PickingResultDetailTm& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailTm payload:\n");

         stream << ::std::endl
                << " Picking result traffic message id = " << m_tmId.toString() << ::std::endl
                << " Picking result traffic message heading = " << m_headline.toString() << ::std::endl
                << " Picking result traffic message description = " << m_description.toString() << ::std::endl
                << " Picking result user defined traffic message = " << m_isUserDefinedTrafficMessage << ::std::endl
                << ::std::endl;

         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailTm(*this);
      }

      ValidValue<uint32_t>               m_tmId;
      ValidValue< ::std::string >        m_headline;
      ValidValue< ::std::string >        m_description;
      bool                               m_isUserDefinedTrafficMessage;
   };

   /** The picking result for VICS parking shares the data type with the traffic icon picking result,
       but has a different type enum to distinguish between both types of information
    */
   typedef PickingResultDetailTm PickingResultDetailVicsParking;

   class PickingResultDetailDestination : public PickingResultDetail
   {
   public:
      PickingResultDetailDestination() {}
      virtual ~PickingResultDetailDestination() {}

      bool operator==(const PickingResultDetailDestination& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_locationString == rhs.m_locationString) &&
                (m_locationAttributeInfos == rhs.m_locationAttributeInfos) &&
                (m_destinationNumber == rhs.m_destinationNumber) &&
                (m_routeSegmentId == rhs.m_routeSegmentId);
      }

      bool operator!=(const PickingResultDetailDestination& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailDestination payload:\n");
         stream << "Location string = " << m_locationString.toString() << ", "
                << "Location attribute Infos = " << m_locationAttributeInfos.toString() << ", "
                << "Icon = " << m_iconData.toString() << ", "
                << "Destination number = " << m_destinationNumber.toString() << ", "
                << "Route Segment Id = " << m_routeSegmentId.toString() << ", " << ::std::endl;
         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailDestination(*this);
      }

      ValidValue< ::std::string >       m_locationString;
      /**
       * The helper function getLocationNameInfo() should be used to get either POI or address name
       * and function getAddressDetailInfos().getAddressValue() should be used to get the required details.
       * This function takes the AddressInfoType enum as a parameter and returns the content as a string.
       */
      LocationAttributeInfos            m_locationAttributeInfos;
      ValidValue<uint16_t>              m_destinationNumber;
      ValidValue<uint16_t>              m_routeSegmentId;
   };

   class PickingResultDetailStormAttributes : public PickingResultDetail
   {
   public:

      /** Enumerates different types of Storm Types Hail that can be picked from weather map*/
      enum StormTypeHail
      {
         STORM_TYPE_HAIL__EMPTY,
         STORM_TYPE_HAIL__PROBABLE,
         STORM_TYPE_HAIL__POSITIVE
      };

      static ::std::string toString(StormTypeHail stormTypeHail)
      {
         switch (stormTypeHail)
         {
         case STORM_TYPE_HAIL__EMPTY:
            return "STORM_TYPE_HAIL__EMPTY";
         case STORM_TYPE_HAIL__PROBABLE:
            return "STORM_TYPE_HAIL__PROBABLE";
         case STORM_TYPE_HAIL__POSITIVE:
            return "STORM_TYPE_HAIL__POSITIVE";
         default:
            ::std::stringstream stream;
            stream << "STORM_TYPE_HAIL__<" << static_cast<unsigned int>(stormTypeHail) << ">";
            return stream.str();
         }
      }

      /** Enumerates different types of Storm Types  that can be picked from weather map*/
      enum StormType
      {
         STORM_TYPE__EMPTY,
         STORM_TYPE__MESOCYCLONIC,
         STORM_TYPE__TORNADIC
      };

      static ::std::string toString(StormType stormType)
      {
         switch (stormType)
         {
         case STORM_TYPE__EMPTY:
            return "STORM_TYPE__EMPTY";
         case STORM_TYPE__MESOCYCLONIC:
            return "STORM_TYPE__MESOCYCLONIC";
         case STORM_TYPE__TORNADIC:
            return "STORM_TYPE__TORNADIC";
         default:
            ::std::stringstream stream;
            stream << "STORM_TYPE__<" << static_cast<unsigned int>(stormType) << ">";
            return stream.str();
         }
      }

      PickingResultDetailStormAttributes():
         m_stormTypeHail(STORM_TYPE_HAIL__EMPTY)
         , m_stormType(STORM_TYPE__EMPTY)
      {}
      virtual ~PickingResultDetailStormAttributes() {}

      PickingResultDetailStormAttributes(const PickingResultDetailStormAttributes& pickingResultDetailStormAttributes)
         : PickingResultDetail(pickingResultDetailStormAttributes)
         , m_stormTypeHail(pickingResultDetailStormAttributes.m_stormTypeHail)
         , m_stormType(pickingResultDetailStormAttributes.m_stormType)
         , m_movementDirection( pickingResultDetailStormAttributes.m_movementDirection)
         , m_movementSpeed(pickingResultDetailStormAttributes.m_movementSpeed)
         , m_elevation(pickingResultDetailStormAttributes.m_elevation)
         , m_timeObserved(pickingResultDetailStormAttributes.m_timeObserved)
      {
      }

      PickingResultDetailStormAttributes& operator=(const PickingResultDetailStormAttributes& pickingResultDetailStormAttributes)
      {
         if(this != &pickingResultDetailStormAttributes)
         {
            PickingResultDetail::operator=(pickingResultDetailStormAttributes);
            m_stormTypeHail = pickingResultDetailStormAttributes.m_stormTypeHail;
            m_stormType = pickingResultDetailStormAttributes.m_stormType;
            m_movementDirection = pickingResultDetailStormAttributes.m_movementDirection;
            m_movementSpeed = pickingResultDetailStormAttributes.m_movementSpeed;
            m_elevation = pickingResultDetailStormAttributes.m_elevation;
            m_timeObserved = pickingResultDetailStormAttributes.m_timeObserved;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailStormAttributes& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_stormTypeHail == rhs.m_stormTypeHail) &&
                (m_stormType == rhs.m_stormType) &&
                (m_movementDirection == rhs.m_movementDirection) &&
                (m_movementSpeed == rhs.m_movementSpeed) &&
                (m_elevation == rhs.m_elevation) &&
                (m_timeObserved == rhs.m_timeObserved);
      }

      bool operator!=(const PickingResultDetailStormAttributes& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailStormAttributes payload:\n");

         stream << ::std::endl
                << " Picking result Weather: StormTypeHail = " << m_stormTypeHail.toString() << ::std::endl
                << " Picking result Weather: StormType = " << m_stormType.toString() << ::std::endl
                << " Picking result Weather: Direction = " << ::navmiddleware::toString(m_movementDirection.getValue()) << ::std::endl
                << " Picking result Weather: Speed = " << m_movementSpeed.toString() << ::std::endl
                << " Picking result Weather: Elevation = " << m_elevation.toString() << ::std::endl
                << " Picking result Weather: time Observed = " << m_timeObserved.toString() << ::std::endl
                << ::std::endl;

         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailStormAttributes(*this);
      }

      ValidValue<StormTypeHail>        m_stormTypeHail;
      ValidValue<StormType>            m_stormType;
      ValidValue<DirectionDescription> m_movementDirection;
      ValidValue< ::std::string >      m_movementSpeed;
      ValidValue< ::std::string >      m_elevation;
      ValidValue< ::std::string >      m_timeObserved;
   };


   class PickingResultDetailStormPosition : public PickingResultDetail
   {
   public:
      enum StormCategory
      {
         STORM_CATEGORY__UNKNOWN,               /**< This value defines an unknown storm category.*/
         STORM_CATEGORY__TROPICAL_DISTURBANCE,  /**< This value defines a tropical disturbance.*/
         STORM_CATEGORY__HURRICANE_CAT_1,       /**< This value defines a hurricane category 1.*/
         STORM_CATEGORY__HURRICANE_CAT_2,       /**< This value defines a hurricane category 2.*/
         STORM_CATEGORY__HURRICANE_CAT_3,       /**< This value defines a hurricane category 3.*/
         STORM_CATEGORY__HURRICANE_CAT_4,       /**< This value defines a hurricane category 4.*/
         STORM_CATEGORY__HURRICANE_CAT_5,       /**< This value defines a hurricane category 5.*/
         STORM_CATEGORY__TROPICAL_STORM,        /**< This value defines a tropical storm.*/
         STORM_CATEGORY__TROPICAL_DEPRESSION,   /**< This value defines a tropical depression.*/
         STORM_CATEGORY__TYPHOON,               /**< This value defines a typhoon.*/
         STORM_CATEGORY__SUPER_TYPHOON,         /**< This value defines a super typhoon.*/
         STORM_CATEGORY__TROPICAL_CYCLONE       /**< This value defines a tropical cyclone.*/
      };

      static ::std::string toString(StormCategory stormCategory)
      {
         switch (stormCategory)
         {
         case STORM_CATEGORY__UNKNOWN:
            return "STORM_CATEGORY__UNKNOWN";
         case STORM_CATEGORY__TROPICAL_DISTURBANCE:
            return "STORM_CATEGORY__TROPICAL_DISTURBANCE";
         case STORM_CATEGORY__HURRICANE_CAT_1:
            return "STORM_CATEGORY__HURRICANE_CAT_1";
         case STORM_CATEGORY__HURRICANE_CAT_2:
            return "STORM_CATEGORY__HURRICANE_CAT_2";
         case STORM_CATEGORY__HURRICANE_CAT_3:
            return "STORM_CATEGORY__HURRICANE_CAT_3";
         case STORM_CATEGORY__HURRICANE_CAT_4:
            return "STORM_CATEGORY__HURRICANE_CAT_4";
         case STORM_CATEGORY__HURRICANE_CAT_5:
            return "STORM_CATEGORY__HURRICANE_CAT_5";
         case STORM_CATEGORY__TROPICAL_STORM:
            return "STORM_CATEGORY__TROPICAL_STORM";
         case STORM_CATEGORY__TROPICAL_DEPRESSION:
            return "STORM_CATEGORY__TROPICAL_DEPRESSION";
         case STORM_CATEGORY__TYPHOON:
            return "STORM_CATEGORY__TYPHOON";
         case STORM_CATEGORY__SUPER_TYPHOON:
            return "STORM_CATEGORY__SUPER_TYPHOON";
         case STORM_CATEGORY__TROPICAL_CYCLONE:
            return "STORM_CATEGORY__TROPICAL_CYCLONE";
         default:
            ::std::stringstream stream;
            stream << "STORM_CATEGORY__<" << static_cast<unsigned int>(stormCategory) << ">";
            return stream.str();
         }
      }

      PickingResultDetailStormPosition() {}
      virtual ~PickingResultDetailStormPosition() {}

      PickingResultDetailStormPosition(const PickingResultDetailStormPosition& pickingResultDetailStormPosition)
         : PickingResultDetail(pickingResultDetailStormPosition)
         , m_stormCategory(pickingResultDetailStormPosition.m_stormCategory)
         , m_stormName(pickingResultDetailStormPosition.m_stormName)
         , m_movementSpeed(pickingResultDetailStormPosition.m_movementSpeed)
         , m_movementDirection(pickingResultDetailStormPosition.m_movementDirection)
         , m_maxWindSpeed(pickingResultDetailStormPosition.m_maxWindSpeed)
         , m_gusts(pickingResultDetailStormPosition.m_gusts)
         , m_pressure(pickingResultDetailStormPosition.m_pressure)
         , m_timeObserved(pickingResultDetailStormPosition.m_timeObserved)
         , m_dateObserved(pickingResultDetailStormPosition.m_dateObserved)
      {
      }

      PickingResultDetailStormPosition& operator=(const PickingResultDetailStormPosition& pickingResultDetailStormPosition)
      {

         if( this != &pickingResultDetailStormPosition)
         {
            PickingResultDetail::operator=(pickingResultDetailStormPosition);
            m_stormCategory = pickingResultDetailStormPosition.m_stormCategory;
            m_stormName = pickingResultDetailStormPosition.m_stormName;
            m_movementSpeed = pickingResultDetailStormPosition.m_movementSpeed;
            m_movementDirection = pickingResultDetailStormPosition.m_movementDirection;
            m_maxWindSpeed = pickingResultDetailStormPosition.m_maxWindSpeed;
            m_gusts = pickingResultDetailStormPosition.m_gusts;
            m_pressure = pickingResultDetailStormPosition.m_pressure;
            m_timeObserved = pickingResultDetailStormPosition.m_timeObserved;
            m_dateObserved = pickingResultDetailStormPosition.m_dateObserved;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailStormPosition& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_stormCategory == rhs.m_stormCategory) &&
                (m_stormName == rhs.m_stormName) &&
                (m_movementSpeed == rhs.m_movementSpeed) &&
                (m_movementDirection == rhs.m_movementDirection) &&
                (m_maxWindSpeed == rhs.m_maxWindSpeed) &&
                (m_gusts == rhs.m_gusts) &&
                (m_pressure == rhs.m_pressure) &&
                (m_timeObserved == rhs.m_timeObserved) &&
                (m_dateObserved == rhs.m_dateObserved);
      }

      bool operator!=(const PickingResultDetailStormPosition& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailStormPosition payload:\n");

         stream << ::std::endl
                << " Picking result Weather position: Storm Category = " << m_stormCategory.toString() << ::std::endl
                << " Picking result Weather position: Storm Name = " << m_stormName.toString() << ::std::endl
                << " Picking result Weather position: Storm Speed = " << m_movementSpeed.toString() << ::std::endl
                << " Picking result Weather position: Storm Direction = " << ::navmiddleware::toString(m_movementDirection.getValue()) << ::std::endl
                << " Picking result Weather position: Storm Max Wind Speed = " << m_maxWindSpeed.toString() << ::std::endl
                << " Picking result Weather position: Storm Wind Gust = " << m_gusts.toString() << ::std::endl
                << " Picking result Weather position: Storm Pressure = " << m_pressure.toString() << ::std::endl
                << " Picking result Weather position: Time Observed = " << m_timeObserved.toString() << ::std::endl
                << " Picking result Weather position: Date Observed = " << m_dateObserved.toString() << ::std::endl
                << ::std::endl;

         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailStormPosition(*this);
      }

      ValidValue< StormCategory >      m_stormCategory;
      ValidValue< ::std::string >      m_stormName;
      ValidValue< ::std::string >      m_movementSpeed;
      ValidValue<DirectionDescription> m_movementDirection;
      ValidValue< ::std::string >      m_maxWindSpeed;
      ValidValue< ::std::string >      m_gusts;
      ValidValue< ::std::string >      m_pressure;
      ValidValue< ::std::string >      m_timeObserved;
      ValidValue< ::std::string >      m_dateObserved;
   };

   class PickingResultDetailWeatherAlert : public PickingResultDetail
   {
   public:
      PickingResultDetailWeatherAlert() {}
      virtual ~PickingResultDetailWeatherAlert() {}

      PickingResultDetailWeatherAlert(const PickingResultDetailWeatherAlert& pickingResultDetailWeatherAlert)
         : PickingResultDetail(pickingResultDetailWeatherAlert)
         , m_startTime(pickingResultDetailWeatherAlert.m_startTime)
         , m_endTime(pickingResultDetailWeatherAlert.m_endTime)
         , m_startDate(pickingResultDetailWeatherAlert.m_startDate)
         , m_endDate(pickingResultDetailWeatherAlert.m_endDate)
         , m_weatherAlertText(pickingResultDetailWeatherAlert.m_weatherAlertText)
         , m_categoryIdList(pickingResultDetailWeatherAlert.m_categoryIdList)
         , m_alertTypeIdList(pickingResultDetailWeatherAlert.m_alertTypeIdList)
      {
      }

      PickingResultDetailWeatherAlert& operator=(const PickingResultDetailWeatherAlert& pickingResultDetailWeatherAlert)
      {
         if(this != &pickingResultDetailWeatherAlert)
         {
            PickingResultDetail::operator=(pickingResultDetailWeatherAlert);
            m_startTime = pickingResultDetailWeatherAlert.m_startTime;
            m_endTime = pickingResultDetailWeatherAlert.m_endTime;
            m_startDate = pickingResultDetailWeatherAlert.m_startDate;
            m_endDate = pickingResultDetailWeatherAlert.m_endDate;
            m_weatherAlertText = pickingResultDetailWeatherAlert.m_weatherAlertText;
            m_categoryIdList = pickingResultDetailWeatherAlert.m_categoryIdList;
            m_alertTypeIdList = pickingResultDetailWeatherAlert.m_alertTypeIdList;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailWeatherAlert& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_startTime == rhs.m_startTime) &&
                (m_endTime == rhs.m_endTime) &&
                (m_startDate == rhs.m_startDate) &&
                (m_endDate == rhs.m_endDate) &&
                (m_weatherAlertText == rhs.m_weatherAlertText) &&
                (m_categoryIdList == rhs.m_categoryIdList) &&
                (m_alertTypeIdList == rhs.m_alertTypeIdList);
      }

      bool operator!=(const PickingResultDetailWeatherAlert& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailWeatherAlert payload:\n");

         stream << ::std::endl
                << " Picking result weather alert text= " << m_weatherAlertText.toString() << ::std::endl
                << " Picking result weather alert time-stamp start = " << m_startTime.toString() << "End time = "<< m_endTime.getValue() << ::std::endl
                << " Picking result weather alert start date = " << m_startDate.toString() << "End date = "<< m_endDate.toString() << ::std::endl
                << ::std::endl;

         if(!m_categoryIdList.empty())
         {
            ::std::vector< uint32_t >::const_iterator categoryIdListIterator =  m_categoryIdList.begin();
            ::std::vector< uint32_t >::const_iterator categoryIdListIteratorEnd =  m_categoryIdList.end();
            for(; categoryIdListIterator !=  categoryIdListIteratorEnd; ++categoryIdListIterator)
            {
               stream << ::std::endl
                      << " Category Id Details = " << *categoryIdListIterator << ::std::endl
                      << ::std::endl;
            }
         }
         else
         {
            stream << "No categoryId found !!! "<< ::std::endl;
         }

         if(!m_alertTypeIdList.empty())
         {
            ::std::vector< uint32_t >::const_iterator alertTypeIdListIterator =  m_alertTypeIdList.begin();
            ::std::vector< uint32_t >::const_iterator alertTypeIdListIteratorEnd =  m_alertTypeIdList.end();
            for(; alertTypeIdListIterator !=  alertTypeIdListIteratorEnd; ++alertTypeIdListIterator)
            {
               stream << ::std::endl
                      << " Alert Type Id Details = " << *alertTypeIdListIterator << ::std::endl
                      << ::std::endl;
            }
         }
         else
         {
            stream << "No AlertTypeIdList found !!! "<< ::std::endl;
         }

         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailWeatherAlert(*this);
      }

      ValidValue< ::std::string >          m_startTime;           /**< start time of the picked weather alert*/
      ValidValue< ::std::string >          m_endTime;             /**< end time of the picked weather alert*/
      ValidValue< ::std::string >          m_startDate;           /**< start date of the picked weather alert*/
      ValidValue< ::std::string >          m_endDate;             /**< end date of the picked weather alert*/
      ValidValue< ::std::string >          m_weatherAlertText;    /**< text to display for the picked weather alert*/
      ::std::vector<uint32_t>              m_categoryIdList;      /**< list of the weather alert category ids*/
      ::std::vector<uint32_t>              m_alertTypeIdList;     /**< list of the weather alert type ids*/
   };

   class PickingResultDetailWeatherIcon : public PickingResultDetail
   {
   public:
      PickingResultDetailWeatherIcon() {}
      virtual ~PickingResultDetailWeatherIcon() {}

      PickingResultDetailWeatherIcon(const PickingResultDetailWeatherIcon& pickingResultDetailWeatherIcon)
         : PickingResultDetail(pickingResultDetailWeatherIcon)
         , m_weatherDataInfo(pickingResultDetailWeatherIcon.m_weatherDataInfo)
      {}

      const WeatherDataInfo& getWeatherDataInfo() const
      {
         return m_weatherDataInfo;
      }

      void setWeatherDataInfo(const WeatherDataInfo& weatherDataInfo)
      {
         m_weatherDataInfo = weatherDataInfo;
      }

      PickingResultDetailWeatherIcon& operator=(const PickingResultDetailWeatherIcon& pickingResultDetailWeatherIcon)
      {
         if (this != &pickingResultDetailWeatherIcon)
         {
            PickingResultDetail::operator=(pickingResultDetailWeatherIcon);
            m_weatherDataInfo = pickingResultDetailWeatherIcon.m_weatherDataInfo;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailWeatherIcon& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_weatherDataInfo == rhs.m_weatherDataInfo);
      }

      bool operator!=(const PickingResultDetailWeatherIcon& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailWeatherIcon payload:\n");
         stream << std::endl
                << "Weather data infos = " << m_weatherDataInfo.toString()
                << std::endl;
         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailWeatherIcon(*this);
      }

   private:
      WeatherDataInfo m_weatherDataInfo;
   };

   class PickingResultDetailArea : public PickingResultDetail
   {
   public:
      PickingResultDetailArea()
      {}
      virtual ~PickingResultDetailArea()
      {}

      bool operator==(const PickingResultDetailArea& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_locationAttributeInfos == rhs.m_locationAttributeInfos);
      }

      bool operator!=(const PickingResultDetailArea& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailArea payload:\n");
         stream << "Location attribute Infos = " << m_locationAttributeInfos.toString() << ::std::endl;
         return stream.str();
      }

      virtual PickingResultDetailArea* clone()
      {
         return new PickingResultDetailArea(*this);
      }

      LocationAttributeInfos m_locationAttributeInfos;
   };

   class PickingResultDetailSafetyCamera : public PickingResultDetail
   {
   public:

      PickingResultDetailSafetyCamera()
         :m_isOnlineSource(false)
      {
      }
      virtual ~PickingResultDetailSafetyCamera() {}

      PickingResultDetailSafetyCamera(const PickingResultDetailSafetyCamera& pickingResultDetailSafetyCamera)
         : PickingResultDetail(pickingResultDetailSafetyCamera)
         , m_isOnlineSource(pickingResultDetailSafetyCamera.m_isOnlineSource)
         , m_safetyCameraType(pickingResultDetailSafetyCamera.m_safetyCameraType)
         , m_safetyCameraInfo(pickingResultDetailSafetyCamera.m_safetyCameraInfo)
         , m_allowedSpeed(pickingResultDetailSafetyCamera.m_allowedSpeed)
      {
      }

      PickingResultDetailSafetyCamera& operator=(const PickingResultDetailSafetyCamera& rhs)
      {
         if (this != &rhs)
         {
            PickingResultDetail::operator=(rhs);
            m_isOnlineSource = rhs.m_isOnlineSource;
            m_safetyCameraType = rhs.m_safetyCameraType;
            m_safetyCameraInfo = rhs.m_safetyCameraInfo;
            m_allowedSpeed = rhs.m_allowedSpeed;
         }
         return *this;
      }

      bool operator==(const PickingResultDetailSafetyCamera& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_isOnlineSource == rhs.m_isOnlineSource) &&
                (m_safetyCameraType == rhs.m_safetyCameraType) &&
                (m_safetyCameraInfo == rhs.m_safetyCameraInfo) &&
                (m_allowedSpeed == rhs.m_allowedSpeed);
      }

      bool operator!=(const PickingResultDetailSafetyCamera& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailDestination payload:\n");
         stream << "Online Source = " << m_isOnlineSource << ", "
                << "Safety Camera Type = " << ::navmiddleware::toString(m_safetyCameraType.getValueOr(SAFETY_CAMERA_TYPE__UNKNOWN)) << ", "
                << "Safety Camera Info = " << m_safetyCameraInfo.toString() << ", "
                << "Allowed Speed = " << m_allowedSpeed.toString()
                << ::std::endl;
         return stream.str();
      }

      virtual PickingResultDetail* clone()
      {
         return new PickingResultDetailSafetyCamera(*this);
      }

      bool                                           m_isOnlineSource;
      ValidValue< SafetyCameraType >                 m_safetyCameraType;
      ValidValue< ::std::string >                    m_safetyCameraInfo;
      ValidValue< int16_t >                          m_allowedSpeed;
   };

   class PickingResultDetailDynamicIcon : public PickingResultDetail
   {
   public:
      PickingResultDetailDynamicIcon()
      {}
      virtual ~PickingResultDetailDynamicIcon()
      {}

      bool operator==(const PickingResultDetailDynamicIcon& rhs) const
      {
         return (PickingResultDetail::operator==(rhs)) &&
                (m_dynamicIconId == rhs.m_dynamicIconId);
      }

      bool operator!=(const PickingResultDetailDynamicIcon& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("PickingResultDetailDynamicIcon payload:\n");
         stream << "DynamicIcon = " << (m_dynamicIconId.isValid() ?
                                        m_dynamicIconId.getValue().toString() : "Invalid" ) << ::std::endl;
         return stream.str();
      }

      virtual PickingResultDetailDynamicIcon* clone()
      {
         return new PickingResultDetailDynamicIcon(*this);
      }

      ValidValue< ::navmiddleware::MapDynamicIconId>      m_dynamicIconId;
   };

   class PickingResult
   {
   public:
      PickingResult()
         : m_elementType(PICKING_RESULT_TYPE__INVALID)
      {
         m_pickingResultDetailData.m_pickingResultDetail = NULL;
      }

      PickingResult(const PickingResult& pickingResult)
         : m_title(pickingResult.m_title)
         , m_elementType(pickingResult.m_elementType)
         , m_geoCoordinate(pickingResult.m_geoCoordinate)
         , m_airDistance(pickingResult.m_airDistance)
         , m_direction(pickingResult.m_direction)
         , m_locationId(pickingResult.m_locationId)
      {
         PickingResultDetail* pickingResultDetail = pickingResult.getPickingResultDetail();
         if(pickingResultDetail)
         {
            m_pickingResultDetailData.m_pickingResultDetail = pickingResultDetail->clone();
         }
         else
         {
            m_pickingResultDetailData.m_pickingResultDetail = NULL;
         }
      }

      PickingResult& operator=(const PickingResult& pickingResult)
      {
         if (this != &pickingResult)
         {
            if(NULL != m_pickingResultDetailData.m_pickingResultDetail)
            {
               delete m_pickingResultDetailData.m_pickingResultDetail;
               m_pickingResultDetailData.m_pickingResultDetail = NULL;
            }

            m_title = pickingResult.m_title;
            m_elementType = pickingResult.m_elementType;
            m_geoCoordinate = pickingResult.m_geoCoordinate;
            m_airDistance = pickingResult.m_airDistance;
            m_direction = pickingResult.m_direction;
            m_locationId = pickingResult.m_locationId;

            PickingResultDetail* pickingResultDetail = pickingResult.getPickingResultDetail();
            if(pickingResultDetail)
            {
               m_pickingResultDetailData.m_pickingResultDetail = pickingResultDetail->clone();
            }
         }
         return *this;
      }

      bool operator==(const PickingResult& rhs) const
      {
         bool isEqual = false;
         if ((m_title == rhs.m_title) && (m_elementType == rhs.m_elementType)
               && (m_geoCoordinate == rhs.m_geoCoordinate) && (m_airDistance == rhs.m_airDistance)
               && (m_direction == rhs.m_direction)
               && (m_locationId == rhs.m_locationId))
         {
            switch (m_elementType)
            {
            case PICKING_RESULT_TYPE__ROAD:
            {
               PickingResultDetailRoad* roadDetailLhs = getPickingResultDetailRoad();
               PickingResultDetailRoad* roadDetailRhs = rhs.getPickingResultDetailRoad();
               if(roadDetailLhs && roadDetailRhs)
               {
                  isEqual = (*roadDetailLhs == *roadDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__POI:
            {
               PickingResultDetailPoi* poiDetailLhs = getPickingResultDetailPoi();
               PickingResultDetailPoi* poiDetailRhs = rhs.getPickingResultDetailPoi();
               if(poiDetailLhs && poiDetailRhs)
               {
                  isEqual = (*poiDetailLhs == *poiDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__ADDRESS_BOOK:
            {
               PickingResultDetailAddressBookPoi* addressBookPoiDetailLhs = getPickingResultDetailAddressBookPoi();
               PickingResultDetailAddressBookPoi* addressBookPoiDetailRhs = rhs.getPickingResultDetailAddressBookPoi();
               if(addressBookPoiDetailLhs && addressBookPoiDetailRhs)
               {
                  isEqual = (*addressBookPoiDetailLhs == *addressBookPoiDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__TRAFFIC_ICON:
            {
               PickingResultDetailTm* tmDetailLhs = getPickingResultDetailTm();
               PickingResultDetailTm* tmDetailRhs = rhs.getPickingResultDetailTm();
               if(tmDetailLhs && tmDetailRhs)
               {
                  isEqual = (*tmDetailLhs == *tmDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__DESTINATION:
            {
               PickingResultDetailDestination* destinationDetailLhs
                  = getPickingResultDetailDestination();
               PickingResultDetailDestination* destinationDetailRhs
                  = rhs.getPickingResultDetailDestination();
               if(destinationDetailLhs && destinationDetailRhs)
               {
                  isEqual = (*destinationDetailLhs == *destinationDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__STORM_ATTRIBUTES:
            {
               PickingResultDetailStormAttributes* stormAttributesDetailLhs = getPickingResultDetailStormAttributes();
               PickingResultDetailStormAttributes* stormAttributesDetailRhs = rhs.getPickingResultDetailStormAttributes();
               if(stormAttributesDetailLhs && stormAttributesDetailRhs)
               {
                  isEqual = (*stormAttributesDetailLhs == *stormAttributesDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__STORM_TRACK_POSITION:
            {
               PickingResultDetailStormPosition* stormPositionDetailLhs = getPickingResultDetailStormPosition();
               PickingResultDetailStormPosition* stormPositionDetailRhs = rhs.getPickingResultDetailStormPosition();
               if(stormPositionDetailLhs && stormPositionDetailRhs)
               {
                  isEqual = (*stormPositionDetailLhs == *stormPositionDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__WEATHER_ALERT:
            {
               PickingResultDetailWeatherAlert* weatherAlertDetailLhs = getPickingResultDetailWeatherAlert();
               PickingResultDetailWeatherAlert* weatherAlertDetailRhs = rhs.getPickingResultDetailWeatherAlert();
               if(weatherAlertDetailLhs && weatherAlertDetailRhs)
               {
                  isEqual = (*weatherAlertDetailLhs == *weatherAlertDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__WEATHER_ICON:
            {
               PickingResultDetailWeatherIcon* weatherIconDetailLhs = getPickingResultDetailWeatherIcon();
               PickingResultDetailWeatherIcon* weatherIconDetailRhs = rhs.getPickingResultDetailWeatherIcon();
               if (weatherIconDetailLhs && weatherIconDetailRhs)
               {
                  isEqual = (*weatherIconDetailLhs == *weatherIconDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__AREA:
            {
               PickingResultDetailArea* areaDetailLhs = getPickingResultDetailArea();
               PickingResultDetailArea* areaDetailRhs = rhs.getPickingResultDetailArea();
               if (areaDetailLhs && areaDetailRhs)
               {
                  isEqual = (*areaDetailLhs == *areaDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON:
            {
               PickingResultDetailSafetyCamera* safetyCameraDetailLhs = getPickingResultDetailSafetyCamera();
               PickingResultDetailSafetyCamera* safetyCameraDetailRhs = rhs.getPickingResultDetailSafetyCamera();
               if (safetyCameraDetailLhs && safetyCameraDetailRhs)
               {
                  isEqual = (*safetyCameraDetailLhs == *safetyCameraDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__VICS_PARKING:
            {
               PickingResultDetailVicsParking* tmDetailLhs = getPickingResultDetailVicsParking();
               PickingResultDetailVicsParking* tmDetailRhs = rhs.getPickingResultDetailVicsParking();
               if ( tmDetailLhs && tmDetailRhs )
               {
                  isEqual = (*tmDetailLhs == *tmDetailRhs);
               }
               break;
            }
            case PICKING_RESULT_TYPE__DYNAMIC_ICON:
            {
               PickingResultDetailDynamicIcon* dynamicIconDetailLhs = getPickingResultDetailDynamicIcon();
               PickingResultDetailDynamicIcon* dynamicIconDetailRhs = rhs.getPickingResultDetailDynamicIcon();
               if ( dynamicIconDetailLhs && dynamicIconDetailRhs )
               {
                  isEqual = (*dynamicIconDetailLhs == *dynamicIconDetailRhs);
               }
               break;
            }
            default:
               break;
            }
         }
         return isEqual;
      }

      bool operator!=(const PickingResult& rhs) const
      {
         return !(*this == rhs);
      }

      ~PickingResult()
      {
         if(NULL != m_pickingResultDetailData.m_pickingResultDetail)
         {
            delete m_pickingResultDetailData.m_pickingResultDetail;
            m_pickingResultDetailData.m_pickingResultDetail = NULL;
         }
      }


      /**getters for the PickingResultDetailData pointers*/

      /** NOTE: Only if the element type of the PickingResult matches the detail type,
       * a valid pointer to the picking result detail is returned.
       * If there is a mis-match, NULL will be returned.
       */

      PickingResultDetail* getPickingResultDetail() const
      {
         return m_pickingResultDetailData.m_pickingResultDetail;
      }

      PickingResultDetailRoad* getPickingResultDetailRoad() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__ROAD)
         {
            return m_pickingResultDetailData.m_pickingResultDetailRoad;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailRoad* getPickingResultDetailRoad()
      {
         if(m_elementType == PICKING_RESULT_TYPE__ROAD)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailRoad )
            {
               m_pickingResultDetailData.m_pickingResultDetailRoad = new PickingResultDetailRoad();
            }
            return m_pickingResultDetailData.m_pickingResultDetailRoad;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailPoi* getPickingResultDetailPoi() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__POI)
         {
            return m_pickingResultDetailData.m_pickingResultDetailPoi;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailPoi* getPickingResultDetailPoi()
      {
         if(m_elementType == PICKING_RESULT_TYPE__POI)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailPoi )
            {
               m_pickingResultDetailData.m_pickingResultDetailPoi = new PickingResultDetailPoi();
            }
            return m_pickingResultDetailData.m_pickingResultDetailPoi;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailAddressBookPoi* getPickingResultDetailAddressBookPoi() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__ADDRESS_BOOK)
         {
            return m_pickingResultDetailData.m_pickingResultDetailAddressBookPoi;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailAddressBookPoi* getPickingResultDetailAddressBookPoi()
      {
         if(m_elementType == PICKING_RESULT_TYPE__ADDRESS_BOOK)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailAddressBookPoi )
            {
               m_pickingResultDetailData.m_pickingResultDetailAddressBookPoi = new PickingResultDetailAddressBookPoi();
            }
            return m_pickingResultDetailData.m_pickingResultDetailAddressBookPoi;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailTm* getPickingResultDetailTm() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__TRAFFIC_ICON)
         {
            return m_pickingResultDetailData.m_pickingResultDetailTm;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailTm* getPickingResultDetailTm()
      {
         if(m_elementType == PICKING_RESULT_TYPE__TRAFFIC_ICON)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailTm )
            {
               m_pickingResultDetailData.m_pickingResultDetailTm = new PickingResultDetailTm();
            }
            return m_pickingResultDetailData.m_pickingResultDetailTm;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailDestination* getPickingResultDetailDestination() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__DESTINATION)
         {
            return m_pickingResultDetailData.m_pickingResultDetailDestination;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailDestination* getPickingResultDetailDestination()
      {
         if(m_elementType == PICKING_RESULT_TYPE__DESTINATION)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailDestination )
            {
               m_pickingResultDetailData.m_pickingResultDetailDestination = new PickingResultDetailDestination();
            }
            return m_pickingResultDetailData.m_pickingResultDetailDestination;
         }
         else
         {
            return NULL;
         }
      }


      PickingResultDetailStormAttributes* getPickingResultDetailStormAttributes() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__STORM_ATTRIBUTES)
         {
            return m_pickingResultDetailData.m_pickingResultDetailStormAttributes;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailStormAttributes* getPickingResultDetailStormAttributes()
      {
         if(m_elementType == PICKING_RESULT_TYPE__STORM_ATTRIBUTES)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailStormAttributes )
            {
               m_pickingResultDetailData.m_pickingResultDetailStormAttributes = new PickingResultDetailStormAttributes();
            }
            return m_pickingResultDetailData.m_pickingResultDetailStormAttributes;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailStormPosition* getPickingResultDetailStormPosition() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__STORM_TRACK_POSITION)
         {
            return m_pickingResultDetailData.m_pickingResultDetailStormPosition;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailStormPosition* getPickingResultDetailStormPosition()
      {
         if(m_elementType == PICKING_RESULT_TYPE__STORM_TRACK_POSITION)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailStormPosition )
            {
               m_pickingResultDetailData.m_pickingResultDetailStormPosition = new PickingResultDetailStormPosition();
            }
            return m_pickingResultDetailData.m_pickingResultDetailStormPosition;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailWeatherAlert* getPickingResultDetailWeatherAlert() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__WEATHER_ALERT)
         {
            return m_pickingResultDetailData.m_pickingResultDetailWeatherAlert;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailWeatherAlert* getPickingResultDetailWeatherAlert()
      {
         if(m_elementType == PICKING_RESULT_TYPE__WEATHER_ALERT)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailWeatherAlert )
            {
               m_pickingResultDetailData.m_pickingResultDetailWeatherAlert = new PickingResultDetailWeatherAlert();
            }
            return m_pickingResultDetailData.m_pickingResultDetailWeatherAlert;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailWeatherIcon* getPickingResultDetailWeatherIcon() const
      {
         if(m_elementType == PICKING_RESULT_TYPE__WEATHER_ICON)
         {
            return m_pickingResultDetailData.m_pickingResultDetailWeatherIcon;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailWeatherIcon* getPickingResultDetailWeatherIcon()
      {
         if(m_elementType == PICKING_RESULT_TYPE__WEATHER_ICON)
         {
            if( !m_pickingResultDetailData.m_pickingResultDetailWeatherIcon )
            {
               m_pickingResultDetailData.m_pickingResultDetailWeatherIcon = new PickingResultDetailWeatherIcon();
            }
            return m_pickingResultDetailData.m_pickingResultDetailWeatherIcon;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailArea* getPickingResultDetailArea() const
      {
         if (m_elementType == PICKING_RESULT_TYPE__AREA)
         {
            return m_pickingResultDetailData.m_pickingResultDetailArea;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailArea* getPickingResultDetailArea()
      {
         if (m_elementType == PICKING_RESULT_TYPE__AREA)
         {
            if (!m_pickingResultDetailData.m_pickingResultDetailArea)
            {
               m_pickingResultDetailData.m_pickingResultDetailArea = new PickingResultDetailArea();
            }
            return m_pickingResultDetailData.m_pickingResultDetailArea;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailSafetyCamera* getPickingResultDetailSafetyCamera() const
      {
         if (m_elementType == PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON)
         {
            return m_pickingResultDetailData.m_pickingResultDetailSafetyCamera;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailSafetyCamera* getPickingResultDetailSafetyCamera()
      {
         if (m_elementType == PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON)
         {
            if (!m_pickingResultDetailData.m_pickingResultDetailSafetyCamera)
            {
               m_pickingResultDetailData.m_pickingResultDetailSafetyCamera = new PickingResultDetailSafetyCamera();
            }
            return m_pickingResultDetailData.m_pickingResultDetailSafetyCamera;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailVicsParking* getPickingResultDetailVicsParking() const
      {
         if ( m_elementType == PICKING_RESULT_TYPE__VICS_PARKING )
         {
            return m_pickingResultDetailData.m_pickingResultDetailVicsParking;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailVicsParking* getPickingResultDetailVicsParking()
      {
         if ( m_elementType == PICKING_RESULT_TYPE__VICS_PARKING )
         {
            if ( !m_pickingResultDetailData.m_pickingResultDetailVicsParking )
            {
               m_pickingResultDetailData.m_pickingResultDetailVicsParking = new PickingResultDetailVicsParking();
            }
            return m_pickingResultDetailData.m_pickingResultDetailVicsParking;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailDynamicIcon* getPickingResultDetailDynamicIcon() const
      {
         if ( m_elementType == PICKING_RESULT_TYPE__DYNAMIC_ICON )
         {
            return m_pickingResultDetailData.m_pickingResultDetailDynamicIcon;
         }
         else
         {
            return NULL;
         }
      }

      PickingResultDetailDynamicIcon* getPickingResultDetailDynamicIcon()
      {
         if ( m_elementType == PICKING_RESULT_TYPE__DYNAMIC_ICON )
         {
            if ( !m_pickingResultDetailData.m_pickingResultDetailDynamicIcon )
            {
               m_pickingResultDetailData.m_pickingResultDetailDynamicIcon = new PickingResultDetailDynamicIcon();
            }
            return m_pickingResultDetailData.m_pickingResultDetailDynamicIcon;
         }
         else
         {
            return NULL;
         }
      }

      void setPickingResultDetail(PickingResultDetail* pickingResultDetail)
      {
         if(pickingResultDetail)
         {
            if(NULL != m_pickingResultDetailData.m_pickingResultDetail)
            {
               delete m_pickingResultDetailData.m_pickingResultDetail;
               m_pickingResultDetailData.m_pickingResultDetail = NULL;
            }
            m_pickingResultDetailData.m_pickingResultDetail = pickingResultDetail->clone();
         }
      }

      ValidValue< ::std::string >                              m_title;          /**< title for the picked element*/
      PickingResultType                                        m_elementType;    /**< type of the picked element*/
      GeoCoordinateDegree                                      m_geoCoordinate;  /**< geocoordinate of the picked element*/

      /** Air distance to the picked element, relative to the car position.*/
      ValidValue< ::std::string >                              m_airDistance;
      /** Direction to the picked element, relative to the car heading.*/
      ValidValue<DirectionDescription>                         m_direction;
      /** [JP]: Unique id for the location, used to identify the picked element*/
      ValidValue< LocationId >                                 m_locationId;

   private:
      union PickingResultDetailData
      {
         PickingResultDetail*                    m_pickingResultDetail;
         PickingResultDetailRoad*                m_pickingResultDetailRoad;
         PickingResultDetailPoi*                 m_pickingResultDetailPoi;
         PickingResultDetailAddressBookPoi*      m_pickingResultDetailAddressBookPoi;
         PickingResultDetailTm*                  m_pickingResultDetailTm;
         PickingResultDetailDestination*         m_pickingResultDetailDestination;
         PickingResultDetailStormAttributes*     m_pickingResultDetailStormAttributes;
         PickingResultDetailStormPosition*       m_pickingResultDetailStormPosition;
         PickingResultDetailWeatherAlert*        m_pickingResultDetailWeatherAlert;
         PickingResultDetailWeatherIcon*         m_pickingResultDetailWeatherIcon;
         PickingResultDetailArea*                m_pickingResultDetailArea;
         PickingResultDetailSafetyCamera*        m_pickingResultDetailSafetyCamera;
         PickingResultDetailVicsParking*         m_pickingResultDetailVicsParking;
         PickingResultDetailDynamicIcon*         m_pickingResultDetailDynamicIcon;
      } m_pickingResultDetailData;
   };

   void setPickingResultList(const ::std::vector<PickingResult>& pickingResultList)
   {
      m_pickingResultList = pickingResultList;
   }

   const ::std::vector<PickingResult>& getPickingResultList() const
   {
      return m_pickingResultList;
   }

   ::std::vector<PickingResult>& getPickingResultListMutable()
   {
      return m_pickingResultList;
   }

   void setPickedGeoCoordinate (const GeoCoordinateDegree& coord)
   {
      m_pickedGeoCoordinate = coord;
   }

   const GeoCoordinateDegree& getPickedGeoCoordinate() const
   {
      return m_pickedGeoCoordinate;
   }

   const ValidValue< ::std::string >& getAirDistance() const
   {
      return m_airDistance;
   }

   void setAirDistance(const ValidValue< ::std::string >& airDistance)
   {
      m_airDistance = airDistance;
   }

   const ValidValue<DirectionDescription>& getDirection() const
   {
      return m_direction;
   }

   void setDirection(const ValidValue<DirectionDescription>& direction)
   {
      m_direction = direction;
   }

   ::std::string getLatitudeAsString() const
   {
      ::std::stringstream stringStream;
      stringStream << m_pickedGeoCoordinate.getLatitude();
      return stringStream.str();
   }

   ::std::string getLongitudeAsString() const
   {
      ::std::stringstream stringStream;
      stringStream << m_pickedGeoCoordinate.getLongitude();
      return stringStream.str();
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setRequestId(const ValidValue<RequestId>& requestId)
   {
      m_requestId = requestId;
   }

   const ValidValue<RequestId>& getRequestId() const
   {
      return m_requestId;
   }

   void setPickingError(const ValidValue<PickingError>& pickingError)
   {
      m_pickingError = pickingError;
   }

   const ValidValue<PickingError>& getPickingError() const
   {
      return m_pickingError;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PickingResults payload:\n");

      stream << "PickingResult coordinates = " << m_pickedGeoCoordinate.toString() << ::std::endl
             << "PickingResult RequestId = " << m_requestId.toString() << ::std::endl
             << "MapViewId = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;

      if(m_pickingError.isValid())
      {
         stream << "PickingResult ErrorCode = " << m_pickingError.toString() << ::std::endl;
         return stream.str();
      }


      for(::std::vector<PickingResult>::const_iterator pickingResultIter =  m_pickingResultList.begin();
            pickingResultIter != m_pickingResultList.end(); ++pickingResultIter)
      {
         const navmiddleware::PickingResults::PickingResult pickingResult = *pickingResultIter;
         stream << "Picking Result list index = "<< pickingResultIter - m_pickingResultList.begin() << ::std::endl
                << "Picking Result element type = " << pickingResult.m_elementType << ::std::endl
                << "Picking Result element title = " << pickingResult.m_title.getValueOr("") << ::std::endl;

         if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__POI )
         {
            PickingResultDetailPoi* pickingResultDetailPoi = pickingResult.getPickingResultDetailPoi();
            if(pickingResultDetailPoi)
            {
               stream << "Picking Results POI info : " << pickingResultDetailPoi->toString() << ::std::endl;
            }

         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__ADDRESS_BOOK)
         {
            PickingResultDetailAddressBookPoi* addressbookDetail = pickingResult.getPickingResultDetailAddressBookPoi();
            if(addressbookDetail)
            {
               stream << "Picking Results Address Book info : " << addressbookDetail->toString() << ::std::endl;
            }
         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__ROAD )
         {
            PickingResultDetailRoad* pickingResultDetailRoad = pickingResult.getPickingResultDetailRoad();
            if(pickingResultDetailRoad)
            {
               stream << "Picking Results Road info : " << pickingResultDetailRoad->toString() << ::std::endl;
            }
         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__TRAFFIC_ICON )
         {
            PickingResultDetailTm* pickingResultDetailTM = pickingResult.getPickingResultDetailTm();
            if(pickingResultDetailTM)
            {
               stream << " Picking Result TM Info: " << pickingResultDetailTM->toString() << ::std::endl;
            }
         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__DESTINATION )
         {
            PickingResultDetailDestination* pickingResultDetailDestination = pickingResult.getPickingResultDetailDestination();
            if(pickingResultDetailDestination)
            {
               stream << " Picking Result Destination Info: "
                      << pickingResultDetailDestination->toString() << ::std::endl;
            }
         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__CARSOR)
         {
            stream << " Picking Result Carsor Info: " << m_pickedGeoCoordinate.toString() << ::std::endl;
         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__STORM_ATTRIBUTES)
         {
            PickingResultDetailStormAttributes* pickingResultDetailStormAttributes = pickingResult.getPickingResultDetailStormAttributes();
            if(pickingResultDetailStormAttributes)
            {
               stream << " Picking Result Storm Attributes Info: " << pickingResultDetailStormAttributes->toString() << ::std::endl;
            }
         }
         else if(pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__STORM_TRACK_POSITION)
         {
            PickingResultDetailStormPosition* pickingResultDetailStormPosition = pickingResult.getPickingResultDetailStormPosition();
            if(pickingResultDetailStormPosition)
            {
               stream << " Picking Result Storm Position Info: " << pickingResultDetailStormPosition->toString() << ::std::endl;
            }
         }
         else if (pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__AREA)
         {
            PickingResultDetailArea* pickingResultDetailArea = pickingResult.getPickingResultDetailArea();
            if(pickingResultDetailArea)
            {
               stream << " Picking Result Area Info: " << pickingResultDetailArea->toString() << ::std::endl;
            }
         }
         else if (pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__SAFETY_CAMERA_ICON)
         {
            PickingResultDetailSafetyCamera* pickingResultDetailSafetyCamera = pickingResult.getPickingResultDetailSafetyCamera();
            if(pickingResultDetailSafetyCamera)
            {
               stream << " Picking Result Safety camera Info: " << pickingResultDetailSafetyCamera->toString() << ::std::endl;
            }
         }
         else if (pickingResult.m_elementType == ::navmiddleware::PICKING_RESULT_TYPE__DYNAMIC_ICON)
         {
            PickingResultDetailDynamicIcon* pickingResultDetailDynamicIcon = pickingResult.getPickingResultDetailDynamicIcon();
            if(pickingResultDetailDynamicIcon)
            {
               stream << " Picking Result DynamicIcon Info: " << pickingResultDetailDynamicIcon->toString() << ::std::endl;
            }
         }
         else
         {
            stream << "INVALID PICKING ELEMENT TYPE";
         }
      }
      return stream.str();
   }

private:
   ::std::vector<PickingResult>                             m_pickingResultList;    /**< list of picked elements*/
   GeoCoordinateDegree                                      m_pickedGeoCoordinate;  /**< geocoordinate of the picked position*/

   /** Air distance to the picked geocoordinate, relative to the car position.*/
   ValidValue< ::std::string >                              m_airDistance;
   /** Direction to the picked geocoordinate, relative to the car heading.*/
   ValidValue<DirectionDescription>                         m_direction;
   MapViewId                                                m_mapViewId;
   ValidValue<RequestId>                                    m_requestId;
   ValidValue<PickingError>                                 m_pickingError;
};

class PickingResultFilter
{
public:
   enum PickingResultFilterType
   {
      PICKING_RESULT_FILTER_TYPE__ALL,          /**< include all matching results*/
      PICKING_RESULT_FILTER_TYPE__FIRST_MATCH   /**< include only the first matched result*/
   };

   ::std::string toString(PickingResultFilterType pickingResultFilterType) const
   {
      switch (pickingResultFilterType)
      {
      case PICKING_RESULT_FILTER_TYPE__ALL:
         return "PICKING_RESULT_FILTER_TYPE__ALL";
      case PICKING_RESULT_FILTER_TYPE__FIRST_MATCH:
         return "PICKING_RESULT_FILTER_TYPE__FIRST_MATCH";
      default:
         ::std::stringstream stream;
         stream << "PICKING_RESULT_FILTER_TYPE__<" << static_cast<unsigned int>(pickingResultFilterType) << ">";
         return stream.str();
      }
   }

   PickingResultFilter(): m_filterList(), m_filterType(PICKING_RESULT_FILTER_TYPE__ALL)
   {
   }

   void setFilterList(const ::std::vector< PickingResultType>& filterList)
   {
      m_filterList = filterList;
   }

   const ::std::vector< PickingResultType>& getFilterList() const
   {
      return m_filterList;
   }

   void setFilterType(PickingResultFilterType filterType)
   {
      m_filterType = filterType;
   }

   PickingResultFilterType getFilterType() const
   {
      return m_filterType;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PickingResultFilter payload:\n");

      stream << ::std::endl
             << "PickingResultFilterType FilterType = " << toString(m_filterType) << std::endl
             << "PickingResultFilterType PickingResultType = [" << ::std::endl;

      if(!m_filterList.empty())
      {
         ::std::vector< PickingResultType >::const_iterator filterListIter;
         for(filterListIter = m_filterList.begin();
               filterListIter !=  m_filterList.end();
               ++filterListIter)
         {
            stream << " ' " << ::navmiddleware::toString(*filterListIter) << " ',"<< ::std::endl;
         }
      }
      else
      {
         stream << "PickingResultType Filter List is Empty!!! "<< ::std::endl;
      }
      stream << "]" << ::std::endl;
      return stream.str();
   }

private:
   ::std::vector<PickingResultType>    m_filterList;
   PickingResultFilterType             m_filterType;
};

class PickedIndexFromPoiIcons
{
public:
   PickedIndexFromPoiIcons()
      : m_mapViewId(MAP_VIEW_ID__PRIMARY)
   {
   }

   virtual ~PickedIndexFromPoiIcons() {}

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setPickedIndexFromPoiIcons(const ValidValue<uint32_t>& pickedIndexFromPoiIcons)
   {
      m_pickedIndexFromPoiIcons = pickedIndexFromPoiIcons;
   }

   const ValidValue<uint32_t>& getPickedIndexFromPoiIcons() const
   {
      return m_pickedIndexFromPoiIcons;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("Picked Index From POI Icons payload:\n");
      stream << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl
             << "Picked Index = " << m_pickedIndexFromPoiIcons.getValueOr(0) << ::std::endl;

      return stream.str();
   }

private:
   MapViewId             m_mapViewId;
   ValidValue<uint32_t>  m_pickedIndexFromPoiIcons;
};

} // namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_MAPPICKINGRESULTS_H_
