/**
* @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_GUIDANCESETTINGS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_GUIDANCESETTINGS_H_

#include <sstream>
#include <string>

namespace navmiddleware
{
namespace settings
{

class GuidanceSettings
{
public:
   GuidanceSettings() :
      m_voiceGuidanceMode(true),
      m_trafficAnnouncements(true),
      m_trafficSignInformation(false),
      m_tollBoothInformation(false),
      m_borderCrossingInformation(false),
      m_isGlobalTrafficSignWarningsEnabled(false),
      //<-- INF4CV
      m_vehicleRestrictionsInformation(false),
      m_isUsbPoiWarningsEnabled(false)
      //->
   {}

   GuidanceSettings(
      bool voiceGuidanceMode,
      bool trafficAnnouncements,
      bool trafficSignInformation,
      bool tollBoothInformation,
      bool borderCrossingInformation,
      bool isGlobalTrafficSignWarningsEnabled,
      //<-- INF4CV
      bool vehicleRestrictionsInformation,
      bool isUsbPoiWarningsEnabled
      //->
   )

      : m_voiceGuidanceMode(voiceGuidanceMode)
      , m_trafficAnnouncements(trafficAnnouncements)
      , m_trafficSignInformation(trafficSignInformation)
      , m_tollBoothInformation(tollBoothInformation)
      , m_borderCrossingInformation(borderCrossingInformation)
      , m_isGlobalTrafficSignWarningsEnabled(isGlobalTrafficSignWarningsEnabled)
      //<-- INF4CV
      , m_vehicleRestrictionsInformation(vehicleRestrictionsInformation)
      , m_isUsbPoiWarningsEnabled(isUsbPoiWarningsEnabled)
      //->
   {}

   bool operator == (const GuidanceSettings& rhs) const
   {
      return (m_voiceGuidanceMode == rhs.m_voiceGuidanceMode &&
              m_trafficAnnouncements == rhs.m_trafficAnnouncements &&
              m_trafficSignInformation == rhs.m_trafficSignInformation &&
              m_tollBoothInformation == rhs.m_tollBoothInformation &&
              m_borderCrossingInformation == rhs.m_borderCrossingInformation &&
              m_isGlobalTrafficSignWarningsEnabled == rhs.m_isGlobalTrafficSignWarningsEnabled &&
              //<-- INF4CV
              m_vehicleRestrictionsInformation == rhs.m_vehicleRestrictionsInformation &&
              m_isUsbPoiWarningsEnabled == rhs.m_isUsbPoiWarningsEnabled);
              //->
   }

   bool operator != (const GuidanceSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool getVoiceGuidanceMode() const
   {
      return m_voiceGuidanceMode;
   }
   void setVoiceGuidanceMode(bool guidanceMode)
   {
      m_voiceGuidanceMode = guidanceMode;
   }

   bool getTrafficAnnouncements() const
   {
      return m_trafficAnnouncements;
   }
   void setTrafficAnnouncements(bool trafficAnnouncements)
   {
      m_trafficAnnouncements = trafficAnnouncements;
   }

   void setTrafficSignInformation(bool trafficSignInformation)
   {
      m_trafficSignInformation = trafficSignInformation;
   }
   bool getTrafficSignInformation() const
   {
      return m_trafficSignInformation;
   }

   bool getTollBoothInformation() const
   {
      return m_tollBoothInformation;
   }
   void setTollBoothInformation(bool tollBoothInformation)
   {
      m_tollBoothInformation = tollBoothInformation;
   }

   bool getBorderCrossingInformation() const
   {
      return m_borderCrossingInformation;
   }

   void setBorderCrossingInformation(bool borderCrossingInformation)
   {
      m_borderCrossingInformation = borderCrossingInformation;
   }

   bool isGlobalTrafficSignWarningsEnabled() const
   {
      return m_isGlobalTrafficSignWarningsEnabled;
   }

   void setGlobalTrafficSignWarningsEnabled(bool isGlobalTrafficSignWarningsEnabled)
   {
      m_isGlobalTrafficSignWarningsEnabled = isGlobalTrafficSignWarningsEnabled;
   }

   //<-- INF4CV
   bool getVehicleRestrictionsInformation() const
   {
      return m_vehicleRestrictionsInformation;
   }

   void setVehicleRestrictionsInformation(bool vehicleRestrictionsInformation)
   {
      m_vehicleRestrictionsInformation = vehicleRestrictionsInformation;
   }

   bool isUsbPoiWarningsEnabled() const
   {
      return m_isUsbPoiWarningsEnabled;
   }

   void setUsbPoiWarningsEnabled(bool isUsbPoiWarningsEnabled)
   {
      m_isUsbPoiWarningsEnabled = isUsbPoiWarningsEnabled;
   }
   //->

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "GuidanceSettings:" << std::endl
             << "voiceGuidanceMode = " << (m_voiceGuidanceMode ? "on" : "off") << ::std::endl
             << "trafficAnnouncements = " << (m_trafficAnnouncements ? "on" : "off") << ::std::endl
             << "trafficSignInformation = " << (m_trafficSignInformation ? "on" : "off") << ::std::endl
             << "tollBoothInformation = " << (m_tollBoothInformation ? "on" : "off") << ::std::endl
             << "borderCrossingInformation = " << (m_borderCrossingInformation ? "on" : "off") << ::std::endl
             << "isGlobalTrafficSignWarningsEnabled = " << (m_isGlobalTrafficSignWarningsEnabled ? "on" : "off") << ::std::endl
             //<-- INF4CV
             << "vehicleRestrictionsInformation = " << (m_vehicleRestrictionsInformation ? "on" : "off") << ::std::endl
             << "usbPoiWarningsEnabled = " << (m_isUsbPoiWarningsEnabled ? "on" : "off") << ::std::endl;
             //->
      return stream.str();
   }

private:
   bool m_voiceGuidanceMode;
   bool m_trafficAnnouncements;
   bool m_trafficSignInformation;
   bool m_tollBoothInformation;
   bool m_borderCrossingInformation;
   /* This field controls acoustic and symbolic warnings related to following traffic signs:
    * speed limit, over speed warning, Traffic enforcement zones (risk areas, speed cams, avg speed zone etc),
    * no overtaking and protected passing
    * */
   bool m_isGlobalTrafficSignWarningsEnabled;
   //<-- INF4CV
   bool m_vehicleRestrictionsInformation;
   bool m_isUsbPoiWarningsEnabled;
   //->
};

class SafetyGuidanceSettings
{
public:
   SafetyGuidanceSettings() :
      m_speedLimit(false),
      m_overSpeedWarning(false),
      m_overSpeedWarningAcoustic(false),
      m_speedLimitSensitivity(0) {}   //<-- INF4CV

   SafetyGuidanceSettings(
      bool speedLimit,
      bool overSpeedWarning,
      bool overSpeedWarningAcoustic,
      int speedLimitSensitivity)

      : m_speedLimit(speedLimit)
      , m_overSpeedWarning(overSpeedWarning)
      , m_overSpeedWarningAcoustic(overSpeedWarningAcoustic)
      , m_speedLimitSensitivity(speedLimitSensitivity)
   {}

   bool operator == (const SafetyGuidanceSettings& rhs) const
   {
      return (m_speedLimit == rhs.m_speedLimit &&
              m_overSpeedWarning == rhs.m_overSpeedWarning &&
              m_overSpeedWarningAcoustic == rhs.m_overSpeedWarningAcoustic &&
              m_speedLimitSensitivity == rhs.m_speedLimitSensitivity);
   }

   bool operator != (const SafetyGuidanceSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool isSpeedLimitActive() const
   {
      return m_speedLimit;
   }
   void setSpeedLimit(bool speedLimit)
   {
      m_speedLimit = speedLimit;
   }

   bool isOverSpeedWarningActive() const
   {
      return m_overSpeedWarning;
   }
   void setOverSpeedWarning(bool overSpeedWarning)
   {
      m_overSpeedWarning = overSpeedWarning;
   }

   bool isOverSpeedWarningAcousticActive() const
   {
      return m_overSpeedWarningAcoustic;
   }
   void setOverSpeedWarningAcoustic(bool overSpeedWarningAcoustic)
   {
      m_overSpeedWarningAcoustic = overSpeedWarningAcoustic;
   }

   int getSpeedLimitSensitivity() const
   {
      return m_speedLimitSensitivity;
   }
   void setSpeedLimitSensitivity(int speedLimitSensitivity)
   {
      m_speedLimitSensitivity = speedLimitSensitivity;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SafetyGuidanceSettings:" << std::endl
             << "speedLimit = " << (m_speedLimit ? "on" : "off") << ::std::endl
             << "overSpeedWarning = " << (m_overSpeedWarning ? "on" : "off") << ::std::endl
             << "overSpeedWarningAcoustic = " << (m_overSpeedWarningAcoustic ? "on" : "off") << ::std::endl
             << "speedLimitSensitivity = " << m_speedLimitSensitivity << ::std::endl;
      return stream.str();
   }

private:
   bool m_speedLimit;
   bool m_overSpeedWarning;
   /** m_overSpeedWarningAcoustic to be used by clients who want to configure visual and acoustic warnings for overspeed independently.
    * state of this member to be checked before triggering an acoustic output.
    */
   bool m_overSpeedWarningAcoustic;
   int m_speedLimitSensitivity;
};

enum TrafficEnforcementZoneWarningSource
{
   TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__NDS_DATA,
   TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__CONNECTED_SERVICES,
   TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__POI_SERVER
};

inline ::std::string toString(const TrafficEnforcementZoneWarningSource& trafficEnforcementZoneWarningSource)
{
   switch (trafficEnforcementZoneWarningSource)
   {
   case TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__NDS_DATA:
      return "TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__NDS_DATA";
   case TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__CONNECTED_SERVICES:
      return "TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__CONNECTED_SERVICES";
   case TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__POI_SERVER:
      return "TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__POI_SERVER";
   }
   return ""; // this code will never be executed
};

class SafetyCameraSettings
{
public:
   SafetyCameraSettings() :
      m_safetyCameraIcon(false),
      m_safetyCameraWarning(false),
      m_trafficEnforcementZoneDisplayOnMap(false),
      m_trafficEnforcementZoneWarningSymbolic(false),
      m_trafficEnforcementZoneWarningAcoustic(false),
      m_trafficEnforcementZoneWarningSource(TRAFFIC_ENFORCEMENT_ZONE_WARNING_SOURCE__NDS_DATA)
   {}

   SafetyCameraSettings(
      bool safetyCameraIcon,
      bool safetyCameraWarning,
      bool trafficEnforcementZoneDisplayOnMap,
      bool trafficEnforcementZoneWarningSymbolic,
      bool trafficEnforcementZoneWarningAcoustic,
      TrafficEnforcementZoneWarningSource trafficEnforcementZoneWarningSource)
      : m_safetyCameraIcon(safetyCameraIcon)
      , m_safetyCameraWarning(safetyCameraWarning)
      , m_trafficEnforcementZoneDisplayOnMap(trafficEnforcementZoneDisplayOnMap)
      , m_trafficEnforcementZoneWarningSymbolic(trafficEnforcementZoneWarningSymbolic)
      , m_trafficEnforcementZoneWarningAcoustic(trafficEnforcementZoneWarningAcoustic)
      , m_trafficEnforcementZoneWarningSource(trafficEnforcementZoneWarningSource)
   {}

   bool operator == (const SafetyCameraSettings& rhs) const
   {
      return (m_safetyCameraIcon == rhs.m_safetyCameraIcon &&
              m_safetyCameraWarning == rhs.m_safetyCameraWarning &&
              m_trafficEnforcementZoneDisplayOnMap == rhs.m_trafficEnforcementZoneDisplayOnMap &&
              m_trafficEnforcementZoneWarningSymbolic == rhs.m_trafficEnforcementZoneWarningSymbolic &&
              m_trafficEnforcementZoneWarningAcoustic == rhs.m_trafficEnforcementZoneWarningAcoustic &&
              m_trafficEnforcementZoneWarningSource == rhs.m_trafficEnforcementZoneWarningSource );
   }

   bool operator != (const SafetyCameraSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool isSafetyCameraIconActive() const
   {
      return m_safetyCameraIcon;
   }
   void setSafetyCameraIcon(bool safetyCameraIcon)
   {
      m_safetyCameraIcon = safetyCameraIcon;
   }

   bool isSafetyCameraWarningActive() const
   {
      return m_safetyCameraWarning;
   }
   void setSafetyCameraWarning(bool safetyCameraWarning)
   {
      m_safetyCameraWarning = safetyCameraWarning;
   }

   bool isTrafficEnforcementZoneDisplayOnMapActive() const
   {
      return m_trafficEnforcementZoneDisplayOnMap;
   }
   void setTrafficEnforcementZoneDisplayOnMap(bool trafficEnforcementZoneDisplayOnMap)
   {
      m_trafficEnforcementZoneDisplayOnMap = trafficEnforcementZoneDisplayOnMap;
   }

   bool isTrafficEnforcementZoneSymbolicWarningActive() const
   {
      return m_trafficEnforcementZoneWarningSymbolic;
   }
   void setTrafficEnforcementZoneSymbolicWarning(bool trafficEnforcementZoneSymbolicWarning)
   {
      m_trafficEnforcementZoneWarningSymbolic = trafficEnforcementZoneSymbolicWarning;
   }

   bool isTrafficEnforcementZoneAcousticWarningActive() const
   {
      return m_trafficEnforcementZoneWarningAcoustic;
   }
   void setTrafficEnforcementZoneAcousticWarning(bool trafficEnforcementZoneSAcousticWarning)
   {
      m_trafficEnforcementZoneWarningAcoustic = trafficEnforcementZoneSAcousticWarning;
   }

   void setTrafficEnforcementWarningSource(const TrafficEnforcementZoneWarningSource warningSource)
   {
      m_trafficEnforcementZoneWarningSource = warningSource;
   }

   TrafficEnforcementZoneWarningSource getTrafficEnforcementWarningSource() const
   {
      return m_trafficEnforcementZoneWarningSource;
   }

private:
   // setting to enable/disable speed cam display on map
   // will be removed for ST2.1 in future, trafficEnforcementZoneWarning
   bool m_safetyCameraIcon;
   // setting to enable/disable speed cam acoustic and symbolic warnings
   // will be removed for ST2.1 in future
   bool m_safetyCameraWarning;

   // ST2.1 onwards: setting to enable/disable trafficEnforcementZone display on map
   bool m_trafficEnforcementZoneDisplayOnMap;
   // ST2.1 onwards: setting to enable/disable trafficEnforcementZoneWarning symbolic warnings
   bool m_trafficEnforcementZoneWarningSymbolic;
   // ST2.1 onwards: setting to enable/disable trafficEnforcementZoneWarning acoustic warnings
   bool m_trafficEnforcementZoneWarningAcoustic;
   // ST2.1 onwards: warning source
   TrafficEnforcementZoneWarningSource m_trafficEnforcementZoneWarningSource;
};

class EvBatteryStatusWarningSettings
{
public:
   EvBatteryStatusWarningSettings () :
      m_lowBatteryWarning(false),
      m_connectToOperatorIfLowBattery(false),
      m_destOutOfRangeWarning(false),
      m_guardianAngel(false)
   {}

   EvBatteryStatusWarningSettings(
      bool lowBatteryWarning,
      bool connectToOperatorIfLowBattery,
      bool destOutOfRangeWarning,
      bool guardianAngel)
      : m_lowBatteryWarning(lowBatteryWarning)
      , m_connectToOperatorIfLowBattery(connectToOperatorIfLowBattery)
      , m_destOutOfRangeWarning(destOutOfRangeWarning)
      , m_guardianAngel(guardianAngel)
   {}

   bool operator == (const EvBatteryStatusWarningSettings& rhs) const
   {
      return (m_lowBatteryWarning == rhs.m_lowBatteryWarning &&
              m_connectToOperatorIfLowBattery == rhs.m_connectToOperatorIfLowBattery &&
              m_destOutOfRangeWarning == rhs.m_destOutOfRangeWarning &&
              m_guardianAngel == rhs.m_guardianAngel);
   }

   bool operator != (const EvBatteryStatusWarningSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool isLowBatteryWarningActive() const
   {
      return m_lowBatteryWarning;
   }

   void setLowBatteryWarning(bool lowBatteryWarning)
   {
      m_lowBatteryWarning = lowBatteryWarning;
   }

   bool isConnectToOperatorIfLowBatteryActive() const
   {
      return m_connectToOperatorIfLowBattery;
   }

   void setConnectToOperatorIfLowBattery(bool connectToOperatorIfLowBattery)
   {
      m_connectToOperatorIfLowBattery = connectToOperatorIfLowBattery;
   }

   bool isDestOutOfRangeWarningActive() const
   {
      return m_destOutOfRangeWarning;
   }

   void setDestOutOfRangeWarning(bool destOutOfRangeWarning)
   {
      m_destOutOfRangeWarning = destOutOfRangeWarning;
   }

   bool isGuardianAngelActive() const
   {
      return m_guardianAngel;
   }

   void setGuardianAngel(bool guardianAngel)
   {
      m_guardianAngel = guardianAngel;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "EvBatteryStatusWarningSettings:" << std::endl
             << "lowBatteryWarning = " << (m_lowBatteryWarning ? "on" : "off") << ::std::endl
             << "connectToOperatorIfLowBattery = " << (m_connectToOperatorIfLowBattery ? "on" : "off") << ::std::endl
             << "destOutOfRangeWarning = " << (m_destOutOfRangeWarning ? "on" : "off") << ::std::endl
             << "guardianAngel = " << (m_guardianAngel ? "on" : "off") << ::std::endl;
      return stream.str();
   }

private:
   bool m_lowBatteryWarning;
   bool m_connectToOperatorIfLowBattery;
   bool m_destOutOfRangeWarning;
   bool m_guardianAngel;
};

class EvMiscellaneousSettings
{
public:
   EvMiscellaneousSettings () :
      m_enableBatteryHeater(false)
   {}

   EvMiscellaneousSettings (
      bool enableBatteryHeater)
      : m_enableBatteryHeater(enableBatteryHeater)
   {}

   bool operator == (const EvMiscellaneousSettings& rhs) const
   {
      return (m_enableBatteryHeater == rhs.m_enableBatteryHeater);
   }

   bool operator != (const EvMiscellaneousSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool isBatteryHeaterEnabled() const
   {
      return m_enableBatteryHeater;
   }

   void setBatteryHeater(bool enableBatteryHeater)
   {
      m_enableBatteryHeater = enableBatteryHeater;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "EvMiscellaneousSettings :" << ::std::endl
             << "enableBatteryHeater = " << (m_enableBatteryHeater ? "ON" : "OFF") << ::std::endl;
      return stream.str();
   }

private:
   bool m_enableBatteryHeater;
};

}
}
#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_GUIDANCESETTINGS_H_
