/*********************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2019
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 *********************************************************************/

#ifndef ASF_STREAM_JSON_H
#define ASF_STREAM_JSON_H

#include "asf/core.h"
#include "asf/core/Types.h"

#include <yajl/yajl_gen.h>
#include <yajl/yajl_parse.h>
#include <stack>

namespace asf {
namespace stream {
namespace json {

void genStringAndCheck(yajl_gen g, const unsigned char* string, size_t length);

enum JsonSerializationOptions {
    // serialize only those fields with a non default value
    JSON_SERIALIZATION_OPTION_SPARSE = 1
};

int compareUnsignedInts(unsigned int* a, unsigned int* b);

int strcmpupper(const char* str1, size_t len1, const char* str2);

struct stringEntry {
    size_t length;
    const char* s;
    unsigned int value;
};

int compareStringEntries(stringEntry* a, stringEntry* b);

bool lookupInStringEntryTable(const char* s,
                              size_t length,
                              stringEntry* entries,
                              size_t numberOfEntries,
                              unsigned int& value);

bool streq(const char* str1, size_t len1, const char* str2);

bool strequpper(const char* str1, size_t len1, const char* str2);

char* ulltoa(uint64_t ll, char* buffer);

char* lltoa(int64_t ll, char* buffer);

enum ParserEvent {
    PARSER_EVENT__NULL,         // 0
    PARSER_EVENT__BOOLEAN,      // 1
    PARSER_EVENT__NUMBER,       // 2
    PARSER_EVENT__STRING,       // 3
    PARSER_EVENT__START_MAP,    // 4
    PARSER_EVENT__MAP_KEY,      // 5
    PARSER_EVENT__END_MAP,      // 6
    PARSER_EVENT__START_ARRAY,  // 7
    PARSER_EVENT__END_ARRAY,    // 8
    PARSER_EVENT__RETURN        // 9
};

enum ParserStatus { PARSER_STATUS__FAIL, PARSER_STATUS__FINISHED, PARSER_STATUS__OK };

struct ParserContext;

typedef ParserStatus (*ParseFunction)(ParserEvent event,
                                      ParserContext* parserContext,
                                      void* p1,
                                      void* p2);

struct Parser {
    ParseFunction parseFunction;

    int state;

    void* object;

    union Data {
        bool bool__;
        int8 int8__;
        int16 int16__;
        int32 int32__;
        int64 int64__;
        uint8 uint8__;
        uint16 uint16__;
        uint32 uint32__;
        uint64 uint64__;
        float float__;
        double double__;
        void* voidptr__;
    } data;

    const char* name;
};

void assignData(Parser& parser, bool value);

void assignData(Parser& parser, int8 value);

void assignData(Parser& parser, int16 value);

void assignData(Parser& parser, int32 value);

void assignData(Parser& parser, int64 value);

void assignData(Parser& parser, uint8 value);

void assignData(Parser& parser, uint16 value);

void assignData(Parser& parser, uint32 value);

void assignData(Parser& parser, uint64 value);

void assignData(Parser& parser, float value);

void assignData(Parser& parser, double value);

void assignData(Parser& parser, void* value);

void assignData(Parser& parser, ::std::string& value);

void assignData(Parser& parser, ::asf::core::Char& char_);

struct ParserContext {
    bool isStrict;

    std::stack< Parser > parsers;
};

ParserStatus parse(ParserEvent event, void* ctx, void* p1, void* p2);

void callParser(ParserContext* context,
                const char* name,
                ParseFunction parseFunction,
                void* object);

ParserStatus leaveParser(ParserContext* context);

void errorMessage(ParserEvent event, Parser& parser);

void setState(ParserContext* context, int state);

// a generic parser function to eat an opening bracket "{".

// Used to for parsing the outer most map. The opening brackets of inner maps or eatten by
// their parent.

ParserStatus parserExpectMap(ParserEvent event, ParserContext* context, void* p1, void* p2);

ParserStatus parserAnyMap(ParserEvent event, ParserContext* context, void* p1, void* p2);

ParserStatus parserAnyArray(ParserEvent event, ParserContext* context, void* p1, void* p2);

ParserStatus parserBlob(ParserEvent event, ParserContext* context, void* p1, void* p2);

extern yajl_callbacks callbacks;

}  // namespace json
}  // namespace stream
}  // namespace asf

#endif  // ASF_JSON_H
