/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DBusUtils.cpp
 *\brief
 *
 *\author   CM-AI/PJ-CF11.1
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSUTILS_H_
#define ASF_DBUS_DBUSUTILS_H_

#include "asf/dbus/DBusConnector.h"
#include "asf/dbus/DBusError.h"

#include "asf/core/Logger.h"

#include "com/bosch/cm/asf/lang/dbus/Connectors.h"

#include <dbus/dbus.h>
#include <cassert>
#include <sstream>

namespace asf {
namespace dbus {

class DBusUtils {
public:
    struct stringEntry {
        size_t length;
        const char* s;
        unsigned int value;
    };

    static bool lookupInStringEntryTable(const char* s,
                                         size_t length,
                                         stringEntry* entries,
                                         int numberOfEntries,
                                         unsigned int& value);

    static ::DBusBusType getDBusBusType(const std::string& busType) {
        if (0 == busType.compare("DBUS_BUS_SYSTEM")) {
            return DBUS_BUS_SYSTEM;
        } else if (0 == busType.compare("DBUS_BUS_SESSION")) {
            return DBUS_BUS_SESSION;
        } else if (0 == busType.compare("DBUS_BUS_STARTER")) {
            return DBUS_BUS_STARTER;
        }
        LOG_FATAL_STATIC("Bus type %s unknown", busType.c_str());
        return static_cast< ::DBusBusType >(0);
    }

    static const char* dbusBusTypeToString(
        const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options) {
        switch (options.getBusType()) {
        case ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__System:
            return "DBUS_BUS_SYSTEM";
        case ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__Session:
            return "DBUS_BUS_SESSION";
        case ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__Address:
            return options.getAddress().c_str();
        default:
            LOG_FATAL_STATIC("Connector bus type unknown");
            return 0;
        }
    }

    static ::DBusBusType getWellKnownBusType(
        const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options) {
        switch (options.getBusType()) {
        case ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__Session:
            return DBUS_BUS_SESSION;
        case ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__System:
            return DBUS_BUS_SYSTEM;
        default:
            LOG_FATAL_STATIC("The DBusConnector does not provide the well-known bus type %d",
                             options.getBusType());
            return static_cast< ::DBusBusType >(0);
        }
    }

    static ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector toOptions(
        ::DBusBusType busType) {
        ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector options;
        switch (busType) {
        case DBUS_BUS_SESSION:
            options.setBusType(
                ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__Session);
            break;
        case DBUS_BUS_SYSTEM:
            options.setBusType(
                ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector::BusType__System);
            break;
        default:
            LOG_FATAL_STATIC("The DBusConnector does not provide the requested bus type %d",
                             busType);
        }
        return options;
    }

    static bool isBasicType(int typecode) {
        switch (typecode) {
        case 'y':
        case 'b':
        case 'n':
        case 'q':
        case 'i':
        case 'u':
        case 'x':
        case 't':
        case 'd':
        case 's':
        case 'o':
        case 'g':
        case 'h':
            return true;
        default:
            return false;
        }
    }

#define DBUS_OBJECTPATH_VALIDATION_ERROR                                                          \
    "Object path must only contain '[A-Z][a-z][0-9]_', must start with a leading slash and must " \
    "not end with a slash"

    static bool isObjectPathValid(const std::string& objectPath) {
#if DBUS_MAJOR_VERSION > 1 || (DBUS_MAJOR_VERSION == 1 && DBUS_MINOR_VERSION >= 6)
        ::asf::dbus::DBusError dbusError;
        dbus_validate_path(objectPath.c_str(), &dbusError._dbusError);
        if (dbusError) {
            LOG_ERROR_STATIC("DBus Error: '%s', '%s' (%s)",
                             dbusError.getName().c_str(),
                             dbusError.getMessage().c_str(),
                             DBUS_OBJECTPATH_VALIDATION_ERROR);
            return false;
        }
        return true;
#else  // For older dbus versions we provide an own basic mechanism for checking the object path
       // validity
        if (0 == objectPath.compare(0, 1, "/")  // check for leading /
            && 0 != objectPath.compare(
                        objectPath.size() - 1, 1, "/"))  // check for absence of trailing /
            return true;
        LOG_ERROR_STATIC(
            "DBus Error: 'org.freedesktop.DBus.Error.InvalidArgs', 'Object path was not valid: "
            "'%s'' (%s)",
            objectPath.c_str(),
            DBUS_OBJECTPATH_VALIDATION_ERROR);
        return false;
#endif
    }

#define DBUS_BUSNAME_VALIDATION_ERROR                                                           \
    "Bus name must only contain '[A-Z][a-z][0-9]_', must not start with a leading digit nor a " \
    "'.' and must contain at least one '.' "

    static bool isBusNameValid(const std::string& busName) {
#if DBUS_MAJOR_VERSION > 1 || (DBUS_MAJOR_VERSION == 1 && DBUS_MINOR_VERSION >= 6)
        ::asf::dbus::DBusError dbusError;
        dbus_validate_bus_name(busName.c_str(), &dbusError._dbusError);
        if (dbusError) {
            LOG_ERROR_STATIC("DBus Error: '%s', '%s' (%s)",
                             dbusError.getName().c_str(),
                             dbusError.getMessage().c_str(),
                             DBUS_BUSNAME_VALIDATION_ERROR);
            return false;
        }
        return true;
#else  // For older dbus versions we provide an own basic mechanism for checking the bus name
       // validity
        if (0 != busName.compare(0, 1, ".")  // check for absence of leading '.'
            && std::count(busName.begin(), busName.end(), '.') >= 1)  // check at least one '.'
            return true;
        LOG_ERROR_STATIC(
            "DBus Error: 'org.freedesktop.DBus.Error.InvalidArgs', 'Bus name was not valid: "
            "'%s'' (%s)",
            busName.c_str(),
            DBUS_BUSNAME_VALIDATION_ERROR);
        return false;
#endif
    }

    static bool isSignatureValid(::DBusMessageIter* iter) {
        char* sig = dbus_message_iter_get_signature(iter);
        if (0 != strcmp(sig, DBUS_TYPE_INVALID_AS_STRING)) {
            dbus_free(sig);
            return true;
        }
        dbus_free(sig);
        return false;
    }

    static const char* getDBusMessageString(const char* string) {
        if (string)
            return string;
        else
            return "null";
    }

    static std::string createMessageLogString(::DBusMessage* dbusMessage);

    /**
     * This function creates a libdbus method message in regards of the provided addressing
     * information. If no message can be
     * created an assertion will be thrown.
     *
     * @param busName
     * @param objectPath
     * @param interfaceName
     * @param methodName
     * @param serial If no serial is given a new serial (ASF unique message id) will be created.
     *
     * @return The pointer to the created libdbus message is returned. The caller has the ownership
     * of the message pointer,
     * which needs to be unreferenced with dbus_messsage_unref().
     */
    static ::DBusMessage* createMethodCall(const std::string& busName,
                                           const std::string& objectPath,
                                           const std::string& interfaceName,
                                           const std::string& methodName,
                                           uint32 serial = 0);

    /**
     * This function creates a libdbus method message in regards of the provided addressing
     * information. If no message can be
     * created an assertion will be thrown. The created message call has the D-Bus header flag
     * NO_REPLY_EXPECTED set to true and no
     * reply nor an error message is expected to be received.
     *
     * @param busName
     * @param objectPath
     * @param interfaceName
     * @param methodName
     * @param serial If no serial is given a new serial (ASF unique message id) will be created.
     *
     * @return The pointer to the created libdbus message is returned. The caller has the ownership
     * of the message pointer,
     * which needs to be unreferenced with dbus_messsage_unref().
     */
    static ::DBusMessage* createMethodCallWithNoReply(const std::string& busName,
                                                      const std::string& objectPath,
                                                      const std::string& interfaceName,
                                                      const std::string& methodName,
                                                      uint32 serial = 0);

    /**
     * This function creates a libdbus signal message in regards of the provided addressing
     * information. If no message can be
     * created an assertion will be thrown.
     *
     * @param objectPath
     * @param interfaceName
     * @param signalName
     *
     * @return The pointer to the created libdbus message is returned. The caller has the ownership
     * of the message pointer,
     * which needs to be unreferenced with dbus_messsage_unref().
     */
    static ::DBusMessage* createSignal(const std::string& objectPath,
                                       const std::string& interfaceName,
                                       const std::string& signalName);

    /**
     * This function returns the D-Bus bus name.
     *
     * If no 'busName' parameter is set in the DBusConnector, the bus name is set to
     * "<application.package>.<application.name>"
     *
     */
    static const std::string getBusName(
        const ::asf::core::Application& application,
        const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options) {
        if (options.hasBusName()) {
            return options.getBusName();
        } else {
            return getDefaultBusName(application);
        }
    }

    static const std::string getDefaultBusName(const ::asf::core::Application& app) {
        return app.getPackageName() + "." + app.getName();
    }

    static std::string replaceEnvironmentVariable(const std::string& environmentVariable);

    static bool isEnvironmentVariable(const std::string& address);

    static const char* readEnvironmentVariable(const std::string& environmentVariable);

    DECLARE_CLASS_LOGGER();
};
}  // namespace dbus
}  // namespace asf

#endif
