/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusProxyDelegate.h
 *\brief
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSPROXYDELEGATE_H
#define ASF_DBUS_DBUSPROXYDELEGATE_H

#include "asf/core/ProxyDelegateBase.h"
#include "asf/dbus/DBusImportedPort.h"
#include "asf/dbus/DBusMessage.h"
#include "asf/dbus/DBusUtils.h"
#include "asf/threading/Mutex.h"

#include <boost/enable_shared_from_this.hpp>

#include <map>
#include <set>

class DBusProxyTest;
class DBusProxyAvailablilityTest;
class DBusMessageFactoryTest;

namespace asf {
namespace dbus {

class DBusProxyRegistry;
class DBusSerialRegistry;
class DBusDaemonProxy;
class DBusPropertiesProxyWrapper;

class DBusProxyCallback {
public:
    DBusProxyCallback(uint16 functionId, void* callback)
        : _functionId(functionId), _callback(callback) {}

    virtual ~DBusProxyCallback() {}

    /**
     * Process a message received for this callback. If it returns
     * true the callback will be removed from the proxy.
     */
    virtual bool processMessage(DBusMessage& message) = 0;

    /**
     * Called whenever the service get unavailable is this callback is
     * still registered
     */
    virtual bool onServiceUnavailable(
        const ::boost::shared_ptr< ::asf::core::Proxy >& baseProxy) = 0;

protected:
    uint16 _functionId;

    void* _callback;
};

/**
 * DBusProxyDelegate provides common methods to all generated DBus proxy classes
 */
class DBusProxyDelegate : public ::asf::core::ProxyDelegateBase,
                          public ::boost::enable_shared_from_this< DBusProxyDelegate > {
public:
    DBusProxyDelegate(const std::string& interfaceName,
                      const ::asf::core::ServiceAvailableIF& serviceAvailable,
                      const ::asf::core::Logger& logger,
                      ::asf::core::Proxy* userProxy,
                      bool internalProxy = false);

    virtual ~DBusProxyDelegate();

    void setDBusBusName(const std::string& busName) { _busName = busName; }

    const std::string& getDBusBusName() const { return _busName; }

    void setDBusObjectPath(const std::string& objectPath) {
        LOG_ASSERT(DBusUtils::isObjectPathValid(objectPath));
        _objectPath = objectPath;
    }

    const std::string& getDBusObjectPath() const { return _objectPath; }

    const std::string& getInterfaceName() const { return _interfaceName; }

    void setConnectorOptions(
        const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options) {
        _connectorOptions = options;
    }

    const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& getConnectorOptions()
        const {
        return _connectorOptions;
    }

    void setSerialRegistry(::boost::shared_ptr< DBusSerialRegistry >& serialRegistry) {
        _serialRegistry = serialRegistry;
    }

    ::boost::shared_ptr< DBusSerialRegistry > getSerialRegistry() const { return _serialRegistry; }

    /**
     * Register the method callback for the given D-Bus serial number in the serial registry.
     * With the matching serial number of the response the correct callback is called when
     * processing the method.
     *
     * @param serial the D-Bus method serial number
     * @param the member name of the D-Bus method
     * @param callback
     */
    void addMethodCallback(const uint32 serial,
                           const std::string& memberName,
                           const ::boost::shared_ptr< DBusProxyCallback >& callback);

    /**
     * Signals don't have serial numbers like methods. Therefore the callback registration uses the
     * given signal name and act.
     *
     * @param signalName
     * @param act
     * @param callback
     */
    void registerSignalCallback(const ::std::string& signalName,
                                const act_t act,
                                const ::boost::shared_ptr< DBusProxyCallback >& callback);

    bool deregisterSignalCallback(const ::std::string& signalName,
                                  const act_t act,
                                  DBusDaemonProxy* dbusDaemonProxy,
                                  bool isProperty = false);

    void connectToImportedPort(DBusImportedPort* importedPort);

    virtual void onConnected();

    virtual void onDisconnected();

private:
    virtual void process(::asf::core::ServiceMessage& msg);

    void startServiceRegistration();

    void onServiceAvailable(const ::asf::core::ServiceStateChange& stateChange);

    void onServiceUnavailable(const ::asf::core::ServiceStateChange& stateChange);

    bool isInternalDaemonProxy() const { return _internalDaemonProxy; }

    ::boost::shared_ptr< DBusSerialRegistry > _serialRegistry;

    std::string _objectPath;

    const std::string _interfaceName;

    std::string _busName;

    ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector _connectorOptions;

    ::asf::core::ServiceAvailableIF& _serviceAvailable;

    ::asf::threading::Mutex
        _lock;  // lock to protect concurrent access to the map _memberNameToCallbackMap

    ::asf::threading::Mutex
        _serialsLock;  // lock to protect concurrent access to _registeredSerials

    typedef std::map< act_t, ::boost::shared_ptr< DBusProxyCallback > > ActToCallbackMap;

    typedef std::map< std::string, ActToCallbackMap > SignalNameToCallbackMap;

    /**
     * Maps the signal name to all act numbers which are registered.
     */
    SignalNameToCallbackMap _signalNameToCallbackMap;

    /**
     * This set holds the registered unique serial numbers for the current proxy.
     */
    std::set< uint32 > _registeredSerials;

    bool hasCallbacks();

    ::asf::core::Logger& _logger;

    bool _internalDaemonProxy;

    friend class DBusMessage;
    friend class DBusProxyRegistry;
    friend class ::DBusProxyTest;
    friend class ::DBusProxyAvailablilityTest;
    friend class DBusPropertiesProxyWrapper;
    friend class DBusDaemonProxy;
    friend class DBusConnectionMessage;
    friend class ::DBusMessageFactoryTest;
};
}  // namespace dbus
}  // namespace asf

#endif
