/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusPropertiesProxyWrapper.h
 *\brief
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_DBUS_PROPERTIES_WRAPPER_PROXY_H
#define ASF_DBUS_PROPERTIES_WRAPPER_PROXY_H

#include "asf/core/Logger.h"
#include "asf/core/nullable.h"

#include "asf/dbus/DBusProxyDelegate.h"
#include "asf/dbus/DBusProxyRegistry.h"

#include "asf/dbus/org/freedesktop/DBus/PropertiesProxy.h"

#include <dbus/dbus.h>
#include <string>
#include "boost/shared_ptr.hpp"

class DBusProxyTest;

namespace asf {
namespace dbus {

class DBusDaemonProxy;

/**
 * The DBusPropertiesProxyWrapper abstracts the generated proxy of the D-Bus properties interface
 * and contains some logic to handle property updates.
 * A DBusPropertiesProxyWrapper instance will be created by the service proxy.
 */
class DBusPropertiesProxyWrapper {
public:
    act_t sendGetProperty(const std::string& property,
                          ::boost::shared_ptr< ::asf::dbus::DBusProxyCallback > callback);

    act_t sendSetProperty(const std::string& interfaceName,
                          const std::string& propertyName,
                          const ::asf::dbus::DBusVariant& propertyContent);

    act_t sendSetProperty(const std::string& interfaceName,
                          const std::string& propertyName,
                          const ::asf::dbus::DBusVariant& propertyContent,
                          ::boost::shared_ptr< ::asf::dbus::DBusProxyCallback > callback);

    void registerProperty();

    void deregisterProperty();

    static DBusPropertiesProxyWrapper* createWrapper(
        const std::string& busName,
        const std::string& objectPath,
        boost::shared_ptr< DBusProxyDelegate >& parentProxy) {
        DBusPropertiesProxyWrapper* pProxy =
            new DBusPropertiesProxyWrapper(busName, objectPath, parentProxy);
        CHECK_ALLOCATION(pProxy);
        return pProxy;
    }

    virtual ~DBusPropertiesProxyWrapper();

protected:
    class PropertiesProxyHandler
        : public ::asf::core::ServiceAvailableIF,
          public ::asf::dbus::org::freedesktop::DBus::Properties::GetCallbackIF,
          public ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesChangedCallbackIF,
          public ::asf::dbus::org::freedesktop::DBus::Properties::GetAllCallbackIF,
          public ::asf::dbus::org::freedesktop::DBus::Properties::SetCallbackIF {
    public:
        PropertiesProxyHandler(DBusPropertiesProxyWrapper* proxy) : _proxy(proxy) {}

    protected:
        virtual void onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& /*proxy*/,
                                 const ::asf::core::ServiceStateChange& /*stateChange*/);

        virtual void onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& /*proxy*/,
                                   const ::asf::core::ServiceStateChange& /*stateChange*/);

        virtual void onGetAllError(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::GetAllError >& error);

        virtual void onGetAllResponse(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::GetAllResponse >& response);

        virtual void onGetError(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr< ::asf::dbus::org::freedesktop::DBus::Properties::GetError >&
                error);

        virtual void onGetResponse(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::GetResponse >& response);

        virtual void onPropertiesChangedError(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesChangedError >& error);

        virtual void onPropertiesChangedSignal(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesChangedSignal >& signal);

        virtual void onSetError(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr< ::asf::dbus::org::freedesktop::DBus::Properties::SetError >&
                error);

        virtual void onSetResponse(
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >& proxy,
            const ::boost::shared_ptr<
                ::asf::dbus::org::freedesktop::DBus::Properties::SetResponse >& response);

    private:
        void processError(const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError >& error);

        DBusPropertiesProxyWrapper* _proxy;
    };

private:
    /**
     * The callback and the content of the property which is needed to invoke the callback of
     * the set or get property reply is stored in this class.
     */
    class PropertySetGetCallbackContent {
    public:
        PropertySetGetCallbackContent(
            const ::boost::shared_ptr< DBusProxyCallback >& callback,
            const ::std::string& propertyName,
            const ::asf::core::nullable< ::asf::dbus::DBusVariant >& propertyContent)
            : _callback(callback), _propertyName(propertyName), _propertyContent(propertyContent) {}

        PropertySetGetCallbackContent(const ::boost::shared_ptr< DBusProxyCallback >& callback,
                                      const ::std::string& propertyName)
            : _callback(callback), _propertyName(propertyName) {}

        ::boost::shared_ptr< DBusProxyCallback > _callback;

        ::std::string _propertyName;

        ::asf::core::nullable< ::asf::dbus::DBusVariant > _propertyContent;

    private:
        PropertySetGetCallbackContent(){};
    };

    DBusPropertiesProxyWrapper(const std::string& busName,
                               const std::string& objectPath,
                               boost::shared_ptr< DBusProxyDelegate >& parentProxyDelegate);

    ::boost::shared_ptr< ::asf::dbus::org::freedesktop::DBus::Properties::PropertiesProxy >
        _propertiesProxy;

    // Store the parent proxy as a weak pointer instead of a raw pointer, because sending a D-Bus
    // error message to the service proxy requires a proxy shared pointer.
    ::boost::weak_ptr< DBusProxyDelegate > _parentProxyDelegate;

    PropertiesProxyHandler _propertiesHandler;

    act_t _registeredPropertyUpdateToken;

    typedef std::map< act_t, ::boost::shared_ptr< PropertySetGetCallbackContent > >
        ActToCallbackContentMap;

    /**
     * This map is needed to store the information which property was requested
     * by the 'sendGetRequest' and 'sendSetRequest', because the property name and property content
     * are not part of the response and need therefore to be reconstructed from this mapping.
     * Further, the map contains the callback object for the get/set property call. For the internal
     * property update messages the D-Bus serial can't be used (like other callbacks which are
     * stored in the global DBusSerialRegistry), because no D-Bus message will be sent and therefore
     * no serial number is available.
     */
    ActToCallbackContentMap _actToCallbackContentMap;

    /**
     * Lock to protect concurrent access to the map _actToCallbackContentMap
     */
    ::asf::threading::Mutex _lock;

    DECLARE_CLASS_LOGGER();

    friend class ::DBusProxyTest;
};

}  // namespace dbus
}  // namespace asf

#endif  // ASF_DBUS_PROPERTIES_WRAPPER_PROXY_H
