/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusPayloadDecorator.h
 *\brief
 *
 *\author   CM-AI/ECA2
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2014-2014 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSPAYLOADDECORATOR_H
#define ASF_DBUS_DBUSPAYLOADDECORATOR_H

#include "asf/core/Payload.h"
#include "asf/stream/json.h"
#include "dbus/dbus.h"

namespace asf {
namespace dbus {

#define DBUS_PAYLOAD_DECORATOR__SERIALIZE_FUNCTION(a)       \
    (::asf::dbus::DBusPayloadDecorator::SerializeFunction)( \
        (void (*)(const a&, ::DBusMessageIter*))::serializeDBus)

#define DBUS_PAYLOAD_DECORATOR__DESERIALIZE_FUNCTION(a)       \
    (::asf::dbus::DBusPayloadDecorator::DeserializeFunction)( \
        (bool (*)(::DBusMessageIter*, a&))::deserializeDBus)

#define DBUS_PAYLOAD_DECORATOR__JSON_FUNCTION(a)                 \
    (::asf::dbus::DBusPayloadDecorator::JsonFunction)((void (*)( \
        const a&, ::std::string&, ::asf::stream::json::JsonSerializationOptions))::serializeJson)

#define DBUS_PAYLOAD_DECORATOR__DESERIALIZE_ERROR_FUNCTION(a)      \
    (::asf::dbus::DBusPayloadDecorator::DeserializeErrorFunction)( \
        (bool (*)(::DBusMessage*, a&))::deserializeDBusError)

#define DBUS_PAYLOAD_DECORATOR__SERIALIZE_ERROR_FUNCTION(a)      \
    (::asf::dbus::DBusPayloadDecorator::SerializeErrorFunction)( \
        (void (*)(const a&, ::DBusMessage*))::serializeDBusError)

// leaves the deserialization function empty in order to not include these in
// code when not needed. Otherwise would for example a client include all
// serialization and deserialization functions for all payload classes
#define INIT_DBUS_PAYLOAD_DECORATOR_FOR_SERIALIZATION(var, type, instance) \
    var.init(::boost::shared_ptr< type >(instance),                        \
             DBUS_PAYLOAD_DECORATOR__SERIALIZE_FUNCTION(type),             \
             0,                                                            \
             DBUS_PAYLOAD_DECORATOR__JSON_FUNCTION(type),                  \
             0,                                                            \
             0);

// leaves the serialization function empty in order to not include these in
// code when not needed. Otherwise would for example a client include all
// serialization and deserialization functions for all payload classes
#define INIT_DBUS_PAYLOAD_DECORATOR_FOR_DESERIALIZATION(var, type) \
    var.init(::boost::shared_ptr< type >(new type()),              \
             0,                                                    \
             DBUS_PAYLOAD_DECORATOR__DESERIALIZE_FUNCTION(type),   \
             DBUS_PAYLOAD_DECORATOR__JSON_FUNCTION(type),          \
             0,                                                    \
             0);

// leaves the serialization and deserialization function empty in order to not include these in
// code when not needed. Otherwise would for example a client include all
// serialization and deserialization functions for all payload classes
#define INIT_DBUS_PAYLOAD_DECORATOR_FOR_ERROR_DESERIALIZATION(var, type) \
    var.init(::boost::shared_ptr< type >(new type()),                    \
             0,                                                          \
             0,                                                          \
             DBUS_PAYLOAD_DECORATOR__JSON_FUNCTION(type),                \
             DBUS_PAYLOAD_DECORATOR__DESERIALIZE_ERROR_FUNCTION(type),   \
             0);

// leaves the serialization and deserialization function empty in order to not include these in
// code when not needed. Otherwise would for example a client include all
// serialization and deserialization functions for all payload classes
#define INIT_DBUS_PAYLOAD_DECORATOR_FOR_ERROR_SERIALIZATION(var, type, instance) \
    var.init(::boost::shared_ptr< type >(instance),                              \
             0,                                                                  \
             0,                                                                  \
             DBUS_PAYLOAD_DECORATOR__JSON_FUNCTION(type),                        \
             0,                                                                  \
             DBUS_PAYLOAD_DECORATOR__SERIALIZE_ERROR_FUNCTION(type));

/**
 * The DBusPayloadDecorator provides functionalities like serialization for a
 * given payload object. These functionalities are realized with the help of
 * function pointers which have to be initialized with the help of the
 * init() method.
 *
 * This class was introduced to add some flexibility to the payload data classes
 * without adding virtual methods to the payload class (in order to not compromise
 * the public API of the data classes) and without requiring any template
 * function which lead to a code bloat.
 *
 * The DBusProxy calls the init() method as soon as it knows the type of the
 * payload during the message dispatching or sending.
 */
class DBusPayloadDecorator {
public:
    typedef void (*SerializeFunction)(const ::asf::core::Payload&, ::DBusMessageIter* s);

    typedef bool (*DeserializeFunction)(::DBusMessageIter* s, ::asf::core::Payload&);

    typedef void (*JsonFunction)(const ::asf::core::Payload&,
                                 std::string&,
                                 ::asf::stream::json::JsonSerializationOptions);

    typedef bool (*DeserializeErrorFunction)(::DBusMessage* m, ::asf::core::Payload&);

    typedef bool (*SerializeErrorFunction)(const ::asf::core::Payload&, ::DBusMessage* m);

    DBusPayloadDecorator();

    void init(::asf::core::PayloadSharedPtr payload,
              SerializeFunction serializeFunction,
              DeserializeFunction deserializeFunction,
              JsonFunction jsonFunction,
              DeserializeErrorFunction _deserializeErrorFunction,
              SerializeErrorFunction _serializeErrorFunction);

    ::boost::shared_ptr< ::asf::core::Payload > getPayload() const;

    void toJson(std::string& s) const;

    void serialize(::DBusMessageIter* s) const;

    void deserialize(::DBusMessageIter* s);

    void deserializeError(::DBusMessage* m) const;

    void serializeError(::DBusMessage* m) const;

private:
    ::asf::core::PayloadSharedPtr _payload;

    SerializeFunction _serializeFunction;

    DeserializeFunction _deserializeFunction;

    JsonFunction _jsonFunction;

    DeserializeErrorFunction _deserializeErrorFunction;

    SerializeErrorFunction _serializeErrorFunction;

    bool _isDeserialized;

    DECLARE_CLASS_LOGGER();
};
}  // namespace dbus
}  // namespace asf

#endif
