/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DBusMessage.h
 *\brief
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_DBUS_DBUSMESSAGE_H
#define ASF_DBUS_DBUSMESSAGE_H

#include "asf/core/Logger.h"
#include "asf/core/Payload.h"
#include "asf/core/Proxy.h"
#include "asf/core/ServiceMessage.h"

#include "asf/threading/Mutex.h"

#include "asf/dbus/DBusPayloadDecorator.h"
#include "asf/dbus/DBusProxy.h"
#include "asf/dbus/DBusTypesJson.h"

#include "asf/dbus/org/freedesktop/DBus/PropertiesDBus.h"

#include <boost/weak_ptr.hpp>

#include <dbus/dbus.h>

class DBusProxyTest;
class DBusMessageTest;

namespace asf {
namespace dbus {

class DBusProxyDelegate;
class DBusStubDelegate;
class DBusProxyCallback;

// these macros are used by the generated code

#define DBUS_MESSAGE_CALL(var, call, payloadType, payloadInstance, hasReply) \
    ::asf::dbus::DBusMessage var(0, _dbusProxyDelegate);                     \
    var.prepareMethodCall(_dbusProxyDelegate->getDBusBusName(),              \
                          _dbusProxyDelegate->getDBusObjectPath(),           \
                          _dbusProxyDelegate->getInterfaceName(),            \
                          call,                                              \
                          hasReply);                                         \
    INIT_DBUS_PAYLOAD_DECORATOR_FOR_SERIALIZATION(                           \
        var.getPayloadDecorator(), payloadType, payloadInstance);

#define DBUS_MESSAGE_REPLY(var, reply, payloadType, payloadInstance)                               \
    ::boost::shared_ptr< ::asf::dbus::DBusMessage > var =                                          \
        static_cast< ::asf::dbus::DBusMessage& >(                                                  \
            *(_stubDelegate->getRegisteredMessage(act, _stubDelegate->getInterfaceName(), reply))) \
            .createMethodReturn(_stubDelegate);                                                    \
    _stubDelegate->deregisterMessage(act);                                                         \
    INIT_DBUS_PAYLOAD_DECORATOR_FOR_SERIALIZATION(                                                 \
        var->getPayloadDecorator(), payloadType, payloadInstance);

#define DBUS_MESSAGE_ERRORREPLY(var, reply, payloadType, payloadInstance)                          \
    CHECK_ALLOCATION(payloadInstance);                                                             \
    ::boost::shared_ptr< ::asf::dbus::DBusMessage > var =                                          \
        static_cast< ::asf::dbus::DBusMessage& >(                                                  \
            *(_stubDelegate->getRegisteredMessage(act, _stubDelegate->getInterfaceName(), reply))) \
            .createMethodError(                                                                    \
                payloadInstance->getName(), payloadInstance->getMessage(), _stubDelegate);         \
    _stubDelegate->deregisterMessage(act);                                                         \
    INIT_DBUS_PAYLOAD_DECORATOR_FOR_ERROR_SERIALIZATION(                                           \
        var->getPayloadDecorator(), payloadType, payloadInstance);

#define DBUS_MESSAGE_SIGNAL(var, signal, payloadType, payloadInstance)                  \
    ::asf::dbus::DBusMessage var(0, _stubDelegate);                                     \
    var.prepareSignal(                                                                  \
        _stubDelegate->getDBusObjectPath(), _stubDelegate->getInterfaceName(), signal); \
    INIT_DBUS_PAYLOAD_DECORATOR_FOR_SERIALIZATION(                                      \
        var.getPayloadDecorator(), payloadType, payloadInstance);

/**
 * A DBusMessage is the ASF representation of a message defined from D-Bus, see
 * http://www.freedesktop.org/wiki/Software/dbus.
 */
class DBusMessage : public ::asf::core::ServiceMessage {
public:
    /**
     * This constructor is used by the DBusPropertiesProxyWrapper to construct an internal property
     * update message.
     *
     * The _isPropertyReply parameter is set when the property update was initiated by a get
     * or set property operation instead
     * of an property update signal.
     */
    DBusMessage(::DBusMessage* libdbusMessage,
                DBusMessageIter* iter,
                ::boost::shared_ptr< DBusProxyDelegate > proxyDelegate,
                bool propertyReply = false);

    DBusMessage(::DBusMessage* libdbusMessage,
                ::boost::shared_ptr< DBusProxyDelegate > proxyDelegate);

    DBusMessage(::DBusMessage* libdbusMessage, DBusStubDelegate* stub);

    DBusMessage(const DBusMessage& src);

    ~DBusMessage();

    /**
     * Returns the decoded payload (runs the deserializeFunction if not deserialized yet)
     */
    ::boost::shared_ptr< ::asf::core::Payload > getPayload();

    virtual ::asf::core::MessageSharedPtr clone() const {
        return ::asf::core::MessageSharedPtr(new DBusMessage(*this));
    };

    virtual void processMessage();

    virtual bool isRemote() const { return true; }

    virtual bool sendRemote();

    void logMessageHeader(::asf::core::Logger& _logger);

    void logInfoMessage(::asf::core::Logger& _logger);

    void prepareMethodCall(const std::string& busName,
                           const std::string& objectPath,
                           const std::string& interfaceName,
                           const std::string& methodName,
                           bool hasReply = true);

    void prepareSignal(const std::string& objectPath,
                       const std::string& interfaceName,
                       const std::string& signalName);

    void prepareSignal(const std::string& signalName);

    ::boost::shared_ptr< DBusMessage > createMethodReturn(DBusStubDelegate* stub) const {
        ::DBusMessage* libdbusMessageReturn = dbus_message_new_method_return(_libdbusMessage);
        LOG_ASSERT(libdbusMessageReturn);
        return ::boost::shared_ptr< DBusMessage >(
            new DBusMessage(libdbusMessageReturn, stub, _memberName));
    }

    ::boost::shared_ptr< DBusMessage > createMethodError(const std::string& name,
                                                         const std::string& message,
                                                         DBusStubDelegate* stub) const {
        ::DBusMessage* libdbusMessageReturn =
            dbus_message_new_error(_libdbusMessage, name.c_str(), message.c_str());
        LOG_ASSERT(libdbusMessageReturn);
        dbus_message_set_reply_serial(libdbusMessageReturn,
                                      dbus_message_get_serial(_libdbusMessage));
        return ::boost::shared_ptr< DBusMessage >(
            new DBusMessage(libdbusMessageReturn, stub, _memberName));
    }

    ::asf::core::Proxy* getProxy() const;

    ::boost::shared_ptr< ::asf::dbus::DBusProxy > getDBusProxyShared() const;

    ::asf::dbus::DBusPayloadDecorator& getPayloadDecorator();

    ::DBusMessage* getLibDBusMessage() { return _libdbusMessage; }

    uint32 getSerial() { return dbus_message_get_serial(_libdbusMessage); }

    uint32 getReplySerial() {
        if (_libdbusMessage) {
            return dbus_message_get_reply_serial(_libdbusMessage);
        } else {
            return 0;
        }
    }

    const char* getMember(void) { return dbus_message_get_member(_libdbusMessage); }

    const std::string& getMemberName(void) const { return _memberName; }

    const char* getInterfaceName(void) const { return dbus_message_get_interface(_libdbusMessage); }

    void setMemberName(std::string memberName) { _memberName = memberName; }

    void setReplyCallback(::boost::shared_ptr< DBusProxyCallback >& cb) { _dbusProxyCallback = cb; }

    const ::boost::shared_ptr< DBusProxyCallback >& getReplyCallback(void) const {
        return _dbusProxyCallback;
    }

    bool isError() const {
        if (_libdbusMessage) {
            return DBUS_MESSAGE_TYPE_ERROR == dbus_message_get_type(_libdbusMessage);
        }
        return false;
    }

    bool isMethodReturn() const {
        if (_libdbusMessage) {
            return DBUS_MESSAGE_TYPE_METHOD_RETURN == dbus_message_get_type(_libdbusMessage);
        }
        return false;
    }

    bool isSignal() const {
        if (_libdbusMessage) {
            return DBUS_MESSAGE_TYPE_SIGNAL == dbus_message_get_type(_libdbusMessage);
        }
        return false;
    }

    bool isProperty() const { return (isPropertyUpdateMessage() || isPropertyReply()); }

    int getDBusMessageType() const {
        if (_libdbusMessage) {
            return dbus_message_get_type(_libdbusMessage);
        }
        return DBUS_MESSAGE_TYPE_INVALID;
    }

    DBusMessageIter* getPropertyPayloadIter() const { return _propPayloadIter; }

    bool isPropertyUpdateMessage() const {
        return !_isPropertyReply && (_propPayloadIter == 0 ? false : true);
    }

    bool isPropertyReply() const { return _isPropertyReply; }

    bool isPropertyError() const { return _isPropertyError; }

    void setPropertyError(::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError > error) {
        _dbusError = error;
        _isPropertyError = true;
    }

    ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError > getPropertyError() const {
        return _dbusError;
    }

    /**
     * D-Bus requires an unique serial number per connection for sending a message. This function
     * is used to get a unique D-Bus serial number (uint32) derived from the unique message id
     * (uint64).
     * If the message id exceeds the limit of UINT32_MAX an assertion is thrown. The assumption is
     * that this scenario is only a theoretical case. E.g. sending out 1000 messages per second
     * per connection the overflow happens after 49 days.
     *
     * @return D-Bus unique serial number derived from the message id
     */
    uint32 getUniqueSerial() const {
        // checked in constructor that the messageId not exceeds limit of uint32
        return static_cast< uint32 >(getMessageId());
    }

private:
    /**
     * This Constructor is used for testing purposes only
     */
    DBusMessage();

    /**
     * This Constructor is used for testing purposes only
     */
    DBusMessage(::DBusMessage* libdbusMessage,
                ::boost::shared_ptr< DBusProxyDelegate > proxyDelegate,
                const std::string& member);

    /**
     * This Constructor is used for testing purposes only
     */
    DBusMessage(::DBusMessage* libdbusMessage, DBusStubDelegate* stub, const std::string& member);

    // Outgoing message

    void serializePayload();

    // Incoming message

    void deserializePayload();

    /**
     * The messageId will be used as the serial number (uint32) of a D-Bus message.
     * An assertion will be thrown when the atomic counter of the ASF base messageId exceeds the
     * limit of UINT32_MAX.
     */
    void assertMessageIdOverflow() const {
        ::asf::msgId_t messageId = getMessageId();
        LOG_ASSERT_FATAL_MSG(messageId <= UINT32_MAX,
                             "Overflow, messageId exceeds limit of UINT32_MAX");
    }

    ::DBusMessage* refDBusMessage(::DBusMessage*) const;

    ::boost::shared_ptr< DBusProxyDelegate > init(::boost::shared_ptr< DBusProxyDelegate > const);

    DBusStubDelegate* init(DBusStubDelegate* const);

    ::DBusMessage* _libdbusMessage;

    DBusStubDelegate* _stub;

    ::boost::weak_ptr< DBusProxyDelegate > _dbusProxyDelegate;

    /**
     * The weak pointer to the DBusProxy is used to check the validity of the DBusProxyDelegate
     * pointer. The DBusProxyDelegate pointer might get invalid when the user deletes the DBusProxy.
     * To avoid message processing on an invalid DBusProxyDelegate pointer, the weak pointer to the
     * DBusProxy is checked for expiration before.
     */
    ::boost::weak_ptr< ::asf::dbus::DBusProxy > _dbusWeakProxy;

    std::string _memberName;

    ::boost::shared_ptr< DBusProxyCallback > _dbusProxyCallback;

    DBusMessageIter* _propPayloadIter;

    bool _isPropertyReply;

    bool _isPropertyError;

    ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError > _dbusError;

    ::asf::dbus::DBusPayloadDecorator _payloadDecorator;

    DECLARE_CLASS_LOGGER();

    friend class ::DBusProxyTest;

    friend class ::DBusMessageTest;
};

}  // namespace dbus
}  // namespace asf

#endif
