/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusConnector.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSCONNECTOR_H
#define ASF_DBUS_DBUSCONNECTOR_H

#include "asf/core/Application.h"
#include "asf/core/CommunicationStack.h"
#include "asf/core/Logger.h"
#include "asf/core/StartStopWaitIF.h"

#include "asf/threading/Mutex.h"

#include "asf/dbus/DBusConnection.h"
#include "asf/dbus/DBusExportedPort.h"
#include "asf/dbus/DBusExports.h"
#include "asf/dbus/DBusImportedPort.h"
#include "asf/dbus/DBusImports.h"
#include "asf/dbus/DBusMessageFactory.h"
#include "asf/dbus/DBusProxyRegistry.h"
#include "asf/dbus/DBusSerialRegistry.h"
#include "asf/dbus/DBusStubRegistry.h"

#include <map>

class DBusProxyAvailablilityTest;
class DBusProxyRegistryTest;
class DBusProxyTest;
class DBusStubTest;
class DBusMessageFactoryTest;

namespace asf {
namespace dbus {

/**
 * Callback to track the addition and removal of D-Bus stubs.
 */
class DBusStubDelegateCallbackIF {
public:
    typedef ::std::vector< DBusStubDelegate* > DBusStubDelegates;

    virtual ~DBusStubDelegateCallbackIF() {}

    /**
     * Notifies listeners about D-Bus interfaces which were added.
     * This callback is invoked after the interfaces are added.
     *
     * @param objectPath the object path on which interfaces are added (may differ from the
     * object path used when registering the callback)
     * @param stubDelegates The list of DBusStubDelegates which have been added.
     * @param addObjectPath A flag to indicate that this is a new objectpath not seen before
     */
    virtual void onDBusStubDelegatesAdded(const ::std::string& objectPath,
                                          const DBusStubDelegates& stubDelegates,
                                          bool addObjectPath) = 0;

    /**
     * Notifies listeners about the removal of D-Bus interfaces.
     * This callback is invoked before the interfaces are removed.
     *
     * @param objectPath the object path on which interfaces are added (may differ from the
     * object path used when registering the callback)
     * @param stubDelegates The list of DBusStubDelegates which have been added.
     * The stubDelegates will be deleted after returning from the callback.
     * @param removeObjectPath A flag to indicate that this is objectpath is to be removed
     */
    virtual void onDBusStubDelegatesRemoval(const ::std::string& objectPath,
                                            const DBusStubDelegates& stubDelegates,
                                            bool removeObjectPath) = 0;
};

class DBusConnector : public ::asf::core::StartStopWaitIF,
                      private ::asf::dbus::DBusStubDelegateCallbackIF {
public:
    DBusConnector(::asf::core::Application& app,
                  const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options);

    DBusConnector(::asf::core::Application& app,
                  ::boost::shared_ptr< ::asf::dbus::DBusStubRegistry > stubRegistry);

    virtual ~DBusConnector();

    virtual void start();

    virtual void stop();

    virtual void waitForCompletion();

    virtual DBusConnectionSharedPtr getConnection() const;

    virtual ::asf::core::CommunicationStack* getCommunicationStack();

    static DBusConnector* getConnector(
        const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options);

    DBusImportedPort& importByName(const std::string& busName, const std::string& objectPath);

    /**
     * This method is called by the generated application to export the appropriate dbus object
     */
    DBusExportedPort& exportByName(::asf::core::ProvidedPort& providedPort,
                                   const std::string& objectPath);

    /**
     * This method is called by the generated application to export the appropriate dbus object
     */
    bool exportAtRuntime(::asf::core::ProvidedPort& providedPort);

    /**
     * This method is called by the framework when the D-Bus connection is established.
     */
    void onConnect(::asf::core::ConnectionIFSharedPtr connection);

    /**
     * This method is called by the framework when the D-Bus connection was lost.
     */
    void onDisconnect(::asf::core::ConnectionIFSharedPtr connection);

    /**
     * Deregister a proxy from the proxy registry.
     */
    bool deregisterProxyDelegate(boost::shared_ptr< DBusProxyDelegate >& proxy);

    /**
     * Register a proxy to the proxy registry.
     */
    static bool registerProxyDelegate(const ::boost::shared_ptr< DBusProxyDelegate >& proxyDelegate,
                                      const ::boost::shared_ptr< ::asf::core::Proxy >& proxy);

    /**
     * Register a wired-at-runtime-proxy to the proxy registry.
     */
    static bool registerProxyDelegate(
        const ::boost::shared_ptr< DBusProxyDelegate >& proxyDelegate,
        const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
        const std::string& busName,
        const std::string& objectPath,
        const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& connectorOptions);

    /**
     * Register a stub to the stub registry. This method is used for the internal stubs only.
     */
    void registerStubDelegate(DBusStubDelegate* stubDelegate) const;

    /**
     * De-register a stub from the stub registry.
     */
    bool deregisterStubDelegate(DBusStubDelegate* stubDelegate) const;

    /**
     * Find the first level of objects underneath an object path in the stub registry.
     */
    void findChildObjects(const std::string& objectPath,
                          std::vector< ::std::string >& objects) const;

    /**
     * Find all objects path underneath an object path in the stub registry.
     */
    void findSubObjectPaths(const std::string& objectPath,
                            std::vector< ::std::string >& objects) const;

    /**
     * Find a stub delegates in the registry by the given parameters.
     */
    DBusStubDelegate* findStubDelegate(const std::string& objectPath,
                                       const std::string& interfaceName) const;

    /**
     * Find all stub delegates for a given object in the stub registry.
     */
    void findStubDelegates(const std::string& objectPath,
                           std::vector< DBusStubDelegate* >& stubDelegates) const;

    /**
     * This method returns a proxy which is bound to a given port of the current component.
     */
    static ::boost::shared_ptr< ::asf::core::Proxy > getProxy(
        const std::string& portName, const ::asf::core::ServiceAvailableIF& serviceAvailable);

    /**
     * This method signals the availability of the service. When all stubs in the server components
     * are created and registered the bus name will
     * be requested. When all stubs are deleted the bus name will be released.
     */
    void announceServiceAvailability(bool available);

    /**
     * This function returns if the connection to the bus is available.
     */
    bool isConnected() const;

    ::boost::shared_ptr< ::asf::dbus::DBusSerialRegistry > getSerialRegistry() const;

    void addInterfaceListener(DBusStubDelegateCallbackIF* cb,
                              const ::std::string& objectPath) const;

    bool removeInterfaceListener(DBusStubDelegateCallbackIF* cb) const;

    const ::std::string& getDBusBusName() const;

    virtual void onDBusStubDelegatesAdded(const ::std::string& objectPath,
                                          const ::std::vector< DBusStubDelegate* >& stubDelegates,
                                          bool addObjectPath);

    virtual void onDBusStubDelegatesRemoval(const ::std::string& objectPath,
                                            const ::std::vector< DBusStubDelegate* >& stubDelegates,
                                            bool removeObjectPath);

private:
    DBusConnector(const DBusConnector&);

    DBusConnector& operator=(const DBusConnector&);

    typedef std::map< ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector, DBusConnector* >
        RegistryMap;

    static RegistryMap _registry;

    struct ServiceId {
        std::string _busName;
        std::string _objectPath;
        std::string _interfaceName;

        bool operator<(const ServiceId& a) const;
    };

    typedef enum {
        STATUS_UNKNOWN = 0,
        STATUS_ALL_STARTED,
        STATUS_ALL_STOPPED,
        STATUS_MIXED
    } ServiceStatus;

    ServiceStatus checkServiceStatus();
    /**
     * Set the service to registered or unregistered in the exported service registry.
     */
    void setExportedServiceRegistration(DBusStubDelegate* stub, bool registered);

    /**
     * Add all exported services to the service registry.
     */
    void addToExportedServiceRegistry(const std::string& busName,
                                      const std::string& objectPath,
                                      const std::string& interfaceName);

    template < class T >
    void createComponentStub(const std::string& portName,
                             const std::string& busName,
                             const std::string& objectPath,
                             const std::string& interface,
                             ::asf::core::ProvidedPort* providedPort);

    ::boost::shared_ptr< ::asf::dbus::DBusSerialRegistry > _serialRegistry;

    ::asf::dbus::DBusProxyRegistry _proxyRegistry;

    ::boost::shared_ptr< ::asf::dbus::DBusStubRegistry > _stubRegistry;

    ::asf::dbus::DBusMessageFactory* _messageFactory;

    ::asf::core::ProxyCommunicationStack* _comStack;

    ::asf::dbus::DBusConnectionSharedPtr _dbusConnection;

    ::asf::core::Application& _app;

    const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector _options;

    std::string _busName;

    typedef std::vector< std::string > PropertyRegistrationList;

    PropertyRegistrationList _propertyRegistry;

    typedef std::map< ServiceId, bool > ServiceRegistrationMap;

    ServiceRegistrationMap _serviceRegistry;

    ::asf::threading::Mutex _lockServiceRegistry;

    bool _isConnected;

    ::asf::threading::Mutex _lockIsConnected;

    DBusImports _imports;

    DBusExports _exports;

    DECLARE_CLASS_LOGGER();

    friend class ::DBusProxyAvailablilityTest;
    friend class ::DBusProxyRegistryTest;
    friend class ::DBusProxyTest;
    friend class ::DBusStubTest;
    friend class ::DBusMessageFactoryTest;
};

}  // namespace dbus
}  // namespace asf

#endif  // ASF_DBUS_DBUSCONNECTOR_H
