/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DBusConnection.h
 *\brief
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_DBUS_DBUSCONNECTION_H
#define ASF_DBUS_DBUSCONNECTION_H

#include "asf/core/IncomingOutgoingIF.h"
#include "asf/core/Logger.h"
#include "asf/core/Types.h"

#include "asf/dbus/DBusDispatcher.h"

#include "asf/threading/Mutex.h"
#include "asf/threading/Signal.h"
#include "asf/threading/ThreadRunnableExecutor.h"

#include "com/bosch/cm/asf/lang/dbus/Connectors.h"

#include <boost/enable_shared_from_this.hpp>
#include <boost/shared_ptr.hpp>

#include <dbus/dbus.h>

class DBusConnectionTest;

namespace asf {
namespace dbus {

class DBusMessage;

class DBusConnection : public ::asf::core::OutgoingIF,
                       public ::asf::core::StartStopWaitIF,
                       public ::asf::core::ConnectionIF,
                       public ::asf::threading::RunnableIF,
                       public ::boost::enable_shared_from_this< DBusConnection > {
public:
    DBusConnection();

    DBusConnection(::asf::core::IncomingIF* incomingEvent,
                   const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& options);

    virtual ~DBusConnection();

    /**
     * Request application name on the bus, see
     * http://dbus.freedesktop.org/doc/dbus-specification.html#message-bus-names
     */
    void requestServiceNameAsync(const std::string& serviceName);

    /**
     * Release application name from the bus, see
     * http://dbus.freedesktop.org/doc/dbus-specification.html#message-bus-names
     */
    void releaseServiceNameAsync(const std::string& serviceName);

    /**
     * Check if the bus is owned by the given service name.
     * This method does a synchronous function call to the libdbus and should only be called from
     * the DBusDispatcher thread to avoid race conditions in the libdbus.
     */
    bool hasBusNameOwner(const std::string& serviceName);

    /**
     * This method registers the object path to libdbus.
     * This method does a synchronous function call to the libdbus and should only be called from
     * the DBusDispatcher thread to avoid race conditions in the libdbus.
     */
    bool registerObjectPath(const std::string& objectPath);

    /**
     * This method unregisters the object path from libdbus.
     * This method does a synchronous function call to the libdbus and should only be called from
     * the DBusDispatcher thread to avoid race conditions in the libdbus.
     */
    bool unregisterObjectPath(const std::string& objectPath);

    // implement OutgoingIF
    virtual ::asf::core::ConnectionIFSharedPtr connectServer(const std::string& /*address*/);

    virtual void disconnectServer(::asf::core::ConnectionIFSharedPtr /*connection*/);

    virtual ssize_t sendData(::asf::core::ConnectionIFSharedPtr connection,
                             size_t size,
                             const uint8* data);

    // implement StartStopWaitIF
    virtual void start();

    virtual void stop();

    virtual void waitForCompletion();

    // RunnableIF implementation.
    virtual ::asf::threading::RunnableIF::tenRun enRun();

    // ConnectionIF implementation
    virtual ssize_t sendData(size_t size, const uint8* data);

    virtual ::asf::identifier_t getConnectionId() {
        return 0;  // The connectionId is not used by the asfdbus binding
    }

    bool isConnected() const {
        return (_dbusConnection && TRUE == dbus_connection_get_is_connected(_dbusConnection));
    }

    const ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& getConnectorOptions()
        const {
        return _options;
    }

    ::boost::shared_ptr< DBusConnection > selfSharedPtr() { return shared_from_this(); }

private:
    DBusConnection(const DBusConnection&);

    DBusConnection& operator=(const DBusConnection&);

    static ::DBusHandlerResult messageFilterCallback(::DBusConnection* dbusConnection,
                                                     ::DBusMessage* libdbusMessage,
                                                     void* userData);

    static ::DBusHandlerResult objectFilterCallback(::DBusConnection* dbusConnection,
                                                    ::DBusMessage* libdbusMessage,
                                                    void* userData);

    static void objectUnregisterCallback(::DBusConnection* dbusConnection, void* userData);

    static DBusObjectPathVTable libdbusObjectPathVTable_;

    /**
     * This method does the connection to libdbus
     */
    bool connect();

    /**
     * Disconnect from libdbus
     */
    void disconnect();

    void onDisconnected();

    void onConnected();

    bool establishConnection();

    void logConnectionAttemps(size_t connectionAttempts) const;

    ::DBusHandlerResult messageFilterHandler(::DBusMessage* dbusMessage);

    ::DBusHandlerResult objectFilterHandler(::DBusMessage* dbusMessage);

    ::DBusBusType getDBusBusType(const std::string& busType);

    bool processDaemonMessage(::DBusMessage* dbusMessage);

    bool isDaemonMessage(::DBusMessage* dbusMessage) const;

    void sendRequestNameRequest(const ::std::string& busName, uint32 flag);

    void onRequestNameResponse(::DBusMessage* dbusMessage) const;

    void sendReleaseNameRequest(const ::std::string& busName);

    void onReleaseNameResponse(::DBusMessage* dbusMessage) const;

    void onDBusError(::DBusMessage* dbusMessage) const;

    bool connectAndRegisterToBus();

    /**
     * The given D-Bus message will be sent by the DBusDispatcher.
     */
    void sendMessage(::DBusMessage* dbusMessage) const;

    void sendDaemonRequestMessage(::DBusMessage* dbusMessage);

    bool isNameAcquiredSignal(::DBusMessage* dbusMessage) const;

    bool isDisconnectedSignal(::DBusMessage* dbusMessage) const;

    bool isNameLostSignal(::DBusMessage* dbusMessage) const;

    void processBusNameQueryReply(const ::std::string& query,
                                  ::DBusMessage* dbusMessage,
                                  uint32 replySerial);

    ::DBusConnection* _dbusConnection;

    ::asf::core::IncomingIF* _incomingEvent;

    ::asf::threading::Signal _waitForCompletion;

    bool _isConnectionEstablished;

    ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector _options;

    ::asf::threading::ThreadRunnableExecutor* _threadRunnableExecutor;

    ::boost::shared_ptr< IDBusDispatcher > _dispatcher;

    // This lock protects the SerialToMemberNameMap map
    ::asf::threading::Mutex _lock;

    typedef std::map< uint32, std::string > SerialToMemberNameMap;

    SerialToMemberNameMap _serialMap;

    ms_t _connectionInterval;

    uint32 _connectionWarningThreshold;

    bool _stopEntered;

    ::std::string _busName;

    // Mixing synchronous and asynchronous calls to the libdbus results in undefined behavior.
    // Therefore a pending synchronous call must not be executed during a pending asynchronous call.
    // Additionally, only one asynchronous call must be executed at a time.
    // This signal protects an asynchronous call to the libdbus against concurrent asynchronous or
    // synchronous calls.
    ::asf::threading::Signal _asyncCallCompletionSignal;

    // This variable is needed to apply the signaling mechanism only in case of an pending
    // asynchronous call
    bool _asyncCallInProgress;

    friend class ::DBusConnectionTest;

    DECLARE_CLASS_LOGGER();
};

typedef ::boost::shared_ptr< DBusConnection > DBusConnectionSharedPtr;

}  // namespace dbus
}  // namespace asf

#endif  // ASF_DBUS_DBUSCONNECTION_H
