/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     Proxy.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

// !!!!!!!! When changing this file please update the ASF User Guide !!!!!!!!!!!

#ifndef ASF_CORE_PROXY_H
#define ASF_CORE_PROXY_H

#include "asf/core.h"
#include "asf/core/Types.h"

#include <boost/shared_ptr.hpp>

namespace asf {
namespace core {

class Proxy;
class ProxyDelegateBase;

/**
 * The ServiceAvailableIF is an interface which signals the service
 * availability of a proxy. Whenever the state of the service changes one
 * of the callbacks is invoked by ASF.
 *
 * The user has to provide an implementation of the ServiceAvailableIF with
 * each proxy instance.
 */
class ServiceAvailableIF {
public:
    virtual ~ServiceAvailableIF() {}

    /**
     * Called whenever the service state changed to state Available.
     */
    virtual void onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
                             const ServiceStateChange& stateChange) = 0;

    /**
     * Called whenever the service state changed to state Disconnected or Suspended.
     */
    virtual void onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
                               const ServiceStateChange& stateChange) = 0;
};

/**
 * The base class of all proxies. A proxy is the client side
 * representation of a service.
 *
 * This class is intended for subclassing.
 */
class Proxy {
public:
    virtual ~Proxy() {}

    /**
     * Returns the port name for which the proxy was created.
     */
    const std::string& getPortName() const { return _portName; }

    /**
     * Returns the state of the service. The proxy can only be used if the service
     * state equals "ServiceState__Available". When the state is
     * "ServiceState__Disconnected" and "ServiceState__Suspended" the service is not usable.
     */
    ::asf::core::ServiceState getServiceState() const { return _serviceState; }

    /**
     * Returns whether the service is available (convenience function)
     */
    bool isAvailable() const { return (_serviceState == ServiceState__Available); }

    ServiceAvailableIF* getServiceAvailableIF() { return &_serviceAvailable; }

protected:
    Proxy(const std::string& portName, ServiceAvailableIF& serviceAvailable)
        : _serviceState(ServiceState__Disconnected),
          _portName(portName),
          _serviceAvailable(serviceAvailable) {}

private:
    ::asf::core::ServiceState _serviceState;

    std::string _portName;

    ServiceAvailableIF& _serviceAvailable;

    friend class ProxyDelegateBase;
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_PROXY_H
