/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012-2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     MessageQueue.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_BASE_MESSAGE_QUEUE_HPP
#define ASF_BASE_MESSAGE_QUEUE_HPP

#include "asf/core/Logger.h"
#include "asf/core/Time.h"
#include "asf/core/Types.h"
#include "asf/threading/Guard.h"
#include "asf/threading/Signal.h"

#include <boost/shared_ptr.hpp>
#include <deque>
#include <string>

namespace asf {
namespace core {

extern ::asf::core::Logger messageQueueLogger;

/**
 * A synchronized and thread safe queue of messages. The elements of the
 * queue are stored as shared pointers.
 *
 * Every ComponentContainer has such a message queue.
 */
template < typename T >
class MessageQueue {
public:
    MessageQueue(const std::string& name)
        : _isClosed(false), _name(name), _maxFillLevel(0), _maxQueuingDuration(0) {
        SET_LOGGER(messageQueueLogger);
        LOG_DEBUG("MessageQueue '%s': created (this=%p)", _name.c_str(), this);
    }

    /**
     * Adds a message to the end of the queue. It can be read later
     * with the pop_front() method.
     */
    void enqueue(::boost::shared_ptr< T > t) {
        SET_LOGGER(messageQueueLogger);
        LOG_DEBUG("MessageQueue '%s': enqueue", _name.c_str());
        LOG_ASSERT(t.get());

        // Acquire lock on the queue
        ::asf::threading::Guard< ::asf::threading::Signal > guard(_signal);

        if (!_isClosed) {
            // Add the data to the queue
            _queue.push_back(t);
            _enqueueTimes.push_back(Time::getElapsedTimeMs());
            _maxFillLevel = std::max(_queue.size(), _maxFillLevel);
            LOG_DEBUG("MessageQueue '%s': size=%d", _name.c_str(), _queue.size());
            // Notify others that data is ready
            _signal.notify_one();
        } else {
            LOG_WARN("MessageQueue '%s': ignored element enqueued, queue is already closed",
                     _name.c_str());
        }
    }

    /**
     * Adds a message to the front (the reader side) of the queue.
     *
     * This function is used when re-adding deferred messages to the
     * queue. Those messages should be processed immediately and are
     * therefore add to the beginning of the queue.
     */
    void enqueue_front(::boost::shared_ptr< T > t) {
        SET_LOGGER(messageQueueLogger);
        LOG_DEBUG("MessageQueue '%s': enqueue_front", _name.c_str());
        LOG_ASSERT(t.get());

        // Acquire lock on the queue
        ::asf::threading::Guard< ::asf::threading::Signal > guard(_signal);

        if (!_isClosed) {
            // Add the data to the queue
            _queue.push_front(t);
            _enqueueTimes.push_front(Time::getElapsedTimeMs());
            _maxFillLevel = std::max(_queue.size(), _maxFillLevel);

            // Notify others that data is ready
            _signal.notify_one();
        } else {
            LOG_INFO(
                "MessageQueue '%s': ignored element enqueued to front, queue is already closed",
                _name.c_str());
        }
    }

    /**
     * Reads a message from the queue. This call will block until
     * the next message is available in the queue
     */
    ::boost::shared_ptr< T > dequeue() {
        SET_LOGGER(messageQueueLogger);
        LOG_DEBUG("MessageQueue '%s': dequeue", _name.c_str());

        // Acquire lock on the queue
        _signal.lock();

        // When there is no data, wait till someone fills it.
        // Lock is automatically released in the wait and obtained
        // again after the wait
        while (_queue.empty() && !isClosed()) _signal.wait();

        if (isClosed()) {
            LOG_DEBUG("MessageQueue '%s': queue is closed, returning null", _name.c_str());

            // Release the lock
            _signal.unlock();

            // Return null
            return ::boost::shared_ptr< T >((T*)0);
        } else {
            // Retrieve the data from the queue
            ::boost::shared_ptr< T > result = _queue.front();
            _queue.pop_front();
            _maxQueuingDuration =
                std::max(Time::getElapsedTimeMs() - _enqueueTimes.front(), _maxQueuingDuration);
            _enqueueTimes.pop_front();

            // Release the lock
            _signal.unlock();

            LOG_DEBUG("MessageQueue '%s': returning element", _name.c_str());

            return result;
        }
    }

    /**
     * Closes the queue. All readers will unblock and will receive a
     * null object.
     *
     * Used during the shut down of an application.
     */
    void close() {
        SET_LOGGER(messageQueueLogger);
        LOG_ASSERT(!_isClosed);
        LOG_DEBUG("MessageQueue '%s': close", _name.c_str());
        LOG_DEBUG("MessageQueue '%s': MaxFillLevel = '%d'", _name.c_str(), _maxFillLevel);
        LOG_DEBUG("MessageQueue '%s': MaxDuration = '%d' ms", _name.c_str(), _maxQueuingDuration);

        // Acquire lock on the queue
        _signal.lock();

        // set the closed flag
        _isClosed = true;

        // notify all readers
        _signal.notify_all();

        // Acquire lock on the queue
        _signal.unlock();
    }

    /**
     * Returns true if the message queue is closed
     */
    bool isClosed() const { return _isClosed; }

    /**
     * Returns the maximal fill level of the message queue.
     * The fill level is the number of current enqueued messages,
     * before they later will be dequeued.
     * Can be used to determine the max amount of memory.
     */
    size_t getMaxFillLevel() const { return _maxFillLevel; }

    /**
     * Returns the maximal queuing duration of an element
     * inside the message queue.
     * The duration is the time from the enqueue
     * till the same element is dequeued.
     * Can be used to determine the runtime.
     */
    size_t getMaxQueuingDuration() const { return _maxQueuingDuration; }

private:
    bool _isClosed;

    ::std::string _name;

    ::std::deque< ::boost::shared_ptr< T > > _queue;

    size_t _maxFillLevel;

    uint32 _maxQueuingDuration;

    ::std::deque< uint32 > _enqueueTimes;

    ::asf::threading::Signal _signal;
};

}  // namespace core
}  // namespace asf

#endif
