/****************************************************************************
 * Automotive Service Framework (ASF)
 *
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

#ifndef ASF_CORE_LOGGER_H
#define ASF_CORE_LOGGER_H

#include <cassert>
#include <cstdarg>
#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <errno.h>

#include "asf/core/Logging.h"
#include "asf/core/LoggingBackendIF.h"
#include "asf/core/LoggingConst.h"
#include "asf/core/Types.h"

namespace asf {
namespace core {

/**
 * A logger represents a specific channel to do log output. Each logger
 * has a name, the so called scope. Logging can happen on several levels
 * such as a warning or error level for instance. Each logger has a
 * current active logging level. This current level serves as a filter
 * which output is generated and which output is suppressed. The current
 * log level can be adjusted by the setLevel() method.
 *
 * Usage example 1:
 *
 * class Boo {
 *
 *  DECLARE_CLASS_LOGGER(); // declares a static instance called _logger
 *
 * };
 *
 * DEFINE_CLASS_LOGGER ("/foo", Bar); // defines the _logger, resulting
 * name is "foo/Bar".
 *
 * Then in a method of Foo:
 *
 * LOG_DEBUG ("Hello %s!", "World");

 * Usage example 2:
 *
 * DEFINE_LOGGER (mylogger, "/foo/bar") // Declare a logger with scope foo/bar
 *
 * Then in a method:
 *
 * SET_LOGGER (mylogger); // we use the previously defined logger
 * LOG_DEBUG ("Hello %s!",  "World"); // Do a log output on level DEBUG
 *
 * Macros like LOG_DEBUG are used to eliminate any overhead when the log
 * statement is suppressed. The formatting of the log output takes only place
 * when the logger is enabled.
 *
 * There is also a concept of conditional code blocks, which are only
 * executed if a certain log level is enabled:
 *
 * if (IS_LOG_DEBUG_ENABLED()) {
 *      // do some stuff here
 * }
 */
class Logger {
public:
    typedef ::asf::core::Logging::Level LoggerLevel;

    static const LoggerLevel Debug = ::asf::core::Logging::Level__Debug;

    static const LoggerLevel Info = ::asf::core::Logging::Level__Info;

    static const LoggerLevel Warn = ::asf::core::Logging::Level__Warn;

    static const LoggerLevel Error = ::asf::core::Logging::Level__Error;

    static const LoggerLevel Fatal = ::asf::core::Logging::Level__Fatal;

    Logger(const ::std::string& scope, Logger::LoggerLevel level = Logger::Error);

    Logger(const Logger& parentLogger,
           const ::std::string& scope,
           Logger::LoggerLevel level = Logger::Error);

    ~Logger();

    const std::string& getName() const { return _name; }

    void setLevel(LoggerLevel level) { _level = level; }

    LoggerLevel getLevel() const { return _level; }

    void setUserData(void* userData) { _userData = userData; }

    void* getUserData() const { return _userData; }

    bool isEnabled(LoggerLevel level) const {
        // The magic flag check is used to prevent logging via an uninitialized
        // logger (global initialization order problem)
        return level >= _level && _magic == 0xDEADBEAF && !_suppressAllLogs;
    }

    void debug(const void* _this, const char* format, unsigned int line, const char* file, ...);

    void debug(const char* format, unsigned int line, const char* file, ...);

    void info(const void* _this, const char* format, unsigned int line, const char* file, ...);

    void info(const char* format, unsigned int line, const char* file, ...);

    void warn(const void* _this, const char* format, unsigned int line, const char* file, ...);

    void warn(const char* format, unsigned int line, const char* file, ...);

    void error(const void* _this, const char* format, unsigned int line, const char* file, ...);

    void error(const char* format, unsigned int line, const char* file, ...);

    void fatal(const void* _this, const char* format, unsigned int line, const char* file, ...);

    void fatal(const char* format, unsigned int line, const char* file, ...);

    void vlog(const void* _this,
              LoggerLevel level,
              size_t line,
              const char* file,
              const char* format,
              va_list argptr);

    void memoryDump(const void* _this,
                    LoggerLevel level,
                    size_t line,
                    const char* file,
                    const uint8* buffer,
                    size_t length);

    /**
     * Sets the logging configuration from a json file. The file must represent
     * a structure from type asf.core.LoggingService.LoggerConfiguration, e.g.:
     *
     *  {
     *      "loggerLevel" : {
     *          "*" : "Error",
     *          "/foo/bar" : "Debug"
     *      }
     *  }
     *
     * The function aborts the process if a parsing error occured.
     */
    static void setupLogging(const std::string& appPackageName,
                             const std::string& appName,
                             const char* fileName = "logConfig.json",
                             const ::asf::core::Logging::LoggerConfiguration& defaultConfiguration =
                                 ::asf::core::Logging::LoggerConfiguration());

    static void setConfiguration(const ::asf::core::Logging::LoggerConfiguration& lc);

    static void getConfiguration(::asf::core::Logging::LoggerConfiguration& lc,
                                 const std::string& matchExpression);

    static Logger _systemLogger;

    static bool readConfigurationFromFile(const std::string& filename,
                                          ::asf::core::Logging::LoggerConfiguration& configuration);

    static bool setLoggerUserData(const ::std::string& logger, void* userData);

    static void* getLoggerUserData(const ::std::string& logger);

    static size_t setLoggingLevel(const ::std::string& logger, LoggerLevel level);

    static void setLoggingLevelToAll(LoggerLevel level);

    static bool readFile(const std::string& filename, std::string& buffer);

    static ::std::map< ::std::string, Logging::Level >& getCurrentLoggerLevels();

    static bool loadExternalBackend(
        const std::string& appPackageName,
        const std::string& appName,
        const std::map< std::string, std::string >& backendConfiguration,
        const ::std::string& backendLibraryPath,
        const ::std::string& backend,
        uint32 apiVersion);

    static void unloadExternalBackend();

    static void setSuppressAllLogs(bool suppress) { _suppressAllLogs = suppress; }

    static bool getSuppressAllLogs() { return _suppressAllLogs; }

private:
    void logFormat(LoggingBackendIF& backend,
                   const void* _this,
                   Logger::LoggerLevel level,
                   size_t line,
                   const char* file,
                   const char* format,
                   va_list argptr);

    void logBuffer(LoggingBackendIF& backend,
                   const void* _this,
                   Logger::LoggerLevel level,
                   size_t line,
                   const char* file,
                   const uint8* buffer,
                   size_t length);

    static bool _suppressAllLogs;

    LoggerLevel _level;

    ::std::string _name;

    void* _userData;

    unsigned int _magic;
};

#define DECLARE_CLASS_LOGGER() static ::asf::core::Logger _logger

#define DEFINE_CLASS_LOGGER(LOGGERNAME, CLASSNAME) \
    ::asf::core::Logger CLASSNAME::_logger(LOGGERNAME "/" #CLASSNAME)

#define DEFINE_CLASS_LOGGER_AND_LEVEL(LOGGERNAME, CLASSNAME, LEVEL) \
    ::asf::core::Logger CLASSNAME::_logger(LOGGERNAME "/" #CLASSNAME, ::asf::core::Logger::LEVEL)

// creates a static variable. The variable is static to guarantee that the variable is only
// available in the current file and
// that loggers in different file do not interleave with each other.

#define DEFINE_LOGGER(VARIABLE, LOGGERNAME) static ::asf::core::Logger VARIABLE(LOGGERNAME)

#define DEFINE_LOGGER_AND_LEVEL(VARIABLE, LOGGERNAME, LEVEL) \
    static ::asf::core::Logger VARIABLE(LOGGERNAME, ::asf::core::Logger::LEVEL)

#define SET_LOGGER(LOGGER) ::asf::core::Logger& _logger = LOGGER

#define LOG_DEBUG(FORMAT, ...) \
    if (IS_LOG_DEBUG_ENABLED()) _logger.debug(this, FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_DEBUG_STATIC(FORMAT, ...) \
    if (IS_LOG_DEBUG_ENABLED()) _logger.debug(FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_INFO(FORMAT, ...) \
    if (IS_LOG_INFO_ENABLED()) _logger.info(this, FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_INFO_STATIC(FORMAT, ...) \
    if (IS_LOG_INFO_ENABLED()) _logger.info(FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_WARN(FORMAT, ...) \
    if (IS_LOG_WARN_ENABLED()) _logger.warn(this, FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_WARN_STATIC(FORMAT, ...) \
    if (IS_LOG_WARN_ENABLED()) _logger.warn(FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_ERROR(FORMAT, ...) \
    if (IS_LOG_ERROR_ENABLED()) _logger.error(this, FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_ERROR_STATIC(FORMAT, ...) \
    if (IS_LOG_ERROR_ENABLED()) _logger.error(FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_SYSTEM_ERROR_STATIC(FORMAT, ...) \
    if (IS_SYSTEMLOG_ERROR_ENABLED())        \
    ::asf::core::Logger::_systemLogger.error(FORMAT, __LINE__, __FILE__, ##__VA_ARGS__)

#define LOG_FATAL(FORMAT, ...)                                              \
    do {                                                                    \
        if (IS_LOG_FATAL_ENABLED()) {                                       \
            _logger.fatal(this, FORMAT, __LINE__, __FILE__, ##__VA_ARGS__); \
        }                                                                   \
        ::std::abort();                                                     \
    } while (0)

#define LOG_FATAL_STATIC(FORMAT, ...)                                 \
    do {                                                              \
        if (IS_LOG_FATAL_ENABLED()) {                                 \
            _logger.fatal(FORMAT, __LINE__, __FILE__, ##__VA_ARGS__); \
        }                                                             \
        ::std::abort();                                               \
    } while (0)

#define LOG_DEBUG_MEMORY_DUMP(buffer, length) \
    if (IS_LOG_DEBUG_ENABLED())               \
    _logger.memoryDump(this, ::asf::core::Logger::Debug, __LINE__, __FILE__, buffer, length)

#define LOG_INFO_MEMORY_DUMP(buffer, length) \
    if (IS_LOG_INFO_ENABLED())               \
    _logger.memoryDump(this, ::asf::core::Logger::Info, __LINE__, __FILE__, buffer, length)

#define LOG_WARN_MEMORY_DUMP(buffer, length) \
    if (IS_LOG_WARN_ENABLED())               \
    _logger.memoryDump(this, ::asf::core::Logger::Warn, __LINE__, __FILE__, buffer, length)

#define LOG_ERROR_MEMORY_DUMP(buffer, length) \
    if (IS_LOG_ERROR_ENABLED())               \
    _logger.memoryDump(this, ::asf::core::Logger::Error, __LINE__, __FILE__, buffer, length)

#define LOG_FATAL_MEMORY_DUMP(buffer, length) \
    if (IS_LOG_FATAL_ENABLED())               \
    _logger.memoryDump(this, ::asf::core::Logger::Fatal, __LINE__, __FILE__, buffer, length)

#define LOG_DEBUG_MEMORY_DUMP_STATIC(buffer, length) \
    if (IS_LOG_DEBUG_ENABLED())                      \
    _logger.memoryDump(0, ::asf::core::Logger::Debug, __LINE__, __FILE__, buffer, length)

#define LOG_INFO_MEMORY_DUMP_STATIC(buffer, length) \
    if (IS_LOG_INFO_ENABLED())                      \
    _logger.memoryDump(0, ::asf::core::Logger::Info, __LINE__, __FILE__, buffer, length)

#define LOG_WARN_MEMORY_DUMP_STATIC(buffer, length) \
    if (IS_LOG_WARN_ENABLED())                      \
    _logger.memoryDump(0, ::asf::core::Logger::Warn, __LINE__, __FILE__, buffer, length)

#define LOG_ERROR_MEMORY_DUMP_STATIC(buffer, length) \
    if (IS_LOG_ERROR_ENABLED())                      \
    _logger.memoryDump(0, ::asf::core::Logger::Error, __LINE__, __FILE__, buffer, length)

#define LOG_FATAL_MEMORY_DUMP_STATIC(buffer, length) \
    if (IS_LOG_FATAL_ENABLED())                      \
    _logger.memoryDump(0, ::asf::core::Logger::Fatal, __LINE__, __FILE__, buffer, length)

#define IS_LOG_DEBUG_ENABLED() _logger.isEnabled(::asf::core::Logger::Debug)

#define IS_LOG_INFO_ENABLED() _logger.isEnabled(::asf::core::Logger::Info)

#define IS_LOG_WARN_ENABLED() _logger.isEnabled(::asf::core::Logger::Warn)

#define IS_LOG_ERROR_ENABLED() _logger.isEnabled(::asf::core::Logger::Error)

#define IS_SYSTEMLOG_ERROR_ENABLED() \
    ::asf::core::Logger::_systemLogger.isEnabled(::asf::core::Logger::Error)

#define IS_LOG_FATAL_ENABLED() _logger.isEnabled(::asf::core::Logger::Fatal)

// -------------------------------------------------------------------
// ASSERT macros
// -------------------------------------------------------------------
//
// --- LOG_ASSERT family macros ---
//
// All macros called LOG_ASSERT do assertions only on *non* release builds.
// If the symbol NDEBUG is defined all assertions will be removed.
//
// LOG_ASSERT - assert
//
// LOG_ASSERT_STATIC - assert static (without printing the this pointer)
//
// LOG_ASSERT_MSG - assert with a message to explain the assertion, this pointer is logged
// automatically
//
// LOG_ASSERT_MSG_STATIC - assert with a message to explain the assertion
//
// --- LOG_ASSERT_FATAL family macros ---
//
// All macros called LOG_ASSERT_FATAL do assertions always. They are
// not removed from release builds
//
// LOG_ASSERT_FATAL - assert
//
// LOG_ASSERT_FATAL_STATIC - assert static (without printing the this pointer)
//
// LOG_ASSERT_FATAL_MSG - assert with a message to explain the assertion, this pointer is logged
// automatically
//
// LOG_ASSERT_FATAL_MSG_STATIC - assert with a message to explain the assertion
//
// -------------------------------------------------------------------

#define LOG_SYSTEM_LOGGER()                                            \
    ::asf::core::Logger& _logger = ::asf::core::Logger::_systemLogger; \
    ((void)_logger)  // to get rid of unused variable warning

// LOG_ASSERT

#ifndef NDEBUG
#define LOG_ASSERT(expr)                                                                    \
    do {                                                                                    \
        if (!(expr)) {                                                                      \
            LOG_ERROR("Assertion failed: %s, file %s, line %d", #expr, __FILE__, __LINE__); \
            ::std::abort();                                                                 \
        }                                                                                   \
    } while (0)
#else
#define LOG_ASSERT(expr) ((void)0)
#endif

#ifndef NDEBUG
#define LOG_ASSERT_STATIC(expr)                                                                    \
    do {                                                                                           \
        if (!(expr)) {                                                                             \
            LOG_ERROR_STATIC("Assertion failed: %s, file %s, line %d", #expr, __FILE__, __LINE__); \
            ::std::abort();                                                                        \
        }                                                                                          \
    } while (0)

#else
#define LOG_ASSERT_STATIC(expr) ((void)0)
#endif

#ifndef NDEBUG
#define LOG_ASSERT_MSG(expr, FORMAT, ...)                               \
    do {                                                                \
        if (!(expr)) {                                                  \
            if (IS_LOG_ERROR_ENABLED()) {                               \
                _logger.error((const void*)this,                        \
                              "Assertion failed: %s, file %s, line %d", \
                              __LINE__,                                 \
                              __FILE__,                                 \
                              "" #expr,                                 \
                              "" __FILE__,                              \
                              __LINE__);                                \
            }                                                           \
            LOG_ERROR(FORMAT, ##__VA_ARGS__);                           \
            ::std::abort();                                             \
        }                                                               \
    } while (0)
#else
#define LOG_ASSERT_MSG(expr, FORMAT, ...) ((void)0)
#endif

#ifndef NDEBUG
#define LOG_ASSERT_MSG_STATIC(expr, FORMAT, ...)                        \
    do {                                                                \
        if (!(expr)) {                                                  \
            if (IS_LOG_ERROR_ENABLED()) {                               \
                _logger.error("Assertion failed: %s, file %s, line %d", \
                              __LINE__,                                 \
                              __FILE__,                                 \
                              "" #expr,                                 \
                              "" __FILE__,                              \
                              __LINE__);                                \
            }                                                           \
            LOG_ERROR_STATIC(FORMAT, ##__VA_ARGS__);                    \
            ::std::abort();                                             \
        }                                                               \
    } while (0)
#else
#define LOG_ASSERT_MSG_STATIC(expr, FORMAT, ...) ((void)0)
#endif

// LOG_ASSERT_FATAL

#define LOG_ASSERT_FATAL(expr)                                          \
    do {                                                                \
        if (!(expr)) {                                                  \
            if (IS_LOG_FATAL_ENABLED()) {                               \
                _logger.fatal((const void*)this,                        \
                              "Assertion failed: %s, file %s, line %d", \
                              __LINE__,                                 \
                              "" __FILE__,                              \
                              "" #expr,                                 \
                              "" __FILE__,                              \
                              __LINE__);                                \
            }                                                           \
            ::std::abort();                                             \
        }                                                               \
    } while (0)

#define LOG_ASSERT_FATAL_STATIC(expr)                                   \
    do {                                                                \
        if (!(expr)) {                                                  \
            if (IS_LOG_FATAL_ENABLED()) {                               \
                _logger.fatal("Assertion failed: %s, file %s, line %d", \
                              __LINE__,                                 \
                              "" __FILE__,                              \
                              "" #expr,                                 \
                              "" __FILE__,                              \
                              __LINE__);                                \
            }                                                           \
            ::std::abort();                                             \
        }                                                               \
    } while (0)

#define LOG_ASSERT_FATAL_MSG(expr, FORMAT, ...)                                                 \
    do {                                                                                        \
        if (!(expr)) {                                                                          \
            if (IS_LOG_FATAL_ENABLED()) {                                                       \
                _logger.fatal((const void*)this,                                                \
                              "Assertion failed: %s, file %s, line %d",                         \
                              __LINE__,                                                         \
                              "" __FILE__,                                                      \
                              "" #expr,                                                         \
                              "" __FILE__,                                                      \
                              __LINE__);                                                        \
                _logger.fatal((const void*)this, FORMAT, __LINE__, "" __FILE__, ##__VA_ARGS__); \
            }                                                                                   \
            ::std::abort();                                                                     \
        }                                                                                       \
    } while (0)

#define LOG_ASSERT_FATAL_MSG_STATIC(expr, FORMAT, ...)                       \
    do {                                                                     \
        if (!(expr)) {                                                       \
            if (IS_LOG_FATAL_ENABLED()) {                                    \
                _logger.fatal("Assertion failed: %s, file %s, line %d",      \
                              __LINE__,                                      \
                              "" __FILE__,                                   \
                              "" #expr,                                      \
                              "" __FILE__,                                   \
                              __LINE__);                                     \
                _logger.fatal(FORMAT, __LINE__, "" __FILE__, ##__VA_ARGS__); \
            }                                                                \
            ::std::abort();                                                  \
        }                                                                    \
    } while (0)

#define LOG_ASSERT_FATAL_ERRNO(status)                                             \
    do {                                                                           \
        if (!(status == 0)) {                                                      \
            if (IS_LOG_FATAL_ENABLED()) {                                          \
                _logger.fatal("Assertion failed: %s, file %s, line %d, error: %s", \
                              __LINE__,                                            \
                              "" __FILE__,                                         \
                              "" #status,                                          \
                              "" __FILE__,                                         \
                              __LINE__,                                            \
                              strerror(errno));                                    \
            }                                                                      \
            ::std::abort();                                                        \
        }                                                                          \
    } while (0)

#define CHECK_ALLOCATION(expr)                                                                   \
    do {                                                                                         \
        if (!(expr)) {                                                                           \
            LOG_SYSTEM_ERROR_STATIC(                                                             \
                "null pointer detected: %s, file %s, line %d", "" #expr, "" __FILE__, __LINE__); \
            ::std::abort();                                                                      \
        }                                                                                        \
    } while (0)

#define LOG_SYSTEM_ASSERT_FATAL(expr)                                                       \
    do {                                                                                    \
        if (!(expr)) {                                                                      \
            LOG_SYSTEM_ERROR_STATIC(                                                        \
                "Assertion failed: %s, file %s, line %d", "" #expr, "" __FILE__, __LINE__); \
            ::std::abort();                                                                 \
        }                                                                                   \
    } while (0)

#ifndef NDEBUG
#define LOG_ASSERT_ERRNO(expr, FORMAT, ...) \
    LOG_ASSERT_MSG(expr, FORMAT ", errno=%d (%s)", ##__VA_ARGS__, errno, strerror(errno))

#define LOG_ASSERT_ERRNO_STATIC(expr, FORMAT, ...) \
    LOG_ASSERT_MSG_STATIC(expr, FORMAT ", errno=%d (%s)", ##__VA_ARGS__, errno, strerror(errno))

#else

#define LOG_ASSERT_ERRNO(expr, FORMAT, ...) ((void)0)

#define LOG_ASSERT_ERRNO_STATIC(expr, FORMAT, ...) ((void)0)

#endif

#define btoa(x) ((x) ? "true" : "false")
}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_LOGGER_H
