/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     ComponentDescription.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_COMPONENTDESCRIPTION_H
#define ASF_CORE_COMPONENTDESCRIPTION_H

#include "asf/core/ComponentFactory.h"
#include "asf/core/IncomingOutgoingIF.h"
#include "asf/core/Logger.h"
#include "asf/core/ProvidedPort.h"
#include "asf/core/RequiredPort.h"
#include "asf/core/Types.h"
#include "asf/threading/Mutex.h"

#include <map>
#include "Internal.h"

#include <cstdio>

namespace asf {
namespace core {

class ComponentContainer;

class ComponentDescription {
public:
    class DeleteHookBase {
    public:
        DeleteHookBase(void* instance, uint16 priority)
            : _instance(instance), _priority(priority) {}

        virtual ~DeleteHookBase() { _instance = 0; }

        bool operator<(const DeleteHookBase& b) const { return (_priority < b._priority); }

    protected:
        void* _instance;
        uint16 _priority;
    };

    template < class T >
    class DeleteHook : public DeleteHookBase {
    public:
        DeleteHook(T* instance, uint16 priority = 0) : DeleteHookBase((void*)instance, priority) {}

        virtual ~DeleteHook() { delete (T*)_instance; }
    };

    typedef std::map< std::string, ProvidedPort > ProvidedPortMap;
    typedef std::pair< std::string, ProvidedPort > ProvidedPortPair;

    typedef std::map< std::string, RequiredPort > RequiredPortMap;
    typedef std::pair< std::string, RequiredPort > RequiredPortPair;

    ComponentDescription(::asf::identifier_t id,
                         const std::string& name,
                         bool immediate,
                         bool threadBridge,
                         ::boost::shared_ptr< ComponentFactoryIF > componentFactory);

    virtual ~ComponentDescription();

    ::asf::identifier_t getId() const { return _id; }

    const std::string& getName() const { return _name; }

    void setContainer(ComponentContainer* container);

    ComponentContainer* getContainer();

    ProvidedPort& addProvidedPort(const ::asf::core::ProvidedPort& providedPort);

    RequiredPort& addRequiredPort(const ::asf::core::RequiredPort& requiredPort);

    ProvidedPort* findProvidedPort(const std::string& portName);

    RequiredPort* findRequiredPort(const std::string& portName);

    BaseComponent* getComponent() const;

    bool isComponentInstantiated() const;

    void deleteComponent();

    static ComponentDescription* getCurrentComponentDescription();

    bool isImmediate() const { return _isImmediate; }

    bool hasThreadBridge() const { return _hasThreadBridge; }

    ProvidedPortMap& getProvidedPorts() { return _providedPorts; }

    RequiredPortMap& getRequiredPorts() { return _requiredPorts; }

    void addDeleteHook(ComponentDescription::DeleteHookBase* deleteHook);

    void movePortsTo(ComponentDescription& destination);

    void onDisconnected(ConnectionIFSharedPtr connection);

    ::boost::shared_ptr< ComponentFactoryIF > getComponentFactory() const;

public:
    ProvidedPortMap _providedPorts;

    RequiredPortMap _requiredPorts;

private:
    void callDeleteHooks();

    ::asf::identifier_t _id;

    std::string _name;

    bool _isImmediate;

    bool _hasThreadBridge;

    ::boost::shared_ptr< ComponentFactoryIF > _componentFactory;

    ComponentContainer* _container;

    ::std::vector< ComponentDescription::DeleteHookBase* > _deleteHooks;

    DECLARE_CLASS_LOGGER();
};

typedef ::boost::shared_ptr< ComponentDescription > ComponentDescriptionSharedPtr;

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_COMPONENTDESCRIPTION_H
