/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     AsyncTask.h
 *\brief
 *
 *\author   CM/ESA2
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2016-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_ASYNCTASK_H
#define ASF_CORE_ASYNCTASK_H

#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include "asf/core/ComponentContainer.h"
#include "asf/core/ComponentMessage.h"
#include "asf/core/ContextData.h"
#include "asf/core/Exceptions.h"
#include "asf/core/LocalMessageToken.h"
#include "asf/core/PayloadTuple.h"
#include "asf/core/Types.h"
#include "asf/core/nullable.h"
#include "asf/threading/RunnableIF.h"

namespace asf {
namespace core {

class Async {
public:
    Async(){};
    static ::asf::core::Logger _asyncLogger;
};

#define LOG_ASYNC_LOGGER() ::asf::core::Logger& _logger = Async::_asyncLogger;

template < typename RESULT >
class AsyncResult : public PayloadTuple1< RESULT > {
public:
    AsyncResult(const RESULT& data) : PayloadTuple1< RESULT >(data) {}

    const RESULT& getData() const { return PayloadTuple1< RESULT >::get0(); }
};

template < typename ERROR_ >
class AsyncError : public PayloadTuple1< ERROR_ > {
public:
    AsyncError(const ERROR_& data) : PayloadTuple1< ERROR_ >(data) {}

    const ERROR_& getData() const { return PayloadTuple1< ERROR_ >::get0(); }
};

class AsyncCompletedCallbackIF {
public:
    virtual ~AsyncCompletedCallbackIF() {}

    virtual void onAsyncCompleted(act_t act) = 0;
};

template < typename RESULT >
class AsyncResultCallbackIF {
public:
    virtual ~AsyncResultCallbackIF() {}

    virtual void onAsyncResult(const ::boost::shared_ptr< AsyncResult< RESULT > >& result) = 0;
};

template < typename ERROR_ >
class AsyncErrorCallbackIF {
public:
    virtual ~AsyncErrorCallbackIF() {}

    virtual void onAsyncCompleted(act_t act) = 0;

    virtual void onAsyncError(const ::boost::shared_ptr< AsyncError< ERROR_ > >& error) = 0;
};

template < typename RESULT, typename ERROR_ >
class AsyncResultErrorCallbackIF {
public:
    virtual ~AsyncResultErrorCallbackIF() {}

    virtual void onAsyncResult(const ::boost::shared_ptr< AsyncResult< RESULT > >& result) = 0;

    virtual void onAsyncError(const ::boost::shared_ptr< AsyncError< ERROR_ > >& error) = 0;
};

class AsyncMessageBase : public ::asf::core::ComponentMessage {
public:
    AsyncMessageBase(act_t act) : ComponentMessage(), _act(act) {}

    virtual bool isRemote() const { return false; }

    virtual bool sendRemote() { return false; }

    act_t getAct() const { return _act; }

    void logProcessMessage(const char* name, void* callback) const {
        LOG_ASYNC_LOGGER();
        LOG_INFO("<- %s, act=%" PRIuPTR ", cb=%p", name, getAct(), callback);
    }

    template < typename P >
    void logProcessMessage(const char* name, P& payload, void* callback) const {
        LOG_ASYNC_LOGGER();
        logProcessMessage(name, callback);
        if (IS_LOG_INFO_ENABLED()) {
            std::ostringstream stream;
            stream << "payload: " << payload;
            LOG_INFO("%s", stream.str().c_str());
        }
    }

protected:
    act_t _act;
};

class AsyncCompletedMessage : public AsyncMessageBase {
public:
    AsyncCompletedMessage(AsyncCompletedCallbackIF& callback, act_t act)
        : AsyncMessageBase(act), _callback(callback) {}

    virtual ~AsyncCompletedMessage() {}

    virtual void processMessage() {
        logProcessMessage("onAsyncCompleted", &_callback);
        _callback.onAsyncCompleted(_act);
    }

    virtual ::asf::core::MessageSharedPtr clone() const {
        return ::asf::core::MessageSharedPtr(new AsyncCompletedMessage(*this));
    }

private:
    AsyncCompletedCallbackIF& _callback;
};

template < typename RESULT >
class AsyncResultMessage : public AsyncMessageBase {
public:
    AsyncResultMessage(AsyncResultCallbackIF< RESULT >& callback, act_t act)
        : AsyncMessageBase(act), _callback(callback) {}

    virtual ~AsyncResultMessage() {}

    virtual void processMessage() {
        logProcessMessage("onAsyncResult", *_payloadResult, &_callback);
        _callback.onAsyncResult(_payloadResult);
    }

    virtual ::asf::core::MessageSharedPtr clone() const {
        return ::asf::core::MessageSharedPtr(new AsyncResultMessage< RESULT >(*this));
    }

    void setResult(const RESULT& result) {
        _payloadResult.reset(new AsyncResult< RESULT >(result));
        _payloadResult->setAct(_act);
    }

private:
    AsyncResultCallbackIF< RESULT >& _callback;

    boost::shared_ptr< AsyncResult< RESULT > > _payloadResult;
};

template < typename ERROR_ >
class AsyncErrorMessage : public AsyncMessageBase {
public:
    AsyncErrorMessage(AsyncErrorCallbackIF< ERROR_ >& callback, act_t act)
        : AsyncMessageBase(act), _callback(callback) {}

    virtual ~AsyncErrorMessage() {}

    virtual void processMessage() {
        if (_payloadError) {
            logProcessMessage("onAsyncError", *_payloadError, &_callback);
            _callback.onAsyncError(_payloadError);
        } else {
            logProcessMessage("onAsyncCompleted", &_callback);
            _callback.onAsyncCompleted(_act);
        }
    }

    virtual ::asf::core::MessageSharedPtr clone() const {
        return ::asf::core::MessageSharedPtr(new AsyncErrorMessage< ERROR_ >(*this));
    }

    void setError(const ERROR_& error) {
        _payloadError.reset(new AsyncError< ERROR_ >(error));
        _payloadError->setAct(_act);
    }

private:
    AsyncErrorCallbackIF< ERROR_ >& _callback;

    boost::shared_ptr< AsyncError< ERROR_ > > _payloadError;
};

template < typename RESULT, typename ERROR_ >
class AsyncResultErrorMessage : public AsyncMessageBase {
public:
    AsyncResultErrorMessage(AsyncResultErrorCallbackIF< RESULT, ERROR_ >& callback, act_t act)
        : AsyncMessageBase(act), _callback(callback) {}

    virtual ~AsyncResultErrorMessage() {}

    virtual void processMessage() {
        if (_payloadResult) {
            logProcessMessage("onAsyncResult", *_payloadResult, &_callback);
            _callback.onAsyncResult(_payloadResult);
        } else if (_payloadError) {
            logProcessMessage("onAsyncError", *_payloadError, &_callback);
            _callback.onAsyncError(_payloadError);
        }
    }

    virtual ::asf::core::MessageSharedPtr clone() const {
        return ::asf::core::MessageSharedPtr(new AsyncResultErrorMessage< RESULT, ERROR_ >(*this));
    }

    void setResult(const RESULT& result) {
        _payloadResult.reset(new AsyncResult< RESULT >(result));
        _payloadResult->setAct(_act);
    }

    void setError(const ERROR_& error) {
        _payloadError.reset(new AsyncError< ERROR_ >(error));
        _payloadError->setAct(_act);
    }

private:
    AsyncResultErrorCallbackIF< RESULT, ERROR_ >& _callback;

    boost::shared_ptr< AsyncResult< RESULT > > _payloadResult;

    boost::shared_ptr< AsyncError< ERROR_ > > _payloadError;
};

class AsyncTaskBase {
public:
    AsyncTaskBase() : _act(0) {
        _componentDescription = ContextData::getThreadLocal()->getComponentDescription();
    }

    virtual ~AsyncTaskBase() { _componentDescription = 0; };

    virtual void interrupt(){};

    virtual void run() = 0;

    void setAct(act_t act) { _act = act; }

    act_t getAct() const { return _act; }

protected:
    ComponentDescription* _componentDescription;

    act_t _act;
};

class AsyncTask : public AsyncTaskBase {
public:
    ::boost::shared_ptr< ComponentMessage > getReturnMessage(AsyncCompletedCallbackIF& cb,
                                                             act_t act) {
        ::boost::shared_ptr< AsyncCompletedMessage > msg(new AsyncCompletedMessage(cb, act));
        msg->setComponentDescription(_componentDescription);
        return msg;
    }
};

class AsyncTaskRunner : public AsyncTask {
public:
    void setRunner(::boost::function< void(AsyncTask&) > runner) { _runner = runner; }

    void run() { _runner(*this); }

private:
    ::boost::function< void(AsyncTask&) > _runner;
};

template < typename RESULT >
class AsyncResultTask : public AsyncTaskBase {
public:
    void setResult(const RESULT& result) {
        LOG_ASYNC_LOGGER();
        LOG_ASSERT_FATAL_MSG(!hasResult(), "A result can be set only once.");
        _result = result;
    }

    const RESULT& getResult() const { return _result.get(); }

    bool hasResult() { return _result.hasValue(); }

    ::boost::shared_ptr< ComponentMessage > getReturnMessage(AsyncResultCallbackIF< RESULT >& cb,
                                                             act_t act) {
        LOG_ASYNC_LOGGER();
        LOG_ASSERT_FATAL_MSG(hasResult(), "No result was set in the run function.");
        ::boost::shared_ptr< AsyncResultMessage< RESULT > > msg(
            new AsyncResultMessage< RESULT >(cb, act));
        msg->setResult(_result.get());
        msg->setComponentDescription(_componentDescription);
        return msg;
    }

private:
    nullable< RESULT > _result;
};

template < typename RESULT >
class AsyncResultTaskRunner : public AsyncResultTask< RESULT > {
public:
    void setRunner(::boost::function< void(AsyncResultTask< RESULT >&) > runner) {
        _runner = runner;
    }

    void run() { _runner(*this); }

private:
    ::boost::function< void(AsyncResultTask< RESULT >&) > _runner;
};

template < typename RESULT, typename ERROR_ >
class AsyncResultErrorTask : public AsyncTaskBase {
public:
    void setResult(const RESULT& result) {
        LOG_ASYNC_LOGGER();
        LOG_ASSERT_FATAL_MSG(!hasResult(), "A result can be set only once.");
        _result = result;
    }

    const RESULT& getResult() const { return _result.get(); }

    bool hasResult() { return _result.hasValue(); }

    void setError(const ERROR_& error) {
        LOG_ASYNC_LOGGER();
        LOG_ASSERT_FATAL_MSG(!hasError(), "A error can be set only once.");
        _error = error;
    }

    const ERROR_& getError() const { return _error.get(); }

    bool hasError() { return _error.hasValue(); }

    ::boost::shared_ptr< ComponentMessage > getReturnMessage(
        AsyncResultErrorCallbackIF< RESULT, ERROR_ >& cb, act_t act) {
        LOG_ASYNC_LOGGER();
        ::boost::shared_ptr< AsyncResultErrorMessage< RESULT, ERROR_ > > msg;
        if (hasResult()) {
            LOG_ASSERT_FATAL_MSG(!hasError(),
                                 "It is not allowed to set error and result in run function.");
            msg = ::boost::shared_ptr< AsyncResultErrorMessage< RESULT, ERROR_ > >(
                new AsyncResultErrorMessage< RESULT, ERROR_ >(cb, act));
            msg->setResult(_result.get());
        } else if (hasError()) {
            msg = ::boost::shared_ptr< AsyncResultErrorMessage< RESULT, ERROR_ > >(
                new AsyncResultErrorMessage< RESULT, ERROR_ >(cb, act));
            msg->setError(_error.get());
        } else {
            LOG_FATAL("No result or error was set in the run function.");
        }
        msg->setComponentDescription(_componentDescription);
        return msg;
    }

private:
    nullable< RESULT > _result;

    nullable< ERROR_ > _error;
};

template < typename RESULT, typename ERROR_ >
class AsyncResultErrorTaskRunner : public AsyncResultErrorTask< RESULT, ERROR_ > {
public:
    void setRunner(::boost::function< void(AsyncResultErrorTask< RESULT, ERROR_ >&) > runner) {
        _runner = runner;
    }

    void run() { _runner(*this); }

private:
    ::boost::function< void(AsyncResultErrorTask< RESULT, ERROR_ >&) > _runner;
};

template < typename ERROR_ >
class AsyncErrorTask : public AsyncTaskBase {
public:
    void setError(const ERROR_& error) { _error = error; }

    const ERROR_& getError() const { return _error.get(); }

    bool hasError() { return _error.hasValue(); }

    ::boost::shared_ptr< ComponentMessage > getReturnMessage(AsyncErrorCallbackIF< ERROR_ >& cb,
                                                             act_t act) {
        ::boost::shared_ptr< AsyncErrorMessage< ERROR_ > > msg(
            new AsyncErrorMessage< ERROR_ >(cb, act));
        if (hasError()) {
            msg->setError(_error.get());
        }
        msg->setComponentDescription(_componentDescription);
        return msg;
    }

private:
    nullable< ERROR_ > _error;
};

template < typename ERROR_ >
class AsyncErrorTaskRunner : public AsyncErrorTask< ERROR_ > {
public:
    void setRunner(::boost::function< void(AsyncErrorTask< ERROR_ >&) > runner) {
        _runner = runner;
    }

    void run() { _runner(*this); }

private:
    ::boost::function< void(AsyncErrorTask< ERROR_ >&) > _runner;
};

class AsyncTaskRunnableFireAndForget : public ::asf::threading::RunnableIF {
public:
    AsyncTaskRunnableFireAndForget(const ::boost::shared_ptr< AsyncTask >& task)
        : _act(::asf::core::LocalMessageToken::getAct()), _task(task) {}

    virtual ~AsyncTaskRunnableFireAndForget() { ::asf::core::LocalMessageToken::returnAct(_act); }

    virtual void interrupt() { _task->interrupt(); }

    virtual tenRun enRun() {
        LOG_ASYNC_LOGGER();
        LOG_INFO("async run start, act=%" PRIuPTR, _act);
        _task->setAct(_act);
        __try {
            _task->run();
        }
        __catch(...) { LOG_FATAL("Exception occurred in async run function."); }
        LOG_INFO("async run end, act=%" PRIuPTR, _act);
        return RunnableIF::EN_SUSPEND;
    }

    act_t getAct() const { return _act; }

private:
    act_t _act;

    ::boost::shared_ptr< AsyncTask > _task;
};

template < typename CB, typename TASK >
class AsyncTaskRunnable : public ::asf::threading::RunnableIF {
public:
    AsyncTaskRunnable(CB& callback, TASK& task, ::asf::core::ComponentContainer* componentContainer)
        : _act(::asf::core::LocalMessageToken::getAct()),
          _callback(callback),
          _task(task),
          _componentContainer(componentContainer),
          _isInterrupted(false) {}

    virtual ~AsyncTaskRunnable() { ::asf::core::LocalMessageToken::returnAct(_act); }

    virtual void interrupt() {
        _isInterrupted = true;
        _task->interrupt();
    }

    virtual tenRun enRun() {
        LOG_ASYNC_LOGGER();
        LOG_INFO("async run start, act=%" PRIuPTR ", cb=%p", _act, &_callback);
        _task->setAct(_act);
        __try {
            _task->run();
        }
        __catch(...) { LOG_FATAL("Exception occurred in async run function."); }
        LOG_INFO("async run end, act=%" PRIuPTR ", cb=%p", _act, &_callback);

        if (!_isInterrupted) {
            _componentContainer->sendMessage(_task->getReturnMessage(_callback, _act));
        }
        return RunnableIF::EN_SUSPEND;
    }

    act_t getAct() { return _act; }

private:
    act_t _act;

    CB& _callback;

    TASK _task;

    ::asf::core::ComponentContainer* _componentContainer;

    bool _isInterrupted;
};

/**
 * You can use the macro ASYNC_CLASS, if you want to write the asynchronous call more compact.
 * The macro just hides the construction of the shared pointer. The macro also accepts a variable
 * count of parameter. This is needed if the constructor of the functor has incoming parameters.
 */
#define ASYNC_CLASS(tasktype, ...) (::boost::shared_ptr< tasktype >(new tasktype(__VA_ARGS__)))

/**
 * You can use the macro ASYNC_FUNCTION, if you want to write the asynchronous function call
 * more compact and simple. It hides the usage of boost bind. Also the correct usage of
 * the boost placeholder for the task type is completely hidden. The macro accepts a variable
 * count of parameters for the function specific parameters.
 */
#define ASYNC_FUNCTION(taskfunction, ...) (::boost::bind(&taskfunction, _1, __VA_ARGS__))

}  // namespace core
}  // namespace asf

#endif
