/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     ApplicationIF.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_APPLICATIONIF_H
#define ASF_CORE_APPLICATIONIF_H

#include "asf/core.h"

#include <string>
#include <vector>
#include "asf/core/ComponentMessage.h"
#include "asf/core/HookContainer.h"
#include "asf/core/Uri.h"

namespace asf {
namespace core {

class CommunicationStack;

class ProvidedPort;

class RequiredPort;

class ExportedPort;

class ImportedPort;

/**
 * This interface is used to bind the application with the ComponentMessageAdapter.
 */
class ApplicationBinderIF {
public:
    virtual ~ApplicationBinderIF() {}

    virtual void sendMessage(ComponentMessage& msg) = 0;

    virtual CommunicationStack* getProxyComStack(const std::string& scheme) = 0;

    virtual bool bindPorts(ProvidedPort& providedPort, ExportedPort& exportedPort) = 0;

    virtual bool bindPorts(RequiredPort& requiredPort, ImportedPort& importedPort) = 0;

    virtual bool bindPorts(RequiredPort& requiredPort, ProvidedPort& providedPort) = 0;

    virtual ProvidedPort* findProvidedPort(const std::string& serviceName,
                                           const std::string& portName) = 0;
};

/**
 * The ApplicationIF is the representation of an ASF application at runtime.
 *
 * An ASF application is described by an application manifest file (cma extension).
 * The ASF code generator (asfc) creates a C++ representation from such a
 * manifest. Part of the runtime representation is an instance of the
 * ApplicationIF. This instance can be retrieved via the static getApplication()
 * method.
 *
 * The ApplicationIF can be used from any context.
 */
class ApplicationIF {
public:
    virtual ~ApplicationIF() {}

    /**
     * Returns the name of the Application which is defined in the
     * application manifest (cma extension)
     */
    virtual const std::string& getName() const = 0;

    /**
     * Returns the package name of the Application which is defined in the
     * application manifest (cma extension)
     */
    virtual const std::string& getPackageName() const = 0;

    /**
     * Returns the version of the application which is defined in the
     * application manifest (cma extension)
     */
    virtual const Version& getVersion() const = 0;

    /**
     * Shuts down the application and exists the current process
     * with the specified exit code.
     *
     * This methods stops the dispatching of messages from all message queues and
     * waits for the completion of the message processing on all threads. Then the
     * application is exiting.
     *
     * A shut down can only be executed once. Once a shutdown is in progress no
     * other shut down can be initiated.
     */
    virtual void shutdown(int exitCode = 0) = 0;

    /**
     * Returns a vector containing the command line arguments of the process. This vector
     * contains only those arguments which are not being processed by ASF itself
     * (e.g. the --logconfig parameter is removed).
     */
    virtual const std::vector< std::string >& getArguments() const = 0;

    /**
     * Check whether vector with arguments (see getArguments() method) isn't empty.
     */
    virtual bool hasArguments() const = 0;

    /**
     * Returns the ApplicationIF instance representing the current application
     * which is being executed.
     */
    static ApplicationIF* getApplication();

    /**
     * Returns true if application is currently shutting down.
     */
    virtual bool isShutdownActive() const = 0;

    /**
     * Adds a function ptr to the application. This function will be called when the
     * application is shut down.
     *
     * @param VoidFunctionPtr the function which will be executed
     * @param priority a higher priority function will be called later on shutting down
     */
    virtual void addShutdownHook(::asf::core::HookContainer::VoidFunctionPtr function,
                                 int priority = 0) = 0;
};

}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_APPLICATIONIF_H
