/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CmsStubRequests.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CMS_CMSSTUBREQUESTS_H
#define ASF_CMS_CMSSTUBREQUESTS_H

#include <map>
#include "asf/cms/CmsMessage.h"
#include "asf/cms/CmsSystemErrorIF.h"
#include "asf/threading/Mutex.h"

namespace asf {
namespace cms {

/**
 * Stores all open REQUEST's, GET's and SET's
 */
class CmsStubRequests {
public:
    CmsStubRequests(CmsSystemErrorIF& systemError);

    ~CmsStubRequests();

    /*
     * Adds a REQUEST, GET or SET message to the message map. The function returns a unique
     * act for each registration. The function returns 0, if the registration fails.
     */
    act_t add(CmsMessage& msg);

    act_t findPendingRequest(const CmsMessage& msg);

    /**
     * The function tries to find a corresponding message to the passed values act, serviceId
     * and functionId. Afterwards the function removes the entry and returns a TRUE.
     * If no act (act=0) is assigned the function tries to read them from the thread local storage.
     * If it was not possible to reconstruct the messageId the function tries to find a unique
     * entry which has the same serviceId and functionId like the passed one. If the search result
     * is ambiguous the system can't evaluate the registration entry without the act.
     * The function returns FALSE if no corresponding message was found. It also returns FALSE if
     * the service is not registered.
     */
    bool getAndRemove(act_t act,
                      const std::string& serviceId,
                      const std::string& functionId,
                      CmsMessage& retMsg) {
        return getAndRemove(act, serviceId, functionId, retMsg, true);
    }

    bool get(act_t act,
             const std::string& serviceId,
             const std::string& functionId,
             CmsMessage& retMsg) {
        return getAndRemove(act, serviceId, functionId, retMsg, false);
    }

    /**
     * Deregisters all entries which have the same serviceId and clientId like the passed
     * message.
     */
    void deregisterService(CmsMessage& msg);

    /**
     * Deregisters all entries which have a relation to the passed connection.
     */
    void deregisterService(::asf::core::ConnectionIFSharedPtr connection);

private:
    bool getAndRemove(act_t act,
                      const std::string& serviceId,
                      const std::string& functionId,
                      CmsMessage& msg,
                      bool removeIt);

    void cleanUpAllActiveRequests();

    ::asf::threading::Mutex _msgMutex;

    typedef std::map< act_t, CmsMessage > MessageMap;

    typedef std::pair< act_t, CmsMessage > MessagePair;

    MessageMap _messages;

    CmsSystemErrorIF& _systemError;

    DECLARE_CLASS_LOGGER();
};
}  // namespace cms
}  // namespace asf

#endif  // ASF_CMS_CMSSTUBREQUESTS_H
