/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CmsProxyDelegate.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CMS_CMSPROXYDELEGATE_H
#define ASF_CMS_CMSPROXYDELEGATE_H

#include <map>
#include "asf/cms/CmsImportedPort.h"
#include "asf/cms/CmsMessage.h"
#include "asf/core/ComponentDescription.h"
#include "asf/core/ProxyDelegateBase.h"
#include "asf/threading/Mutex.h"
#include "com/bosch/cm/asf/lang/cms/ConnectorsConst.h"

namespace asf {
namespace cms {

class CmsProxyDelegate;

class CmsProxyCallbackInfo {
public:
    CmsProxyCallbackInfo(void* obj,
                         ::asf::cms::ProxyCallback callback,
                         std::string functionId,
                         CmsTypes::CmsMessageHeader::CmsMessageType messageType,
                         uint32 regId = 0)
        : _obj(obj),
          _callback(callback),
          _functionId(functionId),
          _messageType(messageType),
          _regId(regId) {}

    void* getObject() { return _obj; }

    ::asf::cms::ProxyCallback getCallback() const { return _callback; }

    const std::string& getFunctionId() const { return _functionId; }

    CmsTypes::CmsMessageHeader::CmsMessageType getCmsMessageType() const { return _messageType; }

    void call(CmsMessage& msg) {
        if (msg.getPayload().get())
            // act and cct have the same value on proxy side
            msg.getPayload().get()->setAct(msg.getCct());
        CHECK_ALLOCATION(_callback);
        _callback(_obj, &msg);
    }

    void setRegId(uint32 regID) { _regId = regID; }

    uint32 getRegId() const { return _regId; }

private:
    CmsProxyCallbackInfo()
        : _obj(0),
          _callback(0),
          _messageType(CmsTypes::CmsMessageHeader::CmsMessageType__Request),
          _regId(0) {}

    void* _obj;

    ProxyCallback _callback;

    std::string _functionId;

    CmsTypes::CmsMessageHeader::CmsMessageType _messageType;

    uint32 _regId;
};

class CommunicationStack;

class CmsProxyDelegate : public ::asf::core::ProxyDelegateBase {
public:
    typedef ::boost::shared_ptr< CmsProxyCallbackInfo > CmsProxyCallbackInfoPtr;

    typedef std::map< act_t, CmsProxyCallbackInfoPtr > CmsProxyCallbackMap;

    typedef std::pair< act_t, CmsProxyCallbackInfoPtr > CmsProxyCallbackPair;

    CmsProxyDelegate(const std::string& serviceId,
                     ::asf::core::Proxy* userProxy,
                     ::asf::core::Logger& logger);

    ~CmsProxyDelegate();

    void setServiceId(const std::string& serviceId) { _serviceId = serviceId; }

    const std::string& getServiceId() const { return _serviceId; }

    bool request(CmsMessage& msg);

    bool abort(CmsMessage& msg);

    bool register_(CmsMessage& msg);

    bool deregister(CmsMessage& msg);

    bool get(CmsMessage& msg);

    bool set(CmsMessage& msg);

    bool sendMsg(CmsMessage& msg);

    void sendErrorLoopBack(CmsMessage& msg);

    act_t registerCallback(void* obj, ProxyCallback cb, CmsMessage& msg);

    void deregisterCallback(act_t registerId);

    void deregisterAll(const std::string& functionId);

    void processCmsMessage(CmsMessage& msg);

    virtual void process(::asf::core::ServiceMessage& msg) {
        processCmsMessage(static_cast< CmsMessage& >(msg));
    }

    CmsServiceStreamer* getStreamer() const;

    bool hasRegistration(act_t registerId, const std::string& functionId);

    bool hasPendingRequest(act_t requestAct, const std::string& functionId);

    static uint32 createUniqueProxyId();

    uint32 getProxyId() const { return _proxyId; }

    void setServerComponentDescription(
        ::asf::core::ComponentDescription* localServerComponentDescription) {
        _localServerComponentDescription = localServerComponentDescription;
    }

    ::asf::core::ComponentDescription* getServerComponentDescription() {
        return _localServerComponentDescription;
    }

    bool connectToProvidedPort(::asf::core::ProvidedPort* providerPort);

    ::asf::core::ProvidedPort* getLocalProviderPort() { return _providedPort; }

    bool connectToImportedPort(::asf::cms::CmsImportedPort* importedPort);

    ::asf::cms::CmsImportedPort* getImportedPort() { return _importedPort; }

    uint32 connectProxyInfo(CmsMessage& msg) const;

    void onConnected(::asf::core::ConnectionIFSharedPtr inConnection, bool inConnected);

    void connect();

    ::com::bosch::cm::asf::lang::cms::Connectors::Encoding getEncoding();

private:
    void cancelPendingSequences(const std::string logName,
                                ::asf::threading::Mutex& mutex,
                                CmsProxyCallbackMap& map);

    void sendServiceRegister();

    void sendServiceDeregister();

    void sendSystemErrorToProxy(CmsMessage& msg, CmsTypes::SystemError error);

    bool registerRegisterCallback(act_t cct, const CmsProxyCallbackInfoPtr& ci);

    bool deregisterRegisterCallback(act_t cct, CmsProxyCallbackInfoPtr& ci);

    bool getRegisterCallback(act_t cct, CmsProxyCallbackInfoPtr& ci);

    bool registerRequestGetCallback(act_t cct, const CmsProxyCallbackInfoPtr& ci);

    bool deregisterRequestGetCallback(act_t cct, CmsProxyCallbackInfoPtr& ci);

    bool getRequestGetCallback(act_t cct, CmsProxyCallbackInfoPtr& ci);

    void callRegisterCallback(CmsMessage& cmsMsg);

    void attachProxyInfos(CmsMessage& msg);

    uint32 _proxyId;

    ::asf::core::ComponentDescription* _localServerComponentDescription;

    ::asf::core::ProvidedPort* _providedPort;

    ::asf::cms::CmsImportedPort* _importedPort;

    std::string _serviceId;

    ::asf::threading::Mutex _registerCallbackMutex;

    CmsProxyCallbackMap _registerCallbacks;

    ::asf::threading::Mutex _requestGetCallbackMutex;

    CmsProxyCallbackMap _requestGetCallbacks;

    ::asf::core::Logger& _logger;

    ::com::bosch::cm::asf::lang::cms::Connectors::Encoding _encoding;

    static ::asf::threading::AtomicCounter< uint32 > _uniqueProxyId;

    static ::asf::threading::AtomicCounter< uint32 > _uniqueRegisterId;

    friend class CmsProxyRegistry;
};

class CmsProxy : public ::asf::core::Proxy {
public:
    CmsProxy(const std::string& portName,
             ::asf::core::ServiceAvailableIF& serviceAvailable,
             const std::string& serviceId,
             ::asf::core::Logger& logger)
        : ::asf::core::Proxy(portName, serviceAvailable),
          _cmsProxyDelegate(serviceId, this, logger) {}

    virtual ~CmsProxy() {}

protected:
    virtual CmsServiceStreamer* getStreamer() = 0;

    CmsProxyDelegate _cmsProxyDelegate;

    friend class CmsProxyDelegate;

    friend class CmsStubDelegate;

    friend class CmsProxyRegistry;

    friend class CmsMessage;
};

}  // namespace cms
}  // namespace asf

#endif  // ASF_CMS_CMSPROXYDELEGATE_H
