/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CcaProxyDelegate.h
 *\brief
 *
 *\author   CM-AI/PJ-CF11.1
 *          christoph.kulla@de.bosch.com
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_CCA_CCAPROXYDELEGATE_H
#define ASF_CCA_CCAPROXYDELEGATE_H

#include "asf/cca/CcaImportedPort.h"
#include "asf/cca/CcaMostStream.h"
#include "asf/cca/CcaProxy.h"
#include "asf/cca/CcaTypes.h"
#include "asf/core/ProxyDelegateBase.h"
#include "asf/core/TokenPool.h"

#include "asf/threading/AtomicCounter.h"
#include "asf/threading/Guard.h"
#include "asf/threading/Mutex.h"

#include "asf/core/ComponentMessageAdapter.h"

#include <boost/enable_shared_from_this.hpp>
#include <list>
#include <map>

class CcaProxyTest;
class CcaProxyTestOnAvailable;
class CcaMostProxyTest;
class CcaMessageFactoryTest;

namespace asf {
namespace core {

class CommunicationStack;
}
}  // namespace asf

namespace asf {
namespace stream {

class MemoryStream;
}
}  // namespace asf

namespace asf {
namespace cca {

class CcaMessage;
class CcaProxyRegistry;
class CcaProxyDelegate;

class CcaProxyCallback {
public:
    CcaProxyCallback(uint16 functionId,
                     CcaTypes::ServiceData::OpCode opCode,
                     act_t act,
                     void* callback)
        : _functionId(functionId),
          _opCode(opCode),
          _act(act),
          _callback(callback),
          _secondaryCallback(0) {}

    virtual ~CcaProxyCallback() {}

    /**
     * Process a message received for this callback. If it returns
     * true the callback will be removed from the proxy.
     */
    virtual bool processMessage(CcaMessage& /*message*/) { return true; }

    /**
     * Called whenever the service get unavailable is this callback is
     * still registered
     */
    virtual bool onServiceUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy) = 0;

    /**
     * Store a second callback (used to store the abort callback for methods)
     */
    virtual void setSecondaryCallback(void* secondaryCallback) {
        _secondaryCallback = secondaryCallback;
    }

    virtual void* getSecondaryCallback() { return _secondaryCallback; }

    /**
     * Stores a payload object in the callback for later use
     */
    void setMessage(::boost::shared_ptr< CcaMessage > message) { _message = message; }

    const ::boost::shared_ptr< CcaMessage > getMessage() const { return _message; }

    uint16 getFunctionId() const { return _functionId; }

    CcaTypes::ServiceData::OpCode getOpCode() const { return _opCode; }

    act_t getAct() const { return _act; }

protected:
    void setActAndLogResult(CcaMessage& message,
                            const char* name,
                            act_t act,
                            ::asf::core::Logger& _logger);

    void setActAndLogResultFirst(CcaMessage& message,
                                 const char* name,
                                 act_t act,
                                 ::asf::core::Logger& _logger);

    void setActAndLogResultMiddle(CcaMessage& message,
                                  const char* name,
                                  act_t act,
                                  ::asf::core::Logger& _logger);

    void setActAndLogResultLast(CcaMessage& message,
                                const char* name,
                                act_t act,
                                ::asf::core::Logger& _logger);

    void setActAndLogAbortResult(CcaMessage& message,
                                 const char* name,
                                 act_t act,
                                 ::asf::core::Logger& _logger);

    void setActAndLogStatus(CcaMessage& message,
                            const char* name,
                            act_t act,
                            ::asf::core::Logger& _logger);

    void setActAndLogError(CcaMessage& message,
                           const char* name,
                           act_t act,
                           ::asf::core::Logger& _logger);

    uint16 _functionId;

    CcaTypes::ServiceData::OpCode _opCode;

    act_t _act;

    void* _callback;

    void* _secondaryCallback;  // used to store the abort callback

    ::boost::shared_ptr< CcaMessage > _message;  // optional message used for RelUpReg messages

    friend class CcaProxyDelegate;
};

/**
 * The CcaProxyDelegate implements cca specific proxy helper methods.
 */
class CcaProxyDelegate : public ::asf::core::ProxyDelegateBase,
                         public boost::enable_shared_from_this< CcaProxyDelegate > {
public:
    CcaProxyDelegate(uint16 appId,
                     uint16 serviceId,
                     uint16 majorVersion,
                     uint16 minorVersion,
                     ::asf::core::ServiceAvailableIF& serviceAvailable,
                     ::asf::core::Logger& logger,
                     ::asf::cca::CcaProxy* userProxy,
                     uint32 flags = CcaProxy::SPM_SERVICE_SUPPLIER,
                     bool isMost = false);

    virtual ~CcaProxyDelegate();

    void connectToImportedPort(CcaImportedPort* ccaImportedPort);

    static void setSourceSubIdSeed(uint16 seed);

public:
    CcaProxy* getCcaProxy() const { return static_cast< CcaProxy* >(getProxy()); }

    virtual ::asf::core::ServiceState getServiceState() const {
        return getProxy()->getServiceState();
    }

    /**
     * Called from the CcaProxyRegistry when the a disconnection of the connection
     * was received with the help of a CcaConnectionMessage.
     */
    // FIXME: the connection is obsolete here, shall be removed
    virtual void onDisconnected(::asf::core::ConnectionIFSharedPtr);

    void addCallback(uint16 commandCounter, ::boost::shared_ptr< CcaProxyCallback > callback);

    void addLoopBackCallback(::asf::msgId_t messageId,
                             ::boost::shared_ptr< CcaProxyCallback > callback);

    void setupCcaMessageHeader(CcaTypes::CcaMessageHeader& header) const;

    CcaTypes::CcaMessageHeader* createCcaServiceDataHeader(CcaTypes::ServiceData_OpCode opCode,
                                                           uint16 functionId);

    CcaTypes::CcaMessageHeader* createCcaServiceDataHeader(CcaTypes::ServiceData_OpCode opCode,
                                                           uint16 functionId,
                                                           uint16 commandCounter) const;

    CcaTypes::CcaMessageHeader* createCcaServiceDataReplyHeader(CcaTypes::ServiceData_OpCode opCode,
                                                                uint16 functionId);

    void setupCcaServiceDataMessage(CcaTypes::CcaMessageHeader& header,
                                    CcaTypes::ServiceData_OpCode opCode,
                                    uint16 functionId);

    void setupCcaServiceDataMessage(CcaTypes::CcaMessageHeader& header,
                                    CcaTypes::ServiceData_OpCode opCode,
                                    uint16 functionId,
                                    uint16 commandCounter) const;

    void setupCcaServiceDataReplyMessage(CcaTypes::CcaMessageHeader& header,
                                         CcaTypes::ServiceData_OpCode opCode,
                                         uint16 functionId);

    void setupCcaServiceDataReplyMessage(CcaTypes::CcaMessageHeader& header,
                                         CcaTypes::ServiceData_OpCode opCode,
                                         uint16 functionId,
                                         uint16 commandCounter) const;

    bool isCallbackPending(act_t act,
                           uint16 functionId,
                           CcaTypes::ServiceData::OpCode opCode) const;

    CcaProxyCallback* getPendingCallback(act_t act,
                                         uint16 functionId,
                                         CcaTypes::ServiceData::OpCode opCode) const;

    uint16 getPendingCommandCounter(act_t act,
                                    uint16 functionId,
                                    CcaTypes::ServiceData::OpCode opCode) const;

    void sendRelUpRegForFunctionId(uint16 functionId);

    bool sendRelUpReg(act_t act, uint16 functionId, const std::string& functionName);

    void sendRelUpReg(uint16 commandCounter, ::boost::shared_ptr< CcaMessage > message);

    void sendLoopBack(CcaMessage& message);

    bool isMost() const { return _isMost; }

    bool upRegAlreadySent(uint16 functionId, const ::asf::core::Payload& payload);

    uint16 getUpRegCommandCounter(uint16 functionId, const ::asf::core::Payload& payload);

    ::boost::shared_ptr< CcaProxyCallback > createCallback(uint16 fid,
                                                           CcaTypes::ServiceData::OpCode opCode,
                                                           uint16 commandCounter,
                                                           void* cb) const;

    ::boost::shared_ptr< CcaProxyCallback > createCallback(CcaMessage& message,
                                                           CcaTypes::ServiceData::OpCode opCode,
                                                           void* cb) const;

    // property helper functions

    act_t sendUpRegMessage(CcaMessage& message,
                           const char* name,
                           void* cb,
                           ::boost::shared_ptr< CcaMessage > relUpRegMessage);

    act_t sendGetMessage(CcaMessage& message, const char* name, void* cb);

    act_t sendSetMessage(CcaMessage& message, const char* name, void* cb);

    act_t sendLoopBackUpReg(CcaMessage& message, const char* name, void* cb);

    act_t sendLoopBackGet(CcaMessage& message, const char* name, void* cb);

    act_t sendLoopBackSet(CcaMessage& message, const char* name, void* cb);

    void sendPureSetMessage(CcaMessage& message, const char* name);

    void sendIncrementMessage(CcaMessage& message, const char* name);

    void sendDecrementMessage(CcaMessage& message, const char* name);

    ::boost::shared_ptr< CcaProxyCallback > addAdditionalStatusCallback(
        CcaMessage& message,
        ::boost::shared_ptr< ::asf::core::Payload > payload,
        const char* name,
        ::boost::shared_ptr< CcaMessage > relUpRegMessage,
        void* cb);

    void sendStatusLoopBack(CcaMessage& message, ::boost::shared_ptr< CcaProxyCallback > callback);

    // method helper functions

    act_t sendMethodStartMessage(CcaMessage& message, const char* name, void* cb);

    act_t sendMethodStartMessage(CcaMessage& message, const char* name);

    act_t sendLoopBackMethodStart(CcaMessage& message, const char* name, void* cb);

    act_t oneWayMethodStartError(const char* name);

    bool sendMethodAbortMessage(CcaMessage& message, const char* name, void* cb, act_t act);

    bool errorHandlingOfMethodAbort(uint16 functionId, const char* name, act_t act);

    act_t callback2act(::boost::shared_ptr< CcaProxyCallback > callback) const;

    const ::asf::threading::Mutex& getLock() const { return _lock; }

    CcaTypes::ProxyState _state;

    uint16 _sourceAppId;

    uint16 _sourceSubId;  // auto generated

    uint16 _destinationAppId;

    uint16 _serviceId;

    uint16 _registerId;  // send by the service confirmation

    uint16 _majorVersion;  // major version of the service, needed for service registration

    uint16 _minorVersion;  // minor version of the service, needed for service registration

    uint16
        _haveMajorVersion;  // received during service registration, used in payload serialization

    uint16 _haveMinorVersion;  // received during service registration

    uint16 _havePatchVersion;  // received during service registration

    uint32 _flags;

    bool _isMost;

    /**
     * Value represents how many tokens were newly created. If peak > gapSize this may mean
     * that (peak - gapSize) simultaneous requests are pending in the system
     */
    const ::asf::core::TokenPool< uint16 >& test_getCommandCounterTokenPool() const {
        return _commandCounterTokenPool;
    };

private:
    virtual void process(::asf::core::ServiceMessage& msg);

    /**
     * Sets the state of the proxy and triggers any actions which are bound to the state
     * transition (like calling the service available hook)
     */
    void setState(CcaTypes::ProxyState state);

    void sendSrvSupplierRegister();

    void sendSrvSupplierUnregister();

    void sendServiceRegister();

    void sendServiceUnregister();

    void setupCcaSpmMessageHeader(CcaTypes::CcaMessageHeader& header) const;

    void sendPseudoServiceRegistration();

    bool isErrorMessage(const CcaMessage& message) const;

    bool isSrvSupplierStatusAvailable(const CcaTypes::SrvSupplierStatus& status) const;

    bool processServiceDataMessage(CcaMessage& message);

    /**
     * Returns whether callbacks are left, used for testing only
     */
    bool hasCallbacks();

    ::asf::core::TokenPool< uint16 > _commandCounterTokenPool;

    ::asf::core::ServiceAvailableIF& _serviceAvailable;

    ::asf::threading::Mutex _lock;  // lock to protect concurrent access to the callback maps

    typedef std::map< uint16, ::boost::shared_ptr< CcaProxyCallback > > CommandCounterToCallbackMap;
    typedef std::multimap< uint16, ::boost::shared_ptr< CcaProxyCallback > >
        CommandCounterToCallbackMultiMap;

    // The _commandCounterToCallbackMultiMap maps cca command counters to ASF callbacks. Whenever a
    // CCA message
    // is received the proxy looks up the callback in this map with the received command counter and
    // calls the
    // processMessage method on the callback.
    CommandCounterToCallbackMap _commandCounterToCallbackMap;

    // This is a multi map to support storing multiple callbacks for a single command counter.
    // Usually only one
    // callback is stored per command counter. In case of properties ASF supports the registration
    // of multiple
    // callbacks for status updates on the same properties. This means multiple callbacks must be
    // processed when
    // a message is received. Therefore this is a multi map.
    CommandCounterToCallbackMultiMap _commandCounterToCallbackMultiMap;

    // This map is used to store callbacks for loop back status messages. Whenever a client
    // registers for
    // a property for which the proxy is already registred, a loop back status message is sent from
    // the
    // proxy to itself. This message is then processed with the help of the
    // _messageIdToCallbackLoopBackMap.
    // When the message is processed the corresponding entry will be removed from this map.
    typedef std::map< ::asf::msgId_t, ::boost::shared_ptr< CcaProxyCallback > >
        MessageIdToCallbackMap;
    MessageIdToCallbackMap _messageIdToCallbackLoopBackMap;

    ::asf::core::ConnectionIFSharedPtr _spmConnection;

    ::asf::core::Logger& _logger;

    friend class CcaMessage;

    friend class CcaProxyRegistry;

    friend class ::CcaProxyTest;

    friend class ::CcaProxyTestOnAvailable;

    friend class ::CcaMostProxyTest;

    friend class ::CcaMessageFactoryTest;
};

::boost::shared_ptr< ::asf::cca::CcaProxyDelegate > to_shared_ptr(
    ::boost::weak_ptr< ::asf::cca::CcaProxyDelegate > wp);

}  // namespace cca
}  // namespace asf

#endif
