/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CcaPayloadDecorator.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.kulla@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_CCA_CCAPAYLOADDECORATOR_H
#define ASF_CCA_CCAPAYLOADDECORATOR_H

#include "asf/core/Payload.h"
#include "asf/stream/MemoryStream.h"
#include "asf/stream/json.h"

namespace asf {
namespace cca {

#define CCA_PAYLOAD_DECORATOR__SERIALIZE_FUNCTION(a)      \
    (::asf::cca::CcaPayloadDecorator::SerializeFunction)( \
        (void (*)(const a&, ::asf::stream::MemoryStream&, unsigned int))::serializeCca)

#define CCA_PAYLOAD_DECORATOR__SERIALIZE_MOST_FUNCTION(a) \
    (::asf::cca::CcaPayloadDecorator::SerializeFunction)( \
        (void (*)(const a&, ::asf::stream::MemoryStream&, unsigned int))::serializeCcaMost)

#define CCA_PAYLOAD_DECORATOR__DESERIALIZE_FUNCTION(a)      \
    (::asf::cca::CcaPayloadDecorator::DeserializeFunction)( \
        (void (*)(::asf::stream::MemoryStream&, a&, unsigned int))::deserializeCca)

#define CCA_PAYLOAD_DECORATOR__DESERIALIZE_MOST_FUNCTION(a) \
    (::asf::cca::CcaPayloadDecorator::DeserializeFunction)( \
        (void (*)(::asf::stream::MemoryStream&, a&, unsigned int))::deserializeCcaMost)

#define CCA_PAYLOAD_DECORATOR__JSON_FUNCTION(a)                \
    (::asf::cca::CcaPayloadDecorator::JsonFunction)((void (*)( \
        const a&, ::std::string&, ::asf::stream::json::JsonSerializationOptions))::serializeJson)

#define CCA_PAYLOAD_DECORATOR__EQUALS_FUNCTION(a) \
    ((::asf::cca::CcaPayloadDecorator::EqualsFunction)&a::operator==)

// leaves the deserialization function empty in order to not include these in
// code when not needed. Otherwise would for example a client include all
// serialization and deserialization functions for all payload classes
#define INIT_CCA_PAYLOAD_DECORATOR_FOR_SERIALIZATION(var, type, instance) \
    var.init(instance,                                                    \
             CCA_PAYLOAD_DECORATOR__SERIALIZE_FUNCTION(type),             \
             0,                                                           \
             CCA_PAYLOAD_DECORATOR__JSON_FUNCTION(type),                  \
             CCA_PAYLOAD_DECORATOR__EQUALS_FUNCTION(type));

#define INIT_CCA_PAYLOAD_DECORATOR_FOR_MOST_SERIALIZATION(var, type, instance) \
    var.init(instance,                                                         \
             CCA_PAYLOAD_DECORATOR__SERIALIZE_MOST_FUNCTION(type),             \
             0,                                                                \
             CCA_PAYLOAD_DECORATOR__JSON_FUNCTION(type),                       \
             CCA_PAYLOAD_DECORATOR__EQUALS_FUNCTION(type));

// leaves the serialization function empty in order to not include these in
// code when not needed. Otherwise would for example a client include all
// serialization and deserialization functions for all payload classes
#define INIT_CCA_PAYLOAD_DECORATOR_FOR_DESERIALIZATION(var, type) \
    var.init(::boost::shared_ptr< type >(new type()),             \
             0,                                                   \
             CCA_PAYLOAD_DECORATOR__DESERIALIZE_FUNCTION(type),   \
             CCA_PAYLOAD_DECORATOR__JSON_FUNCTION(type),          \
             CCA_PAYLOAD_DECORATOR__EQUALS_FUNCTION(type));

#define INIT_CCA_PAYLOAD_DECORATOR_FOR_MOST_DESERIALIZATION(var, type) \
    var.init(::boost::shared_ptr< type >(new type()),                  \
             0,                                                        \
             CCA_PAYLOAD_DECORATOR__DESERIALIZE_MOST_FUNCTION(type),   \
             CCA_PAYLOAD_DECORATOR__JSON_FUNCTION(type),               \
             CCA_PAYLOAD_DECORATOR__EQUALS_FUNCTION(type));

// leaves the serialization and deserialization functions empty in order to not
// include these in the code when not needed. Otherwise would for example a client
// include all serialization and deserialization functions for all payload classes
#define INIT_CCA_PAYLOAD_DECORATOR_FOR_LOOPBACK(var, type, instance) \
    var.initLoopback(instance,                                       \
                     0,                                              \
                     0,                                              \
                     CCA_PAYLOAD_DECORATOR__JSON_FUNCTION(type),     \
                     CCA_PAYLOAD_DECORATOR__EQUALS_FUNCTION(type));

#define INIT_CCA_PAYLOAD_DECORATOR_FOR_MOST_LOOPBACK(var, type, instance) \
    INIT_CCA_PAYLOAD_DECORATOR_FOR_LOOPBACK(var, type, instance)

/**
 * The CcaPayloadDecorator provides functionalities like serialization for a
 * given payload object. These functionalities are realized with the help of
 * function pointers which have to be initialized with the help of the
 * init() method.
 *
 * This class was introduced to add some flexibility to the payload data classes
 * without adding virtual methods to the payload class (in order to not compromise
 * the public API of the data classes) and without requiring any template
 * function which lead to a code bloat.
 *
 * The CcaProxy calls the init() method as soon as it knows the type of the
 * payload during the message dispatching or sending.
 */
class CcaPayloadDecorator {
public:
    typedef void (*SerializeFunction)(const ::asf::core::Payload&,
                                      ::asf::stream::MemoryStream&,
                                      unsigned int majorVersion);

    typedef void (*DeserializeFunction)(::asf::stream::MemoryStream& s,
                                        ::asf::core::Payload&,
                                        unsigned int majorVersion);

    typedef void (*JsonFunction)(const ::asf::core::Payload&,
                                 std::string&,
                                 ::asf::stream::json::JsonSerializationOptions);

    typedef bool (::asf::core::Payload::*EqualsFunction)(const ::asf::core::Payload&) const;

    CcaPayloadDecorator();

    void init(::asf::core::PayloadSharedPtr payload,
              SerializeFunction serializeFunction,
              DeserializeFunction deserializeFunction,
              JsonFunction jsonFunction,
              EqualsFunction equalsFunction);

    void initLoopback(::asf::core::PayloadSharedPtr payload,
                      SerializeFunction serializeFunction,
                      DeserializeFunction deserializeFunction,
                      JsonFunction jsonFunction,
                      EqualsFunction equalsFunction);

    ::boost::shared_ptr< ::asf::core::Payload > getPayload() const;

    void toJson(std::string& s) const;

    void serialize(::asf::stream::MemoryStream& s, unsigned int majorVersion) const;

    void deserialize(::asf::stream::MemoryStream& s, unsigned int majorVersion);

    bool isEqual(const ::asf::core::Payload& payload) const;

private:
    ::asf::core::PayloadSharedPtr _payload;

    SerializeFunction _serializeFunction;

    DeserializeFunction _deserializeFunction;

    JsonFunction _jsonFunction;

    EqualsFunction _equalsFunction;

    bool _isDeserialized;

    bool _isInitializedByLoopback;

    size_t _initCounter;

    DECLARE_CLASS_LOGGER();
};
}  // namespace cca
}  // namespace asf

#endif
