/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     CcaMostStream.h
 *\brief
 *
 *\author   CM-AI/PJ-CF11.1
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2013-2013 Robert Bosch Car Multimedia GmbH
 *
 *\par History:
 * See history of revision control system
 ***************************************************************************/

#ifndef ASF_STREAM_CCAMOSTSTREAM_H
#define ASF_STREAM_CCAMOSTSTREAM_H

#include <string>
#include <vector>
#include "asf/core/Logger.h"
#include "asf/core/Types.h"
#include "asf/stream/MemoryStream.h"

#define CCAMOST_STREAM_CHUNKSIZE 2048  // 16

namespace asf {
namespace cca {

// TODO not all of them documented
class CcaMostStream {
public:
    enum MostStringCode {
        E_MOSTSTRINGCODE_UTF16 = 0,
        E_MOSTSTRINGCODE_ISO8859_15 = 1,
        E_MOSTSTRINGCODE_UTF8 = 2,
        E_MOSTSTRINGCODE_RDS = 3,
        E_MOSTSTRINGCODE_DAB_CHARSET_0001 = 4,
        E_MOSTSTRINGCODE_DAB_CHARSET_0010 = 5,
        E_MOSTSTRINGCODE_DAB_CHARSET_0011 = 6,
        E_MOSTSTRINGCODE_SHIFT_JIS = 7,
        E_MOSTSTRINGCODE_GB_18030 = 0xC0
    };

    CcaMostStream(size_t chunkSize = CCAMOST_STREAM_CHUNKSIZE);

    CcaMostStream(uint8* pu8ExtStream, size_t bufferSize);

    virtual ~CcaMostStream();

    virtual uint8* getStream() const;

    virtual size_t getBufferSize() const { return _bufferSize; }

    size_t getReadIndex() const { return _readIndex; }

    size_t getWriteIndex() const { return _usedSize; }

    uint8* getCurrentReadPos() const { return _stream + _readIndex; }

    void rewindReadIndex() { _readIndex = 0; }

    uint8* getCurrentWritePos() { return _stream + _usedSize; }

    uint8* getEndPos() { return _stream + _usedSize; }

    void writeToPosUint8(uint8* pos, uint8& value);

    void writeToIndexPosUint8(size_t indexPos, uint8& value);

    void writeToIndexPosUint16(size_t indexPos, uint16& value);

    void writeToIndexPosUint32(size_t indexPos, uint32& value);

    size_t getSize() const;

    CcaMostStream& operator<<(const uint8& u8Value);

    CcaMostStream& operator<<(const uint16& u16Value);

    CcaMostStream& operator<<(const uint32& u32Value);

    CcaMostStream& operator<<(const uint64& u64Value);

    CcaMostStream& operator<<(const double& dValue);

    CcaMostStream& operator<<(const float& dValue);

    CcaMostStream& operator<<(const int8& s8Value) { return operator<<((const uint8&)s8Value); }

    CcaMostStream& operator<<(const int16& s16Value) { return operator<<((const uint16&)s16Value); }

    CcaMostStream& operator<<(const int32& s32Value) { return operator<<((const uint32&)s32Value); }

    CcaMostStream& operator<<(const int64& s64Value) { return operator<<((const uint64&)s64Value); }

    CcaMostStream& operator<<(const bool& bValue) { return operator<<((const uint8&)bValue); }

    CcaMostStream& operator>>(uint8& u8Value);

    CcaMostStream& operator>>(uint16& u16Value);

    CcaMostStream& operator>>(uint32& u32Value);

    CcaMostStream& operator>>(uint64& u64Value);

    CcaMostStream& operator>>(double& dValue);

    CcaMostStream& operator>>(float& dValue);

    CcaMostStream& operator>>(int8& s8Value) { return operator>>((uint8&)s8Value); }

    CcaMostStream& operator>>(int16& s16Value) { return operator>>((uint16&)s16Value); }

    CcaMostStream& operator>>(int32& s32Value) { return operator>>((uint32&)s32Value); }

    CcaMostStream& operator>>(int64& s64Value) { return operator>>((uint64&)s64Value); }

    CcaMostStream& operator>>(bool& bValue) { return operator>>((uint8&)bValue); }

    // MOST string (T_String)
    void writeMostString(const std::string& str);

    void writeMostString(const char*);

    void readMostString(std::string& str);

    // ASCII string 0 terminated (tString)
    void writeString(const std::string& str);

    void writeString(const char*);

    void readString(std::string& str);

    void writeBuffer(const void* buffer, size_t length);

    void readBuffer(void* buffer, size_t length);

    CcaMostStream& operator<<(const std::string& str);

    CcaMostStream& operator>>(std::string& str);

    void deleteStream();

    void pushSelector(size_t selector);

    size_t popSelector();

    // fetch data from stream without changing the read pointer
    uint8 fetchUint8() const;

    uint16 fetchUint16() const;

    uint32 fetchUint32() const;

    uint64 fetchUint64() const;

    /*
     * This method gives the ownership of the most stream
     */
    void take(::asf::stream::MemoryStream& memoryStream);

    /*
     * This method release the ownership of the most stream
     */
    void release(::asf::stream::MemoryStream& memoryStream);

private:
    CcaMostStream(const CcaMostStream&);

    CcaMostStream& operator=(const CcaMostStream&);

    DECLARE_CLASS_LOGGER();

    uint8* reserve(size_t numberOfBytes);

    void memcpyEndianness(uint8* dest, const uint8* src, size_t n) const;

    const bool _isRunningOnLittleEndianSystem;

    size_t _chunkSize;

    size_t _readIndex;

    size_t _bufferSize;

    size_t _usedSize;

    uint8* _stream;

    bool _bIsInternalStream;

    std::vector< size_t > _selector;
};

class SerializableIF {
public:
    virtual ~SerializableIF() {}

    virtual void serialize(CcaMostStream& rfOutStream) = 0;

    virtual void deserialize(CcaMostStream& rfInStream) = 0;

    virtual size_t getSize() = 0;
};

CcaMostStream& operator<<(CcaMostStream& rfOutStream, SerializableIF& rfSerial);

CcaMostStream& operator>>(CcaMostStream& rfInStream, SerializableIF& rfSerial);

}  // namespace cca
}  // namespace asf

#endif  // ASF_STREAM_CCAMOSTSTREAM_H
