/************************************************************************
 
 *FILE:           alltypes.h
 
 *SW-COMPONENT:   FI, framework for Functional Interface
 
 *DESCRIPTION:    Interface for all types defined in the different FIs
                  
 *AUTHOR:         Jochen Bruns (CM-DI/ENS41)
 
 *COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 
 *HISTORY:
           $ Log $
 * 
 *    Rev 1.0   Oct 7 2001 17:16:00   bnj2hi
 * Initial revision
 *  
 ************************************************************************/

#ifndef RVC_ALLFITYPES_HEADER
#define RVC_ALLFITYPES_HEADER

#include <vector>

class rvc_fi_tclVisitorBase;

const tU16 RVC_FI_C_U16_ERROR_INTERNALFAILURE = 1;
const tU16 RVC_FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION = 2;
const tU16 RVC_FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA = 3;

class rvc_fi_tclTypeBase : public fi_tclTypeBase
{
public:
  rvc_fi_tclTypeBase() {}
  virtual ~rvc_fi_tclTypeBase() {}
   
  virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& /*oVisitor*/) {}
};

class rvc_fi_tclVisitableTypeBase :public rvc_fi_tclTypeBase
{
public:
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& /*oVisitor*/) {}
};

class rvc_fi_tclU8 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclU8():u8Value(0) {}
   tU8 u8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclU8& roRef) const
      { return (u8Value == roRef.u8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_U8;}
};

class rvc_fi_tclS8 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclS8():s8Value(0) {}
   tS8 s8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclS8& roRef) const
      { return (s8Value == roRef.s8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_S8;}
};

class rvc_fi_tclU16 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclU16():u16Value(0) {}
   tU16 u16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclU16& roRef) const
      { return (u16Value == roRef.u16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_U16;}
};

class rvc_fi_tclS16 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclS16():s16Value(0) {}
   tS16 s16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclS16& roRef) const
      { return (s16Value == roRef.s16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_S16;}
};

class rvc_fi_tclU24 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclU24():u24Value(0) {}
   tU24 u24Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclU24& roRef) const
      { return (u24Value == roRef.u24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_U24;}
};

class rvc_fi_tclS24 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclS24():s24Value(0) {}
   tS24 s24Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclS24& roRef) const
      { return (s24Value == roRef.s24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_S24;}
};

class rvc_fi_tclU32 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclU32():u32Value(0) {}
   tU32 u32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclU32& roRef) const
      { return (u32Value == roRef.u32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_U32;}
};

class rvc_fi_tclS32 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclS32():s32Value(0) {}
   tS32 s32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclS32& roRef) const
      { return (s32Value == roRef.s32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_S32;}
};

class rvc_fi_tclU64 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclU64():u64Value(0) {}
   tU64 u64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclU64& roRef) const
      { return (u64Value == roRef.u64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_U64;}
};

class rvc_fi_tclS64 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclS64():s64Value(0) {}
   tS64 s64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclS64& roRef) const
      { return (s64Value == roRef.s64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_S64;}
};

class rvc_fi_tclF32 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclF32():f32Value(0) {}
   tFloat f32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclF32& roRef) const
      { return (f32Value == roRef.f32Value);  } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_F32;}
};

class rvc_fi_tclF64 : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclF64():f64Value(0) {}
   tDouble f64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclF64& roRef) const
      { return (f64Value == roRef.f64Value); } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_F64;}
};

class rvc_fi_tclBool : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclBool():bValue(0) {}
   tBool bValue;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const rvc_fi_tclBool& roRef) const
      { return (bValue == roRef.bValue); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_BOOL;}
};

class rvc_fi_tclString : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tclString(const rvc_fi_tclString& coRef);
   rvc_fi_tclString& operator=(const rvc_fi_tclString& coRef);
   virtual ~rvc_fi_tclString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const rvc_fi_tclString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   rvc_fi_tclString& operator=(tCString coszString); // ADDED
   rvc_fi_tclString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8,
      FI_EN_ISO8859_1M
   };
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const rvc_fi_tclString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};

class rvc_fi_tclList : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclList():u32ListSize(0) {}
   tU32 u32ListSize;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const rvc_fi_tclList& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oIn);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_LIST;}
};

class rvc_fi_tclMultiLanguageString : public rvc_fi_tclVisitableTypeBase
{
public:
   rvc_fi_tclMultiLanguageString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tclMultiLanguageString(const rvc_fi_tclMultiLanguageString& coRef);
   rvc_fi_tclMultiLanguageString& operator=(const rvc_fi_tclMultiLanguageString& coRef);
   virtual ~rvc_fi_tclMultiLanguageString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const rvc_fi_tclMultiLanguageString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(rvc_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   rvc_fi_tclMultiLanguageString& operator=(tCString coszString); // ADDED
   rvc_fi_tclMultiLanguageString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8
   };
   virtual tS32 s32GetTypeId() const { return rvc_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const rvc_fi_tclMultiLanguageString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};
class rvc_fi_tcl_BinContainer;
class rvc_fi_tcl_Reserved;
class rvc_fi_tcl_u8_Percent;
class rvc_fi_tcl_s16_PerThousand;
class rvc_fi_tcl_u16_IndexInReferenceList;
class rvc_fi_tcl_u16_ErrorCode;
class rvc_fi_tcl_u16_RPM;
class rvc_fi_tcl_s32_Kilogram;
class rvc_fi_tcl_s32_Pound;
class rvc_fi_tcl_s16_Meter;
class rvc_fi_tcl_s32_Meter;
class rvc_fi_tcl_s32_Yards;
class rvc_fi_tcl_s32_Foot;
class rvc_fi_tcl_s32_Centimeter;
class rvc_fi_tcl_s32_Inch;
class rvc_fi_tcl_s32_Millimeter;
class rvc_fi_tcl_s32_Microliter;
class rvc_fi_tcl_s32_Second;
class rvc_fi_tcl_u16_MilliSecond;
class rvc_fi_tcl_s32_MilliSecond;
class rvc_fi_tcl_POSIXTime;
class rvc_fi_tcl_e8_DST_State;
class rvc_fi_tcl_TimeZoneElement;
class rvc_fi_tcl_s16_KmPerHour;
class rvc_fi_tcl_s16_MilesPerHour;
class rvc_fi_tcl_s16_Knot;
class rvc_fi_tcl_s16_CentimeterSecond;
class rvc_fi_tcl_s16_Kelvin;
class rvc_fi_tcl_s16_Fahrenheit;
class rvc_fi_tcl_s32_Pascal;
class rvc_fi_tcl_s32_Watt;
class rvc_fi_tcl_FIVersion;
class rvc_fi_tcl_s32_Longitude;
class rvc_fi_tcl_s32_Latitude;
class rvc_fi_tcl_PositionWGS84;
class rvc_fi_tcl_RectangleWGS84;
class rvc_fi_tcl_e16_ISOCountryCode;
class rvc_fi_tcl_e16_ISOLanguageCode;
class rvc_fi_tcl_LanguageID;
class rvc_fi_tcl_u8_DirectionAbsolute;
class rvc_fi_tcl_u8_DirectionRelative;
class rvc_fi_tcl_s16_CentiDegree;
class rvc_fi_tcl_s16_Degree;
class rvc_fi_tcl_u16_DirectionAbsolute_CentiDegree;
class rvc_fi_tcl_s16_DirectionRelative_CentiDegree;
class rvc_fi_tcl_e8_FuelType;
class rvc_fi_tcl_DistTimeFuel;
class rvc_fi_tcl_e8_RVC_CCAError;
class rvc_fi_tcl_e8_RVC_Camera_System_Type;
class rvc_fi_tcl_e8_RVC_SignalStatus;
class rvc_fi_tcl_e8_RVC_SystemStatus;
class rvc_fi_tcl_e8_RVC_StatusGuideLine;
class rvc_fi_tcl_e8_RVC_StatusWarningText;
class rvc_fi_tcl_e8_RVC_Status;
class rvc_fi_tcl_e8_RVC_ButtonState;
class rvc_fi_tcl_e8_RVC_Language;
class rvc_fi_tcl_e8_AVM_StatusBeep;
class rvc_fi_tcl_e8_AffordableIts_Status;
class rvc_fi_tcl_e8_RVC_VideoParameterType;
class rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode;
class rvc_fi_tcl_e8_AffordableIts_StatusFeature;
class rvc_fi_tcl_e8_Exit_Authorized;
class rvc_fi_tcl_RVC_VideoAnalogLevels;
class rvc_fi_tcl_e8_RVC_CameraState;
class rvc_fi_tcl_e8_IPA_BUTTON_ID;
class rvc_fi_tcl_e8_IPA_PatternID;
class rvc_fi_tcl_e8_IPA_MessageID;
class rvc_fi_tcl_e8_IPA_OperationStatus;
class rvc_fi_tcl_e8_IPA_ModeChangeRequest;
class rvc_fi_tcl_e8_IPA_HMIStatus;
class rvc_fi_tcl_e8_IPA_RegionalSelection;
class rvc_fi_tcl_e8_IPA_AvailabilityStatus;
class rvc_fi_tcl_e8_IPA_SwitchMask;
class rvc_fi_tcl_e8_Set_MicState;
class rvc_fi_tcl_e8_IPA_DirectionIndicator;
class rvc_fi_tcl_e8_IPA_VoiceText;
class rvc_fi_tcl_IPA_Voice_Text_Strings;
class rvc_fi_tcl_e8_Camera_PowerSupplyStatus;
class rvc_fi_tcl_e8_Microphone_PowerSupplyStatus;
class rvc_fi_tcl_e8_RVC_Module;
class rvc_fi_tcl_e8_RVC_ReverseSignalStatus;
class rvc_fi_tcl_e8_ParkAssist_UserConfigurationType;
class rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode;
class rvc_fi_tcl_ParkAssist_UserConfiguration;
class rvc_fi_tcl_e8_GdlAdj_AdjustButton;
class rvc_fi_tcl_e8_GdlAdj_Completion;
class rvc_fi_tcl_e8_GdlAdj_Request;
class rvc_fi_tcl_e8_GdlAdj_ConfigurationItem;
class rvc_fi_tcl_GdlAdj_ConfigParameterData;
class rvc_fi_tcl_e8_MVCViewingState;
class rvc_fi_tcl_e8_AVM_View_Request;
class rvc_fi_tcl_e8_AVM_View_Status;
class rvc_fi_tcl_e8_Tailgate_State;
class rvc_fi_tcl_e8_TouchGesture;
class rvc_fi_tcl_e8_Direction;
class rvc_fi_tcl_e8_FAP_SettingItem;
class rvc_fi_tcl_e8_AVM_FAP_SettingState;
class rvc_fi_tcl_e8_AVM_SettingItem;
class rvc_fi_tcl_FAP_Settings_State;
class rvc_fi_tcl_AVM_Settings_State;
class rvc_fi_tcl_GPS_Data;
class rvc_fi_tcl_MapMatched_Data;
class rvc_fi_tcl_Touch_Data;
class rvc_fi_tcl_e8_AVMph3_SwitchId;
class rvc_fi_tcl_e8_AVMph3_SwitchStatus;
class rvc_fi_tcl_AVMph3_SwitchDescriptor;
class rvc_fi_tcl_e8_AVMph3_DispStatus;
class rvc_fi_tcl_e8_AVMph3_ViewPattern;
class rvc_fi_tcl_e8_AVMph3_CautionId;
class rvc_fi_tcl_e8_AVMph3_IconId;
class rvc_fi_tcl_e8_AVMph3_BuzzerStatus;
class rvc_fi_tcl_e8_AVMph3_ErrorMessage;
class rvc_fi_tcl_e8_AVMph3_SettingItemId;
class rvc_fi_tcl_AVMph3_Setting_Item;
class rvc_fi_tcl_e8_AVMph3_SettingOptionState;
class rvc_fi_tcl_AVMph3_Setting_Option;
class rvc_fi_tcl_e8_AVMph3_Setting_PrioView;
class rvc_fi_tcl_e8_AVMph3_Setting_PrioWide;
class rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos;
class rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl;
class rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl;
class rvc_fi_tcl_e8_AVMph3_SonarSensLvl;
class rvc_fi_tcl_AVMph3_Setting_SonarSensLvl;
class rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl;
class rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl;
class rvc_fi_tcl_e8_AVMph3_RequestType;
class rvc_fi_tcl_e8_AVMph3_VersionInfoType;
class rvc_fi_tcl_AVMph3_VersionInfo;
class rvc_fi_tcl_e8_AVMph3_ConfigurationStatus;
class rvc_fi_tcl_e8_AVMph3_Variant;
class rvc_fi_tcl_AVMph3_Unit_ID;
class rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors;
class rvc_fi_tcl_BinContainer : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_BinContainer();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_BinContainer& operator=(const rvc_fi_tcl_BinContainer& coRef);
   rvc_fi_tcl_BinContainer(const rvc_fi_tcl_BinContainer& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<tU8, std::allocator<tU8> > ContainerDataList;
   virtual ~rvc_fi_tcl_BinContainer();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_BinContainer& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_Reserved : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_Reserved();

   virtual ~rvc_fi_tcl_Reserved();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_Reserved& /* roRef */) const
      {  return true; }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u8_Percent : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u8_Percent(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~rvc_fi_tcl_u8_Percent();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u8_Percent& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_PerThousand : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_PerThousand(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_PerThousand();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_PerThousand& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u16_IndexInReferenceList : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u16_IndexInReferenceList(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~rvc_fi_tcl_u16_IndexInReferenceList();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u16_IndexInReferenceList& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u16_ErrorCode : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u16_ErrorCode(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~rvc_fi_tcl_u16_ErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u16_ErrorCode& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u16_RPM : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u16_RPM(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~rvc_fi_tcl_u16_RPM();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u16_RPM& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Kilogram : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Kilogram(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Kilogram();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Kilogram& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Pound : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Pound(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Pound();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Pound& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_Meter : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_Meter(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_Meter& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Meter : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Meter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Meter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Yards : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Yards(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Yards();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Yards& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Foot : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Foot(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Foot();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Foot& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Centimeter : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Centimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Centimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Centimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Inch : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Inch(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Inch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Inch& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Millimeter : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Millimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Millimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Millimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Microliter : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Microliter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Microliter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Microliter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Second : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Second(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Second();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Second& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u16_MilliSecond : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u16_MilliSecond(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~rvc_fi_tcl_u16_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u16_MilliSecond& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_MilliSecond : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_MilliSecond(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_MilliSecond& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_POSIXTime : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_POSIXTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_POSIXTime& operator=(const rvc_fi_tcl_POSIXTime& coRef);
   rvc_fi_tcl_POSIXTime(const rvc_fi_tcl_POSIXTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS32 Time;
   virtual ~rvc_fi_tcl_POSIXTime();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_POSIXTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_DST_State : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_DST_State();
   enum tenType {
      FI_EN_T_STANDARDTIME = 0UL,
      FI_EN_T_DAYLIGHTSAVINGTIME = 1UL,
      FI_EN_T_EVALUATIONNECESSARY = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_DST_State(rvc_fi_tcl_e8_DST_State::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_DST_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_DST_State& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_TimeZoneElement : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_TimeZoneElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_TimeZoneElement& operator=(const rvc_fi_tcl_TimeZoneElement& coRef);
   rvc_fi_tcl_TimeZoneElement(const rvc_fi_tcl_TimeZoneElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tclMultiLanguageString Abbreviation_StandardTime;
   rvc_fi_tclMultiLanguageString Abbreviation_DST_in_effect;
   rvc_fi_tcl_s32_Second TimeZoneDifference;
   rvc_fi_tcl_e8_DST_State DST_State;
   rvc_fi_tcl_s32_Second DSTDifference;
   std::vector<tU8, std::allocator<tU8> > DST_TimeDomain;
   virtual ~rvc_fi_tcl_TimeZoneElement();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_TimeZoneElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_KmPerHour : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_KmPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_KmPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_KmPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_MilesPerHour : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_MilesPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_MilesPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_MilesPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_Knot : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_Knot(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_Knot();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_Knot& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_CentimeterSecond : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_CentimeterSecond(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_CentimeterSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_CentimeterSecond& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_Kelvin : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_Kelvin(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_Kelvin();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_Kelvin& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_Fahrenheit : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_Fahrenheit(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_Fahrenheit();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_Fahrenheit& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Pascal : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Pascal(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Pascal();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Pascal& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Watt : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Watt(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Watt();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Watt& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_FIVersion : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_FIVersion();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_FIVersion& operator=(const rvc_fi_tcl_FIVersion& coRef);
   rvc_fi_tcl_FIVersion(const rvc_fi_tcl_FIVersion& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 MajorVersion;
   tU8 MinorVersion;
   tU8 PatchVersion;
   virtual ~rvc_fi_tcl_FIVersion();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_FIVersion& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Longitude : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Longitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Longitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Longitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s32_Latitude : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s32_Latitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~rvc_fi_tcl_s32_Latitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s32_Latitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_PositionWGS84 : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_PositionWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_PositionWGS84& operator=(const rvc_fi_tcl_PositionWGS84& coRef);
   rvc_fi_tcl_PositionWGS84(const rvc_fi_tcl_PositionWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_s32_Longitude Longitude;
   rvc_fi_tcl_s32_Latitude Latitude;
   virtual ~rvc_fi_tcl_PositionWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_PositionWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_RectangleWGS84 : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_RectangleWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_RectangleWGS84& operator=(const rvc_fi_tcl_RectangleWGS84& coRef);
   rvc_fi_tcl_RectangleWGS84(const rvc_fi_tcl_RectangleWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_s32_Longitude LongitudeMin;
   rvc_fi_tcl_s32_Latitude LatitudeMin;
   rvc_fi_tcl_s32_Longitude LongitudeMax;
   rvc_fi_tcl_s32_Latitude LatitudeMax;
   virtual ~rvc_fi_tcl_RectangleWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_RectangleWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e16_ISOCountryCode : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e16_ISOCountryCode();
   enum tenType {
      FI_EN_ISO_ALPHA_3_UNDEFINED = 0UL,
      FI_EN_ISO_ALPHA_3_AFG = 1223UL,
      FI_EN_ISO_ALPHA_3_ALB = 1410UL,
      FI_EN_ISO_ALPHA_3_DZA = 4929UL,
      FI_EN_ISO_ALPHA_3_ASM = 1645UL,
      FI_EN_ISO_ALPHA_3_AND = 1476UL,
      FI_EN_ISO_ALPHA_3_AGO = 1263UL,
      FI_EN_ISO_ALPHA_3_AIA = 1313UL,
      FI_EN_ISO_ALPHA_3_ATG = 1671UL,
      FI_EN_ISO_ALPHA_3_ARG = 1607UL,
      FI_EN_ISO_ALPHA_3_ARM = 1613UL,
      FI_EN_ISO_ALPHA_3_ABW = 1111UL,
      FI_EN_ISO_ALPHA_3_AUS = 1715UL,
      FI_EN_ISO_ALPHA_3_AUT = 1716UL,
      FI_EN_ISO_ALPHA_3_AZE = 1861UL,
      FI_EN_ISO_ALPHA_3_BHS = 2323UL,
      FI_EN_ISO_ALPHA_3_BHR = 2322UL,
      FI_EN_ISO_ALPHA_3_BGD = 2276UL,
      FI_EN_ISO_ALPHA_3_BRB = 2626UL,
      FI_EN_ISO_ALPHA_3_BLR = 2450UL,
      FI_EN_ISO_ALPHA_3_BEL = 2220UL,
      FI_EN_ISO_ALPHA_3_BLZ = 2458UL,
      FI_EN_ISO_ALPHA_3_BEN = 2222UL,
      FI_EN_ISO_ALPHA_3_BMU = 2485UL,
      FI_EN_ISO_ALPHA_3_BTN = 2702UL,
      FI_EN_ISO_ALPHA_3_BOL = 2540UL,
      FI_EN_ISO_ALPHA_3_BIH = 2344UL,
      FI_EN_ISO_ALPHA_3_BWA = 2785UL,
      FI_EN_ISO_ALPHA_3_BRA = 2625UL,
      FI_EN_ISO_ALPHA_3_BRN = 2638UL,
      FI_EN_ISO_ALPHA_3_BGR = 2290UL,
      FI_EN_ISO_ALPHA_3_BFA = 2241UL,
      FI_EN_ISO_ALPHA_3_BDI = 2185UL,
      FI_EN_ISO_ALPHA_3_KHM = 11533UL,
      FI_EN_ISO_ALPHA_3_CMR = 3506UL,
      FI_EN_ISO_ALPHA_3_CAN = 3118UL,
      FI_EN_ISO_ALPHA_3_CPV = 3606UL,
      FI_EN_ISO_ALPHA_3_CYM = 3885UL,
      FI_EN_ISO_ALPHA_3_CAF = 3110UL,
      FI_EN_ISO_ALPHA_3_TCD = 20580UL,
      FI_EN_ISO_ALPHA_3_CHL = 3340UL,
      FI_EN_ISO_ALPHA_3_CHN = 3342UL,
      FI_EN_ISO_ALPHA_3_COL = 3564UL,
      FI_EN_ISO_ALPHA_3_COM = 3565UL,
      FI_EN_ISO_ALPHA_3_COG = 3559UL,
      FI_EN_ISO_ALPHA_3_COD = 3556UL,
      FI_EN_ISO_ALPHA_3_COK = 3563UL,
      FI_EN_ISO_ALPHA_3_CRI = 3657UL,
      FI_EN_ISO_ALPHA_3_CIV = 3382UL,
      FI_EN_ISO_ALPHA_3_HRV = 8790UL,
      FI_EN_ISO_ALPHA_3_CUB = 3746UL,
      FI_EN_ISO_ALPHA_3_CYP = 3888UL,
      FI_EN_ISO_ALPHA_3_CZE = 3909UL,
      FI_EN_ISO_ALPHA_3_DNK = 4555UL,
      FI_EN_ISO_ALPHA_3_DJI = 4425UL,
      FI_EN_ISO_ALPHA_3_DMA = 4513UL,
      FI_EN_ISO_ALPHA_3_DOM = 4589UL,
      FI_EN_ISO_ALPHA_3_ECU = 5237UL,
      FI_EN_ISO_ALPHA_3_EGY = 5369UL,
      FI_EN_ISO_ALPHA_3_SLV = 19862UL,
      FI_EN_ISO_ALPHA_3_GNQ = 7633UL,
      FI_EN_ISO_ALPHA_3_ERI = 5705UL,
      FI_EN_ISO_ALPHA_3_EST = 5748UL,
      FI_EN_ISO_ALPHA_3_ETH = 5768UL,
      FI_EN_ISO_ALPHA_3_FLK = 6539UL,
      FI_EN_ISO_ALPHA_3_FRO = 6735UL,
      FI_EN_ISO_ALPHA_3_FJI = 6473UL,
      FI_EN_ISO_ALPHA_3_FIN = 6446UL,
      FI_EN_ISO_ALPHA_3_FRA = 6721UL,
      FI_EN_ISO_ALPHA_3_GUF = 7846UL,
      FI_EN_ISO_ALPHA_3_PYF = 17190UL,
      FI_EN_ISO_ALPHA_3_GAB = 7202UL,
      FI_EN_ISO_ALPHA_3_GMB = 7586UL,
      FI_EN_ISO_ALPHA_3_GEO = 7343UL,
      FI_EN_ISO_ALPHA_3_DEU = 4277UL,
      FI_EN_ISO_ALPHA_3_GHA = 7425UL,
      FI_EN_ISO_ALPHA_3_GIB = 7458UL,
      FI_EN_ISO_ALPHA_3_GRC = 7747UL,
      FI_EN_ISO_ALPHA_3_GRL = 7756UL,
      FI_EN_ISO_ALPHA_3_GRD = 7748UL,
      FI_EN_ISO_ALPHA_3_GLP = 7568UL,
      FI_EN_ISO_ALPHA_3_GUM = 7853UL,
      FI_EN_ISO_ALPHA_3_GTM = 7821UL,
      FI_EN_ISO_ALPHA_3_GIN = 7470UL,
      FI_EN_ISO_ALPHA_3_GNB = 7618UL,
      FI_EN_ISO_ALPHA_3_GUY = 7865UL,
      FI_EN_ISO_ALPHA_3_HTI = 8841UL,
      FI_EN_ISO_ALPHA_3_VAT = 22580UL,
      FI_EN_ISO_ALPHA_3_HND = 8644UL,
      FI_EN_ISO_ALPHA_3_HKG = 8551UL,
      FI_EN_ISO_ALPHA_3_HUN = 8878UL,
      FI_EN_ISO_ALPHA_3_ISL = 9836UL,
      FI_EN_ISO_ALPHA_3_IND = 9668UL,
      FI_EN_ISO_ALPHA_3_IDN = 9358UL,
      FI_EN_ISO_ALPHA_3_INT = 9684UL,
      FI_EN_ISO_ALPHA_3_IRN = 9806UL,
      FI_EN_ISO_ALPHA_3_IRQ = 9809UL,
      FI_EN_ISO_ALPHA_3_IRL = 9804UL,
      FI_EN_ISO_ALPHA_3_ISR = 9842UL,
      FI_EN_ISO_ALPHA_3_ITA = 9857UL,
      FI_EN_ISO_ALPHA_3_JAM = 10285UL,
      FI_EN_ISO_ALPHA_3_JPN = 10766UL,
      FI_EN_ISO_ALPHA_3_JOR = 10738UL,
      FI_EN_ISO_ALPHA_3_KAZ = 11322UL,
      FI_EN_ISO_ALPHA_3_KEN = 11438UL,
      FI_EN_ISO_ALPHA_3_KIR = 11570UL,
      FI_EN_ISO_ALPHA_3_PRK = 16971UL,
      FI_EN_ISO_ALPHA_3_KOR = 11762UL,
      FI_EN_ISO_ALPHA_3_KWT = 12020UL,
      FI_EN_ISO_ALPHA_3_KGZ = 11514UL,
      FI_EN_ISO_ALPHA_3_LAO = 12335UL,
      FI_EN_ISO_ALPHA_3_LVA = 12993UL,
      FI_EN_ISO_ALPHA_3_LBN = 12366UL,
      FI_EN_ISO_ALPHA_3_LSO = 12911UL,
      FI_EN_ISO_ALPHA_3_LBR = 12370UL,
      FI_EN_ISO_ALPHA_3_LBY = 12377UL,
      FI_EN_ISO_ALPHA_3_LIE = 12581UL,
      FI_EN_ISO_ALPHA_3_LTU = 12949UL,
      FI_EN_ISO_ALPHA_3_LUX = 12984UL,
      FI_EN_ISO_ALPHA_3_MAC = 13347UL,
      FI_EN_ISO_ALPHA_3_MKD = 13668UL,
      FI_EN_ISO_ALPHA_3_MDG = 13447UL,
      FI_EN_ISO_ALPHA_3_MWI = 14057UL,
      FI_EN_ISO_ALPHA_3_MYS = 14131UL,
      FI_EN_ISO_ALPHA_3_MDV = 13462UL,
      FI_EN_ISO_ALPHA_3_MLI = 13705UL,
      FI_EN_ISO_ALPHA_3_MLT = 13716UL,
      FI_EN_ISO_ALPHA_3_MHL = 13580UL,
      FI_EN_ISO_ALPHA_3_MTQ = 13969UL,
      FI_EN_ISO_ALPHA_3_MRT = 13908UL,
      FI_EN_ISO_ALPHA_3_MUS = 14003UL,
      FI_EN_ISO_ALPHA_3_MYT = 14132UL,
      FI_EN_ISO_ALPHA_3_MEX = 13496UL,
      FI_EN_ISO_ALPHA_3_FSM = 6765UL,
      FI_EN_ISO_ALPHA_3_MDA = 13441UL,
      FI_EN_ISO_ALPHA_3_MCO = 13423UL,
      FI_EN_ISO_ALPHA_3_MNG = 13767UL,
      FI_EN_ISO_ALPHA_3_MNE = 13765UL,
      FI_EN_ISO_ALPHA_3_MSR = 13938UL,
      FI_EN_ISO_ALPHA_3_MAR = 13362UL,
      FI_EN_ISO_ALPHA_3_MOZ = 13818UL,
      FI_EN_ISO_ALPHA_3_MMR = 13746UL,
      FI_EN_ISO_ALPHA_3_NAM = 14381UL,
      FI_EN_ISO_ALPHA_3_NRU = 14933UL,
      FI_EN_ISO_ALPHA_3_NPL = 14860UL,
      FI_EN_ISO_ALPHA_3_NLD = 14724UL,
      FI_EN_ISO_ALPHA_3_ANT = 1492UL,
      FI_EN_ISO_ALPHA_3_NCL = 14444UL,
      FI_EN_ISO_ALPHA_3_NZL = 15180UL,
      FI_EN_ISO_ALPHA_3_NIC = 14627UL,
      FI_EN_ISO_ALPHA_3_NER = 14514UL,
      FI_EN_ISO_ALPHA_3_NGA = 14561UL,
      FI_EN_ISO_ALPHA_3_NIU = 14645UL,
      FI_EN_ISO_ALPHA_3_NFK = 14539UL,
      FI_EN_ISO_ALPHA_3_MNP = 13776UL,
      FI_EN_ISO_ALPHA_3_NOR = 14834UL,
      FI_EN_ISO_ALPHA_3_PSE = 16997UL,
      FI_EN_ISO_ALPHA_3_OMN = 15790UL,
      FI_EN_ISO_ALPHA_3_PAK = 16427UL,
      FI_EN_ISO_ALPHA_3_PLW = 16791UL,
      FI_EN_ISO_ALPHA_3_PAN = 16430UL,
      FI_EN_ISO_ALPHA_3_PNG = 16839UL,
      FI_EN_ISO_ALPHA_3_PRY = 16985UL,
      FI_EN_ISO_ALPHA_3_PER = 16562UL,
      FI_EN_ISO_ALPHA_3_PHL = 16652UL,
      FI_EN_ISO_ALPHA_3_PCN = 16494UL,
      FI_EN_ISO_ALPHA_3_POL = 16876UL,
      FI_EN_ISO_ALPHA_3_PRT = 16980UL,
      FI_EN_ISO_ALPHA_3_PRI = 16969UL,
      FI_EN_ISO_ALPHA_3_QAT = 17460UL,
      FI_EN_ISO_ALPHA_3_REU = 18613UL,
      FI_EN_ISO_ALPHA_3_ROU = 18933UL,
      FI_EN_ISO_ALPHA_3_RUS = 19123UL,
      FI_EN_ISO_ALPHA_3_RWA = 19169UL,
      FI_EN_ISO_ALPHA_3_KNA = 11713UL,
      FI_EN_ISO_ALPHA_3_LCA = 12385UL,
      FI_EN_ISO_ALPHA_3_VCT = 22644UL,
      FI_EN_ISO_ALPHA_3_WSM = 24173UL,
      FI_EN_ISO_ALPHA_3_SMR = 19890UL,
      FI_EN_ISO_ALPHA_3_STP = 20112UL,
      FI_EN_ISO_ALPHA_3_SAU = 19509UL,
      FI_EN_ISO_ALPHA_3_SEN = 19630UL,
      FI_EN_ISO_ALPHA_3_SRB = 20034UL,
      FI_EN_ISO_ALPHA_3_SYC = 20259UL,
      FI_EN_ISO_ALPHA_3_SLE = 19845UL,
      FI_EN_ISO_ALPHA_3_SGP = 19696UL,
      FI_EN_ISO_ALPHA_3_SVK = 20171UL,
      FI_EN_ISO_ALPHA_3_SVN = 20174UL,
      FI_EN_ISO_ALPHA_3_SLB = 19842UL,
      FI_EN_ISO_ALPHA_3_SOM = 19949UL,
      FI_EN_ISO_ALPHA_3_ZAF = 26662UL,
      FI_EN_ISO_ALPHA_3_ESP = 5744UL,
      FI_EN_ISO_ALPHA_3_LKA = 12641UL,
      FI_EN_ISO_ALPHA_3_SHN = 19726UL,
      FI_EN_ISO_ALPHA_3_SPM = 19981UL,
      FI_EN_ISO_ALPHA_3_SDN = 19598UL,
      FI_EN_ISO_ALPHA_3_SUR = 20146UL,
      FI_EN_ISO_ALPHA_3_SJM = 19789UL,
      FI_EN_ISO_ALPHA_3_SWZ = 20218UL,
      FI_EN_ISO_ALPHA_3_SWE = 20197UL,
      FI_EN_ISO_ALPHA_3_CHE = 3333UL,
      FI_EN_ISO_ALPHA_3_SYR = 20274UL,
      FI_EN_ISO_ALPHA_3_TWN = 21230UL,
      FI_EN_ISO_ALPHA_3_TJK = 20811UL,
      FI_EN_ISO_ALPHA_3_TZA = 21313UL,
      FI_EN_ISO_ALPHA_3_THA = 20737UL,
      FI_EN_ISO_ALPHA_3_TLS = 20883UL,
      FI_EN_ISO_ALPHA_3_TGO = 20719UL,
      FI_EN_ISO_ALPHA_3_TKL = 20844UL,
      FI_EN_ISO_ALPHA_3_TON = 20974UL,
      FI_EN_ISO_ALPHA_3_TTO = 21135UL,
      FI_EN_ISO_ALPHA_3_TUN = 21166UL,
      FI_EN_ISO_ALPHA_3_TUR = 21170UL,
      FI_EN_ISO_ALPHA_3_TKM = 20845UL,
      FI_EN_ISO_ALPHA_3_TCA = 20577UL,
      FI_EN_ISO_ALPHA_3_TUV = 21174UL,
      FI_EN_ISO_ALPHA_3_UGA = 21729UL,
      FI_EN_ISO_ALPHA_3_UKR = 21874UL,
      FI_EN_ISO_ALPHA_3_ARE = 1605UL,
      FI_EN_ISO_ALPHA_3_GBR = 7250UL,
      FI_EN_ISO_ALPHA_3_USA = 22113UL,
      FI_EN_ISO_ALPHA_3_URY = 22105UL,
      FI_EN_ISO_ALPHA_3_UZB = 22338UL,
      FI_EN_ISO_ALPHA_3_VUT = 23220UL,
      FI_EN_ISO_ALPHA_3_VEN = 22702UL,
      FI_EN_ISO_ALPHA_3_VNM = 22989UL,
      FI_EN_ISO_ALPHA_3_VGB = 22754UL,
      FI_EN_ISO_ALPHA_3_VIR = 22834UL,
      FI_EN_ISO_ALPHA_3_WLF = 23942UL,
      FI_EN_ISO_ALPHA_3_ESH = 5736UL,
      FI_EN_ISO_ALPHA_3_YEM = 25773UL,
      FI_EN_ISO_ALPHA_3_YUG = 26279UL,
      FI_EN_ISO_ALPHA_3_ZMB = 27042UL,
      FI_EN_ISO_ALPHA_3_ZWE = 27365UL
   };
   tenType enType;

   rvc_fi_tcl_e16_ISOCountryCode(rvc_fi_tcl_e16_ISOCountryCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e16_ISOCountryCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e16_ISOCountryCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e16_ISOLanguageCode : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e16_ISOLanguageCode();
   enum tenType {
      FI_EN_ISO_639_1_UNDEFINED = 0UL,
      FI_EN_ISO_639_1_AA = 24929UL,
      FI_EN_ISO_639_1_AB = 24930UL,
      FI_EN_ISO_639_1_AE = 24933UL,
      FI_EN_ISO_639_1_AF = 24934UL,
      FI_EN_ISO_639_1_AK = 24939UL,
      FI_EN_ISO_639_1_AM = 24941UL,
      FI_EN_ISO_639_1_AN = 24942UL,
      FI_EN_ISO_639_1_AR = 24946UL,
      FI_EN_ISO_639_1_AS = 24947UL,
      FI_EN_ISO_639_1_AV = 24950UL,
      FI_EN_ISO_639_1_AY = 24953UL,
      FI_EN_ISO_639_1_AZ = 24954UL,
      FI_EN_ISO_639_1_BA = 25185UL,
      FI_EN_ISO_639_1_BE = 25189UL,
      FI_EN_ISO_639_1_BG = 25191UL,
      FI_EN_ISO_639_1_BH = 25192UL,
      FI_EN_ISO_639_1_BI = 25193UL,
      FI_EN_ISO_639_1_BL = 25196UL,
      FI_EN_ISO_639_1_BM = 25197UL,
      FI_EN_ISO_639_1_BN = 25198UL,
      FI_EN_ISO_639_1_BO = 25199UL,
      FI_EN_ISO_639_1_BR = 25202UL,
      FI_EN_ISO_639_1_BS = 25203UL,
      FI_EN_ISO_639_1_BT = 25204UL,
      FI_EN_ISO_639_1_CA = 25441UL,
      FI_EN_ISO_639_1_CE = 25445UL,
      FI_EN_ISO_639_1_CF = 25446UL,
      FI_EN_ISO_639_1_CH = 25448UL,
      FI_EN_ISO_639_1_CL = 25452UL,
      FI_EN_ISO_639_1_CN = 25454UL,
      FI_EN_ISO_639_1_CO = 25455UL,
      FI_EN_ISO_639_1_CR = 25458UL,
      FI_EN_ISO_639_1_CS = 25459UL,
      FI_EN_ISO_639_1_CT = 25460UL,
      FI_EN_ISO_639_1_CU = 25461UL,
      FI_EN_ISO_639_1_CV = 25462UL,
      FI_EN_ISO_639_1_CY = 25465UL,
      FI_EN_ISO_639_1_DA = 25697UL,
      FI_EN_ISO_639_1_DE = 25701UL,
      FI_EN_ISO_639_1_DV = 25718UL,
      FI_EN_ISO_639_1_DZ = 25722UL,
      FI_EN_ISO_639_1_EE = 25957UL,
      FI_EN_ISO_639_1_EG = 25959UL,
      FI_EN_ISO_639_1_EL = 25964UL,
      FI_EN_ISO_639_1_EN = 25966UL,
      FI_EN_ISO_639_1_EO = 25967UL,
      FI_EN_ISO_639_1_ES = 25971UL,
      FI_EN_ISO_639_1_ET = 25972UL,
      FI_EN_ISO_639_1_EU = 25973UL,
      FI_EN_ISO_639_1_FA = 26209UL,
      FI_EN_ISO_639_1_FF = 26214UL,
      FI_EN_ISO_639_1_FI = 26217UL,
      FI_EN_ISO_639_1_FJ = 26218UL,
      FI_EN_ISO_639_1_FO = 26223UL,
      FI_EN_ISO_639_1_FR = 26226UL,
      FI_EN_ISO_639_1_FY = 26233UL,
      FI_EN_ISO_639_1_GA = 26465UL,
      FI_EN_ISO_639_1_GD = 26468UL,
      FI_EN_ISO_639_1_GL = 26476UL,
      FI_EN_ISO_639_1_GN = 26478UL,
      FI_EN_ISO_639_1_GU = 26485UL,
      FI_EN_ISO_639_1_GV = 26486UL,
      FI_EN_ISO_639_1_HA = 26721UL,
      FI_EN_ISO_639_1_HE = 26725UL,
      FI_EN_ISO_639_1_HI = 26729UL,
      FI_EN_ISO_639_1_HO = 26735UL,
      FI_EN_ISO_639_1_HR = 26738UL,
      FI_EN_ISO_639_1_HU = 26741UL,
      FI_EN_ISO_639_1_HY = 26745UL,
      FI_EN_ISO_639_1_HZ = 26746UL,
      FI_EN_ISO_639_1_IA = 26977UL,
      FI_EN_ISO_639_1_ID = 26980UL,
      FI_EN_ISO_639_1_IE = 26981UL,
      FI_EN_ISO_639_1_IG = 26983UL,
      FI_EN_ISO_639_1_II = 26985UL,
      FI_EN_ISO_639_1_IK = 26987UL,
      FI_EN_ISO_639_1_IO = 26991UL,
      FI_EN_ISO_639_1_IS = 26995UL,
      FI_EN_ISO_639_1_IT = 26996UL,
      FI_EN_ISO_639_1_IU = 26997UL,
      FI_EN_ISO_639_1_JA = 27233UL,
      FI_EN_ISO_639_1_JV = 27254UL,
      FI_EN_ISO_639_1_KA = 27489UL,
      FI_EN_ISO_639_1_KG = 27495UL,
      FI_EN_ISO_639_1_KI = 27497UL,
      FI_EN_ISO_639_1_KJ = 27498UL,
      FI_EN_ISO_639_1_KK = 27499UL,
      FI_EN_ISO_639_1_KL = 27500UL,
      FI_EN_ISO_639_1_KM = 27501UL,
      FI_EN_ISO_639_1_KN = 27502UL,
      FI_EN_ISO_639_1_KO = 27503UL,
      FI_EN_ISO_639_1_KR = 27506UL,
      FI_EN_ISO_639_1_KS = 27507UL,
      FI_EN_ISO_639_1_KU = 27509UL,
      FI_EN_ISO_639_1_KV = 27510UL,
      FI_EN_ISO_639_1_KW = 27511UL,
      FI_EN_ISO_639_1_KY = 27513UL,
      FI_EN_ISO_639_1_LA = 27745UL,
      FI_EN_ISO_639_1_LB = 27746UL,
      FI_EN_ISO_639_1_LG = 27751UL,
      FI_EN_ISO_639_1_LI = 27753UL,
      FI_EN_ISO_639_1_LN = 27758UL,
      FI_EN_ISO_639_1_LO = 27759UL,
      FI_EN_ISO_639_1_LT = 27764UL,
      FI_EN_ISO_639_1_LU = 27765UL,
      FI_EN_ISO_639_1_LV = 27766UL,
      FI_EN_ISO_639_1_MC = 28003UL,
      FI_EN_ISO_639_1_MG = 28007UL,
      FI_EN_ISO_639_1_MH = 28008UL,
      FI_EN_ISO_639_1_MI = 28009UL,
      FI_EN_ISO_639_1_MK = 28011UL,
      FI_EN_ISO_639_1_ML = 28012UL,
      FI_EN_ISO_639_1_MN = 28014UL,
      FI_EN_ISO_639_1_MO = 28015UL,
      FI_EN_ISO_639_1_MR = 28018UL,
      FI_EN_ISO_639_1_MS = 28019UL,
      FI_EN_ISO_639_1_MT = 28020UL,
      FI_EN_ISO_639_1_MY = 28025UL,
      FI_EN_ISO_639_1_NA = 28257UL,
      FI_EN_ISO_639_1_NB = 28258UL,
      FI_EN_ISO_639_1_ND = 28260UL,
      FI_EN_ISO_639_1_NE = 28261UL,
      FI_EN_ISO_639_1_NG = 28263UL,
      FI_EN_ISO_639_1_NL = 28268UL,
      FI_EN_ISO_639_1_NN = 28270UL,
      FI_EN_ISO_639_1_NO = 28271UL,
      FI_EN_ISO_639_1_NR = 28274UL,
      FI_EN_ISO_639_1_NV = 28278UL,
      FI_EN_ISO_639_1_NY = 28281UL,
      FI_EN_ISO_639_1_OC = 28515UL,
      FI_EN_ISO_639_1_OJ = 28522UL,
      FI_EN_ISO_639_1_OM = 28525UL,
      FI_EN_ISO_639_1_OR = 28530UL,
      FI_EN_ISO_639_1_OS = 28531UL,
      FI_EN_ISO_639_1_PA = 28769UL,
      FI_EN_ISO_639_1_PI = 28777UL,
      FI_EN_ISO_639_1_PL = 28780UL,
      FI_EN_ISO_639_1_PS = 28787UL,
      FI_EN_ISO_639_1_PT = 28788UL,
      FI_EN_ISO_639_1_QU = 29045UL,
      FI_EN_ISO_639_1_RM = 29293UL,
      FI_EN_ISO_639_1_RN = 29294UL,
      FI_EN_ISO_639_1_RO = 29295UL,
      FI_EN_ISO_639_1_RT = 29300UL,
      FI_EN_ISO_639_1_RU = 29301UL,
      FI_EN_ISO_639_1_RW = 29303UL,
      FI_EN_ISO_639_1_SA = 29537UL,
      FI_EN_ISO_639_1_SB = 29538UL,
      FI_EN_ISO_639_1_SC = 29539UL,
      FI_EN_ISO_639_1_SD = 29540UL,
      FI_EN_ISO_639_1_SE = 29541UL,
      FI_EN_ISO_639_1_SG = 29543UL,
      FI_EN_ISO_639_1_SH = 29544UL,
      FI_EN_ISO_639_1_SI = 29545UL,
      FI_EN_ISO_639_1_SK = 29547UL,
      FI_EN_ISO_639_1_SL = 29548UL,
      FI_EN_ISO_639_1_SM = 29549UL,
      FI_EN_ISO_639_1_SN = 29550UL,
      FI_EN_ISO_639_1_SO = 29551UL,
      FI_EN_ISO_639_1_SQ = 29553UL,
      FI_EN_ISO_639_1_SR = 29554UL,
      FI_EN_ISO_639_1_SS = 29555UL,
      FI_EN_ISO_639_1_ST = 29556UL,
      FI_EN_ISO_639_1_SU = 29557UL,
      FI_EN_ISO_639_1_SV = 29558UL,
      FI_EN_ISO_639_1_SW = 29559UL,
      FI_EN_ISO_639_1_SX = 29560UL,
      FI_EN_ISO_639_1_TA = 29793UL,
      FI_EN_ISO_639_1_TE = 29797UL,
      FI_EN_ISO_639_1_TG = 29799UL,
      FI_EN_ISO_639_1_TH = 29800UL,
      FI_EN_ISO_639_1_TI = 29801UL,
      FI_EN_ISO_639_1_TK = 29803UL,
      FI_EN_ISO_639_1_TL = 29804UL,
      FI_EN_ISO_639_1_TN = 29806UL,
      FI_EN_ISO_639_1_TO = 29807UL,
      FI_EN_ISO_639_1_TR = 29810UL,
      FI_EN_ISO_639_1_TS = 29811UL,
      FI_EN_ISO_639_1_TT = 29812UL,
      FI_EN_ISO_639_1_TW = 29815UL,
      FI_EN_ISO_639_1_TY = 29817UL,
      FI_EN_ISO_639_1_UE = 30053UL,
      FI_EN_ISO_639_1_UG = 30055UL,
      FI_EN_ISO_639_1_UK = 30059UL,
      FI_EN_ISO_639_1_UL = 30060UL,
      FI_EN_ISO_639_1_UR = 30066UL,
      FI_EN_ISO_639_1_UZ = 30074UL,
      FI_EN_ISO_639_1_VA = 30305UL,
      FI_EN_ISO_639_1_VE = 30309UL,
      FI_EN_ISO_639_1_VI = 30313UL,
      FI_EN_ISO_639_1_VO = 30319UL,
      FI_EN_ISO_639_1_WA = 30561UL,
      FI_EN_ISO_639_1_WO = 30575UL,
      FI_EN_ISO_639_1_XH = 30824UL,
      FI_EN_ISO_639_1_YI = 31081UL,
      FI_EN_ISO_639_1_YO = 31087UL,
      FI_EN_ISO_639_1_ZA = 31329UL,
      FI_EN_ISO_639_1_ZH = 31336UL,
      FI_EN_ISO_639_1_ZL = 31340UL,
      FI_EN_ISO_639_1_ZU = 31349UL
   };
   tenType enType;

   rvc_fi_tcl_e16_ISOLanguageCode(rvc_fi_tcl_e16_ISOLanguageCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e16_ISOLanguageCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e16_ISOLanguageCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_LanguageID : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_LanguageID();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_LanguageID& operator=(const rvc_fi_tcl_LanguageID& coRef);
   rvc_fi_tcl_LanguageID(const rvc_fi_tcl_LanguageID& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e16_ISOLanguageCode ISO639_LanguageCode;
   rvc_fi_tcl_e16_ISOCountryCode ISO3166_CountryCode;
   virtual ~rvc_fi_tcl_LanguageID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_LanguageID& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u8_DirectionAbsolute : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u8_DirectionAbsolute(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~rvc_fi_tcl_u8_DirectionAbsolute();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u8_DirectionAbsolute& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u8_DirectionRelative : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u8_DirectionRelative(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~rvc_fi_tcl_u8_DirectionRelative();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u8_DirectionRelative& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_CentiDegree : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_Degree : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_Degree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_Degree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_Degree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_u16_DirectionAbsolute_CentiDegree : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_u16_DirectionAbsolute_CentiDegree(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~rvc_fi_tcl_u16_DirectionAbsolute_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_u16_DirectionAbsolute_CentiDegree& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_s16_DirectionRelative_CentiDegree : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_s16_DirectionRelative_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~rvc_fi_tcl_s16_DirectionRelative_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_s16_DirectionRelative_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_FuelType : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_FuelType();
   enum tenType {
      FI_EN_RESERVED = 0UL,
      FI_EN_PETROL = 1UL,
      FI_EN_DIESEL = 2UL,
      FI_EN_HYDROGEN = 3UL,
      FI_EN_COMPRESSEDNATURALGAS = 4UL,
      FI_EN_LIQUIFIEDPETROLEUMGAS = 5UL
   };
   tenType enType;

   rvc_fi_tcl_e8_FuelType(rvc_fi_tcl_e8_FuelType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_FuelType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_FuelType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_DistTimeFuel : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_DistTimeFuel();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_DistTimeFuel& operator=(const rvc_fi_tcl_DistTimeFuel& coRef);
   rvc_fi_tcl_DistTimeFuel(const rvc_fi_tcl_DistTimeFuel& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_s32_Meter Distance;
   rvc_fi_tcl_s32_Second Time;
   rvc_fi_tcl_s32_Microliter Fuel;
   virtual ~rvc_fi_tcl_DistTimeFuel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_DistTimeFuel& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_CCAError : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_CCAError();
   enum tenType {
      FI_EN_RVC_OPCODE_NOT_SUPPORTED = 16UL,
      FI_EN_RVC_UPREG_FAILURE = 17UL,
      FI_EN_RVC_RELUPREG_FAILURE = 18UL,
      FI_EN_RVC_WRONG_PARAMETERS = 19UL,
      FI_EN_RVC_STRING_PARAMETER_OVERSIZE = 20UL,
      FI_EN_RVC_PARAMETER_LIMITED = 21UL,
      FI_EN_RVC_UNKNOWN_MSG = 22UL,
      FI_EN_RVC_INVALID_OPCODE = 23UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_CCAError(rvc_fi_tcl_e8_RVC_CCAError::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_CCAError();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_CCAError& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_Camera_System_Type : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_Camera_System_Type();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_RVC = 1UL,
      FI_EN_RVC_TAILGATE = 2UL,
      FI_EN_AVM_LOW = 3UL,
      FI_EN_AVM_HIGH = 4UL,
      FI_EN_AVM_IPA = 5UL,
      FI_EN_AVM_ENTRY = 6UL,
      FI_EN_AVM_PHASE3 = 7UL,
      FI_EN_MVC = 8UL,
      FI_EN_RVC_RENAULT = 9UL,
      FI_EN_AVM_ENTRY_RENAULT = 10UL,
      FI_EN_AVM_FAP = 11UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_Camera_System_Type(rvc_fi_tcl_e8_RVC_Camera_System_Type::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_Camera_System_Type();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_Camera_System_Type& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_SignalStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_SignalStatus();
   enum tenType {
      FI_EN_SIGNAL_OK = 0UL,
      FI_EN_CONFIRMED_FAILURE = 1UL,
      FI_EN_CURRENT_FAILURE = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_SignalStatus(rvc_fi_tcl_e8_RVC_SignalStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_SignalStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_SignalStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_SystemStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_SystemStatus();
   enum tenType {
      FI_EN_INITIALISATION = 0UL,
      FI_EN_VISUALISATION_OFF = 1UL,
      FI_EN_VISUALISATION_OFF_VIDEO_SIGNAL_FAILURE = 2UL,
      FI_EN_VISUALISATION_OFF_SYSTEM_MALFUNCTION = 3UL,
      FI_EN_RVC_EARLY_VISUALISATION_BLACK_IN = 10UL,
      FI_EN_RVC_EARLY_VISUALISATION_ON = 11UL,
      FI_EN_RVC_EARLY_VISUALISATION_BLACK_OUT = 12UL,
      FI_EN_RVC_VISUALISATION_BLACK_IN = 20UL,
      FI_EN_RVC_VISUALISATION_ON = 21UL,
      FI_EN_RVC_VISUALISATION_ON_CAMERA_SETTINGS = 22UL,
      FI_EN_RVC_VISUALISATION_BLACK_OUT = 23UL,
      FI_EN_RVC_VISUALISATION_ON_GUIDELINE_ADJUSTMENT = 24UL,
      FI_EN_RVC_VISUALISATION_ON_WITH_UPA = 25UL,
      FI_EN_RVC_VISUALISATION_ON_WITH_UPA_HFP = 26UL,
      FI_EN_AVM_EARLY_VISUALISATION_BLACK_IN = 30UL,
      FI_EN_AVM_EARLY_VISUALISATION_ON = 31UL,
      FI_EN_AVM_EARLY_VISUALISATION_BLACK_OUT = 32UL,
      FI_EN_AVM_VISUALISATION_BLACK_IN = 40UL,
      FI_EN_AVM_VISUALISATION_ON = 41UL,
      FI_EN_AVM_VISUALISATION_ON_CAMERA_SETTINGS = 42UL,
      FI_EN_AVM_VISUALISATION_BLACK_OUT = 43UL,
      FI_EN_AVM_VISUALISATION_BLACK_WHILE_ON = 44UL,
      FI_EN_AVM_VISUALISATION_ON_WITH_UPA = 45UL,
      FI_EN_AVM_VISUALISATION_ON_WITH_UPA_HFP = 46UL,
      FI_EN_VISUALISATION_BLACK_IN_CAMERA_SETTINGS = 50UL,
      FI_EN_VISUALISATION_BLACK_OUT_CAMERA_SETTINGS = 51UL,
      FI_EN_MVC_VISUALISATION_BLACK_IN = 60UL,
      FI_EN_MVC_VISUALISATION_ON = 61UL,
      FI_EN_MVC_VISUALISATION_ON_CAMERA_SETTINGS = 62UL,
      FI_EN_MVC_VISUALISATION_BLACK_OUT = 63UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_SystemStatus(rvc_fi_tcl_e8_RVC_SystemStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_SystemStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_SystemStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_StatusGuideLine : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_StatusGuideLine();
   enum tenType {
      FI_EN_STATICGUIDELINEOFF = 0UL,
      FI_EN_STATICGUIDELINEON = 1UL,
      FI_EN_DYNAMICGUIDELINEOFF = 2UL,
      FI_EN_DYNAMICGUIDELINEON = 3UL,
      FI_EN_GUIDELINENOTAVAILABLE = 15UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_StatusGuideLine(rvc_fi_tcl_e8_RVC_StatusGuideLine::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_StatusGuideLine();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_StatusGuideLine& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_StatusWarningText : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_StatusWarningText();
   enum tenType {
      FI_EN_WARNINGTEXTOFF = 0UL,
      FI_EN_WARNINGTEXTON = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_StatusWarningText(rvc_fi_tcl_e8_RVC_StatusWarningText::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_StatusWarningText();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_StatusWarningText& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_Status : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_Status();
   enum tenType {
      FI_EN_NOT_AVAILABLE = 0UL,
      FI_EN_AVAILABLE = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_Status(rvc_fi_tcl_e8_RVC_Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_ButtonState : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_ButtonState();
   enum tenType {
      FI_EN_RELEASED = 0UL,
      FI_EN_PRESSED = 1UL,
      FI_EN_SHORTPRESSED = 2UL,
      FI_EN_LONGPRESSED = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_ButtonState(rvc_fi_tcl_e8_RVC_ButtonState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_ButtonState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_ButtonState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_Language : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_Language();
   enum tenType {
      FI_EN_ENGLISH_US = 0UL,
      FI_EN_FRENCH_CAN = 1UL,
      FI_EN_SPANISH_LATIN_AMERICAN = 2UL,
      FI_EN_ENGLISH_UK = 3UL,
      FI_EN_GERMAN = 4UL,
      FI_EN_FRENCH_EUR = 5UL,
      FI_EN_ITALIAN = 6UL,
      FI_EN_DUTCH = 7UL,
      FI_EN_SPANISH_EUR = 8UL,
      FI_EN_PORTUGUESE = 9UL,
      FI_EN_RUSSIAN = 10UL,
      FI_EN_TURKISH = 11UL,
      FI_EN_DANISH = 12UL,
      FI_EN_SWEDISH = 13UL,
      FI_EN_FINNISH = 14UL,
      FI_EN_NORWEGIAN = 15UL,
      FI_EN_GREEK = 16UL,
      FI_EN_POLISH = 17UL,
      FI_EN_CZECH = 18UL,
      FI_EN_SLOVAK = 19UL,
      FI_EN_HUNGARIAN = 20UL,
      FI_EN_ARABIC = 21UL,
      FI_EN_THAI = 22UL,
      FI_EN_AUSTRALIAN = 23UL,
      FI_EN_BRAZILIAN = 24UL,
      FI_EN_UKRAINIAN = 25UL,
      FI_EN_KOREAN = 26UL,
      FI_EN_CHINESE_SIMPLIFIED = 27UL,
      FI_EN_CHINESE_TRADITIONAL_CANTONESE = 28UL,
      FI_EN_CHINESE_TRADITIONAL_TAIWANESE = 29UL,
      FI_EN_INDONESIAN = 30UL,
      FI_EN_ROMANIAN = 31UL,
      FI_EN_ALBANESE = 32UL,
      FI_EN_BULGARIAN = 33UL,
      FI_EN_CROATIAN = 34UL,
      FI_EN_ESTONIAN = 35UL,
      FI_EN_FLEMISH = 36UL,
      FI_EN_HEBREW = 37UL,
      FI_EN_HINDI = 38UL,
      FI_EN_JAPANESE = 39UL,
      FI_EN_LATVIAN = 40UL,
      FI_EN_LITHUANIAN = 41UL,
      FI_EN_MALAY = 42UL,
      FI_EN_PERSIAN_FARSI_IRANIAN = 43UL,
      FI_EN_PHILIPPINES = 44UL,
      FI_EN_SERBIAN = 45UL,
      FI_EN_SLOVENIAN = 46UL,
      FI_EN_VIETNAMESE = 47UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_Language(rvc_fi_tcl_e8_RVC_Language::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_Language();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_Language& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVM_StatusBeep : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVM_StatusBeep();
   enum tenType {
      FI_EN_ALLBEEPTYPESNOTACTIVE = 0UL,
      FI_EN_ONLYOPERATIONBEEPACTIVE = 1UL,
      FI_EN_ONLYERRORBEEPACTIVE = 2UL,
      FI_EN_ALLBEEPTYPESACTIVE = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVM_StatusBeep(rvc_fi_tcl_e8_AVM_StatusBeep::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVM_StatusBeep();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVM_StatusBeep& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AffordableIts_Status : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AffordableIts_Status();
   enum tenType {
      FI_EN_NOT_AVAILABLE = 0UL,
      FI_EN_INITIALISING = 1UL,
      FI_EN_AVAILABLE = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AffordableIts_Status(rvc_fi_tcl_e8_AffordableIts_Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AffordableIts_Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AffordableIts_Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_VideoParameterType : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_VideoParameterType();
   enum tenType {
      FI_EN_BRIGHTNESS = 0UL,
      FI_EN_CONTRAST = 1UL,
      FI_EN_COLOR = 2UL,
      FI_EN_HUE = 3UL,
      FI_EN_BLACKLEVEL = 4UL,
      FI_EN_DIMMING = 5UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_VideoParameterType(rvc_fi_tcl_e8_RVC_VideoParameterType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_VideoParameterType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_VideoParameterType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode();
   enum tenType {
      FI_EN_DAY = 0UL,
      FI_EN_NIGHT = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode(rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AffordableIts_StatusFeature : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AffordableIts_StatusFeature();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OFF = 1UL,
      FI_EN_ON = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AffordableIts_StatusFeature(rvc_fi_tcl_e8_AffordableIts_StatusFeature::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AffordableIts_StatusFeature();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AffordableIts_StatusFeature& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_Exit_Authorized : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_Exit_Authorized();
   enum tenType {
      FI_EN_NOT_ALLOWED = 0UL,
      FI_EN_ALLOWED = 1UL,
      FI_EN_UNKNOWN = 255UL
   };
   tenType enType;

   rvc_fi_tcl_e8_Exit_Authorized(rvc_fi_tcl_e8_Exit_Authorized::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_Exit_Authorized();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_Exit_Authorized& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_RVC_VideoAnalogLevels : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_RVC_VideoAnalogLevels();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_RVC_VideoAnalogLevels& operator=(const rvc_fi_tcl_RVC_VideoAnalogLevels& coRef);
   rvc_fi_tcl_RVC_VideoAnalogLevels(const rvc_fi_tcl_RVC_VideoAnalogLevels& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_RVC_VideoParameterType e8Type;
   rvc_fi_tcl_e8_RVC_VideoParameterDimmingMode e8DimmingMode;
   tU8 u8Value;
   virtual ~rvc_fi_tcl_RVC_VideoAnalogLevels();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_RVC_VideoAnalogLevels& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_CameraState : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_CameraState();
   enum tenType {
      FI_EN_RVC_CAMSTATE_INIT = 0UL,
      FI_EN_RVC_CAMSTATE_OFF = 1UL,
      FI_EN_RVC_CAMSTATE_ON = 2UL,
      FI_EN_AVM_VISUALIZATION_ON = 10UL,
      FI_EN_AVM_VISUALIZATION_OFF = 11UL,
      FI_EN_RVC_CAMSTATE_NOT_EXISTENT = 254UL,
      FI_EN_RVC_CAMSTATE_FAILURE = 255UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_CameraState(rvc_fi_tcl_e8_RVC_CameraState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_CameraState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_CameraState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_BUTTON_ID : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_BUTTON_ID();
   enum tenType {
      FI_EN_NO_SWITCH = 0UL,
      FI_EN_P_STEERING_WO_SCAN = 1UL,
      FI_EN_START = 3UL,
      FI_EN_HINT = 4UL,
      FI_EN_CANCEL = 5UL,
      FI_EN_BACK = 6UL,
      FI_EN_ADJUST = 7UL,
      FI_EN_AUTOVIEW_ACTIVE = 8UL,
      FI_EN_AUTOVIEW_INACTIVE = 9UL,
      FI_EN_LEFT_PERPENDICULAR = 10UL,
      FI_EN_RIGHT_PERPENDICULAR = 11UL,
      FI_EN_LEFT_PARALLEL = 12UL,
      FI_EN_RIGHT_PARALLEL = 13UL,
      FI_EN_LEFT_DIAGONAL = 14UL,
      FI_EN_RIGHT_DIAGONAL = 15UL,
      FI_EN_SCROLL_DOWN = 18UL,
      FI_EN_SCROLL_UP = 19UL,
      FI_EN_UP = 20UL,
      FI_EN_DOWN = 21UL,
      FI_EN_LEFT = 22UL,
      FI_EN_RIGHT = 23UL,
      FI_EN_CCW = 24UL,
      FI_EN_CW = 25UL,
      FI_EN_GREYOUT_UP = 26UL,
      FI_EN_GREYOUT_DOWN = 27UL,
      FI_EN_GREYOUT_LEFT = 28UL,
      FI_EN_GREYOUT_RIGHT = 29UL,
      FI_EN_GREYOUT_CCW = 30UL,
      FI_EN_GREYOUT_CW = 31UL,
      FI_EN_P_STEERING = 32UL,
      FI_EN_PARKBOX = 35UL,
      FI_EN_PERPENDICULAR = 36UL,
      FI_EN_VOICE_GUIDANCE = 50UL,
      FI_EN_VOICE_GUIDANCE_VOLUME = 51UL,
      FI_EN_AUTO_SELECT_PARK_TYPE = 52UL,
      FI_EN_AVAIL_PARK_TYPE = 53UL,
      FI_EN_BAY_PARKING = 54UL,
      FI_EN_SLOT_PARKING = 55UL,
      FI_EN_FORWARD_PARKING = 56UL,
      FI_EN_ANGLE_BACKWARD = 57UL,
      FI_EN_ANGLE_FORWARD = 58UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_BUTTON_ID(rvc_fi_tcl_e8_IPA_BUTTON_ID::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_BUTTON_ID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_BUTTON_ID& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_PatternID : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_PatternID();
   enum tenType {
      FI_EN_PATTERN_1 = 0UL,
      FI_EN_PATTERN_2 = 1UL,
      FI_EN_PATTERN_3 = 2UL,
      FI_EN_PATTERN_4 = 3UL,
      FI_EN_PATTERN_5 = 4UL,
      FI_EN_PATTERN_6 = 5UL,
      FI_EN_PATTERN_7 = 6UL,
      FI_EN_PATTERN_8 = 7UL,
      FI_EN_PATTERN_9 = 8UL,
      FI_EN_PATTERN_10 = 9UL,
      FI_EN_PATTERN_11 = 10UL,
      FI_EN_PATTERN_12 = 11UL,
      FI_EN_PATTERN_13 = 12UL,
      FI_EN_PATTERN_14 = 13UL,
      FI_EN_PATTERN_15 = 14UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_PatternID(rvc_fi_tcl_e8_IPA_PatternID::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_PatternID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_PatternID& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_MessageID : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_MessageID();
   enum tenType {
      FI_EN_NO_MESSAGE_DISPLAYED = 0UL,
      FI_EN_ENSURE_IT_IS_SAFE_BEFORE_MANEUVERING = 1UL,
      FI_EN_STOP_BESIDE_PARK_SLOT_TO_START_PARK_ASSIST = 3UL,
      FI_EN_DRIVE_FORWARD = 4UL,
      FI_EN_REDUCE_SPEED = 5UL,
      FI_EN_STOP_AND_SELECT_REVERSE_GEAR = 6UL,
      FI_EN_STOP_AND_SELECT_DRIVE_GEAR = 7UL,
      FI_EN_DRIVE_REARWARD = 8UL,
      FI_EN_STOP_NEXT_TO_EMPTY_PARKING_POSITION = 9UL,
      FI_EN_PRESS_ADJUST_TO_SET_PARKING_POSITION = 10UL,
      FI_EN_ADJUST_PARK_POSITION = 11UL,
      FI_EN_PARK_ASSIST_FINISHED = 16UL,
      FI_EN_SELECT_DRIVE_GEAR = 17UL,
      FI_EN_CENTER_STEERING_WHEEL = 18UL,
      FI_EN_PARK_ASSIST_FAULT = 23UL,
      FI_EN_TURN_SIGNAL_TO_SWITCH_SIDE = 32UL,
      FI_EN_PARK_ASSIST_UNAVAILABLE = 33UL,
      FI_EN_SPACE_FOUND = 38UL,
      FI_EN_STOP = 39UL,
      FI_EN_SELECT_REVERSE_GEAR = 40UL,
      FI_EN_DRIVE_REARWARD_1 = 41UL,
      FI_EN_REDUCE_SPEED_1 = 42UL,
      FI_EN_DRIVE_FORWARD_1 = 43UL,
      FI_EN_PARK_ASSIST_CANCELLED = 44UL,
      FI_EN_PARK_ASSIST_FINISHED_1 = 45UL,
      FI_EN_PARK_ASSIST_CANCELLED_1 = 46UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_MessageID(rvc_fi_tcl_e8_IPA_MessageID::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_MessageID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_MessageID& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_OperationStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_OperationStatus();
   enum tenType {
      FI_EN_NORMAL = 0UL,
      FI_EN_MALFUNCTION = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_OperationStatus(rvc_fi_tcl_e8_IPA_OperationStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_OperationStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_OperationStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_ModeChangeRequest : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_ModeChangeRequest();
   enum tenType {
      FI_EN_NORMAL = 0UL,
      FI_EN_IPA = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_ModeChangeRequest(rvc_fi_tcl_e8_IPA_ModeChangeRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_ModeChangeRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_ModeChangeRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_HMIStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_HMIStatus();
   enum tenType {
      FI_EN_NOTACTIVATED = 0UL,
      FI_EN_ACTIVATED = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_HMIStatus(rvc_fi_tcl_e8_IPA_HMIStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_HMIStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_HMIStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_RegionalSelection : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_RegionalSelection();
   enum tenType {
      FI_EN_JAPAN = 0UL,
      FI_EN_CHINA = 1UL,
      FI_EN_EUROPE = 2UL,
      FI_EN_USA = 3UL,
      FI_EN_INVALID = 7UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_RegionalSelection(rvc_fi_tcl_e8_IPA_RegionalSelection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_RegionalSelection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_RegionalSelection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_AvailabilityStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_AvailabilityStatus();
   enum tenType {
      FI_EN_NOT_AVAILABLE = 0UL,
      FI_EN_INITIALISING = 1UL,
      FI_EN_AVAILABLE = 2UL,
      FI_EN_AVM_IPA_ERROR = 3UL,
      FI_EN_VIDEO_SIGNAL_FAIL = 4UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_AvailabilityStatus(rvc_fi_tcl_e8_IPA_AvailabilityStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_AvailabilityStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_AvailabilityStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_SwitchMask : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_SwitchMask();
   enum tenType {
      FI_EN_NOMASK = 0UL,
      FI_EN_LOWSPEED = 1UL,
      FI_EN_HIGHSPEED = 2UL,
      FI_EN_MIDDLESPEEDMASK = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_SwitchMask(rvc_fi_tcl_e8_IPA_SwitchMask::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_SwitchMask();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_SwitchMask& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_Set_MicState : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_Set_MicState();
   enum tenType {
      FI_EN_OFF = 0UL,
      FI_EN_ON = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_Set_MicState(rvc_fi_tcl_e8_Set_MicState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_Set_MicState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_Set_MicState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_DirectionIndicator : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_DirectionIndicator();
   enum tenType {
      FI_EN_FORWARD = 0UL,
      FI_EN_BACKWARD = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_DirectionIndicator(rvc_fi_tcl_e8_IPA_DirectionIndicator::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_DirectionIndicator();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_DirectionIndicator& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_IPA_VoiceText : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_IPA_VoiceText();
   enum tenType {
      FI_EN_NO_SOUND = 0UL,
      FI_EN_VOICE_1 = 1UL,
      FI_EN_VOICE_2 = 2UL,
      FI_EN_VOICE_3 = 3UL,
      FI_EN_VOICE_4 = 4UL,
      FI_EN_VOICE_5 = 5UL,
      FI_EN_VOICE_6 = 6UL,
      FI_EN_VOICE_7 = 7UL,
      FI_EN_VOICE_8 = 8UL,
      FI_EN_VOICE_9 = 9UL,
      FI_EN_VOICE_10 = 10UL,
      FI_EN_VOICE_11 = 11UL,
      FI_EN_VOICE_12 = 12UL,
      FI_EN_VOICE_13 = 13UL,
      FI_EN_VOICE_14 = 14UL,
      FI_EN_VOICE_15 = 15UL
   };
   tenType enType;

   rvc_fi_tcl_e8_IPA_VoiceText(rvc_fi_tcl_e8_IPA_VoiceText::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_IPA_VoiceText();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_IPA_VoiceText& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_IPA_Voice_Text_Strings : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_IPA_Voice_Text_Strings();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_IPA_Voice_Text_Strings& operator=(const rvc_fi_tcl_IPA_Voice_Text_Strings& coRef);
   rvc_fi_tcl_IPA_Voice_Text_Strings(const rvc_fi_tcl_IPA_Voice_Text_Strings& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_IPA_VoiceText e8RequestId;
   rvc_fi_tclString szText;
   virtual ~rvc_fi_tcl_IPA_Voice_Text_Strings();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_IPA_Voice_Text_Strings& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_Camera_PowerSupplyStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_Camera_PowerSupplyStatus();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OK = 1UL,
      FI_EN_OPENLOAD = 2UL,
      FI_EN_SHORTCIRCUIT = 3UL,
      FI_EN_DEACTIVATED = 4UL,
      FI_EN_DEACTIVATED_LOWVOLTAGE = 5UL,
      FI_EN_DEACTIVATED_HIGHVOLTAGE = 6UL,
      FI_EN_SHORTTOGROUND = 7UL,
      FI_EN_SHORTTOBATTERY = 8UL,
      FI_EN_NUMBEROFITEMS = 9UL
   };
   tenType enType;

   rvc_fi_tcl_e8_Camera_PowerSupplyStatus(rvc_fi_tcl_e8_Camera_PowerSupplyStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_Camera_PowerSupplyStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_Camera_PowerSupplyStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_Microphone_PowerSupplyStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_Microphone_PowerSupplyStatus();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OK = 1UL,
      FI_EN_OPENLOAD = 2UL,
      FI_EN_SHORTCIRCUIT = 3UL,
      FI_EN_DEACTIVATED = 4UL,
      FI_EN_DEACTIVATED_LOWVOLTAGE = 5UL,
      FI_EN_DEACTIVATED_HIGHVOLTAGE = 6UL,
      FI_EN_SHORTTOGROUND = 7UL,
      FI_EN_SHORTTOBATTERY = 8UL,
      FI_EN_NUMBEROFITEMS = 9UL
   };
   tenType enType;

   rvc_fi_tcl_e8_Microphone_PowerSupplyStatus(rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_Microphone_PowerSupplyStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_Microphone_PowerSupplyStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_Module : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_Module();
   enum tenType {
      FI_EN_RESERVED = 0UL,
      FI_EN_VIDEOCAPTUREUNIT = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_Module(rvc_fi_tcl_e8_RVC_Module::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_Module();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_Module& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_RVC_ReverseSignalStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_RVC_ReverseSignalStatus();
   enum tenType {
      FI_EN_OFF = 0UL,
      FI_EN_ON = 1UL,
      FI_EN_ON_NISSANDEBOUNCED = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_RVC_ReverseSignalStatus(rvc_fi_tcl_e8_RVC_ReverseSignalStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_RVC_ReverseSignalStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_RVC_ReverseSignalStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_ParkAssist_UserConfigurationType : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_ParkAssist_UserConfigurationType();
   enum tenType {
      FI_EN_REAR_VIEW_CAMERA = 0UL,
      FI_EN_GUIDELINE = 1UL,
      FI_EN_STATIC_SIDE_GUIDELINE = 2UL,
      FI_EN_STATIC_CENTRE_GUIDELINE = 3UL,
      FI_EN_DYNAMIC_GUIDELINE = 4UL
   };
   tenType enType;

   rvc_fi_tcl_e8_ParkAssist_UserConfigurationType(rvc_fi_tcl_e8_ParkAssist_UserConfigurationType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_ParkAssist_UserConfigurationType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_ParkAssist_UserConfigurationType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode();
   enum tenType {
      FI_EN_OFF = 0UL,
      FI_EN_ON = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode(rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_ParkAssist_UserConfiguration : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_ParkAssist_UserConfiguration();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_ParkAssist_UserConfiguration& operator=(const rvc_fi_tcl_ParkAssist_UserConfiguration& coRef);
   rvc_fi_tcl_ParkAssist_UserConfiguration(const rvc_fi_tcl_ParkAssist_UserConfiguration& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_ParkAssist_UserConfigurationType e8Type;
   rvc_fi_tcl_e8_ParkAssist_UserConfigurationMode e8Mode;
   virtual ~rvc_fi_tcl_ParkAssist_UserConfiguration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_ParkAssist_UserConfiguration& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_GdlAdj_AdjustButton : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_GdlAdj_AdjustButton();
   enum tenType {
      FI_EN_UP = 0UL,
      FI_EN_DOWN = 1UL,
      FI_EN_LEFT = 2UL,
      FI_EN_RIGHT = 3UL,
      FI_EN_ROTATION_LEFT = 4UL,
      FI_EN_ROTATION_RIGHT = 5UL
   };
   tenType enType;

   rvc_fi_tcl_e8_GdlAdj_AdjustButton(rvc_fi_tcl_e8_GdlAdj_AdjustButton::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_GdlAdj_AdjustButton();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_GdlAdj_AdjustButton& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_GdlAdj_Completion : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_GdlAdj_Completion();
   enum tenType {
      FI_EN_CONFIRM = 0UL,
      FI_EN_REFUSED = 1UL,
      FI_EN_RESET = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_GdlAdj_Completion(rvc_fi_tcl_e8_GdlAdj_Completion::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_GdlAdj_Completion();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_GdlAdj_Completion& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_GdlAdj_Request : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_GdlAdj_Request();
   enum tenType {
      FI_EN_GET_ALL = 0UL,
      FI_EN_GET = 1UL,
      FI_EN_SET = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_GdlAdj_Request(rvc_fi_tcl_e8_GdlAdj_Request::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_GdlAdj_Request();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_GdlAdj_Request& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_GdlAdj_ConfigurationItem : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_GdlAdj_ConfigurationItem();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_ANTICIPATORYLINE = 1UL,
      FI_EN_VEHICLEWIDTH = 2UL,
      FI_EN_WHEELBASE = 3UL,
      FI_EN_REAROVERHANG = 4UL,
      FI_EN_STEERWHEELCALIB_F1 = 5UL,
      FI_EN_STEERWHEELCALIB_F3 = 6UL,
      FI_EN_MAXSTEERWHEELANGLE = 7UL,
      FI_EN_MAXREARWHEELANGLE = 8UL,
      FI_EN_RADIUSTUNINGFACT = 9UL,
      FI_EN_FOURWHEELSTEERING = 10UL,
      FI_EN_TIREANGLETURNSENSE = 11UL,
      FI_EN_TIREANGLEREFERENCEPOINT = 12UL,
      FI_EN_GUIDELINESTYLE = 13UL,
      FI_EN_CAMXOFFSET = 14UL,
      FI_EN_CAMYOFFSET = 15UL,
      FI_EN_CAMZOFFSET = 16UL,
      FI_EN_CAMYAW = 17UL,
      FI_EN_CAMPITCH = 18UL,
      FI_EN_CAMROLL = 19UL,
      FI_EN_FRONTTIRETHRESH = 20UL,
      FI_EN_MINFRONTTIRECHANGE = 21UL,
      FI_EN_REARTIRETHRESH = 22UL,
      FI_EN_MINREARTIRECHANGE = 23UL,
      FI_EN_GUIDELINESTARTDISTANCE = 24UL,
      FI_EN_GUIDELINESIDEMARGIN = 25UL,
      FI_EN_RGBCOLORSTATICSIDENEAR_RED = 26UL,
      FI_EN_RGBCOLORSTATICSIDENEAR_GREEN = 27UL,
      FI_EN_RGBCOLORSTATICSIDENEAR_BLUE = 28UL,
      FI_EN_RGBCOLORSTATICSIDEMID_RED = 29UL,
      FI_EN_RGBCOLORSTATICSIDEMID_GREEN = 30UL,
      FI_EN_RGBCOLORSTATICSIDEMID_BLUE = 31UL,
      FI_EN_RGBCOLORSTATICSIDEFAR_RED = 32UL,
      FI_EN_RGBCOLORSTATICSIDEFAR_GREEN = 33UL,
      FI_EN_RGBCOLORSTATICSIDEFAR_BLUE = 34UL,
      FI_EN_RGBCOLORSTATICCENTER_RED = 35UL,
      FI_EN_RGBCOLORSTATICCENTER_GREEN = 36UL,
      FI_EN_RGBCOLORSTATICCENTER_BLUE = 37UL,
      FI_EN_RGBCOLORDYNAMIC_RED = 38UL,
      FI_EN_RGBCOLORDYNAMIC_GREEN = 39UL,
      FI_EN_RGBCOLORDYNAMIC_BLUE = 40UL,
      FI_EN_LINEWIDTHSTATICSIDENEAR = 41UL,
      FI_EN_LINEWIDTHSTATICSIDEFAR = 42UL,
      FI_EN_LINEWIDTHDYNAMICNEAR = 43UL,
      FI_EN_LINEWIDTHDYNAMICFAR = 44UL,
      FI_EN_LINEWIDTHSTATICCENTERNEAR = 45UL,
      FI_EN_LINEWIDTHSTATICCENTERFAR = 46UL,
      FI_EN_STATICLINELENGTHNEAR = 47UL,
      FI_EN_STATICLINELENGTHMID = 48UL,
      FI_EN_STATICLINELENGHTFAR1 = 49UL,
      FI_EN_STATICLINELENGHTFAR2 = 50UL,
      FI_EN_STATICHORIZLINELENGHT = 51UL,
      FI_EN_LINEWIDTHFACTORNEARHLINE = 52UL,
      FI_EN_TUNINGMETHOD = 53UL,
      FI_EN_TUNINGDELTACAMYAW = 54UL,
      FI_EN_TUNINGDELTACAMPITCH = 55UL,
      FI_EN_TUNINGDELTACAMROLL = 56UL,
      FI_EN_TUNINGDELTAXNEARLEFT = 57UL,
      FI_EN_TUNINGDELTAYNEARLEFT = 58UL,
      FI_EN_TUNINGDELTAXFARLEFT = 59UL,
      FI_EN_TUNINGNEARLINELENGTHLEFT = 60UL,
      FI_EN_TUNINGDELTAXNEARRIGHT = 61UL,
      FI_EN_TUNINGDELTAYNEARRIGHT = 62UL,
      FI_EN_TUNINGDELTAXFARRIGHT = 63UL,
      FI_EN_TUNINGNEARLINELENGHTRIGHT = 64UL,
      FI_EN_TUNINGSWAZEROPOINTOFFSET = 65UL,
      FI_EN_TUNINGRWAZEROPOINTOFFSET = 66UL,
      FI_EN_RVCLENSPARAMETER = 67UL,
      FI_EN_LINEWIDTHFACTORVERTLINES = 68UL
   };
   tenType enType;

   rvc_fi_tcl_e8_GdlAdj_ConfigurationItem(rvc_fi_tcl_e8_GdlAdj_ConfigurationItem::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_GdlAdj_ConfigurationItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_GdlAdj_ConfigurationItem& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_GdlAdj_ConfigParameterData : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_GdlAdj_ConfigParameterData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_GdlAdj_ConfigParameterData& operator=(const rvc_fi_tcl_GdlAdj_ConfigParameterData& coRef);
   rvc_fi_tcl_GdlAdj_ConfigParameterData(const rvc_fi_tcl_GdlAdj_ConfigParameterData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_GdlAdj_ConfigurationItem e8Item;
   tF64 f64Value;
   virtual ~rvc_fi_tcl_GdlAdj_ConfigParameterData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_GdlAdj_ConfigParameterData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_MVCViewingState : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_MVCViewingState();
   enum tenType {
      FI_EN_NOVIEW = 0UL,
      FI_EN_FRONT = 1UL,
      FI_EN_REAR = 2UL,
      FI_EN_RIGHT = 3UL,
      FI_EN_LEFT = 4UL
   };
   tenType enType;

   rvc_fi_tcl_e8_MVCViewingState(rvc_fi_tcl_e8_MVCViewingState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_MVCViewingState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_MVCViewingState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVM_View_Request : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVM_View_Request();
   enum tenType {
      FI_EN_NO_REQUEST = 0UL,
      FI_EN_BIRD_VIEW = 1UL,
      FI_EN_SIDE_VIEW = 2UL,
      FI_EN_OFF = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVM_View_Request(rvc_fi_tcl_e8_AVM_View_Request::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVM_View_Request();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVM_View_Request& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVM_View_Status : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVM_View_Status();
   enum tenType {
      FI_EN_FRONT_BIRD_VIEW = 0UL,
      FI_EN_FRONT_SIDE_VIEW = 1UL,
      FI_EN_REAR_BIRD_VIEW = 2UL,
      FI_EN_REAR_SIDE_VIEW = 3UL,
      FI_EN_ONLY_REAR = 4UL,
      FI_EN_DEALERMODE = 5UL,
      FI_EN_NO_REQUEST = 6UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVM_View_Status(rvc_fi_tcl_e8_AVM_View_Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVM_View_Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVM_View_Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_Tailgate_State : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_Tailgate_State();
   enum tenType {
      FI_EN_UNAVAILABLE = 0UL,
      FI_EN_CLOSED = 1UL,
      FI_EN_OPEN = 2UL,
      FI_EN_UNUSED = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_Tailgate_State(rvc_fi_tcl_e8_Tailgate_State::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_Tailgate_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_Tailgate_State& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_TouchGesture : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_TouchGesture();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_PRESS = 1UL,
      FI_EN_DRAG = 2UL,
      FI_EN_PINCH = 3UL,
      FI_EN_FLICK = 4UL
   };
   tenType enType;

   rvc_fi_tcl_e8_TouchGesture(rvc_fi_tcl_e8_TouchGesture::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_TouchGesture();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_TouchGesture& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_Direction : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_Direction();
   enum tenType {
      FI_EN_INVALID = 0UL,
      FI_EN_NORTH = 1UL,
      FI_EN_NORTH_NORTH_EAST = 2UL,
      FI_EN_NORTH_EAST = 3UL,
      FI_EN_EAST_NORTH_EAST = 4UL,
      FI_EN_EAST = 5UL,
      FI_EN_EAST_SOUTH_EAST = 6UL,
      FI_EN_SOUTH_EAST = 7UL,
      FI_EN_SOUTH_SOUTH_EAST = 8UL,
      FI_EN_SOUTH = 9UL,
      FI_EN_SOUTH_SOUTH_WEST = 10UL,
      FI_EN_SOUTH_WEST = 11UL,
      FI_EN_WEST_SOUTH_WEST = 12UL,
      FI_EN_WEST = 13UL,
      FI_EN_WEST_NORTH_WEST = 14UL,
      FI_EN_NORTH_WEST = 15UL,
      FI_EN_NORTH_NORTH_WEST = 16UL
   };
   tenType enType;

   rvc_fi_tcl_e8_Direction(rvc_fi_tcl_e8_Direction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_Direction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_Direction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_FAP_SettingItem : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_FAP_SettingItem();
   enum tenType {
      FI_EN_SETTINGSBUTTONAVAIL = 0UL,
      FI_EN_SETTINGS_1 = 1UL,
      FI_EN_SETTINGS_2 = 2UL,
      FI_EN_SETTINGS_3 = 3UL,
      FI_EN_SETTINGS_4 = 4UL,
      FI_EN_SETTINGS_5 = 5UL,
      FI_EN_SETTINGS_6 = 6UL,
      FI_EN_SETTINGS_7 = 7UL,
      FI_EN_SETTINGS_8 = 8UL,
      FI_EN_SETTINGS_9 = 9UL,
      FI_EN_SETTINGS_10 = 10UL
   };
   tenType enType;

   rvc_fi_tcl_e8_FAP_SettingItem(rvc_fi_tcl_e8_FAP_SettingItem::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_FAP_SettingItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_FAP_SettingItem& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVM_FAP_SettingState : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVM_FAP_SettingState();
   enum tenType {
      FI_EN_NOT_DISPLAYED = 0UL,
      FI_EN_ON = 1UL,
      FI_EN_OFF = 2UL,
      FI_EN_RESERVED = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVM_FAP_SettingState(rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVM_FAP_SettingState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVM_FAP_SettingState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVM_SettingItem : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVM_SettingItem();
   enum tenType {
      FI_EN_SETTINGSBUTTONAVAIL = 0UL,
      FI_EN_SETTINGS_1 = 1UL,
      FI_EN_SETTINGS_2 = 2UL,
      FI_EN_SETTINGS_3 = 3UL,
      FI_EN_SETTINGS_4 = 4UL,
      FI_EN_SETTINGS_5 = 5UL,
      FI_EN_SETTINGS_6 = 6UL,
      FI_EN_SETTINGS_7 = 7UL,
      FI_EN_SETTINGS_8 = 8UL,
      FI_EN_SETTINGS_9 = 9UL,
      FI_EN_SETTINGS_10 = 10UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVM_SettingItem(rvc_fi_tcl_e8_AVM_SettingItem::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVM_SettingItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVM_SettingItem& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_FAP_Settings_State : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_FAP_Settings_State();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_FAP_Settings_State& operator=(const rvc_fi_tcl_FAP_Settings_State& coRef);
   rvc_fi_tcl_FAP_Settings_State(const rvc_fi_tcl_FAP_Settings_State& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_FAP_SettingItem e8Type;
   rvc_fi_tcl_e8_AVM_FAP_SettingState e8State;
   virtual ~rvc_fi_tcl_FAP_Settings_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_FAP_Settings_State& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVM_Settings_State : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVM_Settings_State();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVM_Settings_State& operator=(const rvc_fi_tcl_AVM_Settings_State& coRef);
   rvc_fi_tcl_AVM_Settings_State(const rvc_fi_tcl_AVM_Settings_State& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_AVM_SettingItem e8Type;
   rvc_fi_tcl_e8_AVM_FAP_SettingState e8State;
   virtual ~rvc_fi_tcl_AVM_Settings_State();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVM_Settings_State& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_GPS_Data : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_GPS_Data();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_GPS_Data& operator=(const rvc_fi_tcl_GPS_Data& coRef);
   rvc_fi_tcl_GPS_Data(const rvc_fi_tcl_GPS_Data& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF32 GPS_latitude;
   tF32 GPS_longitude;
   tF32 GPS_altitude;
   tF32 GPS_azimuth;
   tF32 HDOP;
   tF32 VDOP;
   tU32 numOfSatellites;
   tU32 groundSpeed;
   virtual ~rvc_fi_tcl_GPS_Data();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_GPS_Data& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_MapMatched_Data : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_MapMatched_Data();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_MapMatched_Data& operator=(const rvc_fi_tcl_MapMatched_Data& coRef);
   rvc_fi_tcl_MapMatched_Data(const rvc_fi_tcl_MapMatched_Data& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF32 MM_latitude;
   tF32 MM_longitude;
   tF32 MM_azimuth;
   tF32 MM_accuracy;
   virtual ~rvc_fi_tcl_MapMatched_Data();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_MapMatched_Data& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_Touch_Data : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_Touch_Data();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_Touch_Data& operator=(const rvc_fi_tcl_Touch_Data& coRef);
   rvc_fi_tcl_Touch_Data(const rvc_fi_tcl_Touch_Data& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 x_ordinate;
   tU32 y_ordinate;
   tU32 x1_ordinate;
   tU32 y1_ordinate;
   rvc_fi_tcl_e8_TouchGesture e8Gesture;
   rvc_fi_tcl_e8_Direction e8Direction;
   tU8 SpeedOfMovement;
   virtual ~rvc_fi_tcl_Touch_Data();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_Touch_Data& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_SwitchId : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_SwitchId();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_REVERSE_ON = 1UL,
      FI_EN_REVERSE_OFF = 2UL,
      FI_EN_CAMERA = 3UL,
      FI_EN_HIGHANGLEVIEW = 4UL,
      FI_EN_SIDEVIEW = 5UL,
      FI_EN_WIDEVIEW = 6UL,
      FI_EN_CORNERVIEW = 7UL,
      FI_EN_VIEWSWITCHING = 8UL,
      FI_EN_POINTREGISTRATION = 9UL,
      FI_EN_POINTREMOVING = 10UL,
      FI_EN_PARKINGGUIDE = 11UL,
      FI_EN_START = 12UL,
      FI_EN_END = 13UL,
      FI_EN_HINT = 14UL,
      FI_EN_STARTVOICEOUTPUT = 15UL,
      FI_EN_PARALLELATLEFTSIDE = 16UL,
      FI_EN_PARALLELATRIGHTSIDE = 17UL,
      FI_EN_TANDEMATLEFTSIDE = 18UL,
      FI_EN_TANDEMATRIGHTSIDE = 19UL,
      FI_EN_FINERTUNING = 20UL,
      FI_EN_DECIDEFINERTUNING = 21UL,
      FI_EN_BACK = 22UL,
      FI_EN_MODSONAR_IND_ON = 23UL,
      FI_EN_MODSONAR_IND_OFF = 24UL,
      FI_EN_NARROWMODE_IND_ON = 25UL,
      FI_EN_NARROWMODE_IND_OFF = 26UL,
      FI_EN_MOD_IND_ON = 27UL,
      FI_EN_MOD_IND_OFF = 28UL,
      FI_EN_SONAR_IND_ON = 29UL,
      FI_EN_SONAR_IND_OFF = 30UL,
      FI_EN_RESERVESWITCH = 31UL,
      FI_EN_BCI_IND_OFF = 176UL,
      FI_EN_BCI_IND_ON = 177UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_SwitchId(rvc_fi_tcl_e8_AVMph3_SwitchId::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_SwitchId();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_SwitchId& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_SwitchStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_SwitchStatus();
   enum tenType {
      FI_EN_NORMAL = 0UL,
      FI_EN_MASK = 1UL,
      FI_EN_PUSH = 2UL,
      FI_EN_HIGHLIGHT = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_SwitchStatus(rvc_fi_tcl_e8_AVMph3_SwitchStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_SwitchStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_SwitchStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_SwitchDescriptor : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_SwitchDescriptor();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_SwitchDescriptor& operator=(const rvc_fi_tcl_AVMph3_SwitchDescriptor& coRef);
   rvc_fi_tcl_AVMph3_SwitchDescriptor(const rvc_fi_tcl_AVMph3_SwitchDescriptor& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_AVMph3_SwitchId Id;
   rvc_fi_tcl_e8_AVMph3_SwitchStatus Status;
   virtual ~rvc_fi_tcl_AVMph3_SwitchDescriptor();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_SwitchDescriptor& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_DispStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_DispStatus();
   enum tenType {
      FI_EN_OTHERTHANCAMERA = 0UL,
      FI_EN_CAMERA = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_DispStatus(rvc_fi_tcl_e8_AVMph3_DispStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_DispStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_DispStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_ViewPattern : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_ViewPattern();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_PATTERN1_1 = 1UL,
      FI_EN_PATTERN2_1 = 3UL,
      FI_EN_PATTERN3_1 = 5UL,
      FI_EN_PATTERN3_2 = 6UL,
      FI_EN_PATTERN4_1 = 16UL,
      FI_EN_PATTERN4_2 = 17UL,
      FI_EN_PATTERN5_1 = 18UL,
      FI_EN_PATTERN5_2 = 19UL,
      FI_EN_PATTERN6_1 = 20UL,
      FI_EN_PATTERN6_2 = 21UL,
      FI_EN_PATTERN6_3 = 22UL,
      FI_EN_PATTERN7_1 = 23UL,
      FI_EN_PATTERN7_2 = 24UL,
      FI_EN_PATTERN8_1 = 25UL,
      FI_EN_PATTERN8_2 = 26UL,
      FI_EN_PATTERN9_1 = 27UL,
      FI_EN_PATTERN9_2 = 28UL,
      FI_EN_PATTERN10_1 = 29UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_ViewPattern(rvc_fi_tcl_e8_AVMph3_ViewPattern::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_ViewPattern();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_ViewPattern& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_CautionId : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_CautionId();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_CAUTIONPATTERN1 = 1UL,
      FI_EN_CAUTIONPATTERN2 = 2UL,
      FI_EN_CAUTIONPATTERN3 = 3UL,
      FI_EN_CAUTIONPATTERN4 = 4UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_CautionId(rvc_fi_tcl_e8_AVMph3_CautionId::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_CautionId();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_CautionId& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_IconId : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_IconId();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_REARVIEWICON = 1UL,
      FI_EN_FRONTVIEWICON = 2UL,
      FI_EN_REARWIDEVIEWICON = 3UL,
      FI_EN_FRONTWIDEVIEWICON = 4UL,
      FI_EN_FRONTCORNERVIEWICON = 5UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_IconId(rvc_fi_tcl_e8_AVMph3_IconId::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_IconId();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_IconId& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_BuzzerStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_BuzzerStatus();
   enum tenType {
      FI_EN_BUZZER_OFF = 0UL,
      FI_EN_BUZZER_ON = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_BuzzerStatus(rvc_fi_tcl_e8_AVMph3_BuzzerStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_BuzzerStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_BuzzerStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_ErrorMessage : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_ErrorMessage();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_ONLYBEEP = 1UL,
      FI_EN_SONAROFFPOPUPANDERRORBEEP = 2UL,
      FI_EN_REMOVESONAROFFPOPUP = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_ErrorMessage(rvc_fi_tcl_e8_AVMph3_ErrorMessage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_ErrorMessage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_ErrorMessage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_SettingItemId : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_SettingItemId();
   enum tenType {
      FI_EN_VEHICLE_SPEED = 0UL,
      FI_EN_VIEW_LINE = 1UL,
      FI_EN_PRIORITY_VIEW = 2UL,
      FI_EN_MASK_EMPHASIS = 3UL,
      FI_EN_SIDE_SONAR_DISP = 4UL,
      FI_EN_ADJUST_TOPVIEW = 5UL,
      FI_EN_PRIORITY_WIDE = 6UL,
      FI_EN_AUTO_CAM_POSITION = 7UL,
      FI_EN_VOLUME_VOICE = 8UL,
      FI_EN_SONAR_BUZZER = 9UL,
      FI_EN_REAR_WIDE_VIEW = 10UL,
      FI_EN_SONAR_ON_OFF = 11UL,
      FI_EN_MOD_INTERRUPT = 12UL,
      FI_EN_SONAR_INTERRUPT = 13UL,
      FI_EN_MOD_FUNCTION = 14UL,
      FI_EN_MOD_NIGHT_USE = 15UL,
      FI_EN_BCP_FUNCTION = 16UL,
      FI_EN_FRONT_SONAR_ONLY = 17UL,
      FI_EN_CORNER_SONAR_SENSITIVITY = 18UL,
      FI_EN_BUZZER_SOUND_LEVEL = 19UL,
      FI_EN_SONAR_SENSITIVITY_LEVEL = 20UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_SettingItemId(rvc_fi_tcl_e8_AVMph3_SettingItemId::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_SettingItemId();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_SettingItemId& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_Setting_Item : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_Setting_Item();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_Setting_Item& operator=(const rvc_fi_tcl_AVMph3_Setting_Item& coRef);
   rvc_fi_tcl_AVMph3_Setting_Item(const rvc_fi_tcl_AVMph3_Setting_Item& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_AVMph3_SettingItemId e8ItemId;
   tU8 u8ItemState;
   virtual ~rvc_fi_tcl_AVMph3_Setting_Item();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_Setting_Item& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_SettingOptionState : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_SettingOptionState();
   enum tenType {
      FI_EN_UNAVAIL = 0UL,
      FI_EN_OFF = 1UL,
      FI_EN_ON = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_SettingOptionState(rvc_fi_tcl_e8_AVMph3_SettingOptionState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_SettingOptionState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_SettingOptionState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_Setting_Option : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_Setting_Option();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_Setting_Option& operator=(const rvc_fi_tcl_AVMph3_Setting_Option& coRef);
   rvc_fi_tcl_AVMph3_Setting_Option(const rvc_fi_tcl_AVMph3_Setting_Option& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   rvc_fi_tcl_e8_AVMph3_SettingItemId e8ItemId;
   rvc_fi_tcl_e8_AVMph3_SettingOptionState e8ItemState;
   virtual ~rvc_fi_tcl_AVMph3_Setting_Option();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_Setting_Option& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_Setting_PrioView : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_Setting_PrioView();
   enum tenType {
      FI_EN_UNAVAIL = 0UL,
      FI_EN_OVERHEADVIEW = 1UL,
      FI_EN_SIDEVIEW = 2UL,
      FI_EN_DISPLAYSETTINGSWITCH = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_Setting_PrioView(rvc_fi_tcl_e8_AVMph3_Setting_PrioView::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_Setting_PrioView();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_Setting_PrioView& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_Setting_PrioWide : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_Setting_PrioWide();
   enum tenType {
      FI_EN_UNAVAIL = 0UL,
      FI_EN_WIDEVIEW = 1UL,
      FI_EN_CORNERVIEW = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_Setting_PrioWide(rvc_fi_tcl_e8_AVMph3_Setting_PrioWide::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_Setting_PrioWide();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_Setting_PrioWide& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos();
   enum tenType {
      FI_EN_UNAVAIL = 0UL,
      FI_EN_DISPLAYSWITCH = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos(rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_Setting_AutoCamPos& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl();
   enum tenType {
      FI_EN_LEVEL1 = 0UL,
      FI_EN_LEVEL2 = 1UL,
      FI_EN_LEVEL3 = 2UL,
      FI_EN_LEVEL4 = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl(rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl& operator=(const rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl& coRef);
   rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl(const rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bAvail;
   rvc_fi_tcl_e8_AVMph3_CornerSonarSensLvl e8Level;
   virtual ~rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_Setting_CornerSonarSensLvl& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_SonarSensLvl : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_SonarSensLvl();
   enum tenType {
      FI_EN_UNDEFINED = 0UL,
      FI_EN_FAR = 1UL,
      FI_EN_MIDDLE = 2UL,
      FI_EN_NEAR = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_SonarSensLvl(rvc_fi_tcl_e8_AVMph3_SonarSensLvl::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_SonarSensLvl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_SonarSensLvl& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_Setting_SonarSensLvl : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_Setting_SonarSensLvl();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_Setting_SonarSensLvl& operator=(const rvc_fi_tcl_AVMph3_Setting_SonarSensLvl& coRef);
   rvc_fi_tcl_AVMph3_Setting_SonarSensLvl(const rvc_fi_tcl_AVMph3_Setting_SonarSensLvl& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bAvail;
   rvc_fi_tcl_e8_AVMph3_SonarSensLvl e8Level;
   virtual ~rvc_fi_tcl_AVMph3_Setting_SonarSensLvl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_Setting_SonarSensLvl& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl();
   enum tenType {
      FI_EN_UNDEFINED = 0UL,
      FI_EN_HIGH = 1UL,
      FI_EN_MIDDLE = 2UL,
      FI_EN_LOW = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl(rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl& operator=(const rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl& coRef);
   rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl(const rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bAvail;
   rvc_fi_tcl_e8_AVMph3_BuzzerSoundLvl e8Level;
   virtual ~rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_Setting_BuzzerSoundLvl& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_RequestType : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_RequestType();
   enum tenType {
      FI_EN_GET = 0UL,
      FI_EN_SET = 1UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_RequestType(rvc_fi_tcl_e8_AVMph3_RequestType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_RequestType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_RequestType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_VersionInfoType : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_VersionInfoType();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_HW_VERSION = 1UL,
      FI_EN_SW_VERSION = 2UL,
      FI_EN_HW_SW_VERSION = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_VersionInfoType(rvc_fi_tcl_e8_AVMph3_VersionInfoType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_VersionInfoType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_VersionInfoType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_VersionInfo : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_VersionInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_VersionInfo& operator=(const rvc_fi_tcl_AVMph3_VersionInfo& coRef);
   rvc_fi_tcl_AVMph3_VersionInfo(const rvc_fi_tcl_AVMph3_VersionInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8Upper;
   tU8 u8Middle;
   tU8 u8Lower;
   virtual ~rvc_fi_tcl_AVMph3_VersionInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_VersionInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_ConfigurationStatus : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_ConfigurationStatus();
   enum tenType {
      FI_EN_UNDEFINED = 0UL,
      FI_EN_VALID = 1UL,
      FI_EN_INVALID = 2UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_ConfigurationStatus(rvc_fi_tcl_e8_AVMph3_ConfigurationStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_ConfigurationStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_ConfigurationStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_e8_AVMph3_Variant : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_e8_AVMph3_Variant();
   enum tenType {
      FI_EN_VAR_06AVM = 0UL,
      FI_EN_VAR_11AVM = 1UL,
      FI_EN_VAR_08AVM = 2UL,
      FI_EN_VAR_REARWIDEVIEW = 3UL
   };
   tenType enType;

   rvc_fi_tcl_e8_AVMph3_Variant(rvc_fi_tcl_e8_AVMph3_Variant::tenType newValue)
      :enType(newValue)
   {}

   virtual ~rvc_fi_tcl_e8_AVMph3_Variant();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const rvc_fi_tcl_e8_AVMph3_Variant& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_Unit_ID : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_Unit_ID();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_Unit_ID& operator=(const rvc_fi_tcl_AVMph3_Unit_ID& coRef);
   rvc_fi_tcl_AVMph3_Unit_ID(const rvc_fi_tcl_AVMph3_Unit_ID& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8Unit_ID;
   rvc_fi_tcl_e8_AVMph3_Variant e8Variant;
   tBool bLeftHandDrive;
   tBool bWithSonar;
   virtual ~rvc_fi_tcl_AVMph3_Unit_ID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_Unit_ID& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

class rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors : public rvc_fi_tclVisitableTypeBase
{
public:

   rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors& operator=(const rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors& coRef);
   rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors(const rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8BusOff;
   tU8 u8TxCmfToMaster;
   tU8 u8RxCmfFromSonar;
   tU8 u8TxMpdtToSonar;
   virtual ~rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const rvc_fi_tcl_AVMph3_MCAN_CommunicationErrors& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(rvc_fi_tclVisitorBase& rfoVisitor);
};

#endif
