/************************************************************************
 
 *FILE:           alltypes.h
 
 *SW-COMPONENT:   FI, framework for Functional Interface
 
 *DESCRIPTION:    Interface for all types defined in the different FIs
                  
 *AUTHOR:         Jochen Bruns (CM-DI/ENS41)
 
 *COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 
 *HISTORY:
           $ Log $
 * 
 *    Rev 1.0   Oct 7 2001 17:16:00   bnj2hi
 * Initial revision
 *  
 ************************************************************************/

#ifndef CMB_ALLFITYPES_HEADER
#define CMB_ALLFITYPES_HEADER

#include <vector>

class cmb_fi_tclVisitorBase;

const tU16 CMB_FI_C_U16_ERROR_INTERNALFAILURE = 1;
const tU16 CMB_FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION = 2;
const tU16 CMB_FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA = 3;

class cmb_fi_tclTypeBase : public fi_tclTypeBase
{
public:
  cmb_fi_tclTypeBase() {}
  virtual ~cmb_fi_tclTypeBase() {}
   
  virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& /*oVisitor*/) {}
};

class cmb_fi_tclVisitableTypeBase :public cmb_fi_tclTypeBase
{
public:
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& /*oVisitor*/) {}
};

class cmb_fi_tclU8 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclU8():u8Value(0) {}
   tU8 u8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclU8& roRef) const
      { return (u8Value == roRef.u8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_U8;}
};

class cmb_fi_tclS8 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclS8():s8Value(0) {}
   tS8 s8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclS8& roRef) const
      { return (s8Value == roRef.s8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_S8;}
};

class cmb_fi_tclU16 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclU16():u16Value(0) {}
   tU16 u16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclU16& roRef) const
      { return (u16Value == roRef.u16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_U16;}
};

class cmb_fi_tclS16 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclS16():s16Value(0) {}
   tS16 s16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclS16& roRef) const
      { return (s16Value == roRef.s16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_S16;}
};

class cmb_fi_tclU24 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclU24():u24Value(0) {}
   tU24 u24Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclU24& roRef) const
      { return (u24Value == roRef.u24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_U24;}
};

class cmb_fi_tclS24 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclS24():s24Value(0) {}
   tS24 s24Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclS24& roRef) const
      { return (s24Value == roRef.s24Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_S24;}
};

class cmb_fi_tclU32 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclU32():u32Value(0) {}
   tU32 u32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclU32& roRef) const
      { return (u32Value == roRef.u32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_U32;}
};

class cmb_fi_tclS32 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclS32():s32Value(0) {}
   tS32 s32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclS32& roRef) const
      { return (s32Value == roRef.s32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_S32;}
};

class cmb_fi_tclU64 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclU64():u64Value(0) {}
   tU64 u64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclU64& roRef) const
      { return (u64Value == roRef.u64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_U64;}
};

class cmb_fi_tclS64 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclS64():s64Value(0) {}
   tS64 s64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclS64& roRef) const
      { return (s64Value == roRef.s64Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_S64;}
};

class cmb_fi_tclF32 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclF32():f32Value(0) {}
   tFloat f32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclF32& roRef) const
      { return (f32Value == roRef.f32Value);  } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_F32;}
};

class cmb_fi_tclF64 : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclF64():f64Value(0) {}
   tDouble f64Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclF64& roRef) const
      { return (f64Value == roRef.f64Value); } //lint !e777
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_F64;}
};

class cmb_fi_tclBool : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclBool():bValue(0) {}
   tBool bValue;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const cmb_fi_tclBool& roRef) const
      { return (bValue == roRef.bValue); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_BOOL;}
};

class cmb_fi_tclString : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tclString(const cmb_fi_tclString& coRef);
   cmb_fi_tclString& operator=(const cmb_fi_tclString& coRef);
   virtual ~cmb_fi_tclString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const cmb_fi_tclString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   cmb_fi_tclString& operator=(tCString coszString); // ADDED
   cmb_fi_tclString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8,
      FI_EN_ISO8859_1M
   };
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const cmb_fi_tclString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};

class cmb_fi_tclList : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclList():u32ListSize(0) {}
   tU32 u32ListSize;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const cmb_fi_tclList& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oIn);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_LIST;}
};

class cmb_fi_tclMultiLanguageString : public cmb_fi_tclVisitableTypeBase
{
public:
   cmb_fi_tclMultiLanguageString():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tclMultiLanguageString(const cmb_fi_tclMultiLanguageString& coRef);
   cmb_fi_tclMultiLanguageString& operator=(const cmb_fi_tclMultiLanguageString& coRef);
   virtual ~cmb_fi_tclMultiLanguageString();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const cmb_fi_tclMultiLanguageString& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(cmb_fi_tclVisitorBase& oVisitor);
#ifndef FI_DO_NOT_USE_TSTRING
   cmb_fi_tclMultiLanguageString& operator=(tCString coszString); // ADDED
   cmb_fi_tclMultiLanguageString(tCString coszInit); //ADDED
   tBool operator==(tCString coszString) const; // ADDED
   operator const tChar* () const; // ADDED
   operator tString&(); // ADDED
   tBool operator!=(intptr_t iNULL) const; // ADDED
   tChar* operator+(int iOffset); // ADDED
   tChar* operator+(tU8  u8Offset); // ADDED
   tChar* operator+(tU32 u32Offset); // ADDED
   tChar& operator[](tU16 u16Index); //ADDED
   tChar& operator[](tU32 u32Index); //ADDED
   tChar& operator[](tS32 s32Index); //ADDED
#endif
   enum tenCharSet {
      FI_EN_UNDEFINED = 0,
      FI_EN_ISO8859_15M,
      FI_EN_UTF8
   };
   virtual tS32 s32GetTypeId() const { return cmb_typesfi_tclToken::EN_STRING;}
   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   tBool bSet(const cmb_fi_tclMultiLanguageString& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet);
   tString szGet(tenCharSet enCharSet) const;
};
class cmb_fi_tcl_BinContainer;
class cmb_fi_tcl_e8_AudioChannelAdvice;
class cmb_fi_tcl_e8_VoiceCategory;
class cmb_fi_tcl_e8_CatStatus;
class cmb_fi_tcl_AudioChannelAdvice;
class cmb_fi_tcl_b8_AudioOutputLocation;
class cmb_fi_tcl_b8_AudioFormat;
class cmb_fi_tcl_InstancesPerCategory;
class cmb_fi_tcl_InstancesPerCategoryList;
class cmb_fi_tcl_b16_RequestAddInfo;
class cmb_fi_tcl_ChannelRequestStatus;
class cmb_fi_tcl_ChannelRequestStatusList;
class cmb_fi_tcl_e8_OdometerStatus;
class cmb_fi_tcl_e8_OdometerDirection;
class cmb_fi_tcl_e8_GyroError;
class cmb_fi_tcl_e8_GpsManufacturer;
class cmb_fi_tcl_e8_GpsHardwareVersion;
class cmb_fi_tcl_e8_GpsAntennaStatus;
class cmb_fi_tcl_e8_GpsFrontEndStatus;
class cmb_fi_tcl_e8_GpsFrontEndDataPattern;
class cmb_fi_tcl_b8_GpsFixStatus;
class cmb_fi_tcl_b8_GnssSatSys;
class cmb_fi_tcl_CovarianceMatrixUpperTriangle;
class cmb_fi_tcl_OdometerData;
class cmb_fi_tcl_e16_GyroStatus;
class cmb_fi_tcl_e16_AccStatus;
class cmb_fi_tcl_GyroData;
class cmb_fi_tcl_b16_GpsChannelStatus;
class cmb_fi_tcl_GpsTestAdditionalData;
class cmb_fi_tcl_GpsBigConstData;
class cmb_fi_tcl_b32_GpsValidFlags;
class cmb_fi_tcl_GpsExtBigConstData;
class cmb_fi_tcl_b32_MapMatchingHint;
class cmb_fi_tcl_GpsBigVarData;
class cmb_fi_tcl_GpsBigdata;
class cmb_fi_tcl_GpsExtBigData;
class cmb_fi_tcl_GpsCommunicationProtocol;
class cmb_fi_tcl_SetData;
class cmb_fi_tcl_TestModeData;
class cmb_fi_tcl_XoCompensation;
class cmb_fi_tcl_VectorLLA;
class cmb_fi_tcl_VectorNEU;
class cmb_fi_tcl_CovarianceMatrixDiag;
class cmb_fi_tcl_GpsHints;
class cmb_fi_tcl_GpsExtHints;
class cmb_fi_tcl_GpsTightCouplingChannelData;
class cmb_fi_tcl_GpsTightCouplingCommonData;
class cmb_fi_tcl_GpsTightCouplingData;
class cmb_fi_tcl_e8_AbsStatus;
class cmb_fi_tcl_e8_AbsDirection;
class cmb_fi_tcl_AbsData;
class cmb_fi_tcl_e8_SteeringStatus;
class cmb_fi_tcl_SteeringData;
class cmb_fi_tcl_e8_OdometerRangeStatus;
class cmb_fi_tcl_GpsExtTestmodeStart;
class cmb_fi_tcl_GpsExtTestmodeResult;
class cmb_fi_tcl_e8_DefSetGroups;
class cmb_fi_tcl_e8_DefSetMode;
class cmb_fi_tcl_SensorHwInfo;
class cmb_fi_tcl_3dMountAngles;
class cmb_fi_tcl_3dGyroHwInfo;
class cmb_fi_tcl_3dAccHwInfo;
class cmb_fi_tcl_3dGyroData;
class cmb_fi_tcl_3dAccData;
class cmb_fi_tcl_TimeDate;
class cmb_fi_tcl_e8_timeMode;
class cmb_fi_tcl_Reserved;
class cmb_fi_tcl_s16_PerThousand;
class cmb_fi_tcl_s32_Pascal;
class cmb_fi_tcl_u8_Percent;
class cmb_fi_tcl_s32_Centimeter;
class cmb_fi_tcl_s16_CentiDegree;
class cmb_fi_tcl_s16_KmPerHour;
class cmb_fi_tcl_u16_RPM;
class cmb_fi_tcl_s16_CentimeterSecond;
class cmb_fi_tcl_s16_Meter;
class cmb_fi_tcl_s32_Meter;
class cmb_fi_tcl_s32_Yards;
class cmb_fi_tcl_s32_Inch;
class cmb_fi_tcl_s32_Kilogram;
class cmb_fi_tcl_s16_DirectionRelative_CentiDegree;
class cmb_fi_tcl_u16_MilliSecond;
class cmb_fi_tcl_s32_MilliSecond;
class cmb_fi_tcl_s32_Longitude;
class cmb_fi_tcl_s32_Latitude;
class cmb_fi_tcl_PositionWGS84;
class cmb_fi_tcl_s16_MilesPerHour;
class cmb_fi_tcl_s32_Pound;
class cmb_fi_tcl_s16_DeciKelvin;
class cmb_fi_tcl_s16_CentiDegreeSecond;
class cmb_fi_tcl_u32_MilliLux;
class cmb_fi_tcl_u32_Microliter;
class cmb_fi_tcl_s16_CentimeterSquareSecond;
class cmb_fi_tcl_s16_CurvatureValue;
class cmb_fi_tcl_s32_CurvatureChange;
class cmb_fi_tcl_e16_BaseTrafficSign;
class cmb_fi_tcl_e8_BaseTrafficSignCategory;
class cmb_fi_tcl_e8_SpeedSign;
class cmb_fi_tcl_e16_WarningSign;
class cmb_fi_tcl_e16_RegulatorySign;
class cmb_fi_tcl_e16_AdditionalTrafficSign;
class cmb_fi_tcl_TimeDependentString;
class cmb_fi_tcl_b8_LaneDependentValidity;
class cmb_fi_tcl_e8_TrafficSignQuantifier;
class cmb_fi_tcl_AdditionalTrafficSign;
class cmb_fi_tcl_TrafficSignElement;
class cmb_fi_tcl_VeSe_WheelSlip;
class cmb_fi_tcl_VeSe_WheelBrakePressure;
class cmb_fi_tcl_VeSe_WheelAirPressure;
class cmb_fi_tcl_e8_VeSe_Activation;
class cmb_fi_tcl_e8_VeSe_DoorType;
class cmb_fi_tcl_e8_VeSe_WindowType;
class cmb_fi_tcl_e8_VeSe_LineColor;
class cmb_fi_tcl_VeSe_DoorStatus;
class cmb_fi_tcl_VeSe_WindowStatus;
class cmb_fi_tcl_e8_VeSe_SeatOccupation;
class cmb_fi_tcl_e8_VeSe_DirectionIndicator;
class cmb_fi_tcl_e8_VeSe_GearNum;
class cmb_fi_tcl_b8_VeSe_GearBoxState;
class cmb_fi_tcl_e8_VeSe_CCType;
class cmb_fi_tcl_e8_VeSe_CCState;
class cmb_fi_tcl_e8_VeSe_Ignition;
class cmb_fi_tcl_b32_VeSe_BrakeControlMode;
class cmb_fi_tcl_b16_VeSe_CentralLocking;
class cmb_fi_tcl_e8_VeSe_RotationIrregularity;
class cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff;
class cmb_fi_tcl_e8_VeSe_ConverterClutch;
class cmb_fi_tcl_e8_VeSe_GearboxSportSwitch;
class cmb_fi_tcl_e8_VeSe_HeadLights;
class cmb_fi_tcl_VeSe_SeatStatus;
class cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition;
class cmb_fi_tcl_e8_VeSe_KeyValidity;
class cmb_fi_tcl_e8_VeSe_KeyType;
class cmb_fi_tcl_b16_VeSe_KeyLocation;
class cmb_fi_tcl_b16_VeSe_EngineStatus;
class cmb_fi_tcl_e8_VeSe_ReductionRPM;
class cmb_fi_tcl_b16_VeSe_ActiveSeats;
class cmb_fi_tcl_e8_VeSe_SideIndicator;
class cmb_fi_tcl_VeSe_LineInfo;
class cmb_fi_tcl_e8_VeSe_ObjectType;
class cmb_fi_tcl_s16_VeSe_Torque;
class cmb_fi_tcl_VeSe_DetectedObject;
class cmb_fi_tcl_VeSe_TSClassificationQuality;
class cmb_fi_tcl_VeSe_TSQuality;
class cmb_fi_tcl_VeSe_DetectedTrafficSign;
class cmb_fi_tcl_s32_Ws;
class cmb_fi_tcl_u32_Ws;
class cmb_fi_tcl_u16_W;
class cmb_fi_tcl_s32_MilliVolt;
class cmb_fi_tcl_s32_MilliAmpere;
class cmb_fi_tcl_u16_min;
class cmb_fi_tcl_e8_EOperationMode;
class cmb_fi_tcl_b8_CouplingMode;
class cmb_fi_tcl_EnergyConsumption;
class cmb_fi_tcl_PowerDemandAtSpeed;
class cmb_fi_tcl_ElectricalMachine;
class cmb_fi_tcl_e8_VeSe_K0Status;
class cmb_fi_tcl_VeSe_VoltageParams;
class cmb_fi_tcl_VeSe_CurrentParams;
class cmb_fi_tcl_b16_VeSe_Environment;
class cmb_fi_tcl_VeSe_IPPCPlusData;
class cmb_fi_tcl_u8_SD_PackageID;
class cmb_fi_tcl_SD_Package;
class cmb_fi_tcl_e8_DownloadStatus;
class cmb_fi_tcl_SD_PackageLoadStatus;
class cmb_fi_tcl_e8_SD_Damage;
class cmb_fi_tcl_e8_SD_Format;
class cmb_fi_tcl_e8_SpeechCategory;
class cmb_fi_tcl_b16_CategoryAddInfo;
class cmb_fi_tcl_e8_SpeechSequence;
class cmb_fi_tcl_SD_Phrase;
class cmb_fi_tcl_SD_Sentence;
class cmb_fi_tcl_SD_SentenceAndParam;
class cmb_fi_tcl_u16_SD_SampleFrequency;
class cmb_fi_tcl_SD_Settings;
class cmb_fi_tcl_u8_SD_Size;
class cmb_fi_tcl_SD_Reference;
class cmb_fi_tcl_e8_StopVOPriority;
class cmb_fi_tcl_b8_PhraseStatus;
class cmb_fi_tcl_TTSPhraseInfo;
class cmb_fi_tcl_BinContainer : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_BinContainer();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_BinContainer& operator=(const cmb_fi_tcl_BinContainer& coRef);
   cmb_fi_tcl_BinContainer(const cmb_fi_tcl_BinContainer& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<tU8, std::allocator<tU8> > ContainerDataList;
   virtual ~cmb_fi_tcl_BinContainer();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_BinContainer& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_AudioChannelAdvice : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_AudioChannelAdvice();
   enum tenType {
      FI_EN_START = 0UL,
      FI_EN_STOPIMMEDIATELY = 1UL,
      FI_EN_STOPAFTERWORD = 2UL,
      FI_EN_STOPAFTERSENTENCE = 3UL,
      FI_EN_PAUSEIMMEDIATELY = 4UL,
      FI_EN_PAUSEAFTERWORD = 5UL,
      FI_EN_PAUSEAFTERSENTENCE = 6UL,
      FI_EN_RESUME = 7UL
   };
   tenType enType;

   cmb_fi_tcl_e8_AudioChannelAdvice(cmb_fi_tcl_e8_AudioChannelAdvice::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_AudioChannelAdvice();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_AudioChannelAdvice& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VoiceCategory : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VoiceCategory();
   enum tenType {
      FI_EN_DRIVINGRECOMMENDATION = 0UL,
      FI_EN_GENERALINFOAHEAD = 1UL,
      FI_EN_GENERALINFOROUTE = 2UL,
      FI_EN_GENERALINFOHORIZON = 3UL,
      FI_EN_ROUTEINFO = 4UL,
      FI_EN_DESTINFO = 5UL,
      FI_EN_HMIMESSAGE = 6UL,
      FI_EN_DEBUGMESSAGE = 7UL,
      FI_EN_DIALOGPROMPT = 8UL,
      FI_EN_EMAIL = 9UL,
      FI_EN_SMS = 10UL,
      FI_EN_MICAUDIO = 11UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VoiceCategory(cmb_fi_tcl_e8_VoiceCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VoiceCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VoiceCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_CatStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_CatStatus();
   enum tenType {
      FI_EN_SILENT = 0UL,
      FI_EN_NEEDCHANNEL = 1UL,
      FI_EN_PLAYING = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_CatStatus(cmb_fi_tcl_e8_CatStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_CatStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_CatStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_AudioChannelAdvice : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_AudioChannelAdvice();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_AudioChannelAdvice& operator=(const cmb_fi_tcl_AudioChannelAdvice& coRef);
   cmb_fi_tcl_AudioChannelAdvice(const cmb_fi_tcl_AudioChannelAdvice& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_e8_AudioChannelAdvice ChannelAdvice;
   tU8 Index;
   cmb_fi_tclString Device;
   virtual ~cmb_fi_tcl_AudioChannelAdvice();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_AudioChannelAdvice& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_AudioOutputLocation : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_AudioOutputLocation();

   enum {
      FI_C_U8_BIT_STANDARD = 0x1UL,
      FI_C_U8_BIT_FRONTLEFT = 0x2UL,
      FI_C_U8_BIT_FRONTRIGHT = 0x4UL,
      FI_C_U8_BIT_BACKRIGHT = 0x8UL,
      FI_C_U8_BIT_BACKLEFT = 0x10UL
   };
   tBool bStandard() const {return (u8Value & FI_C_U8_BIT_STANDARD) != 0; }
   tVoid vSetStandard(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_STANDARD):(u8Value & ~(tU8)FI_C_U8_BIT_STANDARD)); }
   tBool bFrontLeft() const {return (u8Value & FI_C_U8_BIT_FRONTLEFT) != 0; }
   tVoid vSetFrontLeft(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_FRONTLEFT):(u8Value & ~(tU8)FI_C_U8_BIT_FRONTLEFT)); }
   tBool bFrontRight() const {return (u8Value & FI_C_U8_BIT_FRONTRIGHT) != 0; }
   tVoid vSetFrontRight(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_FRONTRIGHT):(u8Value & ~(tU8)FI_C_U8_BIT_FRONTRIGHT)); }
   tBool bBackRight() const {return (u8Value & FI_C_U8_BIT_BACKRIGHT) != 0; }
   tVoid vSetBackRight(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_BACKRIGHT):(u8Value & ~(tU8)FI_C_U8_BIT_BACKRIGHT)); }
   tBool bBackLeft() const {return (u8Value & FI_C_U8_BIT_BACKLEFT) != 0; }
   tVoid vSetBackLeft(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_BACKLEFT):(u8Value & ~(tU8)FI_C_U8_BIT_BACKLEFT)); }
   struct rBitValues {
      tBool Standard:1;
      tBool FrontLeft:1;
      tBool FrontRight:1;
      tBool BackRight:1;
      tBool BackLeft:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_AudioOutputLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_AudioOutputLocation& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_AudioFormat : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_AudioFormat();

   enum {
      FI_C_U8_BIT_PCM = 0x1UL,
      FI_C_U8_BIT_AMR = 0x2UL,
      FI_C_U8_BIT_AMR_WB = 0x4UL,
      FI_C_U8_BIT_MP3 = 0x8UL
   };
   tBool bPCM() const {return (u8Value & FI_C_U8_BIT_PCM) != 0; }
   tVoid vSetPCM(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_PCM):(u8Value & ~(tU8)FI_C_U8_BIT_PCM)); }
   tBool bAMR() const {return (u8Value & FI_C_U8_BIT_AMR) != 0; }
   tVoid vSetAMR(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_AMR):(u8Value & ~(tU8)FI_C_U8_BIT_AMR)); }
   tBool bAMR_WB() const {return (u8Value & FI_C_U8_BIT_AMR_WB) != 0; }
   tVoid vSetAMR_WB(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_AMR_WB):(u8Value & ~(tU8)FI_C_U8_BIT_AMR_WB)); }
   tBool bMP3() const {return (u8Value & FI_C_U8_BIT_MP3) != 0; }
   tVoid vSetMP3(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_MP3):(u8Value & ~(tU8)FI_C_U8_BIT_MP3)); }
   struct rBitValues {
      tBool PCM:1;
      tBool AMR:1;
      tBool AMR_WB:1;
      tBool MP3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_AudioFormat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_AudioFormat& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_InstancesPerCategory : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_InstancesPerCategory();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_InstancesPerCategory& operator=(const cmb_fi_tcl_InstancesPerCategory& coRef);
   cmb_fi_tcl_InstancesPerCategory(const cmb_fi_tcl_InstancesPerCategory& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_e8_VoiceCategory VoiceCategory;
   tU8 NumberOfInstances;
   virtual ~cmb_fi_tcl_InstancesPerCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_InstancesPerCategory& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_InstancesPerCategoryList : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_InstancesPerCategoryList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_InstancesPerCategoryList& operator=(const cmb_fi_tcl_InstancesPerCategoryList& coRef);
   cmb_fi_tcl_InstancesPerCategoryList(const cmb_fi_tcl_InstancesPerCategoryList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<cmb_fi_tcl_InstancesPerCategory, std::allocator<cmb_fi_tcl_InstancesPerCategory> > InstancesPerCategory;
   tU8 MaxAmountInstances;
   virtual ~cmb_fi_tcl_InstancesPerCategoryList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_InstancesPerCategoryList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_RequestAddInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_RequestAddInfo();

   enum {
      FI_C_U16_BIT_USER = 0x1UL,
      FI_C_U16_BIT_LONG = 0x2UL,
      FI_C_U16_BIT_SHORT = 0x4UL
   };
   tBool bUser() const {return (u16Value & FI_C_U16_BIT_USER) != 0; }
   tVoid vSetUser(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_USER):(u16Value & ~(tU16)FI_C_U16_BIT_USER)); }
   tBool bLong() const {return (u16Value & FI_C_U16_BIT_LONG) != 0; }
   tVoid vSetLong(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_LONG):(u16Value & ~(tU16)FI_C_U16_BIT_LONG)); }
   tBool bShort() const {return (u16Value & FI_C_U16_BIT_SHORT) != 0; }
   tVoid vSetShort(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_SHORT):(u16Value & ~(tU16)FI_C_U16_BIT_SHORT)); }
   struct rBitValues {
      tBool User:1;
      tBool Long:1;
      tBool Short:1;
      tBool dummy3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_RequestAddInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_RequestAddInfo& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_ChannelRequestStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_ChannelRequestStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_ChannelRequestStatus& operator=(const cmb_fi_tcl_ChannelRequestStatus& coRef);
   cmb_fi_tcl_ChannelRequestStatus(const cmb_fi_tcl_ChannelRequestStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_e8_VoiceCategory Category;
   tU8 Index;
   cmb_fi_tcl_e8_CatStatus Status;
   cmb_fi_tcl_b8_AudioFormat AudioFormat;
   cmb_fi_tcl_b8_AudioOutputLocation AudioOutputLocation;
   cmb_fi_tcl_b16_RequestAddInfo RequestAddInfo;
   virtual ~cmb_fi_tcl_ChannelRequestStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_ChannelRequestStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_ChannelRequestStatusList : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_ChannelRequestStatusList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_ChannelRequestStatusList& operator=(const cmb_fi_tcl_ChannelRequestStatusList& coRef);
   cmb_fi_tcl_ChannelRequestStatusList(const cmb_fi_tcl_ChannelRequestStatusList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<cmb_fi_tcl_ChannelRequestStatus, std::allocator<cmb_fi_tcl_ChannelRequestStatus> > ChannelRequestStatusList;
   virtual ~cmb_fi_tcl_ChannelRequestStatusList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_ChannelRequestStatusList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_OdometerStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_OdometerStatus();
   enum tenType {
      FI_EN_ODOMSTATE_CONNECTED_NORMAL = 0UL,
      FI_EN_ODOMSTATE_CONNECTED_NO_CALIBRATION = 248UL,
      FI_EN_ODOMSTATE_CONNECTED_DATA_INVALID = 249UL,
      FI_EN_ODOMSTATE_CONNECTED_INTERNALERROR = 250UL,
      FI_EN_ODOMSTATE_CONNECTED_NOINFO = 251UL,
      FI_EN_ODOMSTATE_CONNECTED_NOMOVEMENT = 252UL,
      FI_EN_ODOMSTATE_CONNECTED_ERROR = 253UL,
      FI_EN_ODOMSTATE_DISCONNECTED = 254UL,
      FI_EN_ODOMSTATE_UNKNOWN = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_OdometerStatus(cmb_fi_tcl_e8_OdometerStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_OdometerStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_OdometerStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_OdometerDirection : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_OdometerDirection();
   enum tenType {
      FI_EN_ODOMDIR_UNKNOWN = 0UL,
      FI_EN_ODOMDIR_FORWARD = 1UL,
      FI_EN_ODOMDIR_REVERSE = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_OdometerDirection(cmb_fi_tcl_e8_OdometerDirection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_OdometerDirection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_OdometerDirection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_GyroError : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_GyroError();
   enum tenType {
      FI_EN_GYRO_ERROR_UNKNOWN = 0UL,
      FI_EN_GYRO_ERROR_ACTIVE = 1UL,
      FI_EN_GYRO_ERROR_INACTIVE = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_GyroError(cmb_fi_tcl_e8_GyroError::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_GyroError();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_GyroError& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_GpsManufacturer : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_GpsManufacturer();
   enum tenType {
      FI_EN_GPS_MAN_UNKNOWN = 0UL,
      FI_EN_GPS_MAN_BP = 1UL,
      FI_EN_GPS_MAN_JRC = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_GpsManufacturer(cmb_fi_tcl_e8_GpsManufacturer::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_GpsManufacturer();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_GpsManufacturer& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_GpsHardwareVersion : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_GpsHardwareVersion();
   enum tenType {
      FI_EN_GPS_HW_GPS1_4 = 0UL,
      FI_EN_GPS_HW_GPS5 = 1UL,
      FI_EN_GPS_HW_GPS6_7 = 2UL,
      FI_EN_GPS_HW_GPS_ELENA = 10UL,
      FI_EN_GPS_HW_BPGPS_ARION = 11UL,
      FI_EN_GPS_HW_BPGPS_DRAGON_ES1 = 12UL,
      FI_EN_GPS_HW_BPGPS_DRAGON = 13UL,
      FI_EN_GPS_HW_NMEA_GPS = 14UL,
      FI_EN_GPS_HW_IPHONE_GPS = 15UL,
      FI_EN_GPS_HW_UNKNOWN = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_GpsHardwareVersion(cmb_fi_tcl_e8_GpsHardwareVersion::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_GpsHardwareVersion();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_GpsHardwareVersion& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_GpsAntennaStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_GpsAntennaStatus();
   enum tenType {
      FI_EN_GPS_ANTENNA_NORMAL = 0UL,
      FI_EN_GPS_ANTENNA_OPEN = 1UL,
      FI_EN_GPS_ANTENNA_SHORT = 2UL,
      FI_EN_GPS_ANTENNA_BURNING = 3UL,
      FI_EN_GPS_ANTENNA_BATTERY_SHORT = 4UL,
      FI_EN_GPS_ANTENNA_UNKNOWN = 5UL
   };
   tenType enType;

   cmb_fi_tcl_e8_GpsAntennaStatus(cmb_fi_tcl_e8_GpsAntennaStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_GpsAntennaStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_GpsAntennaStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_GpsFrontEndStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_GpsFrontEndStatus();
   enum tenType {
      FI_EN_GPS_FRONT_END_OFF = 0UL,
      FI_EN_GPS_FRONT_END_ON = 1UL,
      FI_EN_GPS_FRONT_END_ERROR = 2UL,
      FI_EN_GPS_FRONT_END_UNKNOWN = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_GpsFrontEndStatus(cmb_fi_tcl_e8_GpsFrontEndStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_GpsFrontEndStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_GpsFrontEndStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_GpsFrontEndDataPattern : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_GpsFrontEndDataPattern();
   enum tenType {
      FI_EN_GPS_DATA_PATTERN_OFF = 0UL,
      FI_EN_GPS_DATA_PATTERN_0000 = 4UL,
      FI_EN_GPS_DATA_PATTERN_0101 = 5UL,
      FI_EN_GPS_DATA_PATTERN_1010 = 6UL,
      FI_EN_GPS_DATA_PATTERN_1111 = 7UL
   };
   tenType enType;

   cmb_fi_tcl_e8_GpsFrontEndDataPattern(cmb_fi_tcl_e8_GpsFrontEndDataPattern::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_GpsFrontEndDataPattern();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_GpsFrontEndDataPattern& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_GpsFixStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_GpsFixStatus();

   enum {
      FI_C_U8_BIT_GPS_FIX_TIME = 0x1UL,
      FI_C_U8_BIT_GPS_FIX_2D = 0x2UL,
      FI_C_U8_BIT_GPS_FIX_3D = 0x4UL,
      FI_C_U8_BIT_GPS_FIX_COLDSTART = 0x80UL
   };
   tBool bGPS_FIX_TIME() const {return (u8Value & FI_C_U8_BIT_GPS_FIX_TIME) != 0; }
   tVoid vSetGPS_FIX_TIME(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_GPS_FIX_TIME):(u8Value & ~(tU8)FI_C_U8_BIT_GPS_FIX_TIME)); }
   tBool bGPS_FIX_2D() const {return (u8Value & FI_C_U8_BIT_GPS_FIX_2D) != 0; }
   tVoid vSetGPS_FIX_2D(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_GPS_FIX_2D):(u8Value & ~(tU8)FI_C_U8_BIT_GPS_FIX_2D)); }
   tBool bGPS_FIX_3D() const {return (u8Value & FI_C_U8_BIT_GPS_FIX_3D) != 0; }
   tVoid vSetGPS_FIX_3D(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_GPS_FIX_3D):(u8Value & ~(tU8)FI_C_U8_BIT_GPS_FIX_3D)); }
   tBool bGPS_FIX_COLDSTART() const {return (u8Value & FI_C_U8_BIT_GPS_FIX_COLDSTART) != 0; }
   tVoid vSetGPS_FIX_COLDSTART(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_GPS_FIX_COLDSTART):(u8Value & ~(tU8)FI_C_U8_BIT_GPS_FIX_COLDSTART)); }
   struct rBitValues {
      tBool GPS_FIX_TIME:1;
      tBool GPS_FIX_2D:1;
      tBool GPS_FIX_3D:1;
      tBool dummy3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool GPS_FIX_COLDSTART:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_GpsFixStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_GpsFixStatus& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_GnssSatSys : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_GnssSatSys();

   enum {
      FI_C_U8_BIT_GNSS_SATSYS_GPS = 0x1UL,
      FI_C_U8_BIT_GNSS_SATSYS_GLONASS = 0x2UL
   };
   tBool bGNSS_SATSYS_GPS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_GPS) != 0; }
   tVoid vSetGNSS_SATSYS_GPS(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_GPS):(u8Value & ~(tU8)FI_C_U8_BIT_GNSS_SATSYS_GPS)); }
   tBool bGNSS_SATSYS_GLONASS() const {return (u8Value & FI_C_U8_BIT_GNSS_SATSYS_GLONASS) != 0; }
   tVoid vSetGNSS_SATSYS_GLONASS(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_GNSS_SATSYS_GLONASS):(u8Value & ~(tU8)FI_C_U8_BIT_GNSS_SATSYS_GLONASS)); }
   struct rBitValues {
      tBool GNSS_SATSYS_GPS:1;
      tBool GNSS_SATSYS_GLONASS:1;
      tBool dummy2:1;
      tBool dummy3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_GnssSatSys();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_GnssSatSys& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_CovarianceMatrixUpperTriangle : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_CovarianceMatrixUpperTriangle();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_CovarianceMatrixUpperTriangle& operator=(const cmb_fi_tcl_CovarianceMatrixUpperTriangle& coRef);
   cmb_fi_tcl_CovarianceMatrixUpperTriangle(const cmb_fi_tcl_CovarianceMatrixUpperTriangle& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF32 Elem0;
   tF32 Elem4;
   tF32 Elem5;
   tF32 Elem8;
   tF32 Elem9;
   tF32 Elem10;
   virtual ~cmb_fi_tcl_CovarianceMatrixUpperTriangle();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_CovarianceMatrixUpperTriangle& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_OdometerData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_OdometerData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_OdometerData& operator=(const cmb_fi_tcl_OdometerData& coRef);
   cmb_fi_tcl_OdometerData(const cmb_fi_tcl_OdometerData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   tU16 OdometerCount;
   cmb_fi_tcl_e8_OdometerStatus OdometerStatus;
   cmb_fi_tcl_e8_OdometerDirection OdometerCount_Direction;
   virtual ~cmb_fi_tcl_OdometerData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_OdometerData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e16_GyroStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e16_GyroStatus();
   enum tenType {
      FI_EN_GYROSTATE_CONNECTED_NORMAL = 0UL,
      FI_EN_GYROSTATE_SENSOR_NOT_PRESENT = 65533UL,
      FI_EN_GYROSTATE_CONNECTED_DATA_INVALID = 65534UL,
      FI_EN_GYROSTATE_CONNECTED_INTERNALERROR = 65535UL
   };
   tenType enType;

   cmb_fi_tcl_e16_GyroStatus(cmb_fi_tcl_e16_GyroStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e16_GyroStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e16_GyroStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e16_AccStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e16_AccStatus();
   enum tenType {
      FI_EN_ACCSTATE_CONNECTED_NORMAL = 0UL,
      FI_EN_ACCSTATE_SENSOR_NOT_PRESENT = 65533UL,
      FI_EN_ACCSTATE_CONNECTED_DATA_INVALID = 65534UL,
      FI_EN_ACCSTATE_CONNECTED_INTERNALERROR = 65535UL
   };
   tenType enType;

   cmb_fi_tcl_e16_AccStatus(cmb_fi_tcl_e16_AccStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e16_AccStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e16_AccStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GyroData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GyroData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GyroData& operator=(const cmb_fi_tcl_GyroData& coRef);
   cmb_fi_tcl_GyroData(const cmb_fi_tcl_GyroData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   tU32 GyroValue;
   cmb_fi_tcl_e16_GyroStatus GyroStatus;
   virtual ~cmb_fi_tcl_GyroData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GyroData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_GpsChannelStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_GpsChannelStatus();

   enum {
      FI_C_U16_BIT_GPS_STATUS_ACTIVE = 0x1UL,
      FI_C_U16_BIT_GPS_STATUS_TRACKING = 0x2UL,
      FI_C_U16_BIT_GPS_STATUS_BIT_SYNC = 0x4UL,
      FI_C_U16_BIT_GPS_STATUS_CHAN_SEL = 0x8UL,
      FI_C_U16_BIT_GPS_STATUS_SYNC = 0x10UL,
      FI_C_U16_BIT_GPS_STATUS_SYNC_2 = 0x20UL,
      FI_C_U16_BIT_GPS_STATUS_FREQ_LOCK = 0x40UL,
      FI_C_U16_BIT_GPS_STATUS_TOW_VALID = 0x80UL,
      FI_C_U16_BIT_GPS_STATUS_EPHEMERIS_CHECKED = 0x100UL
   };
   tBool bGPS_STATUS_ACTIVE() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_ACTIVE) != 0; }
   tVoid vSetGPS_STATUS_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_ACTIVE)); }
   tBool bGPS_STATUS_TRACKING() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_TRACKING) != 0; }
   tVoid vSetGPS_STATUS_TRACKING(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_TRACKING):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_TRACKING)); }
   tBool bGPS_STATUS_BIT_SYNC() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_BIT_SYNC) != 0; }
   tVoid vSetGPS_STATUS_BIT_SYNC(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_BIT_SYNC):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_BIT_SYNC)); }
   tBool bGPS_STATUS_CHAN_SEL() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_CHAN_SEL) != 0; }
   tVoid vSetGPS_STATUS_CHAN_SEL(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_CHAN_SEL):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_CHAN_SEL)); }
   tBool bGPS_STATUS_SYNC() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_SYNC) != 0; }
   tVoid vSetGPS_STATUS_SYNC(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_SYNC):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_SYNC)); }
   tBool bGPS_STATUS_SYNC_2() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_SYNC_2) != 0; }
   tVoid vSetGPS_STATUS_SYNC_2(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_SYNC_2):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_SYNC_2)); }
   tBool bGPS_STATUS_FREQ_LOCK() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_FREQ_LOCK) != 0; }
   tVoid vSetGPS_STATUS_FREQ_LOCK(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_FREQ_LOCK):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_FREQ_LOCK)); }
   tBool bGPS_STATUS_TOW_VALID() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_TOW_VALID) != 0; }
   tVoid vSetGPS_STATUS_TOW_VALID(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_TOW_VALID):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_TOW_VALID)); }
   tBool bGPS_STATUS_EPHEMERIS_CHECKED() const {return (u16Value & FI_C_U16_BIT_GPS_STATUS_EPHEMERIS_CHECKED) != 0; }
   tVoid vSetGPS_STATUS_EPHEMERIS_CHECKED(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_GPS_STATUS_EPHEMERIS_CHECKED):(u16Value & ~(tU16)FI_C_U16_BIT_GPS_STATUS_EPHEMERIS_CHECKED)); }
   struct rBitValues {
      tBool GPS_STATUS_ACTIVE:1;
      tBool GPS_STATUS_TRACKING:1;
      tBool GPS_STATUS_BIT_SYNC:1;
      tBool GPS_STATUS_CHAN_SEL:1;
      tBool GPS_STATUS_SYNC:1;
      tBool GPS_STATUS_SYNC_2:1;
      tBool GPS_STATUS_FREQ_LOCK:1;
      tBool GPS_STATUS_TOW_VALID:1;
      tBool GPS_STATUS_EPHEMERIS_CHECKED:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_GpsChannelStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_GpsChannelStatus& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsTestAdditionalData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsTestAdditionalData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsTestAdditionalData& operator=(const cmb_fi_tcl_GpsTestAdditionalData& coRef);
   cmb_fi_tcl_GpsTestAdditionalData(const cmb_fi_tcl_GpsTestAdditionalData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 AuxclockTimeStamp;
   tU32 PinToggle;
   tS32 OscillatorFrequencyOffset;
   virtual ~cmb_fi_tcl_GpsTestAdditionalData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsTestAdditionalData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsBigConstData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsBigConstData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsBigConstData& operator=(const cmb_fi_tcl_GpsBigConstData& coRef);
   cmb_fi_tcl_GpsBigConstData(const cmb_fi_tcl_GpsBigConstData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 Version;
   tU16 Year;
   tU8 Month;
   tU8 Day;
   tU8 Hour;
   tU8 Minute;
   tU8 Second;
   tU8 HundredthSecond;
   tU8 Manufacturer;
   tU8 HardwareVersion;
   tU32 SoftwareVersion;
   tF64 Latitude;
   tF64 Longitude;
   tF32 AltitudeWGS84;
   tF32 AltitudeAboveSealevel;
   tF32 Speed;
   tF32 Heading;
   tF32 PDOP;
   tF32 HDOP;
   tF32 EstAccuracyPlane;
   tF32 EstAccuracyHeight;
   tU8 SatellitesHealthy;
   tU8 SatellitesVisible;
   tU8 SatellitesReceived;
   tU8 SatellitesUsed;
   tU32 ReceivedAlmanac;
   tU32 ReceivedEphemeris;
   tU32 UpdatesSameConstellation;
   tU8 FixStatus;
   tU8 AntennaStatus;
   tU8 UpdateFrequency;
   tU8 NumberOfChannels;
   tF32 EstAccuracySpeed;
   tF32 EstAccuracyCourse;
   virtual ~cmb_fi_tcl_GpsBigConstData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsBigConstData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b32_GpsValidFlags : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b32_GpsValidFlags();

   enum {
      FI_C_U32_BIT_POSITIONVALID = 0x1UL,
      FI_C_U32_BIT_FILTEREDPOSITIONVALID = 0x2UL,
      FI_C_U32_BIT_VELOCITYVALID = 0x4UL,
      FI_C_U32_BIT_FILTEREDVELOCITYVALID = 0x8UL,
      FI_C_U32_BIT_POSITIONCOVARIANCEMATRIXVALID = 0x10UL,
      FI_C_U32_BIT_POSITIONRANGERESIDUALVALID = 0x20UL,
      FI_C_U32_BIT_VELOCITYCOVARIANCEMATRIXVALID = 0x40UL,
      FI_C_U32_BIT_VELOCITYRANGERESIDUALVALID = 0x80UL,
      FI_C_U32_BIT_EPHEMERISCHECKEDVALID = 0x100UL,
      FI_C_U32_BIT_ORIGINALTIMEFIX = 0x200UL,
      FI_C_U32_BIT_ORIGINAL2DFIX = 0x400UL,
      FI_C_U32_BIT_ORIGINAL3DFIX = 0x800UL,
      FI_C_U32_BIT_RAIMALARMPOSITION = 0x40000000UL,
      FI_C_U32_BIT_RAIMALARMVELOCITY = 0x80000000UL
   };
   tBool bPositionValid() const {return (u32Value & FI_C_U32_BIT_POSITIONVALID) != 0; }
   tVoid vSetPositionValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_POSITIONVALID):(u32Value & ~(tU32)FI_C_U32_BIT_POSITIONVALID)); }
   tBool bFilteredPositionValid() const {return (u32Value & FI_C_U32_BIT_FILTEREDPOSITIONVALID) != 0; }
   tVoid vSetFilteredPositionValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_FILTEREDPOSITIONVALID):(u32Value & ~(tU32)FI_C_U32_BIT_FILTEREDPOSITIONVALID)); }
   tBool bVelocityValid() const {return (u32Value & FI_C_U32_BIT_VELOCITYVALID) != 0; }
   tVoid vSetVelocityValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_VELOCITYVALID):(u32Value & ~(tU32)FI_C_U32_BIT_VELOCITYVALID)); }
   tBool bFilteredVelocityValid() const {return (u32Value & FI_C_U32_BIT_FILTEREDVELOCITYVALID) != 0; }
   tVoid vSetFilteredVelocityValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_FILTEREDVELOCITYVALID):(u32Value & ~(tU32)FI_C_U32_BIT_FILTEREDVELOCITYVALID)); }
   tBool bPositionCovarianceMatrixValid() const {return (u32Value & FI_C_U32_BIT_POSITIONCOVARIANCEMATRIXVALID) != 0; }
   tVoid vSetPositionCovarianceMatrixValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_POSITIONCOVARIANCEMATRIXVALID):(u32Value & ~(tU32)FI_C_U32_BIT_POSITIONCOVARIANCEMATRIXVALID)); }
   tBool bPositionRangeResidualValid() const {return (u32Value & FI_C_U32_BIT_POSITIONRANGERESIDUALVALID) != 0; }
   tVoid vSetPositionRangeResidualValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_POSITIONRANGERESIDUALVALID):(u32Value & ~(tU32)FI_C_U32_BIT_POSITIONRANGERESIDUALVALID)); }
   tBool bVelocityCovarianceMatrixValid() const {return (u32Value & FI_C_U32_BIT_VELOCITYCOVARIANCEMATRIXVALID) != 0; }
   tVoid vSetVelocityCovarianceMatrixValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_VELOCITYCOVARIANCEMATRIXVALID):(u32Value & ~(tU32)FI_C_U32_BIT_VELOCITYCOVARIANCEMATRIXVALID)); }
   tBool bVelocityRangeResidualValid() const {return (u32Value & FI_C_U32_BIT_VELOCITYRANGERESIDUALVALID) != 0; }
   tVoid vSetVelocityRangeResidualValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_VELOCITYRANGERESIDUALVALID):(u32Value & ~(tU32)FI_C_U32_BIT_VELOCITYRANGERESIDUALVALID)); }
   tBool bEphemerisCheckedValid() const {return (u32Value & FI_C_U32_BIT_EPHEMERISCHECKEDVALID) != 0; }
   tVoid vSetEphemerisCheckedValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_EPHEMERISCHECKEDVALID):(u32Value & ~(tU32)FI_C_U32_BIT_EPHEMERISCHECKEDVALID)); }
   tBool bOriginalTimeFix() const {return (u32Value & FI_C_U32_BIT_ORIGINALTIMEFIX) != 0; }
   tVoid vSetOriginalTimeFix(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_ORIGINALTIMEFIX):(u32Value & ~(tU32)FI_C_U32_BIT_ORIGINALTIMEFIX)); }
   tBool bOriginal2DFix() const {return (u32Value & FI_C_U32_BIT_ORIGINAL2DFIX) != 0; }
   tVoid vSetOriginal2DFix(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_ORIGINAL2DFIX):(u32Value & ~(tU32)FI_C_U32_BIT_ORIGINAL2DFIX)); }
   tBool bOriginal3DFix() const {return (u32Value & FI_C_U32_BIT_ORIGINAL3DFIX) != 0; }
   tVoid vSetOriginal3DFix(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_ORIGINAL3DFIX):(u32Value & ~(tU32)FI_C_U32_BIT_ORIGINAL3DFIX)); }
   tBool bRaimAlarmPosition() const {return (u32Value & FI_C_U32_BIT_RAIMALARMPOSITION) != 0; }
   tVoid vSetRaimAlarmPosition(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_RAIMALARMPOSITION):(u32Value & ~(tU32)FI_C_U32_BIT_RAIMALARMPOSITION)); }
   tBool bRaimAlarmVelocity() const {return (u32Value & FI_C_U32_BIT_RAIMALARMVELOCITY) != 0; }
   tVoid vSetRaimAlarmVelocity(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_RAIMALARMVELOCITY):(u32Value & ~(tU32)FI_C_U32_BIT_RAIMALARMVELOCITY)); }
   struct rBitValues {
      tBool PositionValid:1;
      tBool FilteredPositionValid:1;
      tBool VelocityValid:1;
      tBool FilteredVelocityValid:1;
      tBool PositionCovarianceMatrixValid:1;
      tBool PositionRangeResidualValid:1;
      tBool VelocityCovarianceMatrixValid:1;
      tBool VelocityRangeResidualValid:1;
      tBool EphemerisCheckedValid:1;
      tBool OriginalTimeFix:1;
      tBool Original2DFix:1;
      tBool Original3DFix:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
      tBool dummy16:1;
      tBool dummy17:1;
      tBool dummy18:1;
      tBool dummy19:1;
      tBool dummy20:1;
      tBool dummy21:1;
      tBool dummy22:1;
      tBool dummy23:1;
      tBool dummy24:1;
      tBool dummy25:1;
      tBool dummy26:1;
      tBool dummy27:1;
      tBool dummy28:1;
      tBool dummy29:1;
      tBool RaimAlarmPosition:1;
      tBool RaimAlarmVelocity:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b32_GpsValidFlags();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b32_GpsValidFlags& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsExtBigConstData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsExtBigConstData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsExtBigConstData& operator=(const cmb_fi_tcl_GpsExtBigConstData& coRef);
   cmb_fi_tcl_GpsExtBigConstData(const cmb_fi_tcl_GpsExtBigConstData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 Version;
   tU16 Year;
   tU8 Month;
   tU8 Day;
   tU8 Hour;
   tU8 Minute;
   tU8 Second;
   tU8 HundredthSecond;
   cmb_fi_tcl_e8_GpsManufacturer Manufacturer;
   cmb_fi_tcl_e8_GpsHardwareVersion HardwareVersion;
   tU32 SoftwareVersion;
   tF64 Latitude;
   tF64 Longitude;
   tF32 AltitudeWGS84;
   tF64 LatitudeFiltered;
   tF64 LongitudeFiltered;
   tF32 AltitudeWGS84Filtered;
   tF32 GeoidalSeparation;
   tF32 VelocityNorth;
   tF32 VelocityEast;
   tF32 VelocityUp;
   tF32 VelocityNorthFiltered;
   tF32 VelocityEastFiltered;
   tF32 VelocityUpFiltered;
   cmb_fi_tcl_CovarianceMatrixUpperTriangle PositionCovarianceMatrix;
   cmb_fi_tcl_CovarianceMatrixUpperTriangle VelocityCovarianceMatrix;
   tF32 PositionRangeResidualMax;
   tF32 VelocityRangeResidualMax;
   cmb_fi_tcl_b32_GpsValidFlags ValidFlags;
   tF32 PDOP;
   tF32 HDOP;
   tU8 SatellitesHealthy;
   tU8 SatellitesVisible;
   tU8 SatellitesReceived;
   tU8 SatellitesUsed;
   tU32 ReceivedAlmanac;
   tU32 ReceivedEphemeris;
   tU32 UpdatesSameConstellation;
   cmb_fi_tcl_b8_GpsFixStatus FixStatus;
   cmb_fi_tcl_e8_GpsAntennaStatus AntennaStatus;
   tU8 UpdateFrequency;
   tU8 NumberOfChannels;
   virtual ~cmb_fi_tcl_GpsExtBigConstData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsExtBigConstData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b32_MapMatchingHint : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b32_MapMatchingHint();

   enum {
      FI_C_U32_BIT_TUNNELVALID = 0x1UL,
      FI_C_U32_BIT_TUNNELFLAG = 0x2UL,
      FI_C_U32_BIT_GARAGEVALID = 0x4UL,
      FI_C_U32_BIT_GARAGEFLAG = 0x8UL,
      FI_C_U32_BIT_CITYVALID = 0x10UL,
      FI_C_U32_BIT_CITYFLAG = 0x20UL,
      FI_C_U32_BIT_FORWARDPOSITIONVALID = 0x10000000UL,
      FI_C_U32_BIT_BACKWARDPOSITIONVALID = 0x20000000UL,
      FI_C_U32_BIT_FORWARDVELOCITYVALID = 0x40000000UL,
      FI_C_U32_BIT_BACKWARDVELOCITYVALID = 0x80000000UL
   };
   tBool bTunnelValid() const {return (u32Value & FI_C_U32_BIT_TUNNELVALID) != 0; }
   tVoid vSetTunnelValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_TUNNELVALID):(u32Value & ~(tU32)FI_C_U32_BIT_TUNNELVALID)); }
   tBool bTunnelFlag() const {return (u32Value & FI_C_U32_BIT_TUNNELFLAG) != 0; }
   tVoid vSetTunnelFlag(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_TUNNELFLAG):(u32Value & ~(tU32)FI_C_U32_BIT_TUNNELFLAG)); }
   tBool bGarageValid() const {return (u32Value & FI_C_U32_BIT_GARAGEVALID) != 0; }
   tVoid vSetGarageValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_GARAGEVALID):(u32Value & ~(tU32)FI_C_U32_BIT_GARAGEVALID)); }
   tBool bGarageFlag() const {return (u32Value & FI_C_U32_BIT_GARAGEFLAG) != 0; }
   tVoid vSetGarageFlag(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_GARAGEFLAG):(u32Value & ~(tU32)FI_C_U32_BIT_GARAGEFLAG)); }
   tBool bCityValid() const {return (u32Value & FI_C_U32_BIT_CITYVALID) != 0; }
   tVoid vSetCityValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_CITYVALID):(u32Value & ~(tU32)FI_C_U32_BIT_CITYVALID)); }
   tBool bCityFlag() const {return (u32Value & FI_C_U32_BIT_CITYFLAG) != 0; }
   tVoid vSetCityFlag(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_CITYFLAG):(u32Value & ~(tU32)FI_C_U32_BIT_CITYFLAG)); }
   tBool bForwardPositionValid() const {return (u32Value & FI_C_U32_BIT_FORWARDPOSITIONVALID) != 0; }
   tVoid vSetForwardPositionValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_FORWARDPOSITIONVALID):(u32Value & ~(tU32)FI_C_U32_BIT_FORWARDPOSITIONVALID)); }
   tBool bBackwardPositionValid() const {return (u32Value & FI_C_U32_BIT_BACKWARDPOSITIONVALID) != 0; }
   tVoid vSetBackwardPositionValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_BACKWARDPOSITIONVALID):(u32Value & ~(tU32)FI_C_U32_BIT_BACKWARDPOSITIONVALID)); }
   tBool bForwardVelocityValid() const {return (u32Value & FI_C_U32_BIT_FORWARDVELOCITYVALID) != 0; }
   tVoid vSetForwardVelocityValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_FORWARDVELOCITYVALID):(u32Value & ~(tU32)FI_C_U32_BIT_FORWARDVELOCITYVALID)); }
   tBool bBackwardVelocityValid() const {return (u32Value & FI_C_U32_BIT_BACKWARDVELOCITYVALID) != 0; }
   tVoid vSetBackwardVelocityValid(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_BACKWARDVELOCITYVALID):(u32Value & ~(tU32)FI_C_U32_BIT_BACKWARDVELOCITYVALID)); }
   struct rBitValues {
      tBool TunnelValid:1;
      tBool TunnelFlag:1;
      tBool GarageValid:1;
      tBool GarageFlag:1;
      tBool CityValid:1;
      tBool CityFlag:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
      tBool dummy16:1;
      tBool dummy17:1;
      tBool dummy18:1;
      tBool dummy19:1;
      tBool dummy20:1;
      tBool dummy21:1;
      tBool dummy22:1;
      tBool dummy23:1;
      tBool dummy24:1;
      tBool dummy25:1;
      tBool dummy26:1;
      tBool dummy27:1;
      tBool ForwardPositionValid:1;
      tBool BackwardPositionValid:1;
      tBool ForwardVelocityValid:1;
      tBool BackwardVelocityValid:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b32_MapMatchingHint();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b32_MapMatchingHint& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsBigVarData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsBigVarData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsBigVarData& operator=(const cmb_fi_tcl_GpsBigVarData& coRef);
   cmb_fi_tcl_GpsBigVarData(const cmb_fi_tcl_GpsBigVarData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 SatId;
   tU8 ChannelStatus;
   tU16 ExtendedStatus;
   tF32 Azimuth;
   tF32 Elevation;
   tF32 CarrierToNoiseRatio;
   virtual ~cmb_fi_tcl_GpsBigVarData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsBigVarData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsBigdata : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsBigdata();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsBigdata& operator=(const cmb_fi_tcl_GpsBigdata& coRef);
   cmb_fi_tcl_GpsBigdata(const cmb_fi_tcl_GpsBigdata& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   cmb_fi_tcl_GpsBigConstData GpsBigConstData;
   std::vector<cmb_fi_tcl_GpsBigVarData, std::allocator<cmb_fi_tcl_GpsBigVarData> > GpsBigVarData;
   virtual ~cmb_fi_tcl_GpsBigdata();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsBigdata& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsExtBigData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsExtBigData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsExtBigData& operator=(const cmb_fi_tcl_GpsExtBigData& coRef);
   cmb_fi_tcl_GpsExtBigData(const cmb_fi_tcl_GpsExtBigData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   cmb_fi_tcl_GpsExtBigConstData GpsExtBigConstData;
   std::vector<cmb_fi_tcl_GpsBigVarData, std::allocator<cmb_fi_tcl_GpsBigVarData> > GpsBigVarData;
   virtual ~cmb_fi_tcl_GpsExtBigData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsExtBigData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsCommunicationProtocol : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsCommunicationProtocol();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsCommunicationProtocol& operator=(const cmb_fi_tcl_GpsCommunicationProtocol& coRef);
   cmb_fi_tcl_GpsCommunicationProtocol(const cmb_fi_tcl_GpsCommunicationProtocol& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 GpsProtocol;
   tU8 CommBaudrate;
   tU8 CommParity;
   tU8 CommStopBits;
   virtual ~cmb_fi_tcl_GpsCommunicationProtocol();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsCommunicationProtocol& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SetData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SetData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SetData& operator=(const cmb_fi_tcl_SetData& coRef);
   cmb_fi_tcl_SetData(const cmb_fi_tcl_SetData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 ClkOffset;
   tU8 SvId;
   tU8 NoPasses;
   virtual ~cmb_fi_tcl_SetData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SetData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_TestModeData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_TestModeData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_TestModeData& operator=(const cmb_fi_tcl_TestModeData& coRef);
   cmb_fi_tcl_TestModeData(const cmb_fi_tcl_TestModeData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 ClkOffset;
   tU32 ClkDrift;
   tU32 TimeToBitSync;
   tU32 ISignal;
   tU32 QSignal;
   tU16 PoorStatusCount;
   tU16 BadBitCount;
   tU16 CNoMean;
   tU16 CNoSigma;
   tU16 CNoEffective;
   tU8 SvId;
   virtual ~cmb_fi_tcl_TestModeData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_TestModeData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_XoCompensation : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_XoCompensation();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_XoCompensation& operator=(const cmb_fi_tcl_XoCompensation& coRef);
   cmb_fi_tcl_XoCompensation(const cmb_fi_tcl_XoCompensation& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Tow;
   tF32 Temperature;
   tF32 C0;
   tF32 C1;
   tF32 C2;
   tF32 C3;
   tU16 Week;
   virtual ~cmb_fi_tcl_XoCompensation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_XoCompensation& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VectorLLA : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VectorLLA();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VectorLLA& operator=(const cmb_fi_tcl_VectorLLA& coRef);
   cmb_fi_tcl_VectorLLA(const cmb_fi_tcl_VectorLLA& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF64 Latitude;
   tF64 Longitude;
   tF64 Altitude;
   virtual ~cmb_fi_tcl_VectorLLA();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VectorLLA& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VectorNEU : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VectorNEU();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VectorNEU& operator=(const cmb_fi_tcl_VectorNEU& coRef);
   cmb_fi_tcl_VectorNEU(const cmb_fi_tcl_VectorNEU& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF32 North;
   tF32 East;
   tF32 Up;
   virtual ~cmb_fi_tcl_VectorNEU();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VectorNEU& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_CovarianceMatrixDiag : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_CovarianceMatrixDiag();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_CovarianceMatrixDiag& operator=(const cmb_fi_tcl_CovarianceMatrixDiag& coRef);
   cmb_fi_tcl_CovarianceMatrixDiag(const cmb_fi_tcl_CovarianceMatrixDiag& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF32 Elem11;
   tF32 Elem22;
   tF32 Elem33;
   virtual ~cmb_fi_tcl_CovarianceMatrixDiag();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_CovarianceMatrixDiag& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsHints : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsHints();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsHints& operator=(const cmb_fi_tcl_GpsHints& coRef);
   cmb_fi_tcl_GpsHints(const cmb_fi_tcl_GpsHints& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_VectorLLA ForwardPosition;
   cmb_fi_tcl_VectorLLA BackwardPosition;
   cmb_fi_tcl_VectorNEU ForwardVelocity;
   cmb_fi_tcl_VectorNEU BackwardVelocity;
   cmb_fi_tcl_CovarianceMatrixDiag PositionCovariances;
   cmb_fi_tcl_CovarianceMatrixDiag VelocityCovariances;
   cmb_fi_tcl_b32_MapMatchingHint Flags;
   virtual ~cmb_fi_tcl_GpsHints();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsHints& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsExtHints : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsExtHints();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsExtHints& operator=(const cmb_fi_tcl_GpsExtHints& coRef);
   cmb_fi_tcl_GpsExtHints(const cmb_fi_tcl_GpsExtHints& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_VectorLLA ForwardPosition;
   cmb_fi_tcl_VectorLLA BackwardPosition;
   cmb_fi_tcl_VectorNEU ForwardVelocity;
   cmb_fi_tcl_VectorNEU BackwardVelocity;
   cmb_fi_tcl_CovarianceMatrixDiag ForwardPositionCovariances;
   cmb_fi_tcl_CovarianceMatrixDiag BackwardPositionCovariances;
   cmb_fi_tcl_CovarianceMatrixDiag ForwardVelocityCovariances;
   cmb_fi_tcl_CovarianceMatrixDiag BackwardVelocityCovariances;
   cmb_fi_tcl_b32_MapMatchingHint Flags;
   tU16 Year;
   tU8 Month;
   tU8 Day;
   tU8 Hour;
   tU8 Minute;
   tU8 Second;
   tU8 HundredthSecond;
   virtual ~cmb_fi_tcl_GpsExtHints();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsExtHints& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsTightCouplingChannelData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsTightCouplingChannelData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsTightCouplingChannelData& operator=(const cmb_fi_tcl_GpsTightCouplingChannelData& coRef);
   cmb_fi_tcl_GpsTightCouplingChannelData(const cmb_fi_tcl_GpsTightCouplingChannelData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tF64 SatPosX;
   tF64 SatPosY;
   tF64 SatPosZ;
   tF64 SatVelX;
   tF64 SatVelY;
   tF64 SatVelZ;
   tF64 SatAccX;
   tF64 SatAccY;
   tF64 SatAccZ;
   tF64 TimeSentCorr;
   tF64 TimeSentUncorr;
   tF64 CarrierPhaseDiff;
   tF64 PseudoRange;
   tF64 DeltaPseudoRange;
   tF32 VarPseudoRange;
   tF32 VarDeltaPseudoRange;
   tF32 PropTimeCorrection;
   tU32 ChannelStatus;
   virtual ~cmb_fi_tcl_GpsTightCouplingChannelData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsTightCouplingChannelData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsTightCouplingCommonData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsTightCouplingCommonData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsTightCouplingCommonData& operator=(const cmb_fi_tcl_GpsTightCouplingCommonData& coRef);
   cmb_fi_tcl_GpsTightCouplingCommonData(const cmb_fi_tcl_GpsTightCouplingCommonData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 TimeStampTC;
   tU32 SampleRate;
   tF64 TimeOfWeek;
   tF64 OscFreq;
   tF32 FreqMixFactor;
   tU32 CalcInterval;
   virtual ~cmb_fi_tcl_GpsTightCouplingCommonData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsTightCouplingCommonData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsTightCouplingData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsTightCouplingData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsTightCouplingData& operator=(const cmb_fi_tcl_GpsTightCouplingData& coRef);
   cmb_fi_tcl_GpsTightCouplingData(const cmb_fi_tcl_GpsTightCouplingData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_GpsTightCouplingCommonData Common;
   std::vector<cmb_fi_tcl_GpsTightCouplingChannelData, std::allocator<cmb_fi_tcl_GpsTightCouplingChannelData> > Channel;
   virtual ~cmb_fi_tcl_GpsTightCouplingData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsTightCouplingData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_AbsStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_AbsStatus();
   enum tenType {
      FI_EN_ABS_STATUS_NORMAL = 0UL,
      FI_EN_ABS_STATUS_DATA_INVALID = 249UL,
      FI_EN_ABS_STATUS_INTERNAL_ERROR = 250UL,
      FI_EN_ABS_STATUS_NO_INFO = 251UL,
      FI_EN_ABS_STATUS_NO_MOVEMENT = 252UL,
      FI_EN_ABS_STATUS_ERROR = 253UL,
      FI_EN_ABS_STATUS_DISCONNECTED = 254UL,
      FI_EN_ABS_STATUS_UNKNOWN = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_AbsStatus(cmb_fi_tcl_e8_AbsStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_AbsStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_AbsStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_AbsDirection : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_AbsDirection();
   enum tenType {
      FI_EN_ABS_DIRECTION_UNKNOWN = 0UL,
      FI_EN_ABS_DIRECTION_FORWARD = 1UL,
      FI_EN_ABS_DIRECTION_REVERSE = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_AbsDirection(cmb_fi_tcl_e8_AbsDirection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_AbsDirection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_AbsDirection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_AbsData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_AbsData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_AbsData& operator=(const cmb_fi_tcl_AbsData& coRef);
   cmb_fi_tcl_AbsData(const cmb_fi_tcl_AbsData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 AbsTimeStamp;
   tU16 WheelCounterFrontLeft;
   tU16 WheelCounterFrontRight;
   tU16 WheelCounterRearLeft;
   tU16 WheelCounterRearRight;
   cmb_fi_tcl_e8_AbsStatus StatusFrontLeft;
   cmb_fi_tcl_e8_AbsStatus StatusFrontRight;
   cmb_fi_tcl_e8_AbsStatus StatusRearLeft;
   cmb_fi_tcl_e8_AbsStatus StatusRearRight;
   cmb_fi_tcl_e8_AbsDirection DirectionFrontLeft;
   cmb_fi_tcl_e8_AbsDirection DirectionFrontRight;
   cmb_fi_tcl_e8_AbsDirection DirectionRearLeft;
   cmb_fi_tcl_e8_AbsDirection DirectionRearRight;
   virtual ~cmb_fi_tcl_AbsData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_AbsData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_SteeringStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_SteeringStatus();
   enum tenType {
      FI_EN_STEERING_STATUS_NORMAL = 0UL,
      FI_EN_STEERING_STATUS_DATA_INVALID = 249UL,
      FI_EN_STEERING_STATUS_INTERNAL_ERROR = 250UL,
      FI_EN_STEERING_STATUS_NO_INFO = 251UL,
      FI_EN_STEERING_STATUS_NO_MOVEMENT = 252UL,
      FI_EN_STEERING_STATUS_ERROR = 253UL,
      FI_EN_STEERING_STATUS_DISCONNECTED = 254UL,
      FI_EN_STEERING_STATUS_UNKNOWN = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_SteeringStatus(cmb_fi_tcl_e8_SteeringStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_SteeringStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_SteeringStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SteeringData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SteeringData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SteeringData& operator=(const cmb_fi_tcl_SteeringData& coRef);
   cmb_fi_tcl_SteeringData(const cmb_fi_tcl_SteeringData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 SteeringTimeStamp;
   tS32 WheelAngle;
   tS32 WheelRate;
   cmb_fi_tcl_e8_SteeringStatus StatusWheelAngle;
   cmb_fi_tcl_e8_SteeringStatus StatusWheelRate;
   virtual ~cmb_fi_tcl_SteeringData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SteeringData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_OdometerRangeStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_OdometerRangeStatus();
   enum tenType {
      FI_EN_ODOMRANGE_UNKNOWN = 0UL,
      FI_EN_ODOMRANGE_ONLYSPEC = 1UL,
      FI_EN_ODOMRANGE_MINLIMITVALID = 2UL,
      FI_EN_ODOMRANGE_MAXLIMITVALID = 3UL,
      FI_EN_ODOMRANGE_MINMAXLIMITVALID = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_OdometerRangeStatus(cmb_fi_tcl_e8_OdometerRangeStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_OdometerRangeStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_OdometerRangeStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsExtTestmodeStart : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsExtTestmodeStart();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsExtTestmodeStart& operator=(const cmb_fi_tcl_GpsExtTestmodeStart& coRef);
   cmb_fi_tcl_GpsExtTestmodeStart(const cmb_fi_tcl_GpsExtTestmodeStart& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 SvId;
   tU16 GpsSearchMask;
   virtual ~cmb_fi_tcl_GpsExtTestmodeStart();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsExtTestmodeStart& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_GpsExtTestmodeResult : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_GpsExtTestmodeResult();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_GpsExtTestmodeResult& operator=(const cmb_fi_tcl_GpsExtTestmodeResult& coRef);
   cmb_fi_tcl_GpsExtTestmodeResult(const cmb_fi_tcl_GpsExtTestmodeResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS32 CurrentClkOffset;
   tU32 CurrentClkValue;
   tU32 GpsDebugChanStatusA;
   tU32 GpsDebugChanStatusB;
   tU32 GpsDebugChanStatusC;
   tU32 GpsDebugChanStatusD;
   tU8 CalcTestResult;
   tU8 TestResult;
   tS16 GpsAverageChannelSNR;
   tS16 GpsActualChannelPower;
   std::vector<tS16, std::allocator<tS16> > GpsActualChannelSNR;
   virtual ~cmb_fi_tcl_GpsExtTestmodeResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_GpsExtTestmodeResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_DefSetGroups : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_DefSetGroups();
   enum tenType {
      FI_EN_DEFSET_GROUP_MAINTENANCE = 13UL
   };
   tenType enType;

   cmb_fi_tcl_e8_DefSetGroups(cmb_fi_tcl_e8_DefSetGroups::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_DefSetGroups();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_DefSetGroups& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_DefSetMode : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_DefSetMode();
   enum tenType {
      FI_EN_DEFSET_SRC_CODING = 5UL
   };
   tenType enType;

   cmb_fi_tcl_e8_DefSetMode(cmb_fi_tcl_e8_DefSetMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_DefSetMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_DefSetMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SensorHwInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SensorHwInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SensorHwInfo& operator=(const cmb_fi_tcl_SensorHwInfo& coRef);
   cmb_fi_tcl_SensorHwInfo(const cmb_fi_tcl_SensorHwInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 AdcRangeMin;
   tU32 AdcRangeMax;
   tU32 SampleMin;
   tU32 SampleMax;
   tF32 MinNoiseValue;
   tF32 EstimOffset;
   tF32 MinOffset;
   tF32 MaxOffset;
   tF32 DriftOffset;
   tF32 MaxUnsteadOffset ;
   tF32 BestCalibOffset;
   tF32 EstimScaleFactor;
   tF32 MinScaleFactor;
   tF32 MaxScaleFactor;
   tF32 DriftScaleFactor;
   tF32 MaxUnsteadScaleFactor ;
   tF32 BestCalibScaleFactor;
   virtual ~cmb_fi_tcl_SensorHwInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SensorHwInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_3dMountAngles : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_3dMountAngles();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_3dMountAngles& operator=(const cmb_fi_tcl_3dMountAngles& coRef);
   cmb_fi_tcl_3dMountAngles(const cmb_fi_tcl_3dMountAngles& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 AngRX;
   tU8 AngRY;
   tU8 AngRZ;
   tU8 AngSX;
   tU8 AngSY;
   tU8 AngSZ;
   tU8 AngTX;
   tU8 AngTY;
   tU8 AngTZ;
   virtual ~cmb_fi_tcl_3dMountAngles();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_3dMountAngles& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_3dGyroHwInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_3dGyroHwInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_3dGyroHwInfo& operator=(const cmb_fi_tcl_3dGyroHwInfo& coRef);
   cmb_fi_tcl_3dGyroHwInfo(const cmb_fi_tcl_3dGyroHwInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 SampleRate;
   cmb_fi_tcl_3dMountAngles MountAngles;
   cmb_fi_tcl_SensorHwInfo RAxes;
   cmb_fi_tcl_SensorHwInfo SAxes;
   cmb_fi_tcl_SensorHwInfo TAxes;
   virtual ~cmb_fi_tcl_3dGyroHwInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_3dGyroHwInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_3dAccHwInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_3dAccHwInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_3dAccHwInfo& operator=(const cmb_fi_tcl_3dAccHwInfo& coRef);
   cmb_fi_tcl_3dAccHwInfo(const cmb_fi_tcl_3dAccHwInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 SampleRate;
   cmb_fi_tcl_3dMountAngles MountAngles;
   cmb_fi_tcl_SensorHwInfo RAxes;
   cmb_fi_tcl_SensorHwInfo SAxes;
   cmb_fi_tcl_SensorHwInfo TAxes;
   virtual ~cmb_fi_tcl_3dAccHwInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_3dAccHwInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_3dGyroData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_3dGyroData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_3dGyroData& operator=(const cmb_fi_tcl_3dGyroData& coRef);
   cmb_fi_tcl_3dGyroData(const cmb_fi_tcl_3dGyroData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 TimeStamp3dGyro;
   tU32 RVal;
   tU32 SVal;
   tU32 TVal;
   cmb_fi_tcl_e16_GyroStatus RStatus;
   cmb_fi_tcl_e16_GyroStatus SStatus;
   cmb_fi_tcl_e16_GyroStatus TStatus;
   virtual ~cmb_fi_tcl_3dGyroData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_3dGyroData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_3dAccData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_3dAccData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_3dAccData& operator=(const cmb_fi_tcl_3dAccData& coRef);
   cmb_fi_tcl_3dAccData(const cmb_fi_tcl_3dAccData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 TimeStamp3dAcc;
   tU32 RVal;
   tU32 SVal;
   tU32 TVal;
   cmb_fi_tcl_e16_AccStatus RStatus;
   cmb_fi_tcl_e16_AccStatus SStatus;
   cmb_fi_tcl_e16_AccStatus TStatus;
   virtual ~cmb_fi_tcl_3dAccData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_3dAccData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_TimeDate : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_TimeDate();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_TimeDate& operator=(const cmb_fi_tcl_TimeDate& coRef);
   cmb_fi_tcl_TimeDate(const cmb_fi_tcl_TimeDate& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 td_year;
   tU8 td_month;
   tU8 td_day;
   tU8 td_hour;
   tU8 td_minute;
   tU8 td_second;
   tU8 td_milliSeconds;
   virtual ~cmb_fi_tcl_TimeDate();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_TimeDate& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_timeMode : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_timeMode();
   enum tenType {
      FI_EN_TIMEMODE_DEFAULT = 0UL,
      FI_EN_TIMEMODE_AUTO = 1UL,
      FI_EN_TIMEMODE_MANUAL = 2UL,
      FI_EN_TIMEMODE_INVALID = 254UL
   };
   tenType enType;

   cmb_fi_tcl_e8_timeMode(cmb_fi_tcl_e8_timeMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_timeMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_timeMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_Reserved : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_Reserved();

   virtual ~cmb_fi_tcl_Reserved();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_Reserved& /* roRef */) const
      {  return true; }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_PerThousand : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_PerThousand(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_PerThousand();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_PerThousand& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Pascal : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Pascal(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Pascal();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Pascal& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u8_Percent : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u8_Percent(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~cmb_fi_tcl_u8_Percent();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u8_Percent& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Centimeter : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Centimeter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Centimeter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Centimeter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_CentiDegree : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_KmPerHour : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_KmPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_KmPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_KmPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u16_RPM : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u16_RPM(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~cmb_fi_tcl_u16_RPM();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u16_RPM& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_CentimeterSecond : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_CentimeterSecond(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_CentimeterSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_CentimeterSecond& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_Meter : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_Meter(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_Meter& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Meter : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Meter(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Meter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Meter& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Yards : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Yards(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Yards();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Yards& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Inch : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Inch(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Inch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Inch& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Kilogram : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Kilogram(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Kilogram();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Kilogram& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_DirectionRelative_CentiDegree : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_DirectionRelative_CentiDegree(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_DirectionRelative_CentiDegree();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_DirectionRelative_CentiDegree& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u16_MilliSecond : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u16_MilliSecond(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~cmb_fi_tcl_u16_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u16_MilliSecond& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_MilliSecond : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_MilliSecond(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_MilliSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_MilliSecond& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Longitude : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Longitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Longitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Longitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Latitude : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Latitude(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Latitude();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Latitude& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_PositionWGS84 : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_PositionWGS84();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_PositionWGS84& operator=(const cmb_fi_tcl_PositionWGS84& coRef);
   cmb_fi_tcl_PositionWGS84(const cmb_fi_tcl_PositionWGS84& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_s32_Longitude Longitude;
   cmb_fi_tcl_s32_Latitude Latitude;
   virtual ~cmb_fi_tcl_PositionWGS84();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_PositionWGS84& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_MilesPerHour : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_MilesPerHour(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_MilesPerHour();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_MilesPerHour& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Pound : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Pound(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Pound();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Pound& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_DeciKelvin : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_DeciKelvin(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_DeciKelvin();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_DeciKelvin& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_CentiDegreeSecond : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_CentiDegreeSecond(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_CentiDegreeSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_CentiDegreeSecond& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u32_MilliLux : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u32_MilliLux(tU32 _u32Value = 0);

   tU32 u32Value;
   virtual ~cmb_fi_tcl_u32_MilliLux();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u32_MilliLux& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u32_Microliter : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u32_Microliter(tU32 _u32Value = 0);

   tU32 u32Value;
   virtual ~cmb_fi_tcl_u32_Microliter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u32_Microliter& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_CentimeterSquareSecond : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_CentimeterSquareSecond(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_CentimeterSquareSecond();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_CentimeterSquareSecond& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_CurvatureValue : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_CurvatureValue(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_CurvatureValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_CurvatureValue& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_CurvatureChange : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_CurvatureChange(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_CurvatureChange();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_CurvatureChange& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e16_BaseTrafficSign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e16_BaseTrafficSign();
   enum tenType {
      FI_EN_UNSPECIFIED = 0UL,
      FI_EN_SPEEDLIMIT = 1UL,
      FI_EN_SPEEDRECOMMEND = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e16_BaseTrafficSign(cmb_fi_tcl_e16_BaseTrafficSign::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e16_BaseTrafficSign();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e16_BaseTrafficSign& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_BaseTrafficSignCategory : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_BaseTrafficSignCategory();
   enum tenType {
      FI_EN_T_RESERVED = 0UL,
      FI_EN_T_E8_SPEEDSIGN = 1UL,
      FI_EN_T_E16_WARNINGSIGN = 2UL,
      FI_EN_T_E16_REGULATORYSIGN = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_BaseTrafficSignCategory(cmb_fi_tcl_e8_BaseTrafficSignCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_BaseTrafficSignCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_BaseTrafficSignCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_SpeedSign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_SpeedSign();
   enum tenType {
      FI_EN_UNSPECIFIED = 0UL,
      FI_EN_SPEEDLIMIT = 1UL,
      FI_EN_SPEEDRECOMMEND = 2UL,
      FI_EN_ENDOFMAXSPEEDLIMIT = 3UL,
      FI_EN_ENDOFSPEEDRECOMMENDATION = 4UL,
      FI_EN_MINSPEEDLIMIT = 5UL,
      FI_EN_ENDOFMINSPEEDLIMIT = 6UL,
      FI_EN_STARTSPEEDLIMITZONE = 7UL,
      FI_EN_ENDSPEEDLIMITZONE = 8UL
   };
   tenType enType;

   cmb_fi_tcl_e8_SpeedSign(cmb_fi_tcl_e8_SpeedSign::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_SpeedSign();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_SpeedSign& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e16_WarningSign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e16_WarningSign();
   enum tenType {
      FI_EN_UNSPECIFIED = 0UL,
      FI_EN_GENERALWARNING = 1010UL,
      FI_EN_DANGEROUSCROSSING = 1020UL,
      FI_EN_LEFTSINGLECURVE = 1031UL,
      FI_EN_RIGHTSINGLECURVE = 1032UL,
      FI_EN_SCURVEFIRSTLEFT = 1051UL,
      FI_EN_SCURVEFIRSTRIGHT = 1052UL,
      FI_EN_SLOPEDOWNGRADE = 1080UL,
      FI_EN_SLOPEUPGRADE = 1100UL,
      FI_EN_SPEEDRAMPS = 1120UL,
      FI_EN_SNOWORBLACKICE = 1130UL,
      FI_EN_SLIPPERYROAD = 1140UL,
      FI_EN_FALLINGROCKSFROMRIGHT = 1151UL,
      FI_EN_FALLINGROCKSFROMLEFT = 1152UL,
      FI_EN_BANK = 1160UL,
      FI_EN_CROSSWINDFROMRIGHT = 1171UL,
      FI_EN_CROSSWINDFROMLEFT = 1172UL,
      FI_EN_CONSTRICTIONBOTHSIDES = 1200UL,
      FI_EN_CONSTRICTIONRIGHTSIDE = 1211UL,
      FI_EN_CONSTRICTIONLEFTSIDE = 1212UL,
      FI_EN_CONSTRUCTIONAHEAD = 1230UL,
      FI_EN_TRAFFICCONGESTIONLIKELY = 1240UL,
      FI_EN_TWOWAYTRAFFIC = 1250UL,
      FI_EN_MOVEABLEBRIDGE = 1280UL,
      FI_EN_COASTFERRYBASSIN = 1290UL,
      FI_EN_TRAFFICLIGHT = 1310UL,
      FI_EN_PEDESTRIANCROSSINGRIGHT = 1331UL,
      FI_EN_PEDESTRIANCROSSINGLEFT = 1332UL,
      FI_EN_ZEBRACROSSINGRIGHT = 1341UL,
      FI_EN_ZEBRACROSSINGLEFT = 1342UL,
      FI_EN_CHILDRENCROSSINGRIGHT = 1361UL,
      FI_EN_CHILDRENCROSSINGLEFT = 1362UL,
      FI_EN_BICYCLESCROSSINGRIGHT = 1381UL,
      FI_EN_BICYCLESCROSSINGLEFT = 1382UL,
      FI_EN_DOMESTICANIMALSCROSSINGRIGHT = 1401UL,
      FI_EN_DOMESTICANIMALSCROSSINGLEFT = 1402UL,
      FI_EN_WILDANIMALSCROSSINGRIGHT = 1421UL,
      FI_EN_WILDANIMALSCROSSINGLEFT = 1422UL,
      FI_EN_AIRTRAFFICRIGHT = 1441UL,
      FI_EN_AIRTRAFFICLEFT = 1442UL,
      FI_EN_BUSESENTERINGORLEAVINGROADWAY = 1450UL,
      FI_EN_RAILROADCROSSINGWITHBARRIER = 1500UL,
      FI_EN_UNPROTECTEDRAILROADCROSSING = 1510UL,
      FI_EN_THREESTRIPEDBAKEBEFOREUNPROTECTEDRAILROADCROSSING = 1530UL,
      FI_EN_THREESTRIPEDBAKEBEFORERAILROADCROSSINGWITHBARRIER = 1560UL,
      FI_EN_THREESTRIPEDBAKERIGHT = 1571UL,
      FI_EN_THREESTRIPEDBAKELEFT = 1572UL,
      FI_EN_TWOSTRIPEDBAKERIGHT = 1591UL,
      FI_EN_TWOSTRIPEDBAKELEFT = 1592UL,
      FI_EN_ONESTRIPEDBAKERIGHT = 1621UL,
      FI_EN_ONESTRIPEDBAKELEFT = 1622UL,
      FI_EN_NOUTURNFORTRUCKS = 6011UL,
      FI_EN_CURVEDIRECTIONPLATELEFT = 6251UL,
      FI_EN_CURVEDIRECTIONPLATERIGHT = 6252UL
   };
   tenType enType;

   cmb_fi_tcl_e16_WarningSign(cmb_fi_tcl_e16_WarningSign::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e16_WarningSign();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e16_WarningSign& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e16_RegulatorySign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e16_RegulatorySign();
   enum tenType {
      FI_EN_UNSPECIFIED = 0UL,
      FI_EN_GENERALPROHIBITION = 2500UL,
      FI_EN_PROHIBITIONFORCARS = 2510UL,
      FI_EN_PROHIBITIONFORTRUCKS = 2530UL,
      FI_EN_PROHIBITIONFORDANGEROUSGOODS = 2610UL,
      FI_EN_MAXIMUMTOTALWEIGHT = 2620UL,
      FI_EN_MAXIMUMLOADPERAXLE = 2630UL,
      FI_EN_MAXIMUMVEHICLEWIDTH = 2640UL,
      FI_EN_MAXIMUMVEHICLEHEIGHT = 2650UL,
      FI_EN_MAXIMUMVEHICLELENGTH = 2660UL,
      FI_EN_PROHIBITIONFORENTERING = 2670UL,
      FI_EN_MANDATORYSNOWCHAINS = 2680UL,
      FI_EN_PROHIBITIONFORWATERPOLLUTINGGOODS = 2690UL,
      FI_EN_PROHIBITIONFOREXPLOSIVEGOODS = 2691UL,
      FI_EN_OVERTAKINGPROHIBITIONFORALLCARS = 2760UL,
      FI_EN_OVERTAKINGPROHIBITIONFORCARSABOVE3500KG = 2770UL,
      FI_EN_ENDOFOVERTAKINGPROHIBITIONFORALLCARS = 2800UL,
      FI_EN_ENDOFOVERTAKINGPROHIBITIONFORCARSABOVE3500KG = 2810UL,
      FI_EN_ENDOFALLPROHIBITIONS = 2820UL
   };
   tenType enType;

   cmb_fi_tcl_e16_RegulatorySign(cmb_fi_tcl_e16_RegulatorySign::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e16_RegulatorySign();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e16_RegulatorySign& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e16_AdditionalTrafficSign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e16_AdditionalTrafficSign();
   enum tenType {
      FI_EN_UNSPECIFIED = 0UL,
      FI_EN_TIMEDEPENDENT = 1UL,
      FI_EN_VEHICLETYPE = 2UL,
      FI_EN_WEATHER = 3UL,
      FI_EN_WETNESS = 4UL,
      FI_EN_WIND = 5UL,
      FI_EN_INDISTANCEMETER = 6UL,
      FI_EN_INDISTANCEKILOMETER = 7UL,
      FI_EN_DIRECTIONLEFTDIRECTARROW = 10UL,
      FI_EN_DIRECTIONLEFTCIRCLEARROW = 11UL,
      FI_EN_DIRECTIONRIGHTDIRECTARROW = 20UL,
      FI_EN_DIRECTIONRIGHTCIRCLEARROW = 21UL,
      FI_EN_OPPOSITETRAFFIC = 31UL,
      FI_EN_BICYCLESCROSSING = 32UL,
      FI_EN_VALIDITYLENGTHMETER = 130UL,
      FI_EN_VALIDITYLENGTHKILOMETER = 131UL,
      FI_EN_STOPINDISTANCEMETER = 431UL,
      FI_EN_OIL = 630UL,
      FI_EN_SMOKE = 631UL,
      FI_EN_BADROADCONDITION = 632UL,
      FI_EN_BUILDINGVEHICLESCOMINGOUT = 633UL,
      FI_EN_STREETDAMAGED = 634UL,
      FI_EN_STREETDIRTY = 635UL,
      FI_EN_DANGEROFCOLLISION = 636UL,
      FI_EN_TOAD = 637UL,
      FI_EN_DANGEROFCONGESTION = 638UL,
      FI_EN_TREECLOSETOSTREETBOUNDARY = 639UL,
      FI_EN_SNOWORBLACKICE = 730UL,
      FI_EN_RIGHTOFWAYCHANGED = 830UL,
      FI_EN_ROADGEOMETRYCHANGED = 831UL,
      FI_EN_INDUSTRIALAREATRAINSHAVERIGHTOFWAY = 832UL,
      FI_EN_PORTAREATRAINSHAVERIGHTOFWAY = 833UL,
      FI_EN_PLAYINGCHILDREN = 1010UL,
      FI_EN_SKIDRIVERSCROSSING = 1011UL,
      FI_EN_START = 1230UL,
      FI_EN_END = 1231UL,
      FI_EN_SCHOOL = 3012UL,
      FI_EN_KINDERGARDEN = 3013UL,
      FI_EN_SKIDRIVERSCROSSINGATCERTAINTIME = 4010UL,
      FI_EN_RADARCONTROL = 4002UL,
      FI_EN_SCHOOLBUSWITHTIME = 4236UL,
      FI_EN_VALIDFORBIKES = 4611UL,
      FI_EN_VALIDFORCARS = 4810UL,
      FI_EN_VALIDFORCARSWITHTRAILER = 4811UL,
      FI_EN_VALIDFORVEHICLESWITHPERMITTEDGROSSWEIGHTOVER3DOT5TONS = 4812UL,
      FI_EN_VALIDFORTRUCKSWITHTRAILER = 4813UL,
      FI_EN_VALIDFORARTICULATEDLORRY = 4814UL,
      FI_EN_VALIDFORTRUCKSWITHTRAILERANDARTICULATEDLORRY = 4815UL,
      FI_EN_VALIDFORBUSSES = 4816UL,
      FI_EN_VALIDFORVEHILCESWITHPERMITTEDGROSSWEIGHTOVER3DOT5TONSANDBUSSESANDCARSWITHTRAILERS = 4913UL,
      FI_EN_VEHICLEWITHEXPLOSIVELOAD = 5230UL,
      FI_EN_VEHICLEWITHWATERPOLLUTINGLOAD = 5231UL,
      FI_EN_BYWETNESS = 5236UL,
      FI_EN_UNSECUREDSTREETBORDER = 5238UL,
      FI_EN_RISKOFSKIDDINGFORCARWITHTRAILER = 6010UL,
      FI_EN_FOG = 10000UL,
      FI_EN_RAIN = 10001UL
   };
   tenType enType;

   cmb_fi_tcl_e16_AdditionalTrafficSign(cmb_fi_tcl_e16_AdditionalTrafficSign::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e16_AdditionalTrafficSign();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e16_AdditionalTrafficSign& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_TimeDependentString : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_TimeDependentString(tString _szValue = 0);

   cmb_fi_tclString szValue;
   virtual ~cmb_fi_tcl_TimeDependentString();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_TimeDependentString& roRef) const
      {  return (szValue == roRef.szValue);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_LaneDependentValidity : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_LaneDependentValidity();

   enum {
      FI_C_U8_BIT_LANE0 = 0x1UL,
      FI_C_U8_BIT_LANE1 = 0x2UL,
      FI_C_U8_BIT_LANE2 = 0x4UL,
      FI_C_U8_BIT_LANE3 = 0x8UL,
      FI_C_U8_BIT_LANE4 = 0x10UL,
      FI_C_U8_BIT_LANE5 = 0x20UL,
      FI_C_U8_BIT_LANE6 = 0x40UL,
      FI_C_U8_BIT_LANE7 = 0x80UL
   };
   tBool bLane0() const {return (u8Value & FI_C_U8_BIT_LANE0) != 0; }
   tVoid vSetLane0(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE0):(u8Value & ~(tU8)FI_C_U8_BIT_LANE0)); }
   tBool bLane1() const {return (u8Value & FI_C_U8_BIT_LANE1) != 0; }
   tVoid vSetLane1(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE1):(u8Value & ~(tU8)FI_C_U8_BIT_LANE1)); }
   tBool bLane2() const {return (u8Value & FI_C_U8_BIT_LANE2) != 0; }
   tVoid vSetLane2(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE2):(u8Value & ~(tU8)FI_C_U8_BIT_LANE2)); }
   tBool bLane3() const {return (u8Value & FI_C_U8_BIT_LANE3) != 0; }
   tVoid vSetLane3(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE3):(u8Value & ~(tU8)FI_C_U8_BIT_LANE3)); }
   tBool bLane4() const {return (u8Value & FI_C_U8_BIT_LANE4) != 0; }
   tVoid vSetLane4(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE4):(u8Value & ~(tU8)FI_C_U8_BIT_LANE4)); }
   tBool bLane5() const {return (u8Value & FI_C_U8_BIT_LANE5) != 0; }
   tVoid vSetLane5(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE5):(u8Value & ~(tU8)FI_C_U8_BIT_LANE5)); }
   tBool bLane6() const {return (u8Value & FI_C_U8_BIT_LANE6) != 0; }
   tVoid vSetLane6(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE6):(u8Value & ~(tU8)FI_C_U8_BIT_LANE6)); }
   tBool bLane7() const {return (u8Value & FI_C_U8_BIT_LANE7) != 0; }
   tVoid vSetLane7(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_LANE7):(u8Value & ~(tU8)FI_C_U8_BIT_LANE7)); }
   struct rBitValues {
      tBool Lane0:1;
      tBool Lane1:1;
      tBool Lane2:1;
      tBool Lane3:1;
      tBool Lane4:1;
      tBool Lane5:1;
      tBool Lane6:1;
      tBool Lane7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_LaneDependentValidity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_LaneDependentValidity& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_TrafficSignQuantifier : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_TrafficSignQuantifier();
   enum tenType {
      FI_EN_T_RESERVED = 0UL,
      FI_EN_T_S16_KMPERHOUR = 1UL,
      FI_EN_T_S16_MILESPERHOUR = 2UL,
      FI_EN_TSTRING = 3UL,
      FI_EN_T_U8_PERCENT = 4UL,
      FI_EN_T_B8_LANEDEPENDENTVALIDITY = 5UL,
      FI_EN_T_TIMEDEPENDENTSTRING = 6UL,
      FI_EN_T_S32_METER = 7UL,
      FI_EN_T_S32_YARDS = 8UL,
      FI_EN_TMULTILANGUAGESTRING = 9UL,
      FI_EN_T_S32_CENTIMETER = 10UL,
      FI_EN_T_S32_INCH = 11UL,
      FI_EN_T_S32_KILOGRAM = 12UL,
      FI_EN_T_S32_POUND = 13UL
   };
   tenType enType;

   cmb_fi_tcl_e8_TrafficSignQuantifier(cmb_fi_tcl_e8_TrafficSignQuantifier::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_TrafficSignQuantifier();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_TrafficSignQuantifier& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_AdditionalTrafficSign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_AdditionalTrafficSign();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_AdditionalTrafficSign& operator=(const cmb_fi_tcl_AdditionalTrafficSign& coRef);
   cmb_fi_tcl_AdditionalTrafficSign(const cmb_fi_tcl_AdditionalTrafficSign& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_e16_AdditionalTrafficSign AdditionalSign;
   cmb_fi_tcl_e8_TrafficSignQuantifier QuantifierOfAddSignType;
   fi_tclTypeBase* poQuantifierOfAddSignData;
   virtual ~cmb_fi_tcl_AdditionalTrafficSign();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_AdditionalTrafficSign& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_TrafficSignElement : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_TrafficSignElement();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_TrafficSignElement& operator=(const cmb_fi_tcl_TrafficSignElement& coRef);
   cmb_fi_tcl_TrafficSignElement(const cmb_fi_tcl_TrafficSignElement& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool DynamicTrafficSign;
   tBool VirtualTrafficSign;
   cmb_fi_tcl_e8_BaseTrafficSignCategory BaseSignUnionType;
   fi_tclTypeBase* poBaseSignUnionData;
   cmb_fi_tcl_e8_TrafficSignQuantifier QuantifierOfBaseSignType;
   fi_tclTypeBase* poQuantifierOfBaseSignData;
   std::vector<cmb_fi_tcl_AdditionalTrafficSign, std::allocator<cmb_fi_tcl_AdditionalTrafficSign> > AdditionalSignList;
   virtual ~cmb_fi_tcl_TrafficSignElement();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_TrafficSignElement& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_WheelSlip : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_WheelSlip();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_WheelSlip& operator=(const cmb_fi_tcl_VeSe_WheelSlip& coRef);
   cmb_fi_tcl_VeSe_WheelSlip(const cmb_fi_tcl_VeSe_WheelSlip& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 Axle;
   tU8 PosOnAxle;
   cmb_fi_tcl_s16_PerThousand Slip;
   virtual ~cmb_fi_tcl_VeSe_WheelSlip();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_WheelSlip& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_WheelBrakePressure : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_WheelBrakePressure();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_WheelBrakePressure& operator=(const cmb_fi_tcl_VeSe_WheelBrakePressure& coRef);
   cmb_fi_tcl_VeSe_WheelBrakePressure(const cmb_fi_tcl_VeSe_WheelBrakePressure& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 Axle;
   tU8 PosOnAxle;
   cmb_fi_tcl_s32_Pascal BrakePressure;
   virtual ~cmb_fi_tcl_VeSe_WheelBrakePressure();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_WheelBrakePressure& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_WheelAirPressure : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_WheelAirPressure();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_WheelAirPressure& operator=(const cmb_fi_tcl_VeSe_WheelAirPressure& coRef);
   cmb_fi_tcl_VeSe_WheelAirPressure(const cmb_fi_tcl_VeSe_WheelAirPressure& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 Axle;
   tU8 PosOnAxle;
   cmb_fi_tcl_s32_Pascal AirPressure;
   virtual ~cmb_fi_tcl_VeSe_WheelAirPressure();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_WheelAirPressure& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_Activation : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_Activation();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OFF = 1UL,
      FI_EN_ON = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_Activation(cmb_fi_tcl_e8_VeSe_Activation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_Activation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_Activation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_DoorType : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_DoorType();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_ENGINE_HOOD = 1UL,
      FI_EN_FUEL_DOOR = 2UL,
      FI_EN_PASSENGER_DOOR = 3UL,
      FI_EN_LUGGAGE_BOOT_DOOR = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_DoorType(cmb_fi_tcl_e8_VeSe_DoorType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_DoorType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_DoorType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_WindowType : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_WindowType();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_WINDSHIELD_WINDOW = 1UL,
      FI_EN_SIDE_WINDOW = 2UL,
      FI_EN_ROOF_WINDOW = 3UL,
      FI_EN_LUGGAGE_BOOT_WINDOW = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_WindowType(cmb_fi_tcl_e8_VeSe_WindowType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_WindowType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_WindowType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_LineColor : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_LineColor();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_NO_LINE = 1UL,
      FI_EN_WHITE = 2UL,
      FI_EN_RED = 3UL,
      FI_EN_YELLOW = 4UL,
      FI_EN_OTHER = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_LineColor(cmb_fi_tcl_e8_VeSe_LineColor::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_LineColor();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_LineColor& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_DoorStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_DoorStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_DoorStatus& operator=(const cmb_fi_tcl_VeSe_DoorStatus& coRef);
   cmb_fi_tcl_VeSe_DoorStatus(const cmb_fi_tcl_VeSe_DoorStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 XPosition;
   tU8 YPosition;
   cmb_fi_tcl_e8_VeSe_Activation Closed;
   cmb_fi_tcl_e8_VeSe_DoorType DoorType;
   virtual ~cmb_fi_tcl_VeSe_DoorStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_DoorStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_WindowStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_WindowStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_WindowStatus& operator=(const cmb_fi_tcl_VeSe_WindowStatus& coRef);
   cmb_fi_tcl_VeSe_WindowStatus(const cmb_fi_tcl_VeSe_WindowStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 XPosition;
   tU8 YPosition;
   cmb_fi_tcl_u8_Percent OpeningLevel;
   cmb_fi_tcl_e8_VeSe_WindowType WindowType;
   virtual ~cmb_fi_tcl_VeSe_WindowStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_WindowStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_SeatOccupation : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_SeatOccupation();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_NO_BODY_DETECTED = 1UL,
      FI_EN_SEAT_OCCUPIED = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_SeatOccupation(cmb_fi_tcl_e8_VeSe_SeatOccupation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_SeatOccupation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_SeatOccupation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_DirectionIndicator : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_DirectionIndicator();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_NEUTRAL = 1UL,
      FI_EN_LEFT = 2UL,
      FI_EN_RIGHT = 3UL,
      FI_EN_HAZARD = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_DirectionIndicator(cmb_fi_tcl_e8_VeSe_DirectionIndicator::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_DirectionIndicator();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_DirectionIndicator& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_GearNum : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_GearNum();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_FIRST = 1UL,
      FI_EN_SECOND = 2UL,
      FI_EN_THIRD = 3UL,
      FI_EN_FOURTH = 4UL,
      FI_EN_FIFTH = 5UL,
      FI_EN_SIXTH = 6UL,
      FI_EN_SEVENTH = 7UL,
      FI_EN_EIGHTH = 8UL,
      FI_EN_NINETH = 9UL,
      FI_EN_P = 100UL,
      FI_EN_D = 101UL,
      FI_EN_N = 102UL,
      FI_EN_R2 = 254UL,
      FI_EN_R1 = 255UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_GearNum(cmb_fi_tcl_e8_VeSe_GearNum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_GearNum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_GearNum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_VeSe_GearBoxState : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_VeSe_GearBoxState();

   enum {
      FI_C_U8_BIT_UNKNOWN = 0x1UL,
      FI_C_U8_BIT_FOURWHEELDRIVE = 0x2UL,
      FI_C_U8_BIT_DIFFERENTIALLOCKFRONT = 0x4UL,
      FI_C_U8_BIT_DIFFERENTIALLOCKREAR = 0x8UL
   };
   tBool bUNKNOWN() const {return (u8Value & FI_C_U8_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_UNKNOWN):(u8Value & ~(tU8)FI_C_U8_BIT_UNKNOWN)); }
   tBool bFourWheelDrive() const {return (u8Value & FI_C_U8_BIT_FOURWHEELDRIVE) != 0; }
   tVoid vSetFourWheelDrive(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_FOURWHEELDRIVE):(u8Value & ~(tU8)FI_C_U8_BIT_FOURWHEELDRIVE)); }
   tBool bDifferentialLockFront() const {return (u8Value & FI_C_U8_BIT_DIFFERENTIALLOCKFRONT) != 0; }
   tVoid vSetDifferentialLockFront(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_DIFFERENTIALLOCKFRONT):(u8Value & ~(tU8)FI_C_U8_BIT_DIFFERENTIALLOCKFRONT)); }
   tBool bDifferentialLockRear() const {return (u8Value & FI_C_U8_BIT_DIFFERENTIALLOCKREAR) != 0; }
   tVoid vSetDifferentialLockRear(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_DIFFERENTIALLOCKREAR):(u8Value & ~(tU8)FI_C_U8_BIT_DIFFERENTIALLOCKREAR)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool FourWheelDrive:1;
      tBool DifferentialLockFront:1;
      tBool DifferentialLockRear:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_VeSe_GearBoxState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_VeSe_GearBoxState& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_CCType : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_CCType();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_NONE = 1UL,
      FI_EN_SPEEDLIMITER = 2UL,
      FI_EN_CRUISECONTROL = 3UL,
      FI_EN_ADAPTIVECRUISECONTROL = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_CCType(cmb_fi_tcl_e8_VeSe_CCType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_CCType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_CCType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_CCState : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_CCState();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_INACTIVE = 1UL,
      FI_EN_ACTIVE = 2UL,
      FI_EN_OVERRIDDEN = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_CCState(cmb_fi_tcl_e8_VeSe_CCState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_CCState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_CCState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_Ignition : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_Ignition();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OFF = 1UL,
      FI_EN_PARKING = 2UL,
      FI_EN_ON = 3UL,
      FI_EN_CRANK = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_Ignition(cmb_fi_tcl_e8_VeSe_Ignition::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_Ignition();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_Ignition& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b32_VeSe_BrakeControlMode : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b32_VeSe_BrakeControlMode();

   enum {
      FI_C_U32_BIT_UNKNOWN = 0x1UL,
      FI_C_U32_BIT_ABS_CONTROL = 0x2UL,
      FI_C_U32_BIT_ASC_CONTROL = 0x4UL,
      FI_C_U32_BIT_DSC_CONTROL = 0x8UL,
      FI_C_U32_BIT_HBA_CONTROL = 0x10UL,
      FI_C_U32_BIT_MSR_CONTROL = 0x20UL,
      FI_C_U32_BIT_EBV_CONTROL = 0x40UL,
      FI_C_U32_BIT_FLR_CONTROL = 0x80UL,
      FI_C_U32_BIT_FDB_CONTROL = 0x100UL
   };
   tBool bUNKNOWN() const {return (u32Value & FI_C_U32_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_UNKNOWN):(u32Value & ~(tU32)FI_C_U32_BIT_UNKNOWN)); }
   tBool bABS_Control() const {return (u32Value & FI_C_U32_BIT_ABS_CONTROL) != 0; }
   tVoid vSetABS_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_ABS_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_ABS_CONTROL)); }
   tBool bASC_Control() const {return (u32Value & FI_C_U32_BIT_ASC_CONTROL) != 0; }
   tVoid vSetASC_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_ASC_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_ASC_CONTROL)); }
   tBool bDSC_Control() const {return (u32Value & FI_C_U32_BIT_DSC_CONTROL) != 0; }
   tVoid vSetDSC_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_DSC_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_DSC_CONTROL)); }
   tBool bHBA_Control() const {return (u32Value & FI_C_U32_BIT_HBA_CONTROL) != 0; }
   tVoid vSetHBA_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_HBA_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_HBA_CONTROL)); }
   tBool bMSR_Control() const {return (u32Value & FI_C_U32_BIT_MSR_CONTROL) != 0; }
   tVoid vSetMSR_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_MSR_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_MSR_CONTROL)); }
   tBool bEBV_Control() const {return (u32Value & FI_C_U32_BIT_EBV_CONTROL) != 0; }
   tVoid vSetEBV_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_EBV_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_EBV_CONTROL)); }
   tBool bFLR_Control() const {return (u32Value & FI_C_U32_BIT_FLR_CONTROL) != 0; }
   tVoid vSetFLR_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_FLR_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_FLR_CONTROL)); }
   tBool bFDB_Control() const {return (u32Value & FI_C_U32_BIT_FDB_CONTROL) != 0; }
   tVoid vSetFDB_Control(tBool bSetBit) { u32Value = (tU32)((bSetBit)?(u32Value | FI_C_U32_BIT_FDB_CONTROL):(u32Value & ~(tU32)FI_C_U32_BIT_FDB_CONTROL)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool ABS_Control:1;
      tBool ASC_Control:1;
      tBool DSC_Control:1;
      tBool HBA_Control:1;
      tBool MSR_Control:1;
      tBool EBV_Control:1;
      tBool FLR_Control:1;
      tBool FDB_Control:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
      tBool dummy16:1;
      tBool dummy17:1;
      tBool dummy18:1;
      tBool dummy19:1;
      tBool dummy20:1;
      tBool dummy21:1;
      tBool dummy22:1;
      tBool dummy23:1;
      tBool dummy24:1;
      tBool dummy25:1;
      tBool dummy26:1;
      tBool dummy27:1;
      tBool dummy28:1;
      tBool dummy29:1;
      tBool dummy30:1;
      tBool dummy31:1;
   };
   union {
      tU32 u32Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b32_VeSe_BrakeControlMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b32_VeSe_BrakeControlMode& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_VeSe_CentralLocking : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_VeSe_CentralLocking();

   enum {
      FI_C_U16_BIT_UNKNOWN = 0x1UL,
      FI_C_U16_BIT_DOOR_UNLOCKED = 0x2UL,
      FI_C_U16_BIT_DOOR_LOCKED = 0x4UL,
      FI_C_U16_BIT_CL_SECURED = 0x8UL
   };
   tBool bUNKNOWN() const {return (u16Value & FI_C_U16_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_UNKNOWN):(u16Value & ~(tU16)FI_C_U16_BIT_UNKNOWN)); }
   tBool bDOOR_UNLOCKED() const {return (u16Value & FI_C_U16_BIT_DOOR_UNLOCKED) != 0; }
   tVoid vSetDOOR_UNLOCKED(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_DOOR_UNLOCKED):(u16Value & ~(tU16)FI_C_U16_BIT_DOOR_UNLOCKED)); }
   tBool bDOOR_LOCKED() const {return (u16Value & FI_C_U16_BIT_DOOR_LOCKED) != 0; }
   tVoid vSetDOOR_LOCKED(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_DOOR_LOCKED):(u16Value & ~(tU16)FI_C_U16_BIT_DOOR_LOCKED)); }
   tBool bCL_SECURED() const {return (u16Value & FI_C_U16_BIT_CL_SECURED) != 0; }
   tVoid vSetCL_SECURED(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_CL_SECURED):(u16Value & ~(tU16)FI_C_U16_BIT_CL_SECURED)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool DOOR_UNLOCKED:1;
      tBool DOOR_LOCKED:1;
      tBool CL_SECURED:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_VeSe_CentralLocking();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_VeSe_CentralLocking& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_RotationIrregularity : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_RotationIrregularity();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OK = 1UL,
      FI_EN_THROUGH_INJECTION = 2UL,
      FI_EN_WITHOUT_INJECTION = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_RotationIrregularity(cmb_fi_tcl_e8_VeSe_RotationIrregularity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_RotationIrregularity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_RotationIrregularity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OFF = 1UL,
      FI_EN_ACTIVE = 2UL,
      FI_EN_OUT_OF_RANGE = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff(cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_OverrunFuelShutoff& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_ConverterClutch : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_ConverterClutch();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_CLOSED = 1UL,
      FI_EN_OPEN = 2UL,
      FI_EN_CONTROLLED_SLIPPING = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_ConverterClutch(cmb_fi_tcl_e8_VeSe_ConverterClutch::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_ConverterClutch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_ConverterClutch& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_GearboxSportSwitch : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_GearboxSportSwitch();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_AUTOMATIC = 1UL,
      FI_EN_SPORT = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_GearboxSportSwitch(cmb_fi_tcl_e8_VeSe_GearboxSportSwitch::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_GearboxSportSwitch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_GearboxSportSwitch& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_HeadLights : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_HeadLights();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_HEAD_LIGHTS_OFF = 1UL,
      FI_EN_PARKING_LIGHT_ONLY = 2UL,
      FI_EN_LOW_BEAM = 3UL,
      FI_EN_MOTORWAY_BEAM = 4UL,
      FI_EN_HIGH_BEAM = 5UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_HeadLights(cmb_fi_tcl_e8_VeSe_HeadLights::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_HeadLights();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_HeadLights& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_SeatStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_SeatStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_SeatStatus& operator=(const cmb_fi_tcl_VeSe_SeatStatus& coRef);
   cmb_fi_tcl_VeSe_SeatStatus(const cmb_fi_tcl_VeSe_SeatStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 Row;
   tU8 PlaceInRow;
   cmb_fi_tcl_e8_VeSe_SeatOccupation Occupation;
   cmb_fi_tcl_e8_VeSe_Activation SeatBeltLock;
   cmb_fi_tcl_u8_Percent SeatClimate;
   cmb_fi_tcl_s16_DeciKelvin RealTemperature;
   cmb_fi_tcl_u8_Percent SeatHeater;
   cmb_fi_tcl_u8_Percent NeckHeater;
   cmb_fi_tcl_s16_DeciKelvin StratSeatHeater;
   virtual ~cmb_fi_tcl_VeSe_SeatStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_SeatStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_VEHICLE_STANDS = 1UL,
      FI_EN_VEHICLE_DRIVES_FORWARD = 2UL,
      FI_EN_VEHICLE_DRIVES_BACKWARDS = 3UL,
      FI_EN_VEHICLE_DRIVES = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition(cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_VehicleDrivingCondition& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_KeyValidity : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_KeyValidity();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_NO_KEY = 1UL,
      FI_EN_KEY_VALID = 2UL,
      FI_EN_KEY_INVALID = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_KeyValidity(cmb_fi_tcl_e8_VeSe_KeyValidity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_KeyValidity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_KeyValidity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_KeyType : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_KeyType();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_NO_KEY = 1UL,
      FI_EN_LAG_KEY = 2UL,
      FI_EN_PURSE_KEY = 3UL,
      FI_EN_HOTEL_KEY = 4UL,
      FI_EN_RADIO_KEY = 5UL,
      FI_EN_PASSIVE_ACCESS = 6UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_KeyType(cmb_fi_tcl_e8_VeSe_KeyType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_KeyType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_KeyType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_VeSe_KeyLocation : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_VeSe_KeyLocation();

   enum {
      FI_C_U16_BIT_UNKNOWN = 0x1UL,
      FI_C_U16_BIT_NO_ID_SENDER = 0x2UL,
      FI_C_U16_BIT_ID_SENDER_POSITION_INSIDE = 0x4UL,
      FI_C_U16_BIT_KEY_POSITION_IN_LOCK = 0x8UL
   };
   tBool bUNKNOWN() const {return (u16Value & FI_C_U16_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_UNKNOWN):(u16Value & ~(tU16)FI_C_U16_BIT_UNKNOWN)); }
   tBool bNO_ID_SENDER() const {return (u16Value & FI_C_U16_BIT_NO_ID_SENDER) != 0; }
   tVoid vSetNO_ID_SENDER(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_NO_ID_SENDER):(u16Value & ~(tU16)FI_C_U16_BIT_NO_ID_SENDER)); }
   tBool bID_SENDER_POSITION_INSIDE() const {return (u16Value & FI_C_U16_BIT_ID_SENDER_POSITION_INSIDE) != 0; }
   tVoid vSetID_SENDER_POSITION_INSIDE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_ID_SENDER_POSITION_INSIDE):(u16Value & ~(tU16)FI_C_U16_BIT_ID_SENDER_POSITION_INSIDE)); }
   tBool bKEY_POSITION_IN_LOCK() const {return (u16Value & FI_C_U16_BIT_KEY_POSITION_IN_LOCK) != 0; }
   tVoid vSetKEY_POSITION_IN_LOCK(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_KEY_POSITION_IN_LOCK):(u16Value & ~(tU16)FI_C_U16_BIT_KEY_POSITION_IN_LOCK)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool NO_ID_SENDER:1;
      tBool ID_SENDER_POSITION_INSIDE:1;
      tBool KEY_POSITION_IN_LOCK:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_VeSe_KeyLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_VeSe_KeyLocation& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_VeSe_EngineStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_VeSe_EngineStatus();

   enum {
      FI_C_U16_BIT_UNKNOWN = 0x1UL,
      FI_C_U16_BIT_OFF = 0x2UL,
      FI_C_U16_BIT_STARTUP = 0x4UL,
      FI_C_U16_BIT_RUNNING = 0x8UL,
      FI_C_U16_BIT_WARMUP = 0x10UL,
      FI_C_U16_BIT_WARM = 0x20UL,
      FI_C_U16_BIT_TEMPERATURE_EXCEEDS = 0x40UL
   };
   tBool bUNKNOWN() const {return (u16Value & FI_C_U16_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_UNKNOWN):(u16Value & ~(tU16)FI_C_U16_BIT_UNKNOWN)); }
   tBool bOFF() const {return (u16Value & FI_C_U16_BIT_OFF) != 0; }
   tVoid vSetOFF(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_OFF):(u16Value & ~(tU16)FI_C_U16_BIT_OFF)); }
   tBool bSTARTUP() const {return (u16Value & FI_C_U16_BIT_STARTUP) != 0; }
   tVoid vSetSTARTUP(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_STARTUP):(u16Value & ~(tU16)FI_C_U16_BIT_STARTUP)); }
   tBool bRUNNING() const {return (u16Value & FI_C_U16_BIT_RUNNING) != 0; }
   tVoid vSetRUNNING(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_RUNNING):(u16Value & ~(tU16)FI_C_U16_BIT_RUNNING)); }
   tBool bWARMUP() const {return (u16Value & FI_C_U16_BIT_WARMUP) != 0; }
   tVoid vSetWARMUP(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_WARMUP):(u16Value & ~(tU16)FI_C_U16_BIT_WARMUP)); }
   tBool bWARM() const {return (u16Value & FI_C_U16_BIT_WARM) != 0; }
   tVoid vSetWARM(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_WARM):(u16Value & ~(tU16)FI_C_U16_BIT_WARM)); }
   tBool bTEMPERATURE_EXCEEDS() const {return (u16Value & FI_C_U16_BIT_TEMPERATURE_EXCEEDS) != 0; }
   tVoid vSetTEMPERATURE_EXCEEDS(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_TEMPERATURE_EXCEEDS):(u16Value & ~(tU16)FI_C_U16_BIT_TEMPERATURE_EXCEEDS)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool OFF:1;
      tBool STARTUP:1;
      tBool RUNNING:1;
      tBool WARMUP:1;
      tBool WARM:1;
      tBool TEMPERATURE_EXCEEDS:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_VeSe_EngineStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_VeSe_EngineStatus& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_ReductionRPM : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_ReductionRPM();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OK = 1UL,
      FI_EN_REDUCTION_SMALL = 2UL,
      FI_EN_REDUCTION_BIG = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_ReductionRPM(cmb_fi_tcl_e8_VeSe_ReductionRPM::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_ReductionRPM();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_ReductionRPM& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_VeSe_ActiveSeats : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_VeSe_ActiveSeats();

   enum {
      FI_C_U16_BIT_UNKNOWN = 0x1UL,
      FI_C_U16_BIT_FRONT_ACTIVE_SEAT_ON = 0x2UL,
      FI_C_U16_BIT_REAR_ACTIVE_SEAT_ON = 0x4UL
   };
   tBool bUNKNOWN() const {return (u16Value & FI_C_U16_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_UNKNOWN):(u16Value & ~(tU16)FI_C_U16_BIT_UNKNOWN)); }
   tBool bFRONT_ACTIVE_SEAT_ON() const {return (u16Value & FI_C_U16_BIT_FRONT_ACTIVE_SEAT_ON) != 0; }
   tVoid vSetFRONT_ACTIVE_SEAT_ON(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_FRONT_ACTIVE_SEAT_ON):(u16Value & ~(tU16)FI_C_U16_BIT_FRONT_ACTIVE_SEAT_ON)); }
   tBool bREAR_ACTIVE_SEAT_ON() const {return (u16Value & FI_C_U16_BIT_REAR_ACTIVE_SEAT_ON) != 0; }
   tVoid vSetREAR_ACTIVE_SEAT_ON(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_REAR_ACTIVE_SEAT_ON):(u16Value & ~(tU16)FI_C_U16_BIT_REAR_ACTIVE_SEAT_ON)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool FRONT_ACTIVE_SEAT_ON:1;
      tBool REAR_ACTIVE_SEAT_ON:1;
      tBool dummy3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_VeSe_ActiveSeats();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_VeSe_ActiveSeats& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_SideIndicator : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_SideIndicator();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_LEFT = 1UL,
      FI_EN_RIGHT = 2UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_SideIndicator(cmb_fi_tcl_e8_VeSe_SideIndicator::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_SideIndicator();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_SideIndicator& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_LineInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_LineInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_LineInfo& operator=(const cmb_fi_tcl_VeSe_LineInfo& coRef);
   cmb_fi_tcl_VeSe_LineInfo(const cmb_fi_tcl_VeSe_LineInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 LineID;
   cmb_fi_tcl_s16_PerThousand Probability;
   cmb_fi_tcl_e8_VeSe_LineColor Linecolor;
   cmb_fi_tcl_s32_Centimeter LateralDistance;
   cmb_fi_tcl_s16_CentiDegree AngleToBegin;
   cmb_fi_tcl_s16_CurvatureValue HorCurvature;
   cmb_fi_tcl_s32_CurvatureChange HorCurvatureChange;
   virtual ~cmb_fi_tcl_VeSe_LineInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_LineInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_ObjectType : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_ObjectType();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_PASSENGER_CAR = 1UL,
      FI_EN_TRUCK = 2UL,
      FI_EN_BIKE = 3UL,
      FI_EN_MERGING_VEHICLE = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_ObjectType(cmb_fi_tcl_e8_VeSe_ObjectType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_ObjectType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_ObjectType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s16_VeSe_Torque : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s16_VeSe_Torque(tS16 _s16Value = 0);

   tS16 s16Value;
   virtual ~cmb_fi_tcl_s16_VeSe_Torque();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s16_VeSe_Torque& roRef) const
      {  return (s16Value == roRef.s16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_DetectedObject : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_DetectedObject();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_DetectedObject& operator=(const cmb_fi_tcl_VeSe_DetectedObject& coRef);
   cmb_fi_tcl_VeSe_DetectedObject(const cmb_fi_tcl_VeSe_DetectedObject& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 ObjectID;
   cmb_fi_tcl_e8_VeSe_ObjectType ObjectType;
   cmb_fi_tcl_e8_VeSe_Activation ObjectMeasured;
   cmb_fi_tcl_s16_PerThousand ExistProbability;
   cmb_fi_tcl_s32_Centimeter LongitudinalDistance;
   cmb_fi_tcl_s32_Centimeter LateralOffset;
   cmb_fi_tcl_s32_Centimeter ObjectWidth;
   cmb_fi_tcl_s32_Centimeter DeviationLongitudinalDistance;
   cmb_fi_tcl_s32_Centimeter DeviationLateralOffset;
   cmb_fi_tcl_s32_Centimeter DeviationWidth;
   virtual ~cmb_fi_tcl_VeSe_DetectedObject();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_DetectedObject& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_TSClassificationQuality : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_TSClassificationQuality();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_TSClassificationQuality& operator=(const cmb_fi_tcl_VeSe_TSClassificationQuality& coRef);
   cmb_fi_tcl_VeSe_TSClassificationQuality(const cmb_fi_tcl_VeSe_TSClassificationQuality& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_u8_Percent TSQuality;
   cmb_fi_tcl_u8_Percent TSQuantifierQuality;
   virtual ~cmb_fi_tcl_VeSe_TSClassificationQuality();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_TSClassificationQuality& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_TSQuality : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_TSQuality();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_TSQuality& operator=(const cmb_fi_tcl_VeSe_TSQuality& coRef);
   cmb_fi_tcl_VeSe_TSQuality(const cmb_fi_tcl_VeSe_TSQuality& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_VeSe_TSClassificationQuality TSQuality;
   std::vector<cmb_fi_tcl_VeSe_TSClassificationQuality, std::allocator<cmb_fi_tcl_VeSe_TSClassificationQuality> > AdditionalTSQualityList;
   virtual ~cmb_fi_tcl_VeSe_TSQuality();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_TSQuality& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_DetectedTrafficSign : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_DetectedTrafficSign();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_DetectedTrafficSign& operator=(const cmb_fi_tcl_VeSe_DetectedTrafficSign& coRef);
   cmb_fi_tcl_VeSe_DetectedTrafficSign(const cmb_fi_tcl_VeSe_DetectedTrafficSign& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 TrafficSignID;
   tU8 IDOfTrafficSignAbove;
   cmb_fi_tcl_TrafficSignElement TrafficSign;
   cmb_fi_tcl_s32_Centimeter TSLongitudinalDistance;
   cmb_fi_tcl_s32_Centimeter TSLateralOffset;
   cmb_fi_tcl_s32_Centimeter TSHeight;
   cmb_fi_tcl_VeSe_TSQuality TSQuality;
   virtual ~cmb_fi_tcl_VeSe_DetectedTrafficSign();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_DetectedTrafficSign& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_Ws : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_Ws(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_Ws();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_Ws& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u32_Ws : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u32_Ws(tU32 _u32Value = 0);

   tU32 u32Value;
   virtual ~cmb_fi_tcl_u32_Ws();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u32_Ws& roRef) const
      {  return (u32Value == roRef.u32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u16_W : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u16_W(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~cmb_fi_tcl_u16_W();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u16_W& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_MilliVolt : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_MilliVolt(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_MilliVolt();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_MilliVolt& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_s32_MilliAmpere : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_s32_MilliAmpere(tS32 _s32Value = 0);

   tS32 s32Value;
   virtual ~cmb_fi_tcl_s32_MilliAmpere();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_s32_MilliAmpere& roRef) const
      {  return (s32Value == roRef.s32Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u16_min : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u16_min(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~cmb_fi_tcl_u16_min();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u16_min& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_EOperationMode : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_EOperationMode();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_OPERATION_NORMAL = 1UL,
      FI_EN_OPERATION_ECO = 2UL,
      FI_EN_OPERATION_ECOPLUS = 3UL,
      FI_EN_OPERATION_SPORT = 4UL
   };
   tenType enType;

   cmb_fi_tcl_e8_EOperationMode(cmb_fi_tcl_e8_EOperationMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_EOperationMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_EOperationMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_CouplingMode : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_CouplingMode();

   enum {
      FI_C_U8_BIT_UNKNOWN = 0x1UL,
      FI_C_U8_BIT_REAR_EMACHINE_ON = 0x2UL,
      FI_C_U8_BIT_FRONT_EMACHINE_ON = 0x4UL
   };
   tBool bUNKNOWN() const {return (u8Value & FI_C_U8_BIT_UNKNOWN) != 0; }
   tVoid vSetUNKNOWN(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_UNKNOWN):(u8Value & ~(tU8)FI_C_U8_BIT_UNKNOWN)); }
   tBool bREAR_EMACHINE_ON() const {return (u8Value & FI_C_U8_BIT_REAR_EMACHINE_ON) != 0; }
   tVoid vSetREAR_EMACHINE_ON(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_REAR_EMACHINE_ON):(u8Value & ~(tU8)FI_C_U8_BIT_REAR_EMACHINE_ON)); }
   tBool bFRONT_EMACHINE_ON() const {return (u8Value & FI_C_U8_BIT_FRONT_EMACHINE_ON) != 0; }
   tVoid vSetFRONT_EMACHINE_ON(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_FRONT_EMACHINE_ON):(u8Value & ~(tU8)FI_C_U8_BIT_FRONT_EMACHINE_ON)); }
   struct rBitValues {
      tBool UNKNOWN:1;
      tBool REAR_EMACHINE_ON:1;
      tBool FRONT_EMACHINE_ON:1;
      tBool dummy3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_CouplingMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_CouplingMode& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_EnergyConsumption : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_EnergyConsumption();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_EnergyConsumption& operator=(const cmb_fi_tcl_EnergyConsumption& coRef);
   cmb_fi_tcl_EnergyConsumption(const cmb_fi_tcl_EnergyConsumption& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_u16_W PowerDemand;
   cmb_fi_tcl_u16_min Duration;
   virtual ~cmb_fi_tcl_EnergyConsumption();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_EnergyConsumption& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_PowerDemandAtSpeed : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_PowerDemandAtSpeed();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_PowerDemandAtSpeed& operator=(const cmb_fi_tcl_PowerDemandAtSpeed& coRef);
   cmb_fi_tcl_PowerDemandAtSpeed(const cmb_fi_tcl_PowerDemandAtSpeed& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_s16_KmPerHour ValidVehicleSpeed;
   cmb_fi_tcl_u16_W PowerDemand;
   virtual ~cmb_fi_tcl_PowerDemandAtSpeed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_PowerDemandAtSpeed& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_ElectricalMachine : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_ElectricalMachine();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_ElectricalMachine& operator=(const cmb_fi_tcl_ElectricalMachine& coRef);
   cmb_fi_tcl_ElectricalMachine(const cmb_fi_tcl_ElectricalMachine& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   cmb_fi_tcl_s32_Ws EnergyConsumption;
   cmb_fi_tcl_s32_Ws TotalEnergyConsumption;
   cmb_fi_tcl_u16_RPM EngineSpeed;
   cmb_fi_tcl_s16_VeSe_Torque EngineTorque;
   cmb_fi_tcl_s16_VeSe_Torque MaxAllowedEngineTorqueNext2Sec;
   cmb_fi_tcl_s16_VeSe_Torque MinAllowedEngineTorqueNext2Sec;
   cmb_fi_tcl_s16_VeSe_Torque MaxAllowedEngineTorqueNext10Sec;
   cmb_fi_tcl_s16_VeSe_Torque MinAllowedEngineTorqueNext10Sec;
   cmb_fi_tcl_e8_EOperationMode OperationMode;
   cmb_fi_tcl_b8_CouplingMode CouplingMode;
   virtual ~cmb_fi_tcl_ElectricalMachine();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_ElectricalMachine& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_VeSe_K0Status : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_VeSe_K0Status();
   enum tenType {
      FI_EN_UNKNOWN = 0UL,
      FI_EN_CLUTCH_OPEN = 1UL,
      FI_EN_CLUTCH_TRANSITION = 2UL,
      FI_EN_CLUTCH_CLOSED = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_VeSe_K0Status(cmb_fi_tcl_e8_VeSe_K0Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_VeSe_K0Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_VeSe_K0Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_VoltageParams : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_VoltageParams();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_VoltageParams& operator=(const cmb_fi_tcl_VeSe_VoltageParams& coRef);
   cmb_fi_tcl_VeSe_VoltageParams(const cmb_fi_tcl_VeSe_VoltageParams& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_s32_MilliVolt CurrentVoltage;
   cmb_fi_tcl_s32_MilliVolt MinAllowedChargingVoltageFor10Sec;
   cmb_fi_tcl_s32_MilliVolt MinAllowedDischargingVoltageFor10Sec;
   virtual ~cmb_fi_tcl_VeSe_VoltageParams();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_VoltageParams& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_CurrentParams : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_CurrentParams();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_CurrentParams& operator=(const cmb_fi_tcl_VeSe_CurrentParams& coRef);
   cmb_fi_tcl_VeSe_CurrentParams(const cmb_fi_tcl_VeSe_CurrentParams& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_s32_MilliAmpere CurrentCurrent;
   cmb_fi_tcl_s32_MilliAmpere MaxAllowedChargingCurrentFor10Sec;
   cmb_fi_tcl_s32_MilliAmpere MaxAllowedDischargingCurrentFor10Sec;
   virtual ~cmb_fi_tcl_VeSe_CurrentParams();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_CurrentParams& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_VeSe_Environment : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_VeSe_Environment();

   enum {
      FI_C_U16_BIT_ACC_ACTIVE = 0x1UL,
      FI_C_U16_BIT_WIPER = 0x2UL,
      FI_C_U16_BIT_FULL_LOAD_ACTIVE = 0x4UL,
      FI_C_U16_BIT_HEAD_LIGHTS_ON = 0x8UL,
      FI_C_U16_BIT_ACCELERATION_PEDAL_PRESSED = 0x10UL,
      FI_C_U16_BIT_BRAKE_PEDAL_PRESSED = 0x20UL,
      FI_C_U16_BIT_ENDURANCE_BRAKE_LEVER_STATE = 0x40UL,
      FI_C_U16_BIT_LOW_TEMPERATURE = 0x80UL,
      FI_C_U16_BIT_ABS_ACTIVE = 0x100UL,
      FI_C_U16_BIT_ASR_ACTIVE = 0x200UL
   };
   tBool bACC_ACTIVE() const {return (u16Value & FI_C_U16_BIT_ACC_ACTIVE) != 0; }
   tVoid vSetACC_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_ACC_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_ACC_ACTIVE)); }
   tBool bWIPER() const {return (u16Value & FI_C_U16_BIT_WIPER) != 0; }
   tVoid vSetWIPER(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_WIPER):(u16Value & ~(tU16)FI_C_U16_BIT_WIPER)); }
   tBool bFULL_LOAD_ACTIVE() const {return (u16Value & FI_C_U16_BIT_FULL_LOAD_ACTIVE) != 0; }
   tVoid vSetFULL_LOAD_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_FULL_LOAD_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_FULL_LOAD_ACTIVE)); }
   tBool bHEAD_LIGHTS_ON() const {return (u16Value & FI_C_U16_BIT_HEAD_LIGHTS_ON) != 0; }
   tVoid vSetHEAD_LIGHTS_ON(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_HEAD_LIGHTS_ON):(u16Value & ~(tU16)FI_C_U16_BIT_HEAD_LIGHTS_ON)); }
   tBool bACCELERATION_PEDAL_PRESSED() const {return (u16Value & FI_C_U16_BIT_ACCELERATION_PEDAL_PRESSED) != 0; }
   tVoid vSetACCELERATION_PEDAL_PRESSED(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_ACCELERATION_PEDAL_PRESSED):(u16Value & ~(tU16)FI_C_U16_BIT_ACCELERATION_PEDAL_PRESSED)); }
   tBool bBRAKE_PEDAL_PRESSED() const {return (u16Value & FI_C_U16_BIT_BRAKE_PEDAL_PRESSED) != 0; }
   tVoid vSetBRAKE_PEDAL_PRESSED(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BRAKE_PEDAL_PRESSED):(u16Value & ~(tU16)FI_C_U16_BIT_BRAKE_PEDAL_PRESSED)); }
   tBool bENDURANCE_BRAKE_LEVER_STATE() const {return (u16Value & FI_C_U16_BIT_ENDURANCE_BRAKE_LEVER_STATE) != 0; }
   tVoid vSetENDURANCE_BRAKE_LEVER_STATE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_ENDURANCE_BRAKE_LEVER_STATE):(u16Value & ~(tU16)FI_C_U16_BIT_ENDURANCE_BRAKE_LEVER_STATE)); }
   tBool bLOW_TEMPERATURE() const {return (u16Value & FI_C_U16_BIT_LOW_TEMPERATURE) != 0; }
   tVoid vSetLOW_TEMPERATURE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_LOW_TEMPERATURE):(u16Value & ~(tU16)FI_C_U16_BIT_LOW_TEMPERATURE)); }
   tBool bABS_ACTIVE() const {return (u16Value & FI_C_U16_BIT_ABS_ACTIVE) != 0; }
   tVoid vSetABS_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_ABS_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_ABS_ACTIVE)); }
   tBool bASR_ACTIVE() const {return (u16Value & FI_C_U16_BIT_ASR_ACTIVE) != 0; }
   tVoid vSetASR_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_ASR_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_ASR_ACTIVE)); }
   struct rBitValues {
      tBool ACC_ACTIVE:1;
      tBool WIPER:1;
      tBool FULL_LOAD_ACTIVE:1;
      tBool HEAD_LIGHTS_ON:1;
      tBool ACCELERATION_PEDAL_PRESSED:1;
      tBool BRAKE_PEDAL_PRESSED:1;
      tBool ENDURANCE_BRAKE_LEVER_STATE:1;
      tBool LOW_TEMPERATURE:1;
      tBool ABS_ACTIVE:1;
      tBool ASR_ACTIVE:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_VeSe_Environment();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_VeSe_Environment& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_VeSe_IPPCPlusData : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_VeSe_IPPCPlusData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_VeSe_IPPCPlusData& operator=(const cmb_fi_tcl_VeSe_IPPCPlusData& coRef);
   cmb_fi_tcl_VeSe_IPPCPlusData(const cmb_fi_tcl_VeSe_IPPCPlusData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 Timestamp;
   cmb_fi_tcl_s16_PerThousand Slope;
   cmb_fi_tcl_s16_PerThousand SlopeConfidence;
   cmb_fi_tcl_s16_CurvatureValue DrivenCurvature;
   cmb_fi_tcl_s16_PerThousand DrivenCurvatureConfidence;
   cmb_fi_tcl_s16_CentimeterSecond VehicleSpeed;
   cmb_fi_tcl_b16_VeSe_Environment Environment;
   virtual ~cmb_fi_tcl_VeSe_IPPCPlusData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_VeSe_IPPCPlusData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u8_SD_PackageID : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u8_SD_PackageID(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~cmb_fi_tcl_u8_SD_PackageID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u8_SD_PackageID& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_Package : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_Package();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_Package& operator=(const cmb_fi_tcl_SD_Package& coRef);
   cmb_fi_tcl_SD_Package(const cmb_fi_tcl_SD_Package& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_u8_SD_PackageID PackageID;
   cmb_fi_tclString SD_Package;
   cmb_fi_tclString SD_Path;
   virtual ~cmb_fi_tcl_SD_Package();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_Package& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_DownloadStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_DownloadStatus();
   enum tenType {
      FI_EN_NOTAVAILABLE = 1UL,
      FI_EN_AVAILABLE = 2UL,
      FI_EN_LOADFAILURE = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_DownloadStatus(cmb_fi_tcl_e8_DownloadStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_DownloadStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_DownloadStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_PackageLoadStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_PackageLoadStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_PackageLoadStatus& operator=(const cmb_fi_tcl_SD_PackageLoadStatus& coRef);
   cmb_fi_tcl_SD_PackageLoadStatus(const cmb_fi_tcl_SD_PackageLoadStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_u8_SD_PackageID PackageID;
   cmb_fi_tcl_e8_DownloadStatus DownloadStatus;
   virtual ~cmb_fi_tcl_SD_PackageLoadStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_PackageLoadStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_SD_Damage : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_SD_Damage();
   enum tenType {
      FI_EN_NODAMAGE = 1UL,
      FI_EN_UNDERVOLTAGE = 2UL,
      FI_EN_CRCFAILURE = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_SD_Damage(cmb_fi_tcl_e8_SD_Damage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_SD_Damage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_SD_Damage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_SD_Format : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_SD_Format();
   enum tenType {
      FI_EN_TYPE_OKI_ADPCM = 1UL,
      FI_EN_TTS = 2UL,
      FI_EN_AMR = 3UL,
      FI_EN_AMR_WB = 4UL,
      FI_EN_PCM = 5UL,
      FI_EN_MP3 = 6UL
   };
   tenType enType;

   cmb_fi_tcl_e8_SD_Format(cmb_fi_tcl_e8_SD_Format::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_SD_Format();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_SD_Format& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_SpeechCategory : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_SpeechCategory();
   enum tenType {
      FI_EN_DRIVINGRECOMMENDATION = 0UL,
      FI_EN_GENERALINFOAHEAD = 1UL,
      FI_EN_GENERALINFOROUTE = 2UL,
      FI_EN_GENERALINFOHORIZON = 3UL,
      FI_EN_ROUTEINFO = 4UL,
      FI_EN_DESTINFO = 5UL,
      FI_EN_HMIMESSAGE = 6UL,
      FI_EN_DEBUGMESSAGE = 7UL
   };
   tenType enType;

   cmb_fi_tcl_e8_SpeechCategory(cmb_fi_tcl_e8_SpeechCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_SpeechCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_SpeechCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b16_CategoryAddInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b16_CategoryAddInfo();

   enum {
      FI_C_U16_BIT_USER = 0x1UL,
      FI_C_U16_BIT_LONG = 0x2UL,
      FI_C_U16_BIT_SHORT = 0x4UL,
      FI_C_U16_BIT_SIGNAL = 0x8UL
   };
   tBool bUser() const {return (u16Value & FI_C_U16_BIT_USER) != 0; }
   tVoid vSetUser(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_USER):(u16Value & ~(tU16)FI_C_U16_BIT_USER)); }
   tBool bLong() const {return (u16Value & FI_C_U16_BIT_LONG) != 0; }
   tVoid vSetLong(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_LONG):(u16Value & ~(tU16)FI_C_U16_BIT_LONG)); }
   tBool bShort() const {return (u16Value & FI_C_U16_BIT_SHORT) != 0; }
   tVoid vSetShort(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_SHORT):(u16Value & ~(tU16)FI_C_U16_BIT_SHORT)); }
   tBool bSignal() const {return (u16Value & FI_C_U16_BIT_SIGNAL) != 0; }
   tVoid vSetSignal(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_SIGNAL):(u16Value & ~(tU16)FI_C_U16_BIT_SIGNAL)); }
   struct rBitValues {
      tBool User:1;
      tBool Long:1;
      tBool Short:1;
      tBool Signal:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b16_CategoryAddInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b16_CategoryAddInfo& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_SpeechSequence : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_SpeechSequence();
   enum tenType {
      FI_EN_IMMEDIATELY = 1UL,
      FI_EN_NEXT = 2UL,
      FI_EN_EOQ = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_SpeechSequence(cmb_fi_tcl_e8_SpeechSequence::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_SpeechSequence();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_SpeechSequence& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_Phrase : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_Phrase();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_Phrase& operator=(const cmb_fi_tcl_SD_Phrase& coRef);
   cmb_fi_tcl_SD_Phrase(const cmb_fi_tcl_SD_Phrase& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_u8_SD_PackageID PackageID;
   tU16 PhraseNumber;
   virtual ~cmb_fi_tcl_SD_Phrase();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_Phrase& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_Sentence : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_Sentence();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_Sentence& operator=(const cmb_fi_tcl_SD_Sentence& coRef);
   cmb_fi_tcl_SD_Sentence(const cmb_fi_tcl_SD_Sentence& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_e8_SpeechSequence SpeechSequence;
   std::vector<cmb_fi_tcl_SD_Phrase, std::allocator<cmb_fi_tcl_SD_Phrase> > Phrases;
   virtual ~cmb_fi_tcl_SD_Sentence();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_Sentence& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_SentenceAndParam : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_SentenceAndParam();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_SentenceAndParam& operator=(const cmb_fi_tcl_SD_SentenceAndParam& coRef);
   cmb_fi_tcl_SD_SentenceAndParam(const cmb_fi_tcl_SD_SentenceAndParam& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   std::vector<cmb_fi_tcl_SD_Phrase, std::allocator<cmb_fi_tcl_SD_Phrase> > Phrases;
   cmb_fi_tclString SentenceText;
   cmb_fi_tcl_e8_SpeechCategory Category;
   tU32 Timeout;
   cmb_fi_tcl_b16_CategoryAddInfo RequestAddInfo;
   virtual ~cmb_fi_tcl_SD_SentenceAndParam();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_SentenceAndParam& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u16_SD_SampleFrequency : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u16_SD_SampleFrequency(tU16 _u16Value = 0);

   tU16 u16Value;
   virtual ~cmb_fi_tcl_u16_SD_SampleFrequency();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u16_SD_SampleFrequency& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_Settings : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_Settings();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_Settings& operator=(const cmb_fi_tcl_SD_Settings& coRef);
   cmb_fi_tcl_SD_Settings(const cmb_fi_tcl_SD_Settings& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_e8_SD_Format SpeechDataFormat;
   cmb_fi_tcl_u16_SD_SampleFrequency SampleFrequency;
   tU8 Resolution;
   tBool Mono;
   virtual ~cmb_fi_tcl_SD_Settings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_Settings& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_u8_SD_Size : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_u8_SD_Size(tU8 _u8Value = 0);

   tU8 u8Value;
   virtual ~cmb_fi_tcl_u8_SD_Size();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_u8_SD_Size& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_SD_Reference : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_SD_Reference();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_SD_Reference& operator=(const cmb_fi_tcl_SD_Reference& coRef);
   cmb_fi_tcl_SD_Reference(const cmb_fi_tcl_SD_Reference& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   cmb_fi_tcl_u8_SD_Size HeaderSize;
   cmb_fi_tcl_u8_SD_Size PhraseNoSize;
   cmb_fi_tcl_u8_SD_Size StartAddSize;
   cmb_fi_tcl_u8_SD_Size StopAddSize;
   virtual ~cmb_fi_tcl_SD_Reference();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_SD_Reference& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_e8_StopVOPriority : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_e8_StopVOPriority();
   enum tenType {
      FI_EN_AFTERCURRENTSENTENCE = 1UL,
      FI_EN_IMMEDIATELY = 2UL,
      FI_EN_AFTERCURRENTPHRASE = 3UL
   };
   tenType enType;

   cmb_fi_tcl_e8_StopVOPriority(cmb_fi_tcl_e8_StopVOPriority::tenType newValue)
      :enType(newValue)
   {}

   virtual ~cmb_fi_tcl_e8_StopVOPriority();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_e8_StopVOPriority& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_b8_PhraseStatus : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_b8_PhraseStatus();

   enum {
      FI_C_U8_BIT_SPEAKABLE = 0x1UL,
      FI_C_U8_BIT_PHONEMEAVAILABLE = 0x2UL
   };
   tBool bSpeakable() const {return (u8Value & FI_C_U8_BIT_SPEAKABLE) != 0; }
   tVoid vSetSpeakable(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_SPEAKABLE):(u8Value & ~(tU8)FI_C_U8_BIT_SPEAKABLE)); }
   tBool bPhonemeAvailable() const {return (u8Value & FI_C_U8_BIT_PHONEMEAVAILABLE) != 0; }
   tVoid vSetPhonemeAvailable(tBool bSetBit) { u8Value = (tU8)((bSetBit)?(u8Value | FI_C_U8_BIT_PHONEMEAVAILABLE):(u8Value & ~(tU8)FI_C_U8_BIT_PHONEMEAVAILABLE)); }
   struct rBitValues {
      tBool Speakable:1;
      tBool PhonemeAvailable:1;
      tBool dummy2:1;
      tBool dummy3:1;
      tBool dummy4:1;
      tBool dummy5:1;
      tBool dummy6:1;
      tBool dummy7:1;
   };
   union {
      tU8 u8Value;
      rBitValues bits;
   };
   virtual ~cmb_fi_tcl_b8_PhraseStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const cmb_fi_tcl_b8_PhraseStatus& roRef) const
      {  return (u8Value == roRef.u8Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

class cmb_fi_tcl_TTSPhraseInfo : public cmb_fi_tclVisitableTypeBase
{
public:

   cmb_fi_tcl_TTSPhraseInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   cmb_fi_tcl_TTSPhraseInfo& operator=(const cmb_fi_tcl_TTSPhraseInfo& coRef);
   cmb_fi_tcl_TTSPhraseInfo(const cmb_fi_tcl_TTSPhraseInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 PhraseNumber;
   cmb_fi_tcl_SD_Phrase Phrase;
   tU16 PhraseLength;
   tU32 PhraseSize;
   tBool Speakable;
   cmb_fi_tcl_b8_PhraseStatus PhraseStatus;
   virtual ~cmb_fi_tcl_TTSPhraseInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const cmb_fi_tcl_TTSPhraseInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(cmb_fi_tclVisitorBase& rfoVisitor);
};

#endif
