/************************************************************************
 
 *FILE:           alltypes.h
 
 *SW-COMPONENT:   FI, framework for Functional Interface
 
 *DESCRIPTION:    Interface for all types defined in the different FIs
                  
 *AUTHOR:         Jochen Bruns (CM-DI/ENS41)
 
 *COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 
 *HISTORY:
           $ Log $
 * 
 *    Rev 1.0   Oct 7 2001 17:16:00   bnj2hi
 * Initial revision
 *  
 ************************************************************************/

#ifndef MOST_ALLFITYPES_HEADER
#define MOST_ALLFITYPES_HEADER

#include <vector>

class most_fi_tclVisitorBase;

const tU16 MOST_FI_C_U16_ERROR_INTERNALFAILURE = 1;
const tU16 MOST_FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION = 2;
const tU16 MOST_FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA = 3;

class most_fi_tclTypeBase : public fi_tclTypeBase
{
public:
  most_fi_tclTypeBase() {}
  virtual ~most_fi_tclTypeBase() {}
   
  virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& /*oVisitor*/) {}
};
///////////////////////////////////////////////////////////////////////////////
//
// special version including project specific MOST definitions
//
///////////////////////////////////////////////////////////////////////////////

#include "most_defines_p.h"

///////////////////////////////////////////////////////////////////////////////
//
// special version including MOST compatible types
//
// classes omitted intentionally for 
// U24, S24 and multilanguage string
//
///////////////////////////////////////////////////////////////////////////////

class most_fi_tclVisitableTypeBase :public most_fi_tclTypeBase
{
public:
   /* inline */ virtual void vTakeVisitor(most_fi_tclVisitorBase& /*oVisitor*/)
      {  }

   // globally define BIG Endian
   /* inline */ virtual fi_tclIOContextBase::tenByteOrder enGetByteOrder() const
      {  return fi_tclIOContextBase::EN_MOTOROLA;  }

};

// -------------------------------------------------------------------------------------

class most_fi_tclU8 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclU8():u8Value(0) {}
   tU8 u8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclU8& roRef) const
      { return (u8Value == roRef.u8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_U8;}

   inline operator tU8&(tVoid) { return u8Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclS8 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclS8():s8Value(0) {}
   tS8 s8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclS8& roRef) const
      { return (s8Value == roRef.s8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_S8;}

   inline operator tS8&(tVoid) { return s8Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclU16 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclU16():u16Value(0) {}
   tU16 u16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclU16& roRef) const
      { return (u16Value == roRef.u16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_U16;}

   inline operator tU16&(tVoid) { return u16Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclS16 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclS16():s16Value(0) {}
   tS16 s16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
  inline tBool operator==(const most_fi_tclS16& roRef) const
      { return (s16Value == roRef.s16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_S16;}

   inline operator tS16&(tVoid) { return s16Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclU32 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclU32():u32Value(0) {}
   tU32 u32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclU32& roRef) const
      { return (u32Value == roRef.u32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_U32;}

   inline operator tU32&(tVoid) { return u32Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclS32 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclS32():s32Value(0) {}
   tS32 s32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclS32& roRef) const
      { return (s32Value == roRef.s32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_S32;}

   inline operator tS32&(tVoid) { return s32Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclBool : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclBool():bValue(0) {}
   tBool bValue;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclBool& roRef) const
      { return (bValue == roRef.bValue); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId(tVoid) const { return most_typesfi_tclToken::EN_BOOL;}
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_String : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tcl_String():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_String(const most_fi_tcl_String& coRef);
   most_fi_tcl_String& operator=(const most_fi_tcl_String& coRef);
   virtual ~most_fi_tcl_String();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_String& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);

   enum tenCharSet {
      FI_EN_UTF16 = 0
     , FI_EN_ISO8859_15M = 1
     , FI_EN_UTF8 = 2
#ifdef FI_MOST_GB18030_CHAR_SET_ID
     , FI_EN_GB18030 = FI_MOST_GB18030_CHAR_SET_ID
#endif
   /*  , FI_EN_RDS
     , FI_EN_DAB_01
     , FI_EN_DAB_10
     , FI_EN_DAB_11
     , FI_EN_SHIFT_JIS */
   /* , 0x08 - 0xBF Reserved
      , 0xC00xEF System Integrator (e.g. Car Maker)
      , 0xF00xFF Supplier */
   };
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_STRING;}

   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   // tBool bSet(const figroup_fi_tcl_String& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet = FI_EN_UTF8);
   tString szGet(tenCharSet enCharSet = FI_EN_UTF8) const;
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_TrailingStream : public most_fi_tclVisitableTypeBase
{
private:
   tU32 _u32DataSize; 
   tU8* au8Data;

public:
   tVoid  vSetData(tU8 const * pau8DataSource, tU32 u32DataSize);
   tVoid  vGetData(tU8* pau8DestinationBuffer, tU32 u32BufferSize) const;
   tU8&  operator[] (tU32 u32Index);
   tU32   u32GetDataSize() { return _u32DataSize; }
   
   most_fi_tcl_TrailingStream() : _u32DataSize(0), au8Data(OSAL_NULL) { }
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TrailingStream(const most_fi_tcl_TrailingStream& coRef);
   most_fi_tcl_TrailingStream& operator=(const most_fi_tcl_TrailingStream& coRef);
   virtual ~most_fi_tcl_TrailingStream();
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY
   virtual tVoid vDestroy(tVoid);
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_TrailingStream& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_TRAILINGSTREAM;}
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_ShortStream : public most_fi_tclVisitableTypeBase
{
private:
   tU8 _u8DataSize;
   tU8* au8Data;

public:   
   // data access functions
   tVoid vSetData(tU8 const * pau8DataSource, tU8 u8DataSize);
   tVoid vGetData(tU8* pau8DestinationBuffer, tU32 u32BufferSize) const;
   tU8&  operator[] (tU8 u8Index);
   tU8   u8GetDataSize() { return _u8DataSize; }
   
   most_fi_tcl_ShortStream() : _u8DataSize(0), au8Data(OSAL_NULL) { }
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_ShortStream(const most_fi_tcl_ShortStream& coRef);
   most_fi_tcl_ShortStream& operator=(const most_fi_tcl_ShortStream& coRef);
   virtual ~most_fi_tcl_ShortStream();
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY
   virtual tVoid vDestroy(tVoid);
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_ShortStream& oRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_SHORTSTREAM;}
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_ClassifiedStream : public most_fi_tclVisitableTypeBase
{
private:
   tU16 _u16DataSize;
   tU8* au8Data;

public:   
   tString sClassifier;
   
   most_fi_tcl_ClassifiedStream() : _u16DataSize(0),au8Data(OSAL_NULL),sClassifier(OSAL_NULL) { }
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_ClassifiedStream(const most_fi_tcl_ClassifiedStream& coRef);
   most_fi_tcl_ClassifiedStream& operator=(const most_fi_tcl_ClassifiedStream& coRef);
   virtual ~most_fi_tcl_ClassifiedStream();
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY
   virtual tVoid vDestroy(tVoid);
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_ClassifiedStream& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_CLASSIFIEDSTREAM;}

   // data access functions
   tVoid vSetData(tU8 const * pau8DataSource, tU16 u16DataSize);
   tVoid vGetData(tU8* pau8DestinationBuffer, tU32 u32BufferSize) const;
   tU8&  operator[] (tU16 u16Index);
   tU16  u16GetDataSize() { return _u16DataSize; }
   
   // Import / export string values.
   // Attention: string values are copied by these functions, thus incoming and/or
   // outgoing strings have to be deleted externally.
   tString szGetDataAsString();
   tVoid   vSetDataFromString(tCString szText);
   tString szGetClassifier() const;
   tVoid   vSetClassifier(tCString szClassifier);
};

// *************************************************************************************

class most_fi_tcl_e8_ErrorCode;
class most_fi_tcl_e8_BTSetPairingType;
class most_fi_tcl_e8_BTSetPairingResponseType;
class most_fi_tcl_e8_BTSetPairingStatusType;
class most_fi_tcl_e8_BTSetDeviceStatus;
class most_fi_tcl_BTSetDateStamp;
class most_fi_tcl_BTSetTimeStamp;
class most_fi_tcl_BTSetDateTimeStamp;
class most_fi_tcl_e8_BTSetChangeDeviceStateAction;
class most_fi_tcl_e8_BTSetConnectionResult;
class most_fi_tcl_BTSetDeviceRingtoneListResultItem;
class most_fi_tcl_BTSetDeviceRingtoneListResult;
class most_fi_tcl_BTSetRingtoneListItem;
class most_fi_tcl_BTSetRingtoneList;
class most_fi_tcl_BTSetDeviceControlSupport;
class most_fi_tcl_e8_BTSetAudioStreamingSupportType;
class most_fi_tcl_e8_BTSetInternetAccessSupportType;
class most_fi_tcl_e8_BTSetMajorDeviceClass;
class most_fi_tcl_BTSetBluetoothFeatureSupport;
class most_fi_tcl_BTSetSupportedProfiles;
class most_fi_tcl_BTSetDeviceIdentification;
class most_fi_tcl_e8_BTSetOrigin;
class most_fi_tcl_e8_BTSetScanningState;
class most_fi_tcl_BTServiceType;
class most_fi_tcl_e8_BTConnectionStatus;
class most_fi_tcl_e8_BTSetDeviceDisconnectedReason;
class most_fi_tcl_BTProfileDisconnectedReason;
class most_fi_tcl_e8_BTDisconnectionReason;
class most_fi_tcl_BTSPPServiceConnectionStatusListItem;
class most_fi_tcl_BTProfileConnectionStatus;
class most_fi_tcl_BTSPPServiceConnectionStatusList;
class most_fi_tcl_e8_BTUsagePreference;
class most_fi_tcl_BTProfileFavoriteIndex;
class most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus;
class most_fi_tcl_BTSetVisibleDeviceItem;
class most_fi_tcl_BTSetSignalStrengthItem;
class most_fi_tcl_BTSetSignalStrengthList;
class most_fi_tcl_BTSetVisibleDevicesList;
class most_fi_tcl_BTSetDeviceListChangeExtended;
class most_fi_tcl_BTSetDeviceListExtendedResultItem;
class most_fi_tcl_e8_BTSetPairingResult;
class most_fi_tcl_BTSetPairableModeType;
class most_fi_tcl_BTLocalModeType;
class most_fi_tcl_e8_BTTechnology;
class most_fi_tcl_e8_BTCommunicationChannel;
class most_fi_tcl_e8_BTLimitationAction;
class most_fi_tcl_e8_BTLimitationActionState;
class most_fi_tcl_BTLimitationModeListItem;
class most_fi_tcl_BTLimitationModeList;
class most_fi_tcl_BTLimitationMode;
class most_fi_tcl_e8_BTOobType;
class most_fi_tcl_e8_BTFavoriteType;
class most_fi_tcl_BTSetDeviceListExtendedResult;
class most_fi_tcl_e8_BTSwitchStatus;
class most_fi_tcl_e8_BTProfile;
class most_fi_tcl_e8_BTProfileUsageType;
class most_fi_tcl_BTProfilesUsageInfo;
class most_fi_tcl_BTProfilesUsageInfoList;
class most_fi_tcl_BTProfilesUsageList;
class most_fi_tcl_e8_BTResolutionType;
class most_fi_tcl_e8_BTConflictTrigger;
class most_fi_tcl_e8_BTConflictType;
class most_fi_tcl_e8_BTConflictState;
class most_fi_tcl_BTConflictInfo;
class most_fi_tcl_BTConflictInfoList;
class most_fi_tcl_e8_BTSetUsagePreference;
class most_fi_tcl_e8_BTAutoconnectionStatus;
class most_fi_tcl_e8_BTUserAvailable;
class most_fi_tcl_e8_BTServiceAvailable;
class most_fi_tcl_e8_MsgFolderType;
class most_fi_tcl_e8_MsgMessageListFilterType;
class most_fi_tcl_e8_MsgMessageListSortType;
class most_fi_tcl_e8_MsgMessageListType;
class most_fi_tcl_e8_MsgMessageType;
class most_fi_tcl_e8_MsgSendMessageResultType;
class most_fi_tcl_e8_MsgPlaybackState;
class most_fi_tcl_MsgMessageHandle;
class most_fi_tcl_MsgMessageDate;
class most_fi_tcl_MsgMessageTime;
class most_fi_tcl_MsgMessageDateTime;
class most_fi_tcl_MsgMessageListSliceResultItem;
class most_fi_tcl_MsgMessageListSliceResult;
class most_fi_tcl_e8_MsgListChangeType;
class most_fi_tcl_MsgChangeItems;
class most_fi_tcl_MsgFolderListResultItem;
class most_fi_tcl_MsgFolderListResult;
class most_fi_tcl_e8_MsgDeletionStatus;
class most_fi_tcl_MsgDeleteMessageHandles;
class most_fi_tcl_e8_MsgRecoveryStatus;
class most_fi_tcl_MsgAccountListResultItem;
class most_fi_tcl_MsgAccountListResult;
class most_fi_tcl_MsgActiveAccount;
class most_fi_tcl_e8_MsgAddressFieldType;
class most_fi_tcl_MsgMsgBodyActionElementTags;
class most_fi_tcl_MsgAddressFieldItem;
class most_fi_tcl_MsgAddressField;
class most_fi_tcl_MsgMessageDetailsResult;
class most_fi_tcl_e8_MsgAttachmentType;
class most_fi_tcl_MsgAttachmentListItem;
class most_fi_tcl_MsgAttachmentList;
class most_fi_tcl_e8_MsgPlaybackAction;
class most_fi_tcl_MsgBodyPartListItem;
class most_fi_tcl_MsgBodyPartList;
class most_fi_tcl_e8_MsgTTSLanguage;
class most_fi_tcl_e8_MsgRecordMessageAction;
class most_fi_tcl_e8_MsgRecordMessageState;
class most_fi_tcl_e8_MsgPredefinedMessageType;
class most_fi_tcl_MsgPredefinedMessageListResultItem;
class most_fi_tcl_MsgPredefinedMessageListResult;
class most_fi_tcl_MsgConversationMessages;
class most_fi_tcl_e8_MsgSentMessageStatus;
class most_fi_tcl_MsgAttachmentHandleStream;
class most_fi_tcl_MsgMessageBodyText;
class most_fi_tcl_e8_MsgActionElementType;
class most_fi_tcl_e8_MsgCreateMessageType;
class most_fi_tcl_e8_MsgNotificationType;
class most_fi_tcl_e8_MapConnectionStatus;
class most_fi_tcl_e8_MsgPredefinedMessageSubCategory;
class most_fi_tcl_e8_MsgPredefinedMessageCategory;
class most_fi_tcl_MsgPredefinedMessageListExtendedResultItem;
class most_fi_tcl_MsgPredefinedMessageListExtendedResult;
class most_fi_tcl_e8_MsgLanguageSet;
class most_fi_tcl_SignatureTextStreamItem;
class most_fi_tcl_SignatureTextStream;
class most_fi_tcl_MsgConversationHandle;
class most_fi_tcl_MsgConversationListSliceResultItem;
class most_fi_tcl_e8_MsgMessageStatus;
class most_fi_tcl_MsgConversationDetailListItem;
class most_fi_tcl_MsgConversationListSliceResult;
class most_fi_tcl_MsgConversationDetailList;
class most_fi_tcl_OnOffListItem;
class most_fi_tcl_OnOffList;
class most_fi_tcl_AutoReplyTextListItem;
class most_fi_tcl_AutoReplyTextList;
class most_fi_tcl_MapDeviceCapabilitiesListItem;
class most_fi_tcl_MapDeviceCapabilitiesList;
class most_fi_tcl_e8_PhonBkPhoneBookSortType;
class most_fi_tcl_PhonBkSearchKeyboardListItem;
class most_fi_tcl_PhonBkSearchKeyboardList;
class most_fi_tcl_e8_PhonBkContactDetailFilter;
class most_fi_tcl_PhonBkHomeAddressLatLong;
class most_fi_tcl_PhonBkWorkAddressLatLong;
class most_fi_tcl_PhonBkOtherAddressLatLong;
class most_fi_tcl_PhonBkContactDetails;
class most_fi_tcl_PhonBkContactPhoto;
class most_fi_tcl_e8_PhonBkListChangeType;
class most_fi_tcl_e8_PhonBkListType;
class most_fi_tcl_PhonBkChangeItems;
class most_fi_tcl_e8_PhonBkContactAddStatus;
class most_fi_tcl_e8_PhonBkContactDeleteStatus;
class most_fi_tcl_e8_PhonBkContactEditStatus;
class most_fi_tcl_e8_PhonBkCallHistorySortType;
class most_fi_tcl_e8_PhonBkCallHistoryType;
class most_fi_tcl_e8_PhonBkCallPhoneType;
class most_fi_tcl_PhonBkCallDate;
class most_fi_tcl_PhonBkCallTime;
class most_fi_tcl_PhonBkCallDateTime;
class most_fi_tcl_PhonBkCallHistoryListSliceResultItem;
class most_fi_tcl_PhonBkCallHistoryListSliceResult;
class most_fi_tcl_e8_PhonBkContactDetailType;
class most_fi_tcl_PhonBkCallHistoryTypeSupport;
class most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem;
class most_fi_tcl_PhonBkAutocompleteMultipleMatchList;
class most_fi_tcl_PhonBkPhoneBookListSliceResultItem;
class most_fi_tcl_PhonBkPhoneBookListSliceResult;
class most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder;
class most_fi_tcl_e8_PhonBkRecentCallListDownloadState;
class most_fi_tcl_e8_PhonBkPhoneBookDownloadState;
class most_fi_tcl_PhonBkDownloadStateStreamItem;
class most_fi_tcl_PhonBkDownloadStateStream;
class most_fi_tcl_e8_PhonBkNumberType;
class most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem;
class most_fi_tcl_PhonBkPhoneNumberItem;
class most_fi_tcl_e8_PhonBkContactType;
class most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem;
class most_fi_tcl_PhonBkPhoneBookListSliceResultExtended;
class most_fi_tcl_PhonBkCallHistoryListSliceResultExtened;
class most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended;
class most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended;
class most_fi_tcl_PhonBkDownloadStateExtendedStreamItem;
class most_fi_tcl_PhonBkDownloadStateExtendedStream;
class most_fi_tcl_PhonBkDeviceHandleList;
class most_fi_tcl_e8_PhonBkSearchSortType;
class most_fi_tcl_PhonBkContactMatchListItem;
class most_fi_tcl_PhonBkContactMatchList;
class most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState;
class most_fi_tcl_PhonBkDownloadOnOffStateListItem;
class most_fi_tcl_PhonBkDownloadOnOffStateList;
class most_fi_tcl_e8_PhonBkContactSortType;
class most_fi_tcl_e8_PhonBkContactListType;
class most_fi_tcl_PhonBkContactDetailsExtended;
class most_fi_tcl_AutomaticContactDownloadListItem;
class most_fi_tcl_AutomaticContactDownloadList;
class most_fi_tcl_DevicePhoneBookFeatureSupportListItem;
class most_fi_tcl_DevicePhoneBookFeatureSupportList;
class most_fi_tcl_PhonBkDownloadProgressListItem;
class most_fi_tcl_PhonBkDownloadProgressList;
class most_fi_tcl_PhonBkMissedCallIndicatorListItem;
class most_fi_tcl_PhonBkMissedCallIndicatorList;
class most_fi_tcl_PhonBkPreferredSortOrderListItem;
class most_fi_tcl_PhonBkPreferredSortOrderList;
class most_fi_tcl_e8_PbkLanguage;
class most_fi_tcl_e8_FavPrvAudioChannel;
class most_fi_tcl_e8_FavPrvVideoChannel;
class most_fi_tcl_e8_FavPrvFavoriteType;
class most_fi_tcl_FavPrvFavoriteInfoItem;
class most_fi_tcl_FavPrvFavoriteInfo;
class most_fi_tcl_e8_FavPrvListChangeType;
class most_fi_tcl_FavPrvChangedFavorites;
class most_fi_tcl_e8_SrcAudioChannelName;
class most_fi_tcl_e8_SrcContentProtection;
class most_fi_tcl_e8_SrcContentType;
class most_fi_tcl_e8_SrcTransmissionClass;
class most_fi_tcl_SrcAudioChannels_Resolution;
class most_fi_tcl_SrcAudioChannelListItem;
class most_fi_tcl_SrcAudioChannelList;
class most_fi_tcl_SrcAudioChannels_AudioChannelList;
class most_fi_tcl_SrcContentDescription;
class most_fi_tcl_e8_SrcActivity;
class most_fi_tcl_e8_TelCallStatus;
class most_fi_tcl_e8_TelHandsetState;
class most_fi_tcl_TelActiveCallinHandsetStreamItem;
class most_fi_tcl_TelActiveCallinHandsetStream;
class most_fi_tcl_TelCallStatusNoticeStreamItem;
class most_fi_tcl_TelCallStatusNoticeStream;
class most_fi_tcl_e8_TelCellService;
class most_fi_tcl_TelStreamOfCallInstances;
class most_fi_tcl_e8_TelIncomingCallEventEnum;
class most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting;
class most_fi_tcl_TelHFEnhancedCallControlStreamItem;
class most_fi_tcl_TelHFEnhancedCallControlStream;
class most_fi_tcl_e8_TelInbandRingingSupport;
class most_fi_tcl_TelDeviceRingtoneListResultItem;
class most_fi_tcl_TelDeviceRingtoneListResult;
class most_fi_tcl_TelSystemWideRingtonesListResultItem;
class most_fi_tcl_TelSystemWideRingtonesListResult;
class most_fi_tcl_TelSuppressRingtoneOnOffListResultItem;
class most_fi_tcl_TelSuppressRingtoneOnOffListResult;
class most_fi_tcl_TelAutoWaitingModeOnOffListResultItem;
class most_fi_tcl_TelAutoWaitingModeOnOffListResult;
class most_fi_tcl_e8_TelWaitingModeAction;
class most_fi_tcl_e8_TelWaitingModeStatus;
class most_fi_tcl_e8_TelSiriAvailabilityState;
class most_fi_tcl_TelCallStatusNoticeExtendedStreamItem;
class most_fi_tcl_TelCallStatusNoticeExtendedStream;
class most_fi_tcl_TelOutgoingCall_Failed;
class most_fi_tcl_TelIncomingCall_Rejected;
class most_fi_tcl_e8_WaitingModeState;
class most_fi_tcl_e8_ExternalVRState;
class most_fi_tcl_e8_VRErrorCode;
class most_fi_tcl_ListOfCallInstanceStreamItem;
class most_fi_tcl_ListOfCallInstanceStream;
class most_fi_tcl_e8_AudioTransferDirection;
class most_fi_tcl_e8_TelPlaybackState;
class most_fi_tcl_e8_ErrorCode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ErrorCode();
   enum tenType {
      FI_EN_FBLOCKIDNOTAVAILABLE = 1UL,
      FI_EN_INSTIDNOTAVAILABLE = 2UL,
      FI_EN_FKTIDNOTAVAILABLE = 3UL,
      FI_EN_OPTYPENOTAVAILABLE = 4UL,
      FI_EN_INVALIDLENGTH = 5UL,
      FI_EN_PARAMETERWRONG_OUTOFRANGE = 6UL,
      FI_EN_PARAMETERNOTAVAILABLE = 7UL,
      FI_EN_PARAMETERMISSING = 8UL,
      FI_EN_TOOMANYPARAMETERS = 9UL,
      FI_EN_FUNCTIONSPECIFIC = 32UL,
      FI_EN_BUSY = 64UL,
      FI_EN_NOTAVAILABLE = 65UL
   };
   tenType enType;

   most_fi_tcl_e8_ErrorCode(most_fi_tcl_e8_ErrorCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ErrorCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetPairingType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetPairingType();
   enum tenType {
      FI_EN_E8PAIRING_TYPE_LEGACY = 0UL,
      FI_EN_E8PAIRING_TYPE_SSP_NUMERIC_COMPARISON = 1UL,
      FI_EN_E8PAIRING_TYPE_SSP_JUST_WORKS = 2UL,
      FI_EN_E8PAIRING_TYPE_CHANGE_PIN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetPairingType(most_fi_tcl_e8_BTSetPairingType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetPairingType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetPairingType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetPairingResponseType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetPairingResponseType();
   enum tenType {
      FI_EN_E8PAIRING_RESPONSE_YES = 0UL,
      FI_EN_E8PAIRING_RESPONSE_NO = 1UL,
      FI_EN_E8PAIRING_RESPONSE_CANCEL = 2UL,
      FI_EN_E8PAIRING_RESPONSE_RETRY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetPairingResponseType(most_fi_tcl_e8_BTSetPairingResponseType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetPairingResponseType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetPairingResponseType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetPairingStatusType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetPairingStatusType();
   enum tenType {
      FI_EN_E8PAIRING_STATUS_SUCCESSFUL = 0UL,
      FI_EN_E8PAIRING_STATUS_UNSUCCESSFUL = 1UL,
      FI_EN_E8PAIRING_STATUS_TIMEOUT = 2UL,
      FI_EN_E8PAIRING_STATUS_CONNECTIONLOST = 3UL,
      FI_EN_E8PAIRING_STATUS_AUTHENTICATIONFAILURE = 4UL,
      FI_EN_E8PAIRING_STATUS_CANCELLED = 5UL,
      FI_EN_E8PAIRING_STATUS_RETRY_KEEP_SESSION = 6UL,
      FI_EN_E8PAIRING_STATUS_RETRY_NEW_SESSION = 7UL,
      FI_EN_E8PAIRING_STATUS_IDLE = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetPairingStatusType(most_fi_tcl_e8_BTSetPairingStatusType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetPairingStatusType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetPairingStatusType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetDeviceStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetDeviceStatus();
   enum tenType {
      FI_EN_E8DEVICE_CONNECTED = 0UL,
      FI_EN_E8DEVICE_CONNECTED_OSD = 1UL,
      FI_EN_E8DEVICE_DISCONNECTED = 2UL,
      FI_EN_E8DEVICE_CHANGED_TO_OSD = 3UL,
      FI_EN_E8DEVICE_DELETED = 4UL,
      FI_EN_E8DEVICE_NO_CHANGE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetDeviceStatus(most_fi_tcl_e8_BTSetDeviceStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetDeviceStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetDeviceStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDateStamp : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDateStamp();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDateStamp& operator=(const most_fi_tcl_BTSetDateStamp& coRef);
   most_fi_tcl_BTSetDateStamp(const most_fi_tcl_BTSetDateStamp& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sCldrYear;
   most_fi_tcl_String sCldrMonth;
   most_fi_tcl_String sCldrDay;
   virtual ~most_fi_tcl_BTSetDateStamp();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDateStamp& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetTimeStamp : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetTimeStamp();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetTimeStamp& operator=(const most_fi_tcl_BTSetTimeStamp& coRef);
   most_fi_tcl_BTSetTimeStamp(const most_fi_tcl_BTSetTimeStamp& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sHours;
   most_fi_tcl_String sMinutes;
   most_fi_tcl_String sSeconds;
   virtual ~most_fi_tcl_BTSetTimeStamp();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetTimeStamp& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDateTimeStamp : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDateTimeStamp();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDateTimeStamp& operator=(const most_fi_tcl_BTSetDateTimeStamp& coRef);
   most_fi_tcl_BTSetDateTimeStamp(const most_fi_tcl_BTSetDateTimeStamp& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_BTSetDateStamp oDateStamp;
   most_fi_tcl_BTSetTimeStamp oTimeStamp;
   virtual ~most_fi_tcl_BTSetDateTimeStamp();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDateTimeStamp& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetChangeDeviceStateAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetChangeDeviceStateAction();
   enum tenType {
      FI_EN_E8SELECT = 0UL,
      FI_EN_E8DISCONNECT = 1UL,
      FI_EN_E8DELETE = 2UL,
      FI_EN_E8DELETE_ALL = 3UL,
      FI_EN_E8BLOCK = 4UL,
      FI_EN_E8UNBLOCK = 5UL,
      FI_EN_E8UNBLOCK_AUTO_CONNECT = 6UL,
      FI_EN_E8BLOCK_ALL_EXCEPT = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetChangeDeviceStateAction(most_fi_tcl_e8_BTSetChangeDeviceStateAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetChangeDeviceStateAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetChangeDeviceStateAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetConnectionResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetConnectionResult();
   enum tenType {
      FI_EN_E8DEVICE_CONNECTED = 0UL,
      FI_EN_E8DEVICE_NOT_CONNECTED = 1UL,
      FI_EN_E8DEVICE_DISCONNECTED = 2UL,
      FI_EN_E8NOT_APPLICABLE = 3UL,
      FI_EN_E8DEVICE_BLOCKED = 4UL,
      FI_EN_E8BLOCK_ALL_EXCEPT = 5UL,
      FI_EN_E8DEVICE_UNBLOCKED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetConnectionResult(most_fi_tcl_e8_BTSetConnectionResult::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetConnectionResult();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetConnectionResult& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceRingtoneListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceRingtoneListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceRingtoneListResultItem& operator=(const most_fi_tcl_BTSetDeviceRingtoneListResultItem& coRef);
   most_fi_tcl_BTSetDeviceRingtoneListResultItem(const most_fi_tcl_BTSetDeviceRingtoneListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_String sDeviceName;
   tU16 u16RingtoneID;
   most_fi_tcl_String sRingtoneName;
   virtual ~most_fi_tcl_BTSetDeviceRingtoneListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceRingtoneListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceRingtoneListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceRingtoneListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceRingtoneListResult& operator=(const most_fi_tcl_BTSetDeviceRingtoneListResult& coRef);
   most_fi_tcl_BTSetDeviceRingtoneListResult(const most_fi_tcl_BTSetDeviceRingtoneListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTSetDeviceRingtoneListResultItem*, std::allocator<most_fi_tcl_BTSetDeviceRingtoneListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTSetDeviceRingtoneListResultItem, std::allocator<most_fi_tcl_BTSetDeviceRingtoneListResultItem> >oItems;

   most_fi_tcl_BTSetDeviceRingtoneListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTSetDeviceRingtoneListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceRingtoneListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetRingtoneListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetRingtoneListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetRingtoneListItem& operator=(const most_fi_tcl_BTSetRingtoneListItem& coRef);
   most_fi_tcl_BTSetRingtoneListItem(const most_fi_tcl_BTSetRingtoneListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16RingtoneID;
   most_fi_tcl_String sRingtoneName;
   tBool bRingtoneAvailable;
   tBool bPlayRingtoneAvailable;
   tBool bActiveRingtone;
   virtual ~most_fi_tcl_BTSetRingtoneListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetRingtoneListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetRingtoneList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetRingtoneList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetRingtoneList& operator=(const most_fi_tcl_BTSetRingtoneList& coRef);
   most_fi_tcl_BTSetRingtoneList(const most_fi_tcl_BTSetRingtoneList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTSetRingtoneListItem*, std::allocator<most_fi_tcl_BTSetRingtoneListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTSetRingtoneListItem, std::allocator<most_fi_tcl_BTSetRingtoneListItem> >oItems;

   most_fi_tcl_BTSetRingtoneListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTSetRingtoneList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetRingtoneList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceControlSupport : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceControlSupport();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceControlSupport& operator=(const most_fi_tcl_BTSetDeviceControlSupport& coRef);
   most_fi_tcl_BTSetDeviceControlSupport(const most_fi_tcl_BTSetDeviceControlSupport& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bPlaySupport;
   tBool bPauseSupport;
   tBool bNextTrackSupport;
   tBool bPreviousTrackSupport;
   tBool bShuffleSupport;
   tBool bRepeatSupport;
   tBool bPlaytimeSupport;
   tBool bMetadataSupport;
   tBool bFolderBrowseSupport;
   tBool bMediaAggregrationSupport;
   tBool bFastForwardSupport;
   tBool bRewindSupport;
   virtual ~most_fi_tcl_BTSetDeviceControlSupport();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceControlSupport& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetAudioStreamingSupportType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetAudioStreamingSupportType();
   enum tenType {
      FI_EN_E8AVRCP_NO_METADATA = 0UL,
      FI_EN_E8AVRCP_METADATA = 1UL,
      FI_EN_E8AVRCP_BROWSING = 2UL,
      FI_EN_E8AVRCP_MYMEDIA = 3UL,
      FI_EN_E8DEFAULT = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetAudioStreamingSupportType(most_fi_tcl_e8_BTSetAudioStreamingSupportType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetAudioStreamingSupportType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetAudioStreamingSupportType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetInternetAccessSupportType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetInternetAccessSupportType();
   enum tenType {
      FI_EN_E8DUN_SUPPORTED = 0UL,
      FI_EN_E8PAN_SUPPORTED = 1UL,
      FI_EN_E8DUN_PAN_SUPPORTED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetInternetAccessSupportType(most_fi_tcl_e8_BTSetInternetAccessSupportType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetInternetAccessSupportType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetInternetAccessSupportType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetMajorDeviceClass : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetMajorDeviceClass();
   enum tenType {
      FI_EN_E8MAJOR_DEVICE_CLASS_PHONE = 0UL,
      FI_EN_E8MAJOR_DEVICE_CLASS_AUDIOVIDEO = 1UL,
      FI_EN_E8MAJOR_DEVICE_CLASS_COMPUTER = 2UL,
      FI_EN_E8MAJOR_DEVICE_CLASS_LANORNAP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetMajorDeviceClass(most_fi_tcl_e8_BTSetMajorDeviceClass::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetMajorDeviceClass();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetMajorDeviceClass& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetBluetoothFeatureSupport : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetBluetoothFeatureSupport();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetBluetoothFeatureSupport& operator=(const most_fi_tcl_BTSetBluetoothFeatureSupport& coRef);
   most_fi_tcl_BTSetBluetoothFeatureSupport(const most_fi_tcl_BTSetBluetoothFeatureSupport& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bHandsFreeSupport;
   tBool bPhoneBookSupport;
   tBool bAudioStreamingSupport;
   most_fi_tcl_e8_BTSetAudioStreamingSupportType e8AudioStreamingSupportType;
   tBool bMessagingSupport;
   tBool bInternetAccessSupport;
   most_fi_tcl_e8_BTSetInternetAccessSupportType e8InternetAccessSupportType;
   tBool bPortableNavigationSupport;
   tBool bSerialPortSupport;
   virtual ~most_fi_tcl_BTSetBluetoothFeatureSupport();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetBluetoothFeatureSupport& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetSupportedProfiles : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetSupportedProfiles();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetSupportedProfiles& operator=(const most_fi_tcl_BTSetSupportedProfiles& coRef);
   most_fi_tcl_BTSetSupportedProfiles(const most_fi_tcl_BTSetSupportedProfiles& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bSPPSupport;
   tBool bHFPSupport;
   tBool bA2DPSupport;
   tBool bAVRCPSupport;
   tBool bMAPSupport;
   tBool bDUNSupport;
   tBool bPANSupport;
   tBool bPBAPSuport;
   tBool bOPPSupport;
   tBool bDIPSupport;
   virtual ~most_fi_tcl_BTSetSupportedProfiles();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetSupportedProfiles& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceIdentification : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceIdentification();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceIdentification& operator=(const most_fi_tcl_BTSetDeviceIdentification& coRef);
   most_fi_tcl_BTSetDeviceIdentification(const most_fi_tcl_BTSetDeviceIdentification& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16VendorIDSource;
   tU16 u16VendorID;
   tU16 u16ProductID;
   tU16 u16ProductVersion;
   virtual ~most_fi_tcl_BTSetDeviceIdentification();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceIdentification& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetOrigin : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetOrigin();
   enum tenType {
      FI_EN_E8DEVICE = 1UL,
      FI_EN_E8SYSTEM = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetOrigin(most_fi_tcl_e8_BTSetOrigin::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetOrigin();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetOrigin& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetScanningState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetScanningState();
   enum tenType {
      FI_EN_E8SCANNING_STOPPED = 0UL,
      FI_EN_E8SCANNING_STARTED = 1UL,
      FI_EN_E8SCANNING_ALREADY_STOPPED = 2UL,
      FI_EN_E8SCANNING_ALREADY_STARTED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetScanningState(most_fi_tcl_e8_BTSetScanningState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetScanningState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetScanningState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTServiceType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTServiceType();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTServiceType& operator=(const most_fi_tcl_BTServiceType& coRef);
   most_fi_tcl_BTServiceType(const most_fi_tcl_BTServiceType& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bHFP;
   tBool bAVP;
   tBool bPAN;
   tBool bPBAP;
   tBool bMAP;
   virtual ~most_fi_tcl_BTServiceType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTServiceType& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTConnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTConnectionStatus();
   enum tenType {
      FI_EN_E8STATUS_NOT_CONNECT = 0UL,
      FI_EN_E8STATUS_CONNECTING = 1UL,
      FI_EN_E8STATUS_CONNECTED = 2UL,
      FI_EN_E8STATUS_DISCONNECTING = 3UL,
      FI_EN_E8STATUS_DISCONNECTED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_BTConnectionStatus(most_fi_tcl_e8_BTConnectionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTConnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTConnectionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetDeviceDisconnectedReason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetDeviceDisconnectedReason();
   enum tenType {
      FI_EN_E8NOT_APPLICABLE = 0UL,
      FI_EN_E8DEVICE_REQUEST = 1UL,
      FI_EN_E8MODULE_REQUEST = 2UL,
      FI_EN_E8AUTOMATIC = 3UL,
      FI_EN_E8OUT_OF_RANGE = 4UL,
      FI_EN_E8INTERMEDIATE = 5UL,
      FI_EN_E8CONNECT_TIMEOUT = 10UL,
      FI_EN_E8CONNECT_REJECTED = 11UL,
      FI_EN_E8PROTOCOL_NOT_SUPPORTED = 12UL,
      FI_EN_E8CONNECT_FAILED = 13UL,
      FI_EN_E8LINKKEY_MISSING = 14UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetDeviceDisconnectedReason(most_fi_tcl_e8_BTSetDeviceDisconnectedReason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetDeviceDisconnectedReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetDeviceDisconnectedReason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTProfileDisconnectedReason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTProfileDisconnectedReason();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTProfileDisconnectedReason& operator=(const most_fi_tcl_BTProfileDisconnectedReason& coRef);
   most_fi_tcl_BTProfileDisconnectedReason(const most_fi_tcl_BTProfileDisconnectedReason& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8HFP;
   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8AVP;
   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8PAN;
   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8PBAP;
   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8MAP;
   virtual ~most_fi_tcl_BTProfileDisconnectedReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTProfileDisconnectedReason& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTDisconnectionReason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTDisconnectionReason();
   enum tenType {
      FI_EN_E8NOT_APPLICABLE = 0UL,
      FI_EN_E8UUID_NOT_SUPPORTED = 1UL,
      FI_EN_E8CONNECTION_FAILED = 2UL,
      FI_EN_E8HMI_DISCONNECT = 3UL,
      FI_EN_E8LOST = 4UL,
      FI_EN_E8REMOTE_DISCONNECT = 5UL,
      FI_EN_E8AUTOMATIC = 6UL,
      FI_EN_E8INTERMEDIATE = 7UL,
      FI_EN_E8CONNECT_TIMEOUT = 8UL,
      FI_EN_E8CONNECT_REJECTED = 9UL,
      FI_EN_E8NONE = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_BTDisconnectionReason(most_fi_tcl_e8_BTDisconnectionReason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTDisconnectionReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTDisconnectionReason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSPPServiceConnectionStatusListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSPPServiceConnectionStatusListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSPPServiceConnectionStatusListItem& operator=(const most_fi_tcl_BTSPPServiceConnectionStatusListItem& coRef);
   most_fi_tcl_BTSPPServiceConnectionStatusListItem(const most_fi_tcl_BTSPPServiceConnectionStatusListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_String sUUID;
   most_fi_tcl_e8_BTConnectionStatus e8SPPConnectionStatus;
   most_fi_tcl_e8_BTDisconnectionReason e8SPPDisconnectionReason;
   most_fi_tcl_String sRFCommID;
   virtual ~most_fi_tcl_BTSPPServiceConnectionStatusListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSPPServiceConnectionStatusListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTProfileConnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTProfileConnectionStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTProfileConnectionStatus& operator=(const most_fi_tcl_BTProfileConnectionStatus& coRef);
   most_fi_tcl_BTProfileConnectionStatus(const most_fi_tcl_BTProfileConnectionStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_BTConnectionStatus e8HFP;
   most_fi_tcl_e8_BTConnectionStatus e8AVP;
   most_fi_tcl_e8_BTConnectionStatus e8PAN;
   most_fi_tcl_e8_BTConnectionStatus e8PBAP;
   most_fi_tcl_e8_BTConnectionStatus e8MAP;
   virtual ~most_fi_tcl_BTProfileConnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTProfileConnectionStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSPPServiceConnectionStatusList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSPPServiceConnectionStatusList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSPPServiceConnectionStatusList& operator=(const most_fi_tcl_BTSPPServiceConnectionStatusList& coRef);
   most_fi_tcl_BTSPPServiceConnectionStatusList(const most_fi_tcl_BTSPPServiceConnectionStatusList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTSPPServiceConnectionStatusListItem*, std::allocator<most_fi_tcl_BTSPPServiceConnectionStatusListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTSPPServiceConnectionStatusListItem, std::allocator<most_fi_tcl_BTSPPServiceConnectionStatusListItem> >oItems;

   most_fi_tcl_BTSPPServiceConnectionStatusListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTSPPServiceConnectionStatusList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSPPServiceConnectionStatusList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTUsagePreference : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTUsagePreference();
   enum tenType {
      FI_EN_E8UNDECIDED = 0UL,
      FI_EN_E8TEMP_CLASSIC_BT = 1UL,
      FI_EN_E8CLASSIC_BT = 2UL,
      FI_EN_E8CPW = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTUsagePreference(most_fi_tcl_e8_BTUsagePreference::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTUsagePreference();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTUsagePreference& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTProfileFavoriteIndex : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTProfileFavoriteIndex();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTProfileFavoriteIndex& operator=(const most_fi_tcl_BTProfileFavoriteIndex& coRef);
   most_fi_tcl_BTProfileFavoriteIndex(const most_fi_tcl_BTProfileFavoriteIndex& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8HFP;
   tU8 u8AVP;
   virtual ~most_fi_tcl_BTProfileFavoriteIndex();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTProfileFavoriteIndex& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus();
   enum tenType {
      FI_EN_E8UNKNOWN = 0UL,
      FI_EN_E8NEW = 1UL,
      FI_EN_E8PAIRED = 2UL,
      FI_EN_E8CONNECTED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus(most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetVisibleDeviceItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetVisibleDeviceItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetVisibleDeviceItem& operator=(const most_fi_tcl_BTSetVisibleDeviceItem& coRef);
   most_fi_tcl_BTSetVisibleDeviceItem(const most_fi_tcl_BTSetVisibleDeviceItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBluetoothDeviceAddress;
   most_fi_tcl_String sDeviceName;
   most_fi_tcl_BTSetSupportedProfiles oSupportedProfiles;
   most_fi_tcl_e8_BTSetVisibleDeviceConnectionStatus e8VisibleDeviceConnectionStatus;
   virtual ~most_fi_tcl_BTSetVisibleDeviceItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetVisibleDeviceItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetSignalStrengthItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetSignalStrengthItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetSignalStrengthItem& operator=(const most_fi_tcl_BTSetSignalStrengthItem& coRef);
   most_fi_tcl_BTSetSignalStrengthItem(const most_fi_tcl_BTSetSignalStrengthItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tS8 s8SignalStrength;
   virtual ~most_fi_tcl_BTSetSignalStrengthItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetSignalStrengthItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetSignalStrengthList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetSignalStrengthList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetSignalStrengthList& operator=(const most_fi_tcl_BTSetSignalStrengthList& coRef);
   most_fi_tcl_BTSetSignalStrengthList(const most_fi_tcl_BTSetSignalStrengthList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_BTSetSignalStrengthItem, std::allocator<most_fi_tcl_BTSetSignalStrengthItem> >oSignalStrengthItems;

   most_fi_tcl_BTSetSignalStrengthItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTSetSignalStrengthList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetSignalStrengthList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetVisibleDevicesList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetVisibleDevicesList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetVisibleDevicesList& operator=(const most_fi_tcl_BTSetVisibleDevicesList& coRef);
   most_fi_tcl_BTSetVisibleDevicesList(const most_fi_tcl_BTSetVisibleDevicesList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTSetVisibleDeviceItem*, std::allocator<most_fi_tcl_BTSetVisibleDeviceItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTSetVisibleDeviceItem, std::allocator<most_fi_tcl_BTSetVisibleDeviceItem> >oVisibleDeviceItems;

   most_fi_tcl_BTSetVisibleDeviceItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTSetVisibleDevicesList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetVisibleDevicesList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceListChangeExtended : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceListChangeExtended();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceListChangeExtended& operator=(const most_fi_tcl_BTSetDeviceListChangeExtended& coRef);
   most_fi_tcl_BTSetDeviceListChangeExtended(const most_fi_tcl_BTSetDeviceListChangeExtended& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_e8_BTSetDeviceStatus e8DeviceStatus;
   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8DeviceDisconnectedReason;
   virtual ~most_fi_tcl_BTSetDeviceListChangeExtended();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceListChangeExtended& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceListExtendedResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceListExtendedResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceListExtendedResultItem& operator=(const most_fi_tcl_BTSetDeviceListExtendedResultItem& coRef);
   most_fi_tcl_BTSetDeviceListExtendedResultItem(const most_fi_tcl_BTSetDeviceListExtendedResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sDeviceAddress;
   tU8 u8DeviceHandle;
   most_fi_tcl_String sDeviceName;
   tBool bDeviceConnectedStatus;
   most_fi_tcl_e8_BTSetDeviceDisconnectedReason e8DeviceDisconnectedReason;
   most_fi_tcl_BTServiceType oDeviceProfileStatus;
   most_fi_tcl_BTProfileDisconnectedReason oDeviceProfileDisconnectedReason;
   most_fi_tcl_BTSetSupportedProfiles oDeviceSupportedProfiles;
   tBool bOutgoingSourceDeviceStatus;
   most_fi_tcl_BTSetDateTimeStamp oDateTimeStamp;
   most_fi_tcl_BTSetDeviceIdentification oDeviceInfo;
   most_fi_tcl_e8_BTConnectionStatus e8DeviceConnectionStatus;
   most_fi_tcl_BTProfileConnectionStatus oDeviceProfileConnectionStatus;
   tU8 u8DeviceFavoriteIndex;
   most_fi_tcl_BTProfileFavoriteIndex oDeviceProfileFavoriteIndex;
   most_fi_tcl_e8_BTUsagePreference e8DeviceUsagePreference;
   most_fi_tcl_BTSetDateTimeStamp oDevicePairedDateTimeStamp;
   tBool biAPoverBTSupport;
   virtual ~most_fi_tcl_BTSetDeviceListExtendedResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceListExtendedResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetPairingResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetPairingResult();
   enum tenType {
      FI_EN_E8PAIRING_STOPPED = 0UL,
      FI_EN_E8PAIRING_STARTED = 1UL,
      FI_EN_E8PAIRING_ALREADY_STOPPED = 2UL,
      FI_EN_E8PAIRING_ALREADY_STARTED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetPairingResult(most_fi_tcl_e8_BTSetPairingResult::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetPairingResult();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetPairingResult& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetPairableModeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetPairableModeType();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8PAIRABLE = 1UL,
      FI_EN_E8PAIRING_IN_PROGRESS = 2UL
   };
   tenType enType;

   most_fi_tcl_BTSetPairableModeType(most_fi_tcl_BTSetPairableModeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_BTSetPairableModeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_BTSetPairableModeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTLocalModeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTLocalModeType();
   enum tenType {
      FI_EN_E8ON = 1UL,
      FI_EN_E8OFF = 2UL,
      FI_EN_E8AUTO = 3UL
   };
   tenType enType;

   most_fi_tcl_BTLocalModeType(most_fi_tcl_BTLocalModeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_BTLocalModeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_BTLocalModeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTTechnology : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTTechnology();
   enum tenType {
      FI_EN_IAP = 1UL,
      FI_EN_CAR_PLAY = 2UL,
      FI_EN_ANDROID_AUTO = 3UL,
      FI_EN_MY_SPIN = 4UL,
      FI_EN_MY_SPIN_IOS = 5UL,
      FI_EN_MIRROR_LINK = 6UL,
      FI_EN_MIRROR_LINK_IOS = 7UL,
      FI_EN_CAR_LIFE = 8UL,
      FI_EN_CAR_LIFE_IOS = 9UL,
      FI_EN_ON_CAR = 10UL,
      FI_EN_ON_CAR_IOS = 11UL
   };
   tenType enType;

   most_fi_tcl_e8_BTTechnology(most_fi_tcl_e8_BTTechnology::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTTechnology();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTTechnology& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTCommunicationChannel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTCommunicationChannel();
   enum tenType {
      FI_EN_USB = 1UL,
      FI_EN_WIFI_2_4GHZ = 2UL,
      FI_EN_WIFI_5GHZ = 3UL,
      FI_EN_BLUETOOTH = 4UL,
      FI_EN_WIFI = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_BTCommunicationChannel(most_fi_tcl_e8_BTCommunicationChannel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTCommunicationChannel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTCommunicationChannel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTLimitationAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTLimitationAction();
   enum tenType {
      FI_EN_PREPARE = 1UL,
      FI_EN_ACTIVATE = 2UL,
      FI_EN_DEACTIVATE = 3UL,
      FI_EN_DEACTIVATE_WAIT = 4UL,
      FI_EN_DEACTIVATE_INTERNAL = 5UL,
      FI_EN_DEACTIVATE_SPMSTATE_OFF = 6UL,
      FI_EN_PREACTIVATE = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_BTLimitationAction(most_fi_tcl_e8_BTLimitationAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTLimitationAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTLimitationAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTLimitationActionState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTLimitationActionState();
   enum tenType {
      FI_EN_IDLE = 0UL,
      FI_EN_PREPARING = 1UL,
      FI_EN_PREPARED = 2UL,
      FI_EN_ACTIVATING = 3UL,
      FI_EN_ACTIVE = 4UL,
      FI_EN_DEACTIVATING = 5UL,
      FI_EN_WAITING = 6UL,
      FI_EN_ERROR = 7UL,
      FI_EN_ERROR_USER_DENIED = 8UL,
      FI_EN_PREACTIVATING = 9UL,
      FI_EN_PREACTIVATED = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_BTLimitationActionState(most_fi_tcl_e8_BTLimitationActionState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTLimitationActionState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTLimitationActionState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTLimitationModeListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTLimitationModeListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTLimitationModeListItem& operator=(const most_fi_tcl_BTLimitationModeListItem& coRef);
   most_fi_tcl_BTLimitationModeListItem(const most_fi_tcl_BTLimitationModeListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBdAddress;
   most_fi_tcl_String sBdDeviceName;
   most_fi_tcl_e8_BTTechnology e8Technology;
   most_fi_tcl_e8_BTCommunicationChannel e8CommunicationChannel;
   most_fi_tcl_e8_BTLimitationActionState e8ActionState;
   virtual ~most_fi_tcl_BTLimitationModeListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTLimitationModeListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTLimitationModeList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTLimitationModeList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTLimitationModeList& operator=(const most_fi_tcl_BTLimitationModeList& coRef);
   most_fi_tcl_BTLimitationModeList(const most_fi_tcl_BTLimitationModeList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTLimitationModeListItem*, std::allocator<most_fi_tcl_BTLimitationModeListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTLimitationModeListItem, std::allocator<most_fi_tcl_BTLimitationModeListItem> >oItems;

   most_fi_tcl_BTLimitationModeListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTLimitationModeList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTLimitationModeList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTLimitationMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTLimitationMode();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTLimitationMode& operator=(const most_fi_tcl_BTLimitationMode& coRef);
   most_fi_tcl_BTLimitationMode(const most_fi_tcl_BTLimitationMode& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBdAddress;
   most_fi_tcl_String sBdDeviceName;
   most_fi_tcl_e8_BTTechnology e8Technology;
   most_fi_tcl_e8_BTCommunicationChannel e8CommunicationChannel;
   most_fi_tcl_e8_BTLimitationAction e8Action;
   virtual ~most_fi_tcl_BTLimitationMode();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTLimitationMode& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTOobType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTOobType();
   enum tenType {
      FI_EN_GENERIC = 0UL,
      FI_EN_APPLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BTOobType(most_fi_tcl_e8_BTOobType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTOobType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTOobType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTFavoriteType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTFavoriteType();
   enum tenType {
      FI_EN_DEVICE = 0UL,
      FI_EN_PROTOCOL_HFP = 1UL,
      FI_EN_PROTOCOL_AVP = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTFavoriteType(most_fi_tcl_e8_BTFavoriteType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTFavoriteType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTFavoriteType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTSetDeviceListExtendedResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTSetDeviceListExtendedResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTSetDeviceListExtendedResult& operator=(const most_fi_tcl_BTSetDeviceListExtendedResult& coRef);
   most_fi_tcl_BTSetDeviceListExtendedResult(const most_fi_tcl_BTSetDeviceListExtendedResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem*, std::allocator<most_fi_tcl_BTSetDeviceListExtendedResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem, std::allocator<most_fi_tcl_BTSetDeviceListExtendedResultItem> >oItems;

   most_fi_tcl_BTSetDeviceListExtendedResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTSetDeviceListExtendedResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTSetDeviceListExtendedResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSwitchStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSwitchStatus();
   enum tenType {
      FI_EN_E8SWITCHED_OFF = 0UL,
      FI_EN_E8SWITCHING_ON = 1UL,
      FI_EN_E8SWITCHED_ON = 2UL,
      FI_EN_E8SWITCHING_OFF = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSwitchStatus(most_fi_tcl_e8_BTSwitchStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSwitchStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSwitchStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTProfile : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTProfile();
   enum tenType {
      FI_EN_E8BT_PROFILE_NONE = 0UL,
      FI_EN_E8BT_PROFILE_HFP = 1UL,
      FI_EN_E8BT_PROFILE_AVP = 2UL,
      FI_EN_E8BT_PROFILE_PAN = 3UL,
      FI_EN_E8BT_PROFILE_PBAP = 4UL,
      FI_EN_E8BT_PROFILE_MAP = 5UL,
      FI_EN_E8BT_PROFILE_SPP = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_BTProfile(most_fi_tcl_e8_BTProfile::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTProfile();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTProfile& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTProfileUsageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTProfileUsageType();
   enum tenType {
      FI_EN_E8PROFILE_USAGE_NONE = 0UL,
      FI_EN_E8PROFILE_USAGE_GENERIC = 1UL,
      FI_EN_E8PROFILE_USAGE_ACTIVE_CALL = 2UL,
      FI_EN_E8PROFILE_USAGE_AUDIO_STREAMING = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BTProfileUsageType(most_fi_tcl_e8_BTProfileUsageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTProfileUsageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTProfileUsageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTProfilesUsageInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTProfilesUsageInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTProfilesUsageInfo& operator=(const most_fi_tcl_BTProfilesUsageInfo& coRef);
   most_fi_tcl_BTProfilesUsageInfo(const most_fi_tcl_BTProfilesUsageInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_BTProfile e8BTProfile;
   most_fi_tcl_String sUUID;
   most_fi_tcl_e8_BTProfileUsageType e8BTProfileUsageType;
   tBool bInUse;
   virtual ~most_fi_tcl_BTProfilesUsageInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTProfilesUsageInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTProfilesUsageInfoList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTProfilesUsageInfoList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTProfilesUsageInfoList& operator=(const most_fi_tcl_BTProfilesUsageInfoList& coRef);
   most_fi_tcl_BTProfilesUsageInfoList(const most_fi_tcl_BTProfilesUsageInfoList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_BTProfilesUsageInfo oBTProfilesUsageInfo;
   virtual ~most_fi_tcl_BTProfilesUsageInfoList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTProfilesUsageInfoList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTProfilesUsageList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTProfilesUsageList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTProfilesUsageList& operator=(const most_fi_tcl_BTProfilesUsageList& coRef);
   most_fi_tcl_BTProfilesUsageList(const most_fi_tcl_BTProfilesUsageList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTProfilesUsageInfoList*, std::allocator<most_fi_tcl_BTProfilesUsageInfoList*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTProfilesUsageInfoList, std::allocator<most_fi_tcl_BTProfilesUsageInfoList> >oItems;

   most_fi_tcl_BTProfilesUsageInfoList& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTProfilesUsageList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTProfilesUsageList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTResolutionType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTResolutionType();
   enum tenType {
      FI_EN_E8RESOLUTION_TYPE_PROCEED = 0UL,
      FI_EN_E8RESOLUTION_TYPE_CANCEL = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BTResolutionType(most_fi_tcl_e8_BTResolutionType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTResolutionType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTResolutionType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTConflictTrigger : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTConflictTrigger();
   enum tenType {
      FI_EN_E8NONE = 0UL,
      FI_EN_E8BT_PAIRING = 1UL,
      FI_EN_E8OOB_BT_PAIRING = 2UL,
      FI_EN_E8AAP_USB = 3UL,
      FI_EN_E8AAP_WIRELESS = 4UL,
      FI_EN_E8CP_USB = 5UL,
      FI_EN_E8CP_WIRELESS = 6UL,
      FI_EN_E8CARLIFE_USB = 7UL,
      FI_EN_E8CARLIFE_IOS_USB = 8UL,
      FI_EN_E8ONCAR_USB = 9UL,
      FI_EN_E8ONCAR_IOS_USB = 10UL,
      FI_EN_E8MIRRORLINK_USB = 11UL,
      FI_EN_E8MIRRORLINK_IOS_USB = 12UL,
      FI_EN_E8MYSPIN_USB = 13UL,
      FI_EN_E8MYSPIN_IOS_USB = 14UL
   };
   tenType enType;

   most_fi_tcl_e8_BTConflictTrigger(most_fi_tcl_e8_BTConflictTrigger::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTConflictTrigger();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTConflictTrigger& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTConflictType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTConflictType();
   enum tenType {
      FI_EN_E8GENERIC = 0UL,
      FI_EN_E8MAX_NUM_OF_PAIRED_DEVICES = 1UL,
      FI_EN_E8PROFILE_IN_USE = 2UL,
      FI_EN_E8INCORRECT_WIFI_MODE = 3UL,
      FI_EN_E8CHECK_MAX_NUM_OF_PAIRED_DEVICES = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_BTConflictType(most_fi_tcl_e8_BTConflictType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTConflictType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTConflictType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTConflictState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTConflictState();
   enum tenType {
      FI_EN_E8CONFLICT_STATE_NEW = 0UL,
      FI_EN_E8CONFLICT_STATE_UNCHANGED = 1UL,
      FI_EN_E8CONFLICT_STATE_SOLVED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTConflictState(most_fi_tcl_e8_BTConflictState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTConflictState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTConflictState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTConflictInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTConflictInfo();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTConflictInfo& operator=(const most_fi_tcl_BTConflictInfo& coRef);
   most_fi_tcl_BTConflictInfo(const most_fi_tcl_BTConflictInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_BTConflictType e8ConflictType;
   tU8 u8DeviceHandle;
   most_fi_tcl_e8_BTProfile e8BTProfile;
   most_fi_tcl_String sUUID;
   most_fi_tcl_e8_BTProfileUsageType e8BTProfileUsageType;
   most_fi_tcl_e8_BTConflictState e8ConflictState;
   virtual ~most_fi_tcl_BTConflictInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTConflictInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_BTConflictInfoList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_BTConflictInfoList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_BTConflictInfoList& operator=(const most_fi_tcl_BTConflictInfoList& coRef);
   most_fi_tcl_BTConflictInfoList(const most_fi_tcl_BTConflictInfoList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_BTConflictInfo*, std::allocator<most_fi_tcl_BTConflictInfo*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_BTConflictInfo, std::allocator<most_fi_tcl_BTConflictInfo> >oItems;

   most_fi_tcl_BTConflictInfo& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_BTConflictInfoList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_BTConflictInfoList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTSetUsagePreference : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTSetUsagePreference();
   enum tenType {
      FI_EN_E8CLASSIC_BT = 0UL,
      FI_EN_E8CPW = 1UL,
      FI_EN_E8TEMP_CLASSIC_BT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTSetUsagePreference(most_fi_tcl_e8_BTSetUsagePreference::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTSetUsagePreference();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTSetUsagePreference& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTAutoconnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTAutoconnectionStatus();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8PREPARING = 1UL,
      FI_EN_E8IN_PROGRESS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTAutoconnectionStatus(most_fi_tcl_e8_BTAutoconnectionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTAutoconnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTAutoconnectionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTUserAvailable : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTUserAvailable();
   enum tenType {
      FI_EN_E8USER_NOT_AVAILABLE = 0UL,
      FI_EN_E8USER_AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BTUserAvailable(most_fi_tcl_e8_BTUserAvailable::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTUserAvailable();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTUserAvailable& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BTServiceAvailable : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BTServiceAvailable();
   enum tenType {
      FI_EN_E8SERVICE_UNAVAILABLE = 0UL,
      FI_EN_E8SERVICE_AVAILABLE_RESTRICTED = 1UL,
      FI_EN_E8SERVICE_AVAILABLE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BTServiceAvailable(most_fi_tcl_e8_BTServiceAvailable::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BTServiceAvailable();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BTServiceAvailable& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgFolderType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgFolderType();
   enum tenType {
      FI_EN_E8MSG_FOLDER_INBOX = 0UL,
      FI_EN_E8MSG_FOLDER_DELETED = 1UL,
      FI_EN_E8MSG_FOLDER_DRAFTS = 2UL,
      FI_EN_E8MSG_FOLDER_SENT = 3UL,
      FI_EN_E8MSG_FOLDER_USERDEFINED = 4UL,
      FI_EN_E8MSG_FOLDER_INACCESSIBLE = 5UL,
      FI_EN_E8MSG_FOLDER_OUTBOX = 6UL,
      FI_EN_E8MSG_FOLDER_CONVERSATION = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgFolderType(most_fi_tcl_e8_MsgFolderType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgFolderType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgFolderType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgMessageListFilterType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgMessageListFilterType();
   enum tenType {
      FI_EN_E8MSG_LIST_FILTER_ALL = 0UL,
      FI_EN_E8MSG_LIST_FILTER_UNREAD = 1UL,
      FI_EN_E8MSG_LIST_FILTER_CURRENT_MAP_SESSION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgMessageListFilterType(most_fi_tcl_e8_MsgMessageListFilterType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgMessageListFilterType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgMessageListFilterType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgMessageListSortType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgMessageListSortType();
   enum tenType {
      FI_EN_E8MSG_LIST_SORT_TIMESTAMP = 0UL,
      FI_EN_E8MSG_LIST_SORT_LASTNAME = 1UL,
      FI_EN_E8MSG_LIST_SORT_FIRSTNAME = 2UL,
      FI_EN_E8MSG_LIST_SORT_ADDRESS = 3UL,
      FI_EN_E8MSG_LIST_SORT_PHONENUMBER = 4UL,
      FI_EN_E8MSG_LIST_SORT_CONVERSATION = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgMessageListSortType(most_fi_tcl_e8_MsgMessageListSortType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgMessageListSortType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgMessageListSortType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgMessageListType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgMessageListType();
   enum tenType {
      FI_EN_E8MSG_LIST_SMS_MMS = 0UL,
      FI_EN_E8MSG_LIST_EMAIL = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgMessageListType(most_fi_tcl_e8_MsgMessageListType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgMessageListType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgMessageListType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgMessageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgMessageType();
   enum tenType {
      FI_EN_E8MSG_TYPE_SMS = 0UL,
      FI_EN_E8MSG_TYPE_MMS = 1UL,
      FI_EN_E8MSG_TYPE_EMAIL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgMessageType(most_fi_tcl_e8_MsgMessageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgMessageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgMessageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgSendMessageResultType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgSendMessageResultType();
   enum tenType {
      FI_EN_E8DELIVERY_SUCCESS = 0UL,
      FI_EN_E8DELIVERY_FAILURE = 1UL,
      FI_EN_E8SENDING_SUCCESS = 2UL,
      FI_EN_E8SENDING_FAILURE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgSendMessageResultType(most_fi_tcl_e8_MsgSendMessageResultType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgSendMessageResultType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgSendMessageResultType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgPlaybackState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgPlaybackState();
   enum tenType {
      FI_EN_E8PBS_PLAYING = 0UL,
      FI_EN_E8PBS_PAUSED = 1UL,
      FI_EN_E8PBS_STOPPED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgPlaybackState(most_fi_tcl_e8_MsgPlaybackState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgPlaybackState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgPlaybackState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageHandle : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageHandle();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageHandle& operator=(const most_fi_tcl_MsgMessageHandle& coRef);
   most_fi_tcl_MsgMessageHandle(const most_fi_tcl_MsgMessageHandle& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleUpper;
   tU32 u32MsgHandleLower;
   virtual ~most_fi_tcl_MsgMessageHandle();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageHandle& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageDate : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageDate();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageDate& operator=(const most_fi_tcl_MsgMessageDate& coRef);
   most_fi_tcl_MsgMessageDate(const most_fi_tcl_MsgMessageDate& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sCldrYear;
   most_fi_tcl_String sCldrMonth;
   most_fi_tcl_String sCldrDay;
   virtual ~most_fi_tcl_MsgMessageDate();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageDate& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageTime : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageTime& operator=(const most_fi_tcl_MsgMessageTime& coRef);
   most_fi_tcl_MsgMessageTime(const most_fi_tcl_MsgMessageTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sHours;
   most_fi_tcl_String sMinutes;
   most_fi_tcl_String sSeconds;
   virtual ~most_fi_tcl_MsgMessageTime();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageDateTime : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageDateTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageDateTime& operator=(const most_fi_tcl_MsgMessageDateTime& coRef);
   most_fi_tcl_MsgMessageDateTime(const most_fi_tcl_MsgMessageDateTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_MsgMessageDate oMessageDate;
   most_fi_tcl_MsgMessageTime oMessageTime;
   virtual ~most_fi_tcl_MsgMessageDateTime();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageDateTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageListSliceResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageListSliceResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageListSliceResultItem& operator=(const most_fi_tcl_MsgMessageListSliceResultItem& coRef);
   most_fi_tcl_MsgMessageListSliceResultItem(const most_fi_tcl_MsgMessageListSliceResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_MsgMessageHandle oMessageHandle;
   most_fi_tcl_MsgMessageDateTime oMessageDateTime;
   most_fi_tcl_String sSubject;
   most_fi_tcl_String sShortMessage;
   tBool bAttachmentFlag;
   most_fi_tcl_e8_MsgMessageType e8MessageType;
   most_fi_tcl_e8_MsgSendMessageResultType e8SendMessageResultType;
   tBool bReadStatusFlag;
   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sEmailAddress;
   most_fi_tcl_String sPhoneNumber;
   tU32 u32ContactHandle;
   most_fi_tcl_e8_MsgPlaybackState e8PlaybackState;
   virtual ~most_fi_tcl_MsgMessageListSliceResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageListSliceResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageListSliceResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageListSliceResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageListSliceResult& operator=(const most_fi_tcl_MsgMessageListSliceResult& coRef);
   most_fi_tcl_MsgMessageListSliceResult(const most_fi_tcl_MsgMessageListSliceResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgMessageListSliceResultItem*, std::allocator<most_fi_tcl_MsgMessageListSliceResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgMessageListSliceResultItem, std::allocator<most_fi_tcl_MsgMessageListSliceResultItem> >oItems;

   most_fi_tcl_MsgMessageListSliceResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgMessageListSliceResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageListSliceResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgListChangeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgListChangeType();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgListChangeType(most_fi_tcl_e8_MsgListChangeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgListChangeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgListChangeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgChangeItems : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgChangeItems() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgChangeItems& operator=(const most_fi_tcl_MsgChangeItems& coRef);
   most_fi_tcl_MsgChangeItems(const most_fi_tcl_MsgChangeItems& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgChangeItems();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgChangeItems& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgFolderListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgFolderListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgFolderListResultItem& operator=(const most_fi_tcl_MsgFolderListResultItem& coRef);
   most_fi_tcl_MsgFolderListResultItem(const most_fi_tcl_MsgFolderListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32FolderHandle;
   most_fi_tcl_String sFolderName;
   most_fi_tcl_e8_MsgFolderType e8FolderType;
   tU8 u8NumSubfolders;
   tU16 u16NumMessages;
   tU16 u16NumUnreadMessages;
   virtual ~most_fi_tcl_MsgFolderListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgFolderListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgFolderListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgFolderListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgFolderListResult& operator=(const most_fi_tcl_MsgFolderListResult& coRef);
   most_fi_tcl_MsgFolderListResult(const most_fi_tcl_MsgFolderListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgFolderListResultItem*, std::allocator<most_fi_tcl_MsgFolderListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgFolderListResultItem, std::allocator<most_fi_tcl_MsgFolderListResultItem> >oItems;

   most_fi_tcl_MsgFolderListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgFolderListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgFolderListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgDeletionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgDeletionStatus();
   enum tenType {
      FI_EN_E8MSG_DELETE_SUCCESS = 0UL,
      FI_EN_E8MSG_DELETE_UNSUCCESSFUL = 1UL,
      FI_EN_E8MSG_DELETE_FAIL_DEVICE_UNAVAILABLE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgDeletionStatus(most_fi_tcl_e8_MsgDeletionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgDeletionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgDeletionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgDeleteMessageHandles : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgDeleteMessageHandles() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgDeleteMessageHandles& operator=(const most_fi_tcl_MsgDeleteMessageHandles& coRef);
   most_fi_tcl_MsgDeleteMessageHandles(const most_fi_tcl_MsgDeleteMessageHandles& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_ShortStream*, std::allocator<most_fi_tcl_ShortStream*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_ShortStream, std::allocator<most_fi_tcl_ShortStream> >oItems;

   most_fi_tcl_ShortStream& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgDeleteMessageHandles();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgDeleteMessageHandles& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgRecoveryStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgRecoveryStatus();
   enum tenType {
      FI_EN_E8MSG_RECOVER_SUCCESS = 0UL,
      FI_EN_E8MSG_RECOVER_UNSUCCESSFUL = 1UL,
      FI_EN_E8MSG_RECOVER_FAIL_DEVICE_UNAVAILABLE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgRecoveryStatus(most_fi_tcl_e8_MsgRecoveryStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgRecoveryStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgRecoveryStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAccountListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAccountListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAccountListResultItem& operator=(const most_fi_tcl_MsgAccountListResultItem& coRef);
   most_fi_tcl_MsgAccountListResultItem(const most_fi_tcl_MsgAccountListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16AccountHandle;
   most_fi_tcl_String sAccountName;
   tBool bActiveAccountStatus;
   virtual ~most_fi_tcl_MsgAccountListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAccountListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAccountListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAccountListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAccountListResult& operator=(const most_fi_tcl_MsgAccountListResult& coRef);
   most_fi_tcl_MsgAccountListResult(const most_fi_tcl_MsgAccountListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgAccountListResultItem*, std::allocator<most_fi_tcl_MsgAccountListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgAccountListResultItem, std::allocator<most_fi_tcl_MsgAccountListResultItem> >oItems;

   most_fi_tcl_MsgAccountListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgAccountListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAccountListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgActiveAccount : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgActiveAccount();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgActiveAccount& operator=(const most_fi_tcl_MsgActiveAccount& coRef);
   most_fi_tcl_MsgActiveAccount(const most_fi_tcl_MsgActiveAccount& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16AccountHandle;
   most_fi_tcl_String sAccountName;
   virtual ~most_fi_tcl_MsgActiveAccount();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgActiveAccount& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgAddressFieldType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgAddressFieldType();
   enum tenType {
      FI_EN_E8ADDRESS_FIELD_FROM = 0UL,
      FI_EN_E8ADDRESS_FIELD_TO = 1UL,
      FI_EN_E8ADDRESS_FIELD_CC = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgAddressFieldType(most_fi_tcl_e8_MsgAddressFieldType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgAddressFieldType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgAddressFieldType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMsgBodyActionElementTags : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMsgBodyActionElementTags() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMsgBodyActionElementTags& operator=(const most_fi_tcl_MsgMsgBodyActionElementTags& coRef);
   most_fi_tcl_MsgMsgBodyActionElementTags(const most_fi_tcl_MsgMsgBodyActionElementTags& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgMsgBodyActionElementTags();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMsgBodyActionElementTags& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAddressFieldItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAddressFieldItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAddressFieldItem& operator=(const most_fi_tcl_MsgAddressFieldItem& coRef);
   most_fi_tcl_MsgAddressFieldItem(const most_fi_tcl_MsgAddressFieldItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_MsgAddressFieldType e8AddressFieldType;
   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sEmailAddress;
   most_fi_tcl_String sPhoneNumber;
   virtual ~most_fi_tcl_MsgAddressFieldItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAddressFieldItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAddressField : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAddressField() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAddressField& operator=(const most_fi_tcl_MsgAddressField& coRef);
   most_fi_tcl_MsgAddressField(const most_fi_tcl_MsgAddressField& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgAddressFieldItem*, std::allocator<most_fi_tcl_MsgAddressFieldItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgAddressFieldItem, std::allocator<most_fi_tcl_MsgAddressFieldItem> >oItems;

   most_fi_tcl_MsgAddressFieldItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgAddressField();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAddressField& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageDetailsResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageDetailsResult();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageDetailsResult& operator=(const most_fi_tcl_MsgMessageDetailsResult& coRef);
   most_fi_tcl_MsgMessageDetailsResult(const most_fi_tcl_MsgMessageDetailsResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_MsgMessageDateTime oMessageDateTime;
   most_fi_tcl_String sSubject;
   most_fi_tcl_String sShortMessage;
   tU8 u8NumAttachments;
   tU8 u8NumBodyParts;
   most_fi_tcl_MsgMsgBodyActionElementTags oMsgBodyActionElementTags;
   most_fi_tcl_MsgAddressField oAddressField;
   virtual ~most_fi_tcl_MsgMessageDetailsResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageDetailsResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgAttachmentType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgAttachmentType();
   enum tenType {
      FI_EN_E8MSG_ATT_TYPE_UNSUPPORTED = 0UL,
      FI_EN_E8MSG_ATT_TYPE_AUDIO = 1UL,
      FI_EN_E8MSG_ATT_TYPE_IMAGE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgAttachmentType(most_fi_tcl_e8_MsgAttachmentType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgAttachmentType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgAttachmentType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAttachmentListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAttachmentListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAttachmentListItem& operator=(const most_fi_tcl_MsgAttachmentListItem& coRef);
   most_fi_tcl_MsgAttachmentListItem(const most_fi_tcl_MsgAttachmentListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16AttachmentHandle;
   most_fi_tcl_e8_MsgAttachmentType e8AttachmentType;
   most_fi_tcl_String sFilename;
   virtual ~most_fi_tcl_MsgAttachmentListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAttachmentListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAttachmentList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAttachmentList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAttachmentList& operator=(const most_fi_tcl_MsgAttachmentList& coRef);
   most_fi_tcl_MsgAttachmentList(const most_fi_tcl_MsgAttachmentList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgAttachmentListItem*, std::allocator<most_fi_tcl_MsgAttachmentListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgAttachmentListItem, std::allocator<most_fi_tcl_MsgAttachmentListItem> >oItems;

   most_fi_tcl_MsgAttachmentListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgAttachmentList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAttachmentList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgPlaybackAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgPlaybackAction();
   enum tenType {
      FI_EN_E8PBA_PLAY = 0UL,
      FI_EN_E8PBA_PAUSE = 1UL,
      FI_EN_E8PBA_STOP = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgPlaybackAction(most_fi_tcl_e8_MsgPlaybackAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgPlaybackAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgPlaybackAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgBodyPartListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgBodyPartListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgBodyPartListItem& operator=(const most_fi_tcl_MsgBodyPartListItem& coRef);
   most_fi_tcl_MsgBodyPartListItem(const most_fi_tcl_MsgBodyPartListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBodyTitle;
   most_fi_tcl_String sBodyText;
   virtual ~most_fi_tcl_MsgBodyPartListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgBodyPartListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgBodyPartList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgBodyPartList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgBodyPartList& operator=(const most_fi_tcl_MsgBodyPartList& coRef);
   most_fi_tcl_MsgBodyPartList(const most_fi_tcl_MsgBodyPartList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgBodyPartListItem*, std::allocator<most_fi_tcl_MsgBodyPartListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgBodyPartListItem, std::allocator<most_fi_tcl_MsgBodyPartListItem> >oItems;

   most_fi_tcl_MsgBodyPartListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgBodyPartList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgBodyPartList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgTTSLanguage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgTTSLanguage();
   enum tenType {
      FI_EN_E8ENGLISHUS = 0UL,
      FI_EN_E8FRENCH_CANADIAN = 1UL,
      FI_EN_E8MEXICAN_SPANISH = 2UL,
      FI_EN_E8EUROPEAN_ENGLISH = 3UL,
      FI_EN_E8EUROPEAN_SPANISH = 4UL,
      FI_EN_E8EUROPEAN_FRENCH = 5UL,
      FI_EN_E8EUROPEAN_GERMAN = 6UL,
      FI_EN_E8CLASSICAL_ARABIC = 7UL,
      FI_EN_E8KOREAN = 8UL,
      FI_EN_E8STANDARD_CHINESE = 9UL,
      FI_EN_E8ITALIAN = 10UL,
      FI_EN_E8SWEDISH = 11UL,
      FI_EN_E8DUTCH = 12UL,
      FI_EN_E8HUNGARIAN = 13UL,
      FI_EN_E8CZECK = 14UL,
      FI_EN_E8NORWEGIAN = 15UL,
      FI_EN_E8DANISH = 16UL,
      FI_EN_E8SLOVAK = 17UL,
      FI_EN_E8PORTUGESE = 18UL,
      FI_EN_E8RUSSIAN = 19UL,
      FI_EN_E8GREEK = 20UL,
      FI_EN_E8FINNISH = 21UL,
      FI_EN_E8JAPANESE = 22UL,
      FI_EN_E8TURKISH = 23UL,
      FI_EN_E8USE_SYSTEM_LANGUAGE_SELECTION = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgTTSLanguage(most_fi_tcl_e8_MsgTTSLanguage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgTTSLanguage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgTTSLanguage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgRecordMessageAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgRecordMessageAction();
   enum tenType {
      FI_EN_E8REC_MSG_ACTION_START = 0UL,
      FI_EN_E8REC_MSG_ACTION_STOP = 1UL,
      FI_EN_E8REC_MSG_ACTION_CANCEL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgRecordMessageAction(most_fi_tcl_e8_MsgRecordMessageAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgRecordMessageAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgRecordMessageAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgRecordMessageState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgRecordMessageState();
   enum tenType {
      FI_EN_E8REC_MSG_STATE_RECORDING = 0UL,
      FI_EN_E8REC_MSG_STATE_STOPPED = 1UL,
      FI_EN_E8REC_MSG_STATE_CANCELLED = 2UL,
      FI_EN_E8REC_MSG_STATE_STOP_LIMIT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgRecordMessageState(most_fi_tcl_e8_MsgRecordMessageState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgRecordMessageState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgRecordMessageState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgPredefinedMessageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgPredefinedMessageType();
   enum tenType {
      FI_EN_E8PRE_MSG_STATIC = 0UL,
      FI_EN_E8PRE_MSG_DYNAMIC_ETA = 1UL,
      FI_EN_E8PRE_MSG_DYNAMIC_INTERSECTION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgPredefinedMessageType(most_fi_tcl_e8_MsgPredefinedMessageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgPredefinedMessageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgPredefinedMessageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgPredefinedMessageListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgPredefinedMessageListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgPredefinedMessageListResultItem& operator=(const most_fi_tcl_MsgPredefinedMessageListResultItem& coRef);
   most_fi_tcl_MsgPredefinedMessageListResultItem(const most_fi_tcl_MsgPredefinedMessageListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16PredefinedMessageHandle;
   most_fi_tcl_e8_MsgPredefinedMessageType e8PredefinedMessageType;
   most_fi_tcl_String sPredefinedMessageText;
   virtual ~most_fi_tcl_MsgPredefinedMessageListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgPredefinedMessageListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgPredefinedMessageListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgPredefinedMessageListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgPredefinedMessageListResult& operator=(const most_fi_tcl_MsgPredefinedMessageListResult& coRef);
   most_fi_tcl_MsgPredefinedMessageListResult(const most_fi_tcl_MsgPredefinedMessageListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgPredefinedMessageListResultItem*, std::allocator<most_fi_tcl_MsgPredefinedMessageListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgPredefinedMessageListResultItem, std::allocator<most_fi_tcl_MsgPredefinedMessageListResultItem> >oItems;

   most_fi_tcl_MsgPredefinedMessageListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgPredefinedMessageListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgPredefinedMessageListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgConversationMessages : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgConversationMessages() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgConversationMessages& operator=(const most_fi_tcl_MsgConversationMessages& coRef);
   most_fi_tcl_MsgConversationMessages(const most_fi_tcl_MsgConversationMessages& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_ShortStream*, std::allocator<most_fi_tcl_ShortStream*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_ShortStream, std::allocator<most_fi_tcl_ShortStream> >oItems;

   most_fi_tcl_ShortStream& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgConversationMessages();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgConversationMessages& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgSentMessageStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgSentMessageStatus();
   enum tenType {
      FI_EN_E8MSG_SENT_STATUS_ERROR = 0UL,
      FI_EN_E8MSG_SENT_STATUS_SUCCESS = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgSentMessageStatus(most_fi_tcl_e8_MsgSentMessageStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgSentMessageStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgSentMessageStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgAttachmentHandleStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgAttachmentHandleStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgAttachmentHandleStream& operator=(const most_fi_tcl_MsgAttachmentHandleStream& coRef);
   most_fi_tcl_MsgAttachmentHandleStream(const most_fi_tcl_MsgAttachmentHandleStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU16, std::allocator<tU16> >u16Items;

   tU16& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgAttachmentHandleStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgAttachmentHandleStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgMessageBodyText : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgMessageBodyText() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgMessageBodyText& operator=(const most_fi_tcl_MsgMessageBodyText& coRef);
   most_fi_tcl_MsgMessageBodyText(const most_fi_tcl_MsgMessageBodyText& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_String*, std::allocator<most_fi_tcl_String*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_String, std::allocator<most_fi_tcl_String> >sItems;

   most_fi_tcl_String& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgMessageBodyText();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgMessageBodyText& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgActionElementType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgActionElementType();
   enum tenType {
      FI_EN_E8ACTION_ELEMENT_PHONE_NUMBER = 0UL,
      FI_EN_E8ACTION_ELEMENT_EMAIL_ADDRESS = 1UL,
      FI_EN_E8ACTION_ELEMENT_ADDRESS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgActionElementType(most_fi_tcl_e8_MsgActionElementType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgActionElementType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgActionElementType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgCreateMessageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgCreateMessageType();
   enum tenType {
      FI_EN_E8NEW_MESSAGE = 0UL,
      FI_EN_E8REPLY = 1UL,
      FI_EN_E8REPLY_TO_ALL = 2UL,
      FI_EN_E8FORWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgCreateMessageType(most_fi_tcl_e8_MsgCreateMessageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgCreateMessageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgCreateMessageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgNotificationType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgNotificationType();
   enum tenType {
      FI_EN_E8NOTIFY_NEW_MSG = 0UL,
      FI_EN_E8NOTIFY_MSG_SEND_SUCCESS = 1UL,
      FI_EN_E8NOTIFY_MSG_DELIVERY_SUCCESS = 2UL,
      FI_EN_E8NOTIFY_MSG_SEND_FAILURE = 3UL,
      FI_EN_E8NOTIFY_MSG_DELIVERY_FAILURE = 4UL,
      FI_EN_E8NOTIFY_MSG_DELETED = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgNotificationType(most_fi_tcl_e8_MsgNotificationType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgNotificationType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgNotificationType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MapConnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MapConnectionStatus();
   enum tenType {
      FI_EN_E8MAP_DISCONNECTED = 0UL,
      FI_EN_E8MAP_CONNECTING = 1UL,
      FI_EN_E8MAP_CONNECTED = 2UL,
      FI_EN_E8MAP_CONNECTION_FAILED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MapConnectionStatus(most_fi_tcl_e8_MapConnectionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MapConnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MapConnectionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgPredefinedMessageSubCategory : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgPredefinedMessageSubCategory();
   enum tenType {
      FI_EN_E8NO_SUB_CATEGORY = 0UL,
      FI_EN_E8PRE_MSG_SUB_CAT_1 = 1UL,
      FI_EN_E8PRE_MSG_SUB_CAT_2 = 2UL,
      FI_EN_E8PRE_MSG_SUB_CAT_3 = 3UL,
      FI_EN_E8PRE_MSG_SUB_CAT_4 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgPredefinedMessageSubCategory(most_fi_tcl_e8_MsgPredefinedMessageSubCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgPredefinedMessageSubCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgPredefinedMessageSubCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgPredefinedMessageCategory : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgPredefinedMessageCategory();
   enum tenType {
      FI_EN_E8PRE_MSG_CAT_USER_DEF = 0UL,
      FI_EN_E8PRE_MSG_CAT_SYS_DEF = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgPredefinedMessageCategory(most_fi_tcl_e8_MsgPredefinedMessageCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgPredefinedMessageCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgPredefinedMessageCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgPredefinedMessageListExtendedResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgPredefinedMessageListExtendedResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgPredefinedMessageListExtendedResultItem& operator=(const most_fi_tcl_MsgPredefinedMessageListExtendedResultItem& coRef);
   most_fi_tcl_MsgPredefinedMessageListExtendedResultItem(const most_fi_tcl_MsgPredefinedMessageListExtendedResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU16 u16PredefinedMessageHandle;
   most_fi_tcl_e8_MsgPredefinedMessageType e8PredefinedMessageType;
   most_fi_tcl_e8_MsgPredefinedMessageCategory e8PredefinedMessageCategory;
   most_fi_tcl_e8_MsgPredefinedMessageSubCategory e8PredefinedMessageSubCategory;
   most_fi_tcl_String sPredefinedMessageText;
   virtual ~most_fi_tcl_MsgPredefinedMessageListExtendedResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgPredefinedMessageListExtendedResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgPredefinedMessageListExtendedResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgPredefinedMessageListExtendedResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgPredefinedMessageListExtendedResult& operator=(const most_fi_tcl_MsgPredefinedMessageListExtendedResult& coRef);
   most_fi_tcl_MsgPredefinedMessageListExtendedResult(const most_fi_tcl_MsgPredefinedMessageListExtendedResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgPredefinedMessageListExtendedResultItem*, std::allocator<most_fi_tcl_MsgPredefinedMessageListExtendedResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgPredefinedMessageListExtendedResultItem, std::allocator<most_fi_tcl_MsgPredefinedMessageListExtendedResultItem> >oItems;

   most_fi_tcl_MsgPredefinedMessageListExtendedResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgPredefinedMessageListExtendedResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgPredefinedMessageListExtendedResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgLanguageSet : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgLanguageSet();
   enum tenType {
      FI_EN_LANGUAGE_SELECTION_NA_ENGLISH = 0UL,
      FI_EN_LANGUAGE_SELECTION_GERMAN = 1UL,
      FI_EN_LANGUAGE_SELECTION_ITALIAN = 2UL,
      FI_EN_LANGUAGE_SELECTION_SWEDISH = 3UL,
      FI_EN_LANGUAGE_SELECTION_FRENCH = 4UL,
      FI_EN_LANGUAGE_SELECTION_SPANISH = 5UL,
      FI_EN_LANGUAGE_SELECTION_DUTCH = 6UL,
      FI_EN_LANGUAGE_SELECTION_PORTUGUESE = 7UL,
      FI_EN_LANGUAGE_SELECTION_NORWEGIAN = 8UL,
      FI_EN_LANGUAGE_SELECTION_FINNISH = 9UL,
      FI_EN_LANGUAGE_SELECTION_DANISH = 10UL,
      FI_EN_LANGUAGE_SELECTION_GREEK = 11UL,
      FI_EN_LANGUAGE_SELECTION_JAPANESE = 12UL,
      FI_EN_LANGUAGE_SELECTION_ARABIC = 13UL,
      FI_EN_LANGUAGE_SELECTION_STANDARD_CHINESE = 14UL,
      FI_EN_LANGUAGE_SELECTION_POLISH = 15UL,
      FI_EN_LANGUAGE_SELECTION_TURKISH = 16UL,
      FI_EN_LANGUAGE_SELECTION_KOREAN = 17UL,
      FI_EN_LANGUAGE_SELECTION_TRADITIONAL_CHINESE = 18UL,
      FI_EN_LANGUAGE_SELECTION_UK_ENGLISH = 19UL,
      FI_EN_LANGUAGE_SELECTION_HUNGARIAN = 20UL,
      FI_EN_LANGUAGE_SELECTION_CZECH = 21UL,
      FI_EN_LANGUAGE_SELECTION_SLOVAK = 22UL,
      FI_EN_LANGUAGE_SELECTION_RUSSIAN = 23UL,
      FI_EN_LANGUAGE_SELECTION_BRAZILIAN_PORTUGUESE = 24UL,
      FI_EN_LANGUAGE_SELECTION_THAI = 25UL,
      FI_EN_LANGUAGE_SELECTION_BULGARIAN = 26UL,
      FI_EN_LANGUAGE_SELECTION_ROMANIAN = 27UL,
      FI_EN_LANGUAGE_SELECTION_SLOVENIAN = 28UL,
      FI_EN_LANGUAGE_SELECTION_CROATIAN = 29UL,
      FI_EN_LANGUAGE_SELECTION_UKRAINIAN = 30UL,
      FI_EN_LANGUAGE_SELECTION_NA_FRENCH = 31UL,
      FI_EN_LANGUAGE_SELECTION_NA_SPANISH = 32UL,
      FI_EN_LANGUAGE_SELECTION_CANTONESE = 33UL,
      FI_EN_LANGUAGE_SELECTION_UZBEK = 34UL,
      FI_EN_LANGUAGE_SELECTION_SERBIAN = 35UL,
      FI_EN_LANGUAGE_SELECTION_LITHUANIAN = 36UL,
      FI_EN_LANGUAGE_SELECTION_ESTONIAN = 37UL,
      FI_EN_LANGUAGE_SELECTION_LATVIAN = 38UL,
      FI_EN_LANGUAGE_SELECTION_ALBANESE = 39UL,
      FI_EN_LANGUAGE_SELECTION_MANDARIN_CHINESE = 40UL,
      FI_EN_LANGUAGE_SELECTION_AUSTRALIAN_ENGLISH = 41UL,
      FI_EN_LANGUAGE_SELECTION_US_JAPAN_ENGLISH = 42UL,
      FI_EN_LANGUAGE_SELECTION_US_PRC_ENGLISH = 43UL,
      FI_EN_LANGUAGE_SELECTION_CANADIAN_ENGLISH = 44UL,
      FI_EN_LANGUAGE_SELECTION_INDIA_ENGLISH = 45UL,
      FI_EN_LANGUAGE_SELECTION_FILIPINO = 46UL,
      FI_EN_LANGUAGE_SELECTION_FLEMISH = 47UL,
      FI_EN_LANGUAGE_SELECTION_CANADIAN_FRENCH = 48UL,
      FI_EN_LANGUAGE_SELECTION_HEBREW = 49UL,
      FI_EN_LANGUAGE_SELECTION_HINDI = 50UL,
      FI_EN_LANGUAGE_SELECTION_INDONESIAN = 51UL,
      FI_EN_LANGUAGE_SELECTION_MALAY = 52UL,
      FI_EN_LANGUAGE_SELECTION_PERSIAN = 53UL,
      FI_EN_LANGUAGE_SELECTION_LATIN_AMERICAN_SPANISH = 54UL,
      FI_EN_LANGUAGE_SELECTION_VIETNAMESE = 55UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgLanguageSet(most_fi_tcl_e8_MsgLanguageSet::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgLanguageSet();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgLanguageSet& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SignatureTextStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SignatureTextStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SignatureTextStreamItem& operator=(const most_fi_tcl_SignatureTextStreamItem& coRef);
   most_fi_tcl_SignatureTextStreamItem(const most_fi_tcl_SignatureTextStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tBool bSignatureOnOff;
   most_fi_tcl_String sSignatureText;
   virtual ~most_fi_tcl_SignatureTextStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SignatureTextStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SignatureTextStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SignatureTextStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SignatureTextStream& operator=(const most_fi_tcl_SignatureTextStream& coRef);
   most_fi_tcl_SignatureTextStream(const most_fi_tcl_SignatureTextStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_SignatureTextStreamItem*, std::allocator<most_fi_tcl_SignatureTextStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_SignatureTextStreamItem, std::allocator<most_fi_tcl_SignatureTextStreamItem> >oItems;

   most_fi_tcl_SignatureTextStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_SignatureTextStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SignatureTextStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgConversationHandle : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgConversationHandle();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgConversationHandle& operator=(const most_fi_tcl_MsgConversationHandle& coRef);
   most_fi_tcl_MsgConversationHandle(const most_fi_tcl_MsgConversationHandle& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32ConversationID;
   virtual ~most_fi_tcl_MsgConversationHandle();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgConversationHandle& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgConversationListSliceResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgConversationListSliceResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgConversationListSliceResultItem& operator=(const most_fi_tcl_MsgConversationListSliceResultItem& coRef);
   most_fi_tcl_MsgConversationListSliceResultItem(const most_fi_tcl_MsgConversationListSliceResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_MsgConversationHandle oConversationHandle;
   most_fi_tcl_MsgMessageDateTime oLastActivity;
   most_fi_tcl_String sSubject;
   tU32 u32AttachmentSize;
   most_fi_tcl_e8_MsgMessageType e8MessageType;
   tU32 u32NumberOfMessages;
   tU8 u8UnreadMessageCount;
   most_fi_tcl_MsgAddressFieldItem oAddressField;
   tU32 u32ContactHandle;
   virtual ~most_fi_tcl_MsgConversationListSliceResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgConversationListSliceResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MsgMessageStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MsgMessageStatus();
   enum tenType {
      FI_EN_E8DELIVERY_SUCCESS = 0UL,
      FI_EN_E8DELIVERY_FAILURE = 1UL,
      FI_EN_E8SENDING_SUCCESS = 2UL,
      FI_EN_E8SENDING_FAILURE = 3UL,
      FI_EN_E8RECEIVED_COMPLETE = 4UL,
      FI_EN_E8RECEIVED_FRACTIONED = 5UL,
      FI_EN_E8RECEIVED_NOTIFICATION = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_MsgMessageStatus(most_fi_tcl_e8_MsgMessageStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MsgMessageStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MsgMessageStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgConversationDetailListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgConversationDetailListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgConversationDetailListItem& operator=(const most_fi_tcl_MsgConversationDetailListItem& coRef);
   most_fi_tcl_MsgConversationDetailListItem(const most_fi_tcl_MsgConversationDetailListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_MsgMessageHandle oMessageHandle;
   most_fi_tcl_MsgMessageDateTime oMessageDateTime;
   most_fi_tcl_String sMessageContent;
   tU32 u32AttachmentSize;
   most_fi_tcl_e8_MsgMessageType e8MessageType;
   most_fi_tcl_e8_MsgMessageStatus e8MessageStatus;
   tBool bReadStatusFlag;
   most_fi_tcl_MsgAddressFieldItem oAddressField;
   tU32 u32ContactHandle;
   virtual ~most_fi_tcl_MsgConversationDetailListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgConversationDetailListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgConversationListSliceResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgConversationListSliceResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgConversationListSliceResult& operator=(const most_fi_tcl_MsgConversationListSliceResult& coRef);
   most_fi_tcl_MsgConversationListSliceResult(const most_fi_tcl_MsgConversationListSliceResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgConversationListSliceResultItem*, std::allocator<most_fi_tcl_MsgConversationListSliceResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgConversationListSliceResultItem, std::allocator<most_fi_tcl_MsgConversationListSliceResultItem> >oItems;

   most_fi_tcl_MsgConversationListSliceResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgConversationListSliceResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgConversationListSliceResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MsgConversationDetailList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MsgConversationDetailList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MsgConversationDetailList& operator=(const most_fi_tcl_MsgConversationDetailList& coRef);
   most_fi_tcl_MsgConversationDetailList(const most_fi_tcl_MsgConversationDetailList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MsgConversationDetailListItem*, std::allocator<most_fi_tcl_MsgConversationDetailListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MsgConversationDetailListItem, std::allocator<most_fi_tcl_MsgConversationDetailListItem> >oItems;

   most_fi_tcl_MsgConversationDetailListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MsgConversationDetailList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MsgConversationDetailList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnOffListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnOffListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnOffListItem& operator=(const most_fi_tcl_OnOffListItem& coRef);
   most_fi_tcl_OnOffListItem(const most_fi_tcl_OnOffListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tBool bOnOffState;
   virtual ~most_fi_tcl_OnOffListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnOffListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnOffList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnOffList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnOffList& operator=(const most_fi_tcl_OnOffList& coRef);
   most_fi_tcl_OnOffList(const most_fi_tcl_OnOffList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_OnOffListItem, std::allocator<most_fi_tcl_OnOffListItem> >oItems;

   most_fi_tcl_OnOffListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_OnOffList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnOffList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AutoReplyTextListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AutoReplyTextListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AutoReplyTextListItem& operator=(const most_fi_tcl_AutoReplyTextListItem& coRef);
   most_fi_tcl_AutoReplyTextListItem(const most_fi_tcl_AutoReplyTextListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU16 u16PredefinedMsgHandle;
   virtual ~most_fi_tcl_AutoReplyTextListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AutoReplyTextListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AutoReplyTextList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AutoReplyTextList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AutoReplyTextList& operator=(const most_fi_tcl_AutoReplyTextList& coRef);
   most_fi_tcl_AutoReplyTextList(const most_fi_tcl_AutoReplyTextList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AutoReplyTextListItem, std::allocator<most_fi_tcl_AutoReplyTextListItem> >oItems;

   most_fi_tcl_AutoReplyTextListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AutoReplyTextList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AutoReplyTextList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MapDeviceCapabilitiesListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MapDeviceCapabilitiesListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MapDeviceCapabilitiesListItem& operator=(const most_fi_tcl_MapDeviceCapabilitiesListItem& coRef);
   most_fi_tcl_MapDeviceCapabilitiesListItem(const most_fi_tcl_MapDeviceCapabilitiesListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU16 u16SupportedMessageTypes;
   tU16 u16SupportedMapFeatures;
   most_fi_tcl_e8_MapConnectionStatus e8MapConnectionState;
   most_fi_tcl_MsgActiveAccount oActiveAccount;
   virtual ~most_fi_tcl_MapDeviceCapabilitiesListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MapDeviceCapabilitiesListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MapDeviceCapabilitiesList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MapDeviceCapabilitiesList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MapDeviceCapabilitiesList& operator=(const most_fi_tcl_MapDeviceCapabilitiesList& coRef);
   most_fi_tcl_MapDeviceCapabilitiesList(const most_fi_tcl_MapDeviceCapabilitiesList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MapDeviceCapabilitiesListItem*, std::allocator<most_fi_tcl_MapDeviceCapabilitiesListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MapDeviceCapabilitiesListItem, std::allocator<most_fi_tcl_MapDeviceCapabilitiesListItem> >oItems;

   most_fi_tcl_MapDeviceCapabilitiesListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MapDeviceCapabilitiesList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MapDeviceCapabilitiesList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkPhoneBookSortType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkPhoneBookSortType();
   enum tenType {
      FI_EN_E8PB_LIST_SORT_LASTNAME = 0UL,
      FI_EN_E8PB_LIST_SORT_FIRSTNAME = 1UL,
      FI_EN_E8PB_LIST_SORT_ADDRESS_LASTNAME = 2UL,
      FI_EN_E8PB_LIST_SORT_ADDRESS_FIRSTNAME = 3UL,
      FI_EN_E8PB_LIST_SORT_NUMBER_LASTNAME = 4UL,
      FI_EN_E8PB_LIST_SORT_NUMBER_FIRSTNAME = 5UL,
      FI_EN_E8PB_LIST_SORT_EMAIL_LASTNAME = 6UL,
      FI_EN_E8PB_LIST_SORT_EMAIL_FIRSTNAME = 7UL,
      FI_EN_E8PB_LIST_SORT_CATEGORY_LASTNAME = 8UL,
      FI_EN_E8PB_LIST_SORT_CATEGORY_FIRSTNAME = 9UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkPhoneBookSortType(most_fi_tcl_e8_PhonBkPhoneBookSortType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkPhoneBookSortType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkPhoneBookSortType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkSearchKeyboardListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkSearchKeyboardListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkSearchKeyboardListItem& operator=(const most_fi_tcl_PhonBkSearchKeyboardListItem& coRef);
   most_fi_tcl_PhonBkSearchKeyboardListItem(const most_fi_tcl_PhonBkSearchKeyboardListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sSearchKeyboardLetter;
   tBool bLetterAvailable;
   tU32 u32LetterStartIndex;
   tU32 u32LetterEndIndex;
   virtual ~most_fi_tcl_PhonBkSearchKeyboardListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkSearchKeyboardListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkSearchKeyboardList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkSearchKeyboardList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkSearchKeyboardList& operator=(const most_fi_tcl_PhonBkSearchKeyboardList& coRef);
   most_fi_tcl_PhonBkSearchKeyboardList(const most_fi_tcl_PhonBkSearchKeyboardList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkSearchKeyboardListItem*, std::allocator<most_fi_tcl_PhonBkSearchKeyboardListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkSearchKeyboardListItem, std::allocator<most_fi_tcl_PhonBkSearchKeyboardListItem> >oItems;

   most_fi_tcl_PhonBkSearchKeyboardListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkSearchKeyboardList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkSearchKeyboardList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactDetailFilter : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactDetailFilter();
   enum tenType {
      FI_EN_E8CDF_NONE = 0UL,
      FI_EN_E8CDF_ADDRESS = 1UL,
      FI_EN_E8CDF_TELEPHONE = 2UL,
      FI_EN_E8CDF_EMAIL = 3UL,
      FI_EN_E8CDF_NOTE = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactDetailFilter(most_fi_tcl_e8_PhonBkContactDetailFilter::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactDetailFilter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactDetailFilter& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkHomeAddressLatLong : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkHomeAddressLatLong();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkHomeAddressLatLong& operator=(const most_fi_tcl_PhonBkHomeAddressLatLong& coRef);
   most_fi_tcl_PhonBkHomeAddressLatLong(const most_fi_tcl_PhonBkHomeAddressLatLong& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bLatLongValidity;
   tS32 s32Latitude;
   tS32 s32Longitude;
   virtual ~most_fi_tcl_PhonBkHomeAddressLatLong();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkHomeAddressLatLong& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkWorkAddressLatLong : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkWorkAddressLatLong();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkWorkAddressLatLong& operator=(const most_fi_tcl_PhonBkWorkAddressLatLong& coRef);
   most_fi_tcl_PhonBkWorkAddressLatLong(const most_fi_tcl_PhonBkWorkAddressLatLong& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bLatLongValidity;
   tS32 s32Latitude;
   tS32 s32Longitude;
   virtual ~most_fi_tcl_PhonBkWorkAddressLatLong();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkWorkAddressLatLong& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkOtherAddressLatLong : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkOtherAddressLatLong();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkOtherAddressLatLong& operator=(const most_fi_tcl_PhonBkOtherAddressLatLong& coRef);
   most_fi_tcl_PhonBkOtherAddressLatLong(const most_fi_tcl_PhonBkOtherAddressLatLong& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bLatLongValidity;
   tS32 s32Latitude;
   tS32 s32Longitude;
   virtual ~most_fi_tcl_PhonBkOtherAddressLatLong();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkOtherAddressLatLong& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkContactDetails : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkContactDetails();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkContactDetails& operator=(const most_fi_tcl_PhonBkContactDetails& coRef);
   most_fi_tcl_PhonBkContactDetails(const most_fi_tcl_PhonBkContactDetails& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPOIName;
   most_fi_tcl_String sPreferredNumber;
   most_fi_tcl_String sCellNumber1;
   most_fi_tcl_String sCellNumber2;
   most_fi_tcl_String sHomeNumber1;
   most_fi_tcl_String sHomeNumber2;
   most_fi_tcl_String sWorkNumber1;
   most_fi_tcl_String sWorkNumber2;
   most_fi_tcl_String sOtherNumber;
   most_fi_tcl_String sEmailAddress1;
   most_fi_tcl_String sEmailAddress2;
   most_fi_tcl_String sEmailAddress3;
   most_fi_tcl_String sHomeAddress;
   most_fi_tcl_String sWorkAddress;
   most_fi_tcl_String sOtherAddress;
   most_fi_tcl_String sCategory;
   most_fi_tcl_String sNotes;
   tU32 u32FirstNameTag;
   tU32 u32LastNameTag;
   tU32 u32POINameTag;
   tU32 u32PrefNumTag;
   tU32 u32CellNum1Tag;
   tU32 u32CellNum2Tag;
   tU32 u32HomeNum1Tag;
   tU32 u32HomeNum2Tag;
   tU32 u32WorkNum1Tag;
   tU32 u32WorkNum2Tag;
   tU32 u32OtherNumTag;
   tU32 u32EmailAdd1Tag;
   tU32 u32EmailAdd2Tag;
   tU32 u32EmailAdd3Tag;
   tU32 u32HomeAddTag;
   tU32 u32WorkAddTag;
   tU32 u32OtherAddTag;
   tU32 u32CategoryTag;
   tU32 u32NotesTag;
   tBool bPhotoAvailable;
   most_fi_tcl_PhonBkHomeAddressLatLong oHomeAddressLatLong;
   most_fi_tcl_PhonBkWorkAddressLatLong oWorkAddressLatLong;
   most_fi_tcl_PhonBkOtherAddressLatLong oOtherAddressLatLong;
   virtual ~most_fi_tcl_PhonBkContactDetails();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkContactDetails& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkContactPhoto : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkContactPhoto() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkContactPhoto& operator=(const most_fi_tcl_PhonBkContactPhoto& coRef);
   most_fi_tcl_PhonBkContactPhoto(const most_fi_tcl_PhonBkContactPhoto& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkContactPhoto();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkContactPhoto& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkListChangeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkListChangeType();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkListChangeType(most_fi_tcl_e8_PhonBkListChangeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkListChangeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkListChangeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkListType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkListType();
   enum tenType {
      FI_EN_E8PHONEBOOK_LIST = 0UL,
      FI_EN_E8CALLHISTORY_LIST = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkListType(most_fi_tcl_e8_PhonBkListType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkListType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkListType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkChangeItems : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkChangeItems() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkChangeItems& operator=(const most_fi_tcl_PhonBkChangeItems& coRef);
   most_fi_tcl_PhonBkChangeItems(const most_fi_tcl_PhonBkChangeItems& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkChangeItems();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkChangeItems& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactAddStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactAddStatus();
   enum tenType {
      FI_EN_E8CONTACT_ADD_SUCCESS = 0UL,
      FI_EN_E8CONTACT_ADD_FAILURE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactAddStatus(most_fi_tcl_e8_PhonBkContactAddStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactAddStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactAddStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactDeleteStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactDeleteStatus();
   enum tenType {
      FI_EN_E8CONTACT_DELETE_SUCCESS = 0UL,
      FI_EN_E8CONTACT_DELETE_FAILURE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactDeleteStatus(most_fi_tcl_e8_PhonBkContactDeleteStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactDeleteStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactDeleteStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactEditStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactEditStatus();
   enum tenType {
      FI_EN_E8CONTACT_EDIT_SUCCESS = 0UL,
      FI_EN_E8CONTACT_EDIT_FAILURE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactEditStatus(most_fi_tcl_e8_PhonBkContactEditStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactEditStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactEditStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkCallHistorySortType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkCallHistorySortType();
   enum tenType {
      FI_EN_E8CH_SORT_DATETIME = 0UL,
      FI_EN_E8CH_SORT_LASTNAME = 1UL,
      FI_EN_E8CH_SORT_FIRSTNAME = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkCallHistorySortType(most_fi_tcl_e8_PhonBkCallHistorySortType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkCallHistorySortType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkCallHistorySortType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkCallHistoryType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkCallHistoryType();
   enum tenType {
      FI_EN_E8CCH = 0UL,
      FI_EN_E8MCH = 1UL,
      FI_EN_E8ICH = 2UL,
      FI_EN_E8OCH = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkCallHistoryType(most_fi_tcl_e8_PhonBkCallHistoryType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkCallHistoryType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkCallHistoryType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkCallPhoneType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkCallPhoneType();
   enum tenType {
      FI_EN_E8PREFERRED = 0UL,
      FI_EN_E8CELL1 = 1UL,
      FI_EN_E8CELL2 = 2UL,
      FI_EN_E8HOME1 = 3UL,
      FI_EN_E8HOME2 = 4UL,
      FI_EN_E8WORK1 = 5UL,
      FI_EN_E8WORK2 = 6UL,
      FI_EN_E8OTHER = 7UL,
      FI_EN_E8UNKNOWN = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkCallPhoneType(most_fi_tcl_e8_PhonBkCallPhoneType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkCallPhoneType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkCallPhoneType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallDate : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallDate();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallDate& operator=(const most_fi_tcl_PhonBkCallDate& coRef);
   most_fi_tcl_PhonBkCallDate(const most_fi_tcl_PhonBkCallDate& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sCldrYear;
   most_fi_tcl_String sCldrMonth;
   most_fi_tcl_String sCldrDay;
   virtual ~most_fi_tcl_PhonBkCallDate();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallDate& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallTime : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallTime& operator=(const most_fi_tcl_PhonBkCallTime& coRef);
   most_fi_tcl_PhonBkCallTime(const most_fi_tcl_PhonBkCallTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sHours;
   most_fi_tcl_String sMinutes;
   most_fi_tcl_String sSeconds;
   virtual ~most_fi_tcl_PhonBkCallTime();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallDateTime : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallDateTime();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallDateTime& operator=(const most_fi_tcl_PhonBkCallDateTime& coRef);
   most_fi_tcl_PhonBkCallDateTime(const most_fi_tcl_PhonBkCallDateTime& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_PhonBkCallDate oCallDate;
   most_fi_tcl_PhonBkCallTime oCallTime;
   virtual ~most_fi_tcl_PhonBkCallDateTime();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallDateTime& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallHistoryListSliceResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallHistoryListSliceResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallHistoryListSliceResultItem& operator=(const most_fi_tcl_PhonBkCallHistoryListSliceResultItem& coRef);
   most_fi_tcl_PhonBkCallHistoryListSliceResultItem(const most_fi_tcl_PhonBkCallHistoryListSliceResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ContactHandle;
   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPhoneNumber;
   most_fi_tcl_e8_PhonBkCallPhoneType e8CallPhoneType;
   most_fi_tcl_e8_PhonBkCallHistoryType e8CallHistoryType;
   most_fi_tcl_PhonBkCallDateTime oCallDateTime;
   virtual ~most_fi_tcl_PhonBkCallHistoryListSliceResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallHistoryListSliceResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallHistoryListSliceResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallHistoryListSliceResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallHistoryListSliceResult& operator=(const most_fi_tcl_PhonBkCallHistoryListSliceResult& coRef);
   most_fi_tcl_PhonBkCallHistoryListSliceResult(const most_fi_tcl_PhonBkCallHistoryListSliceResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkCallHistoryListSliceResultItem*, std::allocator<most_fi_tcl_PhonBkCallHistoryListSliceResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkCallHistoryListSliceResultItem, std::allocator<most_fi_tcl_PhonBkCallHistoryListSliceResultItem> >oItems;

   most_fi_tcl_PhonBkCallHistoryListSliceResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkCallHistoryListSliceResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallHistoryListSliceResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactDetailType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactDetailType();
   enum tenType {
      FI_EN_E8NONE = 0UL,
      FI_EN_E8PREFERRED_NUMBER = 1UL,
      FI_EN_E8CELL_NUMBER_1 = 2UL,
      FI_EN_E8CELL_NUMBER_2 = 3UL,
      FI_EN_E8HOME_NUMBER_1 = 4UL,
      FI_EN_E8HOME_NUMBER_2 = 5UL,
      FI_EN_E8WORK_NUMBER_1 = 6UL,
      FI_EN_E8WORK_NUMBER_2 = 7UL,
      FI_EN_E8OTHER_NUMBER = 8UL,
      FI_EN_E8EMAIL_ADDRESS_1 = 9UL,
      FI_EN_E8EMAIL_ADDRESS_2 = 10UL,
      FI_EN_E8EMAIL_ADDRESS_3 = 11UL,
      FI_EN_E8HOME_ADDRESS = 12UL,
      FI_EN_E8WORK_ADDRESS = 13UL,
      FI_EN_E8OTHER_ADDRESS = 14UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactDetailType(most_fi_tcl_e8_PhonBkContactDetailType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactDetailType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactDetailType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallHistoryTypeSupport : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallHistoryTypeSupport();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallHistoryTypeSupport& operator=(const most_fi_tcl_PhonBkCallHistoryTypeSupport& coRef);
   most_fi_tcl_PhonBkCallHistoryTypeSupport(const most_fi_tcl_PhonBkCallHistoryTypeSupport& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bCCHSupport;
   tBool bMCHSupport;
   tBool bICHSupport;
   tBool bOCHSupport;
   virtual ~most_fi_tcl_PhonBkCallHistoryTypeSupport();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallHistoryTypeSupport& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem& operator=(const most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem& coRef);
   most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem(const most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPhoneNumber;
   virtual ~most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkAutocompleteMultipleMatchList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkAutocompleteMultipleMatchList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkAutocompleteMultipleMatchList& operator=(const most_fi_tcl_PhonBkAutocompleteMultipleMatchList& coRef);
   most_fi_tcl_PhonBkAutocompleteMultipleMatchList(const most_fi_tcl_PhonBkAutocompleteMultipleMatchList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem*, std::allocator<most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem, std::allocator<most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem> >oItems;

   most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkAutocompleteMultipleMatchList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkAutocompleteMultipleMatchList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPhoneBookListSliceResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPhoneBookListSliceResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPhoneBookListSliceResultItem& operator=(const most_fi_tcl_PhonBkPhoneBookListSliceResultItem& coRef);
   most_fi_tcl_PhonBkPhoneBookListSliceResultItem(const most_fi_tcl_PhonBkPhoneBookListSliceResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ContactHandle;
   tU32 u32ContactNameTag;
   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPOIName;
   tBool bQuickDial;
   tBool bQuickRoute;
   tBool bQuickText;
   virtual ~most_fi_tcl_PhonBkPhoneBookListSliceResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPhoneBookListSliceResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPhoneBookListSliceResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPhoneBookListSliceResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPhoneBookListSliceResult& operator=(const most_fi_tcl_PhonBkPhoneBookListSliceResult& coRef);
   most_fi_tcl_PhonBkPhoneBookListSliceResult(const most_fi_tcl_PhonBkPhoneBookListSliceResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkPhoneBookListSliceResultItem*, std::allocator<most_fi_tcl_PhonBkPhoneBookListSliceResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkPhoneBookListSliceResultItem, std::allocator<most_fi_tcl_PhonBkPhoneBookListSliceResultItem> >oItems;

   most_fi_tcl_PhonBkPhoneBookListSliceResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkPhoneBookListSliceResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPhoneBookListSliceResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder();
   enum tenType {
      FI_EN_E8PREFERRED_SORT_ORDER_FIRSTNAME = 0UL,
      FI_EN_E8PREFERRED_SORT_ORDER_LASTNAME = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder(most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkRecentCallListDownloadState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkRecentCallListDownloadState();
   enum tenType {
      FI_EN_E8RCDS_NOT_STARTED = 0UL,
      FI_EN_E8RCDS_IN_PROCESS = 1UL,
      FI_EN_E8RCDS_COMPLETE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkRecentCallListDownloadState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkRecentCallListDownloadState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkPhoneBookDownloadState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkPhoneBookDownloadState();
   enum tenType {
      FI_EN_E8PBDS_NOT_STARTED = 0UL,
      FI_EN_E8PBDS_CONTACT_INFO = 1UL,
      FI_EN_E8PBDS_CONTACT_PHOTO = 2UL,
      FI_EN_E8PBDS_COMPLETE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkPhoneBookDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkPhoneBookDownloadState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkPhoneBookDownloadState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadStateStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadStateStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadStateStreamItem& operator=(const most_fi_tcl_PhonBkDownloadStateStreamItem& coRef);
   most_fi_tcl_PhonBkDownloadStateStreamItem(const most_fi_tcl_PhonBkDownloadStateStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_e8_PhonBkRecentCallListDownloadState e8RecentCallListDownloadState;
   most_fi_tcl_e8_PhonBkPhoneBookDownloadState e8PhoneBookDownloadState;
   virtual ~most_fi_tcl_PhonBkDownloadStateStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadStateStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadStateStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadStateStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadStateStream& operator=(const most_fi_tcl_PhonBkDownloadStateStream& coRef);
   most_fi_tcl_PhonBkDownloadStateStream(const most_fi_tcl_PhonBkDownloadStateStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_PhonBkDownloadStateStreamItem, std::allocator<most_fi_tcl_PhonBkDownloadStateStreamItem> >oItems;

   most_fi_tcl_PhonBkDownloadStateStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkDownloadStateStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadStateStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkNumberType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkNumberType();
   enum tenType {
      FI_EN_NONE = 0UL,
      FI_EN_PREFERRED = 1UL,
      FI_EN_CELL = 2UL,
      FI_EN_HOME = 3UL,
      FI_EN_WORK = 4UL,
      FI_EN_OTHER = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkNumberType(most_fi_tcl_e8_PhonBkNumberType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkNumberType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkNumberType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem& operator=(const most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem& coRef);
   most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem(const most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ContactHandle;
   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPhoneNumber;
   most_fi_tcl_e8_PhonBkNumberType e8CallPhoneType;
   most_fi_tcl_e8_PhonBkCallHistoryType e8CallHistoryType;
   most_fi_tcl_PhonBkCallDateTime oCallDateTime;
   tU8 u8CallCount;
   virtual ~most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPhoneNumberItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPhoneNumberItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPhoneNumberItem& operator=(const most_fi_tcl_PhonBkPhoneNumberItem& coRef);
   most_fi_tcl_PhonBkPhoneNumberItem(const most_fi_tcl_PhonBkPhoneNumberItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sNumber;
   most_fi_tcl_e8_PhonBkNumberType e8Type;
   virtual ~most_fi_tcl_PhonBkPhoneNumberItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPhoneNumberItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactType();
   enum tenType {
      FI_EN_STANDARD = 0UL,
      FI_EN_FAVORITE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactType(most_fi_tcl_e8_PhonBkContactType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem& operator=(const most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem& coRef);
   most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem(const most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ContactHandle;
   tU32 u32ContactNameTag;
   most_fi_tcl_e8_PhonBkContactType e8ContactType;
   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPOIName;
   tBool bQuickDial;
   tBool bQuickRoute;
   tBool bQuickText;
   tU8 u8PhoneNumberCount;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber1;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber2;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber3;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber4;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber5;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber6;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber7;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber8;
   tU8 u8EmailCount;
   most_fi_tcl_String oEmailAddr1;
   most_fi_tcl_String oEmailAddr2;
   most_fi_tcl_String oEmailAddr3;
   tU8 u8AddressesCount;
   tBool bPhotoAvailable;
   virtual ~most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPhoneBookListSliceResultExtended : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPhoneBookListSliceResultExtended() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPhoneBookListSliceResultExtended& operator=(const most_fi_tcl_PhonBkPhoneBookListSliceResultExtended& coRef);
   most_fi_tcl_PhonBkPhoneBookListSliceResultExtended(const most_fi_tcl_PhonBkPhoneBookListSliceResultExtended& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem*, std::allocator<most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem, std::allocator<most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem> >oItems;

   most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkPhoneBookListSliceResultExtended();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPhoneBookListSliceResultExtended& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkCallHistoryListSliceResultExtened : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkCallHistoryListSliceResultExtened() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkCallHistoryListSliceResultExtened& operator=(const most_fi_tcl_PhonBkCallHistoryListSliceResultExtened& coRef);
   most_fi_tcl_PhonBkCallHistoryListSliceResultExtened(const most_fi_tcl_PhonBkCallHistoryListSliceResultExtened& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem*, std::allocator<most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem, std::allocator<most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem> >oItems;

   most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkCallHistoryListSliceResultExtened();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkCallHistoryListSliceResultExtened& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended();
   enum tenType {
      FI_EN_E8RCDS_NOT_STARTED = 0UL,
      FI_EN_E8RCDS_IN_PROCESS = 1UL,
      FI_EN_E8RCDS_COMPLETE = 2UL,
      FI_EN_E8RCDS_ERROR = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended(most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended();
   enum tenType {
      FI_EN_E8PBDS_NOT_STARTED = 0UL,
      FI_EN_E8PBDS_CONTACT_INFO = 1UL,
      FI_EN_E8PBDS_CONTACT_PHOTO = 2UL,
      FI_EN_E8PBDS_COMPLETE = 3UL,
      FI_EN_E8PBDS_LIMIT_REACHED = 4UL,
      FI_EN_E8PBDS_ERROR = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadStateExtendedStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadStateExtendedStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadStateExtendedStreamItem& operator=(const most_fi_tcl_PhonBkDownloadStateExtendedStreamItem& coRef);
   most_fi_tcl_PhonBkDownloadStateExtendedStreamItem(const most_fi_tcl_PhonBkDownloadStateExtendedStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended e8RecentCallListDownloadState;
   most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended e8PhoneBookDownloadState;
   virtual ~most_fi_tcl_PhonBkDownloadStateExtendedStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadStateExtendedStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadStateExtendedStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadStateExtendedStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadStateExtendedStream& operator=(const most_fi_tcl_PhonBkDownloadStateExtendedStream& coRef);
   most_fi_tcl_PhonBkDownloadStateExtendedStream(const most_fi_tcl_PhonBkDownloadStateExtendedStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_PhonBkDownloadStateExtendedStreamItem, std::allocator<most_fi_tcl_PhonBkDownloadStateExtendedStreamItem> >oItems;

   most_fi_tcl_PhonBkDownloadStateExtendedStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkDownloadStateExtendedStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadStateExtendedStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDeviceHandleList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDeviceHandleList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDeviceHandleList& operator=(const most_fi_tcl_PhonBkDeviceHandleList& coRef);
   most_fi_tcl_PhonBkDeviceHandleList(const most_fi_tcl_PhonBkDeviceHandleList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >oDeviceHandles;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkDeviceHandleList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDeviceHandleList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkSearchSortType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkSearchSortType();
   enum tenType {
      FI_EN_E8PB_SEARCH_SORT_LASTNAME = 0UL,
      FI_EN_E8PB_SEARCH_SORT_FIRSTNAME = 1UL,
      FI_EN_E8PB_SEARCH_SORT_ADDRESS_LASTNAME = 2UL,
      FI_EN_E8PB_SEARCH_SORT_ADDRESS_FIRSTNAME = 3UL,
      FI_EN_E8PB_SEARCH_SORT_NUMBER_LASTNAME = 4UL,
      FI_EN_E8PB_SEARCH_SORT_NUMBER_FIRSTNAME = 5UL,
      FI_EN_E8PB_SEARCH_SORT_EMAIL_LASTNAME = 6UL,
      FI_EN_E8PB_SEARCH_SORT_EMAIL_FIRSTNAME = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkSearchSortType(most_fi_tcl_e8_PhonBkSearchSortType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkSearchSortType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkSearchSortType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkContactMatchListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkContactMatchListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkContactMatchListItem& operator=(const most_fi_tcl_PhonBkContactMatchListItem& coRef);
   most_fi_tcl_PhonBkContactMatchListItem(const most_fi_tcl_PhonBkContactMatchListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sReportedFirstName;
   most_fi_tcl_String sReportedLastName;
   tU32 u32ContactHandle;
   tU8 u8PhoneNumberCount;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber1;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber2;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber3;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber4;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber5;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber6;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber7;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber8;
   tU8 u8EmailCount;
   most_fi_tcl_String oEmailAddr1;
   most_fi_tcl_String oEmailAddr2;
   most_fi_tcl_String oEmailAddr3;
   tU8 u8AddressesCount;
   tBool bPhotoAvailable;
   virtual ~most_fi_tcl_PhonBkContactMatchListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkContactMatchListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkContactMatchList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkContactMatchList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkContactMatchList& operator=(const most_fi_tcl_PhonBkContactMatchList& coRef);
   most_fi_tcl_PhonBkContactMatchList(const most_fi_tcl_PhonBkContactMatchList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_PhonBkContactMatchListItem*, std::allocator<most_fi_tcl_PhonBkContactMatchListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_PhonBkContactMatchListItem, std::allocator<most_fi_tcl_PhonBkContactMatchListItem> >oItems;

   most_fi_tcl_PhonBkContactMatchListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkContactMatchList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkContactMatchList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState();
   enum tenType {
      FI_EN_E8PB_SWITCH_DL_ON = 0UL,
      FI_EN_E8PB_SWITCH_DL_OFF = 1UL,
      FI_EN_E8PB_SWITCH_DL_IN_TRANSITION_TO_ON = 2UL,
      FI_EN_E8PB_SWITCH_DL_IN_TRANSITION_TO_OFF = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState(most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadOnOffStateListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadOnOffStateListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadOnOffStateListItem& operator=(const most_fi_tcl_PhonBkDownloadOnOffStateListItem& coRef);
   most_fi_tcl_PhonBkDownloadOnOffStateListItem(const most_fi_tcl_PhonBkDownloadOnOffStateListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 DeviceHandle;
   most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState DownloadOnOffState;
   virtual ~most_fi_tcl_PhonBkDownloadOnOffStateListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadOnOffStateListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadOnOffStateList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadOnOffStateList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadOnOffStateList& operator=(const most_fi_tcl_PhonBkDownloadOnOffStateList& coRef);
   most_fi_tcl_PhonBkDownloadOnOffStateList(const most_fi_tcl_PhonBkDownloadOnOffStateList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_PhonBkDownloadOnOffStateListItem, std::allocator<most_fi_tcl_PhonBkDownloadOnOffStateListItem> >oDownloadOnOffStateListItem;

   most_fi_tcl_PhonBkDownloadOnOffStateListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkDownloadOnOffStateList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadOnOffStateList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactSortType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactSortType();
   enum tenType {
      FI_EN_E8PB_LIST_SORT_LASTNAME = 0UL,
      FI_EN_E8PB_LIST_SORT_FIRSTNAME = 1UL,
      FI_EN_E8PB_LIST_SORT_ADDRESS_LASTNAME = 2UL,
      FI_EN_E8PB_LIST_SORT_ADDRESS_FIRSTNAME = 3UL,
      FI_EN_E8PB_LIST_SORT_NUMBER_LASTNAME = 4UL,
      FI_EN_E8PB_LIST_SORT_NUMBER_FIRSTNAME = 5UL,
      FI_EN_E8PB_LIST_SORT_EMAIL_LASTNAME = 6UL,
      FI_EN_E8PB_LIST_SORT_EMAIL_FIRSTNAME = 7UL,
      FI_EN_E8PB_LIST_SORT_CATEGORY_LASTNAME = 8UL,
      FI_EN_E8PB_LIST_SORT_CATEGORY_FIRSTNAME = 9UL,
      FI_EN_E8PB_LIST_SORT_POSITION = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactSortType(most_fi_tcl_e8_PhonBkContactSortType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactSortType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactSortType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PhonBkContactListType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PhonBkContactListType();
   enum tenType {
      FI_EN_PHONEBOOK = 0UL,
      FI_EN_FAVORITE = 1UL,
      FI_EN_VEHICLE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_PhonBkContactListType(most_fi_tcl_e8_PhonBkContactListType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PhonBkContactListType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PhonBkContactListType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkContactDetailsExtended : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkContactDetailsExtended();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkContactDetailsExtended& operator=(const most_fi_tcl_PhonBkContactDetailsExtended& coRef);
   most_fi_tcl_PhonBkContactDetailsExtended(const most_fi_tcl_PhonBkContactDetailsExtended& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sFirstName;
   most_fi_tcl_String sLastName;
   most_fi_tcl_String sPOIName;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber1;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber2;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber3;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber4;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber5;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber6;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber7;
   most_fi_tcl_PhonBkPhoneNumberItem oPhoneNumber8;
   most_fi_tcl_String sEmailAddress1;
   most_fi_tcl_String sEmailAddress2;
   most_fi_tcl_String sEmailAddress3;
   most_fi_tcl_String sHomeAddress;
   most_fi_tcl_String sWorkAddress;
   most_fi_tcl_String sOtherAddress;
   most_fi_tcl_String sCategory;
   most_fi_tcl_String sNotes;
   tBool bPhotoAvailable;
   most_fi_tcl_PhonBkHomeAddressLatLong oHomeAddressLatLong;
   most_fi_tcl_PhonBkWorkAddressLatLong oWorkAddressLatLong;
   most_fi_tcl_PhonBkOtherAddressLatLong oOtherAddressLatLong;
   virtual ~most_fi_tcl_PhonBkContactDetailsExtended();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkContactDetailsExtended& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AutomaticContactDownloadListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AutomaticContactDownloadListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AutomaticContactDownloadListItem& operator=(const most_fi_tcl_AutomaticContactDownloadListItem& coRef);
   most_fi_tcl_AutomaticContactDownloadListItem(const most_fi_tcl_AutomaticContactDownloadListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 DeviceHandle;
   tBool bAutomaticDL;
   virtual ~most_fi_tcl_AutomaticContactDownloadListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AutomaticContactDownloadListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AutomaticContactDownloadList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AutomaticContactDownloadList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AutomaticContactDownloadList& operator=(const most_fi_tcl_AutomaticContactDownloadList& coRef);
   most_fi_tcl_AutomaticContactDownloadList(const most_fi_tcl_AutomaticContactDownloadList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AutomaticContactDownloadListItem, std::allocator<most_fi_tcl_AutomaticContactDownloadListItem> >oAutomaticContactDownloadListItem;

   most_fi_tcl_AutomaticContactDownloadListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AutomaticContactDownloadList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AutomaticContactDownloadList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevicePhoneBookFeatureSupportListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevicePhoneBookFeatureSupportListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevicePhoneBookFeatureSupportListItem& operator=(const most_fi_tcl_DevicePhoneBookFeatureSupportListItem& coRef);
   most_fi_tcl_DevicePhoneBookFeatureSupportListItem(const most_fi_tcl_DevicePhoneBookFeatureSupportListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tBool bPhoneBookSupport;
   tBool bCCHListSupport;
   tBool bICHListSupport;
   tBool bOCHListSupport;
   tBool bMCHListSupport;
   tBool bContactListSortView;
   virtual ~most_fi_tcl_DevicePhoneBookFeatureSupportListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevicePhoneBookFeatureSupportListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevicePhoneBookFeatureSupportList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevicePhoneBookFeatureSupportList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevicePhoneBookFeatureSupportList& operator=(const most_fi_tcl_DevicePhoneBookFeatureSupportList& coRef);
   most_fi_tcl_DevicePhoneBookFeatureSupportList(const most_fi_tcl_DevicePhoneBookFeatureSupportList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_DevicePhoneBookFeatureSupportListItem, std::allocator<most_fi_tcl_DevicePhoneBookFeatureSupportListItem> >oItems;

   most_fi_tcl_DevicePhoneBookFeatureSupportListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DevicePhoneBookFeatureSupportList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevicePhoneBookFeatureSupportList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadProgressListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadProgressListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadProgressListItem& operator=(const most_fi_tcl_PhonBkDownloadProgressListItem& coRef);
   most_fi_tcl_PhonBkDownloadProgressListItem(const most_fi_tcl_PhonBkDownloadProgressListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32NumberOfContacts;
   tU8 u8Percentage;
   tU32 u32ContactsReportedByPhone;
   virtual ~most_fi_tcl_PhonBkDownloadProgressListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadProgressListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkDownloadProgressList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkDownloadProgressList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkDownloadProgressList& operator=(const most_fi_tcl_PhonBkDownloadProgressList& coRef);
   most_fi_tcl_PhonBkDownloadProgressList(const most_fi_tcl_PhonBkDownloadProgressList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_PhonBkDownloadProgressListItem, std::allocator<most_fi_tcl_PhonBkDownloadProgressListItem> >oItems;

   most_fi_tcl_PhonBkDownloadProgressListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkDownloadProgressList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkDownloadProgressList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkMissedCallIndicatorListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkMissedCallIndicatorListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkMissedCallIndicatorListItem& operator=(const most_fi_tcl_PhonBkMissedCallIndicatorListItem& coRef);
   most_fi_tcl_PhonBkMissedCallIndicatorListItem(const most_fi_tcl_PhonBkMissedCallIndicatorListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32NumMissedCalls;
   virtual ~most_fi_tcl_PhonBkMissedCallIndicatorListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkMissedCallIndicatorListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkMissedCallIndicatorList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkMissedCallIndicatorList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkMissedCallIndicatorList& operator=(const most_fi_tcl_PhonBkMissedCallIndicatorList& coRef);
   most_fi_tcl_PhonBkMissedCallIndicatorList(const most_fi_tcl_PhonBkMissedCallIndicatorList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_PhonBkMissedCallIndicatorListItem, std::allocator<most_fi_tcl_PhonBkMissedCallIndicatorListItem> >oItems;

   most_fi_tcl_PhonBkMissedCallIndicatorListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkMissedCallIndicatorList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkMissedCallIndicatorList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPreferredSortOrderListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPreferredSortOrderListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPreferredSortOrderListItem& operator=(const most_fi_tcl_PhonBkPreferredSortOrderListItem& coRef);
   most_fi_tcl_PhonBkPreferredSortOrderListItem(const most_fi_tcl_PhonBkPreferredSortOrderListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder e8PreferredPhoneBookSortOrder;
   virtual ~most_fi_tcl_PhonBkPreferredSortOrderListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPreferredSortOrderListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_PhonBkPreferredSortOrderList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_PhonBkPreferredSortOrderList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_PhonBkPreferredSortOrderList& operator=(const most_fi_tcl_PhonBkPreferredSortOrderList& coRef);
   most_fi_tcl_PhonBkPreferredSortOrderList(const most_fi_tcl_PhonBkPreferredSortOrderList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_PhonBkPreferredSortOrderListItem, std::allocator<most_fi_tcl_PhonBkPreferredSortOrderListItem> >oItems;

   most_fi_tcl_PhonBkPreferredSortOrderListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_PhonBkPreferredSortOrderList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_PhonBkPreferredSortOrderList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PbkLanguage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PbkLanguage();
   enum tenType {
      FI_EN_NA_ENGLISH = 0UL,
      FI_EN_GERMAN = 1UL,
      FI_EN_ITALIAN = 2UL,
      FI_EN_SWEDISH = 3UL,
      FI_EN_FRENCH = 4UL,
      FI_EN_SPANISH = 5UL,
      FI_EN_DUTCH = 6UL,
      FI_EN_PORTUGUESE = 7UL,
      FI_EN_NORWEGIAN = 8UL,
      FI_EN_FINNISH = 9UL,
      FI_EN_DANISH = 10UL,
      FI_EN_GREEK = 11UL,
      FI_EN_JAPANESE = 12UL,
      FI_EN_ARABIC = 13UL,
      FI_EN_STANDARD_CHINESE = 14UL,
      FI_EN_POLISH = 15UL,
      FI_EN_TURKISH = 16UL,
      FI_EN_KOREAN = 17UL,
      FI_EN_TRADITIONAL_CHINESE = 18UL,
      FI_EN_UK_ENGLISH = 19UL,
      FI_EN_HUNGARIAN = 20UL,
      FI_EN_CZECH = 21UL,
      FI_EN_SLOVAK = 22UL,
      FI_EN_RUSSIAN = 23UL,
      FI_EN_BRAZILIAN_PORTUGUESE = 24UL,
      FI_EN_THAI = 25UL,
      FI_EN_BULGARIAN = 26UL,
      FI_EN_ROMANIAN = 27UL,
      FI_EN_SLOVENIAN = 28UL,
      FI_EN_CROATIAN = 29UL,
      FI_EN_UKRAINIAN = 30UL,
      FI_EN_NA_FRENCH = 31UL,
      FI_EN_NA_SPANISH = 32UL,
      FI_EN_CANTONESE = 33UL,
      FI_EN_UZBEK = 34UL,
      FI_EN_SERBIAN = 35UL,
      FI_EN_LITHUANIAN = 36UL,
      FI_EN_ESTONIAN = 37UL,
      FI_EN_LATVIAN = 38UL,
      FI_EN_ALBANESE = 39UL,
      FI_EN_MANDARIN_CHINESE = 40UL,
      FI_EN_AUSTRALIAN_ENGLISH = 41UL,
      FI_EN_US_JAPAN_ENGLISH = 42UL,
      FI_EN_US_PRC_ENGLISH = 43UL,
      FI_EN_CANADIAN_ENGLISH = 44UL,
      FI_EN_INDIA_ENGLISH = 45UL,
      FI_EN_FILIPINO = 46UL,
      FI_EN_FLEMISH = 47UL,
      FI_EN_CANADIAN_FRENCH = 48UL,
      FI_EN_HEBREW = 49UL,
      FI_EN_HINDI = 50UL,
      FI_EN_INDONESIAN = 51UL,
      FI_EN_MALAY = 52UL,
      FI_EN_PERSIAN = 53UL,
      FI_EN_LATIN_AMERICAN_SPANISH = 54UL,
      FI_EN_VIETNAMESE = 55UL
   };
   tenType enType;

   most_fi_tcl_e8_PbkLanguage(most_fi_tcl_e8_PbkLanguage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PbkLanguage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PbkLanguage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavPrvAudioChannel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavPrvAudioChannel();
   enum tenType {
      FI_EN_E8LC_NONE = 0UL,
      FI_EN_E8LC_MAIN_AUDIO = 1UL,
      FI_EN_E8LC_PHONE = 5UL,
      FI_EN_E8LC_REAR_AUDIO = 17UL,
      FI_EN_E8LC_REAR_PHONE = 20UL,
      FI_EN_E8LC_HEADPHONE_1 = 33UL,
      FI_EN_E8LC_HEADPHONE_2 = 34UL,
      FI_EN_E8LC_HEADPHONE_3 = 35UL,
      FI_EN_E8LC_HEADPHONE_4 = 36UL
   };
   tenType enType;

   most_fi_tcl_e8_FavPrvAudioChannel(most_fi_tcl_e8_FavPrvAudioChannel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavPrvAudioChannel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavPrvAudioChannel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavPrvVideoChannel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavPrvVideoChannel();
   enum tenType {
      FI_EN_E8LC_NONE = 0UL
   };
   tenType enType;

   most_fi_tcl_e8_FavPrvVideoChannel(most_fi_tcl_e8_FavPrvVideoChannel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavPrvVideoChannel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavPrvVideoChannel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavPrvFavoriteType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavPrvFavoriteType();
   enum tenType {
      FI_EN_E8FT_ALL = 0UL,
      FI_EN_E8FT_BLANK = 1UL,
      FI_EN_E8FT_AM_FREQUENCY = 2UL,
      FI_EN_E8FT_FM_FREQUENCY = 3UL,
      FI_EN_E8FT_SDARS_CHANNEL = 4UL,
      FI_EN_E8FT_PANDORA_STATION = 5UL,
      FI_EN_E8FT_INTERNET_STATION = 6UL,
      FI_EN_E8FT_PLAYLIST = 7UL,
      FI_EN_E8FT_MEDIA = 8UL,
      FI_EN_E8FT_PODCAST = 9UL,
      FI_EN_E8FT_VIDEO = 10UL,
      FI_EN_E8FT_AUDIO_BOOK = 11UL,
      FI_EN_E8FT_POI_CATEGORY = 12UL,
      FI_EN_E8FT_DESTINATION = 13UL,
      FI_EN_E8FT_FUNCTION = 14UL,
      FI_EN_E8FT_APPLICATION = 15UL,
      FI_EN_E8FT_CONTACT_NAME = 16UL,
      FI_EN_E8FT_PHONE_NUMBER = 17UL,
      FI_EN_E8FT_TONE_CONTROLS = 18UL,
      FI_EN_E8FT_SEARCH_TERM_POI = 19UL,
      FI_EN_E8FT_SEARCH_TERM_ADDRESS = 20UL,
      FI_EN_E8FT_POI_CHAIN = 21UL,
      FI_EN_E8FT_MYROUTE = 22UL,
      FI_EN_E8FT_SAVED_TRIP = 23UL,
      FI_EN_E8FT_DAB_STATION = 24UL
   };
   tenType enType;

   most_fi_tcl_e8_FavPrvFavoriteType(most_fi_tcl_e8_FavPrvFavoriteType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavPrvFavoriteType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavPrvFavoriteType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavPrvFavoriteInfoItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavPrvFavoriteInfoItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavPrvFavoriteInfoItem& operator=(const most_fi_tcl_FavPrvFavoriteInfoItem& coRef);
   most_fi_tcl_FavPrvFavoriteInfoItem(const most_fi_tcl_FavPrvFavoriteInfoItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bActive;
   tBool bAvailable;
   tU16 u16FavoriteID;
   most_fi_tcl_String sLabelText;
   most_fi_tcl_String sDescriptorText;
   most_fi_tcl_e8_FavPrvFavoriteType e8FavoriteType;
   virtual ~most_fi_tcl_FavPrvFavoriteInfoItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavPrvFavoriteInfoItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavPrvFavoriteInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavPrvFavoriteInfo() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavPrvFavoriteInfo& operator=(const most_fi_tcl_FavPrvFavoriteInfo& coRef);
   most_fi_tcl_FavPrvFavoriteInfo(const most_fi_tcl_FavPrvFavoriteInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_FavPrvFavoriteInfoItem*, std::allocator<most_fi_tcl_FavPrvFavoriteInfoItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_FavPrvFavoriteInfoItem, std::allocator<most_fi_tcl_FavPrvFavoriteInfoItem> >oItems;

   most_fi_tcl_FavPrvFavoriteInfoItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_FavPrvFavoriteInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavPrvFavoriteInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavPrvListChangeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavPrvListChangeType();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_FavPrvListChangeType(most_fi_tcl_e8_FavPrvListChangeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavPrvListChangeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavPrvListChangeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavPrvChangedFavorites : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavPrvChangedFavorites() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavPrvChangedFavorites& operator=(const most_fi_tcl_FavPrvChangedFavorites& coRef);
   most_fi_tcl_FavPrvChangedFavorites(const most_fi_tcl_FavPrvChangedFavorites& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU16, std::allocator<tU16> >u16Items;

   tU16& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_FavPrvChangedFavorites();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavPrvChangedFavorites& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SrcAudioChannelName : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SrcAudioChannelName();
   enum tenType {
      FI_EN_E8AUDIO_CHANNEL_NOT_IN_USE = 0UL,
      FI_EN_E8AUDIO_CHANNEL_FRONT_LEFT = 1UL,
      FI_EN_E8AUDIO_CHANNEL_FRONT_RIGHT = 2UL,
      FI_EN_E8AUDIO_CHANNEL_FRONT_CENTER = 3UL,
      FI_EN_E8AUDIO_CHANNEL_LOW_FREQUENCY = 4UL,
      FI_EN_E8AUDIO_CHANNEL_BACK_LEFT = 5UL,
      FI_EN_E8AUDIO_CHANNEL_BACK_RIGHT = 6UL,
      FI_EN_E8AUDIO_CHANNEL_FRONT_LEFT_CENTER = 7UL,
      FI_EN_E8AUDIO_CHANNEL_FRONT_RIGHT_CENTER = 8UL,
      FI_EN_E8AUDIO_CHANNEL_BACK_CENTER = 9UL,
      FI_EN_E8AUDIO_CHANNEL_SIDE_LEFT = 10UL,
      FI_EN_E8AUDIO_CHANNEL_SIDE_RIGHT = 11UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_CENTER = 12UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_FRONT_LEFT = 13UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_FRONT_CENTER = 14UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_FRONT_RIGHT = 15UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_BACK_LEFT = 16UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_BACK_CENTER = 17UL,
      FI_EN_E8AUDIO_CHANNEL_TOP_BACK_RIGHT = 18UL,
      FI_EN_E8AUDIO_CHANNEL_BACK_LEFT_CENTER = 19UL,
      FI_EN_E8AUDIO_CHANNEL_BACK_RIGHT_CENTER = 20UL
   };
   tenType enType;

   most_fi_tcl_e8_SrcAudioChannelName(most_fi_tcl_e8_SrcAudioChannelName::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SrcAudioChannelName();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SrcAudioChannelName& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SrcContentProtection : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SrcContentProtection();
   enum tenType {
      FI_EN_E8CONTENT_PROTECTION_NONE = 0UL,
      FI_EN_E8CONTENT_PROTECTION_SERIAL_COPY_MANAGEMENT = 1UL,
      FI_EN_E8CONTENT_PROTECTION_MOST_DTCP = 2UL,
      FI_EN_E8CONTENT_PROTECTION_DTCP_IP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_SrcContentProtection(most_fi_tcl_e8_SrcContentProtection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SrcContentProtection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SrcContentProtection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SrcContentType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SrcContentType();
   enum tenType {
      FI_EN_E8CONTENT_TYPE_AUDIO = 0UL,
      FI_EN_E8CONTENT_TYPE_MULTI_CHANNEL_AUDIO = 16UL,
      FI_EN_E8CONTENT_TYPE_MPEG1_SYSTEM_STREAM = 32UL,
      FI_EN_E8CONTENT_TYPE_MPEG2_PROGRAM_STREAM = 33UL,
      FI_EN_E8CONTENT_TYPE_MPEG2_TRANSPORT_STREAM = 34UL
   };
   tenType enType;

   most_fi_tcl_e8_SrcContentType(most_fi_tcl_e8_SrcContentType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SrcContentType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SrcContentType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SrcTransmissionClass : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SrcTransmissionClass();
   enum tenType {
      FI_EN_E8TRANSMISSION_CLASS_SYNCHRONOUS = 0UL
   };
   tenType enType;

   most_fi_tcl_e8_SrcTransmissionClass(most_fi_tcl_e8_SrcTransmissionClass::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SrcTransmissionClass();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SrcTransmissionClass& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SrcAudioChannels_Resolution : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SrcAudioChannels_Resolution();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SrcAudioChannels_Resolution& operator=(const most_fi_tcl_SrcAudioChannels_Resolution& coRef);
   most_fi_tcl_SrcAudioChannels_Resolution(const most_fi_tcl_SrcAudioChannels_Resolution& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8AudioChannels;
   tU8 u8Resolution;
   virtual ~most_fi_tcl_SrcAudioChannels_Resolution();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SrcAudioChannels_Resolution& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SrcAudioChannelListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SrcAudioChannelListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SrcAudioChannelListItem& operator=(const most_fi_tcl_SrcAudioChannelListItem& coRef);
   most_fi_tcl_SrcAudioChannelListItem(const most_fi_tcl_SrcAudioChannelListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_SrcAudioChannelName e8AudioChannelName;
   tU8 u8BitsPerSample;
   virtual ~most_fi_tcl_SrcAudioChannelListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SrcAudioChannelListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SrcAudioChannelList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SrcAudioChannelList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SrcAudioChannelList& operator=(const most_fi_tcl_SrcAudioChannelList& coRef);
   most_fi_tcl_SrcAudioChannelList(const most_fi_tcl_SrcAudioChannelList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_SrcAudioChannelListItem, std::allocator<most_fi_tcl_SrcAudioChannelListItem> >oItems;

   most_fi_tcl_SrcAudioChannelListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_SrcAudioChannelList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SrcAudioChannelList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SrcAudioChannels_AudioChannelList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SrcAudioChannels_AudioChannelList();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SrcAudioChannels_AudioChannelList& operator=(const most_fi_tcl_SrcAudioChannels_AudioChannelList& coRef);
   most_fi_tcl_SrcAudioChannels_AudioChannelList(const most_fi_tcl_SrcAudioChannels_AudioChannelList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8AudioChannels;
   most_fi_tcl_SrcAudioChannelList oAudioChannelList;
   virtual ~most_fi_tcl_SrcAudioChannels_AudioChannelList();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SrcAudioChannels_AudioChannelList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_SrcContentDescription : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_SrcContentDescription():u32DynamicSize(0), _ActiveMemberOffset(0u), pau8Data(OSAL_NULL), _oAudioChannels_Resolution(), _oAudioChannels_AudioChannelList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_SrcContentDescription& operator=(const most_fi_tcl_SrcContentDescription& coRef);
   most_fi_tcl_SrcContentDescription(const most_fi_tcl_SrcContentDescription& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   tU32 u32DynamicSize;
   size_t _ActiveMemberOffset;
   tU8* pau8Data;

   most_fi_tcl_SrcAudioChannels_Resolution _oAudioChannels_Resolution;
   most_fi_tcl_SrcAudioChannels_AudioChannelList _oAudioChannels_AudioChannelList;

   // helper functions
   inline size_t Offset(const most_fi_tclVisitableTypeBase& roMember) const
      { return (size_t)(&roMember) - (size_t)this; }
   const most_fi_tclVisitableTypeBase* pcActiveMember() const;
   most_fi_tclVisitableTypeBase& roVariant(most_fi_tclVisitableTypeBase& roSelectedVariant);

public:
   // variant access functions
   most_fi_tcl_SrcAudioChannels_Resolution& oAudioChannels_Resolution();
   most_fi_tcl_SrcAudioChannels_AudioChannelList& oAudioChannels_AudioChannelList();
   virtual ~most_fi_tcl_SrcContentDescription();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_SrcContentDescription& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SrcActivity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SrcActivity();
   enum tenType {
      FI_EN_E8SA_OFF = 0UL,
      FI_EN_E8SA_RESTRICTED_DO_NOT_USE = 1UL,
      FI_EN_E8SA_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_SrcActivity(most_fi_tcl_e8_SrcActivity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SrcActivity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SrcActivity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelCallStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelCallStatus();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8DIALING = 1UL,
      FI_EN_E8ACTIVE = 2UL,
      FI_EN_E8ON_HOLD = 3UL,
      FI_EN_E8DISCONNECTING = 4UL,
      FI_EN_E8BUSY = 5UL,
      FI_EN_E8CONFERENCE = 6UL,
      FI_EN_E8IN_VOICEMAIL = 7UL,
      FI_EN_E8RINGTONE = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_TelCallStatus(most_fi_tcl_e8_TelCallStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelCallStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelCallStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelHandsetState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelHandsetState();
   enum tenType {
      FI_EN_E8NO_CALL = 0UL,
      FI_EN_E8ACTIVE_CALL = 1UL,
      FI_EN_E8HOLD_CALL = 2UL,
      FI_EN_E8ACTIVEANDHOLD_CALL = 4UL,
      FI_EN_E8MULTIPARTY_CALL = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_TelHandsetState(most_fi_tcl_e8_TelHandsetState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelHandsetState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelHandsetState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelActiveCallinHandsetStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelActiveCallinHandsetStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelActiveCallinHandsetStreamItem& operator=(const most_fi_tcl_TelActiveCallinHandsetStreamItem& coRef);
   most_fi_tcl_TelActiveCallinHandsetStreamItem(const most_fi_tcl_TelActiveCallinHandsetStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_TelHandsetState e8HandsetState;
   tU8 u8DeviceHandle;
   virtual ~most_fi_tcl_TelActiveCallinHandsetStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelActiveCallinHandsetStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelActiveCallinHandsetStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelActiveCallinHandsetStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelActiveCallinHandsetStream& operator=(const most_fi_tcl_TelActiveCallinHandsetStream& coRef);
   most_fi_tcl_TelActiveCallinHandsetStream(const most_fi_tcl_TelActiveCallinHandsetStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_TelActiveCallinHandsetStreamItem, std::allocator<most_fi_tcl_TelActiveCallinHandsetStreamItem> >oItems;

   most_fi_tcl_TelActiveCallinHandsetStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelActiveCallinHandsetStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelActiveCallinHandsetStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelCallStatusNoticeStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelCallStatusNoticeStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelCallStatusNoticeStreamItem& operator=(const most_fi_tcl_TelCallStatusNoticeStreamItem& coRef);
   most_fi_tcl_TelCallStatusNoticeStreamItem(const most_fi_tcl_TelCallStatusNoticeStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16CallInstance;
   most_fi_tcl_e8_TelCallStatus e8CallStatus;
   tBool bUsingVehicleAudio;
   tU8 u8CallDurationHr;
   tU8 u8CallDurationMin;
   tU8 u8CallDurationSec;
   tU8 u8DeviceHandle;
   virtual ~most_fi_tcl_TelCallStatusNoticeStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelCallStatusNoticeStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelCallStatusNoticeStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelCallStatusNoticeStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelCallStatusNoticeStream& operator=(const most_fi_tcl_TelCallStatusNoticeStream& coRef);
   most_fi_tcl_TelCallStatusNoticeStream(const most_fi_tcl_TelCallStatusNoticeStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_TelCallStatusNoticeStreamItem, std::allocator<most_fi_tcl_TelCallStatusNoticeStreamItem> >oItems;

   most_fi_tcl_TelCallStatusNoticeStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelCallStatusNoticeStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelCallStatusNoticeStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelCellService : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelCellService();
   enum tenType {
      FI_EN_E8NO_SERVICE = 0UL,
      FI_EN_E8SERVICE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_TelCellService(most_fi_tcl_e8_TelCellService::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelCellService();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelCellService& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelStreamOfCallInstances : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelStreamOfCallInstances() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelStreamOfCallInstances& operator=(const most_fi_tcl_TelStreamOfCallInstances& coRef);
   most_fi_tcl_TelStreamOfCallInstances(const most_fi_tcl_TelStreamOfCallInstances& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU16, std::allocator<tU16> >u16Items;

   tU16& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelStreamOfCallInstances();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelStreamOfCallInstances& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelIncomingCallEventEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelIncomingCallEventEnum();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8RINGING = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_TelIncomingCallEventEnum(most_fi_tcl_e8_TelIncomingCallEventEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelIncomingCallEventEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelIncomingCallEventEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting();
   enum tenType {
      FI_EN_E8ECNR_NOCHANGE = 0UL,
      FI_EN_E8ECNR_VOICE = 1UL,
      FI_EN_E8ECNR_SERVER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting(most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelHFEnhancedCallControlStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelHFEnhancedCallControlStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelHFEnhancedCallControlStreamItem& operator=(const most_fi_tcl_TelHFEnhancedCallControlStreamItem& coRef);
   most_fi_tcl_TelHFEnhancedCallControlStreamItem(const most_fi_tcl_TelHFEnhancedCallControlStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tBool bEndTargetCall;
   tBool bSplitMergedCalls;
   virtual ~most_fi_tcl_TelHFEnhancedCallControlStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelHFEnhancedCallControlStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelHFEnhancedCallControlStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelHFEnhancedCallControlStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelHFEnhancedCallControlStream& operator=(const most_fi_tcl_TelHFEnhancedCallControlStream& coRef);
   most_fi_tcl_TelHFEnhancedCallControlStream(const most_fi_tcl_TelHFEnhancedCallControlStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_TelHFEnhancedCallControlStreamItem, std::allocator<most_fi_tcl_TelHFEnhancedCallControlStreamItem> >oItems;

   most_fi_tcl_TelHFEnhancedCallControlStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelHFEnhancedCallControlStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelHFEnhancedCallControlStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelInbandRingingSupport : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelInbandRingingSupport();
   enum tenType {
      FI_EN_E8NOT_SUPPORTED = 0UL,
      FI_EN_E8SUPPORTED_ENABLED = 1UL,
      FI_EN_E8SUPPORTED_DISABLED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_TelInbandRingingSupport(most_fi_tcl_e8_TelInbandRingingSupport::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelInbandRingingSupport();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelInbandRingingSupport& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelDeviceRingtoneListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelDeviceRingtoneListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelDeviceRingtoneListResultItem& operator=(const most_fi_tcl_TelDeviceRingtoneListResultItem& coRef);
   most_fi_tcl_TelDeviceRingtoneListResultItem(const most_fi_tcl_TelDeviceRingtoneListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBdAddress;
   tU8 u8RingtoneID;
   most_fi_tcl_String sRingtoneName;
   most_fi_tcl_e8_TelInbandRingingSupport e8InbandRingingSupport;
   virtual ~most_fi_tcl_TelDeviceRingtoneListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelDeviceRingtoneListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelDeviceRingtoneListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelDeviceRingtoneListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelDeviceRingtoneListResult& operator=(const most_fi_tcl_TelDeviceRingtoneListResult& coRef);
   most_fi_tcl_TelDeviceRingtoneListResult(const most_fi_tcl_TelDeviceRingtoneListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_TelDeviceRingtoneListResultItem*, std::allocator<most_fi_tcl_TelDeviceRingtoneListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_TelDeviceRingtoneListResultItem, std::allocator<most_fi_tcl_TelDeviceRingtoneListResultItem> >oItems;

   most_fi_tcl_TelDeviceRingtoneListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelDeviceRingtoneListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelDeviceRingtoneListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelSystemWideRingtonesListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelSystemWideRingtonesListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelSystemWideRingtonesListResultItem& operator=(const most_fi_tcl_TelSystemWideRingtonesListResultItem& coRef);
   most_fi_tcl_TelSystemWideRingtonesListResultItem(const most_fi_tcl_TelSystemWideRingtonesListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8RingtoneID;
   most_fi_tcl_String sRingtoneName;
   virtual ~most_fi_tcl_TelSystemWideRingtonesListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelSystemWideRingtonesListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelSystemWideRingtonesListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelSystemWideRingtonesListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelSystemWideRingtonesListResult& operator=(const most_fi_tcl_TelSystemWideRingtonesListResult& coRef);
   most_fi_tcl_TelSystemWideRingtonesListResult(const most_fi_tcl_TelSystemWideRingtonesListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_TelSystemWideRingtonesListResultItem*, std::allocator<most_fi_tcl_TelSystemWideRingtonesListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_TelSystemWideRingtonesListResultItem, std::allocator<most_fi_tcl_TelSystemWideRingtonesListResultItem> >oItems;

   most_fi_tcl_TelSystemWideRingtonesListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelSystemWideRingtonesListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelSystemWideRingtonesListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelSuppressRingtoneOnOffListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelSuppressRingtoneOnOffListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelSuppressRingtoneOnOffListResultItem& operator=(const most_fi_tcl_TelSuppressRingtoneOnOffListResultItem& coRef);
   most_fi_tcl_TelSuppressRingtoneOnOffListResultItem(const most_fi_tcl_TelSuppressRingtoneOnOffListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBdAddress;
   tBool bSuppressRingtoneState;
   virtual ~most_fi_tcl_TelSuppressRingtoneOnOffListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelSuppressRingtoneOnOffListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelSuppressRingtoneOnOffListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelSuppressRingtoneOnOffListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelSuppressRingtoneOnOffListResult& operator=(const most_fi_tcl_TelSuppressRingtoneOnOffListResult& coRef);
   most_fi_tcl_TelSuppressRingtoneOnOffListResult(const most_fi_tcl_TelSuppressRingtoneOnOffListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_TelSuppressRingtoneOnOffListResultItem*, std::allocator<most_fi_tcl_TelSuppressRingtoneOnOffListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_TelSuppressRingtoneOnOffListResultItem, std::allocator<most_fi_tcl_TelSuppressRingtoneOnOffListResultItem> >oItems;

   most_fi_tcl_TelSuppressRingtoneOnOffListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelSuppressRingtoneOnOffListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelSuppressRingtoneOnOffListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelAutoWaitingModeOnOffListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelAutoWaitingModeOnOffListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelAutoWaitingModeOnOffListResultItem& operator=(const most_fi_tcl_TelAutoWaitingModeOnOffListResultItem& coRef);
   most_fi_tcl_TelAutoWaitingModeOnOffListResultItem(const most_fi_tcl_TelAutoWaitingModeOnOffListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sBdAddress;
   tBool bAutoWaitingModeState;
   virtual ~most_fi_tcl_TelAutoWaitingModeOnOffListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelAutoWaitingModeOnOffListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelAutoWaitingModeOnOffListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelAutoWaitingModeOnOffListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelAutoWaitingModeOnOffListResult& operator=(const most_fi_tcl_TelAutoWaitingModeOnOffListResult& coRef);
   most_fi_tcl_TelAutoWaitingModeOnOffListResult(const most_fi_tcl_TelAutoWaitingModeOnOffListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_TelAutoWaitingModeOnOffListResultItem*, std::allocator<most_fi_tcl_TelAutoWaitingModeOnOffListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_TelAutoWaitingModeOnOffListResultItem, std::allocator<most_fi_tcl_TelAutoWaitingModeOnOffListResultItem> >oItems;

   most_fi_tcl_TelAutoWaitingModeOnOffListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelAutoWaitingModeOnOffListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelAutoWaitingModeOnOffListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelWaitingModeAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelWaitingModeAction();
   enum tenType {
      FI_EN_E8START = 0UL,
      FI_EN_E8STOP = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_TelWaitingModeAction(most_fi_tcl_e8_TelWaitingModeAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelWaitingModeAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelWaitingModeAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelWaitingModeStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelWaitingModeStatus();
   enum tenType {
      FI_EN_E8WAITINGMODE_SUCCESS = 0UL,
      FI_EN_E8WAITINGMODE_FAILURE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_TelWaitingModeStatus(most_fi_tcl_e8_TelWaitingModeStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelWaitingModeStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelWaitingModeStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelSiriAvailabilityState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelSiriAvailabilityState();
   enum tenType {
      FI_EN_E8NOT_AVAILABLE = 0UL,
      FI_EN_E8AVAILABLE_ENABLED = 1UL,
      FI_EN_E8AVAILABLE_DISABLED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_TelSiriAvailabilityState(most_fi_tcl_e8_TelSiriAvailabilityState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelSiriAvailabilityState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelSiriAvailabilityState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelCallStatusNoticeExtendedStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelCallStatusNoticeExtendedStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelCallStatusNoticeExtendedStreamItem& operator=(const most_fi_tcl_TelCallStatusNoticeExtendedStreamItem& coRef);
   most_fi_tcl_TelCallStatusNoticeExtendedStreamItem(const most_fi_tcl_TelCallStatusNoticeExtendedStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16CallInstance;
   most_fi_tcl_e8_TelCallStatus e8CallStatus;
   tBool bUsingVehicleAudio;
   tU8 u8CallDurationHr;
   tU8 u8CallDurationMin;
   tU8 u8CallDurationSec;
   tU8 u8DeviceHandle;
   most_fi_tcl_String sTelephoneNumber;
   most_fi_tcl_String sName;
   virtual ~most_fi_tcl_TelCallStatusNoticeExtendedStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelCallStatusNoticeExtendedStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelCallStatusNoticeExtendedStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelCallStatusNoticeExtendedStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelCallStatusNoticeExtendedStream& operator=(const most_fi_tcl_TelCallStatusNoticeExtendedStream& coRef);
   most_fi_tcl_TelCallStatusNoticeExtendedStream(const most_fi_tcl_TelCallStatusNoticeExtendedStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem*, std::allocator<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem, std::allocator<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem> >oItems;

   most_fi_tcl_TelCallStatusNoticeExtendedStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_TelCallStatusNoticeExtendedStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelCallStatusNoticeExtendedStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelOutgoingCall_Failed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelOutgoingCall_Failed();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelOutgoingCall_Failed& operator=(const most_fi_tcl_TelOutgoingCall_Failed& coRef);
   most_fi_tcl_TelOutgoingCall_Failed(const most_fi_tcl_TelOutgoingCall_Failed& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_String sTelephoneNumber;
   most_fi_tcl_String sName;
   most_fi_tcl_String sOutgoingCallFailedReason;
   virtual ~most_fi_tcl_TelOutgoingCall_Failed();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelOutgoingCall_Failed& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_TelIncomingCall_Rejected : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_TelIncomingCall_Rejected();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TelIncomingCall_Rejected& operator=(const most_fi_tcl_TelIncomingCall_Rejected& coRef);
   most_fi_tcl_TelIncomingCall_Rejected(const most_fi_tcl_TelIncomingCall_Rejected& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   most_fi_tcl_String sTelephoneNumber;
   most_fi_tcl_String sIncomingCallRejectedReason;
   virtual ~most_fi_tcl_TelIncomingCall_Rejected();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_TelIncomingCall_Rejected& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_WaitingModeState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_WaitingModeState();
   enum tenType {
      FI_EN_E8WAITING_MODE_IDLE = 0UL,
      FI_EN_E8WAITING_MODE_ACTIVE = 1UL,
      FI_EN_E8WAITING_MODE_STARTING = 2UL,
      FI_EN_E8WAITING_MODE_STOPPING = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_WaitingModeState(most_fi_tcl_e8_WaitingModeState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_WaitingModeState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_WaitingModeState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ExternalVRState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ExternalVRState();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8READY = 1UL,
      FI_EN_E8ACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ExternalVRState(most_fi_tcl_e8_ExternalVRState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ExternalVRState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ExternalVRState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_VRErrorCode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_VRErrorCode();
   enum tenType {
      FI_EN_E8NO_ERROR = 0UL,
      FI_EN_E8ERROR_IN_STREAMING = 1UL,
      FI_EN_E8ERROR_PHONE_CALL_ACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_VRErrorCode(most_fi_tcl_e8_VRErrorCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_VRErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_VRErrorCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_ListOfCallInstanceStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_ListOfCallInstanceStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_ListOfCallInstanceStreamItem& operator=(const most_fi_tcl_ListOfCallInstanceStreamItem& coRef);
   most_fi_tcl_ListOfCallInstanceStreamItem(const most_fi_tcl_ListOfCallInstanceStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16CallInstance;
   virtual ~most_fi_tcl_ListOfCallInstanceStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_ListOfCallInstanceStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_ListOfCallInstanceStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_ListOfCallInstanceStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_ListOfCallInstanceStream& operator=(const most_fi_tcl_ListOfCallInstanceStream& coRef);
   most_fi_tcl_ListOfCallInstanceStream(const most_fi_tcl_ListOfCallInstanceStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_ListOfCallInstanceStreamItem, std::allocator<most_fi_tcl_ListOfCallInstanceStreamItem> >oItems;

   most_fi_tcl_ListOfCallInstanceStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_ListOfCallInstanceStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_ListOfCallInstanceStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AudioTransferDirection : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AudioTransferDirection();
   enum tenType {
      FI_EN_E8HF_TO_AG = 0UL,
      FI_EN_E8AG_TO_HF = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AudioTransferDirection(most_fi_tcl_e8_AudioTransferDirection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AudioTransferDirection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AudioTransferDirection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_TelPlaybackState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_TelPlaybackState();
   enum tenType {
      FI_EN_E8NOT_PLAYING = 0UL,
      FI_EN_E8STARTING = 1UL,
      FI_EN_E8PLAYING = 2UL,
      FI_EN_E8STOPPING = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_TelPlaybackState(most_fi_tcl_e8_TelPlaybackState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_TelPlaybackState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_TelPlaybackState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

#endif
