#include "TtfisLoggingBackend.h"
#include "asf/core/CallPthreadNoLogging.h"
#include <adit-components/trace_interface.h>
#include <algorithm>


using namespace ::asf::core;


#define DEFAULT_TRACE_CLASS           0x97C0


LoggingBackendIF* createLoggingBackend()
{
   return new TtfisLoggingBackend();
}


void destroyLoggingBackend(LoggingBackendIF* lb)
{
   delete lb;
}


void getLoggingBackendVersion(uint32& major, uint32& minor, uint32& patch)
{
   major = LoggingBackendIF::API_MAJOR_VERSION;
   minor = 0;
   patch = 0;
}


TtfisLoggingBackend::TtfisLoggingBackend()
   : _detailedHeader(false)
   , _traceClass(DEFAULT_TRACE_CLASS)
{
    CALL_PTHREAD(pthread_mutexattr_init(&_attr));
    CALL_PTHREAD(pthread_mutexattr_settype(&_attr, PTHREAD_MUTEX_RECURSIVE));
    CALL_PTHREAD(pthread_mutex_init(&_mutex, &_attr));
}


TtfisLoggingBackend::~TtfisLoggingBackend()
{
    CALL_PTHREAD(pthread_mutex_destroy(&_mutex));
    CALL_PTHREAD(pthread_mutexattr_destroy(&_attr));
}


/**
 * Returns the currently configured TTFis trace level for 'traceClass'.
 * Since the trace lib does not provide a function to retrieve the
 * current trace level, we have to iterate through all trace levels
 * from highest to lowest level until reaching an enabled level.
 */
static TR_tenTraceLevel getConfiguredTtfisLevel(unsigned short traceClass)
{
   for (size_t level = TR_LEVEL_USER_4; level > TR_LEVEL_FATAL; level--)
   {
      if (TR_core_bIsClassSelected(traceClass, (TR_tenTraceLevel)level))
      {
         return (TR_tenTraceLevel)level;
      }
   }
   return TR_LEVEL_FATAL;
}


static inline std::string toString(size_t value)
{
   char str[20];
   snprintf(str, sizeof str, "%zd", value);
   return str;
}


static inline void traceTtfis(unsigned short traceClass, TR_tenTraceLevel level, const std::string& text)
{
   if (!text.empty())
   {
   TR_core_uwTraceOut(static_cast<U32>(text.size()), traceClass, level, (U8*)text.c_str());
}
}


/**
 * Logs each line in the 'text' string to a separate TTFis trace line.
 * For instance, this is important for correctly tracing message payloads.
 */
static void traceMultiline(unsigned short traceClass, TR_tenTraceLevel level, const std::string& text)
{
   size_t start = 0;
   while (1)
   {
      size_t end = text.find('\n', start);
      if (end == std::string::npos)
      {
         std::string sub = text.substr(start);
         traceTtfis(traceClass, level, sub);
         break;
      }
      else
      {
         std::string sub = text.substr(start, end - start);
         traceTtfis(traceClass, level, sub);
         start = end + 1;  // skip linebreak
      }
   }
}


/**
 * Set the logger level of all loggers of the application to 'level'.
 */
static void enforceAsfLogLevel(Logging::Level level, LoggerAccessorIF& accessor)
{
   const std::map<std::string, Logging::Level>& loggerLevels = accessor.getCurrentLoggerLevels();
   std::map<std::string, Logging::Level>::const_iterator iter = loggerLevels.begin();
   while (iter != loggerLevels.end())
   {
      accessor.setLoggingLevel(iter->first, level);
      iter++;
   }
}


/**
 * Determine how to deal with the ASF logger levels based on the TTFis trace level.
 * Some TTFis trace levels enforce a certain level on all ASF loggers, others retain
 * the configuration of the log config file.
 */
static void setupAsfLogLevels(unsigned short traceClass, TR_tenTraceLevel ttfisLevel, LoggerAccessorIF& accessor)
{
   switch (ttfisLevel)
   {
      case TR_LEVEL_FATAL:
         traceTtfis(traceClass, TR_LEVEL_FATAL, "TTFis trace level 'TR_LEVEL_FATAL' enforces ASF log level 'Error' for all loggers on startup.");
         enforceAsfLogLevel(Logging::Level__Error, accessor);
         break;

      case TR_LEVEL_USER_2:
         traceTtfis(traceClass, TR_LEVEL_FATAL, "TTFis trace level 'TR_LEVEL_USER_2' enforces ASF log level 'Info' for all loggers on startup.");
         enforceAsfLogLevel(Logging::Level__Info, accessor);
         break;

      case TR_LEVEL_USER_3:
         traceTtfis(traceClass, TR_LEVEL_FATAL, "TTFis trace level 'TR_LEVEL_USER_3' enforces ASF log level 'Debug' for all loggers on startup.");
         enforceAsfLogLevel(Logging::Level__Debug, accessor);
         break;

      default:
         traceTtfis(traceClass, TR_LEVEL_FATAL, "TTFis trace level " + toString(ttfisLevel) + " retains ASF log levels as defined by log config.");
         break;
   }
}


/**
 * Wrap the creation of the initialization message into a separate function.
 */
static std::string formatInitLog(const struct LoggingBackendIF::InitParameters& parameters, unsigned short traceClass)
{
   char tc[20];
   snprintf(tc, sizeof tc, "0x%04x", traceClass);
   std::string text = "ASF logs of '" + parameters.appPackageName + "." + parameters.appName + "' are forwarded to TTFis trace class " + tc;
   return text;
}


bool TtfisLoggingBackend::init(struct InitParameters& parameters)
{
   _detailedHeader = getBoolValue(parameters.backendConfiguration, "detailedHeader");
   _traceClass = (unsigned short) getInteger(parameters.backendConfiguration, "traceClass", DEFAULT_TRACE_CLASS);
   /**
    * NOTE: The first call to traceTtfis() / TR_core_uwTraceOut() with trace
    *       level FATAL initializes the trace library as a side-effect!
    *       Any library call prior to this initialization will *NOT* succeed!
    */
   traceTtfis(_traceClass, TR_LEVEL_FATAL, formatInitLog(parameters, _traceClass));
   TR_tenTraceLevel ttfisLevel = getConfiguredTtfisLevel(_traceClass);
   setupAsfLogLevels(_traceClass, ttfisLevel, parameters.accessor);
   return true;
}


std::string TtfisLoggingBackend::getBackendDescription()
{
   return "The TtfisLoggingBackend forwards ASF logs to TTFis.";
}


std::string TtfisLoggingBackend::getBackendName()
{
   return "TtfisLoggingBackend";
}


/**
 * Return the minimum required TTFis level in order to print a log
 * with the given 'asfLevel'.
 */
static TR_tenTraceLevel minimumRequiredTtfisLevel(Logging::Level asfLevel)
{
   switch (asfLevel)
   {
      case Logging::Level__Fatal:
      case Logging::Level__Error:   // ASF errors are usually critical, thus ensure the trace is printed
         return TR_LEVEL_FATAL;

      case Logging::Level__Warn:
         return TR_LEVEL_WARNING;

      case Logging::Level__Info:
         return TR_LEVEL_COMPONENT;

      case Logging::Level__Debug:
         return TR_LEVEL_USER_3;

      default:
         return TR_LEVEL_USER_4;
   }
}


/**
 * This function is called by the ASF loggers to format and print a log message.
 */
void TtfisLoggingBackend::logFormat(const struct LoggingHeader& header, const char* format, va_list argptr)
{
   CALL_PTHREAD(pthread_mutex_lock((pthread_mutex_t*)&_mutex));
   TR_tenTraceLevel ttfisLevel = minimumRequiredTtfisLevel(header.level);
   if (TR_core_bIsClassSelected(_traceClass, ttfisLevel))
   {
      std::string headerString;
      if (_detailedHeader)
      {
         headerString = formatDetailedHeader(header);
      }
      char log[2000];   // this array may have to hold a long multi-line string from ASF, e.g. a message payload
      int fullSize = vsnprintf(log, sizeof log, format, argptr);
      if (fullSize >= static_cast<int>(sizeof log))
      {
         traceTtfis(_traceClass, ttfisLevel, "ATTENTION: Below log is truncated! TTFis logging backend only supports strings of up to 2000 bytes");
      }
      traceMultiline(_traceClass, ttfisLevel, headerString + log);
   }
   CALL_PTHREAD(pthread_mutex_unlock((pthread_mutex_t*)&_mutex));
}


void TtfisLoggingBackend::logBuffer(const struct LoggingHeader& header, const uint8* buffer, size_t length)
{
   CALL_PTHREAD(pthread_mutex_lock((pthread_mutex_t*)&_mutex));
   TR_tenTraceLevel ttfisLevel = minimumRequiredTtfisLevel(header.level);
   if (TR_core_bIsClassSelected(_traceClass, ttfisLevel))
   {
      std::string headerString;
      if (_detailedHeader)
      {
         headerString = formatDetailedHeader(header);
      }
      char log[200];
      snprintf(log, sizeof log, "buffer: %p, length=%zd (memory dump not supported)", buffer, length);
      traceTtfis(_traceClass, ttfisLevel, headerString + log);
   }
   CALL_PTHREAD(pthread_mutex_unlock((pthread_mutex_t*)&_mutex));
}


std::string TtfisLoggingBackend::formatDetailedHeader(const struct LoggingHeader& header) const
{
   char buffer[200];
   const char* loggerName = header.logger.c_str();

   if (header.logger.size() > 30)
   {
      loggerName += header.logger.size() - 30;
   }
   snprintf(
         buffer,
         sizeof buffer,
         "[%-5s|%-16s|%08lx|%6zd|%-16s|%6" PRIu64 "|%-30.30s|%p] ",
         getLevelName(header.level),
         header.threadName.c_str(),
         header.threadHandle,
         header.activityIndex,
         header.componentName.c_str(),
         header.messageId,
         loggerName,
         header._this);
   return buffer;
}


bool TtfisLoggingBackend::getBoolValue(
      const std::map<std::string, std::string>& backendConfiguration,
      const std::string& key,
      bool defaultValue) const
{
   std::map<std::string, std::string>::const_iterator it = backendConfiguration.find(key);
   if (it != backendConfiguration.end())
   {
      std::string value = it->second.c_str();
      std::transform(value.begin(), value.end(), value.begin(), ::toupper);
      return (value.compare("TRUE") == 0);
   }
   return defaultValue;
}


long int TtfisLoggingBackend::getInteger(
      const std::map<std::string, std::string>& backendConfiguration,
      const std::string& key,
      long int defaultValue) const
{
   std::map<std::string, std::string>::const_iterator it = backendConfiguration.find(key);
   if (it != backendConfiguration.end())
   {
      return strtol(it->second.c_str(), NULL, 0);
   }
   return defaultValue;
}


const char* TtfisLoggingBackend::getLevelName(Logging::Level value) const
{
    static const char* levelName[] =
    {
        "Debug",
        "Info",
        "Warn",
        "Error",
        "Fatal",
        "Off",
        "Unknown"
    };

    if (value < Logging::Level__Off)
    {
        return levelName[value];
    }
    else
    {
        return levelName[6];
    }
}
