/*******************************************************************************
*Copyright (C) 2017 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
*******************************************************************************/
/*******************************************************************************
* FILE         : tddrd_main.c
*             
* DESCRIPTION  :This is the main impementation of the tddrd daemon. 
* 				This file contains the implementation of the main interface 
*				that gets connected to the tddr frontend,backend 
*
*                            
* AUTHOR(s)    :  Ravindra Prabha (Ravindra.Prabha@in.bosch.com)
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*02.May.2017| Initial version 1.0|Ravindra Prabha(Ravindra.Prabha@in.bosch.com)
* -----------------------------------------------------------------------
* -----------------------------------------------------------------------
*******************************************************************************/

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
#include <sys/un.h>
#include <sys/epoll.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <syslog.h>
#include <unistd.h>
#include <dirent.h>
#include <tddrd.h>

#define TDDR_DEV_NAME "/dev/tddr"

/*  IMPORTANT NOTE :To store complete tddr data from frontend sub device,
 *  approximately~ 2.5MB space required.
 *	This number may vary for future driver enhancements.THis is initial value
 */

static int32_t tddrd_do_read_store();
static int tddrd_establish_infrastructre();

const char* version = "0.01 (Add tddr / Requires Driver Version >= 0.01)";

tddrd_t tddrd ;
char errlog[256];
//char *source_files[32] = {"tddrd_read_store.c","tddrd_main.c",NULL};

int is_tddrd_running()
{
	int ret = 0;
	
	if(tddrd.tddrd_running)
	{
		ret = 1;
	}
	return ret;
}

/******************************************************************************
* FUNCTION       : tddrd_log_error
* PARAMETER      : void                                       
* RETURNVALUE    : -
* DESCRIPTION    : This function captures logs 
*
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
******************************************************************************/
void tddrd_log_error( unsigned int line_no,short int fileid,
							const char *text)
{
error_packet_t packet;
int fd;
struct stat filestats;
memset(&filestats,0,sizeof(struct stat));

	packet.file_id = fileid;
	packet.err = (short int)errno;
	packet.line_no = line_no;

	snprintf(packet.text,sizeof(packet.text),text);
	snprintf(errlog, 256,"%stddr.log" ,tddrd.tddr_config.backend_storage);

	/* create error log file if it doesn't exist.Create the file with
	 Read/Write privileges */
	fd = open(errlog, O_WRONLY | O_APPEND | O_CREAT,
				S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH) ;
	if( -1 != fd )
	{
		memset(&filestats,0,sizeof(struct stat));
		
		if(stat(errlog, &filestats) == 0)
		{
			/* log file should not exceed 128k.If the log file reaches 
				128k then it overwrites the first entry onwards*/
			if(filestats.st_size > (128 * 1024))
			{
				if(-1 != lseek(fd,0,SEEK_SET))
				{
					write(fd,&packet,sizeof(packet));
				}
			}
			else
			{
				/* log file size is less than 128 k.so entry can be written */
				write(fd,&packet,sizeof(packet));
			}
		}
		close(fd);
	}
}

 /******************************************************************************
 * FUNCTION 	  : tddrd_get_devinfo
 * PARAMETER	  : void									   
 * RETURNVALUE	  : returns the device ifo structure
 *				   -1 : FAILURE
 * DESCRIPTION	  : This function returns the address of frontend object that is 
 *					 matched with the requested devid
 *
 *------------------------------------------------------------------------
 * Date 	 |		 Version	   | Author & comments
 *-----------|---------------------|--------------------------------------
 *12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
 * -----------------------------------------------------------------------
 ******************************************************************************/
tddr_dev_info_t* tddrd_get_devinfo()
{
	return  &tddrd.tddr_dev_info;
}
 
/******************************************************************************
* FUNCTION       : tddrd_get_frontend_node
* PARAMETER      : void                                       
* RETURNVALUE    : Address of frontend object : SUCCESS
*		  		  -1 : FAILURE
* DESCRIPTION    : This function returns the address of frontend object that is 
* 					matched with the requested devid
*
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
******************************************************************************/
tddrd_frontend_t* tddrd_get_frontend_node(int devid)
{
	tddrd_frontend_t *fe = tddrd.tddr_dev_info.frontend;

		while((fe) )	
		{
			if(devid == fe->devid) 
			{
				return fe;
			}
			
			fe = fe->next;
		}	
	
	return (tddrd_frontend_t*)NULL;
}
 
/******************************************************************************
* FUNCTION       : tddrd_delete_frontend
* PARAMETER      : void                                       
* RETURNVALUE    : 0 : SUCCESS
* DESCRIPTION    : This function deletes the frontend object that is 
* 					matched with the requested devid
*
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
******************************************************************************/
int tddrd_delete_frontend(int devid)
{
	tddrd_frontend_t *fe = tddrd.tddr_dev_info.frontend;
	tddrd_frontend_t *temp;

		while((fe) )	
		{
			if(devid == fe->devid) 
			{
				temp = fe;
				fe=fe->next;
				free(temp);
			}
			else
			{
				fe = fe->next;
			}
		}	
		tddrd.tddr_dev_info.frontend = NULL;

		return 0;
}

/******************************************************************************
* FUNCTION       : tddrd_add_frontend
* PARAMETER      : void                                       
* RETURNVALUE    : 0 : SUCCESS
*				  -1 : FAILURE
* DESCRIPTION    :This function adds the frontend object with the requested node
*
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
******************************************************************************/
static int tddrd_add_frontend(tddrd_frontend_t *node)
{
	tddrd_frontend_t *fe = tddrd.tddr_dev_info.frontend;
	int ret = -1;

	if(!node)
	{
		tddrd_log_error(__LINE__,1,"invalid arg to tddrd_add_frontend ");	
		return ret;
	}
	if(!fe) 
	{
		fe = (tddrd_frontend_t *) malloc(sizeof(tddrd_frontend_t));
		if(fe)
		{
			memcpy(fe,node,sizeof(tddrd_frontend_t));
			fe->next = NULL;
			tddrd.tddr_dev_info.frontend = fe;
			
			/* node has been added successfully */
			ret = 0;
		}
	}
	else
	{
		while(fe->next) 
		{
			fe = fe->next;
		}
		fe->next  = (tddrd_frontend_t *) malloc(sizeof(tddrd_frontend_t));
		if(fe->next) 
		{
			memcpy(fe->next,node,sizeof(tddrd_frontend_t));
			fe = fe->next;
			fe->next = NULL;
			
			/* node has been added successfully */
			ret = 0;
		}
	}
	return ret;	
}

/******************************************************************************
* FUNCTION       : tddrd_do_read_store
* PARAMETER      : void                                       
* RETURNVALUE    : 0 : SUCCESS
*		 		  -1 : FAILURE
* DESCRIPTION    : This function 
*		 	1). Detect number of frontend sub devices
*		 	2). Add each frontend subdevice to the frontend list
*		 	3). Allocate backend unit for each frontend sub device
*		 	4). Create worker thread to read all the data from froneend sub
* 				device and stores into the configured backend
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
******************************************************************************/
static int32_t tddrd_do_read_store()
{
int ret = -1;
unsigned int ii = 0;
tddrd_backend_t *pbackend;
char errbuf[64];
int *thread_args;

	/* Create the worker thread per tddr sub device .The worker thread reads 
	   the recorded data from tddr correpsonding sub device (frontend) and 
	   stores into the configured persistent file locations(backend) */
			
	pbackend = tddrd.tddr_dev_info.backend;
	if(pbackend)
	{
			thread_args = (int *) malloc(tddrd.num_tddr_devices * sizeof(int));
			if(thread_args)
			{
				for (ii=0; ii < tddrd.num_tddr_devices; ii++) 
				{
					/* Initialize worker thread attributes */
					ret = pthread_attr_init(&pbackend[ii].worker_thread_attr);
					if (0 != ret) 
					{
						/* file id=1 for this source file*/
						snprintf(errbuf,
								sizeof(errbuf),"pthread_attr_init %d fails",ii);
						tddrd_log_error(__LINE__,1,errbuf);						
					}
					else 
					{
						ret = pthread_attr_setdetachstate
									(&pbackend[ii].worker_thread_attr,
									PTHREAD_CREATE_JOINABLE);  
						if (0 != ret) 
						{
							snprintf(errbuf,sizeof(errbuf),
								"pthread_attr_setdetachstate %d fails",ii);
							tddrd_log_error(__LINE__,1,errbuf);		
						}
						else 
						{	
							thread_args[ii] = ii;
							/* create worker thread . The argument to the 
							thread is value not the pointer !!! */
							ret = pthread_create(&pbackend[ii].worker_thread_id,
										&pbackend[ii].worker_thread_attr,
								tddrd_worker_thread,(void *)&thread_args[ii]);
							if (0 != ret)
							{ 						
								snprintf(errbuf,sizeof(errbuf),
										"pthread_create %d fails",ii);
								tddrd_log_error(__LINE__,1,errbuf);				
							}						
						}
					}
				}
				/* ensure that all the workers threads finished their job*/
				for (ii=0; ii < tddrd.num_tddr_devices; ii++) 
				{
					ret = pthread_join(pbackend[ii].worker_thread_id, NULL);
					pbackend[ii].worker_thread_id = -1; 
					if(0 != ret)
					{
						snprintf(errbuf,sizeof(errbuf),
								"pthread_join %d fails",ii);
						tddrd_log_error(__LINE__,1,errbuf);
					}
				}
				
				free(thread_args);
			}
			else
			{
				tddrd_log_error(__LINE__,1,"malloc error for thread_args");
			
}

	}
	else
	{
		tddrd_log_error(__LINE__,1,"backend is not setup");
	
}
	return ret;
}


/*******************************************************************************
* FUNCTION       : tddrd_establish_infrastructre
* PARAMETER      :                                                    
* RETURNVALUE    : 
*                   
* DESCRIPTION    : This function establishes the infrastructure 
* 					It creates allocates and backend lists
*------------------------------------------------------------------------
* Date      |       Version       | Author
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
*******************************************************************************/
static int tddrd_establish_infrastructre()
{
unsigned int ii = 0;
int ret = -1;
tddrd_frontend_t frontend;
tddrd_backend_t *pbackend;

	/* Set Daemon running state to TRUE */
	tddrd.tddrd_running = 1;
	tddrd.tddr_dev_info.max_file_length = tddrd.tddr_config.size_max;

	while(1)
	{
		/* Add the device information like file descriptor,device name
			into a front-end list*/
		snprintf(frontend.dev, TDDRD_MAX_NAME_LEN,"%s%d",TDDR_DEV_NAME,ii);
		frontend.fd = open(frontend.dev,O_RDONLY);
		if(frontend.fd < 0) {

		/* The device does not present then break the loop */
			break;
		}
		frontend.devid = ii;
		tddrd_add_frontend(&frontend);
		ii++;		
	}
	
	/* There is not even single frontend sub device existing. */
	if(0 == ii) {
		tddrd_log_error(__LINE__,1,"frontend not found");
	}
	else 
	{
		tddrd.num_tddr_devices = ii;

		/* Add each frontend subdevice to the frontend list */
		pbackend = (tddrd_backend_t *) 
					malloc(sizeof(tddrd_backend_t) * tddrd.num_tddr_devices);
		if(pbackend)
		{			
			for (ii=0; ii < tddrd.num_tddr_devices; ii++)
			{
				/*Store the backend device information into the corresponding
					device info entry*/
				snprintf(pbackend[ii].dev, TDDRD_MAX_NAME_LEN,"%stddr%d" ,
										tddrd.tddr_config.backend_storage,ii);
			}			
			tddrd.tddr_dev_info.backend = pbackend;

			ret = 0;	
		}
		else
		{		
			tddrd_log_error(__LINE__,1,"malloc failed");		
		}
	}
	return ret;
}

/******************************************************************************
* FUNCTION       : tddrd_destroy_infrastructure
* PARAMETER      :                                                    
* RETURNVALUE    : 
*                   
* DESCRIPTION    : This function destroys the frontend and backend objects
*------------------------------------------------------------------------
* Date      |       Version       | Author
*-----------|---------------------|--------------------------------------
*12.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
******************************************************************************/
static void tddrd_destroy_infrastructure()
{
	unsigned int ii;

	/* Set Daemon running state to FALSE */
	tddrd.tddrd_running = 0;

	for(ii=0;ii < tddrd.num_tddr_devices;ii++)
	{
		tddrd_delete_frontend(ii);
	}
	if(tddrd.tddr_dev_info.backend)
	{
		free(tddrd.tddr_dev_info.backend);
		tddrd.tddr_dev_info.backend = NULL;
	}
}

/* daemon binary   daemon option   output file path    max size for output file
 /bin/tddrd     -d              -o /var/opt/bosch/dynamic/tddr -s 5242880 */
int main (int argc, char  *const argv[])
{
int32_t ret ;
int32_t option_index = 0;
int32_t c;
const char *sh_opt  = "hido:s:e:ow";


/* h:help i:info o:output s:maxsize e:extended output ow:overwrite the output
 * If  a character is followed by a colon, the option requires an argument, 
 * Two colons mean an option takes an optional arg */
static const struct option long_options[] =	{
	{"help",no_argument, 0,  0 }, /*h*/
	{"info",no_argument, 0,  1 }, /*i*/
	{"debuginfo",no_argument, 0,  2 }, /*d*/
	{"outputfile",  required_argument, 0,  3 },/*o*/
	{"overwrite", optional_argument, 0,  4 },/*s*/
	{0,	       0,		  0,  0 }
   };

	memset(&tddrd,0,sizeof(tddrd));
	
	/* 1. Parsing the arguments so that frontend,backend,daemon attributes 
	* can be configured with the provided information
	*/
	while (1) 
	{	
	      c = getopt_long(argc, argv, sh_opt,long_options, &option_index);
	       if (c == -1) 
		   {
				break;
	       }

		/* help,info,debug options are not supported in the initial version */
		switch (c) 
		{
			case 'o':
				/* configure the output file i.e.backend location based on the 
				 * provided information.
				 * Currently it is "/var/opt/bosch/dynamic/tddr"
				 */
				 
		   		if (optarg) 
				{
					DIR *dir;
					strncpy(tddrd.tddr_config.backend_storage,optarg,TDDRD_MAX_NAME_LEN);
					/* NULL terminate the string */
					tddrd.tddr_config.backend_storage[TDDRD_MAX_NAME_LEN-1]= 0 ;					

					/* Check if backend storage folder is existing,
					 * if not create the folder
					 */
					dir = opendir(tddrd.tddr_config.backend_storage);
					if(!dir)
					{
						if(0 == mkdir(tddrd.tddr_config.backend_storage,0x777))
						{					
							dir = opendir(tddrd.tddr_config.backend_storage);
							if(!dir)
							{
								/* Without backend storage directory available,
								 * nothing can be done
								 */
								return -1;
							}
							closedir(dir);
						}
					}
					else
					{
						closedir(dir);
					}
				}
		   		break;
	
			case 's':
				/* output file (i.e. backend) shall be overwritten with new
				 * data upon reaching the max size while storing the data*/
				if(optarg) 
				{
					tddrd.tddr_config.size_max = strtol(optarg, NULL, 0);
				}
				break;

			/* fall through is intentional.This shall be considered in the 
				future enhancements */
			case 'h':
			case 'i':
			case 'd':
			default:
				break;
	       }
	   }

	/*Parsing the arguments END*/

	/* Initialize the daemon structure which consists the complete information 
	 * of daemon
	 */
	ret = tddrd_establish_infrastructre();

	if(0 == ret)	
	{
		/* can not be done much handling when tddrd_do_read_store() 
		 * returns error */
		ret = tddrd_do_read_store();
		if(ret < 0)
		{
			tddrd_log_error(__LINE__,1,"tddrd_do_read_store failed");
		}
		
		/* 11.Finally destroy the daemon frontend and backend objects*/	
		tddrd_destroy_infrastructure();
	}
	else
	{	
		tddrd_log_error(__LINE__,1,"tddrd_establish_infrastructre failed");
	}
	return ret;
}

