/************************************************************************
| FILE:         dispatcher_table.c
| PROJECT:      
| SW-COMPONENT: OSAL I/O
|------------------------------------------------------------------------
| DESCRIPTION:  This is the source file for OSAL I/O dispatcher
|               implementation
|
|------------------------------------------------------------------------
| COPYRIGHT:    (c) 2005 Bosch GmbH
| HISTORY:      
| Date      | Modification                    | Author
| 18.01.06  | Initial revision                | MRK2HI
| 03.04.13  | To provide fix to NIKAI_4347    | 
|           | Fix Path Name's Max lenght      | SWM2KOR
| --.--.--  | ----------------                | -------, -----
|
|************************************************************************/


/************************************************************************
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Linux_osal.h"

#include "osfile.h"



void  vCloseTripFile(void);
tBool bInitTripFileRecord( tCString strFileName);
tBool bInitTripFileReplay( tString strFileName, tU8 u8Mode, tU32 u32Interval);
tBool bSendTripMessages( tU32 u32Count );


extern u32PrmFunc           pPrmFunc;
extern pFuncTraceBinOutput         TraceBinOutput;
extern pFunc_chan_acess_bRegChan   chan_acess_bRegChan;
extern pFunc_chan_acess_bUnRegChan chan_acess_bUnRegChan;

#define BACKSLASH '\\'


/************************************************************************
*                                                                       *
*                         OsalDeviceDescriptor                          *
*                                                                       *
*                       File descriptor definition                      *
*                                                                       *
************************************************************************/
/************************************************************************
*                                                                       *
*                   static function prototyps                           *
*                                                                       *
************************************************************************/

/************************************************************************
*                                                                       *
*                   static definitons exported to headers               *
*                                                                       *
************************************************************************/


/* include wrappers  that have not the conforming calling convention */
#include "dispatcher_wrappers.h"

/* include generated dispatcher code */
#include "dispatcher_table.h"

/* this function will be generated by dispather_table.h" */
/* extern struct drivers *dispatcher_table_lookup (register const char *str, register unsigned int len); */

/* this type will be produced for driver */
/* struct drivers {const char*   name;
                tOpenPtr      ptr_s32IOOpen;
                tClosePtr     ptr_s32IOClose;
                tCreatePtr    ptr_s32IOCreate;
                tRemovePtr    ptr_s32IORemove;
                tIOControlPtr ptr_s32IOControl;
                tReadPtr      ptr_s32IORead;
                tWritePtr     ptr_s32IOWrite;
                tBool         bFileSystem;
                OSAL_tenDevAccess *bExclusiveAcces;};
*/

/******************************************************************************
* FUNCTION:     u32CheckMagic
*
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  
*
* HISTORY:
* Date      |   Modification                       | Authors
*****************************************************************************/

static tU32 u32CheckMagic( OsalDeviceDescriptor * fd )
{

 /*  if( ( fd == NULL ) || ( (tU32)fd == OSAL_C_INVALID_HANDLE ) )
   {                                 * fd contains above defines, not good code, but no assert *
      return OSAL_E_INVALIDVALUE;
   }*/

   /* check for mempool area used for device descriptor pool ,more robust against overwritten handles */
   if(((uintptr_t)fd <= DescMemPoolHandle.u32memstart )||((uintptr_t)fd >= DescMemPoolHandle.u32memend))
   {
      trOSAL_MPF* pPtr = (trOSAL_MPF*)DescMemPoolHandle.pMem;
      if(pPtr->u32ErrCnt == 0)
      {
         TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Handle Check OSAL FD:%p failed due memory range %p - %p ",
                               OSAL_ProcessWhoAmI(),fd,DescMemPoolHandle.u32memstart,DescMemPoolHandle.u32memend);
         return OSAL_E_INVALIDVALUE;
      }
      else
      {
          /* possible that Handle Memory was allocated via malloc */
          if(( fd == NULL)||((uintptr_t)fd == OSAL_C_INVALID_HANDLE))
          {
              TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Handle Check OSAL FD:%p failed due file descriptor value ",
                                    OSAL_ProcessWhoAmI(),fd);
              return OSAL_E_INVALIDVALUE;
          }
      }
   }

   if( OSAL_DESCRIPTOR_MAGIC != fd->magic )
   {      
      struct OsalDevices* pdevice =  (struct OsalDevices*)fd->device_ptr;
      if(pdevice)
      {
         TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Handle Check OSAL FD:%p failed due magic DevId:%d Device:%s",
                               OSAL_ProcessWhoAmI(),fd,pdevice->device_local_id,pdevice->name );
      }
      else
      {
         TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Handle Check OSAL FD:%p failed due magic DevId:? Device:?",
                               OSAL_ProcessWhoAmI(),fd);
      }
      return OSAL_E_BADFILEDESCRIPTOR;
   }

   if(fd->s32Pid != OSAL_ProcessWhoAmI())
   {                                         /* fd hande is used from wrong process */
       TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Handle Check OSAL FD:%p failed due wrong PID:%d",
                             OSAL_ProcessWhoAmI(),fd,fd->s32Pid);
       return OSAL_E_BADFILEDESCRIPTOR;
   }

   return OSAL_E_NOERROR;
}


/******************************************************************************
* FUNCTION:     GetOsalDeviceName
*
* DESCRIPTION:  returns Osal Device Name from handle
*
* PARAMETER:    
*
* RETURNVALUE:  
*
* HISTORY:
* Date      |   Modification                       | Authors
* 24.01.07  |   Initial revision                   | CM-DI/PJ-CF14 Storre
*****************************************************************************/
tCString GetOsalDeviceName( OSAL_tIODescriptor prIoH )
{
   OsalDeviceDescriptor * prFH = (OsalDeviceDescriptor *)prIoH;

   if( prFH != NULL )
   {
      return( prFH->device_ptr->name );
   }
   else
   {
      return( NULL );
   }
}



/******************************************************************************
* FUNCTION:     vSetTraceForAllDeviceIo
*
* DESCRIPTION:  de-/activate traces for all OSAL device access
*
* PARAMETER:    char on/off value
*               tS32 process ID
*
* RETURNVALUE:  none
*
* HISTORY:
* Date      |   Modification                       | Authors
* 24.06.15  |   Initial revision                   | mrk2hi
*****************************************************************************/
void vSetTraceForAllDeviceIo(char cVal, OSAL_tProcessID Pid)
{
   int i;
   for(i = 0;i< (int)OSAL_EN_DEVID_LAST;i++)
   {
      pOsalData->rDevTrace[i].cActive = cVal;
      pOsalData->rDevTrace[i].Pid  = Pid;
   }
}

/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_InitTable                                                   *
*                                                                       *
* DESCRIPTION                                                           *
*      inits the full system table                                      *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      -                                                                *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      -                                                                *
*                                                                       *
***********************************************************************
void OSAL_InitTable(void)
{
  int i;
 
  for( i = 0; i < (int)OSAL_EN_DEVID_FSLAST; ++i)
  {
    pOsalData->exclusive_drive_handling[i].u16AppID   = OSAL_C_U16_INVALID_APPID;
    pOsalData->exclusive_drive_handling[i].u16ExAppID = OSAL_C_U16_INVALID_APPID;
  }
}*/




/************************************************************************
*                                                                       *
*                                 SEARCH_SLASH                          *
*         MACRO to find the next slash or the end of the string         *
*                                                                       *
************************************************************************/
#define SEARCH_SLASH(ptr) \
  if (!(*ptr)) \
  { \
    ptr = NULL; \
  } \
  else \
  { \
    do \
    {\
      ++ptr; \
    } while ((*ptr) && (*ptr != '/')); \
  }

void vCheckLoadedPrm(void)
{
   if(pPrmFunc == NULL)
   {
      if(u32LoadSharedObject(pPrmFunc,pOsalData->rLibrary[EN_SHARED_OSAL2].cLibraryNames) == OSAL_E_NOTSUPPORTED)
      {
         TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSAL PRM cannot be loaded");
      }
   }
}

/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_Get_Driver_by_name                                          *
*                                                                       *
* DESCRIPTION                                                           *
*      checks if the no-fs-file-system device access variable is used   *
*      set the no fs file system device access variable to used         *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      tCString name : full name of access                              *
*      pointer to rest of the name for the result                       *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      struct OsalDevices : the device or NULL                          *
*      *filename : the filesystem part of the device                    *
*                                                                       *
************************************************************************/
static struct OsalDevices *OSAL_Get_Driver_by_name( tCString name, tCString *filename )
{
  struct OsalDevices *device;
  const char *ptr;

  device = NULL;
  /* no empty names */
  if (!name || !(*name))
  {
    return device;
  }
  ptr = name + 1;
  SEARCH_SLASH(ptr);
  if (!ptr)
  {
    return device;
  }
  SEARCH_SLASH(ptr);
  while (ptr)
  {
    device = dispatcher_table_lookup( name, (tU32)((uintptr_t)ptr - (uintptr_t)name));
    if (device && filename)
    {
      if (*ptr)
      {
        *filename = ptr+1;
      }
      else
      {
        *filename = NULL;
      }
      return device;
    }
    else
    {
      SEARCH_SLASH(ptr);
    }
  }
  return device;
}




/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_get_device_name                                             *
*                                                                       *
* DESCRIPTION                                                           *
*      returns the drive name                                           *
*      or NULL                                                          *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*      OSAL_tIODescriptor dd : descriptor                               *
*                                                                       *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*       char* : the device name                                         *
*                                                                       *
************************************************************************/
char* OSAL_get_device_name( OSAL_tIODescriptor descriptor )
{
   OsalDeviceDescriptor *dd_n;

   dd_n = (OsalDeviceDescriptor *)descriptor;
 
   return (char*)dd_n->device_ptr->name;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_get_drive_id                                                *
*                                                                       *
* DESCRIPTION                                                           *
*      returns the drive id                                             *
*      or NULL                                                          *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*      OSAL_tIODescriptor dd : descriptor                               *
*      tCString name : full name of access                              *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*       OSAL_tenFSDevID : the device id                            *
*                                                                       *
************************************************************************/
OSAL_tenFSDevID OSAL_get_drive_id( OSAL_tIODescriptor descriptor )
{
   OsalDeviceDescriptor *dd_n = (OsalDeviceDescriptor *)descriptor;
   
   return (OSAL_tenFSDevID)dd_n->device_ptr->prm_index;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      s32OSAL_get_device_id_and_filename_by_Path                       *
*                                                                       *
* DESCRIPTION                                                           *
*      returns the drive id                                             *
*      or OSAL_ERROR (-1)                                               *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*      adress of the pointer to the osal path                           *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*       OSAL_tenFSDevID : the device id                                 *
*       name adress is set to the string after the device name          *
*                                                                       *
************************************************************************/
tS32 s32OSAL_get_device_id_and_filename_by_Path( tString *name )
{
   struct OsalDevices *device;
   tString filename;
   
   device = OSAL_Get_Driver_by_name( *name, (tCString*)(&filename) );
   if( device == NULL )
   {
      return OSAL_ERROR;
   }
   else
   {
      *name = filename;
      return (tS32)device->device_local_id;
   }
}


tBool bCheckRestoreFd(OSAL_tIODescriptor fd)
{
   if((pOsalData->u32RemountTime)// was remount happen ?
	&&(!strncmp(((OSAL_tfdIO*)((OsalDeviceDescriptor*)fd)->local_ptr)->szFileName,"/media",strlen("/media")))) // file from USB stick ?
   {
      close((int)((OSAL_tfdIO*)((OsalDeviceDescriptor*)fd)->local_ptr)->fd);
      ((OSAL_tfdIO*)((OsalDeviceDescriptor*)fd)->local_ptr)->fd = open(((OSAL_tfdIO*)((OsalDeviceDescriptor*)fd)->local_ptr)->szFileName,
                                                                       O_RDONLY|O_CLOEXEC,OSAL_FILE_ACCESS_RIGTHS);
      TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "Restoring File Handle for ",((OSAL_tfdIO*)((OsalDeviceDescriptor*)fd)->local_ptr)->szFileName);
      return TRUE;
   }
   return FALSE;
}

/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_Create                                                      *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io create function, create request file (device)            *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      tCString szName file name                                        *
*      OSAL_tenAccess enAccess access type                              *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      OSAL_tIODescriptor file descriptor                               *
*                                                                       *
* HISTORY:
* Date      |   Modification                       | Authors
* 03.04.13  | To provide fix to NIKAI-4347         |
*           | To provide name lenght Check         | SWM2KOR
************************************************************************/
OSAL_tIODescriptor OSAL_IOCreate(tCString coszName, OSAL_tenAccess enAccess)
{
   struct OsalDevices *device;
   tCString filename=NULL;
   uintptr_t return_value = 0;
   tS32 error_code = OSAL_E_NOERROR;
   OSAL_tIODescriptor result = OSAL_E_UNKNOWN;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   device = OSAL_Get_Driver_by_name( coszName, &filename );
   if((filename!=NULL) && (OSAL_u32StringLength(filename) > (tU32)OSAL_C_U32_MAX_PATHLENGTH))
   {
      error_code=OSAL_E_NAMETOOLONG;
   }

   if (device && device->ptr_s32IOCreate && (error_code == OSAL_E_NOERROR) )
   {
      {
         if(device->bFileSystem == TRUE)
         {
            error_code = device->ptr_s32IOCreate((tS32)device->device_local_id,
                                                  filename,
                                                  enAccess,
                                                  &return_value, 
                                                  OSAL_C_U16_INVALID_APPID );
         }
         else
         {
             error_code = OSAL_E_NOTSUPPORTED;
         }
      }
      if( (tU32)error_code != OSAL_E_NOERROR )
      {
         OSAL_vSetErrorCode( (tU32)error_code );      /* Set error code */
         return OSAL_ERROR;
      }
      else
      {
         OsalDeviceDescriptor *ret = (OsalDeviceDescriptor *)OSAL_pvMemPoolFixSizeGetBlockOfPool(&DescMemPoolHandle);
         if(ret)
         {
            ret->magic       = OSAL_DESCRIPTOR_MAGIC;
            ret->device_ptr  = device;
            ret->local_ptr   = return_value;
            ret->nJobQueue   = C_INVALID_JOB;
            ret->u16AppID    = OSAL_C_U16_INVALID_APPID;
            ret->u16FileSystem = (tU16)((struct OsalDevices*)device)->bFileSystem;
            ret->u16PrmIndex = (tU16)((struct OsalDevices*)device)->ExclusiveAccess;
            ret->pAIO        = NULL;
            ret->u32ID       = device->device_local_id;
            ret->s32Pid      = (tS32)OSAL_ProcessWhoAmI();
            /* make type  opaque */
            result = (OSAL_tIODescriptor)ret;
         }
         else
         {
            FATAL_M_ASSERT_ALWAYS();
         }
      }
   }
   else
   {
      OSAL_vSetErrorCode((tU32)error_code);      /* Set error code */
      result = OSAL_ERROR;                          /* Return OSAL ERROR */
   }
   return result;
}

tU32 u32UpperCaseWrk(tS32 error_code,const struct OsalDevices *device,OSAL_tenAccess enAccess,uintptr_t *preturn_value,tCString filename)
{
    tU32 ercd = (tU32)error_code;
            // RAD3HI workaround for uppercase
            if(filename)
            {
               tU32 u32Index=0;
               char archFileName[OSAL_C_U32_MAX_PATHLENGTH] = {'\0'};
               strncpy(&archFileName[0], filename,OSAL_C_U32_MAX_PATHLENGTH);
               while( archFileName[u32Index] )
               {
                     archFileName[u32Index] = (char)toupper(archFileName[u32Index] );/*lint !e666 !e516 !e774*/
                     u32Index++;
               }
               // try to open again with uppercase filename
               ercd = (tU32)device->ptr_s32IOOpen( (tS32)device->device_local_id,
               archFileName,
               enAccess,
               preturn_value,
               OSAL_C_U16_INVALID_APPID );
               if( ercd == OSAL_E_NOERROR )  // workaround was successfull
               {
                  char au8Buf[239];
                  int len;

                  memset(au8Buf, 0, 239);
                  au8Buf[0] = (char)0xF1;
                  strcpy( &au8Buf[1], "Uppercase Workaround Task= " );
                  len = (int)strlen(au8Buf);
                  bGetThreadNameForTID( &au8Buf[len], 32, OSAL_ThreadWhoAmI() );
                  len = (int)strlen(au8Buf);
                  if (au8Buf[len-1] == '\n')
                  {
                     au8Buf[len-1] = 0;   // eleminate new line char
                     --len;
                  }
                  strncat( &au8Buf[len], " File= ",(size_t)(237 - len));
                  len = (int)strlen(au8Buf);
                  strncat( &au8Buf[len], archFileName, (size_t)(237 - len) );
                  len = (int)strlen(au8Buf);
                  au8Buf[len+1] = 0;
                  
                  LLD_vTrace(TR_COMP_OSALIO, TR_LEVEL_ERROR, au8Buf, (tU32)(len+1));
               }                                                               
            } // end workaround
            return ercd;
}
/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_Open                                                        *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io open function, open request file (device)                *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      tCString szName file name                                        *
*      OSAL_tenAccess enAccess access type                              *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      OSAL_tIODescriptor file descriptor                               *
*                                                                       *
* HISTORY:
* Date      |   Modification                               | Authors
* 03.04.13  | To provide fix to NIKAI-4347                 |
*           | To provide name lenght Check                 | SWM2KOR
************************************************************************/
OSAL_tIODescriptor OSAL_IOOpen( tCString coszName, OSAL_tenAccess enAccess )
{
   struct OsalDevices *device = NULL; 
   uintptr_t return_value = 0;
   tS32 error_code = OSAL_E_NOERROR;
   tU32 u32DevId = 0;
   tCString filename=NULL;
   OSAL_tIODescriptor result = OSAL_E_UNKNOWN;
   tBool bTrace = TRUE;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   if(enAccess > OSAL_EN_READWRITE)
   {
      if(enAccess == OSAL_EN_ACCESS_DIR) 
      {
         enAccess = OSAL_EN_READWRITE;
      }
      else
      {
         error_code = OSAL_E_INVALIDVALUE;
      }
   }
   if(error_code == OSAL_E_NOERROR) 
   {
      device = OSAL_Get_Driver_by_name( coszName, &filename );
      /* if no device found */
      if( device == NULL )
      {
         error_code = OSAL_E_DOESNOTEXIST;
      }
      else
      {
         if((filename!=NULL) && (OSAL_u32StringLength(filename) > (tU32)OSAL_C_U32_MAX_PATHLENGTH))
         {
               error_code=OSAL_E_NAMETOOLONG;
         }
      }
      if (device && (error_code == OSAL_E_NOERROR))
      {
       /*  if( (device->bFileSystem == FALSE)&& OSAL_test_and_set_inuse( device ))
         {
            error_code = OSAL_E_ALREADYOPENED;
         }
         else*/
          bTrace = FALSE;
          {
            u32DevId  = device->device_local_id;
            if(device->bFileSystem == TRUE)
            {
               if(device->ptr_s32IOOpen)
               {
                 if((device->device_local_id == OSAL_EN_DEVID_DEV_MEDIA)
                  ||(device->device_local_id == OSAL_EN_DEVID_CRYPT_CARD))
                 {
                    vCheckLoadedPrm();
                 }
                 error_code = device->ptr_s32IOOpen( (tS32)device->device_local_id,
                                                    filename,
                                                    enAccess,
                                                    &return_value,
                                                    OSAL_C_U16_INVALID_APPID );
                 if((device->bFileSystem)&&(device->device_local_id != OSAL_EN_DEVID_REGISTRY))
                 {
                    if(((tU32)error_code != OSAL_E_NOERROR)&&(pOsalData->u32UpperCase))
                    {
                        error_code = (tS32)u32UpperCaseWrk(error_code,device,enAccess,(uintptr_t*)&return_value,filename);
                    }
                 }
               }
               else
               {
                  error_code = OSAL_E_NOTSUPPORTED;
               }
            }
            else
            {
               if(rOsalDrvFuncTable[device->device_local_id].pFuncOpn)
               {
                  error_code =  rOsalDrvFuncTable[device->device_local_id].pFuncOpn((tS32)device->device_local_id,
                                                                                     filename,
                                                                                     enAccess,
                                                                                     &return_value,
                                                                                     OSAL_C_U16_INVALID_APPID);
               }
               else
               {
                  error_code = OSAL_E_NOTSUPPORTED;
               }
               if(pOsalData->rDevTrace[device->device_local_id].cActive == 1)
               {
                  bTrace = TRUE;
                  if(pOsalData->rDevTrace[device->device_local_id].Pid)
                  {
                     if(pOsalData->rDevTrace[device->device_local_id].Pid != OSAL_ProcessWhoAmI())
                     {
                         bTrace = FALSE;
                     }
                  }
               }
               if((tU32)error_code != OSAL_E_NOERROR )
               {
                  bTrace = TRUE;
               }
            }
            /*   if( ( (tU32)error_code != OSAL_E_NOERROR) && !(device->bFileSystem) )
               {
                  * device cannot be opened, give the exclusive access free *
                     OSAL_set_device_unused(device);
               }*/
         }
         if((tU32)error_code == OSAL_E_NOERROR )
         {
            OsalDeviceDescriptor *ret = (OsalDeviceDescriptor *)OSAL_pvMemPoolFixSizeGetBlockOfPool(&DescMemPoolHandle);
            if(ret)
            {
               ret->magic       = OSAL_DESCRIPTOR_MAGIC;
               ret->device_ptr  = device;
               ret->local_ptr   = return_value;
               ret->nJobQueue   = C_INVALID_JOB;
               ret->u16AppID    = OSAL_C_U16_INVALID_APPID;
               ret->u16FileSystem = (tU16)((struct OsalDevices*)device)->bFileSystem;
               ret->u16PrmIndex = (tU16)((struct OsalDevices*)device)->ExclusiveAccess;
               ret->pAIO        = NULL;
               ret->u32ID       = u32DevId;
               ret->s32Pid      = (tS32)OSAL_ProcessWhoAmI();
               /* make type  opaque */
               result = (OSAL_tIODescriptor)ret;
            }
            else
            {
               FATAL_M_ASSERT_ALWAYS();
            }
         }
      }
   }
   if(pOsalData->u32DisableAssertCS < 2)
   {
      if(bTrace)
      {
         if(device)
         {
             TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Open DevId:%d Device:%s ErrorCode:%d OSAL FD:%p",
                                   OSAL_ProcessWhoAmI(),device->device_local_id,coszName,error_code,result);
         }
         else
         {
             TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Open DevId:? Device:%s ErrorCode:%d OSAL FD:%p",
                                   OSAL_ProcessWhoAmI(),coszName,error_code,result);
         }
      }
   }
   if( (tU32)error_code != OSAL_E_NOERROR )
   {
       OSAL_vSetErrorCode((tU32)error_code);
       result = OSAL_ERROR;
   }
   return result;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_s32IOClose                                                  *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io close function, close request file (device)              *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      OSAL_tIODescriptor fd File descriptor                            *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      tS32 OSAL_NOERROR or error code                                  *
*                                                                       *
************************************************************************/
tS32 OSAL_s32IOClose( OSAL_tIODescriptor fd )
{
   struct OsalDevices *device = NULL;
   tS32 error_code;
   OsalDeviceDescriptor* fd_local;
   tBool bTrace = FALSE;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   fd_local = (OsalDeviceDescriptor *)fd;
   error_code = (tS32)u32CheckMagic( fd_local );
  
   if( (tU32)error_code == OSAL_E_NOERROR )
   {
      device = fd_local->device_ptr;
      if (device )
      { 
         if(device->bFileSystem == TRUE)
         {
             if(device->ptr_s32IOClose)
             {
                error_code = device->ptr_s32IOClose((tS32)device->device_local_id, 
                                                    fd_local->local_ptr );
             }
             else
             {
                error_code = OSAL_E_NOTSUPPORTED;
             }
         }
         else
         {
            if(rOsalDrvFuncTable[device->device_local_id].pFuncCls)
            {
               error_code = rOsalDrvFuncTable[device->device_local_id].pFuncCls((tS32)device->device_local_id, 
                                                                                fd_local->local_ptr );
            }
            else
            {
               error_code = OSAL_E_NOTSUPPORTED;
            }
            if(pOsalData->rDevTrace[device->device_local_id].cActive == 1)
            {
               bTrace = TRUE;
               if(pOsalData->rDevTrace[device->device_local_id].Pid)
               {
                  if(pOsalData->rDevTrace[device->device_local_id].Pid != OSAL_ProcessWhoAmI())
                  {
                     bTrace = FALSE;
                  }
               }
            }
         }
         if(bTrace)TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Close DevId:%d Device:%s ErrorCode:%d  OSAL FD:%p",
                                         OSAL_ProcessWhoAmI(),device->device_local_id,device->name,error_code,fd_local);
         if( (tU32)error_code == OSAL_E_NOERROR )
         { 
          /*  OSAL_set_device_unused(device);*/
            /* destroy magic explizit */
            fd_local->magic  = 0;
            fd_local->s32Pid = 0;
            fd_local->device_ptr = 0;
            fd_local->local_ptr = 0; /* set to zero to identify double close for registry key which uses malloced memory*/
            /* set File Descriptor for AIO Jobs as invalid */
            if(fd_local->pAIO)
            { 
               fd_local->pAIO->s32Status = OSAL_ERROR;
               TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "AIO Close:0x%08x | Time:%d",fd_local->pAIO,(int)OSAL_ClockGetElapsedTime());
            }
            if( OSAL_s32MemPoolFixSizeRelBlockOfPool(&DescMemPoolHandle, 
                                                     (void*)fd_local) == OSAL_ERROR )
            {
            }
            fd_local = NULL;
         } 
      }  
      else
      { 
         error_code = OSAL_E_DOESNOTEXIST;
      } 
   } 
   if( (tU32)error_code != OSAL_E_NOERROR )
   {
      OSAL_vSetErrorCode( (tU32)error_code );      /* Set error code */
      error_code = OSAL_ERROR;
   }
   else
   {
       error_code = OSAL_OK;
   }
   return error_code;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_s32IOControl                                                *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io control function, calls requested function with specified*
*      argument                                                         *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*                                                                       *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      OSAL_tIODescriptor fd: File descriptor                           *
*      tS32 fun: Function identificator                                 *
*      tS32 arg: Function argument                                      *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      tS32 OSAL_OK or OSAL_ERROR                                       *
*                                                                       *
************************************************************************/
tS32 OSAL_s32IOControl( OSAL_tIODescriptor fd, tS32 fun, intptr_t arg )
{
   struct OsalDevices *device = NULL;
   tS32 error_code;
   OsalDeviceDescriptor *fd_local;
   tBool bTrace = FALSE;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   fd_local = (OsalDeviceDescriptor *)fd;
   error_code = (tS32)u32CheckMagic( fd_local );
   if( (tU32)error_code == OSAL_E_NOERROR )
   {
      device = fd_local->device_ptr;
      if (device )
      {
         tBool bIsAPrmFunc = FALSE;

         if(  ( fun <= OSAL_C_S32_IOCTRL_PRM_MAX_VALUE )
           && ( fun >= OSAL_C_S32_IOCTRL_PRM_MIN_VALUE ) )
         {
            bIsAPrmFunc = TRUE;
            /* check if cd driver package is loaded */
            if(pOsalData->bCdActive == TRUE)
            {
               /*GEN3 Linux behaviour: 
                 all CDCTRL-IOctrls especially OSAL_C_S32_IOCTRL_REG_NOTIFICATION and
                 OSAL_C_S32_IOCTRL_UNREG_NOTIFICATION will be handles by DEV_CDCTRL itself and not by PRM!*/
                 if((tU32)OSAL_EN_DEVID_CDCTRL == (tU32)OSAL_get_drive_id(fd))
                 {
                    bIsAPrmFunc = FALSE; 
                 }
            }
         }

         if( TRUE == bIsAPrmFunc) /*PRM new*/
         {
            vCheckLoadedPrm();
            if(pPrmFunc)
            {
               error_code = (tS32)pPrmFunc( fd, fun, arg);
            }
            else
            {
               error_code = OSAL_E_NOTSUPPORTED;
            }
         }
         else
         {
            {
               if(device->bFileSystem == TRUE)
               {
                  if(device->ptr_s32IOControl)
                  {
                     error_code = device->ptr_s32IOControl((tS32)device->device_local_id,
                                                           fd_local->local_ptr,
                                                           fun,
                                                           arg );
                     if(error_code == OSAL_E_BADFILEDESCRIPTOR)
                     {
                        if(bCheckRestoreFd(fd))
                        {
                           error_code = device->ptr_s32IOControl((tS32)device->device_local_id,
                                                           fd_local->local_ptr,
                                                           fun,
                                                           arg );
                        }
                     }
                  }
                  else
                  {
                     error_code = OSAL_E_NOTSUPPORTED;
                  }
               }
               else
               {
                  if(rOsalDrvFuncTable[device->device_local_id].pFuncCtl)
                  {
                     error_code = rOsalDrvFuncTable[device->device_local_id].pFuncCtl((tS32)device->device_local_id,
                                                                                      fd_local->local_ptr,
                                                                                      fun,
                                                                                      arg);
                  }
                  else
                  {
                     error_code = OSAL_E_NOTSUPPORTED;
                  }
                  if(pOsalData->rDevTrace[device->device_local_id].cActive == 1)
                  {
                      bTrace = TRUE;
                      if(pOsalData->rDevTrace[device->device_local_id].Pid)
                      {
                         if(pOsalData->rDevTrace[device->device_local_id].Pid != OSAL_ProcessWhoAmI())
                         {
                             bTrace = FALSE;
                         }
                      }
                   }
               }
            }
         }
      }
   }
   if(bTrace)TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d IoControl DevId:%d Device:%s ErrorCode:%d OSAL FD:%p Func:%d Arg:%d",
                                  OSAL_ProcessWhoAmI(),device->device_local_id,device->name,error_code,fd_local, fun,arg);
   if( (tU32)error_code != OSAL_E_NOERROR )
   {
      OSAL_vSetErrorCode( (tU32)error_code );      /* Set error code */
      error_code = OSAL_ERROR;
   }
   else
   {
      error_code = OSAL_OK;
   }
   return error_code;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_s32IOControl_plain                                          *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io control function, calls requested function with specified*
*      argument                                                         *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*      u32NofsDispatcher, FsDispatcher                                  *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      OSAL_tIODescriptor fd: File descriptor                           *
*      tS32 fun: Function identificator                                 *
*      tS32 arg: Function argument                                      *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      tS32 OSAL_NO_ERROR or error code                                 *
*                                                                       *
************************************************************************/
tS32 OSAL_s32IOControl_plain( tCString name, uintptr_t local_ptr, tS32 fun, intptr_t arg )
{
   struct OsalDevices *device;
   device = OSAL_Get_Driver_by_name( name, NULL );

   if( device )
   {
      return device->ptr_s32IOControl((tS32)device->device_local_id,
                                      local_ptr,
                                      fun,
                                      arg );
   }
   else
   {
      return OSAL_E_NOTSUPPORTED;
   }
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_s32IOWrite                                                  *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io write function, write data on requested device           *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*      u32NofsDispatcher, FsDispatcher, Nucleus semaphore               *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      OSAL_tIODescriptor fd: File descriptor                           *
*      tPCS8 pBuffer: Pointer to buffer containing data                 *
*      tU32 nbytes: Buffer length                                       *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      tS32 OSAL_NOERROR or error code                                  *
*                                                                       *
************************************************************************/
tS32 OSAL_s32IOWrite( OSAL_tIODescriptor fd, tPCS8 pBuffer, tU32 nbytes )
{
   struct OsalDevices *device = NULL;
   tS32 error_code;
   tS32 s32Ret = 0;
   uintptr_t real_written_bytes = 0;
   OsalDeviceDescriptor *fd_local;
   tBool bTrace = FALSE;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   /* prove for valid read buffer */
   fd_local = (OsalDeviceDescriptor *)fd;
   if( !pBuffer )
   {
      error_code = OSAL_E_INVALIDVALUE; 
   }
   else
   {   
      error_code = (tS32)u32CheckMagic( fd_local );
   }
   if( (tU32)error_code == OSAL_E_NOERROR )
   {
      device = fd_local->device_ptr;
      if( device )
      {
            if(device->bFileSystem == TRUE)
            {
               if(device->ptr_s32IOWrite)
               {
                  error_code = device->ptr_s32IOWrite( (tS32)device->device_local_id, 
                                                      fd_local->local_ptr,
                                                      pBuffer, 
                                                      nbytes, 
                                                      &real_written_bytes );
                  if(error_code == OSAL_E_BADFILEDESCRIPTOR)
                  {
                     if(bCheckRestoreFd(fd))
                     {
                         error_code = device->ptr_s32IOWrite( (tS32)device->device_local_id, 
                                                      fd_local->local_ptr,
                                                      pBuffer, 
                                                      nbytes, 
                                                      &real_written_bytes );
                     }
                  }
               }
               else
               {
                  error_code = OSAL_E_NOTSUPPORTED;
               }
            }
            else
            {
               if(rOsalDrvFuncTable[device->device_local_id].pFuncWri)
               {
                  error_code = rOsalDrvFuncTable[device->device_local_id].pFuncWri((tS32)device->device_local_id, 
                                                                                   fd_local->local_ptr,
                                                                                   pBuffer, 
                                                                                   nbytes, 
                                                                                   &real_written_bytes );
               }
               else
               {
                  error_code = OSAL_E_NOTSUPPORTED;
               }
            }
            if( device->bFileSystem != FALSE )
            {
               /* fs descriptor functions */
               if( (tU32)error_code == OSAL_E_NOERROR )
               {
                  s32Ret = (tS32)real_written_bytes;
               }
            }
            else
            {
               /* nFS devices cannot read streams bigger than 0x720000 bytes ! */
               if( (tU32)error_code <= OSAL_E_NOERROR )
               {
                  s32Ret = error_code;
                  if( (tU32)error_code < nbytes )
                  {
                     OSAL_vSetErrorCode(OSAL_E_TIMEOUT);    /* Set error code for timeout*/
                  }
                  error_code = OSAL_E_NOERROR;
               }
               if(pOsalData->rDevTrace[device->device_local_id].cActive == 1)
               {
                   bTrace = TRUE;
                   if(pOsalData->rDevTrace[device->device_local_id].Pid)
                   {
                      if(pOsalData->rDevTrace[device->device_local_id].Pid != OSAL_ProcessWhoAmI())
                      {
                          bTrace = FALSE;
                      }
                   }
               }
            }
       }
      else
      {
         error_code = OSAL_E_DOESNOTEXIST;      /* Set error code */
      }
   }

   if(bTrace)TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Write DevId:%d Device:%s ErrorCode:%d OSAL FD:%p BytesToWrite:%d WrittenBytes:%d",
                                   OSAL_ProcessWhoAmI(),device->device_local_id,device->name,error_code,fd_local, nbytes,real_written_bytes);
   if(error_code != OSAL_E_NOERROR)
   {
      OSAL_vSetErrorCode( (tU32)error_code );      /* Set error code */
      s32Ret = OSAL_ERROR;
   }
   return s32Ret;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_s32IORead                                                   *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io read function, read data frm requested device            *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*      u32NofsDispatcher, FsDispatcher, Nucleus semaphore               *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      OSAL_tIODescriptor fd: File descriptor                           *
*      tPS8 pBuffer:  Pointer to buffer for data storing                *
*      tU32 maxbytes: Max bytes to be read                              *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      tS32 OSAL_NOERROR or error code                                  *
*                                                                       *
************************************************************************/
tS32 OSAL_s32IORead( OSAL_tIODescriptor fd, tPS8 pBuffer, tU32 maxbytes )
{
   struct OsalDevices *device = NULL;
   tS32 error_code;
   tS32 s32Ret = 0;
   uintptr_t real_read_bytes = 0;
   OsalDeviceDescriptor *fd_local;
   tBool bTrace = FALSE;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   fd_local = (OsalDeviceDescriptor *)fd;
   /* prove for valid read buffer */
   if( !pBuffer ) 
   {
      error_code = OSAL_E_INVALIDVALUE; 
   }
   else
   {
      error_code = (tS32)u32CheckMagic( fd_local );
   }
   if( (tU32)error_code == OSAL_E_NOERROR )
   {
      device = fd_local->device_ptr;
      if( device )
      {
            if(device->bFileSystem == TRUE)
            {
               if(device->ptr_s32IORead )
               {
                  error_code = device->ptr_s32IORead( (tS32)device->device_local_id, 
                                                     fd_local->local_ptr,
                                                     pBuffer, 
                                                     maxbytes, 
                                                     &real_read_bytes );
                  if(error_code == OSAL_E_BADFILEDESCRIPTOR)
                  {
                     if(bCheckRestoreFd(fd))
                     {
                        error_code = device->ptr_s32IORead( (tS32)device->device_local_id, 
                                                     fd_local->local_ptr,
                                                     pBuffer, 
                                                     maxbytes, 
                                                     &real_read_bytes );
                     }
                  }
               }
               else
               {
                  error_code = OSAL_E_NOTSUPPORTED;
               }
            }
            else
            {
               if(rOsalDrvFuncTable[device->device_local_id].pFuncRea)
               {
                  error_code = rOsalDrvFuncTable[device->device_local_id].pFuncRea((tS32)device->device_local_id, 
                                                                                   fd_local->local_ptr,
                                                                                   pBuffer, 
                                                                                   maxbytes, 
                                                                                   &real_read_bytes );
               }
               else
               {
                  error_code = OSAL_E_NOTSUPPORTED;
               }
            }
            if( device->bFileSystem != FALSE )
            {
               /* fs descriptor functions */
               if ( (tU32)error_code == OSAL_E_NOERROR )
               {
                  s32Ret = (tS32)real_read_bytes;
               }
            }
            else
            {
               /* nFS devices cannot read streams bigger than 0x720000 bytes ! */
               if ( (tU32)error_code <= OSAL_E_NOERROR )
               {
                  s32Ret = error_code;
                  if((tU32)error_code < maxbytes )
                  {
                     OSAL_vSetErrorCode(OSAL_E_TIMEOUT);    /* Set error code for timeout */
                  }
                  error_code = OSAL_E_NOERROR;
               }
               if(pOsalData->rDevTrace[device->device_local_id].cActive == 1)
               {
                   bTrace = TRUE;
                   if(pOsalData->rDevTrace[device->device_local_id].Pid)
                   {
                      if(pOsalData->rDevTrace[device->device_local_id].Pid != OSAL_ProcessWhoAmI())
                      {
                          bTrace = FALSE;
                      }
                   }
               }
            }
      }
      else
      {
         error_code = OSAL_E_DOESNOTEXIST;
      }
   }
   if(bTrace)TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "OSALIO Pid:%d Read DevId:%d Device:%s ErrorCode:%d OSAL FD:%p BytesToRead:%d ReadBytes:%d ",
                                         OSAL_ProcessWhoAmI(),device->device_local_id,device->name,error_code,fd_local, maxbytes,real_read_bytes);
   if(error_code != OSAL_E_NOERROR)
   {
      OSAL_vSetErrorCode( (tU32)error_code );      /* Set error code */
      s32Ret = OSAL_ERROR;
   }
   return s32Ret;
}


/************************************************************************
*                                                                       *
* FUNCTIONS                                                             *
*                                                                       *
*      OSAL_Remove                                                      *
*                                                                       *
* DESCRIPTION                                                           *
*      OSAL io remove function, remove request file (device)            *
*                                                                       *
* CALLS                                                                 *
*                                                                       *
*      NoFsDispatcher, FsDispatcher, Nucleus semaphore                  *
*                                                                       *
* INPUTS                                                                *
*                                                                       *
*      tCString szName file name                                        *
*      OSAL_tenAccess enAccess access type                              *
*                                                                       *
* OUTPUTS                                                               *
*                                                                       *
*      OSAL_tIODescriptor file descriptor                               *
*                                                                       *
************************************************************************/
tS32 OSAL_s32IORemove( tCString coszName )
{
   struct OsalDevices *device;
   tS32 error_code;
   tCString filename = NULL;

   if(u32PrcExitStep >= 2) return OSAL_ERROR;

   device = OSAL_Get_Driver_by_name( coszName, &filename );
   if((filename!=NULL) &&( device && device->ptr_s32IORemove ))
   {
      if(device->bFileSystem == TRUE)
      {
         error_code = device->ptr_s32IORemove((tS32)device->device_local_id,filename);
      }
      else
      {
         error_code = OSAL_E_NOTSUPPORTED;
      }
   }
   else
   {
      error_code = OSAL_E_DOESNOTEXIST; 
   }
   if( (tU32)error_code != OSAL_E_NOERROR )
   {
      OSAL_vSetErrorCode( (tU32)error_code );      /* Set error code */
      error_code = OSAL_ERROR;
   }
   else
   {
      error_code = OSAL_OK;
   }
   return error_code;
}


/******************************************************FunctionHeaderBegin******
 * FUNCTION    : to_upper
 * CREATED     : 2004-10-11
 * AUTHOR      :
 * DESCRIPTION :   -
 * SYNTAX      : char to_upper(const char c);
 * ARGUMENTS   : c : character to convert to upper value
 * RETURN VALUE: -
 *
 * NOTES       : -
 *
 *******************************************************FunctionHeaderEnd******/

char to_upper( const char c )
{
  int x;
  
  x = c;
  if ((x >= 'a') && (x <= 'z'))
  {
    x += (char)('A' - 'a');
  }
  return (char)x;
}



void vSetDevTrace(char* pcPid ,const char* name)
{
   struct OsalDevices *device;
   tCString dummy;
   OSAL_tProcessID Pid = *((tS32*)(pcPid));/*lint !e826 */

   device = OSAL_Get_Driver_by_name( name, &dummy );
   if(device)
   {
     if(pOsalData->rDevTrace[device->device_local_id].cActive == 0)
     {
        pOsalData->rDevTrace[device->device_local_id].cActive = 1;
        pOsalData->rDevTrace[device->device_local_id].Pid  = Pid;
        TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "Set Trace Active for Device %s PID:%d ",name,Pid);
     }
     else
     {
        pOsalData->rDevTrace[device->device_local_id].cActive = 0;
        pOsalData->rDevTrace[device->device_local_id].Pid  = 0;
        TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "Set Trace InActive for Device %s PID:%d ",name,Pid);
     }
   }
   else
   {
      TraceStringClassLevel(TR_COMP_OSALIO,TR_LEVEL_FATAL, "Invalid Device %s for DevTrace used",name);
   }
}


/*****************************************************************************
*
* FUNCTION:    TRACE_s32IOCreate
*
* DESCRIPTION: Dummy implementation                
*
* PARAMETER:   tS32 s32DevId
*
* RETURNVALUE: tS32 error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tS32 TRACE_s32IOCreate( tS32 s32DevId )
{
   ((void)s32DevId);/* satisfy lint*/
   return (tS32)OSAL_E_NOERROR;
}


/*****************************************************************************
*
* FUNCTION:    TRACE_s32IORemove
*
* DESCRIPTION: Dummy implementation                
*
* PARAMETER:   tS32 s32DevId
*
* RETURNVALUE: tS32 error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tS32 TRACE_s32IORemove( tS32 s32DevId )
{
   ((void)s32DevId);/* satisfy lint*/
   return(tS32)OSAL_E_NOERROR;
}


/*****************************************************************************
*
* FUNCTION:    TRACE_s32IOOpen
*
* DESCRIPTION: get the OSAL Trace handle                
*
* PARAMETER:   none
*
* RETURNVALUE: tS32 error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tS32 TRACE_s32IOOpen(tVoid)
{
   return (tS32)OSAL_E_NOERROR;
}


/*****************************************************************************
*
* FUNCTION:    TRACE_s32IOClose
*
* DESCRIPTION: Dummy implementation                
*
* PARAMETER:   none
*
* RETURNVALUE: tS32 error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tS32 TRACE_s32IOClose(tVoid)
{
    return (tS32)OSAL_E_NOERROR;
}


/*****************************************************************************
*
* FUNCTION:    TRACE_s32IOControl
*
* DESCRIPTION: This functions gets an trace handle for OSAL using applications                
*
* PARAMETER:   tS32 s32DevId, device ID
*              tS32 s32fun,   function type
*              tS32 s32arg    parameter for function type
*
* RETURNVALUE: tS32  error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tS32 TRACE_s32IOControl(tS32 s32DevId, tS32 s32fun, intptr_t s32arg)
{
    tU32 u32Errorcode = OSAL_E_NOERROR;
   ((void)s32DevId);/* satisfy lint*/

    /* Parameter check */
   if(s32arg)
   {
      switch(s32fun)
      {
         case OSAL_C_S32_IOCTRL_VERSION:
                   u32Errorcode = OSAL_E_NOTSUPPORTED;
            //*(tPS32)s32arg = TRACE_C_S32_IO_VERSION;
            break;
         case OSAL_C_S32_IOCTRL_ISACTIVE:
            {
               OSAL_trIOCtrlActivTrace *prIOCtrlTemp = (OSAL_trIOCtrlActivTrace *)s32arg;
               if(LLD_bIsTraceActive(prIOCtrlTemp->enTraceClass,prIOCtrlTemp->enTraceLevel) !=FALSE)
               {
                  prIOCtrlTemp->bIsActive = TRUE;
               }
            }
            break;
#ifdef USE_TRACE_IF
         case OSAL_C_S32_IOCTRL_CALLBACK_REG:
            {
               OSAL_trIOCtrlLaunchChannel *pChanData = (OSAL_trIOCtrlLaunchChannel*)s32arg;
               if(chan_acess_bRegChan)
               {
                  if(!chan_acess_bRegChan(pChanData->enTraceChannel,pChanData->pCallback))
                  {
                     u32Errorcode = OSAL_E_ALREADYEXISTS;
                  }
               }
            }
            break;
         case OSAL_C_S32_IOCTRL_CALLBACK_UNREG:
            {
               OSAL_trIOCtrlLaunchChannel *pChanData = (OSAL_trIOCtrlLaunchChannel*)s32arg;
               if(chan_acess_bUnRegChan)
               {
                  if(!chan_acess_bUnRegChan(pChanData->enTraceChannel,pChanData->pCallback))
                  {
                      u32Errorcode = OSAL_E_DOESNOTEXIST;
                  }
               }
            }
            break;
         case OSAL_C_S32_IOCTRL_SEND_BINARY_DATA:
            {
               OSAL_trIOCtrlBinaryData *pBinaryData = (OSAL_trIOCtrlBinaryData*)s32arg;
               if(TraceBinOutput)
               {
                  tU32 n_size = TraceBinOutput(pBinaryData->u32CompId,
                                                   pBinaryData->u8SockId,
                                                   pBinaryData->u32BufferLen,
                                                   pBinaryData->puchDataBuffer);
                  if(n_size <= pBinaryData->u32BufferLen)
                  {
                      u32Errorcode = OSAL_E_UNKNOWN;
                  }
               }
            }
            break;
#endif
         case OSAL_C_S32_IOCTRL_INIT_TRIP_DATA_REC:
            {  
               OSAL_trIOCtrlInitTripData *pTripData = (OSAL_trIOCtrlInitTripData*)s32arg;
               if(!bInitTripFileRecord(pTripData->strFileName))
               {
                        u32Errorcode = OSAL_E_UNKNOWN;
               }
            }
            break;
         case OSAL_C_S32_IOCTRL_INIT_TRIP_DATA_REPLAY :
            {
               OSAL_trIOCtrlInitTripData *pTripData = (OSAL_trIOCtrlInitTripData*)s32arg;
               if(!bInitTripFileReplay(pTripData->strFileName,
                                       pTripData->u8Mode,
                                       pTripData->u32Interval))
               {
                         u32Errorcode = OSAL_E_UNKNOWN;
               }
            }
            break;
         case OSAL_C_S32_IOCTRL_STOP_TRIP_DATA_REPLAY :
               vCloseTripFile();
            break;
         case OSAL_C_S32_IOCTRL_GET_NEXT_TRIP_DATA :
               if(!bSendTripMessages((tU32)s32arg))
               {
                      u32Errorcode = OSAL_E_UNKNOWN;
            }
            break;
         case OSAL_C_S32_IOCTRL_GET_OUTPUTCHANNEL :
                   u32Errorcode = OSAL_E_NOTSUPPORTED;
            break;
         case OSAL_C_S32_IOCTRL_SET_OUTPUTCHANNEL :
            {
                   u32Errorcode = OSAL_E_NOTSUPPORTED;
            }
            break;
         default:
                   u32Errorcode = OSAL_E_NOTSUPPORTED;
            break;
      }
   }
   else
   {
        u32Errorcode = OSAL_E_INVALIDVALUE;
   }
   return (tS32)(u32Errorcode);
}


/*****************************************************************************
*
* FUNCTION:    bInitTripFileRecord
*
* DESCRIPTION:               
*
* PARAMETER:   tCString strFileName   Tip file name
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if everything goes right;
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tBool bInitTripFileRecord( tCString strFileName)
{
   tBool bResult = TRUE;

   (void)strFileName;   // lint
   // This code forced an exception so it is commented out 
   //   tU8 u8Buffer = 0x0A;
   //   ((void)strFileName);/* satisfy lint*/
   //    TraceOut(1,TR_CLASS_TRIP_SESSION,(TR_tenTraceLevel)TR_LEVEL_USER_4,(tU8*)u8Buffer);

   return bResult;
}


/*****************************************************************************
*
* FUNCTION:    bInitTripFileRecord
*
* DESCRIPTION:               
*
* PARAMETER:   tCString strFileName   Tip file name
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if everything goes right;
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void vPutU32InTripData(const tU32 u32Lenght,tU8* puchBuffer)
{
   puchBuffer[0] = (tU8)((u32Lenght )>> 24);
   puchBuffer[1] = (tU8)((u32Lenght )>> 16);
   puchBuffer[2] = (tU8)((u32Lenght )>> 8);
   puchBuffer[3] = (tU8)(u32Lenght  );
}


/*****************************************************************************
*
* FUNCTION:    bInitTripFileRecord
*
* DESCRIPTION:               
*
* PARAMETER:   tCString strFileName   Tip file name
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if everything goes right;
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tBool bInitTripFileReplay( tString strFileName, tU8 u8Mode, tU32 u32Interval)
{
   if( pOsalData->bIsInitialised != TRUE)
   {
      //get shared memory
      char *pu8Buffer = (char*)malloc(5/*TRACE_HEADER_SIZE*/+1/*TR_TRIP_FILE_LENGTH_BYTE*/+ strlen(strFileName)+1);
      if( pu8Buffer != NULL )
      {
         pu8Buffer[0] = 0;//TODO -> TR_TRIP_MODE_TRIP_REPLAY;
         pu8Buffer[1] = (char)u8Mode;
         vPutU32InTripData(u32Interval,(tU8*)(&pu8Buffer[2]));
         pu8Buffer[6] = (char)strlen(strFileName);
         sprintf(&pu8Buffer[7],"%s",strFileName);
         LLD_vTrace(TR_CLASS_TRIP,(TR_tenTraceLevel)TR_LEVEL_USER_4,(const void*)pu8Buffer,(tU32)(pu8Buffer[6]+7));
 
         free(pu8Buffer);
         pu8Buffer = NULL;
         pOsalData->bIsInitialised = TRUE;
      }
   }
   return pOsalData->bIsInitialised;
}


/*****************************************************************************
*
* FUNCTION:    bInitTripFileRecord
*
* DESCRIPTION:               
*
* PARAMETER:   tCString strFileName   Tip file name
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if everything goes right;
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tBool bSendTripMessages( tU32 u32Count )
{
   if( pOsalData->bIsInitialised == TRUE )
   {
      char u8Buffer[5] = {0};
      
      u8Buffer[0] = 0x03;
      vPutU32InTripData(u32Count,(tU8*)(&u8Buffer[1]));
      LLD_vTrace(TR_CLASS_TRIP,(TR_tenTraceLevel)TR_LEVEL_USER_4,(const void*)&u8Buffer[0],5);
   }
   return TRUE;
}


/*****************************************************************************
*
* FUNCTION:    bInitTripFileRecord
*
* DESCRIPTION:               
*
* PARAMETER:   tCString strFileName   Tip file name
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if everything goes right;
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void vCloseTripFile(void)
{
   char u8Buffer[1] = {0};
   u8Buffer[0] = 0x02;
   LLD_vTrace(TR_CLASS_TRIP,(TR_tenTraceLevel)TR_LEVEL_USER_4,(const void*)&u8Buffer[0],1);
   pOsalData->bIsInitialised = FALSE;
}


/*****************************************************************************
*
* FUNCTION:    TRACE_s32IOWrite
*
* DESCRIPTION: Write to Trace device             
*
* PARAMETER:   tS32 s32DevId,                     device handle
*              OSAL_trIOWriteTrace *prIOTrData,   trace data
*              tU32 u32nbytes                     size of trace data
*
* RETURNVALUE: tS32 error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tS32 TRACE_s32IOWrite( tS32 s32DevId,const OSAL_trIOWriteTrace *prIOTrData, tU32 u32nbytes )
{
   ((void)s32DevId); /* satisfy lint */
   (void)u32nbytes;  // lint

   LLD_vTrace(prIOTrData->u32Class,(TR_tenTraceLevel)prIOTrData->u32Level,(const void*)prIOTrData->pcos8Buffer,prIOTrData->u32Length);
   return OSAL_OK;
}

/************************************************************************
|end of file dispatcher_table.c
|-----------------------------------------------------------------------*/
