/************************************************************************
| FILE:         osaltrace.cpp
| PROJECT:      platform
| SW-COMPONENT: OSAL CORE
|------------------------------------------------------------------------
| DESCRIPTION:  Assistance functions for OSAL trace
|------------------------------------------------------------------------
| COPYRIGHT:    (c) 2010 Robert Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 03.10.05  | Initial revision           | MRK2HI
| 11.03.09  | MMS 223874 :               | JEV1KOR 
|           | Corrections in FileCopy()  |
| --.--.--  | ----------------           | -------, -----
| 31.03.09  | Made new TTFis command      | anc3kor
|            |    to Format SDCard            |
| 22.08.17  | Coverity Fix               | boc7kor
|************************************************************************/

/************************************************************************ 
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Linux_osal.h"

#include <syslog.h>


#ifdef __cplusplus
extern "C" {
#endif


extern dbg_func pCoreDbgFunc;
extern dbg_func pIoDbgFunc;


/************************************************************************ 
|defines and macros (scope: module-local) 
|-----------------------------------------------------------------------*/
//#define EARLY_STARTUP_TRACE_ACTIVE

/************************************************************************ 
|typedefs (scope: module-local) 
|-----------------------------------------------------------------------*/
    
/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/
static int fd_log = -1;

#ifndef QNX
static char LogIdName[16];
static const char* LogId = &LogIdName[0];
#endif

#ifdef USE_TRACE_IF
static void* pHandle = NULL;
static  tBool bTraceLoadDone = FALSE;
pFuncIsClassSelected        IsClassSelected = NULL;
OSAL_DECL pFuncTraceOut     TraceOut = NULL;
pFuncTraceBinOutput         TraceBinOutput = NULL;
pFunc_chan_acess_bRegChan   chan_acess_bRegChan = NULL;
pFunc_chan_acess_bUnRegChan chan_acess_bUnRegChan = NULL;
OSAL_DECL pFunc_get_blockmode_status  get_blockmode_status = NULL;
OSAL_DECL pFunc_SendCmd     SendCmd = NULL;
#endif

#ifdef USE_DLT_IF
DLT_DECLARE_CONTEXT(OsalCntxt); /* declare context */

static void* pDltHandle = NULL;
static  tBool bDltLoadDone = FALSE;
pFuncdlt_register_app       fpdlt_register_app = NULL;
pFuncdlt_unregister_app     fpdlt_unregister_app = NULL;
pFuncdlt_register_context   fpdlt_register_context = NULL;
pFuncdlt_unregister_context fpdlt_unregister_context = NULL;
pFuncdlt_register_injection_callback fpdlt_register_injection_callback = NULL;
OSAL_DECL pFuncdlt_log_string fpdlt_log_string = NULL;
pFuncdlt_log_raw            fpFuncdlt_log_raw = NULL;
#endif

/************************************************************************ 
| variable definition (scope: global)
|-----------------------------------------------------------------------*/
extern tBool bConstructFlag;

/************************************************************************
|function prototype (scope: module-local)
|-----------------------------------------------------------------------*/


/************************************************************************ 
|function implementation (scope: module-local) 
|-----------------------------------------------------------------------*/



void InitOsalTrace(void)
{
   tU32 i;
   pOsalData->u32TraceLevel = 2;

   for(i=0;i< 50;i++)
   {
      pOsalData->u32Class[i] = 0xffffffff;
   }
}

#ifdef USE_TRACE_IF
tBool LoadTrace(void)
{
   OSAL_tProcessID Pid =  OSAL_ProcessWhoAmI();
//   char *error;
   bTraceLoadDone = TRUE;
   dlerror();    /* Clear any existing error */
   pHandle = dlopen("libtrace.so", RTLD_NOW );
   if(pHandle == NULL)
   {
       return FALSE;
   }
   else
   {
      IsClassSelected       = (pFuncIsClassSelected)(dlsym(pHandle, (const char*)"TR_core_bIsClassSelected"));/*lint !e611 */;
      TraceOut              = (pFuncTraceOut)(dlsym(pHandle, (const char*)"TR_core_uwTraceOut"));/*lint !e611 */
      TraceBinOutput        = (pFuncTraceBinOutput)(dlsym(pHandle, (const char*)"TR_core_uwTraceBinOutput"));/*lint !e611 */
      chan_acess_bRegChan   = (pFunc_chan_acess_bRegChan)(dlsym(pHandle, (const char*)"TR_chan_acess_bRegChan"));/*lint !e611 */
      chan_acess_bUnRegChan = (pFunc_chan_acess_bUnRegChan)(dlsym(pHandle, (const char*)"TR_chan_acess_bUnRegChan"));/*lint !e611 */
      get_blockmode_status  = (pFunc_get_blockmode_status)(dlsym(pHandle, (const char*)"TR_get_blockmode_status"));/*lint !e611 */
      SendCmd               = (pFunc_SendCmd)(dlsym(pHandle, (const char*)"TR_core_s32SendCmd"));/*lint !e611 */
         
      if((!IsClassSelected)||(!TraceOut)||(!TraceBinOutput)||(!chan_acess_bRegChan)||(!chan_acess_bUnRegChan)||(!get_blockmode_status)||(!SendCmd))
      {
         TraceOut = NULL;
         dlclose(pHandle);
         return FALSE;
      }
      if(LLD_bIsTraceActive(TR_COMP_OSALCORE,TR_LEVEL_COMPONENT))
      {
         tS32 s32PidEntry = s32FindProcEntry(Pid);
         if(s32PidEntry == OSAL_ERROR)
         {
           //NORMAL_M_ASSERT_ALWAYS();
           TraceString("Loaded Trace for unknown PID OSAL during load?");
         }
         else
         {
            TraceString("Loaded Trace PID:%d %s",(int)Pid,prProcDat[s32PidEntry].pu8CommandLine);
         }
      }
      return TRUE;
   }
}
#endif

#ifdef USE_DLT_IF

tBool LoadDlt(void)
{
   OSAL_tProcessID Pid =  OSAL_ProcessWhoAmI();
//   char *error;
   bDltLoadDone = TRUE;
   dlerror();    /* Clear any existing error */
   pDltHandle = dlopen(LIB_DLT, RTLD_NOW );
   if(pDltHandle == NULL)
   {
       return FALSE;
   }
   else
   {
      fpdlt_register_app       = (pFuncdlt_register_app)(dlsym(pDltHandle, (const char*)"dlt_register_app"));/*lint !e611 */;
      fpdlt_unregister_app     = (pFuncdlt_unregister_app)(dlsym(pDltHandle, (const char*)"dlt_unregister_app"));/*lint !e611 */
      fpdlt_register_context   = (pFuncdlt_register_context)(dlsym(pDltHandle, (const char*)"dlt_register_context"));/*lint !e611 */
      fpdlt_unregister_context = (pFuncdlt_unregister_context)(dlsym(pDltHandle, (const char*)"dlt_unregister_context"));/*lint !e611 */
      fpdlt_register_injection_callback = (pFuncdlt_register_injection_callback)(dlsym(pDltHandle, (const char*)"dlt_register_injection_callback"));/*lint !e611 */
      fpdlt_log_string         = (pFuncdlt_log_string)(dlsym(pDltHandle, (const char*)"dlt_log_string"));/*lint !e611 */
      fpFuncdlt_log_raw        = (pFuncdlt_log_raw)(dlsym(pDltHandle, (const char*)"dlt_log_raw"));
      if((!fpdlt_register_app)||(!fpdlt_unregister_app)||(!fpdlt_register_context)||(!fpdlt_unregister_context)||(!fpdlt_register_injection_callback)
      ||(!fpdlt_log_string) || (!fpFuncdlt_log_raw))
      {
         fpdlt_log_string = NULL;
         dlclose(pDltHandle);
         return FALSE;
      }
      if(LLD_bIsTraceActive(TR_COMP_OSALCORE,TR_LEVEL_COMPONENT))
      {
         tS32 s32PidEntry = s32FindProcEntry(Pid);
         if(s32PidEntry == OSAL_ERROR)
         {
           //NORMAL_M_ASSERT_ALWAYS();
           TraceString("Loaded DLT for unknown PID OSAL during load?");
         }
         else
         {
            TraceString("Loaded DLT PID:%d %s",(int)Pid,prProcDat[s32PidEntry].pu8CommandLine);
         }
      }
      return TRUE;
   }
}
#endif

/*****************************************************************************
*
* FUNCTION:    LLD_bOpenTrace
*
* DESCRIPTION: This functions gets an trace handle for OSAL using applications                
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if everything goes right;
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tBool LLD_bOpenTrace()
{
#ifdef QNX
#else
   if(pOsalData->u32OsalTrace == 2)
   {
       snprintf(LogIdName,16,"OSAL_%d",(int)OSAL_ProcessWhoAmI());
       openlog(LogId, LOG_CONS,LOG_USER );
   }
#endif
   return TRUE;
}


                                                                      
 

/*****************************************************************************
*
* FUNCTION:    LLD_vCloseTrace
*
* DESCRIPTION: This functions closes the trace handle for OSAL using 
*              applications, if it isn' used furthermore
*
* PARAMETER:   none
*
* RETURNVALUE: none
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tVoid LLD_vCloseTrace(void)
{
#ifdef USE_DLT_IF
   fpdlt_unregister_context(&OsalCntxt);
   fpdlt_unregister_app();
#endif
    if(fd_log != -1)
    {
       close(fd_log);
    }
}

tBool bCheckTraceActive(tU32 u32Class, tU32 u32Level)
{
   tBool bRet = FALSE;
   tU32 i;
   
   if(pOsalData == NULL)return FALSE;
   
   // evalute global trace level first
   if(u32Level <= pOsalData->u32TraceLevel)
   {
      bRet = TRUE;
   }

   for(i=0;i< pOsalData->u32NrOfClass;i++)
   {
      if((pOsalData->u32Class[i])&&(pOsalData->u32Class[i] == u32Class))
      {
         if(u32Level <= pOsalData->u32Level[i])
         {
            bRet = TRUE;
         }
         else
         {
             // override global trace level if specific class was set to lower level
             bRet = FALSE;
         }
         break;
      }
   }
   return bRet;
}


/*****************************************************************************
*
* FUNCTION:    LLD_bIsTraceActive
*
* DESCRIPTION: This functions proves if the required trace class/level is 
*              switched on                
*
* PARAMETER:   tU32 u32Class    required trace class
*              tU32 u32Level    required trace level
*
* RETURNVALUE: tBool
*                 it is the function return value: 
*                 - TRUE if class/level is selected
*                 - FALSE otherwise.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tBool LLD_bIsTraceActive(tU32 u32Class, tU32 u32Level)
{
#ifdef USE_DLT_IF
   ((void)u32Class);
   ((void)u32Level);
   return TRUE;
#endif

#ifdef USE_TRACE_IF
   if((bTraceLoadDone == FALSE))
   {
       LoadTrace();
   }
   if(IsClassSelected)
   {
      return (tBool)IsClassSelected((tU16)u32Class,(TR_tenTraceLevel)u32Level);
   }
#endif
   return bCheckTraceActive(u32Class,u32Level);
}


void vWriteTraceRawLog(int fd,uint32_t u32Class,void* pvData, uint32_t u32Length)
{
   uint32_t datalen = 0;
   uint32_t len = u32Length;
   char cDest[4096];
   char Destination[4];
   unsigned char* pData = (unsigned char*)&u32Class;
   
   if(fd)
   {
      uint32_t i, j;
      datalen = (uint32_t)strlen("DATA:");
      memcpy(&cDest[0],"DATA:",datalen);
      sprintf(Destination,"%02x", *pData);  
      cDest[datalen+3] = Destination[0];
      cDest[datalen+4] = Destination[1];
      pData++;
 
      sprintf(Destination,"%02x", *pData);  
      cDest[datalen] = Destination[0];
      cDest[datalen+1] = Destination[1];
      cDest[datalen+2] = ' ';
      strcpy(&cDest[datalen+5]," 00 00 00 ");

      j=datalen + 15;
      for(i=0;i<len;i++)
      {
        sprintf(Destination,"%02x",*((unsigned char*)pvData+i));  
        cDest[j] = Destination[0];
        j++;
        cDest[j] = Destination[1];
        j++;
        cDest[j] = ' ';
        j++;
      }
      cDest[j] = 0x0d;
      j++;
      cDest[j] = 0x0a;
      if(write(fd,cDest,j+1)){}
   }
}

#ifdef USE_DLT_IF
tU32 u32MapDltLevel(tU32 u32Level)
{
      switch(u32Level)
      {
       case TR_LEVEL_FATAL:
            u32Level = DLT_LOG_FATAL;
           break;
       case TR_LEVEL_ERRORS:
            u32Level = DLT_LOG_ERROR;
           break;
       case TR_LEVEL_SYSTEM_MIN:
            u32Level = DLT_LOG_WARN;
           break;
       case TR_LEVEL_SYSTEM:
            u32Level = DLT_LOG_INFO;
           break;
       case TR_LEVEL_COMPONENT:
            u32Level = DLT_LOG_INFO;
           break;
       case TR_LEVEL_USER_1:
            u32Level = DLT_LOG_DEBUG;
           break;
       case TR_LEVEL_USER_2:
            u32Level = DLT_LOG_DEBUG;
           break;
       case TR_LEVEL_USER_3:
            u32Level = DLT_LOG_VERBOSE;
           break;
       case TR_LEVEL_USER_4:
            u32Level = DLT_LOG_VERBOSE;
           break;
       default:
            u32Level = DLT_LOG_OFF;
           break;
      }
      return u32Level;
}
#endif
#ifdef ANDROID
tU32 u32MapAndroidLevel(tU32 u32Level)
{
   tU32 u32RetLevel;
   switch(u32Level)
   {
      case TR_LEVEL_USER_3:
      case TR_LEVEL_USER_4:
           u32RetLevel = ANDROID_LOG_VERBOSE;
          break;
      case TR_LEVEL_USER_1:
      case TR_LEVEL_USER_2:
           u32RetLevel = ANDROID_LOG_DEBUG;
          break;
      case TR_LEVEL_COMPONENT:
           u32RetLevel = ANDROID_LOG_INFO;
          break;
      case TR_LEVEL_SYSTEM_MIN:
           u32RetLevel = ANDROID_LOG_WARN;
          break;
      case TR_LEVEL_ERRORS:
           u32RetLevel = ANDROID_LOG_ERROR;
          break;
      case TR_LEVEL_FATAL:
           u32RetLevel = ANDROID_LOG_FATAL;
          break;
      default:
           u32RetLevel = ANDROID_LOG_UNKNOWN;
          break;
   }
   return u32RetLevel;
}
#endif

#ifdef QNX
tU32 u32MapQnxLevel(tU32 u32Level)
{
   tU32 u32RetLevel;
   switch(u32Level)
   {
      case TR_LEVEL_USER_3:
      case TR_LEVEL_USER_4:
           u32RetLevel = _SLOG_DEBUG2;
          break;
      case TR_LEVEL_USER_1:
      case TR_LEVEL_USER_2:
           u32RetLevel = _SLOG_INFO;
          break;
      case TR_LEVEL_COMPONENT:
           u32RetLevel = _SLOG_NOTICE;
          break;
      case TR_LEVEL_SYSTEM_MIN:
           u32RetLevel = _SLOG_WARNING;
          break;
      case TR_LEVEL_ERRORS:
           u32RetLevel = _SLOG_ERROR;
          break;
      case TR_LEVEL_FATAL:
           u32RetLevel = _SLOG_CRITICAL;
          break;
      default:
           u32RetLevel = _SLOG_SHUTDOWN;
          break;
   }
   return u32RetLevel;
}
#endif

void LoadSharedObjectTraceInterface(void)
{
#ifdef USE_TRACE_IF
   if((bTraceLoadDone == FALSE))
   {
      LoadTrace();
      if((TraceOut == NULL)&&(pOsalData->u32OsalTrace == 0))
      {
         pOsalData->u32OsalTrace = 1;
      }
   }
#endif 
#ifdef USE_DLT_IF
   if(!fpdlt_log_string)
   {
      /* loading DLT in shred library contructor causes deadlock */
      if((bDltLoadDone == FALSE)&&(!bConstructFlag))
      {
         pOsalData->u32OsalTrace = 0;
         LoadDlt();
         if(fpdlt_register_app && fpdlt_register_context)
         {
             fpdlt_register_app("OAPP", "OSAL for process");
             fpdlt_register_context(&OsalCntxt, "OSAL", "OSAL Context for Logging");
         }
         if((fpdlt_log_string == NULL)&&(pOsalData->u32OsalTrace == 0))
         {
             pOsalData->u32OsalTrace = 1;
         }
      }
   }
#endif //USE_DLT_IF	   
}

#if defined QNX || defined ANDROID
int stdout_copy = -1;
int stdout_new = -1;

void PrepareStdOut(void)
{
   if(stdout_copy == -1)
   {
      stdout_copy = dup(1);
      close(0);
      stdout_new = open("/dev/tty",O_WRONLY);
   }
}
/*
close(file1);
dup2(stdout_copy, 1);
close(stdout_copy);
*/
#endif

static int32_t s32FillTracePrefix(char* Prefix, size_t prefixSize, uint32_t u32Class, uint32_t u32Level)
{
   uint32_t u32Time = OSAL_ClockGetElapsedTime();
   int32_t s32Size = snprintf(Prefix, prefixSize, "[%02d:%02d:%02d.%03d|%05d|%u]",
                           (int)(u32Time / (60 * 60 * 1000) % 24),
                           (int)(u32Time / (60 * 1000) % 60),
                           (int)(u32Time / 1000 % 60),
                           (int)(u32Time % 1000),(int)u32Class,(unsigned int)u32Level);
   return s32Size;
}


void vNativeTrace(uint32_t u32Class, uint32_t u32Level, const void* pvData, uint32_t u32Length)
{
    char cTraceBuffer[300];
    int32_t s32Size;
    if(pOsalData && pOsalData->u32OsalTrace)
    {
       tBool bDoTrace = bCheckTraceActive(u32Class,u32Level);
       if(bDoTrace == TRUE)
       {
          char Prefix[50]= "_";
          char* pTmp;
          s32FillTracePrefix(Prefix, sizeof(Prefix), u32Class, u32Level); 
          if(*((unsigned char*)pvData) == 0xf1)pTmp = ((char*)pvData+1);
          else pTmp = ((char*)pvData);

          if(pOsalData->u32OsalTrace == 2)//make sure syslogd is started in QNX
          {
             syslog(LOG_EMERG,"%s\n",pTmp);
          }
          else
          {
             s32Size = snprintf(cTraceBuffer,300,"%s %s %s",Prefix,pTmp,"\n");
             if(write(STDOUT_FILENO,cTraceBuffer, (s32Size<300)?s32Size+1:300)){}
          }

          if(pOsalData->u32WritToFile)
          {
             if(fd_log == -1)
             {
                fd_log = open(&pOsalData->szPathTraceLog[0],O_RDWR | O_CREAT | O_APPEND ,OSAL_ACCESS_RIGTHS);
                if((fd_log == -1)&&(errno == EEXIST))
                {
                   fd_log = open(&pOsalData->szPathTraceLog[0],O_RDWR | O_APPEND ,OSAL_ACCESS_RIGTHS);
                }
                else
                {
                   if(s32OsalGroupId && (fd_log != -1))
                   {
                     /* adapt rights for shared memory */
                     if(fchown(fd_log,(uid_t)-1,s32OsalGroupId) == -1)
                     {
                        vWritePrintfErrmem("open fd_log -> fchown error %d \n",errno);
                     }
                     // umask (022) is overwriting the permissions on creation, so we have to set it again
                     if(fchmod(fd_log, OSAL_ACCESS_RIGTHS) == -1)
                     {
                        vWritePrintfErrmem("open fd_log -> fchmod error %d \n",errno);
                     }
                   }
                }
             }
             if(fd_log != -1)
             {
                if(pOsalData->u32WritToFile == 2)
                {
                   /* do original trace to allow trace via file@card in TTFIS */
                   vWriteTraceRawLog(fd_log,u32Class,(void*)pvData,u32Length);/*lint !e1773 */
                }
                else
                {
                   s32Size = snprintf(cTraceBuffer, 300, "%s %s\n", Prefix, pTmp);
			       if(write(fd_log,cTraceBuffer, (s32Size<300)?s32Size+1:300)){}
                }
             }
          }
       }
    }
#ifdef EARLY_STARTUP_TRACE_ACTIVE
    else
    {
          char Prefix[50]= "_";
          char* pTmp;
          s32FillTracePrefix(Prefix, sizeof(Prefix), u32Class, u32Level);
          if(*((unsigned char*)pvData) == 0xf1)pTmp = ((char*)pvData+1);
          else pTmp = ((char*)pvData);

          s32Size = snprintf(cTraceBuffer,300,"%s %s %s",Prefix,pTmp,"\n");
          if(write(STDOUT_FILENO,cTraceBuffer, (s32Size<300)?s32Size+1:300)){}
    }		
#endif	
}

/*****************************************************************************
*
* FUNCTION:    LLD_vTrace
*
* DESCRIPTION: This functions generates trace output for OSAL using applications                
*
* PARAMETER:   uint32_t u32Class    required trace class
*              uint32_t u32Level    required trace level
*              void* pvData     pointer to trace data
*              uint32_t u32Length   trace data length
*
* RETURNVALUE: none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void LLD_vTrace(uint32_t u32Class, uint32_t u32Level, const void* pvData, uint32_t u32Length)
{
#ifdef ANDROID
   char ClassName[16];
   char* pTemp = (char*)pvData;
   tU32 u32AndroidLevel = u32MapAndroidLevel(u32Level);
   snprintf(ClassName,16,"OSAL_%d",u32Class);
   /* check if incoming data are in string format */
   if(*pTemp == 0xf1)
   {
      __android_log_print(u32AndroidLevel, ClassName, "%s",pTemp+1);
   }
   else
   {
      __android_log_print(u32AndroidLevel, ClassName, "%s",pTemp);	   
   }
#endif

#ifdef QNX
/* #define _SLOGC_PRIVATE_START 10000  * For private use by OEM apps *
 * #define _SLOGC_PRIVATE_END	 19999  * End of private OEM range */
   uint32_t u32QnxLevel = u32MapQnxLevel(u32Level);
   unsigned char* pTemp = (unsigned char*)(pvData);
   /* check if incoming data are in string format */
   if(*pTemp == 0xf1)
   {
      slogf(_SLOG_SETCODE(_SLOGC_PRIVATE_START+1, 2),u32QnxLevel,(const char*)(pTemp+1)); // -> printf like
   }
   else
   {
      slogb(_SLOG_SETCODE(_SLOGC_PRIVATE_START+1, 2),u32QnxLevel,(void*)pvData,u32Length); // -> binary trace
   }
#endif

#ifdef USE_TRACE_IF // Bosch Trace IF shall be used for logging
   if(TraceOut != NULL)
   {
      TraceOut(u32Length,(unsigned short)u32Class,(TR_tenTraceLevel)u32Level,(tU8*)pvData); /*lint !e1773 */
   }
#endif

#ifdef USE_DLT_IF // Bosch DLT IF shall be used for logging
   if(fpdlt_log_string != NULL)
   {
       u32Level = u32MapDltLevel(u32Level);
       char* pTemp = (char*)pvData;
       if(*pTemp == 0xf1)
       {
          fpdlt_log_string(&OsalCntxt, (DltLogLevelType)u32Level,(char*)pTemp+1);
       }
       else
       {
         fpFuncdlt_log_raw(&OsalCntxt, (DltLogLevelType)u32Level,(void*)pvData,u32Length);
       }
   }
#endif //USE_DLT_IF	   

   /* check for tracing via syslog, console or logging into a file */
   vNativeTrace(u32Class,u32Level,pvData,u32Length);
}



/*****************************************************************************
*
* FUNCTION:    LLD_vRegTraceCallback
*
* DESCRIPTION: This functions registers the OSAL Callback function to the
*              trace device
*
* PARAMETER:    TR_tenTraceChan enTraceChannel,   Channel for register
*               OSAL_tpfCallback pfCallback       Callback to register
*
* RETURNVALUE: none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void LLD_vRegTraceCallback( TR_tenTraceChan enTraceChannel, OSAL_tpfCallback pfCallback )
{
#ifdef USE_TRACE_IF
   if(chan_acess_bRegChan == NULL)
   {
      LoadTrace();
   }
   if(chan_acess_bRegChan != NULL)
   {
      chan_acess_bRegChan(enTraceChannel,pfCallback);
   }
#else
 ((void)enTraceChannel); 
 ((void)pfCallback);
#endif
}



void LLD_vUnRegTraceCallback( TR_tenTraceChan enTraceChannel)
{
#ifdef USE_TRACE_IF
  if(chan_acess_bUnRegChan == NULL)
  {
     LoadTrace();
  }
  if(chan_acess_bUnRegChan != NULL)
  {
     chan_acess_bUnRegChan(enTraceChannel, NULL);
  }
#else
  ((void)enTraceChannel); 
#endif
}



void TraceString(const char* cBuffer,...)
{
  tU32 u32Len;
  char cBuf[OSAL_C_U32_MAX_PATHLENGTH];
  OSAL_tVarArgList argList; /*lint -e530 */
  cBuf[0] = (char)OSAL_STRING_OUT;
  OSAL_VarArgStart(argList, cBuffer);  //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  
  u32Len = (tU32)OSALUTIL_s32SaveVarNPrintFormat(&cBuf[1], OSAL_C_U32_MAX_PATHLENGTH-1, cBuffer, argList); //lint !e530
  if(u32Len > OSAL_C_U32_MAX_PATHLENGTH-1)u32Len = OSAL_C_U32_MAX_PATHLENGTH-1;
  OSAL_VarArgEnd(argList);
  LLD_vTrace(TR_COMP_OSALCORE, TR_LEVEL_FATAL,cBuf,u32Len+1);
}

void TraceStringLevel(tU32 u32Level, const char* cBuffer,...)
{
  tU32 u32Len;
  char cBuf[OSAL_C_U32_MAX_PATHLENGTH];
  OSAL_tVarArgList argList; /*lint -e530 */
  cBuf[0] = (char)OSAL_STRING_OUT;
  OSAL_VarArgStart(argList, cBuffer);  //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  
  u32Len = (tU32)OSALUTIL_s32SaveVarNPrintFormat(&cBuf[1], OSAL_C_U32_MAX_PATHLENGTH-1, cBuffer, argList); //lint !e530
  if(u32Len > OSAL_C_U32_MAX_PATHLENGTH-1)u32Len = OSAL_C_U32_MAX_PATHLENGTH-1;
  OSAL_VarArgEnd(argList);
  LLD_vTrace(TR_COMP_OSALCORE, u32Level,cBuf,u32Len+1);
}

void TraceStringClassLevel(tU32 u32Class,tU32 u32Level, const char* cBuffer,...)
{
  tU32 u32Len;
  char cBuf[OSAL_C_U32_MAX_PATHLENGTH];
  OSAL_tVarArgList argList; /*lint -e530 */
  cBuf[0] = (char)OSAL_STRING_OUT;
  OSAL_VarArgStart(argList, cBuffer);  //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  
  u32Len = (tU32)OSALUTIL_s32SaveVarNPrintFormat(&cBuf[1], OSAL_C_U32_MAX_PATHLENGTH-1, cBuffer, argList); //lint !e530
  if(u32Len > OSAL_C_U32_MAX_PATHLENGTH-1)u32Len = OSAL_C_U32_MAX_PATHLENGTH-1;
  OSAL_VarArgEnd(argList);
  LLD_vTrace(u32Class, u32Level,cBuf,u32Len);
}


void TraceToConsole(const char* cBuffer,...)
{
   OSAL_tVarArgList argList; /*lint -e530 */
   char cBuf[OSAL_C_U32_MAX_PATHLENGTH];
   tU32 u32Len = 0;
   OSAL_VarArgStart(argList, cBuffer);  //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  
   tS32 s32Return = OSALUTIL_s32SaveVarNPrintFormat(&cBuf[0], OSAL_C_U32_MAX_PATHLENGTH-1, cBuffer, argList); //lint !e530
   u32Len = (s32Return < 0) ? OSAL_C_U32_MAX_PATHLENGTH-1 : (tU32)s32Return;
   OSAL_VarArgEnd(argList);
   if(u32Len > (OSAL_C_U32_MAX_PATHLENGTH-1))u32Len = OSAL_C_U32_MAX_PATHLENGTH-1;
   if(write(1,cBuf,u32Len)){}
}


void vCoreCallbackHandler(void* pvBuffer )
{
   if(pCoreDbgFunc == NULL)
   {
      if(u32LoadSharedObject((void*)pCoreDbgFunc,pOsalData->rLibrary[EN_SHARED_OSAL2].cLibraryNames) != OSAL_E_NOERROR)/*lint !e611 otherwise linker warning */
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
  }

  if(pCoreDbgFunc)
  {
      pCoreDbgFunc(pvBuffer );
  }
}

/*****************************************************************************
*
* FUNCTION:    vRegisterSysCallback
*
* DESCRIPTION: Register OSAL Callback
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
#ifdef USE_DLT_IF
int osal_injection_callback(uint32_t service_id, void *data, uint32_t length)
{
   ((void)service_id);	
   ((void)length);
   vCoreCallbackHandler(data);
   return 0;
}
#endif
void vRegisterSysCallback(void)
{
#ifdef USE_DLT_IF
   if(fpdlt_register_injection_callback)
   {
      fpdlt_register_injection_callback(&OsalCntxt, 0x1000, osal_injection_callback);
   }
   else
#endif
   {
      LLD_vRegTraceCallback((TR_tenTraceChan)TR_TTFIS_LINUX_OSALCORE,(OSAL_tpfCallback)&vCoreCallbackHandler);
   }
}

/*****************************************************************************
*
* FUNCTION:    vUnregisterSysCallback
*
* DESCRIPTION: Unregister OSAL Callback
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void vUnregisterSysCallback (void)
{
}

void vIoCallbackHandler(void* pvBuffer )
{
   if(pIoDbgFunc == NULL)
   {
      if(u32LoadSharedObject((void*)pIoDbgFunc,pOsalData->rLibrary[EN_SHARED_OSAL2].cLibraryNames) != OSAL_E_NOERROR)
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   if(pIoDbgFunc)
   {
      pIoDbgFunc(pvBuffer );
   }
}

/*****************************************************************************
*
* FUNCTION:    vUnregisterOsalIO_Callback
*
* DESCRIPTION: Register OSAL Callback
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
#ifdef USE_DLT_IF
int osalio_injection_callback(uint32_t service_id, void *data, uint32_t length)
{
   ((void)service_id);	
   ((void)length);	
   vIoCallbackHandler(data);
   return 0;
}
#endif
void vRegisterOsalIO_Callback(void)
{
#ifdef USE_DLT_IF
   if(fpdlt_register_injection_callback)
   {
      fpdlt_register_injection_callback(&OsalCntxt, 0x1001, osalio_injection_callback);
   }
   else
#endif
   {
      LLD_vRegTraceCallback((TR_tenTraceChan)TR_TTFIS_LINUX_OSALIO,(OSAL_tpfCallback)vIoCallbackHandler);
   }
}

/*****************************************************************************
*
* FUNCTION:    vUnregisterOsalIO_Callback
*
* DESCRIPTION: Unregister OSAL Callback
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.05  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void vUnregisterOsalIO_Callback(void)
{
 /*   LaunchChannel  oTraceChannel;

    oTraceChannel.enTraceChannel = TR_TTFIS_OSALIO;
    oTraceChannel.p_Callback = (OSAL_tpfCallback)vOsalIoCallbackHandler;*/
}

static long tickspersec = sysconf(_SC_CLK_TCK);

tU32 u32ScanPidTaskTidStat(char* Path,tBool bErrMemEntry,tS32 s32Idx)
{
#ifdef QNX
   char filepath[64];
   char options[64];
   char command[128];
   if((s32Idx == 0) || (s32Idx == -1))
   {
	   s32Idx = (tS32)OSAL_ProcessWhoAmI();
   }  
   if(Path == NULL)
   {
     snprintf(options,64,"threads times sched");
   }
   else
   {
     snprintf(options,64,"%s",Path);
   }	   
   snprintf(filepath,64,"%s/pidinfo.txt",VOLATILE_DIR);
   snprintf(command,128,"pidin -p %d %s > %s",s32Idx,options, filepath);
   system(command);
   vReadFile(filepath,bErrMemEntry);
   remove(filepath);
   return 0;
#else	
   FILE* fdstr = NULL;
   char Buffer[246]; //needed for /proc/<PID>/tsak/<TID>/wchan & /proc/<PID>/tsak/<TID>/stat
   tU32 u32Ret = 0;
   char wchan[100];
   int fd = -1;

   /*Coverity Fix:CID 17361*/
   OSAL_szStringNCopy(Buffer,Path,sizeof(Buffer)-1);
   Buffer[245]='\0';
   OSAL_szStringConcat(Buffer,"/wchan");
   if((fd = open(Buffer, O_RDONLY,0)) != -1)
   {
       memset(wchan,0,100);
       if(read(fd,wchan,100) == -1){}
       close(fd);
   }
   /*Coverity Fix:CID 17361*/
   OSAL_szStringNCopy(Buffer,Path,sizeof(Buffer)-1);
   Buffer[245]='\0';
   OSAL_szStringConcat(Buffer,"/stat");

   fdstr = fopen(Buffer, "r");
   if(fdstr)
   {
     int Dummy;
     long int lDummy;
     unsigned uDummy;
     long unsigned luDummy;
     long long unsigned lluDummy;
     int TID   = 0;
     int PPID  = 0;
     char cTskName[PATH_MAX] = {0};
     char cStatus = '?';
     char* pPolicy = (char*)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
     long unsigned UTim = 0;
     long unsigned KTim = 0;
     long long unsigned StartTim = 0;
     long int cuTim = 0;
     long int csTim = 0;
     long int Prio  = 0;
     long int Nice  = 0;
     unsigned rtPrio = 0;
     unsigned uPolicy= 0;
     if(fscanf(fdstr,"%d ",&TID)){}     // 1. TID
     if(fscanf(fdstr,"%s ",&cTskName[0])){} // 2. Taskname
     if(fscanf(fdstr,"%c ",&cStatus)){}  // 3. %c ne character from the string "RSDZTW" where R is running, S is sleeping in an interruptible wait, D is waiting 
                                 //   in uninterruptible disk sleep, Z is zombie, T is traced or stopped (on a signal), and W is paging.*/
     if(fscanf(fdstr,"%d ",&PPID )){}    // 4. %d The PID of the parent.
     if(fscanf(fdstr,"%d ",&Dummy)){}    // 5.
     if(fscanf(fdstr,"%d ",&Dummy)){}    // 6.
     if(fscanf(fdstr,"%d ",&Dummy)){}    // 7.
     if(fscanf(fdstr,"%d ",&Dummy)){}    // 8.
     if(fscanf(fdstr,"%u ",&uDummy)){}   // 9.
     if(fscanf(fdstr,"%lu ",&luDummy)){} // 10.
     if(fscanf(fdstr,"%lu ",&luDummy)){} // 11.
     if(fscanf(fdstr,"%lu ",&luDummy)){} // 12.
     if(fscanf(fdstr,"%lu ",&luDummy)){} // 13.
     if(fscanf(fdstr,"%lu ",&UTim)){}    // 14 %ld Amount of time that this process's waited-for children have been scheduled in user mode, 
                                  //    measured in clock ticks (divide by sysconf(_SC_CLK_TCK)).(See also times(2).) This includes guest time, cguest_time 
     if(fscanf(fdstr,"%lu ",&KTim)){}    // 15. %ld Amount of time that this process's waited-for children have been scheduled in kernel mode,
                                   //     measured in clock ticks (divide by sysconf(_SC_CLK_TCK)). */
     if(fscanf(fdstr,"%ld ",&cuTim)){}   // 16.
     if(fscanf(fdstr,"%ld ",&csTim)){}   // 17.
     if(fscanf(fdstr,"%ld ",&Prio)){}    /* 18.priority  this is the negated scheduling priority, minus one; that is, a number in the range -2 to -100,
                                      corresponding to real-time priorities 1 to 99. For processes running under a non-real-time scheduling policy, this is 
                                      the raw nice value (setpriority(2)) as represented in the kernel. The kernel stores nice values as numbers in the range
                                      0 (high) to 39 (low), corresponding to the user-visible nice range of -20 to 19. */
     if(fscanf(fdstr,"%ld ",&Nice)){}    // 19. The nice value (see setpriority(2)), a value in the range 19 (low priority) to -20 (high priority). t_priority %u (since Linux 2.5.19; was %lu before Linux 2.6.22) 
     if(fscanf(fdstr,"%ld ",&lDummy)){}  // 20.
     if(fscanf(fdstr,"%ld ",&lDummy)){}    // 21.
     if(fscanf(fdstr,"%llu ",&StartTim)){}  /* 22.The time the process started after system boot. In kernels before Linux 2.6, this value was expressed in jiffies. 
                                            Since Linux 2.6, the value is expressed in clock ticks (divide by sysconf(_SC_CLK_TCK)).*/

     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 23.
     if(fscanf(fdstr,"%ld ",&lDummy)){}    // 24.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 25.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 26.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 27.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 28.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 29.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 30.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 31.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 32.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 33.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 34.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 35.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 36.
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 37.
     if(fscanf(fdstr,"%lu ",&luDummy)){}     // 38.
     if(fscanf(fdstr,"%d ",&Dummy)){}    // 39.
     if(fscanf(fdstr,"%u ",&rtPrio)){}   /* 40. Real-time scheduling priority, a number in the range 1 to 99 for processes scheduled under a real-time policy, or 0,
                                          for non-real-time processes */
     if(fscanf(fdstr,"%u ",&uPolicy)){}     // 41
     if(fscanf(fdstr,"%llu ",&lluDummy)){}   // 42
     if(fscanf(fdstr,"%lu ",&luDummy)){}    // 43
     if(fscanf(fdstr,"%ld ",&lDummy)){}     // 44
     switch(uPolicy)
     {
        case SCHED_OTHER:
             pPolicy = (char*)"SCHED_OTHER";/*lint !e1773 */  /*otherwise linker warning */
            break;
        /*	  case SCHED_BATCH:
             pPolicy = "SCHED_BATCH";
            break;
        case SCHED_IDLE:
             pPolicy = "SCHED_IDLE";
            break;*/
        case SCHED_FIFO:
             pPolicy = (char*)"SCHED_FIFO";/*lint !e1773 */  /*otherwise linker warning */
            break;
        case SCHED_RR:
             pPolicy = (char*)"SCHED_RR";/*lint !e1773 */  /*otherwise linker warning */
            break;
        default:
            break;
     }
     if(bErrMemEntry == FALSE)
     {
        TraceString("PID:%s PPID:%d TID:%d %s Status:%c Prio:%ld RTPrio:%u Policy:%s Nice:%ld stopped at:%s UserTime:%.10f sec  KernelTime:%.10f sec Started after %.10f ",
                    Path,PPID,TID,&cTskName[0], cStatus,Prio,rtPrio,pPolicy,Nice,wchan,(((double)UTim) / (double)tickspersec),(((double)KTim) / (double)tickspersec),(((double)StartTim) / (double)tickspersec));
     }
     else
     {
        vWritePrintfErrmem("PID:%s PPID:%d TID:%d %s Status:%c Prio:%ld RTPrio:%u Policy:%s Nice:%ld stopped at:%s UserTime:%f sec  KernelTime:%f sec Started after %f \n",
                           Path,PPID,TID,&cTskName[0], cStatus,Prio,rtPrio,pPolicy,Nice,wchan, (((double)UTim) / (double)tickspersec),(((double)KTim) / (double)tickspersec),(((double)StartTim) / (double)tickspersec));
     }
     if(s32Idx != -1)
     {
         switch(s32Idx)
         {
         case 1: 
             u32Ret = TID;
             break;
         case 3:
             u32Ret = (tU32)cStatus;
             break;
         case 4:
             u32Ret = PPID;
             break;
         case 18:
     		 u32Ret = (tU32)Prio;
             break;
         case 19:
             u32Ret = (tU32)Nice;
             break;
         case 40:
             u32Ret = rtPrio;
             break;
         default:
             break;
         }
     }
     fclose(fdstr);
   }
   return u32Ret;
#endif
}

void vMemInfoForProc(char* pName,tBool bErrMemEntry)
{
    char szPrcPath[50];
    FILE* fdstr = NULL;
    long int VmSize = 0;
    long int VmRSS  = 0;
    long int Share  = 0;
    long int Text    = 0;
    long int Lib     = 0;
    long int data    = 0;

    snprintf(szPrcPath,50,"/proc/%s",pName);
    if(chdir(szPrcPath) != -1)
    {
       fdstr = fopen("statm", "r");
       if(fdstr)
       {
          int PageSize = getpagesize();
          if(fscanf(fdstr,"%lu ",&VmSize)){}     // 1. TID
          if(fscanf(fdstr,"%lu ",&VmRSS)){}       // 1. TID
          if(fscanf(fdstr,"%lu ",&Share)){}       // 1. TID
          if(fscanf(fdstr,"%lu ",&Text)){}       // 1. TID
          if(fscanf(fdstr,"%lu ",&Lib)){}       // 1. TID
          if(fscanf(fdstr,"%lu ",&data)){}       // 1. TID
          fclose(fdstr);
    
/*                     size         (1) total program size
                                      (same as VmSize in /proc/[pid]/status)
                        resident    (2) resident set size
                                      (same as VmRSS in /proc/[pid]/status)
                        share        (3) shared pages (i.e., backed by a file)
                        text         (4) text (code)
                        lib          (5) library (unused in Linux 2.6)
                        data         (6) data + stack*/
          if(VmSize)
          {
             if(bErrMemEntry == FALSE)
             {
                TraceString("PID:%s VmSize:%d kB VmRSS:%d kB SharedMem:%d kB Code:%d kB Lib:%d kB Stack&Data:%d kB",
                            pName,VmSize*PageSize/1024,VmRSS*PageSize/1024,Share*PageSize/1024, 
                            Text*PageSize/1024,Lib*PageSize/1024,data*PageSize/1024);
             }
             else
             {
                vWritePrintfErrmem("PID:%s VmSize:%d kB VmRSS:%d kB SharedMem:%d kB Code:%d kB Lib:%d kB Stack&Data:%d kB \n",
                                   pName,VmSize*PageSize/1024,VmRSS*PageSize/1024,Share*PageSize/1024, 
                                   Text*PageSize/1024,Lib*PageSize/1024,data*PageSize/1024);
             }	
             VmSize = 0;
          }
       }
    }
}

void vWriteMemStatToErrmem(char *pcReadBuf,unsigned int uSize, int Err)
{
   if(pcReadBuf)
   {
      struct dirent*    pDirEntp=NULL;
      DIR* dirproc =0;
      int fd;
      ssize_t size;
      char Path[64];
      /* get global memory status*/
       vReadFile("/proc/meminfo",FALSE);
	  
      if((fd = open("/proc/meminfo",O_RDONLY,0)) != -1)
      {
         if((size = read( fd,pcReadBuf,uSize)) > 0)
         {
            *(pcReadBuf+size)= '\0';
            if(Err)
            {
               vWritePrintfErrmem("%s",pcReadBuf);
            }
            else
            {
                TraceString("%s",pcReadBuf);				
            }
          }
          close(fd);
      }
      /* iterate over processes to get meminfo*/
      dirproc = opendir("/proc");
      if(dirproc)
      {
         pDirEntp = readdir((DIR*)dirproc);
         while(pDirEntp)
         {
            if((*pDirEntp->d_name>= 0x30 /* 0 */)&&(*pDirEntp->d_name < 0x39 /* 9 */ ))
            {
               /* write process name */
               snprintf(Path,64,"/proc/%s/comm",pDirEntp->d_name);
               if((fd = open(Path,O_RDONLY,0)) != -1)
               {
                  if((size = read( fd,pcReadBuf,uSize)) > 0)
                  {
                     *(pcReadBuf+size)= '\0';
                     if((strncmp(pcReadBuf,"uname",5))&&(strncmp(pcReadBuf,"sh",2)))
                     {
                        if(Err)
                        {
                           vWritePrintfErrmem("%s",pcReadBuf);
                        }
                        else
                        {
                           TraceString("%s",pcReadBuf);				
                        }
                        /* write memory info */
                        vMemInfoForProc(pDirEntp->d_name, (tBool)Err);      
                     }
                  }
                  close(fd);
               }
            }
            pDirEntp = readdir(dirproc);
         }//while(pDirEntp)
         closedir(dirproc);
      }//if(dirproc)
   }
}

/*****************************************************************************
*
* FUNCTION:     vReadFile
*
* DESCRIPTION: read file content
*
* PARAMETER:    void*  file name as string
*
* RETURNVALUE: none
*
* HISTORY:
* Date        |    Modification                                 | Authors
* 03.07.06  | Initial revision                              | MRK2HI
* --.--.--  | ----------------                              | -----
*
*****************************************************************************/
void vReadFile(const void* pvBuffer,tBool bWriteErrmem)
{
#define LINE_LENGTH  250
  tS32 s32Val;
  tS32 fd;
  int size,i;
  int offset = 0;
  char cTraceBuf[LINE_LENGTH+5];
  tU16 u16OpenMode = 0;//O_TEXT;
  unsigned char* pBuffer = NULL;
  tBool bIgnoreSize= FALSE;
  memset(&cTraceBuf[0],0,LINE_LENGTH+5);

    if((fd = open((const char*)pvBuffer, O_RDONLY,u16OpenMode)) != -1)
    {
        size = s32GetFileSize(fd);
        if(size == OSAL_ERROR)
        {
             size = 100000;
             bIgnoreSize = TRUE;
        }
        if((pBuffer = (unsigned char*)OSAL_pvMemoryAllocate((tU32)size)) != 0)
        {
          memset(pBuffer,0,(tU32)size);
          if((s32Val = (tS32)lseek(fd,0,SEEK_SET)) < OSAL_OK)
          {
             TraceString((const char*)"Seek File Error");
             s32Val = -1;
          }
          else
          {
             if((s32Val = (tS32)read( fd,pBuffer,(tU32)size)) != size)
             {  
                 if(bIgnoreSize != TRUE)
                 {
                     TraceString((const char*)"Read File Error");
                     s32Val = -1;
                 }
             }
        }
          
        if(s32Val >= 0)
        {
            cTraceBuf[0] = (char)OSAL_STRING_OUT;
            while(size)
            {
                /* search next line if available*/
                for (i = 0; i<LINE_LENGTH ;i++)
                {
                  if((*(pBuffer+offset+i) == /*EOF*/0xff) || (*(pBuffer+offset+i) == 0x0a/*EOL*/)) 
                  {
                     i++;
                     break;
                  }
                  if(offset+i > s32Val)break;
                }
                memcpy(&cTraceBuf[1],pBuffer+offset,(tU32)i);
                LLD_vTrace(TR_COMP_OSALCORE, TR_LEVEL_FATAL,&cTraceBuf[0],(tU32)i+1);
                if(bWriteErrmem)
                {
                    vWriteToErrMem((tS32)TR_COMP_OSALCORE,(char*)&cTraceBuf[0],i+1,OSAL_STRING_OUT);
                }
                offset += (i);
                memset(&cTraceBuf[1],0,i);
                if(offset >= s32Val)break;
          }// end while
        }
        else
        {
            TraceString((const char*)"Read File Error , Try it again");
        }
        OSAL_vMemoryFree(pBuffer);
      }
      else
      {
         TraceString((const char*)"Allocate Memory Error");
      }
      close(fd);
    }
    else
    {
     TraceString((const char*)"Open File Error");
    }
}


void vGetSystemCallInfo(tCString Command, tBool bErrMemEntry)
{
    char Buffer[256];
    snprintf(Buffer,256,"%s > %s",Command,OSAL_LOG_FILE);

    if(system(Buffer) != -1)
    {
       if(bErrMemEntry)
       {
          vReadFile(OSAL_LOG_FILE,TRUE);
       }
       else
       {
          vReadFile(OSAL_LOG_FILE,FALSE);
       }
       if(remove(OSAL_LOG_FILE) == -1)
       {
           TraceString("system call remove %s failed",OSAL_LOG_FILE);
       }
    }
    else
    {
       TraceString("%s failed",Buffer);
    }
}




#ifdef __cplusplus
}
#endif
/************************************************************************
|end of file
|-----------------------------------------------------------------------*/
