/******************************************************************************
| FILE:         osalmempool.cpp
| PROJECT:      ADIT GEN1 
| SW-COMPONENT: OSAL
|------------------------------------------------------------------------------
| DESCRIPTION:  This is the implementation file for the Memory Pool-Functions.
|
|------------------------------------------------------------------------------
| COPYRIGHT:    (c) 2009 Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 03.10.05  | Initial revision           | MRK2HI
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/

/************************************************************************ 
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "Linux_osal.h"

#include "ostrace.h"

#ifdef __cplusplus
extern "C" {
#endif 

/************************************************************************
|defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
#define POOLMAGIC       0x504F4F4C504F4F4C /* POOL Magic */
#define ALLOCMAGIC      0x414C4C4F414C4C4F /* Malloc Magic */
#else
#define POOLMAGIC       0x504F4F4C /* POOL Magic */
#define ALLOCMAGIC      0x414C4C4F /* Malloc Magic */
#endif

#define GETBLOCK_MPF_FAILED     0xb2 /* u32ErrorCode , OSAL_ThreadWhoAmI */
#define RELBLOCK_MPF_FAILED     0xb3 /* u32ErrorCode , OSAL_ThreadWhoAmI */



#define MPF_INFO                0xb6

/************************************************************************
|typedefs (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/
static tBool bPrintOwner = FALSE;
/************************************************************************
| function definition (scope: global)
|-----------------------------------------------------------------------*/
#ifndef USE_SEM_FOR_POOL
int CreateMutex(pthread_mutexattr_t* mutexattr,pthread_mutex_t* hMutex,uint32_t u32Opt);
#endif

/******************************************************************************
 * FUNCTION:      vTraceOsalMpf
 *
 * DESCRIPTION:   gets memory pool index for OSAL mempool
 *
 * PARAMETER:     TR_tenTraceLevel  trace level
 *                UB*               pointer to trace buffer
 *                int               size of trace buffer
 *               
 * RETURNVALUE:   none
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.11.09  |   Initial revision                     | MRK2HI
 *****************************************************************************/
void vTraceOsalMpf(TR_tenTraceLevel enLevel, tU8* buf, int size)
{
   if(LLD_bIsTraceActive(OSAL_C_TR_CLASS_SYS_MEMPOOL,enLevel) == TRUE)
   {
       LLD_vTrace(OSAL_C_TR_CLASS_SYS_MEMPOOL,enLevel,buf,size); /*lint -e1773*/
   }
}

uintptr_t GetMemPoolOffset(trHandleMpf* pHandle, uintptr_t* pU32Mem)
{
    return (uintptr_t)pU32Mem - (uintptr_t)pHandle->u32memstart;
}


/******************************************************************************
 * FUNCTION:      OSAL_u32MemPoolFixSizeCreate
 *
 * DESCRIPTION:   generates memory pool for OSAL
 *
 * PARAMETER:     tCString      Pool Name
 *                tU32          u32MpfBlockSize,
 *                tU32          u32MpfBlockCnt,
 *                trHandleMpf*  pointer to Pool handle structure
 *                tBool         Flag i f internal lock is needed
 *                
 *               
 * RETURNVALUE:   OSAL_ERROR or OSAL_OK
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.04.13  |   Initial revision                     | MRK2HI
 ***************** ************************************************************/
tS32 OSAL_u32MemPoolFixSizeCreate(tCString coszName, 
                                  tU32 u32MpfBlockSize,
                                  tU32 u32MpfBlockCnt,
                                  trHandleMpf* pHandle,
                                  tBool bLock,
                                  tBool bMalloc)
{
  tU32 blfsz;
  trOSAL_MPF* pPool = NULL;
  char cBuffer[32];
  tU32 i;
 
  /* determine pool element size */
  blfsz = u32MpfBlockSize % (tU32)sizeof(uintptr_t);
  if(blfsz == 0)
  {
     blfsz = (tU32)u32MpfBlockSize + (2*(tU32)sizeof(uintptr_t))/*size info + magic field*/;
  }
  else
  {
     blfsz = (tU32)u32MpfBlockSize + ((tU32)sizeof(uintptr_t) - blfsz) + (2*(tU32)sizeof(uintptr_t))/*size info + magic field*/;
  }
  /* extend each element with place for the owner PID */
  blfsz += (tU32)sizeof(uintptr_t); /* PID info */

  if(bLock)
  {
     /* use for create and open always a lock */
     snprintf(cBuffer,32,"/%s",coszName);
#ifdef USE_SEM_FOR_POOL
     for(i=0;i<SEM_MEM_RES;i++)
     {
        if(pOsalData->rSemMem[i].bUsed == FALSE)
        {
            pOsalData->rSemMem[i].bUsed = TRUE;
            pHandle->Sem = (sem_t*)&pOsalData->rSemMem[i].lock;
            break;
        }
     }
     if(i >= SEM_MEM_RES) 
     { 
       abort();
     }
     if(0 != sem_init(pHandle->Sem, 1,0))
     {
        TraceString("OSAL_MP sem_init failed errno:%d",errno);
        FATAL_M_ASSERT_ALWAYS();
     }
#endif	
  }
  if((pHandle->hShMem = (tS32)OSAL_SharedMemoryCreate(coszName,  // return value is an array index
                                                OSAL_EN_READWRITE,
                                                (u32MpfBlockCnt * blfsz) + (tU32)sizeof(trOSAL_MPF))) == OSAL_ERROR)
  {
    TraceString("OSAL_MP OSAL_SharedMemoryCreate failed errno:%d",errno);
    return OSAL_ERROR;
  }
  if((pPool = (trOSAL_MPF*)OSAL_pvSharedMemoryMap(pHandle->hShMem,
                                                  OSAL_EN_READWRITE,
                                                  (u32MpfBlockCnt * blfsz) + (tU32)sizeof(trOSAL_MPF),
                                                  0)) == NULL)
  {
    TraceString("OSAL_MP OSAL_pvSharedMemoryMap failed errno:%d",errno);
    return OSAL_ERROR;
  }
  if(bLock)
  {
#ifdef USE_SEM_FOR_POOL
     pPool->Sem         = i;
#else
     pthread_mutexattr_t mutexattr;
     if(CreateMutex(&mutexattr,&pPool->Mtx,0) == 0)
     {
        pthread_mutex_lock(&pPool->Mtx);
        pHandle->Mtx = &pPool->Mtx;
     }
     else
     {  
        FATAL_M_ASSERT_ALWAYS();
     }
#endif
  }
  /* Fill handle */
  pHandle->pMem            = (void*)pPool;
  pHandle->u32memstart     = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF);
  pHandle->u32memend       = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF) + (blfsz * u32MpfBlockCnt);
  pHandle->Magic           = HANDLE_MAGIC;

  pPool->bLock     = bLock;
  pPool->bMalloc   = bMalloc;

  /* fill internal structure */
  strncpy(pPool->szName,coszName,31);
  pPool->blfsz       = blfsz;
  pPool->mpfcnt      = u32MpfBlockCnt;
  pPool->freecnt     = pPool->mpfcnt;
  pPool->u32OpenCnt  = 1;
  pPool->u32MaxCount = 0;
  pPool->u32CurCount = 0;
  pPool->u32ErrCnt   = 0;
  pPool->u32nextIdx  = 0;
  /* prepare memory */
  uintptr_t *pMem;
  for(i = 0;i<pPool->mpfcnt;i++)
  {
     pMem = (uintptr_t*)((uintptr_t)pHandle->u32memstart + (i * pPool->blfsz));
     /* initialize the PID place in the memory*/
     *pMem = 0;
     pMem++;
     /* mark each element as free */
     *pMem = FREEMARKER;
     /* set magic for pool memory at the end of each lement */
     pMem = (uintptr_t*)((uintptr_t)pHandle->u32memstart + (((i+1) * pPool->blfsz) - (tU32)sizeof(uintptr_t)));
     *pMem = POOLMAGIC;
  }

  if(bLock)
  {
#ifdef USE_SEM_FOR_POOL
     sem_post(pHandle->Sem);
#else
     pthread_mutex_unlock(pHandle->Mtx);
#endif
  }
  return OSAL_OK;
}

tU32 u32CalculatePoolMemSize(tU32 u32MpfBlockSize,
                             tU32 u32MpfBlockCnt)
{
  tU32 	blfsz;
  /* determine pool element size */
  blfsz = u32MpfBlockSize % (tU32)sizeof(uintptr_t);
  if(blfsz == 0)
  {
     blfsz = (tU32)u32MpfBlockSize + (2*(tU32)sizeof(uintptr_t))/*size info + magic field*/;
  }
  else
  {
     blfsz = (tU32)u32MpfBlockSize + ((tU32)sizeof(uintptr_t) - blfsz) + (2*(tU32)sizeof(uintptr_t))/*size info + magic field*/;
  }
  /* extend each element with place for the owner PID */
  blfsz += (tU32)sizeof(uintptr_t); /* PID info */
  
  return (u32MpfBlockCnt*blfsz) + (tU32)sizeof(trOSAL_MPF);
  
}

tS32 OSAL_u32MemPoolFixSizeCreateOnMem(tCString coszName, 
                                  tU32 u32MpfBlockSize,
                                  trHandleMpf* pHandle,
                                  tBool bLock,
                                  void* pPoolMem,
								  tU32 u32Size)
{
  tU32 blfsz;
  trOSAL_MPF* pPool = (trOSAL_MPF*)pPoolMem;
  char cBuffer[32];
  tU32 u32MpfBlockCnt;
  tU32 i = 0;
   
  u32Size -= (tU32)sizeof(trOSAL_MPF);
  /* determine pool element size */
  blfsz = u32MpfBlockSize % (tU32)sizeof(uintptr_t);
  if(blfsz == 0)
  {
     blfsz = (tU32)u32MpfBlockSize + (2*(tU32)sizeof(uintptr_t))/*size info + magic field*/;
  }
  else
  {
     blfsz = (tU32)u32MpfBlockSize + ((tU32)sizeof(uintptr_t) - blfsz) + (2*(tU32)sizeof(uintptr_t))/*size info + magic field*/;
  }
  
  /* extend each element with place for the owner PID */
  blfsz += (tU32)sizeof(uintptr_t); /* PID info */

  u32MpfBlockCnt = u32Size/blfsz;
  if(bLock)
  {
     snprintf(cBuffer,32,"/%s",coszName);
#ifdef USE_SEM_FOR_POOL
     for(i=0;i<SEM_MEM_RES;i++)
     {
        if(pOsalData->rSemMem[i].bUsed == FALSE)
        {
            pOsalData->rSemMem[i].bUsed = TRUE;
            pHandle->Sem = (sem_t*)&pOsalData->rSemMem[i].lock;
            break;
        }
     }
     if(i >= SEM_MEM_RES) 
     { 
       abort();
     }
     if(0 != sem_init(pHandle->Sem,1,0))
     {
        TraceString("OSAL_MP sem_init failed errno:%d",errno);
        FATAL_M_ASSERT_ALWAYS();
     }
#else
     pthread_mutexattr_t mutexattr;
     if(CreateMutex(&mutexattr,&pPool->Mtx,0) == 0)
     {
        pthread_mutex_lock(&pPool->Mtx);
        pHandle->Mtx = &pPool->Mtx;
     }
#endif
  }
  pHandle->hShMem = 0;
  /* Fill handle */
  pHandle->pMem            = (void*)pPool;
  pHandle->u32memstart     = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF);
  pHandle->u32memend       = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF) + (blfsz * u32MpfBlockCnt);
  pHandle->Magic           = HANDLE_MAGIC;

  pPool->bMalloc   = TRUE;
  pPool->bLock   = bLock;

  /* fill internal structure */
  strncpy(pPool->szName,coszName,31);
  pPool->blfsz       = blfsz;
  pPool->mpfcnt      = u32MpfBlockCnt;
  pPool->freecnt     = pPool->mpfcnt;
  pPool->u32OpenCnt  = 1;
  pPool->u32MaxCount = 0;
  pPool->u32CurCount = 0;
  pPool->u32ErrCnt   = 0;
  pPool->u32nextIdx  = 0;
#ifdef USE_SEM_FOR_POOL
  pPool->Sem         = i;
#endif
  /* prepare memory */
  uintptr_t *pMem;
  for(i = 0;i<pPool->mpfcnt;i++)
  {
     pMem = (uintptr_t*)((uintptr_t)pHandle->u32memstart + (i * pPool->blfsz));
     /* initialize the PID place in the memory*/
     *pMem = 0;
     pMem++;
     /* mark each element as free */
     *pMem = FREEMARKER;
     /* set magic for pool memory at the end of each lement */
     pMem = (uintptr_t*)((uintptr_t)pHandle->u32memstart + (((i+1) * pPool->blfsz) - (tU32)sizeof(uintptr_t)));
     *pMem = POOLMAGIC;
  }
  
  if(bLock)
  {
#ifdef USE_SEM_FOR_POOL
     sem_post(pHandle->Sem);
#else
     pthread_mutex_unlock(pHandle->Mtx);
#endif
  }
  return OSAL_OK;
}


tS32 OSAL_u32MemPoolFixSizeOpen(tCString coszName, 
                                trHandleMpf* pHandle)
{
  char cBuffer[32];
  trOSAL_MPF* pPool = NULL;

  /* use for create and open always a lock */
  snprintf(cBuffer,32,"/%s",coszName);
  if((pHandle->hShMem = (tS32)OSAL_SharedMemoryOpen(coszName,// return value is an arry index
                                                    OSAL_EN_READWRITE)) == OSAL_ERROR)
  {
      TraceString("OSAL_MP OSAL_SharedMemoryOpen failed errno:%d",errno);
      return OSAL_ERROR;
  }
  if((pPool = (trOSAL_MPF*)OSAL_pvSharedMemoryMap(pHandle->hShMem,
                                                  OSAL_EN_READWRITE,
                                                  (tU32)sizeof(trOSAL_MPF),
                                                  0)) == NULL)
  {
     TraceString("OSAL_MP OSAL_pvSharedMemoryMap failed errno:%d",errno);
     return OSAL_ERROR;
  }
 
  if(pPool->bLock)
  {
#ifdef USE_SEM_FOR_POOL
     if(pOsalData->rSemMem[pPool->Sem].bUsed)
     {	   
        pHandle->Sem = (sem_t*)&pOsalData->rSemMem[pPool->Sem].lock;
     }
     else
     {
        FATAL_M_ASSERT_ALWAYS();
        return OSAL_ERROR;
     }
     if(u32SemWait(pHandle->Sem) != OSAL_E_NOERROR)
     {
        return OSAL_ERROR;
     }
#else	
     pthread_mutex_lock(&pPool->Mtx); 
     pHandle->Mtx = &pPool->Mtx;
#endif
  }

  if((pPool = (trOSAL_MPF*)OSAL_pvSharedMemoryMap(pHandle->hShMem,
                                                  OSAL_EN_READWRITE,
                                                  (pPool->mpfcnt * pPool->blfsz) + (tU32)sizeof(trOSAL_MPF),
                                                  0)) == NULL)
  {
     TraceString("OSAL_MP OSAL_pvSharedMemoryMap 2 failed errno:%d",errno);
     return OSAL_ERROR;
  }

  /* Fill handle */
  pHandle->pMem               = (void*)pPool;
  pHandle->u32memstart        = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF);
  pHandle->u32memend          = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF) + (pPool->blfsz * pPool->mpfcnt);
  pHandle->Magic              = HANDLE_MAGIC;

  pPool->u32OpenCnt++;

  if(pPool->bLock)
  {
#ifdef USE_SEM_FOR_POOL
     sem_post(pHandle->Sem);
#else
     pthread_mutex_unlock(pHandle->Mtx); 
#endif
  }
  return OSAL_OK;
}

tS32 OSAL_u32MemPoolFixSizeOpenOnMem(tCString coszName, 
                                  trHandleMpf* pHandle,
                                  void* pMem)
{
  char cBuffer[32];
  trOSAL_MPF* pPool = (trOSAL_MPF*)pMem;

  if(pPool->bLock)
  {
     /* use for create and open always a lock */
     snprintf(cBuffer,32,"/%s",coszName);
#ifdef USE_SEM_FOR_POOL
     if(pOsalData->rSemMem[pPool->Sem].bUsed)
     {	   
        pHandle->Sem  = &pOsalData->rSemMem[pPool->Sem].lock;
     }
     else
     {
        pHandle->Sem  = SEM_FAILED;
     }
     if (pHandle->Sem  == SEM_FAILED)
     {
        TraceString("OSAL_MP pHandle->Sem  == SEM_FAILED");
        FATAL_M_ASSERT_ALWAYS();
        return OSAL_ERROR;
     }
     if(u32SemWait(pHandle->Sem ) != OSAL_E_NOERROR)
#else	
     pHandle->Mtx = &pPool->Mtx;
     if(pthread_mutex_lock(pHandle->Mtx) != 0)
#endif
     {
        TraceString("OSAL_MP take a lock failed errno:%d",errno);
        return OSAL_ERROR;
     }
  }

  pHandle->hShMem = 0;

  /* Fill handle */
  pHandle->pMem               = (void*)pPool;
  pHandle->u32memstart        = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF);
  pHandle->u32memend          = (uintptr_t)pPool + (tU32)sizeof(trOSAL_MPF) + (pPool->blfsz * pPool->mpfcnt);
  pHandle->Magic              = HANDLE_MAGIC;
  pPool->u32OpenCnt++;

  if(pPool->bLock)
  {
#ifdef USE_SEM_FOR_POOL
     sem_post(pHandle->Sem);
#else
     pthread_mutex_unlock(pHandle->Mtx); 
#endif
  }
  return OSAL_OK;
}


/******************************************************************************
 * FUNCTION:      OSAL_s32MsgPoolFixSizeDelete
 *
 * DESCRIPTION:   delete memory pool for OSAL
 *
 * PARAMETER:     trHandleMpf*     pointer to Pool handle structure 
 *               
 * RETURNVALUE:   none
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.11.09  |   Initial revision                     | MRK2HI
 *****************************************************************************/
tS32 OSAL_s32MemPoolFixSizeDelete (tCString szName)
{
   tS32 s32Ret = OSAL_OK;

   char cBuffer[32];
   /* use for create and open always a lock */
   snprintf(cBuffer,32,"/%s",szName);
   if(OSAL_s32SharedMemoryDelete(szName) == OSAL_ERROR)
   {
      s32Ret = OSAL_ERROR;
   }
   return s32Ret;
}

/******************************************************************************
 * FUNCTION:      OSAL_s32MemPoolFixSizeClose
 *
 * DESCRIPTION:   close memory pool for OSAL process
 *
 * PARAMETER:     trHandleMpf*     pointer to Pool handle structure 
 *               
 * RETURNVALUE:   none
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.11.09  |   Initial revision                     | MRK2HI
 *****************************************************************************/
tS32 OSAL_s32MemPoolFixSizeClose (trHandleMpf* pHandle)
{
   trOSAL_MPF* pPool;
   tS32 s32Ret = OSAL_OK;
   char cBuffer[32] = {0};

   if(pHandle->Magic != HANDLE_MAGIC)
   { 
     TraceString("Mempool Invalid Handle was used !!!");
     return OSAL_ERROR;
   }

   pPool = (trOSAL_MPF*)pHandle->pMem;
   snprintf(cBuffer,32,"/%s",pPool->szName);

   if(pPool->bLock)
   {
#ifdef USE_SEM_FOR_POOL
      if(u32SemWait(pHandle->Sem) != OSAL_E_NOERROR)
#else	
      if(pthread_mutex_lock(pHandle->Mtx) != 0)
#endif
      {
         return OSAL_ERROR;
      }
   }
   if(pPool->u32OpenCnt)pPool->u32OpenCnt--;
  
   if(pPool->bLock)
   {
#ifdef USE_SEM_FOR_POOL
      sem_post(pHandle->Sem);
      sem_close(pHandle->Sem);
#else
      pthread_mutex_unlock(pHandle->Mtx); 
      if(!pPool->u32OpenCnt)pthread_mutex_destroy(pHandle->Mtx);
#endif
   }   

   if(pHandle->hShMem != 0)
   {
      if(OSAL_s32SharedMemoryClose(pHandle->hShMem) == OSAL_ERROR)
      {
         s32Ret = OSAL_ERROR;
      }
   }
   pHandle->Magic = 0;
   return s32Ret;
}

void vReadFileContent(const void* pvBuffer,tBool bWriteErrmem)
{
#define LINE_LENGTH  250
  intptr_t s32Val;
  tS32 fd;
  int i; 
  intptr_t size;
  int offset = 0;
  char cTraceBuf[LINE_LENGTH+5];
  tU16 u16OpenMode = 0;//O_TEXT;
  unsigned char* pBuffer = NULL;
  tBool bIgnoreSize= FALSE;

   if((fd = open((char*)pvBuffer, O_RDONLY,u16OpenMode)) != -1)
   {
      struct stat stat_data;
      if(fstat( fd, &stat_data ) == 0)
      {
         size = stat_data.st_size;
      }
      else
      {
          size = 100000;
          bIgnoreSize = TRUE;
      }
      if((pBuffer = (unsigned char*)malloc((tU32)size)) != 0)
      {
        memset(pBuffer,0,(tU32)size);
        if((s32Val = lseek(fd,0,SEEK_SET)) < OSAL_OK)
        {
//          TraceString((const char*)"Seek File Error");
          s32Val = -1;
        }
        else
        {
          if((s32Val = read( fd,pBuffer,(tU32)size)) != size)
          {  
             if(bIgnoreSize != TRUE)
             {
//                TraceString((const char*)"Read File Error");
                s32Val = -1;
             }
          }
      }
        
      if(s32Val >= 0)
      {
         OSAL_M_INSERT_T8(&cTraceBuf[0],OSAL_STRING_OUT);
         while(size)
         {
            /* search next line if available*/
            for (i = 0; i<LINE_LENGTH ;i++)
            {
              if((*(pBuffer+offset+i) == /*EOF*/0xff) || (*(pBuffer+offset+i) == 0x0a/*EOL*/)) 
              {
                i++;
                break;
              }
              if(offset+i > s32Val)break;
            }
            memcpy(&cTraceBuf[1],pBuffer+offset,(tU32)i);
            LLD_vTrace(TR_COMP_OSALCORE, TR_LEVEL_FATAL,&cTraceBuf[0],(tU32)i);
            if(bWriteErrmem)
            {
               vWriteToErrMem((tS32)TR_COMP_OSALCORE,(char*)&cTraceBuf[0],i,OSAL_STRING_OUT);
            }
            offset += (i);
            if(offset >= s32Val)break;
        }// end while
      }
      
     }
     close(fd);
   }
   if(pBuffer)free(pBuffer);
}

void vPrintAllBlocks(const trHandleMpf* pHandle)
{
  uintptr_t *pMem,Pid,Size,Magic;
  trOSAL_MPF* pPool = (trOSAL_MPF*)(pHandle->u32memstart - (tU32)sizeof(trOSAL_MPF));
  tU32 i;
  if(CreateProcess("ps auxw","/tmp/process.txt") == -1)
  {
	  TraceString("ps auxw > /tmp/process.txt failed");
  }
  for(i = 0;i<pPool->mpfcnt;i++)
  {
     pMem = (uintptr_t*)((uintptr_t)pHandle->u32memstart + (i * pPool->blfsz));
     Pid = *pMem;
     pMem++;
     Size = *pMem;
     Magic = *((uintptr_t*)(pHandle->u32memstart + (((i+1) * pPool->blfsz) - sizeof(uintptr_t))));
     TraceString("Pool %s Idx:%d Pid:%d Size:%d, Magic:%d",pPool->szName,i,(tU32)Pid,(tU32)Size,Magic);
  }
  vReadFileContent("/tmp/process.txt",FALSE);/*lint !e1773 */  /*otherwise linker warning */
  if(remove("/tmp/process.txt") == -1)
  {
     TraceString("Pool %s Cannot remove/tmp/process.txt",pPool->szName);
  }
}


/******************************************************************************
 * FUNCTION:      OSAL_pvMemPoolFixSizeGetBlockOfPool
 *
 * DESCRIPTION:   allocate a memory block from the specified OSAL memory pool
 *
 * PARAMETER:     trHandleMpf*     pointer to Pool handle structure 
 *               
 * RETURNVALUE:   VP Pointer to allocated memory 
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.11.09  |   Initial revision                     | MRK2HI
 *****************************************************************************/
void* OSAL_pvMemPoolFixSizeGetBlockOfPool(const trHandleMpf* pHandle)
{
  void* pBlock = NULL;
  trOSAL_MPF* pPool;
  tU32 u32ErrorCode = OSAL_E_NOERROR;
  tU32 u32AllocSize;
  tU8* pu8Pointer = NULL;
  tBool bMarker = FALSE;
  uintptr_t* pPointer = NULL;
  uintptr_t* pCurrent;
  
  if(pHandle->Magic != HANDLE_MAGIC)
  { 
     TraceString("Mempool Invalid Handle was used !!!");
     return NULL;
  }

  pPool = (trOSAL_MPF*)pHandle->pMem;
  u32AllocSize = pPool->blfsz;
	  
  if(pPool->bLock)
  {
#ifdef USE_SEM_FOR_POOL
    if(u32SemWait(pHandle->Sem) != OSAL_E_NOERROR)return NULL;
#else
    int Ret;	
    if((Ret = pthread_mutex_lock(pHandle->Mtx)) != 0)
    {
#ifndef ANDROID // functionality not supported in ANDROID
       if(Ret == EOWNERDEAD)
       {
          pthread_mutex_consistent(pHandle->Mtx);
       }
       else
#endif
       {
         return NULL;
       }
    }
#endif
  }


  /* get memory from pool */
  if(pPool->freecnt)
  {
     for(;;)
     {  
         pCurrent = (uintptr_t*)(pHandle->u32memstart + (pPool->u32nextIdx * pPool->blfsz));
         if(*((uintptr_t*)((uintptr_t)pCurrent+(tU32)sizeof(uintptr_t))) == FREEMARKER)
         {
            *pCurrent = 0;
            pPool->u32nextIdx++;
            if(pPool->u32nextIdx == pPool->mpfcnt)
            {
                pPool->u32nextIdx = 0;
            }
            pBlock = (void*)pCurrent;
            pPool->freecnt--;
            break;
         }
         else
         {
            pPool->u32nextIdx++;
            if(pPool->u32nextIdx == pPool->mpfcnt)
            {
                pPool->u32nextIdx = 0;
            }
         }
     }

     /* increment message counter */
     pPool->u32CurCount++;
     if(pPool->u32CurCount > pPool->u32MaxCount)pPool->u32MaxCount = pPool->u32CurCount;
  }
  else
  {
     pPool->u32ErrCnt++;
     if(!pPool->bMalloc)
     {
        TraceString("Pool %s is empty no malloc configured Pid:%d!",pPool->szName,(int)OSAL_ProcessWhoAmI());
     }
     else
     {
        bMarker = TRUE;
        if(pPool->u32ErrCnt < 100)TraceString("Pool %s needs malloc in Pid:%d!",pPool->szName,(int)OSAL_ProcessWhoAmI());
        /* no pool available use standard malloc*/
        pBlock = malloc(u32AllocSize);
        if(bPrintOwner ==FALSE)
        {
            bPrintOwner = TRUE;
       //     vPrintAllBlocks(pHandle);
        }
     }
  }

  if(pPool->bLock)
  {
#ifdef USE_SEM_FOR_POOL
     sem_post(pHandle->Sem);
#else	
     pthread_mutex_unlock(pHandle->Mtx); 
#endif
  }


  /* check for usable memory block */
  if(pBlock == NULL)
  {
     TraceString("no memory from Pool %s!",pPool->szName);
     u32ErrorCode = OSAL_E_NOSPACE;
  }
  
  /* check for errors */
  if(u32ErrorCode != OSAL_E_NOERROR)
  {
#ifdef SHORT_TRACE_OUTPUT
     tU8 u8Buffer[50];
     tU32 u32Tid = (tU32)OSAL_ThreadWhoAmI();
     OSAL_M_INSERT_T8(&u8Buffer[0],GETBLOCK_MPF_FAILED);
     OSAL_M_INSERT_T32(&u8Buffer[1],u32ErrorCode);
     OSAL_M_INSERT_T32(&u8Buffer[5],u32Tid);
     OSAL_M_INSERT_T32(&u8Buffer[9],(tU32)pPool->blfsz);
     vTraceOsalMpf(TR_LEVEL_FATAL,u8Buffer,13);
#else //SHORT_TRACE_OUTPUT
      TraceString("TID:%d Release memory block size %d failed with error 0x%x",OSAL_ThreadWhoAmI(),pPool->blfsz,u32ErrorCode);
#endif	//SHORT_TRACE_OUTPUT
     OSAL_vSetErrorCode(u32ErrorCode);
  }
  else
  {
      /* prepare memblock ,generate pointer to memblock*/
      pu8Pointer  = (tU8*)pBlock;
      pPointer = (uintptr_t*)pBlock;
      /* store Pid info at begin of mem block for analysis*/
      *pPointer = (uintptr_t)OSAL_ProcessWhoAmI();/*lint !e613 pu32Pointer is here always !=0 */
      pPointer++;/*lint !e613 pu32Pointer is here always !=0 */
      /* store size info at begin of mem block */
      *pPointer = u32AllocSize;/*lint !e613 pu32Pointer is here always !=0 */
      /*determine return value address */
      pPointer++;/*lint !e613 pu32Pointer is here always !=0 */
      pBlock = (void*)pPointer;

      /* store magic at last 4Byte, alignment is in pool implementation ensured*/
      pPointer = (uintptr_t*)((uintptr_t)pu8Pointer + pPool->blfsz - (tU32)sizeof(uintptr_t));/*lint !e826 */
      if(bMarker)*pPointer = ALLOCMAGIC;
      else       *pPointer = POOLMAGIC;
  }
  return pBlock;
}


/******************************************************************************
 * FUNCTION:      OSAL_s32MemPoolFixSizeRelBlockOfPool
 *
 * DESCRIPTION:   releases a memory block from the OSAL memory pool
 *
 * PARAMETER:     trHandleMpf*     pointer to Pool handle structure 
 *                VP  Pointer to allocated memory 
 *               
 * RETURNVALUE:   OSAL_OK/OSAL_ERROR 
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.11.09  |   Initial revision                     | MRK2HI
 *****************************************************************************/
tS32 OSAL_s32MemPoolFixSizeRelBlockOfPool(const trHandleMpf* pHandle, void* pBlock)
{
  tS32 s32Return    = OSAL_OK;
  tU32 u32ErrorCode = OSAL_E_NOERROR;
  uintptr_t* puPointer = (uintptr_t*)pBlock;
  uintptr_t* puTemp;
  tU8*  pu8Pointer;
  trOSAL_MPF* pPool;

  if(pHandle->Magic != HANDLE_MAGIC)
  { 
     TraceString("Mempool Invalid Handle was used !!!");
     return OSAL_ERROR;
  }

  pPool = (trOSAL_MPF*)pHandle->pMem;
  /* go to block start position to get size info*/
  puPointer--;

  pu8Pointer = (tU8*)puPointer;

  /* check for correkt size */
 if(*puPointer != (uintptr_t)pPool->blfsz)
 {
     /* go to magic info and check magic*/
     puTemp = (uintptr_t*)(pu8Pointer + (uintptr_t)pPool->blfsz - (tU32)sizeof(uintptr_t));/*lint !e826 */
     puTemp--;
     if(*puTemp == ALLOCMAGIC)
     {
         /* go to begin of mem block */
         puPointer--;/*lint !e613 pu32Pointer is here always !=0 */
         /* check for valid pointer and possible mallocs */
         if(pPool->u32ErrCnt > 0)
//       if(((tU32)pu8Pointer < pHandle->u32memstart)&&((tU32)pu8Pointer > pHandle->u32memend))
         {
            /* memory was allocated via standard malloc */
            free((void*)puPointer);/*lint !e424 */
         }
         else
         {
            TraceString("Mempool wrong free adress 0x% Value:%d",puPointer,pPool->u32ErrCnt);
         }
     }
     else
     {
        TraceString("wrong memory for Pool %s!",pPool->szName);
        u32ErrorCode = OSAL_E_INVALIDVALUE;
     }
  }
  else
  {
     /* go to magic info and check magic*/
     puTemp = (uintptr_t*)(pu8Pointer + (uintptr_t)pPool->blfsz - (tU32)sizeof(uintptr_t));/*lint !e826 */
     puTemp--;
     if(*puTemp == POOLMAGIC)
     {
         /* mark block as freed */
        if(pPool->bLock)
        {
#ifdef USE_SEM_FOR_POOL
           if(u32SemWait(pHandle->Sem) != OSAL_E_NOERROR)return OSAL_ERROR;
#else	
	       int Ret;
           if((Ret = pthread_mutex_lock(pHandle->Mtx)) != 0) 
           {
#ifndef ANDROID // functionality not supported in ANDROID
              if(Ret == EOWNERDEAD)
              {
                 pthread_mutex_consistent(pHandle->Mtx);
              }
              else
#endif
              {
                 return OSAL_ERROR;
              }
           }
#endif
        }
        /* release block */
        *puPointer = FREEMARKER;

        /* reset Pid info at begin of mem block */
        puPointer--;/*lint !e613 pu32Pointer is here always !=0 */
        *puPointer = 0;

        pPool->freecnt++;
        pPool->u32CurCount--;
        if(pPool->bLock)
        {
#ifdef USE_SEM_FOR_POOL
           sem_post(pHandle->Sem);
#else	
           pthread_mutex_unlock(pHandle->Mtx); 
#endif
        }
     }
     else if(*puTemp == ALLOCMAGIC)
     {
         /* reset Pid info at begin of mem block */
         puPointer--;/*lint !e613 pu32Pointer is here always !=0 */
         /* check for valid pointer and possible mallocs */
         if(pPool->u32ErrCnt > 0)
//       if(((uintptr_t)pu8Pointer < pHandle->u32memstart)&&((uintptr_t)pu8Pointer > pHandle->u32memend))
         {
            /* memory was allocated via standard malloc */
            free((void*)puPointer);/*lint !e424 */
         }
         else
         {
            TraceString("Mempool wrong free adress 0x% Value:%d",puPointer,pPool->u32ErrCnt);
         }
     }
     else
     {
         TraceString("wrong memory free for Pool %s!",pPool->szName);
         /* try to repair if overwritten block is used */
         if(((uintptr_t)puPointer > pHandle->u32memstart)&&((uintptr_t)puPointer < pHandle->u32memend))
         {
           /**/
         }
     }
  }

  /* check for errors */
  if(u32ErrorCode != OSAL_E_NOERROR)
  {
#ifdef SHORT_TRACE_OUTPUT
      tU8 u8Buffer[50]  = {0};
      tU32 u32Tid = (tU32)OSAL_ThreadWhoAmI();
      OSAL_M_INSERT_T8(&u8Buffer[0],RELBLOCK_MPF_FAILED);
      OSAL_M_INSERT_T32(&u8Buffer[1],u32ErrorCode);
      OSAL_M_INSERT_T32(&u8Buffer[5],u32Tid);
      OSAL_M_INSERT_T32(&u8Buffer[9],pPool->blfsz -12);
      vTraceOsalMpf(TR_LEVEL_FATAL,u8Buffer,13);
#else //SHORT_TRACE_OUTPUT
      TraceString("TID:%d Release memory block size %d failed with error 0x%x",OSAL_ThreadWhoAmI(),pPool->blfsz -12,u32ErrorCode);
#endif	//SHORT_TRACE_OUTPUT
     OSAL_vSetErrorCode(u32ErrorCode);
     s32Return    = OSAL_ERROR;
  }
  return s32Return;
}


/******************************************************************************
 * FUNCTION:      TracePoolInfo
 *
 * DESCRIPTION:   traces the related OSAL memory pool information
 *
 * PARAMETER:     VP   Pointer to allocated memory 
 *               
 * RETURNVALUE:   OSAL_OK/OSAL_ERROR 
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 18.11.09  |   Initial revision                     | MRK2HI
 *****************************************************************************/
void TraceSpecificPoolInfo(const trHandleMpf* pHandle)
{
  tU8 u8Buffer[40];
  u8Buffer[0] = MPF_INFO;
  trOSAL_MPF* pPool = (trOSAL_MPF*)pHandle->pMem;
  
  if((pPool != NULL)&&(pHandle->Magic == HANDLE_MAGIC))
  {
      if(pPool->bLock)
      {
#ifdef USE_SEM_FOR_POOL
        if(u32SemWait(pHandle->Sem) != OSAL_E_NOERROR)
#else	
        if(pthread_mutex_lock(pHandle->Mtx) != 0) 
#endif
        {
           return ;
        }
     }
     OSAL_M_INSERT_T32(&u8Buffer[1],((tU32)pPool->blfsz - (tU32)(2*(tU32)sizeof(uintptr_t))));
   //  OSAL_M_INSERT_T32(&u8Buffer[1],pPool->rMpf.blfsz);
     OSAL_M_INSERT_T32(&u8Buffer[5],(tU32)pPool->mpfcnt);
     OSAL_M_INSERT_T32(&u8Buffer[9],pPool->u32MaxCount);
     OSAL_M_INSERT_T32(&u8Buffer[13],pPool->u32CurCount);
     OSAL_M_INSERT_T32(&u8Buffer[17],pPool->u32ErrCnt);
     OSAL_M_INSERT_T32(&u8Buffer[21],pPool->u32OpenCnt);
     strncpy((char*)&u8Buffer[25],(const char*)&pPool->szName,8);
  
     if(pPool->bLock)
     {
#ifdef USE_SEM_FOR_POOL
        sem_post(pHandle->Sem);
#else	
        pthread_mutex_unlock(pHandle->Mtx); 
#endif
     }
     vTraceOsalMpf(TR_LEVEL_FATAL,u8Buffer,33);
  }
  else
  {
     TraceString("TraceSpecificPoolInfo with invalid handle used");
  }
}


void vGetPoolInfo(const trHandleMpf* pHandle,char* Buffer,tU32 u32Len)
{
   if(pHandle)
   {	   
      trOSAL_MPF* pPool = (trOSAL_MPF*)pHandle->pMem;
      if(pHandle->pMem != NULL)
      {
         snprintf(Buffer,u32Len,"%s Elements:%d Max In use:%d Current in use:%d",
                  pPool->szName,(int)pPool->mpfcnt,(int)pPool->u32MaxCount,(int)pPool->u32CurCount);
      }
      else
      { 
         snprintf(Buffer,u32Len,"Pool not initialzed");
      }
   }
}

void vTracePoolInfo(const trHandleMpf* pHandle)
{
   if(pHandle)
   {	   
      trOSAL_MPF* pPool = (trOSAL_MPF*)pHandle->pMem;
      TraceString("%s Elements:%d Size:%d Max In use:%d Current in use:%d",
                  pPool->szName,pPool->mpfcnt,pPool->blfsz,pPool->u32MaxCount,pPool->u32CurCount);
   }
}


void* pvGetFirstElementForPid(const trHandleMpf* pHandle,OSAL_tProcessID Pid)
{
  if(pHandle)
  {	   
     trOSAL_MPF* pPool = (trOSAL_MPF*)pHandle->pMem;
     tU32 i;
     uintptr_t* puVal = NULL;
  
     if(pHandle->Magic != HANDLE_MAGIC)return NULL;

     for(i = 0;i<pPool->mpfcnt;i++)
     {
    //    TraceString("OSAL MemPool Element 0x%x for PID:%d found",pu32Val.*pu32Val);
        puVal = (uintptr_t*)((uintptr_t)pHandle->u32memstart + (i * pPool->blfsz));
        if(*puVal == (uintptr_t)Pid)
        {
          puVal++; //go to size field and check if element is in use
          puVal++; //go to start of element data
   //     TraceString("OSAL Pool:0x%x Element 0x%x for PID:%d found",pPool, pu32Val,u32Pid);
          return ((void*)puVal);
        }
     }
  }
  //TraceString("No Element found");
  return NULL;
}

#ifdef __cplusplus
}
#endif

/************************************************************************ 
|end of file osalmempool.cpp
|-----------------------------------------------------------------------*/

