/*****************************************************************************
| FILE:         osal_dbus_msg.cpp
| PROJECT:      Platform
| SW-COMPONENT: OSAL
|-----------------------------------------------------------------------------
| DESCRIPTION:  This is the implementation file for the OSAL
|               (Operating System Abstraction Layer) . Implementation of DBUS
|               messaging via OSAL MQ
|
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2017 Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 18.04.17  | Initial revision           | MRK2HI
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/
/************************************************************************
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/

#include "OsalConf.h"

#ifdef DBUS_MQ_SUPPORT

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#ifdef __cplusplus
extern "C" {
#endif


#include "Linux_osal.h"
#include "ostrace.h"
#include <dbus/dbus.h>

/************************************************************************
|defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/
#define MAX_DBUS_HANDLE  50

#define OSAL_MQ_BUS_NAME_BASE "org.bosch.osal_mq"
#define OSAL_MQ_DBUS_IF       "org.bosch.osal_mq"
#define OSAL_MQ_DBUS_OBJ_BASE "/org/bosch/osal_mq"
#define OSAL_MSG_METHOD       "osal_message"

//#define USE_MSG_HANDLER
//#define PRIVATE_CON
/************************************************************************
|typedefs (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/
/* Use process local Handle struct because storing to shared memory fails, reason unknown */
static DBusConnection* Connection[MAX_DBUS_HANDLE];

/************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
|function prototype (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
|function prototype (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
|function implementation (scope: module-local)
|-----------------------------------------------------------------------*/
tS32 GetFreeEntryIndex(void)
{
   tS32 i;
   for(i=0;i<MAX_DBUS_HANDLE;i++)
   {
      if(Connection[i] == 0)
      {
         return i;
      }
   }
   return OSAL_ERROR;
}

#ifdef USE_MSG_HANDLER
DBusHandlerResult filter_func(DBusConnection *connection, 
      DBusMessage *message, void *pData)
{
   dbus_bool_t handled = false;
   char *pReadData;
   int len;

   DBusError dberr;
   dbus_error_init(&dberr);

   ((void)connection);
   TraceString("filter_func %s",pData);
   if(FALSE == dbus_message_get_args(message, &dberr, DBUS_TYPE_ARRAY,
   DBUS_TYPE_BYTE, &pReadData, &len,  DBUS_TYPE_INVALID) && 0 != len)
   {
     return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
   }
   if(0 == len)
      return DBUS_HANDLER_RESULT_HANDLED;
   
   TraceString("filter_func pReadData %s",pReadData);
   return (handled ? DBUS_HANDLER_RESULT_HANDLED :
   DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

}
#endif

/*****************************************************************************
*
* FUNCTION:    ConnectToDBus
*
* DESCRIPTION: This function connect the caller of a metod to the dbus subsystem
*
* PARAMETER:     char*  pointer to name of the DBUS caller, 
*
* RETURNVALUE:   DBusConnection* pointer to DBUS connection handle
*
* HISTORY:
* Date      |   Modification                         | Authors
* 18.04.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
OSAL_DECL tS32 ConnectToDBus(OSAL_tMQueueHandle hMQ, tU32 u32MsgSize,tBool bOwn)
{
   DBusError err;
   int ret = 0;
   int Idx = 0;
   char Name[64];
 
/*   if(Connection == NULL)
   {
      pConnection = (DBusConnection*)malloc(sizeof(DBusConnection*)*MAX_DBUS_HANDLE);
      memset(pConnection,0,sizeof(DBusConnection*)*MAX_DBUS_HANDLE);
   }*/
   Idx = GetFreeEntryIndex();

   if(Idx == OSAL_ERROR)
   {
       //OSAL_E_NOSPACE;
       return Idx;
   }
   else
   {
//     dbus_threads_init_default();
     // initialiset the errors
     dbus_error_init(&err);
     if(Connection[Idx] == NULL)
     {
        // connect to the system bus and check for errors
#ifdef PRIVATE_CON
        Connection[Idx] = dbus_bus_get_private(DBUS_BUS_SYSTEM, &err);
#else
        Connection[Idx] = dbus_bus_get(DBUS_BUS_SYSTEM, &err);
#endif
        if (dbus_error_is_set(&err)) 
        { 
           TraceString("OSALMQ_DBUS Connection Error (%s)\n", err.message); 
           dbus_error_free(&err);
        }
        if (NULL == Connection[Idx]) 
        { 
           return OSAL_ERROR;
        }
     }
   }

   /* define a IF name */
   snprintf(Name,64,"%s.%s.server",OSAL_MQ_BUS_NAME_BASE,(char*)((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
   if(bOwn)
   {
      if(!dbus_bus_name_has_owner(Connection[Idx],Name,&err))
      {
         if(dbus_validate_interface(Name,&err))
         { 
             // request our name on the bus
            ret = dbus_bus_request_name(Connection[Idx], Name, DBUS_NAME_FLAG_REPLACE_EXISTING , &err);
            switch(ret) 
             {
                case -1: 
                     TraceString("OSALMQ_DBUS Couldn't acquire name %s for our connection: %s\n", Name ,err.message);
                    break;
                case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
                     TraceString("OSALMQ_DBUS dbus_bus_request_name(): We now own the name %s!\n",Name);
                    break;
                case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
                     TraceString("OSALMQ_DBUS dbus_bus_request_name(): We are standing in queue for our name %s!\n",Name);
                    break;
                case DBUS_REQUEST_NAME_REPLY_EXISTS:
                     TraceString("OSALMQ_DBUS dbus_bus_request_name(): :-( The %d we asked for already exists!\n",Name);
                    break;
                case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
                     TraceString("OSALMQ_DBUS dbus_bus_request_name(): Eh? We already own this name %d! \n",Name);
                    break;
                default:
                     TraceString("OSALMQ_DBUS dbus_bus_request_name(): Unknown result = %d %d\n", ret,err.message);
                    break;
            }
            if (dbus_error_is_set(&err)) 
            { 
               dbus_error_free(&err);
               return OSAL_ERROR;
            }
            if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret) 
            { 
              TraceString("OSALMQ_DBUS Name Error DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret"); 
            }
            /* set filter otpion*/
            char Buffer[DBUS_MAXIMUM_MATCH_RULE_LENGTH];
            snprintf(Buffer, DBUS_MAXIMUM_MATCH_RULE_LENGTH,"type='method_call',interface='%s.%s'",OSAL_MQ_DBUS_IF,(char*)((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
            // add a rule for which messages we want to see
            dbus_bus_add_match(Connection[Idx], Buffer, &err); // see signals from the given interface, no error code because we don't need a reply
            if (dbus_error_is_set(&err)) 
            { 
                TraceString("OSALMQ_DBUS dbus_bus_add_match Error (%s)\n", err.message);
                dbus_error_free(&err);
            } 
            dbus_connection_flush(Connection[Idx]);
            /* set the message size for this DBUS queue*/
#ifdef USE_MSG_HANDLER
            if (!dbus_connection_add_filter(Connection[Idx], filter_func, NULL, NULL))
            {
               TraceString("OSALMQ_DBUS dbus_connection_add_filter failed ");
            }
#endif
   //         dbus_connection_set_max_message_size(Connection[Idx],u32MsgSize);
              ((void)u32MsgSize);
         }
         else
         {
            TraceString("OSALMQ_DBUS dbus_validate_interface Error "); 	  
         }
      }
   }
   else
   {
       // request our name on the bus
       /*ret = dbus_bus_request_name(pConnection[Idx], Name, DBUS_NAME_FLAG_REPLACE_EXISTING , &err);
       if (dbus_error_is_set(&err)) 
       { 
          TraceString("PRM_DBUS Name Error (%s)\n", err.message); 
          dbus_error_free(&err);
       }*/
  //     Connection[Idx] = dbus_connection_ref(Connection[Idx]);
       if(!dbus_connection_get_is_connected(Connection[Idx]))
       {
          TraceString("OSALMQ_DBUS dbus_connection_get_is_connected Error "); 
          return OSAL_ERROR;
       }
   }
   
   ((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle = Idx;
   
   if(!dbus_connection_get_is_connected(Connection[Idx]))
   {
      TraceString("OSALMQ_DBUS dbus_connection_get_is_connected Error "); 
      return OSAL_ERROR;
   }
   return OSAL_OK;
}

/*****************************************************************************
*
* FUNCTION:    DisconnectFromDBus
*
* DESCRIPTION: This function decremnts the referenz count to the DBUS connection
*
* PARAMETER:     OSAL_tMQueueHandle Handle of the MQ
*
* RETURNVALUE:   always OSAL_OK
*
* HISTORY:
* Date      |   Modification                         | Authors
* 18.04.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
OSAL_DECL tS32 DisconnectFromDBus(OSAL_tMQueueHandle hMQ)
{
#ifdef PRIVATE_CON
   dbus_connection_close(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]);
#endif
   dbus_connection_unref(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]);
   Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle] = 0;
   ((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle = 0;
   return OSAL_OK;
}

OSAL_DECL tS32 ReleaseIfDBus(void)
{
   DBusError error;
   int Ret;
   dbus_error_init(&error);
   if((Ret = dbus_bus_release_name(Connection[0], OSAL_MQ_DBUS_IF, &error)) == -1)
   {
      TraceString("OSALMQ_DBUS Couldn't dbus_bus_release_name %s for our connection: %s\n", OSAL_MQ_DBUS_IF ,error.message);
   }
   return Ret;
}

//#define SEND_WITH_REPLY

/*****************************************************************************
*
* FUNCTION:    DBusMsgQueueSend
*
* DESCRIPTION: This function send data from a given buffer to the DBUS connection
*
* PARAMETER:     OSAL_tMQueueHandle Handle of the MQ
*                char*              pointer to the send buffer   
*                tU32               size of the send buffer
*
* RETURNVALUE:   OSAL_OK if succeeded otherwise OSAL_ERROR
*
* HISTORY:
* Date      |   Modification                         | Authors
* 18.04.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
OSAL_DECL tS32 DBusMsgQueueSend(OSAL_tMQueueHandle hMQ, char* pData , tU32 u32Size)
{
   DBusMessage* pMsg = NULL;
   DBusError err;
   dbus_error_init(&err);	 
   dbus_uint32_t serial = 0;
  
   char Path[64];
   char Name[64];
   char Server[64];
   memset(Path,0,64);
   memset(Name,0,64);
   memset(Server,0,64);
   snprintf(Path,64,"%s/%s",OSAL_MQ_DBUS_OBJ_BASE,(char*)((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
   snprintf(Name,64,"%s.%s",OSAL_MQ_DBUS_IF,(char*)((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
   snprintf(Server,64,"%s.%s.server",OSAL_MQ_BUS_NAME_BASE,(char*)((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
   pMsg = dbus_message_new_method_call(Server, /* destination */
                                       Path,  /* obj. path */
                                       Name,  /* interface */
                                       OSAL_MSG_METHOD); /* method str */
   if(pMsg)
   {
      if(dbus_message_append_args(pMsg,DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &pData, u32Size, DBUS_TYPE_INVALID) == FALSE)
      {
           TraceString("OSALMQ_DBUS dbus_message_append_args failed");
      }
#ifdef SEND_WITH_REPLY
      DBusMessageIter args;
      DBusPendingCall* pending;
      dbus_uint32_t func;
      // send message and get a handle for a reply
      if ( !dbus_connection_send_with_reply (connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle], pMsg, &pending, -1 ) ) 
      {
         TraceString("OSALMQ_DBUS dbus_connection_send_with_reply Out Of Memory!\n" );
         return OSAL_ERROR;
      }
      if ( NULL == pending ) 
      {
         TraceString("OSAL_DBUS Pending Call Null" );
         return OSAL_ERROR;
      }
      dbus_connection_flush(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]);
      // free message
      dbus_message_unref(pMsg);
      // block until we receive a reply
      dbus_pending_call_block( pending );
      // get the reply message
      pMsg = dbus_pending_call_steal_reply( pending );
      if ( NULL == pMsg ) 
      {
         TraceString("OSAL_DBUS Reply Null" );
         return OSAL_ERROR;
      }
      // free the pending message handle
      dbus_pending_call_unref( pending );
      // read the parameters
      if( !dbus_message_iter_init( pMsg, &args ) )
      TraceString("OSAL_DBUS Message has no arguments!" );

      if (DBUS_TYPE_UINT32 == dbus_message_iter_get_arg_type(&args)) 
      {
          dbus_message_iter_get_basic(&args, &func);
          TraceString("OSAL_DBUS Function Argument is 0x%x!",func); 
      }
      return OSAL_OK;
#else
   //   dbus_message_set_no_reply (pMsg,TRUE);
      if(dbus_connection_send(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle],pMsg,&serial) == TRUE)
      {
       //  dbus_connection_flush(connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]);//only needed when exit will follow
         dbus_message_unref(pMsg);
         return OSAL_OK;
      }
      else
      {
         TraceString("OSALMQ_DBUS dbus_connection_send failed ");
      }
      dbus_message_unref(pMsg);
#endif 
   }
   else
   {
      TraceString("OSALMQ_DBUS dbus_message_new_method_call failed ");
   }
   return OSAL_ERROR;   
}


/*****************************************************************************
*
* FUNCTION:    bCheckForValidMessage
*
* DESCRIPTION: This function check validity of a specific DBUS message for a given connection
*
* PARAMETER:     OSAL_tMQueueHandle Handle of the MQ
*                DBusMessage*       pointer to DBUS message
*
* RETURNVALUE:   TRUE if valid message otherwise FALSE
*
* HISTORY:
* Date      |   Modification                         | Authors
* 18.04.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
tBool bCheckForValidMessage(OSAL_tMQueueHandle hMQ, DBusMessage* pMsg)
{
  char Name[64];
  snprintf(Name, 64,"%s.%s", OSAL_MQ_DBUS_IF,(char*)((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
  if(!dbus_message_is_method_call(pMsg, Name,OSAL_MSG_METHOD))
  {
    if(LLD_bIsTraceActive((tU32)OSAL_C_TR_CLASS_MQ_DBUS, (tU32)TR_LEVEL_DATA))
    {        
      int Type = dbus_message_get_type(pMsg);
	  switch(Type)
	  {
        case DBUS_MESSAGE_TYPE_METHOD_CALL:
             TraceString("OSALMQ_DBUS unexpected DBUS_MESSAGE_TYPE_METHOD_CALL");
            break;
        case DBUS_MESSAGE_TYPE_METHOD_RETURN: 
             TraceString("OSALMQ_DBUS unexpected DBUS_MESSAGE_TYPE_METHOD_RETURN");
            break;
        case DBUS_MESSAGE_TYPE_ERROR:
             TraceString("OSALMQ_DBUS unexpected DBUS_MESSAGE_TYPE_ERROR");
            break;
        case DBUS_MESSAGE_TYPE_SIGNAL:
             TraceString("OSALMQ_DBUS unexpected DBUS_MESSAGE_TYPE_SIGNAL");
            break;
        default:
             TraceString("OSALMQ_DBUS unexpected unknown type call");
            break;
	  }
      const char* Name = dbus_message_get_interface(pMsg);
      TraceString("OSALMQ_DBUS message for %s IF received",Name);
      Name = dbus_message_get_path(pMsg);
      TraceString("OSALMQ_DBUS message for %s Path received",Name);
      Name = dbus_message_get_destination(pMsg);
      TraceString("OSALMQ_DBUS message for %s destination received",Name);
    }
    return FALSE;
  }
  else
  {
     return TRUE;
  }
}

/*****************************************************************************
*
* FUNCTION:    DBusMsgQueueRecv
*
* DESCRIPTION: This function receives data in a given buffer from the DBUS connection
*
* PARAMETER:     OSAL_tMQueueHandle Handle of the MQ
*                char*              pointer to the reception buffer   
*                tU32               size of the reception buffer
*
* RETURNVALUE:   OSAL_OK if succeeded otherwise OSAL_ERROR
*
* HISTORY:
* Date      |   Modification                         | Authors
* 18.04.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
OSAL_DECL tS32 DBusMsgQueueRecv(OSAL_tMQueueHandle hMQ, char* pData , tU32 u32Size, tU32 u32Timeout)
{
   DBusMessage* pMsg = NULL;
   DBusError err;  
   dbus_error_init(&err);
   int len=0;   
   char* pReadData = NULL;
  
   if(!dbus_connection_get_is_connected(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]))
   {
      TraceString("OSALMQ_DBUS dbus_connection_get_is_connected Error "); 
      return OSAL_ERROR;
   }
  int Tmo = (int)u32Timeout;
  while(1)
  {
     /* all sorts of authentication protocol stuff has to be parsed before the first message arrives. */ 
     pMsg = dbus_connection_pop_message(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]);
     if(pMsg == NULL)dbus_connection_read_write(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle],Tmo);
//     pMsg = dbus_connection_pop_message(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]);
     if(pMsg)
     {
        if(!dbus_connection_get_is_connected(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]))
        {
           TraceString("OSALMQ_DBUS dbus_connection_get_is_connected Error pMsg: 0x%x",pMsg); 
        }
        if(bCheckForValidMessage(hMQ,pMsg) == TRUE)break;
     }
     else
     {
        if(!dbus_connection_get_is_connected(Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]))
        {
          TraceString("OSALMQ_DBUS dbus_connection_get_is_connected Error pMsg: 0x%x",pMsg); 
          return OSAL_ERROR;
        }
        if(LLD_bIsTraceActive((tU32)OSAL_C_TR_CLASS_MQ_DBUS, (tU32)TR_LEVEL_DATA))
        {        
           TraceString("OSALMQ_DBUS dbus_connection_read_write_ failed ");
        }
        if(u32Timeout)
        {
           OSAL_s32ThreadWait(500);
           if(u32Timeout != OSAL_C_TIMEOUT_FOREVER)
           {
              len = (int)u32Timeout; 
              len = len - 500;
              if(len <= 0)break;
           }
        }
     }
  }

	   
  if(pMsg)
  {
     len = 0;
     if(dbus_message_get_args(pMsg,&err,DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE,&pReadData, &len,DBUS_TYPE_INVALID)&& (0 != len))
     {
        memcpy(pData,pReadData,len);
        if(len != (int)u32Size)
        {
           TraceString("OSALMQ_DBUS short message received %d instead of %d ",len,u32Size); 
        }
     }   
     else
     {
        TraceString("OSALMQ_DBUS dbus_message_get_args failed %s",err.name);
     }
     dbus_message_unref(pMsg);
   }
  
  
#ifdef SEND_WITH_REPLY
   DBusMessage *reply;
   dbus_uint32_t Val = len;
   if ((reply = dbus_message_new_method_return (pMsg)) == NULL) 
   {
      TraceString("OSALMQ_DBUS Error in dbus_message_new_method_return"); 
   }
   else
   {	   
      DBusMessageIter iter; 
      dbus_message_iter_init_append (reply, &iter); 
      if (!dbus_message_iter_append_basic (&iter, DBUS_TYPE_UINT32, &Val))
      {
          TraceString("OSALMQ_DBUS Error in dbus_message_iter_append_basic"); 
 //  exit (1); 
      }
      else
      {	      
         if (!dbus_connection_send (Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle], reply, NULL)) 
         {        
            TraceString("Error in dbus_connection_send\n"); 
     //       exit (1); 
         } 
         dbus_connection_flush (Connection[((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->s32LiHandle]); 
         dbus_message_unref (reply); 
      }
   }
#endif
   return len;
}


#ifdef __cplusplus
}
#endif

#endif //DBUS_SUPPORT

/************************************************************************
|end of file osal_dbus_msg.cpp
|-----------------------------------------------------------------------*/