/************************************************************************
| FILE:         dgram_service.h
| PROJECT:      platform
| SW-COMPONENT: INC
|------------------------------------------------------------------------------
| DESCRIPTION:  Datagram service to be used on TCP stream socket.
|
|               Written data is prepended with a header which is used to find
|               message boundaries on receive.
|
|               Each call to dgram_send will result in one call to dgram_recv
|               returning.
|
|------------------------------------------------------------------------------
| COPYRIGHT:    (c) 2013 Robert Bosch GmbH
| HISTORY:
| Date      | Modification               | Author
| 15.01.13  | Initial revision           | Andreas Pape
| 21.02.13  | Fix cpp compiler issues    | Matthias Thomae
| 06.02.13  | Move functions to .c file  | Matthias Thomae
| 07.10.16  | Support for TCP/IP         | Mai Daftedar
|	    | based Gen4 INC             |
| 17.10.16  | Fix RADAR and lint issues	 | Venkatesh Parthasarathy
| 10.01.17  | Dgram Header update and    | Venkatesh Parthasarathy
|           | data types changed for 64  |
|           | bit arch.                  |
| 24.07.17  | fixed compilation errors in|
|           |  Rcar & g4Lsim environment | Ravi Kiran Gundeti
| 7.09.17   | dgram_connect new api added| Shahida Mohammed Ashraf
| 7.12.17   | Added modifications to tests | Mai Daftedar
|27.02.18	| Removed Warnings			   | Mai Daftedar
|*****************************************************************************/
#ifndef __DGRAM_SERVICE__H__
#define __DGRAM_SERVICE__H__

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <unistd.h>


#ifdef __cplusplus
extern "C" {
#endif

//#define OSAL_GEN4

#ifdef OSAL_GEN4
#define MSG_PROTOCOL
#endif

#ifdef DG_ENABLE_DEBUG
#define DG_DBG(a, ...) fprintf(stdout,a, ## __VA_ARGS__)
#else
#define DG_DBG(a,...)
#endif

#define DG_ERR(a, ...) fprintf(stderr,a, ## __VA_ARGS__)

#define UNUSED_VARIABLE(VAR) (void)VAR

#define PAD(x) char pad[x]

/*Define INC domain errors*/
#define   INC_ERR_PORT_IN_USE	-103
#define	  ER_LOCAL_HOST_FAIL	-104
#define   ER_REMOTE_HOST_FAIL	-105

#define INTERFACE_NAME_MAX      256
#define PORT_NUM_MAX            10
#define INC_SUCCESS              0    
/*standard header for datagram service*/
typedef struct {
#ifdef MSG_PROTOCOL
   /* Header to include the PDU ID.
      PDU ID is first byte of message followed by LUN */
   unsigned int pduid;
   unsigned int dglen; /* 4-byte length of dgram */
#else
   unsigned short dglen;   /* 2-byte length of dgram */
#endif
}dgram_header_std;

/*maximum size of datagram*/
#define DGRAM_MAX (0xffff-sizeof(dgram_header_std))

typedef struct {
   int sk;     /*socket filedescriptor*/
   int proto;  /*used protocol*/
   size_t len;    /*len of receive buffer*/
   size_t hlen;   /*size of used dgram header*/
   dgram_header_std h; /*copy of received header*/
   char *buf;  /*receive buffer*/
   ssize_t received;  /*received bytes*/
#if __WORDSIZE == 64
   PAD(4);
#endif
}sk_dgram;

sk_dgram* dgram_init(int sk, size_t dgram_max, void *options);
int dgram_exit(sk_dgram *skd);
ssize_t dgram_send(sk_dgram *skd, void *ubuf, size_t ulen);
ssize_t dgram_recv(sk_dgram *skd, void* ubuf, size_t ulen);

int dgram_connect(int *sockfd, char *intf_name, unsigned int timeout, unsigned int serverportno, unsigned int clientportno, sk_dgram **dgram,unsigned int rcv_timeout);
#ifdef __cplusplus
}
#endif

#endif /*__DGRAM_SERVICE__H__*/
