/************************************************************************
| FILE:         inc_send.c
| PROJECT:      platform
| SW-COMPONENT: INC
|------------------------------------------------------------------------------
| DESCRIPTION:  Test program for sending arbitrary INC messages.
|
|               Example:
|               /opt/bosch/base/bin/inc_send_out.out -b 50944 -p 50944 -r scc 01-02-03-04
|------------------------------------------------------------------------------
| COPYRIGHT:    (C) 2013 Robert Bosch GmbH
| HISTORY:
| Date      | Modification               | Author
| 24.01.13  | Initial revision           | Andreas Pape
| 28.01.13  | Use datagram service       | tma2hi
| 06.02.13  | Use local hostname         | tma2hi
| 06.02.13  | Integrate Lothar's SPM ext | tma2hi
| 19.03.13  | Update description         | tma2hi
| 19.06.14  | Text error msgs, recv only | tma2hi
| 06.01.17  | used getaddrinfo() instead |
|           | of gethostbyname() and     | gur6kor
|           | added vPrintHelp function  |
| 27.2.18   | Add Lingr socket option + 
			  removed warnings			 | Mai Daftedar
|*****************************************************************************/

#define USE_DGRAM_SERVICE

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h> 
#include <sys/time.h>
#include <time.h>
#include <netinet/tcp.h>
#include <sys/poll.h>
#include <signal.h>
#include "inc.h"

#ifdef USE_DGRAM_SERVICE
#include "dgram_service.h"
sk_dgram *pdgram = NULL;
#endif

#ifdef MSG_PROTOCOL
#define DEFAULT_MSGSZ DGRAM_MAX
#else
#define DEFAULT_MSGSZ 1500
#endif

int iSocketfd = -1;

static volatile int bTerminate;

#define PR_ERROR(fmt, args...) \
      { fprintf(stderr, "inc_send (line: %d): %s: " fmt "\n", __LINE__, __func__, ## args );}
#define PR_WARN(fmt, args...) \
      { fprintf(stdout, fmt "\n", ## args );}

/* Timeout in seconds */
#define TIMEOUT_POLL_RECV 200


static void vPrintHelp()
{
   printf( "\nThis (inc_send_out.out) is a test program for sending arbitrary INC messages\n");
   printf( "\nUsage: /opt/bosch/base/bin/inc_send_out.out [options] <Host name> <data/message>\n");
   printf( "\nOptions:\n");
   printf( "   -h   show this help message and exit\n");
   printf( "   -b   Local host port number to bind\n");
   printf( "   -p   Remote host port number for connect\n");
   printf( "   -r   receive data after sending\n");
   printf( "   -s   Activate  spm test mode\n");
   printf( "   -m   receives multiple data after sending\n");
   printf( "\nHost name:   scc/adr3/fake0\n");
   printf( "\nExamples:\n");
   printf( "   /opt/bosch/base/bin/inc_send_out.out -b 50944 -p 50944 -r scc 01-02-03-04\n");
   printf( "   /opt/bosch/base/bin/inc_send_out.out -s 0 -r scc\n\n");
   printf( "Example for receiving multiple data\n");
   printf( "   /opt/bosch/base/bin/inc_send_out.out -b 50944 -p 50944 -m scc 01-02-03-04\n");
   exit(0);
}

/******************************************************************************
* FUNCTION: hexdump(const char *cpIntro, char *cpMsgBuf, int iLength)
*
* DESCRIPTION: Priniting the passed buffer in Hex Format
*
* PARAMETERS:
*      cpIntro: Intro name printed before HEX
*      cpMsgBuf: Buffer to be printed in HEX
*      iLength: Length of buffer
*
* RETURNS: NONE
*
*****************************************************************************/
static void hexdump(const char *cpIntro, char *cpMsgBuf, int iLength)
{
   int i;
   printf("%s:",cpIntro);
   for (i=0;i<iLength;i++)
      printf(" 0x%02x", cpMsgBuf[i]);
   printf("\n");
}

/******************************************************************************
* FUNCTION: static int arg2payload(unsigned char *cpBuffer, int iBufLen, char* cpString)
*
* DESCRIPTION: Helper function to convert from string to HEX Format
*           input: payload as hex-ASCII string: e.g: a9-de-56-03
*
* PARAMETERS:
*      cpBuffer: output buffer
*      iBufLen: Buffer length
*      cpString: input string buffer to be converted to HEX
*
* RETURNS: The length of the output buffer
*
*****************************************************************************/
static int arg2payload(unsigned char *cpBuffer, int iBufLen, char* cpString)
{
   char *pos, *end;
   unsigned char ret;
   long result;
   int len = 0;
   if(!cpString)
      return -1;
   pos = cpString;
   end = cpString;
   while(*end != '\0' && (len < iBufLen)) {
      result = strtol(pos, &end ,16);
      if(result > 255)
         return -1;
      if(result < 0)
         return -1;
		/* result is within one byte range: Casting, removes warnings and wouldn't affect
		 * the functionality 
		 */
		ret = (unsigned char)result;
      cpBuffer[len++] = ret;
      if(*end != '\0') {
         if(*end == '-')
            end++;
         else
            return -1;
      }
      pos = end;
   }
   return len;
}

/******************************************************************************
* FUNCTION: bStartINCCommunication( char* cpIncHostName, int iLocalHostPortNum, int iRemoteHostPortNum )
*
* DESCRIPTION: Establish INC communication. This function opens a socket and
*               initialize the connection with Host.
*
* PARAMETERS:
*      cpIncHostName: Host name ex: scc/adr3/fake0
*      iLocalHostPortNum: Local host Port Number
*      iRemoteHostPortNum: Remote host Port Number
*
* RETURNS: True - If connection successfully established. else False
*
*****************************************************************************/
static bool bStartINCCommunication( char* cpIncHostName, int iLocalHostPortNum, int iRemoteHostPortNum )
{
   int ret;
   char cIncHostLocal[256] = {0}, cIncHostRemote[256] = {0};
   char cLocalPortNum[10] = {0}, cRemotePortNum[10] = {0};
   struct addrinfo hints, *pHostAddrLocal = NULL, *pHostAddrRemote = NULL;
   struct sockaddr_in *local_addr, *remote_addr;
   struct linger lingstruct;
   sprintf(cLocalPortNum, "%d", iLocalHostPortNum);
   sprintf(cRemotePortNum, "%d", iRemoteHostPortNum);

   errno = 0;
   iSocketfd = socket(AF_BOSCH_INC_AUTOSAR, SOCK_STREAM, 0);
   if ( 0 > iSocketfd )
   {
      PR_ERROR("ERROR failed to create a socket. error %d: %s", errno, strerror(errno));
      return ( false );
   }
#ifdef USE_DGRAM_SERVICE
   pdgram = dgram_init(iSocketfd, DGRAM_MAX, NULL);
   if ( NULL == pdgram )
   {
      PR_ERROR("ERROR dgram_init failed\n");
      return ( false );
   }
#endif
 	/* Using the lingr socket option */
   	lingstruct.l_linger = 0;
	/* Set the paramater "l_onoff" to 1 to ensure that the Linger socket option is active */
	lingstruct.l_onoff  = 1;
   	ret = setsockopt(iSocketfd, SOL_SOCKET, SO_LINGER,&lingstruct, sizeof(lingstruct));
   	if(ret) {
		PR_ERROR("Failure to set socket options\n");   
   	}
	memset( &hints, 0, sizeof( hints ) );
	hints.ai_family = AF_INET;

   if ( NULL == cpIncHostName )
   {
      PR_ERROR("ERROR cpIncHostName NULL\n");
      return ( false );
   }

   sprintf(cIncHostLocal, "%s-local", cpIncHostName);
   sprintf(cIncHostRemote, "%s", cpIncHostName);

   /**Get the Local Host and Bind to that Host,
      which is dynamically created at runtime by kernal in the file /etc/hosts*/
   ret = getaddrinfo( cIncHostLocal, cLocalPortNum, &hints, &pHostAddrLocal );
   if ( 0 != ret )
   {
      PR_ERROR( "Failed to getaddrinfo for '%s' : with error (%d): %s", \
                          cIncHostLocal, ret, gai_strerror( ret ) );
      return ( false );
   }

   if ( NULL == pHostAddrLocal )
   {
      PR_ERROR("ERROR getaddrinfo failed for port 0x%x ('%s')\n", iLocalHostPortNum, cIncHostLocal );
      return ( false );
   }

   local_addr = (struct sockaddr_in *)pHostAddrLocal->ai_addr;
   printf( "local %s:%d\n", inet_ntoa(local_addr->sin_addr), ntohs(local_addr->sin_port));

   /** Bind to a local port to recieve response*/
   errno = 0;
   //lint -e64
   if ( bind( iSocketfd, pHostAddrLocal->ai_addr, pHostAddrLocal->ai_addrlen ) < 0 )
   {
      freeaddrinfo( pHostAddrLocal );           /* addr info no longer needed */
      PR_ERROR( "Failed to bind the socket port 0x%x ('%s'), error %d: %s", \
                          iLocalHostPortNum, cIncHostLocal, errno, strerror( errno ) );
      return ( false );
   }
   freeaddrinfo( pHostAddrLocal );           /* addr info no longer needed */

   memset( &hints, 0, sizeof( hints ) );
   hints.ai_family = AF_INET;

   /**Get the Remote Host and Connect to that Host,
      which is dynamically created at runtime by kernal in the file /etc/hosts*/
   ret = getaddrinfo( cIncHostRemote, cRemotePortNum, &hints, &pHostAddrRemote );
   if ( 0 != ret )
   {
      PR_ERROR( "Failed to getaddrinfo for '%s' : with error (%d): %s", \
                          cIncHostRemote, ret, gai_strerror( ret ) );
      return ( false );
   }

   if ( NULL == pHostAddrRemote )
   {
      PR_ERROR("ERROR getaddrinfo failed for port 0x%x ('%s')\n", iRemoteHostPortNum, cIncHostRemote );
      return ( false );
   }

   remote_addr = (struct sockaddr_in *)pHostAddrRemote->ai_addr;
   printf( "remote %s:%d\n", inet_ntoa(remote_addr->sin_addr), ntohs(remote_addr->sin_port));

   /* Connect to remote port*/
   errno = 0;
   if ( connect( iSocketfd, pHostAddrRemote->ai_addr, pHostAddrRemote->ai_addrlen ) < 0 )
   {
      freeaddrinfo( pHostAddrRemote );           /* addr info no longer needed */
      PR_ERROR( "Failed to connect to socket port 0x%x ('%s'), error %d: %s", \
                          iRemoteHostPortNum, cIncHostRemote, errno, strerror( errno ) );
      return ( false );
   }
   freeaddrinfo( pHostAddrRemote );           /* addr info no longer needed */

   return ( true );
} // bStartINCCommunication

/******************************************************************************
* FUNCTION: vEndINCCommunication()
*
* DESCRIPTION: End established INC communication
*
* PARAMETERS: NONE
*
* RETURNS: NONE
*
*****************************************************************************/
static void vEndINCCommunication()
{
#ifdef USE_DGRAM_SERVICE
   if ( NULL != pdgram )
   {
      if (dgram_exit(pdgram) < 0)
         PR_ERROR( "dgram_exit failed" );
      pdgram = NULL;
   }
#endif
   if ( - 1 != iSocketfd )
   {
      #ifdef MSG_PROTOCOL 
          /* Added for the Autosar issue of not forwarding the Txconfirmation to pDUR*/
          usleep(50000); 
      #endif
      close( iSocketfd );
      iSocketfd = - 1;
   }
} // vEndINCCommunication

int inc_send( void *ubuf )
{
   unsigned char cBuf[DEFAULT_MSGSZ] = {0};
   int iBytesSent = -1, len;
   ssize_t ret;  /* Added to prevent warning for the GEN4 related size changes in the dgram code */
   /* We are expecting unsigned char values from 0 to 255 : Use unsigend char prevents warnings
	*/
   len = arg2payload(cBuf,DEFAULT_MSGSZ, ubuf );  // start message
   if(len<0)
   {
      PR_ERROR("ERROR on parsing payload\n");
	  /*Coverity Fix : 77261, 77257 */
      return ( iBytesSent );
   }

   hexdump("SEND", cBuf , len);

   errno = 0;
#ifdef USE_DGRAM_SERVICE
   if ( NULL == pdgram )
   {
      PR_ERROR("ERROR pdgram NULL\n");
      return ( iBytesSent );
   }
   /* Added to prevent warnings: dgram_send and write functions return ssize_t */
   ret = (ssize_t)dgram_send(pdgram, cBuf, (size_t)len);
   iBytesSent = (int)ret;
#else
   if ( 0 < iSocketfd ) {
		ret = write(iSocketfd,ubuf,len);
		iBytesRecv = (int)ret;
   }
#endif
   if (iBytesSent < 0) 
   {
     PR_ERROR("ERROR writing to socket: %d %s", errno, strerror(errno));
   }

   return iBytesSent;
}

static inline char *signo2signame(int signal)
{
    const char *signalname;

    switch(signal) 
    {
        case SIGHUP:
            signalname = "SIGHUP";
            break;
        case SIGINT:
            signalname = "SIGINT";
            break;
        case SIGQUIT:
            signalname = "SIGQUIT";
            break;
        default:
            PR_ERROR("Error Unknown signal (%d) received", signal);
            signalname = "unknown";
    }
    return signalname;
}

static void signal_handler(int signal)
{
      bTerminate = signal;
}

static void handle_signals()
{
    struct sigaction signal;
    signal.sa_handler = signal_handler;
    sigfillset(&signal.sa_mask);

    /* RESTART functions if interrupted by handler */
    signal.sa_flags = SA_RESTART;

    /* Hangup detected on controlling terminal or death of controlling process */
    if(sigaction(SIGHUP, &signal, NULL) == -1)
    {
        PR_WARN("WARNING SIGHUP signal not installed/handled");
    }
    /* Interrupt from keyboard */
    if(sigaction(SIGINT, &signal, NULL) == -1)
    {
        PR_WARN("WARNING SIGINT signal not installed/handled");
    }
    /* Quit from keyboard */
    if(sigaction(SIGQUIT, &signal, NULL) == -1)
    {
        PR_WARN("WARNING SIGQUIT signal not installed/handled");
    }
    /* SIGKILL and SIGSTOP Can't be captured */
}

int inc_multiple_recv(void* arg,void* buffer,size_t len)
{
   int pfd,i,rc = -1;
   int iBytesRecv = -1;
   struct pollfd fds[1];

   sk_dgram* pdgram = (sk_dgram*)(arg);
   if(pdgram == NULL)
   {
      PR_ERROR("ERROR pdgram NULL\n");
      return -1;
   }

   pfd = pdgram->sk;
   fds[0].fd = pfd;
   fds[0].events = POLLIN;

   while(!bTerminate)
   {

       rc = poll(fds, 1, TIMEOUT_POLL_RECV * 1000);

       if (rc > 0)
       {
           if(fds[0].revents & POLLIN)
           {
               memset(buffer, 0, len);
               iBytesRecv = dgram_recv(pdgram, buffer, len);
               hexdump("RECEIVED", buffer, iBytesRecv);
           }
       } else if(rc == 0)
       {
           PR_WARN("WARNING %d Seconds timeout elapsed. Exiting...\n", TIMEOUT_POLL_RECV);
           break;
       } else
       {
           if(!bTerminate)
           {
               PR_ERROR("Error Polling failed\n");
           }
           break;
       }
   }

   if(bTerminate)
   {
       PR_WARN("WARNING Caught %s signal. Exiting process...", signo2signame(bTerminate));
       /* Due to signal handling, poll system call will return with error code. */
       rc = 0;
   }

   return rc;
}

int inc_recv( void* ubuf, size_t ulen )
{
   int iBytesRecv = -1;
   ssize_t ret;  /* Added to prevent warning for the GEN4 related size changes in the dgram code */
   errno = 0;
#ifdef USE_DGRAM_SERVICE
   if ( NULL == pdgram )
   {
      PR_ERROR("ERROR pdgram NULL\n");
      return ( iBytesRecv );
   }
   /* Added to prevent warnings: dgram_recv and read functions return ssize_t */
   ret = dgram_recv(pdgram, ubuf, (size_t)ulen);
   iBytesRecv = (int)ret;
#else
   if ( 0 < iSocketfd ) {
    ret = read(iSocketfd, ubuf, ulen);
    iBytesRecv = (int)ret;
   }
#endif
   if (iBytesRecv < 0)
   {
      PR_ERROR("ERROR reading from socket: %d %s", errno, strerror(errno));
      return ( iBytesRecv );
   }

   hexdump("RECEIVED", ubuf, iBytesRecv);

   return iBytesRecv;
}

int main(int argc, char *argv[])
{
   int portno = 0, localportno = 0, receive = 0, spm = 0, spmseq = 0;
   int multiple_receive = 0;
   int wr, rd, opt;
   unsigned char cMsgBuf[DEFAULT_MSGSZ] = {0};
   unsigned char *buffer = NULL;
   char* cpIncHostRemote;

   if(argc <= 1 )
   {
      vPrintHelp();
   }

   for (opterr = 0; (opt = getopt(argc, argv, "hrp:b:s:m")) != -1;) {
      switch (opt) {
         case 'h':
            vPrintHelp();
            break;
      case 'p':
         portno = atoi(optarg);
         break;
      case 'b':
         localportno = atoi(optarg);
         break;
      case 'r':
         receive = 1;/*received data after sending*/
         break;
      case 's':
         spm    = 1; /* spm test mode active */
         spmseq = atoi(optarg);
         portno      = 0xC701;  /* LUN 1 */
         localportno = 0xC701;  /* LUN 1 */
         break;
      case 'm':
         handle_signals();
         multiple_receive = 1;
         break;
         
      default:
         printf("Invalid option found\n");
         vPrintHelp();
      }
   }
   if( ( !( portno <= 51199 && portno >= 50944 ) ) || ( !( localportno <= 51199 && localportno >= 50944 ) ) )
   {
      PR_ERROR("Invalid port found (%d or %d)- use option -p (allowed range 50944 - 51199)", localportno, portno);
      return ( -1 );
   }

   if(optind>=argc)
   {
      printf("ERROR host missing\n");
      vPrintHelp();
   }

   cpIncHostRemote = argv[optind];

   if( true != bStartINCCommunication( cpIncHostRemote, localportno, portno ) )
   {

      PR_ERROR( "Failed to setup socket connection. Host '%s' (localport -0x%x & remoteport -0x%x)", \
                 argv[optind], localportno, portno );
      return ( -1 );
      }
   optind++;
   if( spm == 0 ) /* no payload, if sequences are send/received */
   {
      if(optind < argc) 
      {
         buffer = argv[optind];
         optind++;
      }else{
         PR_ERROR("ERROR data missing");
         vEndINCCommunication();
         return ( -1 );
      }
   }

   if(optind<argc)
   {
      PR_ERROR("ERROR unexpected argument");
      vEndINCCommunication();
      return ( -1 );
      }

   if( spm == 0 ) /* simple send receive */
   {
      wr = inc_send(buffer);
      if (wr < 0)
      {
         PR_ERROR("ERROR writing from socket");
         vEndINCCommunication();
         return ( -1 );
      }

      /*read*/
      if (receive) {
         rd = inc_recv( cMsgBuf, DEFAULT_MSGSZ );
         if (rd < 0) {
            PR_ERROR("ERROR reading from socket");
            vEndINCCommunication();
            return ( -1 );
         }
      }
      else if(multiple_receive)
      {
          rd = inc_multiple_recv(pdgram,cMsgBuf, DEFAULT_MSGSZ );
          if (rd < 0) {
              PR_ERROR("ERROR reading from socket");
              vEndINCCommunication();
              return ( -1 );
          }
      }
   }
   else{
      char cSpmMsg[DEFAULT_MSGSZ] = {0};
      // Startup message C_INDICATE_CLIENT_APP_STATE (msgid: C0) with ApplicationMode 00:State_Normal, 01 State_download, 02 State_testmanager
      sprintf(cSpmMsg, "c0-%x-00-02-0b", spmseq);
      wr = inc_send(cSpmMsg);
      if (wr < 0)
      {
         PR_ERROR("ERROR writing from socket");
         vEndINCCommunication();
         return ( -1 );
      }

      do
      {
         rd = inc_recv( cMsgBuf, DEFAULT_MSGSZ );
         if (rd < 0) {
            PR_ERROR("ERROR reading from socket");
            vEndINCCommunication();
            return ( -1 );
         }

         switch( cMsgBuf[0] )
         {
            case 0xe8:
               switch( cMsgBuf[1] )
               {
                  case 0xC2:  // C_Startup   buffer[2] = 00: Coldstart, 01: restart, 02: warmstart
                     inc_send("e8-c3-00"); // R_STARTUP - STATE_NORMAL
                     break;

                  case 0x62:  // C_watchdog
                     inc_send("e8-63"); // R_Watchdog
                     break;

                  default:
                     break;
               }
               break;
            default:
               break;
         }
      }while( rd > 0);
   }

   vEndINCCommunication();
   return 0;
}

