/************************************************************************
| FILE:         dgram_service.c
| PROJECT:      platform
| SW-COMPONENT: INC
|------------------------------------------------------------------------------
| DESCRIPTION:  Datagram service to be used on TCP stream socket.
|
|               Written data is prepended with a header which is used to find
|               message boundaries on receive.
|
|               Each call to dgram_send will result in one call to dgram_recv
|               returning.
|
|------------------------------------------------------------------------------
| COPYRIGHT:    (c) 2013 Robert Bosch GmbH
| HISTORY:
| Date      | Modification               | Author
| 15.01.13  | Initial revision           | Andreas Pape
| 21.02.13  | Fix cpp compiler issues    | Matthias Thomae
| 06.02.13  | Move functions to .c file  | Matthias Thomae
| 25.06.15  | LintFix in dgram_init()    | Shahida Mohammed Ashraf(ECF5)
|           | CFG3-1266                  |
| 07.10.16  | Support for TCP/IP         | Mai Daftedar
|           | based Gen4 INC             |
| 17.10.16  | Fix RADAR and lint issues  | Venkatesh Parthasarathy
| 10.01.17  | Dgram Header update and    | Venkatesh Parthasarathy
|           | data types changed for 64  |
|           | bit arch.                  |
| 29.05.17  | Added NULL checks          | Ravi Kiran Gundeti
| 02.06.17  | 64 bit arch change for     |
|           | dgram_send                 | Ravi Kiran Gundeti
| 24.07.17  | fixed compilation errors in|
|           |  Rcar & g4Lsim environment | Ravi Kiran Gundeti
| 7.09.17   | dgram_connect new api added| Shahida Mohammed Ashraf
| 10.11.17  | dgram_connect, dgram_recv  |
|           | error handling corrected   | Shahida Mohammed Ashraf
| 7.12.17   | Added modifications to tests | Mai Daftedar
| 27.2.18   | Removed warnings, dgram_recv issues | Mai Daftedar
| 04.04.18  | Added parameter to dgram_connect | Mai Daftedar
|*****************************************************************************/
#include "dgram_service.h"
#include <signal.h>
#include <sys/socket.h>
#include <netdb.h>
#include <sys/time.h>
#include "inc.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef AF_INC
#define AF_INC 41
#endif


/* signal handler requirements */
struct sigaction act;                   /* signal handler structure */
static bool  timerExpiredFlag = false;    	/* informing the threads to complete the current cycle and exit */


/*****************************************************************
void signalHandle(int signum)
* PURPOSE : Signal handle
*

* RETURN :  void
*
*          
* Author :  
*****************************************************************/
static void signalHandle(int signum)
{
   (void)signum;
   timerExpiredFlag = true;   
}

/*****************************************************************
int dgram_connect(int *sockfd, char *intf_name, unsigned int timeout, unsigned int serverportno,
                  unsigned int clientportno, sk_dgram **dgram,unsigned int rcv_timeout)
* PURPOSE : Connecting with Timeout if server is not responding
* Note: The function parameter types are changed to remove the warnings, the basic types are used.

* RETURN :  Error code if dgram_connect fails 
*
*          
* Author :  

*****************************************************************/
int dgram_connect(int *sockfd, char *intf_name, unsigned int timeout, unsigned int serverportno, unsigned int clientportno,
                  sk_dgram **dgram,unsigned int rcv_timeout)
{
    int32_t err = 0;
	uint32_t ret = 0;
    char cIncHostLocal[INTERFACE_NAME_MAX] = {0}, cIncHostRemote[INTERFACE_NAME_MAX] = {0};
    char cLocalPortNum[PORT_NUM_MAX] = {0}, cRemotePortNum[PORT_NUM_MAX] = {0};
    struct addrinfo hints, *pHostAddrLocal = NULL, *pHostAddrRemote = NULL;
    bool setalarm = true;
    uint32_t enable = 1;
    struct timeval timeout_val={rcv_timeout,0}; //set default timeout in seconds.
	struct linger lingstruct;
	/* Initialize the errno before any system calls */
	errno = 0;
    sprintf(cLocalPortNum, "%d", clientportno);
    sprintf(cRemotePortNum, "%d", serverportno);
	
	
    do
    {
        #ifdef MSG_PROTOCOL
        *sockfd = socket(AF_INET, SOCK_STREAM, 0);

        #else
        *sockfd = socket(AF_INC, SOCK_STREAM, 0);
        #endif
       
        if (*sockfd < 0) {
            err = errno;
            DG_ERR("ERROR opening socket: %d %s\n", err, strerror(err));
            goto ERR;
        }

        *dgram = dgram_init(*sockfd, DGRAM_MAX, NULL);
        #ifdef MSG_PROTOCOL
	
        /*set socket to non blocking and allow port reuse*/
		lingstruct.l_linger = 0;
		/* Set the paramater "l_onoff" to 1 to ensure that the Linger socket option is active */
		lingstruct.l_onoff  = 1;
		err = setsockopt(*sockfd, SOL_SOCKET, SO_LINGER,&lingstruct, sizeof(lingstruct));
		if(err) {
			err = errno;
			DG_ERR("Failure to set socket options\n");
			goto ERR;
		}
	
		if ( (setsockopt(*sockfd,SOL_SOCKET,SO_REUSEADDR,&enable,sizeof(enable))) == -1) {
			err = errno;
            DG_ERR("ERROR: Set socket opts : %d %s\n", err, strerror(err));
            goto ERR;
      	}
	
        if ( (setsockopt(*sockfd,SOL_SOCKET,SO_RCVTIMEO,(char*)&timeout_val,sizeof(struct timeval))) == -1) {
            err = errno;
            DG_ERR("ERROR: Set socket opts : %d %s\n", err, strerror(err));
            goto ERR;
        }
        #endif
       
        memset( &hints, 0, sizeof( hints ) );
        hints.ai_family = AF_INET;
        sprintf(cIncHostRemote, "%s", intf_name);
        sprintf(cIncHostLocal, "%s-local", cIncHostRemote);
       
        /* Get the Local Host and Bind to that Host,
         * which is dynamically created at runtime by kernal in the file /etc/hosts
		 */
        err = getaddrinfo( cIncHostLocal, cLocalPortNum, &hints, &pHostAddrLocal );
        if ( 0 != err ) {
            DG_ERR( "Failed to getaddrinfo for '%s' : with error (%d): %s", \
                                      cIncHostLocal, err, gai_strerror( err ) );
            return ( err );
        }
        if ( NULL == pHostAddrLocal )
        {
            DG_ERR("ERROR getaddrinfo failed for port %s ('%s')\n", cLocalPortNum, cIncHostLocal );
            return ( err );
        }
	
        /** Bind to a local port to recieve response*/
        err = bind( *sockfd, pHostAddrLocal->ai_addr, pHostAddrLocal->ai_addrlen );
        /* addr info no longer needed */
        freeaddrinfo( pHostAddrLocal ); 
        /*If bind error return err*/
        if (err) {

            DG_ERR("Error in binding: %d %s\n",err, strerror(errno));
            return err;
        } else {
            DG_DBG("Successfully binded\n");
        }
		
        memset( &hints, 0, sizeof( hints ) );
        hints.ai_family = AF_INET;

        /**Get the Remote Host and Connect to that Host,
        which is dynamically created at runtime by kernal in the file /etc/hosts*/
        err = getaddrinfo( cIncHostRemote, cRemotePortNum, &hints, &pHostAddrRemote );
        if ( 0 != err ) {
            DG_ERR( "Failed to getaddrinfo for '%s' : with error (%d): %s", \
                                      cIncHostRemote, err, gai_strerror( err ) );
            return ( err );
        }

        if ( NULL == pHostAddrRemote ) {
            DG_ERR("ERROR getaddrinfo failed for port %s ('%s')\n", cRemotePortNum, cIncHostRemote );
            return ( err );
        }


        /* Connect to remote port*/
        err = connect( *sockfd, pHostAddrRemote->ai_addr, pHostAddrRemote->ai_addrlen );

        /* addr info no longer needed */
        freeaddrinfo( pHostAddrRemote );           

        /*If connection not established retry until timeout*/
        if(err) {
            DG_ERR("ERROR connecting: %d %s waiting for server to respond port %s\n", errno, strerror(errno), cRemotePortNum);
            if (dgram_exit(*dgram) < 0)
            {
               DG_ERR("dgram_exit failed\n");
            }
            err = close(*sockfd);
            if(err)                                                                                                                           
				DG_DBG("Close fail\n");
            /* Sleep giving more time to server to start up*/
            /* Autosar server is responding late so sleep of 20s is kept*/
            /* To do update the sleep time*/
            sleep(20);
            if(setalarm) {
                act.sa_handler = signalHandle;
                sigemptyset(&act.sa_mask);
                act.sa_flags = 0;
                err = sigaction(SIGALRM, &act, NULL);
				/* return of alarm is unsigned int for compilation warings using the same return type */
                ret = alarm(timeout);
                if(ret) {
                    DG_ERR("There is an existing alarm to be delivered after '%d' seconds; it has been cancelled.\n",err);
                } else 
                    DG_DBG("Alarm set succesfully\n");
				
               setalarm = false;
            }  
			err = (int32_t)ret;				
            }
            else {
                DG_DBG("Successfully connected\n");
                break;    
            }
            
    }while(!timerExpiredFlag);

ERR:
    return err;
 
}
/*****************************************************************
sk_dgram* dgram_init(int sk, size_t dgram_max, void *options)
* PURPOSE : init datagram servive on given socket
*
* RETURN :  Pointer to the dgram structure initialized
*          
* Author :  

*****************************************************************/
sk_dgram* dgram_init(int sk, size_t dgram_max, void *options)
{
   sk_dgram *skd;
   struct sockaddr_in addr = {0}; //lint fix CFG3-1266
   socklen_t len = sizeof(struct sockaddr);
//lint -e64

   if(getsockname(sk, (struct sockaddr *)&addr, &len)<0)
      return NULL;
  
   DG_DBG("dgram init on socket af %d\n",addr.sin_family);

   if(dgram_max > DGRAM_MAX) { /* satisfy lint */
      DG_ERR("exceed maxsize (%zu>%zu)\n",dgram_max, DGRAM_MAX);
      return NULL;
   }

    switch(addr.sin_family) 
	{
      case AF_INET:
      case AF_INC:
         break;
      default:
         DG_ERR("not supported on AF %d\n",addr.sin_family);
         return NULL;
	}
   
   skd = (sk_dgram *) malloc(sizeof(sk_dgram));
	
	if(!skd) {
      DG_ERR("failed allocating memory\n");
      return NULL;
	}
   
	memset(skd, 0,sizeof(*skd));
	skd->proto = addr.sin_family;
	skd->hlen = sizeof(dgram_header_std);
	skd->len = dgram_max;

   skd->buf = (char *) malloc(skd->len+skd->hlen);

   if(!skd->buf) {
      DG_ERR("failed allocating rcv buffer\n");
      free(skd);
      return NULL;
   }
   
   skd->sk = sk;
   /*ignore options*/
   UNUSED_VARIABLE(options);

   return skd;
}

/*****************************************************************
uint32_t dgram_exit(sk_dgram *skd)

* PURPOSE : Freeing the dgram structure created
*
* RETURN :  Error code, if sk_dgram is invalid or success otherwise
*          
* Author :  

*****************************************************************/
int dgram_exit(sk_dgram *skd)
{
	if(!skd) {
		DG_ERR("Invalid datagram handle\n");
		errno = EINVAL;
		return -1;
	}

 	free(skd->buf);
	skd->buf = NULL;

	free(skd);
 	return 0;
}

/*****************************************************************
ssize_t dgram_send(sk_dgram *skd, void *ubuf, size_t ulen)

* PURPOSE : Send message as datagram, just prepend header,
* for GEN4 use 
*
* RETURN :  
*                       
*          
* Author :  

*****************************************************************/
ssize_t dgram_send(sk_dgram *skd, void *ubuf, size_t ulen)
{
   struct msghdr msg;
   struct iovec iov[2];
   dgram_header_std h;
   ssize_t ret = 0; /* Same type as return function */
   /* Zero the errno before any system call */
   errno = 0;

#ifdef MSG_PROTOCOL
   unsigned short lun, signal_id;
   unsigned int pduid = 0;
   struct sockaddr_in sin;
#endif

   if(!skd) 
   {
      DG_ERR("Invalid datagram handle\n");
      errno = EINVAL;
      return -1;
   }
 
   if(skd->proto != AF_INET) {
      return send(skd->sk, ubuf, ulen, 0);
   }
   
   if(ulen > skd->len) { /* satisfy lint */
      DG_ERR("send: dgram exceeds bufsize (%zu>%zu)\n", ulen, skd->len);
      errno = EMSGSIZE;
      return -1;
   }
   
   memset(&msg, 0, sizeof(msg));
   msg.msg_iovlen = 2;
   msg.msg_iov = iov;
   
#ifdef MSG_PROTOCOL
      /* Header to include the PDU ID and message length.
	  PDU ID is first byte of message followed by LUN */
	socklen_t len = sizeof(sin);
	if (getsockname(skd->sk, (struct sockaddr *)&sin, &len) == -1) {
	   DG_ERR("invalid handle\n");
	   return -1;
	}
	/* htonl expects uint32 cast the ulen variable*/
	h.dglen = htonl((uint32_t)ulen);
	lun = ((sin.sin_port & 0XFF00) >> 8);
	signal_id = (uint16_t)*(char *)ubuf;
	pduid = (lun);
	pduid =(pduid << 16) | signal_id;
	h.pduid =htonl( pduid);
	/* After each systemcall zero the errno */
	errno = 0;
#else
   h.dglen = htons((uint16_t)ulen);
#endif
   
   iov[0].iov_base = &h;
   iov[0].iov_len = sizeof(h);
   iov[1].iov_base = ubuf;
   iov[1].iov_len = ulen;
  

  /* On success, sendmsg return the number of characters sent.
   * On error, -1 is returned, and errno is set accordingly
   */
   ret = sendmsg(skd->sk, &msg, 0);
   if(ret < 0)
	   return ret;
   else {
		/* Sizeof return size_t no need for casting */
		if(ret >= (ssize_t)sizeof(h)) /* satisfy lint */
			ret -= (ssize_t)sizeof(h);
   }
  

   return ret;
}
/*****************************************************************
ssize_t dgram_recv(sk_dgram *skd, void* ubuf, size_t ulen)

* PURPOSE : Messaging protocol to return the atomic message data
* based on the size added in the message header. For GEN4 it is not required
* to loop to obtain the full sized data required.
*
* RETURN : On error return -1, check the errno to check the errors
		   On Success return the number of 
*                       
* NOTES :   
*          
* Author :  

*****************************************************************/
ssize_t dgram_recv(sk_dgram *skd, void* ubuf, size_t ulen)
{
	size_t totalrd = 0;
	ssize_t rd = 0;
	/* Set errno to zero before calling the system call */	
    errno = 0;

    if(!skd) {
        DG_ERR("Invalid datagram handle\n");
        errno = EINVAL;
        return -1;
    }
	
	if (skd->proto != AF_INET)
      return recv(skd->sk, ubuf, ulen, 0);

#ifdef MSG_PROTOCOL
                dgram_header_std *h;
                size_t size = 0;
				
                /* Use two system calls to receive the stream of data in packets
				 * the first receive is used to retreive the header information
				 * based on the size placed in the header, the second receive 
				 * is called receive the data based on the size requested.
				 */
                rd = recv(skd->sk , skd->buf , sizeof(dgram_header_std) , 0);
				/* recv returns either 0, -1 or the number of bytes received */
                if (rd <= 0) {
                               if( errno == EAGAIN) {
									DG_DBG("Can't receive data now, try again later  \n");
                               } else {
									DG_ERR("Receive failed with err(%d) rd(%zd)\n", errno, rd);
                               }
							   /* If an issue occured with the first receive return here */
                               return -1;
                }
				/* if we successfully received data we are here */
                h = (dgram_header_std *)skd->buf;
                /* The size of the data included in the buffer */
                size = ntohl(h->dglen);
				/* Print the size after the ntohl conversion */
                DG_DBG("Size of message as per header : %d\n", size);

                if (size > DGRAM_MAX || size > ulen) {
                               DG_ERR("Maximum size of message exceeded \n");
                               errno = EMSGSIZE;
                               return -1;
                }
                /*append buffer by the size of the initial header*/
                totalrd = sizeof(dgram_header_std);
                
                /* Retrieve all the size of data requested in the buffer header */
                 do {
                        rd = recv(skd->sk, (skd->buf)+totalrd, size , 0);

                        DG_DBG("dgram_recv() => recv() returns with = %zd\n", rd);
                                 
                        if (rd < 0) {
                            if (errno == EAGAIN) {
                                DG_DBG("Can't receive data now, try again later \n");
                            } else {
                                DG_DBG("Receive failed with err% d \n", errno);
                            }
                                return -1;
                        } else {
							/* If we are here we are sure that rd is > 0 and it
							 * is okay to cast it
							 * Only decrement these flags incase of successful transmission
							 */
							size -= (size_t)rd;
                               
							totalrd += (size_t)rd;
                        }
						/* Set errno to zero before calling the system call */	
						errno = 0;
                } while( size > 0);
  
   totalrd-=sizeof(dgram_header_std);
   memcpy(ubuf, (skd->buf)+sizeof(dgram_header_std), totalrd);

   
   DG_DBG("finished DGRAM of len %zu\n", totalrd);
#else
   
   if ((skd->received < (ssize_t) skd->hlen) || ((skd->received >= (ssize_t) skd->hlen) && (skd->received < (ssize_t) (skd->h.dglen+skd->hlen)))) {
      rd = recv(skd->sk, skd->buf+skd->received, (skd->len + skd->hlen)-(size_t)skd->received, 0);
      if(rd <= 0) {
         DG_DBG("receive failed with err %zd %d \n",rd, errno);
         return rd;
      }
      skd->received += rd;
   }

   if (skd->received < (ssize_t) skd->hlen) {
      errno = EAGAIN;
      return -1;
   }

   /*complete header available*/
   memcpy((char*)&skd->h, skd->buf, skd->hlen);
   skd->h.dglen = ntohs(skd->h.dglen);

   if (skd->received < (ssize_t) (skd->h.dglen+skd->hlen )) {
      errno = EAGAIN;
      return -1;
   }

   /*full dgram available*/
   if (ulen < skd->h.dglen) {
      DG_ERR("recv: dgram exceeds bufsize (%d>%zu)\n", skd->h.dglen, ulen);
      errno = EMSGSIZE;
      return -1;
   }

   memcpy(ubuf, skd->buf+skd->hlen, skd->h.dglen);
   
   skd->received-= (ssize_t)(skd->h.dglen+skd->hlen);
   
   rd = (size_t)skd->h.dglen;
   
   if(skd->received > 0 ) {
      memmove(skd->buf, skd->buf+skd->hlen+skd->h.dglen, (size_t)skd->received);
      if(skd->received >= (ssize_t) skd->hlen) {
         memcpy((char*)&skd->h, skd->buf, skd->hlen);
         skd->h.dglen = ntohs(skd->h.dglen);
      }
   }
   totalrd = (size_t)rd;
   DG_DBG("finished DGRAM of len %zd\n", rd);

#endif

	/* Casting is required to prevent compiler warnings for the size conversions */
   return (ssize_t)totalrd;

}

#ifdef __cplusplus
}
#endif
