/*****************************************************************************************
* Copyright (C) RBEI, 2019
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
******************************************************************************************/
/*******************************************************************************
*
* FILE			:	writeprofiler_comm.c
*
* SW-COMPONENT	:	eMMC Write profiler
*
* PROJECT		:	This will be common for all the projects.
*
* DESCRIPTION	: This contains the implementation to offer the message
* 				  communication and event notification functionalities

* AUTHOR		:	Ravindra Prabha (RBEI/ECF1)
*
* COPYRIGHT:    (c) 2019 Robert Bosch GmbH, Hildesheim
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*27.Sep.2019| Initial version 1.0 | Ravindra Prabha (RBEI/ECF1) & Code refactoring
* -----------------------------------------------------------------------
* -----------------------------------------------------------------------

/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "writeprofiler.h"
/******************************************************************************/

/********************************************************************************
* FUNCTION		  : createEvent
* PARAMETER 	  : pevHandle	: Address of event peventObj
* RETURNVALUE	  : 0 : Success ;
* 				   -1 : Failure ;
* DESCRIPTION	  : Creates the new event object
**********************************************************************************/
tS32 createEvent(sem_t *peventObj)
{
	tS32 ret = -1;
	if (peventObj) {
		ret = sem_init(peventObj, 0, 0);
	}
	return ret;
}

/******************************************************************************
* FUNCTION		  : postEvent
* PARAMETER 	  : eventObj	: Address of event peventObj
* RETURNVALUE	  : 0 : Success ;
* 				   -1 : Failure ;
* DESCRIPTION	  : posts the new event
******************************************************************************/
tS32 postEvent(sem_t *peventObj)
{
tS32 ret = -1;
	if (peventObj) {
		ret = sem_post(peventObj);
	}
	return ret;
}
/******************************************************************************
* FUNCTION		  : waitEvent
* PARAMETER 	  : eventObj	: Address of event peventObj
* 					timeout : timeout for the event wait (in msec)
* 						if timeout = -1,it is infinite timeout
* RETURNVALUE	  : 0 : Success
* 					1 : Timeout , Timeout need not be an error as per design
* 				   -1 : Failure
* DESCRIPTION	  : waits the new event
******************************************************************************/
tS32 waitEvent(sem_t *peventObj,tS32 timeout)
{
tS32 ret = -1;
struct	 timespec tm;

	if (peventObj) {
		if (-1 != timeout) {
			if(clock_gettime(CLOCK_REALTIME, &tm) != 0)	{
				ERROR("\n clock_gettime_error...errno=%d \n",errno);
			}
			/* timeout is in msec*/
			tm.tv_sec = tm.tv_sec + (timeout / 1000) ;
			tm.tv_nsec = tm.tv_nsec + ((timeout % (1000)) * (1000 * 1000));

			/*adjust tv_nsec value if goes beyond 1000Million val*/
			if (tm.tv_nsec >= NUM_NANOSEC_PER_SEC)	{
				tm.tv_sec = tm.tv_sec + tm.tv_nsec/NUM_NANOSEC_PER_SEC;
				tm.tv_nsec = tm.tv_nsec % NUM_NANOSEC_PER_SEC;
			}

			ret = sem_timedwait(peventObj, &tm);
			if ((ret < 0))	{
				if (errno != ETIMEDOUT)	{
					ERROR("!!ERROR:sem_timedwait Failed...errno=%d \n",errno);
					ret = -1;
				}
				else {
					/* if it is timed-out, then return 1*/
					ret = 1;
				}
			}
		}
		else {
			ret = sem_wait(peventObj);
		}
	}
	else {
		ERROR("!!Invalid argument in function %s \n",__FUNCTION__);
	}
	return ret;
}

/******************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_CreateMsgObj
* PARAMETER 	  : pevt_q	: Address of message object
* 					name : name of the message queue
* RETURNVALUE	  : void
* DESCRIPTION	  : Creates the message queue with the requested name
* 					upon successful operation,the message queue handle is
* 					stored into the provided "pevt_q" object
* 					Upon error,-1 value is stored into the pevt_q
******************************************************************************/
void eMMC_ExcessiveWP_CreateMsgObj(mqd_t *pevt_q,const char *name)
{
	struct mq_attr attr;
	mqd_t mq;

	if ((NULL == name) || (NULL == pevt_q)) {
		ERROR("!!ERROR:eMMC_ExcessiveWP_CreateMsgObj invalid argument \n");
		return;
	}

	/*
	Maximum 10 message are sufficient enough because queuing of 10 events
	is very unlikely to happen
	message size 4 is enough as it used to post the events only
	*/
	attr.mq_maxmsg  = 10;
	attr.mq_msgsize = sizeof(tU32);
	attr.mq_flags   = 0;
	attr.mq_curmsgs = 0;
    (void)mq_unlink(name);

	mq = mq_open(name,O_CREAT | O_RDWR,S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH ,&attr);

	if(mq < 0) {
		ERROR ("mq_open error for msag name %s errno=%d\n:",name,errno);
	}

	/*
		The caller function shall check the value of pevt_q and decides whether
		the message is created successfully or not
	 */
	*pevt_q = mq;
}

/******************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_SendMsg
* PARAMETER 	  : evt_q	: message queue object
* 					data : data to send
* 					timeout : timeout for the message send (in msec)
* 						if timeout = -1,it is infinite timeout
* RETURNVALUE	  : 0 : Success
* 					1 : Timeout , Timeout need not be an error as per design
* 				   -1 : Failure
* DESCRIPTION	  : sends (posts) the message to the message queue
* 					The client who waits for the message on the queue shall
* 					be woken up.
******************************************************************************/
tS32 eMMC_ExcessiveWP_SendMsg (mqd_t evt_q,tU32 data,tS32 timeout)
{
tS32 ret;
tU32 prio = 1;
struct	 timespec tm;

	if (-1 != timeout) {
		clock_gettime(CLOCK_REALTIME, &tm);

		/* timeout is in msec*/
		tm.tv_sec = tm.tv_sec + (timeout / 1000);
		tm.tv_nsec = tm.tv_nsec + ((timeout % (1000)) * (1000 * 1000));

		/*adjust tv_nsec value if goes beyond 1000Million val*/
		if (tm.tv_nsec >= NUM_NANOSEC_PER_SEC)	{
			tm.tv_sec = tm.tv_sec + tm.tv_nsec/NUM_NANOSEC_PER_SEC;
			tm.tv_nsec = tm.tv_nsec % NUM_NANOSEC_PER_SEC;
		}

		ret = mq_timedsend(evt_q,(tChar*)&data,sizeof(tU32),prio,&tm);
		if ((ret < 0))
		{
			if (errno != ETIMEDOUT)	{
				ERROR("!!ERROR:mq_timedsend %s Failed...errno=%d \n",EXCESSIVEWP_EVTQUEUE_NAME,errno);
			}
			else {
				/* Timeout need not be an error as per design */
				ret = 1;
			}
		}
	}
	else {
		ret = mq_send(evt_q,(tChar*)&data,sizeof(tU32),prio);
		if ((ret < 0))	{
			ERROR("!!ERROR:mq_send %s Failed...errno=%d \n",EXCESSIVEWP_EVTQUEUE_NAME,errno);
		}
	}

return ret;
}

/******************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_ReceiveMsg
* PARAMETER 	  : evt_q	: message queue object
* 					data : data buffer to stored the received message
* 					timeout : timeout for the message send (in msec)
* 						if timeout = -1,it is infinite timeout
* RETURNVALUE	  : 0 : Success
* 					1 : Timeout , Timeout need not be an error as per design
* 				   -1 : Failure
* DESCRIPTION	  : receives the message in the message queue
* 					The client who waits for the message on the queue shall
* 					be woken up when the message is posted in the queue
******************************************************************************/
tS32 eMMC_ExcessiveWP_ReceiveMsg (mqd_t evt_q,tU32 *data,tS32 timeout) //timeout in msec
{
tS32 ret;
tU32 prio = 1;
struct	 timespec tm;

	if (-1 != timeout) {
		if(clock_gettime(CLOCK_REALTIME, &tm)!=0)
		{
			ERROR("\n clock_gettime_error in function %s\n",__FUNCTION__);
		}
		/* timeout is in msec*/
		tm.tv_sec = tm.tv_sec + (timeout / 1000) ;
		tm.tv_nsec = tm.tv_nsec + ((timeout % (1000)) * (1000 * 1000));

		/*adjust tv_nsec value if goes beyond 1000Million val*/
		if (tm.tv_nsec >= NUM_NANOSEC_PER_SEC)
		{
			tm.tv_sec = tm.tv_sec + tm.tv_nsec/NUM_NANOSEC_PER_SEC;
			tm.tv_nsec = tm.tv_nsec % NUM_NANOSEC_PER_SEC;
		}
		ret = mq_timedreceive(evt_q,(tChar*)data,sizeof(tU32),&prio,&tm);

		if ((ret < 0))	{
			if (errno != ETIMEDOUT)	{
				ERROR ("ERROR:mq_receive %s Failed...errno=%d in function %s\n",EXCESSIVEWP_EVTQUEUE_NAME,errno,__FUNCTION__);
				return -1;
			}
			else {
				/* if it is timed-out, then return 1*/
				return 1;
			}
		}
	}
	else {
		ret = mq_receive(evt_q,(tChar*)data,sizeof(tU32),&prio);
		if ((ret < 0))	{
			ERROR ("ERROR--:mq_receive %s Failed...errno=%d in function %s\n",EXCESSIVEWP_EVTQUEUE_NAME,errno,__FUNCTION__);
			exit(0);
			return -1;
		}
	}
	return 0;
}
/******************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_DeleteMsgObj
* PARAMETER 	  : pevt_q	: Address of message object
* 					name : name of the message queue
* RETURNVALUE	  : 0 : Success
* 				   -1 : Failure
* DESCRIPTION	  : Deletes the message object. This function is invoked during
*					the resource cleanup
******************************************************************************/
tS32 eMMC_ExcessiveWP_DeleteMsgObj (mqd_t evt_q,const char*name)
{
tS32 ret;

	if (NULL == name) {
		ERROR("!!ERROR:eMMC_ExcessiveWP_DeleteMsgObj invalid argument \n");
		return -1;
	}

	ret = mq_close(evt_q);
	if (-1 == ret) {
		ERROR("!!ERROR:mq_close %s Failed...errno=%d in function %s\n",name,errno,__FUNCTION__);
		return -1;
	}
	ret = mq_unlink(name);
	if (-1 == ret) {
		ERROR("!!ERROR:mq_unlink %s Failed...errno=%d in function %s\n",name,errno,__FUNCTION__);
		return -1;
	}
	return ret;
}


