/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         writeprofilerAppComponent.cpp
  * @brief        This file contains writeprofiler Application Implemetation.
  * @copyright    (C) 2013 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

/***************************************************************************************************
 **                                   Includes                                                     **
 ***************************************************************************************************/
extern "C" {
int eMMC_ExcessiveWrite_main (int argc, char** argv);
int eMMC_ExcessiveWP_LoadPersistency (  void );
int eMMC_ExcessiveWP_SavePersistency (  void );
}

#include "writeprofilerAppComponent.h"

/***************************************************************************************************
 **                                   Namespaces                                                   **
 ***************************************************************************************************/

namespace excessive_write { namespace sources {



using namespace ::org::genivi::NodeStateManager::Consumer;
using namespace ::org::genivi::NodeStateManager::LifeCycleConsumer;
using namespace ::org::genivi::NodeStateManager::LifecycleControl;
using namespace ::org::bosch::cm::lcm;
using namespace ::org::bosch::cm::lcm::lcmbaseappcomponent;
using namespace ::asf::core;



/***************************************************************************************************
 **                                    Variable                                                    **
 ***************************************************************************************************/

/***************************************************************************************************
 **                                   Function                                                     **
 ***************************************************************************************************/

/**
  **************************************************************************************************
  * \brief Constructor of your application
  *
  * \param[in] config
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
writeprofilerAppComponent::writeprofilerAppComponent( )
// ASF Base
   : BaseComponent( ),
   _lcmAppComponent( /*busname*/ "excessive_write.sources.writeprofilerApp",
/*busname == <package name as set in your cma file>.<application name>*/
                                 /*objname*/ "/org/genivi/NodeStateManager/LifeCycleConsumer/writeprofilerAppComponent",
/*objname == <object name given in your cma file at exportByName for the LifeCycleConsumerPort */
                                 * this ){
   LOG_INFO( "writeprofilerAppComponent called" );

   /* you can use the following function to get the command line arguments
      getCommandLineArguments()*/


    /*eMMC_ExcessiveWrite_main sets up the infrastructure of the excessive write component, therefore it is called before_lcmAppComponent.setAppStartReady( )*/

   eMMC_ExcessiveWrite_main(0,NULL);

   /* call function setAppStartReady() whenever the application is fully started and dependend apps
      can be started by systemd.
      ATTENTION: This must not necessarily be done here. Any other place e.g. onAvailable might be
      more suitable for your application
     */
   _lcmAppComponent.setAppStartReady( );

}

/**
  **************************************************************************************************
  * \brief Destructor of your application
  *
  * \param[in] None
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
writeprofilerAppComponent::~writeprofilerAppComponent( ){
   LOG_INFO( "~writeprofilerAppComponent called" );
}

// ServiceAvailableIF implementation

/**
  **************************************************************************************************
  * \brief this function is called from the ASF framework if any of your services that you have a
  *       proxy created for is available
  *
  * \param[in] proxy - shared pointer to the proxy that the service is now available for
  * \param[in] stateChange - contains the old and the new state of the service (see /home/rmi2hi/ASF_Git/apollo/src/cpp/asf/src/asf/core.cmt)
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onAvailable( const boost::shared_ptr < Proxy >& proxy,
                                       const ServiceStateChange         & stateChange ){
   if ( proxy && stateChange.hasCurrentState( ) ){
      LOG_INFO( "writeprofilerAppComponent::onAvailable: counsumer proxy service available, ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
   } else {
      LOG_INFO( "writeprofilerAppComponent::onAvailable:: consumer proxy service not available" );
   }

/****************************
  *   Example How-To
   if(proxy == _yourProxy)
   {
      LOG_INFO("_yourProxy service now available");
   }
  *****************************/
}

/**
  **************************************************************************************************
  * \brief this function is called from the ASF framework if any of your services that you have a
  *       proxy created for is not available anymore
  *
  * \param[in] proxy - shared pointer to the proxy that the service is now not available anymore
  * \param[in] stateChange - contains the old and the new state of the service (see /home/rmi2hi/ASF_Git/apollo/src/cpp/asf/src/asf/core.cmt)
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onUnavailable( const boost::shared_ptr < Proxy >& proxy,
                                         const ServiceStateChange         & stateChange ){
   if ( proxy && stateChange.hasCurrentState( ) ){
      LOG_INFO( "writeprofilerAppComponent::onUnavailable: counsumer proxy service not available anymore, ServiceState: %s", ServiceState_Name( stateChange.getCurrentState( ) ) );
   } else {
      LOG_INFO( "writeprofilerAppComponent::onUnavailable:: unknown service" );
   }

/****************************
  *   Example How-To
   if(proxy == _yourProxy)
   {
      LOG_INFO("_yourProxy service not available anymore");
   }
  *****************************/
}

// TimerCallbackIF implementation

/**
  **************************************************************************************************
  * \brief this function is called from the ASF framework if any of your timers has expired.
  *
  * \param[in] timer - shared pointer to the timer that has expired. Check the ACT
  * (ACT = Asynchronous Completion Token)
  * \param[in] payload - contains data about timer that has run out (see /home/rmi2hi/ASF_Git/apollo/src/cpp/asf/src/asf/core.cmt)
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onExpired( asf::core::Timer                            & timer,
                                     boost::shared_ptr < asf::core::TimerPayload > payload ){
   LOG_INFO( "writeprofilerAppComponent::onExpired::Timer:%u", timer.getAct( ) );

   if ( payload->hasReason( ) && payload->hasRepeatCount( ) ){
      LOG_INFO( "writeprofilerAppComponent::onExpired: TimerPayload Reason: %s, Repeat Count:%u", TimerPayload_Reason_Name( payload->getReason( ) ), payload->getRepeatCount( ) );
   }

/****************************
  *   Example How-To
   if( (timer.getAct() == _yourTimer.getAct())
    && (&timer         == &_yourTimer )
     ) {
      LOG_INFO("_yourTimer expired");
   }
  *****************************/
}

// LCM Interface implementation

/**
  **************************************************************************************************
  * \brief this function is called from the LcmBaseAppComponent if a new application mode is requested from LCM
  *        here the transition actions need to be done
  *        This function is mandatory
  *
  * \param[in] newAppMode - new application mode that is requested
  * \param[in] oldAppMode - previous application mode
  * \param[out] None
  * \param[inout] None
  * \return error code
  * typedef enum _NsmErrorStatus_e
  * {
  *    NsmErrorStatus_NotSet,               *< Initial value when error type is not set             *
  *    NsmErrorStatus_Ok,                   *< Value when no error occurred                         *
  *    NsmErrorStatus_Error,                *< A general, non-specific error occurred               *
  *    NsmErrorStatus_Dbus,                 *< Error in D-Bus communication                         *
  *    NsmErrorStatus_Internal,             *< Internal error (memory alloc. failed, etc.)          *
  *    NsmErrorStatus_Parameter,            *< A passed parameter was incorrect                     *
  *    NsmErrorStatus_WrongSession,         *< The requested session is unknown.                    *
  *    NsmErrorStatus_ResponsePending,      *< Command accepted, return value delivered asynch.     *
  *    NsmErrorStatus_Last                  *< Last error value to identify valid errors            *
  * } NsmErrorStatus_e;
  *
  **************************************************************************************************
  */
_NsmErrorStatus_e writeprofilerAppComponent::OnAppModeChange( _Nsm_Shutdown_Type_e newAppMode,
                                                        _Nsm_Shutdown_Type_e oldAppMode ){
   _NsmErrorStatus_e returnVal = _NsmErrorStatus_e__NsmErrorStatus_NotSet;

   LOG_INFO( "writeprofilerAppComponent::OnAppModeChange -NewAppMode=%s, OldAppMode=%s", _Nsm_Shutdown_Type_e_Name( newAppMode ), _Nsm_Shutdown_Type_e_Name( oldAppMode ) );
   if ( newAppMode == _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup ){
      // goto Normal Mode
      returnVal = _NsmErrorStatus_e__NsmErrorStatus_Ok; // if successfull
   } else if ( newAppMode == _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal ){
      // goto OFF Mode
      returnVal = _NsmErrorStatus_e__NsmErrorStatus_Ok; // if successfull
   }

   return( returnVal );
}

/**
  **************************************************************************************************
  * \brief this function is called from the LCMBaseApp if the application watchdog timer has run out
  *        This function is mandatory
  *
  * \param[in]
  * \param[out] None
  * \param[inout] None
  * \return TRUE  - all is OK
  *         FALSE - some inconsistency in the application state was detected
  *
  **************************************************************************************************
  */
bool writeprofilerAppComponent::OnAppWatchdog( ){
   bool returnVal = false;

/*   LOG_INFO( "OnAppWatchdog ..........." );*/

/****************************
  *   Example How-To
   // check consistency of your app here
  *****************************/
   returnVal = true; // only if all is OK
   return( returnVal );
}

/**
  **************************************************************************************************
  * \brief this function is called from the LCMBaseApp if all LCM services have been registered
  *
  * \param[in]
  * \param[out] None
  * \param[inout] None
  * \return none
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::OnLcmRegistered( ){
   LOG_INFO( "OnLcmRegistered ..........." );

/****************************
  *   Example How-To
   // now all SPM services are available
  *****************************/
}

/**
  **************************************************************************************************
  * \brief this function is called from the LCMBaseApp before the App change request to RunUp
  *
  * \param[in]
  * \param[out] None
  * \param[inout] None
  * \return none
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onLoadPersistency( ){
   LOG_INFO( "OnAppWatchdog ..........." );

	eMMC_ExcessiveWP_LoadPersistency();
/****************************
  *   Example How-To
   // load your persistent data
  *****************************/
}

/**
  **************************************************************************************************
  * \brief this function is called from the LCMBaseApp after the App change request to Shutdown
  *
  * \param[in]
  * \param[out] None
  * \param[inout] None
  * \return none
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onSavePersistency( ){
   LOG_INFO( "OnAppWatchdog ..........." );

	eMMC_ExcessiveWP_SavePersistency();
/****************************
  *   Example How-To
   // save your persistent data
  *****************************/
}

/**
  **************************************************************************************************
  * \brief This function is called when there is an error on LifecycleRequestComplete .
  *
  * \param[in] error
  * \param[in] error name
  * \param[in] error message
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::OnAppLifecycleRequestCompleteError( _NsmErrorStatus_e  error,
                                                              const std::string& name,
                                                              const std::string& message ){
   LOG_WARN( "writeprofilerAppComponent::OnAppLifecycleRequestCompleteError , Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

/**
  **************************************************************************************************
  * \brief This function is called when there is a response on LifecycleRequestComplete .
  *
  * \param[in]  response
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::OnAppLifecycleRequestCompleteResponse( _NsmErrorStatus_e response ){
   LOG_INFO( "writeprofilerAppComponent::OnAppLifecycleRequestCompleteResponse, Response=%s", _NsmErrorStatus_e_Name( response ) );
}

/**
  **************************************************************************************************
  * \brief This function is called when there is any error on RegisterShutdownClient .
  *
  * \param[in] error
  * \param[in] error name
  * \param[in] error message
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onAppRegisterShutdownClientError( _NsmErrorStatus_e  error,
                                                            const std::string& name,
                                                            const std::string& message ){
   LOG_WARN( "writeprofilerAppComponent::onAppRegisterShutdownClientError, Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

/**
  **************************************************************************************************
  * \brief This function is called when there is a response on RegisterShutdownClient .
  *
  * \param[in]  response
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onAppRegisterShutdownClientResponse( _NsmErrorStatus_e response ){
   LOG_INFO( "writeprofilerAppComponent::onAppRegisterShutdownClientResponse , Response=%s", _NsmErrorStatus_e_Name( response ) );
}

/**
  **************************************************************************************************
  * \brief This function is called when there is any error on UnRegisterShutdownClient .
  *
  * \param[in] error
  * \param[in] error name
  * \param[in] error message
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onAppUnRegisterShutdownClientError( _NsmErrorStatus_e  error,
                                                              const std::string& name,
                                                              const std::string& message ){
   LOG_WARN( "writeprofilerAppComponent::onAppUnRegisterShutdownClientError, Error=%s, Name=%s, Message=%s", _NsmErrorStatus_e_Name( error ), name.c_str( ), message.c_str( ) );
}

/**
  **************************************************************************************************
  * \brief This function is called when there is a response on UnRegisterShutdownClient .
  *
  * \param[in]  response
  * \param[out] None
  * \param[inout] None
  * \return None
  *
  **************************************************************************************************
  */
void writeprofilerAppComponent::onAppUnRegisterShutdownClientResponse( _NsmErrorStatus_e response ){
   LOG_INFO( "writeprofilerAppComponent::onAppUnRegisterShutdownClientResponse, Response=%s", _NsmErrorStatus_e_Name( response ) );
}

DEFINE_CLASS_LOGGER_AND_LEVEL( "excessive_write.sources.writeprofilerAppComponent", writeprofilerAppComponent, Info );

}
}
