/*****************************************************************************************
* Copyright (C) RBEI, 2017
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
******************************************************************************************/
/*******************************************************************************
*
* FILE			:	write_profiler.c
*
* SW-COMPONENT	:	eMMC Write profiler
*
* PROJECT		:	This will be common for all the projects.
*
* DESCRIPTION	:	This contains the implementation to identify the excessive
*					write on eMMC device. Due to the some improper application NVM
*					date write handling, device life is ended before the designed life
*					of the device. This write profiler can identify application which doing
*					unwanted write to the eMMC device.
*               Algorithm followed here is as follows:
*					- Read the configuration file. Configuration file contains device or container
*					  to monitor, how long the device has to be monitor and the maximum limit for the
*					  respective duration.
*					- To monitor container and Native system:
*					  read the block write count from the cgroup and compared with the respective limit.
*					- To monitor the eMMC partition:
*					  read the block write count from /proc/diskstats file and compared with the respective limit.
*					- Excessive write notification is post to D-Bus.
*
* AUTHOR		:	Balaji Vishwanadhula (RBEI/ECF2)
*					Selvakumar Kalimuthu (RBEI/ECF2)
*
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*10.July.2017| Initial version 1.0 | SELVAKUMAR KALIMUTHU (RBEI/ECF2)
* -----------------------------------------------------------------------
*-----------|---------------------|--------------------------------------
*10.July.2017| Initial version 1.1 | SELVAKUMAR KALIMUTHU (RBEI/ECF2)| Find Container path in Group logic cahnged.
* -----------------------------------------------------------------------
*-----------|---------------------|--------------------------------------
*04.Dec.2018| 			version 1.2 | SELVAKUMAR KALIMUTHU (RBEI/ECF2)| Bug fixes.
* -----------------------------------------------------------------------
*-----------|---------------------|--------------------------------------
*04.Oct.2019| 			version 1.3 | Ravindra Prabha (RBEI/ECF1)| Bug fixes and code refactor.
* -----------------------------------------------------------------------

*******************************************************************************/
/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "writeprofiler.h"
/******************************************************************************/
/*                                                                            */
/* Private Variable                                                           */
/*                                                                            */
/******************************************************************************/
static MonitorThreadState AppInterryptedBy = PROFILER_THREAD_RUNNING;
static pthread_t	HndleWatchThrd;
static monitor_global_info 	*stcfgInfo= NULL;

/******************************************************************************/
/*                                                                            */
/* LOCAL FUNCTIONS                                                            */
/*                                                                            */
/******************************************************************************/
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_SignalHandler
* PARAMETER 	  :
* RETURNVALUE	  :
* DESCRIPTION	  : Signal handler
**********************************************************************************/
tVoid eMMC_ExcessiveWP_SignalHandler(tS32 signo)
{
	DEBUG("Signal Got %d\n",signo);
	AppInterryptedBy = PROFILER_THREAD_SIGNAL_COUGHT;
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_SignalHandlerSetup
* PARAMETER 	  :
* RETURNVALUE	  :
* DESCRIPTION	  : Register the required signal handler.
**********************************************************************************/
tVoid eMMC_ExcessiveWP_SignalHandlerSetup(tVoid)
{

	 if (signal(SIGUSR2, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGUSR2\n");

	 if (signal(SIGABRT, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGABRT\n");

	 if (signal(SIGSEGV, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGSEGV\n");

	 if (signal(SIGHUP, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGHUP\n");

	 if (signal(SIGTERM, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGTERM\n");

	 if (signal(SIGINT, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGINT\n");

	 if (signal(SIGUSR1, eMMC_ExcessiveWP_SignalHandler) == SIG_ERR)
		 DEBUG("\ncan't catch SIGUSR1\n");
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_AllocateMemory
* PARAMETER 	  : size : Size to allocate dynamic memory
* RETURNVALUE	  : Allocated memory handler
* DESCRIPTION	  : Allocation dynamic memory and initializes to 0 for application usage.
**********************************************************************************/
tPVoid	eMMC_ExcessiveWP_AllocateMemory ( size_t size )
{
char *p = malloc(size);
	if(p) {
		memset(p,0,size);
	}
	return (tPVoid)p;
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_DeallocateMemory
* PARAMETER 	  : ptr : Handle to free the allocated memory area
* RETURNVALUE	  :
* DESCRIPTION	  : Deallocation dynamic memory used by application
**********************************************************************************/
void	eMMC_ExcessiveWP_DeallocateMemory ( tPVoid ptr )
{
	if( ptr != NULL )
	{
		free (ptr);
	}
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_DeleteRingBuffer
* PARAMETER 	  : blkioLocationInfo	: Location details which read from cfg file
* RETURNVALUE	  :
* DESCRIPTION	  : Delete the ring buffers
**********************************************************************************/
void eMMC_ExcessiveWP_DeleteRingBuffer(monitor_blkio *blkioLocationInfo)
{
	node *rmNode,*rootNode;

	if( blkioLocationInfo->ringnode != NULL )
	{
		rootNode	= blkioLocationInfo->ringnode;

		do
		{
			rmNode = rootNode->prev;
            /*Check only if one node remains to be removed */
            if( rootNode == rmNode )
            {
                rootNode = NULL;
                blkioLocationInfo->ringnode = NULL;
            }
            else
            {
                /*
                Isolate the rmnode from the list by disconnecting
                the next of its previous and previous of its next
                */
    			rootNode->prev = rmNode->prev;

                /*
                 NULL check for (rmNode->prev) not necessary,because we are in
                 condition that rmNode->prev never be NULL
                */
                ((node *)(rmNode->prev))->next = rootNode;
            }
			eMMC_ExcessiveWP_DeallocateMemory(rmNode);
		} while ( rootNode != NULL );
	}
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_DeleteMemory
* PARAMETER 	  : stcfgInfo: This will hold all the application details.
* RETURNVALUE	  :
* DESCRIPTION	  : Deallocation all the allocated memory by this application
**********************************************************************************/
tVoid eMMC_ExcessiveWP_DeleteMemory ( monitor_global_info	*stcfgInfo )
{
	monitor_blkio 			*blkio,*blkiotmp;
	write_profiler_params 	*monitor,*monitortmp;

	blkio = stcfgInfo->blkio;

	while ( blkio  != NULL )
	{
		if( blkio->ringnode != NULL )
		{
			eMMC_ExcessiveWP_DeleteRingBuffer(blkio);
		}
		monitor = blkio->monitor;
		while ( monitor != NULL )
		{
			monitortmp=monitor->next;
			eMMC_ExcessiveWP_DeallocateMemory(monitor);
			monitor = monitortmp;
		}
		blkiotmp = blkio->next;
		eMMC_ExcessiveWP_DeallocateMemory(blkio);
		blkio = blkiotmp;
	}

	eMMC_ExcessiveWP_DeallocateMemory(stcfgInfo);
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_PrintConfigInfo
* PARAMETER 	  : stcfgInfo: This will hold all the application details.
* RETURNVALUE	  :
* DESCRIPTION	  : Print the configuration file info
**********************************************************************************/
void eMMC_ExcessiveWP_PrintConfigInfo ( monitor_global_info	*stcfgInfo )
{
	monitor_blkio 			*blkio;
	write_profiler_params 	*monitor;

	blkio = stcfgInfo->blkio;

	DEBUG( "\nTotal Monitor Locations: %d\n",stcfgInfo->MonitorLocationCnt );
	while ( blkio  != NULL )
	{
		DEBUG( "\nMonitor Name : %s \n",blkio->unique_name);
		DEBUG( "Monitor Location : %s \n",blkio->location );

		monitor = blkio->monitor;

		DEBUG("\nTotal Trace Points : %d\n\n",blkio->TracePointCnt);
		DEBUG("\nMax Duration : %d\n\n",blkio->MaxDuration);
		while ( monitor != NULL )
		{
			DEBUG( "Trace Point Name : %s\n",monitor->name);
			DEBUG( "Trace Point Duration : %d\n",monitor->duration);
			DEBUG( "Trace Point Threshold : %lld\n",monitor->threshold);
			monitor = monitor->next;
		}
		DEBUG("\n");
		blkio = blkio->next;
	}
	DEBUG("\nPersistentDataLocation : %s\n",stcfgInfo->PersistentDataLocation);
	DEBUG("PersistentDataSaveCycle : %d\n\n",stcfgInfo->PersistentDataSaveCycle);
	DEBUG("ErrmemNotification : %d\n\n",stcfgInfo->ErrmemNotification);
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_CreateNode
* PARAMETER 	  :
* RETURNVALUE	  : Valid node structure handle
* DESCRIPTION	  : This will create the new mode for ringbuffer.
**********************************************************************************/
node* eMMC_ExcessiveWP_CreateNode( void )
{
	node *temp;

	temp=(node*)eMMC_ExcessiveWP_AllocateMemory(sizeof(node));
	if(temp == NULL)
	{
		ERROR("!!ERROR:Mem Fail Size<%lu>\n",sizeof(node));
		return NULL;
	}
	temp->next=NULL;
	temp->prev=NULL;

	return temp;
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_SyncData
* PARAMETER 	  : fptr: File pointer
* RETURNVALUE	  :
* DESCRIPTION	  : Sync the file system
**********************************************************************************/
void eMMC_ExcessiveWP_SyncData(FILE *fptr)
{
	tS32 file_ptr;
	fflush(fptr);

	file_ptr=fileno(fptr);
	if(file_ptr < 0)
	{
		(void)system("sync");
	}
	else
	{
		(void)fsync(file_ptr);
	}
}
/********************************************************************************
* FUNCTION		  : month
* PARAMETER 	  : mm (month number) and yy(year number)
* RETURNVALUE	  :
* DESCRIPTION	  :
**********************************************************************************/
tS32 month(tU32 mm,tU32 yy)
{
	tU32 x = 0;
	tU32 c;
	tU32 mon[12]={31,28,31,30,31,30,31,31,30,31,30,31};

	/* Just make sure month is within in range */
	if( mm <= 12) {
			for(c=0;c<mm-1;c++)
			{
				/* leap year handling for Feb month */
				if(c==1) {
					if(yy%4==0) {
						if (yy%100 == 0) {
							if (yy%400 == 0) {
								x = x + 29;
							}
							else {
								x = x + 28;
							}
						}
						else {
							x = x + 29;
						}
					}
					else {
						x = x + 28;
					}
				}
				else {
					x = x + mon[c];
				}
		}
	}
	return(x);
}
/********************************************************************************
* FUNCTION		  : days
* PARAMETER 	  :
* RETURNVALUE	  :
* DESCRIPTION	  :
**********************************************************************************/
tS32 days(tS32 y1,tS32 y2,tS32 m1,tS32 m2,tS32 d1,tS32 d2)
{
	tS32 count=0,yy;

	for(yy=y1;yy<y2;yy++)
	{
		if(yy%4 == 0) {
			if (yy%100 == 0) {
				if (yy%400 == 0) {
					count = count + 366;
				}
				else {
					count = count + 365;
				}
			}
			else {
				count = count + 366;
			}
		}
		else {
			count = count + 365;
		}
	}

	count = count - month(m1,y1);
	count = count - d1;
	count = count + month(m2,y2);
	count = count + d2;

	DEBUG("The no. of days b/w the 2 dates = %d days\n",count);

	return (count);
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_CompareDates
* PARAMETER 	  : last and current power on day.
* RETURNVALUE	  : count - difference b/w last and current power on day.
* DESCRIPTION	  : This will give the difference b/w last and current power on day.
**********************************************************************************/
tS32 eMMC_ExcessiveWP_CompareDates(struct tm previous_time,struct tm current_time)
{
	tS32 count;

	DEBUG("year:{C:%d==P:%d}\n",current_time.tm_year+PROFIER_SYSTEM_BASE_DATE,previous_time.tm_year+PROFIER_SYSTEM_BASE_DATE);
	DEBUG("Month:{C:%d==P:%d}\n",current_time.tm_mon,previous_time.tm_mon);
	DEBUG("Day:{C:%d==P:%d}\n",current_time.tm_mday,previous_time.tm_mday);

	if((current_time.tm_year+PROFIER_SYSTEM_BASE_DATE)>= (previous_time.tm_year+PROFIER_SYSTEM_BASE_DATE)) {
		count=days(	(previous_time.tm_year+PROFIER_SYSTEM_BASE_DATE),
					(current_time.tm_year+PROFIER_SYSTEM_BASE_DATE),
					(previous_time.tm_mon+1),
					(current_time.tm_mon+1),
					(previous_time.tm_mday),
					(current_time.tm_mday));
	}
	else {
		count=days(	(current_time.tm_year+PROFIER_SYSTEM_BASE_DATE),
					(previous_time.tm_year+PROFIER_SYSTEM_BASE_DATE),
					(current_time.tm_mon+1),
					(previous_time.tm_mon+1),
					(current_time.tm_mday),
					(previous_time.tm_mday));
	}

	if ( current_time.tm_year < previous_time.tm_year )
				count*=-1;

	if ( count < 0 )
	{
		DEBUG("Current Date is Updated wrongly-1 C<%d:%d:%d> P<%%d:%d;%d>\n",current_time.tm_mday,current_time.tm_mon,current_time.tm_year,previous_time.tm_mday,previous_time.tm_mon,previous_time.tm_year);
		if( (current_time.tm_year != previous_time.tm_year) &&
			(current_time.tm_year+PROFIER_SYSTEM_BASE_DATE == (PROFIER_SYSTEM_BASE_DATE+PROFIER_SYSTEM_BASE_YEAR)))
		{
			DEBUG("Current Date is Updated wrongly-2 C<%d:%d:%d> P<%%d:%d;%d>\n",current_time.tm_mday,current_time.tm_mon,current_time.tm_year,previous_time.tm_mday,previous_time.tm_mon,previous_time.tm_year);
			count = 1;
		}
	}

	return(count);
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_ObtainPresentDate
* PARAMETER 	  : date - Buffer to store the current date
* RETURNVALUE	  :
* DESCRIPTION	  : This will read the system time and store to the user buffer
**********************************************************************************/
tVoid eMMC_ExcessiveWP_ObtainPresentDate ( struct tm *date)
{
  time_t now = time(NULL);
  struct tm *t = localtime(&now);
  memcpy(date,t,sizeof(struct tm));
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_WriteErrmem
* PARAMETER 	  : Buffer - Buffer for errmem write
* RETURNVALUE	  :
* DESCRIPTION	  : This will write the date to errmem.
**********************************************************************************/
tBool eMMC_ExcessiveWP_WriteErrmem (tChar *Buffer)
{
	static tS32 		fdErrmem = 0;

	if ( fdErrmem == 0 )
	{
		fdErrmem = open("/dev/errmem", O_WRONLY);
		if( fdErrmem == -1)
		{
			ERROR("!!ERROR:Errmem Open Fails\n");
			fdErrmem = 0;
			return FALSE;
		}
	}

	if( fdErrmem > 0 )
	{
		if(write(fdErrmem, Buffer,strlen(Buffer)) <= 0)
		{
		   /* Trace Error*/
		   ERROR("!!ERROR: Unable to write into error memory %d\n",errno);
		   close (fdErrmem);
		   fdErrmem = 0;
		   return FALSE;
		}
	}
	return TRUE;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_LoadCumulativeBlockCnt
* PARAMETER 	  : blkioLocationInfo	: Location details which read from cfg file
* RETURNVALUE	  : TRUE - Success ; FALSE - Failure
* DESCRIPTION	  : Load the cumulative write cycle count.
**********************************************************************************/
tBool eMMC_ExcessiveWP_LoadCumulativeBlockCnt(monitor_blkio *blkioLocationInfo)
{
	tChar	DevNameuf[CFG_NAME_LEN];
	long long	TotalWriteCount = 0;
	FILE *fpBlkCountFile;
	tBool	found= FALSE;

	blkioLocationInfo->CumulativeBlkCount = 0;

	fpBlkCountFile = fopen(TOT_BLK_COUNT_PERSISTENT_FILE,MODE_RW);
	if( fpBlkCountFile != NULL ) {
		/*Check if the number of paramters scanned are 2 */
		while( fscanf(fpBlkCountFile,TOTAL_BLK_CNT_READ_PATTERN,DevNameuf,&TotalWriteCount) == 2 )
		{
			if(!strncmp(DevNameuf,blkioLocationInfo->location,sizeof(DevNameuf)))	{
				blkioLocationInfo->CumulativeBlkCount = TotalWriteCount;
				found = TRUE ;
				break;
			}
		}
		fclose(fpBlkCountFile);
	}
	return found;
}


/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWrite_PopulateCriticalNodes
* PARAMETER 	  : blkioLocationInfo	: 	Location details which read from cfg file
*					BlkCount			:	Blk count
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : This function will handle the functionality if the day
*					go back by one day
**********************************************************************************/
tBool eMMC_ExcessiveWrite_PopulateCriticalNodes (monitor_blkio *blkioLocationInfo, unsigned long long BlkCount, struct tm CurrentDate)
{
	node 	*rootNode,
			*LastEntry,
			*NewNode;

	rootNode	= blkioLocationInfo->ringnode;
	LastEntry 	= rootNode->prev;

	DEBUG("\n<%s>Current date : %2d/%2d/%4d BlkCount:%lu\n",blkioLocationInfo->location,CurrentDate.tm_mday,CurrentDate.tm_mon,CurrentDate.tm_year,BlkCount);
 	/*If it is one day window period, then update the same blockcount, total block count to the single node*/
	if ((blkioLocationInfo->MaxDuration == 1 /*1 Day*/) ||
		(blkioLocationInfo->MaxDuration == 0 /*0 Day*/))
	{
		DEBUG("Maximum Duration is one/zero.\n");
		rootNode->date = CurrentDate;

		rootNode->total_write_cycles_count = rootNode->total_write_cycles_count + rootNode->blk_count;
		if( blkioLocationInfo->PowerCycleStats != PROFILER_NEW_STARTUP )
		{
			blkioLocationInfo->PrevBlkcount = -(LastEntry->blk_count - blkioLocationInfo->PrevBlkcount);
			BlkCount = BlkCount + blkioLocationInfo->PrevBlkcount;
		}
		rootNode->blk_count = BlkCount;
		blkioLocationInfo->PowerCycleStats = PROFILER_NEW_POWERCYCLE;
	}
	/*If only one node is available :
	 * Create the new node and make that node as root node. Update block count, total block count and
	 * Cumulative count to the respective new node.
	 * Here we consider this is not a new power cycle. Instead of that, we treat as this is power cycel with
	 * 	a day bahind.
	 */
	else if( rootNode == LastEntry )
	{
		DEBUG("Created New node \n");
		NewNode = eMMC_ExcessiveWP_CreateNode( );
		if( NewNode == NULL )
		{
			return FALSE;
		}

		if( blkioLocationInfo->PowerCycleStats != PROFILER_NEW_STARTUP )
		{
			blkioLocationInfo->PrevBlkcount = -(LastEntry->blk_count - blkioLocationInfo->PrevBlkcount);
			BlkCount += blkioLocationInfo->PrevBlkcount;
		}
		NewNode->blk_count = BlkCount;
		NewNode->date = CurrentDate;
		NewNode->total_write_cycles_count = rootNode->total_write_cycles_count;
		rootNode->total_write_cycles_count += BlkCount;
		NewNode->next = NewNode->prev = rootNode;
		rootNode->next = rootNode->prev = NewNode;
		blkioLocationInfo->ringnode = NewNode;
		blkioLocationInfo->PowerCycleStats = PROFILER_NEW_POWERCYCLE_DAY_BACKWARD;
	}
	else
	{

		if(blkioLocationInfo->PowerCycleStats == PROFILER_NEW_POWERCYCLE_IN_SAME_DAY )
		{
			blkioLocationInfo->blockcnt = blkioLocationInfo->blockcnt + (LastEntry->blk_count - blkioLocationInfo->PrevBlkcount);
		}

		if( rootNode == LastEntry->prev )
		{
			DEBUG("Two nodes Available\n");
			/*If the previous date is already exist then update the block count to the respective node.
		 	 * Here we consider this is not a new power cycle. Instead of that, we treat as this is power cycel with
			 * a day bahind.
			 */
			if (blkioLocationInfo->PowerCycleStats != PROFILER_NEW_POWERCYCLE_DAY_BACKWARD) {
				DEBUG("Backup Previous Blk count LT:%llu BLK:%llu\n",LastEntry->total_write_cycles_count,rootNode->blk_count);
				blkioLocationInfo->PrevBlkcount = rootNode->blk_count;
			}
			rootNode->blk_count = blkioLocationInfo->PrevBlkcount  +  BlkCount - blkioLocationInfo->blockcnt;
			LastEntry->total_write_cycles_count = rootNode->blk_count + rootNode->total_write_cycles_count;
		}
		else
		{
			DEBUG("More then Two nodes Available\n");
			NewNode = LastEntry->prev;
			if ( blkioLocationInfo->PowerCycleStats != PROFILER_NEW_POWERCYCLE_DAY_BACKWARD)
			{
				DEBUG("Backup Previous Blk count\n");
				blkioLocationInfo->PrevBlkcount = NewNode->blk_count;
			}
			NewNode->blk_count = blkioLocationInfo->PrevBlkcount + BlkCount - blkioLocationInfo->blockcnt;
			LastEntry->total_write_cycles_count = NewNode->blk_count + NewNode->total_write_cycles_count;
		}
		blkioLocationInfo->PowerCycleStats = PROFILER_NEW_POWERCYCLE_DAY_BACKWARD;
	}

	blkioLocationInfo->CumulativeBlkCount = LastEntry->blk_count + LastEntry->total_write_cycles_count;
	DEBUG("CumulativeCnt: %llu<P:%llu L:%llu>\n"
												,blkioLocationInfo->CumulativeBlkCount
												,blkioLocationInfo->PrevBlkcount
												,LastEntry->total_write_cycles_count);
	return TRUE;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_PostEvent
* PARAMETER 	  : TriggerEvent		: Hold the details of excessive write
*					ExcessiveBytes		: Total count which exceeds the cfg threshold
*					blkioLocationInfo	: Location details which read from cfg file
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  :
**********************************************************************************/
tBool eMMC_ExcessiveWP_PostEvent ( write_profiler_params	*TriggerEvent, long long ExcessiveBytes,monitor_blkio *blkioLocationInfo )
{
	static tChar		*ErrmemBuffer = NULL;

	ERROR(ERRMEM_ENTRY 	,blkioLocationInfo->unique_name
						,blkioLocationInfo->location
						,TriggerEvent->name
					   	,TriggerEvent->duration
						,TriggerEvent->threshold
						,ExcessiveBytes);
	eMMC_ExcessiveWriteSendDbusNotification(TriggerEvent, ExcessiveBytes, blkioLocationInfo);
	if ( blkioLocationInfo->ErrmemNotification )
	{
		if(ErrmemBuffer == NULL)	{
			ErrmemBuffer = (tChar *)eMMC_ExcessiveWP_AllocateMemory( sizeof(ERRMEM_ENTRY)+SHELL_CMD_BYTES);
			if( ErrmemBuffer == NULL )
			{
				ERROR("!!ERROR:Mem Fail Size <%d>\n",(tS32)( sizeof(ERRMEM_ENTRY)+SHELL_CMD_BYTES));
				return FALSE;
			}
		}

		if (ErrmemBuffer != NULL) {
			snprintf(ErrmemBuffer,( sizeof(ERRMEM_ENTRY)+SHELL_CMD_BYTES),ERRMEM_ENTRY	,blkioLocationInfo->unique_name
												,blkioLocationInfo->location
												,TriggerEvent->name
											   	,TriggerEvent->duration
												,TriggerEvent->threshold
												,ExcessiveBytes);

			eMMC_ExcessiveWP_WriteErrmem(ErrmemBuffer);
		}
	}
#ifdef DUMP_TRACE_INFO
	TRACE_SYNC(fpTraceFile);
#endif /* DUMP_TRACE_INFO */
	return TRUE;
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_CheckWatchWindow
* PARAMETER 	  : blkioLocationInfo	: Location details which read from cfg file
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : Cumulatively add the last n number of entries based on the
*					configuration window period and check with threshold period. If
*					cumulative count exceeds the threshold period, Trigger the excessive
*					write event via D-Bus.
*					Only one trigger event is allowed in single power cycle.
**********************************************************************************/
tBool eMMC_ExcessiveWP_CheckWatchWindow(monitor_blkio *blkioLocationInfo)
{
	write_profiler_params	*TriggerEvent	=	blkioLocationInfo->monitor;
	node					*RingBuffer		=	blkioLocationInfo->ringnode;
	node					*LastEntry;
	tS32					DurationCnt = 1;
	long long				CumulativeCnt;

	if(( TriggerEvent == NULL ) || ( RingBuffer ==NULL )) {
		ERROR("!!ERROR: No event or no Ring Buffer Or All events are notified");
		return FALSE;
	}

	while ( TriggerEvent )
	{
		LastEntry = RingBuffer->prev;

		CumulativeCnt = 0;

		if (  TriggerEvent->duration == 0 )	{
			CumulativeCnt = LastEntry->blk_count + LastEntry->total_write_cycles_count;
		}
		else if (  TriggerEvent->duration == 1 ) {
			CumulativeCnt = LastEntry->blk_count;
		}
		else if (  TriggerEvent->duration > 1 )
		{
			do
			{
				CumulativeCnt += LastEntry->blk_count;
				LastEntry = LastEntry->prev;
			}while (( DurationCnt++ < TriggerEvent->duration ) && ( LastEntry != RingBuffer->prev ));
		}

		if ( CumulativeCnt > TriggerEvent->threshold )	{
			eMMC_ExcessiveWP_PostEvent(TriggerEvent, CumulativeCnt,blkioLocationInfo);
			TriggerEvent->duration = -1;
		}

		TriggerEvent = TriggerEvent->next;
	}
	return TRUE;
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_GetContainerName
* PARAMETER 	  : Location	: Location details which read from cfg file
*					ContainerName : Get the lxc-container anme
* RETURNVALUE	  :
* DESCRIPTION	  : This function check lxc location name and paser the opt lxc name for monitor
**********************************************************************************/
tBool eMMC_ExcessiveWP_GetContainerName(tChar *Location, tChar* ContainerName)
{
	if (!strncmp(LXC_NAME_PREFIX,Location,strlen(LXC_NAME_PREFIX)))
	{
		if ((NULL != strchr(Location,LXC_NAME_SPLIT)) &&
			((strchr(Location,LXC_NAME_SPLIT)+1)[0]!='\0') &&
			((strchr(Location,LXC_NAME_SPLIT)+1)[0]!=' '))
		{
			strcpy(ContainerName,strchr(Location,LXC_NAME_SPLIT)+1);
			return TRUE;
		}
	}
	strcpy(ContainerName,Location);
	return FALSE;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_CreateThread
* PARAMETER 	  : ptr	: Argument to the new thread
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : Create the new thread and apps the argument
**********************************************************************************/
tBool eMMC_ExcessiveWP_CreateThread ( tPVoid ptr)
{
	monitor_blkio	*blkioLocationInfo = (monitor_blkio *)ptr;

	if(pthread_create(&blkioLocationInfo->LocationThread, NULL, eMMC_ExcessiveWP_MonitorThread, ptr))
	{
		ERROR("Error creating thread %s\n",blkioLocationInfo->unique_name);
		return FALSE;
	}
	return TRUE;
}
/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_CheckCurrentDate
* PARAMETER       :
* RETURNVALUE	  :
* DESCRIPTION	  : Write profiler is supported after 2017. So check the current time
					is greater than 2017.
**********************************************************************************/
tS32 eMMC_ExcessiveWP_CheckCurrentDate (tVoid)
{
struct tm 	EntryTime,current_time;
tS32 		count = CURRENT_DATE_CHECK_TIME_SEC;
tS32 ret;
tU32 evt;

/* timeout is in ms and we want here to wait for 120 seconds */
tS32 timeout;

	/* This thread runs forever until stcfgInfo becomes NULL*/
	while(NULL != stcfgInfo)
	{
		/*timeout for 120 seconds for the time updation from the GPS */
		timeout = 120 * 1000;

		ret = eMMC_ExcessiveWP_ReceiveMsg(stcfgInfo->evt_lcm,&evt,timeout);
		if (-1 == ret)	{
			DEBUG ("eMMC_ExcessiveWP_ReceiveMsg failed \n",ret );
			break;
		}
		else if (1 == ret)	{
			/*
			We are in the condition where no message has been received
			(i.e. SHUTDOWN / WAKEUP	to store/load persistent information)
			We JUST WAITED for the time updation from the GPS
			 Get the Current date and wait for the System date reachs 2017 or
			 wait for 2min 	 to ensure the system time stable
		 	*/
			break;
		}
		else {
			/*
				We are in the condition where the message has been received
				(i.e. SHUTDOWN / WAKEUP to store/load persistent information)
			 	Resume the application and continue sleep for another 120 sec.
			 	post the event and nothing else can be done at this point
			*/
			postEvent(&stcfgInfo->ackevent_lcm);
			continue;
		}
	}
	return ret;
}
/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_InfraSetup
* PARAMETER       :
* RETURNVALUE	  : On SUCCESS, returns pointer of type monitor_global_info .
*                            On FAILURE, returns NULL.
* DESCRIPTION	  : Setting up the infrastructure for excessive write.
**********************************************************************************/

monitor_global_info* eMMC_ExcessiveWP_InfraSetup()
{
	FILE *fpBlkCountFile;
	monitor_global_info	*cfgInfo = NULL;

/* Open the log file */
#ifdef DUMP_TRACE_INFO
		TRACE_OPEN(fpTraceFile);
#endif

	/*Created semaphore for shared file access between different location */
	if (sem_init(&RestrictSharedAccessSem, 0, 1) == -1)
	{
		ERROR("!!ERROR:Sem Crete<RestrictSharedAccessSem Failed.. \n");
		return NULL;
	}

	cfgInfo	= (monitor_global_info *)eMMC_ExcessiveWP_AllocateMemory (sizeof(monitor_global_info));
	if( NULL == cfgInfo )
	{
		ERROR("!!ERROR:Mem fail size monitor_cfg <%d>\n",(tS32)sizeof(monitor_global_info));
		goto error_exit;
	}

	/* configuration file parsing */
	if (( eMMC_ExcessiveWP_ParseConfigFile (cfgInfo) != TRUE ) && ( cfgInfo->blkio == 	NULL))
	{
		ERROR("!!ERROR:Parse Config file %s\n",JSON_CONFIG_FILE);
		ERROR("!!ERROR:Wrong CFG file , No location found\n" );
		goto error_exit;
	}
	/*register the signal handler to cought the exception and other signal to the app*/
	eMMC_ExcessiveWP_SignalHandlerSetup();
	eMMC_ExcessiveWP_CreateMsgObj(&cfgInfo->evt_lcm,EXCESSIVEWP_EVTQUEUE_NAME);

	if( -1 == cfgInfo->evt_lcm)
	{
		ERROR("!!ERROReMMC_ExcessiveWP_CreateMsgObj error for queue %s\n",EXCESSIVEWP_EVTQUEUE_NAME);
		goto error_exit;
	}
	/*Created semaphore for shared file access between different location */
	if (createEvent(&cfgInfo->ackevent_lcm) == -1)
	{
		ERROR("!!ERROR:Sem Create <ackevent_lcm > Failed.. \n");
		goto error_exit;
	}

	/*
		This below code is just to create "TOT_BLK_COUNT_PERSISTENT_FILE",
		incase if it does not exist
	*/
	fpBlkCountFile = fopen(TOT_BLK_COUNT_PERSISTENT_FILE,MODE_RW);
	if( fpBlkCountFile == NULL ) {
		fpBlkCountFile = fopen(TOT_BLK_COUNT_PERSISTENT_FILE,MODE_W);
		if( fpBlkCountFile == NULL ) {
			ERROR("!!ERRROR:FileOpen Fail %s \n",TOT_BLK_COUNT_PERSISTENT_FILE);
			goto error_exit;
		}
	}

	fclose(fpBlkCountFile);
	fpBlkCountFile = NULL;

	return cfgInfo;

error_exit:

	/* Nothing can be done in error handling of sem_destroy */
	(void)sem_destroy(&RestrictSharedAccessSem);
#ifdef DUMP_TRACE_INFO
	TRACE_CLOSE(fpTraceFile);
#endif /* DUMP_TRACE_INFO */
	if (cfgInfo)	{
		if(-1 != cfgInfo->evt_lcm)	{
			eMMC_ExcessiveWP_DeleteMsgObj (cfgInfo->evt_lcm,EXCESSIVEWP_EVTQUEUE_NAME);
		}
		eMMC_ExcessiveWP_DeleteMemory (cfgInfo);
		cfgInfo = NULL;
	}
	return NULL;
}

/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_WatchThread
* PARAMETER       :	Arg - Excessive write info
* RETURNVALUE	  : tPVoid
* DESCRIPTION	  : This thread will Start the profilling and watch the status of it.
**********************************************************************************/
tPVoid	eMMC_ExcessiveWP_WatchThread ( tPVoid Arg )
{
	monitor_blkio 	*blkioLocationInfo;
	tS32 timeout;
	tS32 ret;
	tU32 evt;


	if( stcfgInfo == NULL ) {
		ERROR("!!! stcfgInfo is NULL !!! \n");
		return NULL;
	}

	/*Created the thread the monitor the D-Bus and share the excessive write details to the D-Bus*/
	strncpy(stcfgInfo->trDbusInfo.DBusName,DBUS_NAME,sizeof(stcfgInfo->trDbusInfo.DBusName));
	stcfgInfo->trDbusInfo.IsBusAcquired = FALSE;
	if( pthread_create(&stcfgInfo->trDbusInfo.DbusThrdhandle,NULL,eMMC_ExcessiveWrite_DbusThread,(tPVoid)(&(stcfgInfo->trDbusInfo))) )
	{
		ERROR("!!ERROR:D-Bus Thread Create Fail.. \n");
		goto end;
	}

	/*Wait for the time updation from the GPS.*/
	if (((tS32)eMMC_ExcessiveWP_CheckCurrentDate() < 0)) {
		goto end;
	}
	blkioLocationInfo = stcfgInfo->blkio;
	/*Create the individual thread to each device monitor*/
	/*Maintained the cumulative write count w.r.t the device in sepearate file.*/
	while ( blkioLocationInfo != NULL )
	{
		eMMC_ExcessiveWP_LoadCumulativeBlockCnt(blkioLocationInfo);
		blkioLocationInfo->PowerCycleStats 		= PROFILER_NEW_STARTUP;
		blkioLocationInfo->MonitorThrdState		= PROFILER_THREAD_RUNNING;
		blkioLocationInfo->trDbusInfo			= &stcfgInfo->trDbusInfo;
		blkioLocationInfo->ErrmemNotification	= stcfgInfo->ErrmemNotification;
		blkioLocationInfo->blockcnt				= 0;
		blkioLocationInfo->PrevBlkcount			= 0;
		/*Based on the interval period write to be issued for persistent file.*/
		blkioLocationInfo->DataWriteInterval = stcfgInfo->PersistentDataSaveCycle*PERSISTENT_WRITE_INTERVAL_SEC;
		/* Initially,there is no event received in the monitor thread */
		blkioLocationInfo->evt_state = EVT_LCM_NOTHING;
		eMMC_ExcessiveWP_CreateThread ((tPVoid)blkioLocationInfo);
		blkioLocationInfo = blkioLocationInfo->next;
	}

	/* timeout infinite */
timeout = -1;

	while(NULL != stcfgInfo)
	{
		blkioLocationInfo = stcfgInfo->blkio;
		ret = eMMC_ExcessiveWP_ReceiveMsg(stcfgInfo->evt_lcm,&evt,timeout);
		if (-1 == ret) {
			DEBUG ("eMMC_ExcessiveWP_ReceiveMsg failed \n",ret );
			break;
		}
		else if (1 == ret)	{
			/*
			We are in the condition where no message has been received
			(i.e. SHUTDOWN / WAKEUP	to store/load persistent information)
			We JUST WAITED for the time updation from the GPS
			 Get the Current date and wait for the System date reachs 2017 or
			 wait for 2min 	 to ensure the system time stable
		 	*/
			continue;
		}
		else {
			/*
				We are in the condition where the message has been received
				(i.e. SHUTDOWN / WAKEUP to store/load persistent information)
				 Resume the application and reload the sleep time
			*/
			if ((evt == EVT_LCM_SAVE_PERSISTENCY) || (evt == EVT_LCM_LOAD_PERSISTENCY))
			{

				while ( blkioLocationInfo != NULL )
				{
					/* Important: No other thread modifies "evt_state" */
					blkioLocationInfo->evt_state = evt;

					/* Signal the monitor thread */
					postEvent(&blkioLocationInfo->eventObj);
					blkioLocationInfo = blkioLocationInfo->next;
				}
			}
		}

			/* Wait for ack from each thread,after each thread preserve the data */
			blkioLocationInfo = stcfgInfo->blkio;

			/*timeout = 200 ms*/
			timeout = 200;
			while ( blkioLocationInfo != NULL )
			{
				/* Wait for the signal from monitor thread */
				waitEvent(&blkioLocationInfo->AckeventObj,timeout);
				blkioLocationInfo = blkioLocationInfo->next;
			}
			/*notify LCM SavePersistency/LoadPersistencycalls */
			postEvent(&stcfgInfo->ackevent_lcm);

		}

	while ( blkioLocationInfo != NULL )
	{
			pthread_join(blkioLocationInfo->LocationThread,NULL);
			blkioLocationInfo = blkioLocationInfo->next;
	}

	/*Stop the D-Bus thread the kill the d_bus in case of signal cought.*/
	g_main_loop_quit(stcfgInfo->trDbusInfo.main_loop);
	g_bus_unown_name(stcfgInfo->trDbusInfo.watcher_id);
	g_main_loop_unref(stcfgInfo->trDbusInfo.main_loop);
	pthread_join(stcfgInfo->trDbusInfo.DbusThrdhandle,NULL);

end:
	DEBUG("Excessive Profilling End \n");

#ifdef DUMP_TRACE_INFO
	TRACE_CLOSE(fpTraceFile);
#endif /* DUMP_TRACE_INFO */

	pthread_exit(NULL);
}
/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWrite_main
* PARAMETER       :	tS32 argc, tS8* argv[]
* RETURNVALUE	  : tS32
* DESCRIPTION	  : This function initializes the parse the configuration file,
*					signal handler, eMMC profiler structures and start write profiler
*					by creating the threads.
**********************************************************************************/
tS32 eMMC_ExcessiveWrite_main (tS32 argc, tChar** argv)
{
	/*All the debug and info prints are redirected to the trace file*/
#ifdef DUMP_TRACE_INFO
	TRACE_OPEN(fpTraceFile);
#endif /* DUMP_TRACE_INFO */
	DEBUG("Excessive write Main Entered \n");
	stcfgInfo = eMMC_ExcessiveWP_InfraSetup();
	if( NULL == stcfgInfo ) {
		ERROR("!!ERROR:eMMC_ExcessiveWP_InfraSetup is failed \n");
		return -1;
	}
	if(pthread_create(&HndleWatchThrd, NULL, eMMC_ExcessiveWP_WatchThread, (tPVoid) stcfgInfo))
	{
		ERROR("Watch Error creating thread \n");
		return -1;
	}

	pthread_setname_np(HndleWatchThrd,"ExcessWriteWatchThread");

	DEBUG("-----ExcessWriteWatchThread Started---- \n");
	return 0;
}
/********************************************************************************
* FUNCTION        : PostLCMEvent
* PARAMETER       :
* RETURNVALUE	  :
* DESCRIPTION	  : This is to notify  the watch thread
**********************************************************************************/
static void PostLCMEvent(tU32 evt ,tU32 snd_timeout,tU32 ack_timeout)
{
	tS32 ret;

	/*Infrastrure not set,it seems...just return*/
	if(NULL == stcfgInfo)
	{
		return ;
	}

	ret = eMMC_ExcessiveWP_SendMsg (stcfgInfo->evt_lcm,evt,snd_timeout);
	if (0 != ret)
	{
		DEBUG ("eMMC_ExcessiveWP_LoadPersistency failed \n",ret );
	}
	else {
		/*
		  Wait for Signal the monitor thread for 500ms
		  For whatever the reason, if there is no signal received from Watch thread,
		  may be all the monitor threads are busy in processing or system cpu load
		  is quite high that they could get scheduled within this time.
		  then it just simply returns.In this case we may loose the opportunity
		  of saving the data during shutdown but it is helpless situation
		   No point of waiting for long time.
		 */

		ret = waitEvent(&stcfgInfo->ackevent_lcm,ack_timeout);
		if (0 == ret)
		{
			DEBUG ("\n waitEvent SUCCESS in ret = %d evt = %d \n",ret,evt );
		}
		else if (1 == ret) {
			DEBUG ("\n waitEvent timeout in ret = %d evt = %d \n",ret,evt );
		}
		else {
			DEBUG ("\n waitEvent Failure in ret = %d evt = %d \n",ret, evt );
		}
	}
return ;
}

/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_LoadPersistency
* PARAMETER       :
* RETURNVALUE	  :
* DESCRIPTION	  : This is the notification to start the application process
**********************************************************************************/
tS32 eMMC_ExcessiveWP_LoadPersistency (tVoid )
{

	/*
		This function eMMC_ExcessiveWP_SendMsg does not take beyond 500 ms time
		Incase it can not send the message within 500ms for whatever the reason
		it is,just ignore it and return
	*/

	PostLCMEvent(EVT_LCM_LOAD_PERSISTENCY,200,500);
	DEBUG("Profiler Load Persistency \n");

	return 0;
}

/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_SavePersistency
* PARAMETER       :
* RETURNVALUE	  :
* DESCRIPTION	  : This is the notification to store the last mode date to the disk.
*					Hereafter no more write to the disk.
					This function promises that it returns within 800ms time
**********************************************************************************/
tS32 eMMC_ExcessiveWP_SavePersistency (  tVoid )
{
	/*
		This function eMMC_ExcessiveWP_SendMsg does not take beyond 250 ms time
		Incase it can not send the message within 250ms for whatever the reason
		it is,just ignore it and return
	*/

	PostLCMEvent(EVT_LCM_SAVE_PERSISTENCY,250,700);

	DEBUG("Profiler Save Persistency \n");

	return 0;
}
