/*****************************************************************************************
 * Copyright (C) RBEI, 2013
 * This software is property of Robert Bosch.
 * Unauthorized duplication and disclosure to third parties is prohibited.
 ******************************************************************************************/
/******************************************************************************
 * FILE         : emmc_refresh-signalHandler.c
 *
 * DESCRIPTION  : This contains the all signal handling for emmc refresh
 *
 *
 * AUTHOR(s)    :  (RBEI/ECF5)
 *
 * HISTORY      :
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *-----------|---------------------|--------------------------------------
 *01.AUG.2013| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *12.DEC.2014| Version 1.1         | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *08.OCT.2015| Version 1.3         | SWATHI BOLAR (RBEI/ECF5)
 *           |                     | eMMC refresh enabled for all projects.
 * -----------------------------------------------------------------------
 ***************************************************************************/
/************************************************************************
 * Header file declaration
 *-----------------------------------------------------------------------*/

#include "emmc_refresh.h"
#include "platform_app_manager.h"

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <nvm_refresh.h>
#include <dirent.h>
#include <libgen.h>


// It is not defined in QNX
# if !defined(SA_RESTART)
#  define SA_RESTART 0
# endif
# if !defined(SA_ONSTACK)
#  define SA_ONSTACK 0
# endif

/************************************************************************
 * Function declaration (scope: Local to file)
 *-----------------------------------------------------------------------*/
/********************************************************************************
 * FUNCTION        : stop_signal_handler
 * PARAMETER       :
 * RETURNVALUE     :
 *
 * DESCRIPTION     :  This contains the signal handling
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *1.AUG.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 **********************************************************************************/
void stop_signal_handler(tS32 sig)
{
	log_debug("OUCH! - I got signal %d-%s\n", sig,strsignal(sig));
	application_tearDown();
	exit(0);
}
/********************************************************************************
 * FUNCTION        : signal_handler_setup
 * PARAMETER       :
 * RETURNVALUE     :
 *
 * DESCRIPTION     :  This contains the signal handler setup
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *21.AUG.2013| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 **********************************************************************************/
void tVSignalHandlerSetup(void)
{
	struct sigaction 	act;
	tS32 result = 0;

	act.sa_handler = stop_signal_handler;

	result = sigemptyset(&act.sa_mask);

	if (result != 0)
	{
		perror("sigemptyset failed: ");
		log_debug("sigemptyset failed RetValue: %d", result);
	}

	act.sa_flags = SA_RESTART | SA_ONSTACK;
	result = sigaction( SIGINT | SIGTERM | SIGABRT | SIGKILL,	&act, NULL);

	if (result != 0)
	{
		perror("sigaction failed: ");
		log_debug("Signal handler registration failed RetValue: %d", result);
	}
}
/********************************************************************************
 * FUNCTION        : concatPath
 * PARAMETER       : s1 -> First Part of the path String
 * 					 s2 -> Second part of the path String
 * RETURNVALUE     : tPC8 -> Concatenated path string
 *
 * DESCRIPTION     : This function concatenate the path strings given and returns it
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Initialversion 1.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
tPC8 concatPath(tPC8 s1 , tPC8 s2)
{
	tPC8 result = NULL;
	if(s1[strlen(s1)-1] != '/' && s2[0] != '/' )
		//Checking whether last char of s1 and first char of s2 is have '/'.
	{
		//Allocating one char for storing '/' and one char for storing null char
		result = (char*)malloc((strlen(s1)+1) +strlen(s2)+1);
		sprintf((tPS8)result,"%s/%s",s1,s2);
	}
	else
	{
		result = (char*)malloc(strlen(s1) +strlen(s2)+1);
		sprintf((tPS8)result,"%s%s",s1,s2);
	}
	return result;
}

/********************************************************************************
 * FUNCTION        : alignDeviceSize
 * PARAMETER       : tS32 num -> Size in Gb
 *
 * RETURNVALUE     : tS32 -> Size in Gb aligned with 2^n
 *
 * DESCRIPTION     : Returns the size of num aligned with respect to 2^n
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/

//Returns the next greatest multiple of 2
tU32 u32AlignDeviceSize(tU32 num)
{
	tU32 inc = 1;
	//Return num if it is already power of 2
	if (!(num & (num - 1)))
		return num;

	while (inc < num)
		inc <<= 1;

	return inc;
}

/********************************************************************************
 * FUNCTION        :  s32CheckAndCreateDir
 * PARAMETER       :  s32CheckAndCreateDir - Dir to be checked or created
 * RETURNVALUE     :  REFRESH_NOERROR if the dir exists or created
 * DESCRIPTION     :  Creates the dir if it doesn't exist
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
tS32 s32CheckAndCreateDir(tPC8 cMetaDataFilePath)
{
    tS32 s32ReturnValue = REFRESH_NOERROR;
    DIR *dir;
    dir = opendir(cMetaDataFilePath);
    if(dir == NULL)
    {
		//printf("Given MetaData Folder '%s' does not exist.Hence Creating one");
        log_debug("Given MetaData Folder %s does not exist.Hence Creating one",cMetaDataFilePath);
        //If the given dir doesnt exist then create it
        s32ReturnValue = s32CreateDirs(cMetaDataFilePath,S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
		if(s32ReturnValue == REFRESH_NOERROR)
        {
            //Set the group id for the created folder.This function is implemented
            // only for linux platform

            vSetGroupId(cMetaDataFilePath);
		}
        else
        {
           log_error("Unable to create dir : %s ErrorCode : %d ErrorString : %s", cMetaDataFilePath,errno,strerror(errno));
           s32ReturnValue = (tS32) REFRESH_ACCESS_ERROR;
        }
    }
    else
    {
        closedir(dir);
	}
    return s32ReturnValue;
}

/********************************************************************************
 * FUNCTION        : s32CreateDirs
 * PARAMETER       : s8Path -> Dir path to be created
 * 					 mode   -> Dir modes
 * RETURNVALUE     : 0 if success | < 0 in case of error
 *
 * DESCRIPTION     : This is a recursive function which does the functionality
 * 					 same as mkdir -p
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Version 2.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
tS32 s32CreateDirs(tPC8 s8Path,mode_t mode)
{
    tPS8 s8FullPath = strdup(s8Path);
    if(access(dirname(s8FullPath), F_OK) != 0)
    {
        //If the parent dir doesnt exist create it.This function iterates until
        // all the parent dirs are created
        (void)s32CreateDirs(s8FullPath, mode);
	}
	if(s8FullPath)
		free(s8FullPath);
    return mkdir(s8Path,mode);
}

