/*
 * emmc_access_new.c
 *
 *  Created on: Aug 8, 2019
 *      Author: RAP5COB
 */
 
#include "emmc_platform_access.h"
#include "platform_app_manager.h"

#include <stdint.h>
#include <fcntl.h>
#include <linux/fs.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <libgen.h>
typedef unsigned long long	u64;

struct mmc_ioc_cmd {
    /* Implies direction of data.  true = write, false = read */
    int write_flag;

    /* Application-specific command.  true = precede with CMD55 */
    int is_acmd;

    tU32 opcode;
    int arg;
    tU32 response[4];  /* CMD response */
    unsigned int flags;
    unsigned int blksz;
    unsigned int blocks;

    /*
     * Sleep at least postsleep_min_us useconds, and at most
     * postsleep_max_us useconds *after* issuing command.  Needed for
     * some read commands for which cards have no other way of indicating
     * they're ready for the next command (i.e. there is no equivalent of
     * a "busy" indicator for read operations).
     */
    unsigned int postsleep_min_us;
    unsigned int postsleep_max_us;

    /*
     * Override driver-computed timeouts.  Note the difference in units!
     */
    unsigned int data_timeout_ns;
    unsigned int cmd_timeout_ms;

    /*
     * For 64-bit machines, the next member, ``__u64 data_ptr``, wants to
     * be 8-byte aligned.  Make sure this struct is the same size when
     * built for 32-bit.
     */
    tU32 __pad;

    /* DAT buffer */
    u64 data_ptr;
};

#ifdef UTEST

tS32 ioctl(tS32 a,unsigned long b,void *c);
#define PATH /tmp/%s
#define DEV_PATH_EXPAND(PATH) #PATH
#define DEV_PATH(PATH) DEV_PATH_EXPAND(PATH)

#define VENDOR_ID_PATH /tmp/%s/manfid
#define VID_EXP(VENDOR_ID_PATH) #VENDOR_ID_PATH
#define VID_PATH(VENDOR_ID_PATH) VID_EXP(VENDOR_ID_PATH)

#define VENDOR_ID_OTHERPATH /tmp/%s
#define VID_EXPO(VENDOR_ID_OTHERPATH) #VENDOR_ID_OTHERPATH
#define VID_OTHERPATH(VENDOR_ID_OTHERPATH) VID_EXPO(VENDOR_ID_OTHERPATH)

#define SYS_DIR_MMCHOST_PATH	"/tmp/"

#else
#define PATH /dev/%s
#define DEV_PATH_EXPAND(PATH) #PATH
#define DEV_PATH(PATH) DEV_PATH_EXPAND(PATH)


#define VENDOR_ID_PATH /sys/block/%s/device/manfid
#define VID_EXP(VENDOR_ID_PATH) #VENDOR_ID_PATH
#define VID_PATH(VENDOR_ID_PATH) VID_EXP(VENDOR_ID_PATH)

#define VENDOR_ID_OTHERPATH /sys/block/%s/device/device/manfid
#define VID_EXPO(VENDOR_ID_OTHERPATH) #VENDOR_ID_OTHERPATH
#define VID_OTHERPATH(VENDOR_ID_OTHERPATH) VID_EXPO(VENDOR_ID_OTHERPATH)
#define SYS_DIR_MMCHOST_PATH	"/sys/class/mmc_host/"
#include <sys/ioctl.h>

#endif

#define mmc_ioc_cmd_set_data(ic, ptr) ic.data_ptr = (u64)(unsigned long) ptr




#define MMC_SEND_CID             10   /* ac   [31:16] RCA        R2  */
#define MMC_SEND_EXT_CSD          8
#define MMC_SEND_STATUS          13   /* ac   [31:16] RCA        R1  */

#define MMC_BLOCK_MAJOR		179
#define MMC_IOC_CMD _IOWR(MMC_BLOCK_MAJOR, 0, struct mmc_ioc_cmd)
#define MMC_RSP_PRESENT	(1 << 0)
#define MMC_RSP_136	(1 << 1)		/* 136 bit response */
#define MMC_RSP_CRC	(1 << 2)		/* expect valid crc */
#define MMC_RSP_BUSY	(1 << 3)		/* card may send busy */
#define MMC_RSP_OPCODE	(1 << 4)		/* response contains opcode */

#define MMC_CMD_MASK	(3 << 5)		/* non-SPI command type */
#define MMC_CMD_AC	(0 << 5)
#define MMC_CMD_ADTC	(1 << 5)
#define MMC_CMD_BC	(2 << 5)
#define MMC_CMD_BCR	(3 << 5)
#define MMC_RSP_SPI_S1	(1 << 7)		/* one status byte */
#define MMC_RSP_SPI_S2	(1 << 8)		/* second byte */
#define MMC_RSP_SPI_B4	(1 << 9)		/* four data bytes */
#define MMC_RSP_SPI_BUSY (1 << 10)		/* card may send busy */

#define MMC_DATA_WRITE	(1 << 8)
#define MMC_DATA_READ	(1 << 9)
#define MMC_DATA_STREAM	(1 << 10)


#define MMC_RSP_R1	(MMC_RSP_PRESENT|MMC_RSP_CRC|MMC_RSP_OPCODE)
#define MMC_RSP_SPI_R1	(MMC_RSP_SPI_S1)
#define MMC_RSP_SPI_R1B	(MMC_RSP_SPI_S1|MMC_RSP_SPI_BUSY)
#define MMC_RSP_R1B	(MMC_RSP_PRESENT|MMC_RSP_CRC|MMC_RSP_OPCODE|MMC_RSP_BUSY)
#define MMC_CMD_AC	(0 << 5)

#define MMC_RSP_SPI_R2	(MMC_RSP_SPI_S1|MMC_RSP_SPI_S2)
#define MMC_RSP_R2	(MMC_RSP_PRESENT|MMC_RSP_136|MMC_RSP_CRC)
#define MMC_SWITCH_MODE_WRITE_BYTE	0x03	/* Set target to value */
#define EXT_CSD_CMD_SET_NORMAL		(1<<0)
#define MMC_SWITCH                6   /* ac   [31:0] See below   R1b */


tU8 s32ScanForValidDevices(tPS8 s8DeviceName[],emmc_dev_data *mmc_dev_info,tU8 u8DeviceCount);
tS32 s32FetchDeviceInformation(emmc_dev_data *dev_data,tU32 *u32VendorId,tU32 *u32Mmc_Revision);
tS32 s32GetManFid(tPS8 mmcPath, tU32 *u32Manfid);


/********************************************************************************
 * FUNCTION        : inline tSortStruct
 * PARAMETER       : a    -  Pointer of the data to be sorted
 *                   b    -  Pointer of the data to be sorted
 * RETURNVALUE     : 1 If needs to be sorted else 0
 * DESCRIPTION     : This function does the strcmp and returns 1 if string1 is greater else 0
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
tS32 tSortStruct(const void *a, const void *b)
{
	if(a && b)
	{
		const emmc_dev_data *data_a = (emmc_dev_data*)a;
		const emmc_dev_data *data_b = (emmc_dev_data*)b;

		int res = strcmp(data_a->dev_path_name,data_b->dev_path_name);
		if(res) return res;
		return 0;
	}
	return -1;

}


/********************************************************************************
 * FUNCTION        : S32CheckAndPopulateDeviceDetails
 * PARAMETER       : s8DeviceNames    -  eMMC partitions to be refreshed seperated by ':'
 *                   s8Numdevices     -  number of vaid partition count will be updated
 *                   s8VendorId       -  VendorId of the eMMC will be updated
 * RETURNVALUE     : emmc_dev_data instance
 * DESCRIPTION     : This function manipulates the s8DeviceNames passed and invokes
 * 					 respective functions for validating and populating the eMMC device information
 * 					 in the emmc_dev_data structure instance, which will be returned.
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
emmc_dev_data *s32CheckAndPopulateDeviceDetails(tPS8 s8DeviceNames,refresh_data *emmc_refresh_data)
{
	tU8 u8DeviceCount = 0;
	char* dev_name = strtok(s8DeviceNames,":");
	//Make sure that always last is NULL,So adding +1
	char *dev_names[MAX_SUB_DEVICES+1] = {0};
	emmc_dev_data *dev_data_start = NULL;
	while(dev_name != NULL)
	{
		dev_names[u8DeviceCount] = (char*)malloc(strlen(dev_name)+1);
		memset(dev_names[u8DeviceCount],0,strlen(dev_name)+1);
		strncpy(dev_names[u8DeviceCount],dev_name,strlen(dev_name));
		log_debug("Input devices name are %s",dev_names[u8DeviceCount]);
		u8DeviceCount++;
		if(u8DeviceCount > MAX_SUB_DEVICES)
			break;
		dev_name = strtok(NULL,":");
	}
	dev_names[u8DeviceCount] = NULL;
	if(u8DeviceCount > 0)
	{
		emmc_dev_data *dev_data = (emmc_dev_data*)malloc(sizeof(emmc_dev_data)*u8DeviceCount);
		if(dev_data)
		{
			memset(dev_data,0,sizeof(emmc_dev_data)*u8DeviceCount);	
			dev_data_start = dev_data;
			emmc_refresh_data->u8NumOfPartitions = s32ScanForValidDevices(dev_names,dev_data,u8DeviceCount);			
			if(emmc_refresh_data->u8NumOfPartitions > 0 )
			{
				log_debug("Found Devices %d",emmc_refresh_data->u8NumOfPartitions);
				//Sorting the structure based on device name. mmcblk1 should be at the start of the structure
				qsort(dev_data_start,emmc_refresh_data->u8NumOfPartitions,sizeof(emmc_dev_data),tSortStruct);
				tS32 s32ReturnValue = s32FetchDeviceInformation(dev_data_start,&emmc_refresh_data->u32mmc_vendor,&emmc_refresh_data->u32mmc_revision);	
				if(s32ReturnValue != REFRESH_NOERROR)
					dev_data_start = NULL;
			}
			else
			{
                                        free(dev_data);
					dev_data_start = NULL;
			}
		}
		else
		{
			log_error("Memory allocation failed for emmc_dev_data of size %ld",sizeof(emmc_dev_data)*u8DeviceCount);
		}
	}
	return dev_data_start;
}

/********************************************************************************
 * FUNCTION        : u64GetSize
 * PARAMETER       : s8DevicePath    -  eMMC device path("/dev/emmcnode")
 * RETURNVALUE     : tU64 Size of the eMMC else 0
 * DESCRIPTION     : This function uses ioctl interface for calculating the device size
 * 					 and returns it
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
tU64 u64GetSize(tPS8 s8DevicePath)
{
	tU64 u64Size = 0;
	if(s8DevicePath)
	{
		tS32 fp = open(s8DevicePath, O_RDONLY);
		if(fp >= 0 )
		{
			//Using ioctl for calculating device size. Earlier device size is read from sysfs file.
			//This method was used so that calculation logic will be the same across different OS paltforms
			tS32 res = ioctl(fp,BLKGETSIZE64,&u64Size);
			if(res<0)
			{
				log_debug("Unable to fetch size for the device %s ErrorCode : %d ErrorString : %s",s8DevicePath,errno, strerror(errno));
			}
			close(fp);
		}
		else
		{
			log_debug("Unable to open the device %s ErrorCode : %d ErrorString : %s",s8DevicePath,errno, strerror(errno));
		}
	}
	else
	{
		log_debug("s8DevicePath is NULL");
	}
	return u64Size;
}


/********************************************************************************
 * FUNCTION        : s32ScanForValidDevices
 * PARAMETER       : s8DeviceName    -  eMMC device Node's
 * 					 mmc_dev_info    -  structure where the valid eMMC device partition names
 * 					 will be populated
 * RETURNVALUE     : s32DevicesFound (no.ofd evices found or 0)
 * DESCRIPTION     : This function parses the s8DeviceName array and checks whether the
 * 					 entries are of type MMC from the sysfs 'type' file
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

tU8 s32ScanForValidDevices(tPS8 s8DeviceName[],emmc_dev_data *mmc_dev_info,tU8 u8DevicePassed)
{
	tS32 s32DevNumber = 0;
	tU8 u8DevicesFound = 0;
	struct dirent   *dirMmcCardEntry = {0};
	DIR *dirAccess = NULL;
	char sysDirMmcHostPath[EMMC_MAX_PATH_LEN]= {0};
	char sysMmcCardTypeFile[EMMC_MAX_PATH_LEN]= {0};
	char devType[8]= {0};
	FILE *fpMmcDevType = NULL;
	tS8 s8hostEntry[16] = {0};
	tU8 count=0;
	//Iterating the devices and checking whether it belongs to MMC type. Because we should reject devices like SD-Cards from refresh

	while(s8DeviceName[count] != NULL)
	{
		log_debug("Device names are %s",s8DeviceName[count]);
		sscanf(s8DeviceName[count],"mmcblk%d",&s32DevNumber);
		if(s32DevNumber >= 0)
		{
			snprintf(s8hostEntry,sizeof(s8hostEntry)-1,"mmc%d",s32DevNumber);
			snprintf(sysDirMmcHostPath,sizeof(sysDirMmcHostPath)-1,"%s%s",SYS_DIR_MMCHOST_PATH,s8hostEntry);
			log_debug("HostEntry Path for the device %s in sysfs :  %s",s8DeviceName[count],sysDirMmcHostPath);
			if((dirAccess = opendir(sysDirMmcHostPath)) != NULL)
			{
				while((dirMmcCardEntry = readdir(dirAccess)) != NULL)
				{
					if(!strncmp(dirMmcCardEntry->d_name,s8hostEntry, strlen(s8hostEntry)))
					{
						snprintf(sysMmcCardTypeFile,sizeof(sysMmcCardTypeFile)-1,"%s/%s/type",sysDirMmcHostPath,dirMmcCardEntry->d_name);

						fpMmcDevType = fopen(sysMmcCardTypeFile,"r");
						if(fpMmcDevType != NULL)
						{
							(void)fgets(devType,sizeof(devType)-1,fpMmcDevType);
							fclose(fpMmcDevType);
							fpMmcDevType = NULL;
							if(!strncmp(devType,"MMC",strlen("MMC")))
							{
								tU64 u64size =0;
								snprintf(mmc_dev_info->dev_path_name,sizeof(mmc_dev_info->dev_path_name)-1,DEV_PATH(PATH),s8DeviceName[count]);
								if((u64size = u64GetSize(mmc_dev_info->dev_path_name)) > 0 )
								{
									mmc_dev_info->dev_size = u64size;
									mmc_dev_info->s32fd = -1;
									u8DevicesFound++;
									//check to aviod increment while updating for the last device
									if(u8DevicesFound < u8DevicePassed )
										mmc_dev_info++;
								}
								else
								{
									log_debug("Size of the device %s returned is not greater than zero",mmc_dev_info->dev_path_name);
								}
							}
							else
							{
								log_debug("Device %s seems not be an MMC device",s8DeviceName[count]);
							}
						}
						else
						{
							log_debug("Unable to open the file %s for the device %s Errorcode : %d ErrorString : %s" \
									,sysMmcCardTypeFile,s8DeviceName[count],errno,strerror(errno));
							memset(mmc_dev_info, 0 , sizeof(emmc_dev_data));
						}
					}
				}
				closedir(dirAccess);
				dirAccess = NULL;
			}
			else
			{
				log_debug("Error accessing the Folder : %s Errorcode : %d ErrorString : %s ",sysDirMmcHostPath,errno,strerror(errno));
			}
		}
		count++;
	}
	return u8DevicesFound;
}

/********************************************************************************
 * FUNCTION        : s32FetchDeviceInformation
 * PARAMETER       : s8devName    -  eMMC device Node
 * 					 u32VendorId    - Vendor id to be populated
 * 					 u32Mmc_Revision - MMC_Revision to be populated
 *
 * RETURNVALUE     : REFRESH_NOERROR || Error Values
 * DESCRIPTION     : Reads and populates EXT_CSD_revision id and Manfid.
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

tS32 s32FetchDeviceInformation(emmc_dev_data *dev_data,tU32 *u32VendorId,tU32 *u32Mmc_Revision)
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	if(dev_data && u32VendorId && u32Mmc_Revision)
	{
		tS8 ext_csd_data[512];
		dev_data->s32fd = open(dev_data->dev_path_name, O_RDWR | O_LARGEFILE | O_SYNC | O_DIRECT);
		if(dev_data->s32fd >= 0)
		{
			s32ReturnValue = s32Read_MMC_EXT_CSD(dev_data,ext_csd_data);
			if(s32ReturnValue == REFRESH_NOERROR)
			{
				*u32Mmc_Revision = ext_csd_data[EXT_CSD_MMC_REVISION];
				log_debug("MMC Revision of the device %s is : %d",dev_data->dev_path_name,*u32Mmc_Revision);
				s32ReturnValue = s32GetManFid(dev_data->dev_path_name,u32VendorId);
				if(s32ReturnValue == REFRESH_NOERROR)
				{
					log_debug("Vendor Id for the device %s is : %d",dev_data->dev_path_name,*u32VendorId);
				}
				else
				{
					log_debug("Unable to get Vendor id for the device %s ",dev_data->dev_path_name);
				}
			}
			else
			{
				*u32Mmc_Revision = 0;
				log_error("Unable to get MMC_Revision id from the device %s ",dev_data->dev_path_name);
			}
			close(dev_data->s32fd);
		}	
		else
		{
			log_error("Unable to open the file %s ErrorCode : %d ErrorString : %s",dev_data->dev_path_name,errno, strerror(errno));
			s32ReturnValue = REFRESH_ACCESS_ERROR;
		}
		
	}
	else
	{
		log_debug("Either s8DeviceName or u32VendorId or u32Mmc_Revision is NULL");
		s32ReturnValue = REFRESH_ERROR;
	}
	return s32ReturnValue;

}

/********************************************************************************
 * FUNCTION        : s32GetManFid
 * PARAMETER       : mmcPath    -  eMMC device Node
 * 					 s32Manfid    - Vendor id to be populated
 *
 * RETURNVALUE     : REFRESH_NOERROR || Error Values
 * DESCRIPTION     : Reads the respective device sysfs manfid file and populates
 * 					 the vendor id
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

tS32 s32GetManFid(tPS8 mmcPath, tU32 *u32Manfid)
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	if(mmcPath && u32Manfid)
	{
	    char* baseFileName = basename(mmcPath);
	    char s8ManidFile[EMMC_MAX_PATH_LEN];
	    FILE *fpManIdFile = NULL;
	    if(baseFileName != NULL)
	    {
	    	snprintf(s8ManidFile,sizeof(s8ManidFile)-1,VID_PATH(VENDOR_ID_PATH),baseFileName);
			fpManIdFile = fopen(s8ManidFile,"r");
			if(fpManIdFile == NULL)
			{
				snprintf(s8ManidFile,sizeof(s8ManidFile)-1,VID_OTHERPATH(VENDOR_ID_OTHERPATH),baseFileName);
				fpManIdFile = fopen(s8ManidFile,"r");
			}
			if(fpManIdFile != NULL)
	        {
	        	(void)fscanf(fpManIdFile, "%x",u32Manfid); // CHECK this type
	            fclose(fpManIdFile);
			}
	        else
	        {
	        	log_error("Unable to open the file %s ErrorCode : %d ErrorString : %s",s8ManidFile,errno, strerror(errno));
	            s32ReturnValue = REFRESH_ACCESS_ERROR;
	        }
	    }
	    else
	    {
	    	log_error("Unable to extract baseName from the path %s",mmcPath);
	        s32ReturnValue = REFRESH_ERROR;
	    }
	}
	else
	{
		log_error("Either mmcPath or manfid is Null");
		s32ReturnValue = REFRESH_ERROR;
	}
	return s32ReturnValue;
}

/********************************************************************************
 * FUNCTION        : s32Read_MMC_EXT_CSD
 * PARAMETER       : mmcPath    -  eMMC device Node
 * 					 buffer    -  ext_csd register values
 *
 * RETURNVALUE     : REFRESH_NOERROR || Error Values
 * DESCRIPTION     : Reads the EXT_CSD values by sending MMC cmds via ioctl
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
tS32 s32Read_MMC_EXT_CSD(emmc_dev_data* dev_data,tS8 buffer[512])
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	if((dev_data != NULL) && (dev_data->s32fd >=0))
	{			
		struct mmc_ioc_cmd 	idata;
		memset(&idata, 0, sizeof(idata));
		idata.opcode	 = MMC_SEND_EXT_CSD;
		idata.arg		 = 0;
		idata.flags 	 = MMC_RSP_SPI_R1| MMC_RSP_R1 | MMC_CMD_ADTC|MMC_DATA_READ;
		idata.blksz		 = 512;
		idata.blocks	 = 1;
		mmc_ioc_cmd_set_data(idata, buffer);
		tS32 s32Ret = ioctl(dev_data->s32fd,MMC_IOC_CMD, &idata);
		if(s32Ret)
		{
			log_error("Unable to access the device using ioctl %s ErrorCode : %d ErrorString : %s",dev_data->dev_path_name,errno, strerror(errno));
			s32ReturnValue = REFRESH_ACCESS_ERROR;
		}			
	}
	else
	{
		log_debug("eMMC file path or eMMC file handle is NULL");
		s32ReturnValue = REFRESH_ACCESS_ERROR;
	}
	return s32ReturnValue;
}

/********************************************************************************
 * FUNCTION        : s32write_ExtCsd_Value
 * PARAMETER       : pc8MMCPath    -  eMMC device Node
 * 					 u8ext_csd_index    -  Register index to be updated
 * 					 u8value - Index value
 *
 * RETURNVALUE     : REFRESH_NOERROR || Error Values
 * DESCRIPTION     : Writes the value of EXT_CSD given index by sending MMC cmds via ioctl
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

tS32 s32write_ExtCsd_Value(emmc_dev_data* dev_data, tU8 u8ext_csd_index, tU8 u8value)
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	if((dev_data != NULL) && (dev_data->s32fd >=0))
	{
		struct mmc_ioc_cmd idata;
		memset(&idata, 0, sizeof(struct mmc_ioc_cmd));
		idata.write_flag = 1;
		idata.opcode = MMC_SWITCH;
		idata.arg = (MMC_SWITCH_MODE_WRITE_BYTE << 24) |
				(u8ext_csd_index << 16) |
				(u8value << 8) |
				EXT_CSD_CMD_SET_NORMAL;
		idata.flags = MMC_RSP_SPI_R1B | MMC_RSP_R1B | MMC_CMD_AC;
		tS32 s32Ret = ioctl(dev_data->s32fd, MMC_IOC_CMD, &idata);
		if(s32Ret)
		{
			log_error("Unable to access the device using ioctl %s ErrorCode : %d ErrorString : %s",dev_data->dev_path_name,errno, strerror(errno));
			s32ReturnValue = REFRESH_ACCESS_ERROR;
		}		
	}
	else
	{
		log_debug("Either emmc_dev_data is NULL or invalid file handle");
		s32ReturnValue = REFRESH_ACCESS_ERROR;
	}
	return s32ReturnValue;
}

/********************************************************************************
 * FUNCTION        : s32Read_MMC_StatusReg
 * PARAMETER       : mmcPath    -  eMMC device Node
 * 					 out_result    -  status_regiser value
 *
 * RETURNVALUE     : REFRESH_NOERROR || Error Values
 * DESCRIPTION     : Reads the cmd response values by sending MMC cmds via ioctl 
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
tS32 s32Read_MMC_StatusReg(emmc_dev_data* dev_data, tU32 *out_result)
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	tS32 ret = 0;
	if((dev_data != NULL) && (dev_data->s32fd >=0) && out_result)
	{		
		struct mmc_ioc_cmd idata;
		memset(&idata,0,sizeof(struct mmc_ioc_cmd));
		idata.opcode = MMC_SEND_STATUS;
		idata.arg = (1 << 16);
		idata.flags = MMC_RSP_R1 | MMC_CMD_AC;
		ret = ioctl(dev_data->s32fd, MMC_IOC_CMD, &idata);
		if (ret)
		{
			log_error("Unable to access the device using ioctl %s ErrorCode : %d ErrorString : %s",dev_data->dev_path_name,errno, strerror(errno));
			s32ReturnValue = REFRESH_ACCESS_ERROR;
		}
		else
			*out_result = idata.response[0];
	}
	else
	{
		log_debug("NULL Pointer exception");
		s32ReturnValue  = REFRESH_ACCESS_ERROR;
	}
	return s32ReturnValue;
}
