/*
 * Platform_logging_VRTE.cpp
 *
 *  Created on: Aug 1, 2019
 *      Author: RAP5COB
 */

#include "platform_app_manager.h"
#include "nvm_refresh.h"

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include "types.h"
#include <grp.h>
#include <sys/stat.h>
#include <errno.h>
#include <libgen.h>

#ifdef UTEST
extern	tBool s8LogDebugMessages;
extern 	tPC8 ps8DebugFilePath;
#define DIAGNOSTIC_GROUP_NAME "DUMMY"
#define ERRMEM_FILE_PATH "/tmp/errmem.txt"
#else
static tBool s8LogDebugMessages = FALSE;
static tPC8 ps8DebugFilePath = NULL;
#define ERRMEM_FILE_PATH "/dev/errmem"
#define DIAGNOSTIC_GROUP_NAME "aid_diagnosis"
#define REFRESH_FS_ERROR_FILE "/var/opt/bosch/dynamic/refresh_error.txt"
#endif
void s32WriteToErrorMemory(tPC8 pc8Message);

/********************************************************************************
 * FUNCTION        : InitApplicationLogging
 * PARAMETER       : s8LogDebugMsgs    -  Logging should be enabled or not
 * 					 s8DebugFilePath    - Dir for storing debug files
 *
 * RETURNVALUE     : void
 * DESCRIPTION     : Initializes the application logging
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

void InitApplicationLogging(tBool s8LogDebugMsgs, tPC8 s8DebugFilePath)
{
	s8LogDebugMessages = s8LogDebugMsgs;
	if(s8LogDebugMsgs == TRUE)
	{
		ps8DebugFilePath = concatPath(s8DebugFilePath,DEBUG_FILE);
		if(ps8DebugFilePath)
		{
			vRemoveFile(ps8DebugFilePath);
		}
	}
}

/********************************************************************************
 * FUNCTION        : vRemoveFile
 * PARAMETER       : filePath - file to be removed
 *
 * RETURNVALUE     : void
 * DESCRIPTION     : Removes the file
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

void vRemoveFile(tPC8 filePath)
{
	tS32 ret = remove(filePath);
	if(ret < 0)
	{
		log_debug("eMMC refresh: Unable to remove the file/folder %s ,Error code : %d ErrorString : %s\n",filePath,errno,strerror(errno));
	}
}

/********************************************************************************
 * FUNCTION        : s32WriteDebugMessage
 * PARAMETER       : message - Messages to be written to the debug file
 *
 * RETURNVALUE     : void
 * DESCRIPTION     : Writes the message into the debug file
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

void s32WriteDebugMessage(tPC8 message)
{
	if(s8LogDebugMessages == TRUE)
	{
		FILE* fDebugPtr = NULL;
		struct stat st;
		memset(&st,0,sizeof(struct stat));
		//Log file has to be re-created when it exceeds 5MB of size.
		// Using fopen instead of stat to prevent time-of-check coverity warning.
		fDebugPtr = fopen(ps8DebugFilePath,"r");
		if(fDebugPtr != NULL)
		{
			(void)fstat(fileno(fDebugPtr),&st);
			fclose(fDebugPtr);
			fDebugPtr = NULL;
		}
		if(SIZE_IN_MB(st.st_size) < 5)
			fDebugPtr = fopen(ps8DebugFilePath, "a+");
		else
			fDebugPtr = fopen(ps8DebugFilePath, "w");
		if(fDebugPtr != NULL)
		{			
			fprintf(fDebugPtr,"%s\n",message);
			fclose(fDebugPtr);
		}
		else
		{
			fprintf(stderr,"%s\n",message);
		}
	}

}

//Platform specific implementation. Do not remove it
void application_tearDown()
{

}

//Platform specific implementation. Do not remove it
void ReportApplicationStartup()
{

}

/********************************************************************************
 * FUNCTION        : log_message
 * PARAMETER       : s32LogLevel -> debug,warn,error,info
 * 					 pc8FileName -> Filename where the message comes from
 * 					 s32LineNo -> Line no of the message
 * 					 pc8FuncName -> Func name where the message comes from
 * 					 pc8Message -> Messages to be logged
 * RETURNVALUE     : void
 * DESCRIPTION     : Reports Application startup to the execution manager
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
void log_message(tS32 s32LogLevel, tPC8 pc8FileName,tS32 s32LineNo, tPC8 pc8FuncName,tPC8 pc8Message,...)
{
	if(NULL != pc8Message)
	{
		va_list vaargs;
		tPS8 s8lineBuffer = (tPS8)malloc(2048);
		if(NULL != s8lineBuffer)
		{
			memset(s8lineBuffer,0,2048);
			va_start(vaargs,pc8Message);
			snprintf(s8lineBuffer,2047,"FileName: %s FuncName: %s LineNo: %d ",basename((tPS8)pc8FileName),pc8FuncName,s32LineNo);
			vsnprintf(s8lineBuffer+strlen(s8lineBuffer),2048-strlen(s8lineBuffer),pc8Message,vaargs);
			switch(s32LogLevel)
			{
			case PLF_LOG_INFO:
			case PLF_LOG_WARN:
			case PLF_LOG_ERROR:
				if(s32LogLevel >= PLF_LOG_ERROR)
				{
					s32WriteToErrorMemory(s8lineBuffer);
					s32WriteDebugMessage(s8lineBuffer);
				}
				break;
			case PLF_LOG_DEBUG:
				s32WriteDebugMessage(s8lineBuffer);
				break;
			default:
				fprintf(stderr,"%s\n","Illegal logging level deteceted for emmc_refresh application");
				break;
			}
			va_end(vaargs);
			free(s8lineBuffer);
		}
	 }
}

/********************************************************************************
 * FUNCTION        : s32WriteToErrorMemory
 * PARAMETER       : pc8Message -> Messages to be logged
 * RETURNVALUE     : void
 * DESCRIPTION     : Writes the message to errmem
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/

void s32WriteToErrorMemory(tPC8 pc8Message)
{
	tS32 fd = open(ERRMEM_FILE_PATH, O_WRONLY );
	if( fd != -1)
	{
		if(write(fd, pc8Message,strlen(pc8Message)) <= 0)
		{
			log_debug("eMMC refresh: Unable to write into error memory\n");
		}
		(void)close(fd);
	}
	else
	{
		/* Trace Error*/
		log_debug("eMMC refresh: Unable to Open error memory Error code : %d ErrorString : %s\n",errno,strerror(errno));
	}
	// anyhow, output errormessage on stderr.
	fprintf(stderr,"%s\n",pc8Message);
}

/********************************************************************************
 * FUNCTION        : vSetGroupId
 * PARAMETER       : pc8FilePath
 * RETURNVALUE     : void
 * DESCRIPTION     : Sets the groupid for the given filePath
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
void vSetGroupId(tPC8 pc8FilePath)
{
	struct group *groupInfo = NULL;

	if (pc8FilePath == NULL)
	{
		log_debug("Folder Path is NULL\n");
	}
	else
	{
		if (groupInfo == NULL)
		{
			/*Get the group id of the diagnostic application*/
			groupInfo = getgrnam(DIAGNOSTIC_GROUP_NAME);
			if (groupInfo == NULL)
			{
				log_debug("Fail to get group Info of %s",DIAGNOSTIC_GROUP_NAME);
			}
		}

		if (groupInfo != NULL)
		{
			struct stat     fileStat;
			memset(&fileStat,0,sizeof(struct stat));
			mode_t          modes;
			int             fd;
			/* File Privilege was changed using fd : to handle the time-of-check,
			 * time-of-use coverity error */
			fd = open(pc8FilePath, O_RDONLY);
			if( fd != -1 )
			{
				if(fstat (fd, &fileStat) != -1)
				{
					if (groupInfo->gr_gid != fileStat.st_gid)
					{
						if (S_ISDIR (fileStat.st_mode))
						{
							modes = S_IRWXU |S_IRWXG |S_IROTH|S_IXOTH;
						}
						else
						{
							modes=S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP |S_IROTH;
						}
						if(fchown (fd,fileStat.st_uid,groupInfo->gr_gid) != 0)
						{
							 log_debug("%s Chown failed ErrorCode : %d ErrorString : %s", \
									pc8FilePath,errno,strerror(errno));

						}
						if(fchmod (fd, modes) != 0)
						{
							log_debug("%s chmod failed ErrorCode : %d ErrorString : %s", \
									pc8FilePath,errno,strerror(errno));

						}

					}
				}
				else
				{
					log_debug("File %s is not available ErrorCode : %d ErrorString : %s",
							pc8FilePath,errno,strerror((errno)));
				}
				close(fd);
			}
			else
			{
				log_debug("File %s is open failed ErrorCode : %d ErrorString : %s",
						pc8FilePath,errno,strerror((errno)));
			}
		}
	}
}
