/* ******************************************************FileHeaderBegin** *//**
 *
 * @file        early_config_main.c
 *
 * This file contains the main function of the process.
 *
 * global function:
 * -- main(): entry function, checks between different options
 *
 * local function:
 *                  -- vEarlyConfig_PddConfigDisplay()
 *                        function to set the display configuration
 *                  -- s32EarlyConfig_PddSetConfigDisplay():
 *                        set the display configuration
 *                  -- bEarlyConfig_PddSetConfigDisplayDefault():
 *                        set the default display configuration
 *                  -- s32EarlyConfig_PddConfigLoadDriver():
 *                        This function loads a linux driver.
 *                  -- vEarlyConfig_PrintHelp():
 *                        helper function for the process
 *
 * @date        2014-07-07
 *
 * @note
 *
 *  &copy; Copyright BoschSoftTec GmbH Hildesheim. All Rights reserved!
 *
 *//* ***************************************************FileHeaderEnd******* */
 /******************************************************************************/
 /* include the system interface                                               */
 /******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <systemd/sd-daemon.h>
#include "helper.h"
#include "system_types.h"
#include "system_definition.h"
#include "pdd.h"
#include "pdd_config_nor_user.h"
#include "pdd_config_emmc_user.h"
#include "early_config_private.h"

/*******************************************************************************
|defines and macros
|------------------------------------------------------------------------------*/

/********************************************************************************
|typedefs and struct defs
|------------------------------------------------------------------------------*/

//#define DISPLAY_PIXEL_CLOCK_EDGE_SYSFS_PATH       "find /sys/devices/soc0/soc.0/2100000.aips-bus/ -iname pixel_clock_edge"
//#define DISPLAY_SSCG_SYSFS_PATH                   "find /sys/devices/soc0/soc.0/2100000.aips-bus/ -iname SSCG"

/* NCG3D-63819: Used the ls cmd for finding the pixel_clock_edge & sscg path for the degradation in Startup Performance */
/* NCG3D-128176: Added the backward compatibility for finding the old & new path for  pixel_clock_edge,sscg & fwpatch */

#define DISPLAY_PIXEL_CLOCK_EDGE_SYSFS_PATH         "ls /sys/class/i2c-dev/i2c-*/device/*/pixel_clock_edge"
#define DISPLAY_PIXEL_CLOCK_EDGE_SYSFS_PATH_NEW     "ls /sys/class/i2c-dev/i2c-*/device/*/*/remote_pixel_clock_edge"
#define DISPLAY_DESERIALIZER_FW_PATH                "/lib/firmware/display-patches/"
#define MAXLENGTH 200
#define MAX_TOUCH 2

tsDISPLAY_CONFIG DiplayConfigLVDS[2] = { 0 };

/******************************************************************************/
/* static  variable                                                           */
/******************************************************************************/


/******************************************************************************/
/* local function                                                 */
/******************************************************************************/
static void  vEarlyConfig_PddConfigDisplay(tChar* PcStr);
static void  vEarlyConfig_PddConfigOneDisplay(void);
static void  vEarlyConfig_PddConfigMoreDisplays(tU8 Pu8Number);
static tS32  s32EarlyConfig_PddConfigLoadDriver(tChar* PcDriver);
static tS32  s32EarlyConfig_PddSetCSC(void);
static tS32  s32EarlyConfig_PddSetConfigDisplay(tsEarlyConfig_Info *PpsEarlyConfigInfo);
static tBool bEarlyConfig_PddSetConfigDisplayDefault(tsEarlyConfig_Info *PpsEarlyConfigInfo);
static void  vEarlyConfig_PrintHelp(void);
static int   EarlyConfig_FindPixelclockSSCGAndFwpatchPath(void);

static tS32 s32EarlyConfig_PddConfigMoreTouch(tChar* PcDriver);
static tS32 s32EarlyConfig_PddConfigLoadRtcTouchDriver(tChar* PcDriver);


/******************************************************************************
* FUNCTION: int main(int argc, tChar *argv[])
*
* DESCRIPTION: main thread: checks between different options
*              - t: default display timing for LVDS1 and LVDS2
*              - s: default display clock edge select for LVDS1 and LVDS2
*              - d: display configuration
*              - l: load driver (expample load RTC driver -lRtc)
*              - h: help
*              if no valid parameter given => display configuration starts
*              if target not configured, default display parameter will be use
*
* PARAMETERS:
*      argc: number of strings
*      argv: string table
*
* RETURNS:
*
* HISTORY:Created by Andrea Bueter 2015 28 07
*****************************************************************************/
static tChar* g_cpDefaultTimingLVDS = NULL;
static tChar* g_cpDefaultClockEdgeSelectLVDS = NULL;

int main(int argc, tChar *argv[])
{
    const tChar* VcOptString = "ht:s:d:l:c";
    int         ViOpt;
    tChar*       VcpStr = NULL;
    tS32        Vs32ExitCode = 0;
    tBool       VbDone = FALSE;
    tBool       VbNotify = FALSE;

    M_DEBUG_TIME();
    while ((ViOpt = getopt(argc, argv, VcOptString)) != -1)
    {
        switch (ViOpt)
        {
        case 'h':
        {
            vEarlyConfig_PrintHelp();
            VbDone = TRUE;
        }break;
        case 't':
        {/*read and set default timing LVDS1 ... will be used for unconfigured display timings*/
            if (NULL != optarg)
            {
                g_cpDefaultTimingLVDS = strdup(optarg);
                M_DEBUG_STR_STR("early_config_out: g_cpDefaultTimingLVDS is '%s' \n", g_cpDefaultTimingLVDS);
            }
            else
            {
                g_cpDefaultTimingLVDS = NULL;
            }
        }break;
        case 's':
        {/*read and set default clock edge select*/
            if (NULL != optarg)
            {
                g_cpDefaultClockEdgeSelectLVDS = strdup(optarg);
                M_DEBUG_STR_STR("early_config_out: g_cpDefaultClockEdgeSelectLVDS is '%s' \n", g_cpDefaultClockEdgeSelectLVDS);
            }
            else
            {
                g_cpDefaultClockEdgeSelectLVDS = NULL;
            }
        }break;
        case 'd':
        {/*read and set display configuration*/
            if (NULL != optarg) //Coverity Fix:117124 (FORWARD_NULL)
            {
                VcpStr = strdup(optarg);
                M_DEBUG_STR_STR("early_config_out: displays is '%s' \n", VcpStr);
                vEarlyConfig_PddConfigDisplay(VcpStr);
                free(VcpStr);
            }
            VbDone = TRUE;
            VbNotify = TRUE;
        }break;
        case 'l':
        { /*load driver */
            if (NULL != optarg) //Coverity fix:117124 (FORWARD_NULL)
            {
                VcpStr = strdup(optarg);
                Vs32ExitCode = s32EarlyConfig_PddConfigLoadDriver(VcpStr);
                free(VcpStr);
            }
            VbDone = TRUE;
        }break;
        case 'c':
        {//set Color Space Conversion 
            Vs32ExitCode = s32EarlyConfig_PddSetCSC();
            VbDone = TRUE;
            VbNotify = TRUE;
        }break;
        default:
        {/*read and set display configuration*/
            vEarlyConfig_PrintHelp();
            VbDone = TRUE;
        }break;
        }
    }
    if (VbDone == FALSE)
    { /*with no parameter => display configuration*/
        vEarlyConfig_PddConfigOneDisplay();
        VbNotify = TRUE;
    }
    M_DEBUG_TIME();
    // --- notify systemd that process early_config ready
    if (VbNotify == TRUE)
    {
        sd_notify(0, "READY=1");
        while (1)
        {
            sleep(1000); //second
        }
    }
    if (NULL != g_cpDefaultTimingLVDS)
    { /* free default timing LVDS1*/
        free(g_cpDefaultTimingLVDS);
        g_cpDefaultTimingLVDS = NULL;
    }
    if (NULL != g_cpDefaultClockEdgeSelectLVDS)
    { /* free default clock edge select LVDS1*/
        free(g_cpDefaultClockEdgeSelectLVDS);
        g_cpDefaultClockEdgeSelectLVDS = NULL;
    }
    exit(Vs32ExitCode);
}
/******************************************************************************
* FUNCTION: vEarlyConfig_PddConfigDisplay
*
* DESCRIPTION: function to set the display configuration
*                  --- read configuration data from PDD;
*                  --- if get valid data from PDD set configuration data
*                  --- if read not valid data from PDD, get default displayand set the default BC mode and LF mode
*                  --- if setting change, save diaplay configuration to NOR
*                  --- notify systemd that process early_config ready
*
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2015 07 27
*****************************************************************************/
static void  vEarlyConfig_PddConfigDisplay(tChar* PcStr)
{
    if (strncmp("2", PcStr, strlen("2")) == 0)
    {
        vEarlyConfig_PddConfigMoreDisplays(2);
    }
    else
    {
        vEarlyConfig_PddConfigOneDisplay();
    }
}
/******************************************************************************
* FUNCTION: vEarlyConfig_PddConfigOneDisplay
*
* DESCRIPTION: function to set the display configuration
*                  --- read configuration data from PDD;
*                  --- if get valid data from PDD set configuration data
*                  --- if read not valid data from PDD, get default displayand set the default BC mode and LF mode
*                  --- if setting change, save diaplay configuration to NOR
*                  --- notify systemd that process early_config ready
*
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2015 07 27
*****************************************************************************/
static void  vEarlyConfig_PddConfigOneDisplay(void)
{
    tS32               Vs32ReturnCode;
    tsEarlyConfig_Info VtsEarlyConfigInfo;
    // --- read configuration data from PDD*/
    Vs32ReturnCode = s32EarlyConfig_PddReadDisplayConfig(&VtsEarlyConfigInfo.tsPddDisplay);
    // --- print out read status
    M_DEBUG_STR_3VALUES("early_config: read state resolution: %d read state BCMode:%d read state LFMode:%d\n",
        VtsEarlyConfigInfo.tsPddDisplay.tResolution.bReadInfo,
        VtsEarlyConfigInfo.tsPddDisplay.tLFMode.bReadInfo,
        VtsEarlyConfigInfo.tsPddDisplay.tBCMode.bReadInfo);
    // --- if read valid data from PDD set configuration data
    if (Vs32ReturnCode >= 0)
    { /*set configuration of the display*/
        Vs32ReturnCode = s32EarlyConfig_PddSetConfigDisplay(&VtsEarlyConfigInfo);
    }
    // --- if read not valid data from PDD, get default display and set the default BC mode and LF mode
    else if (bEarlyConfig_PddSetConfigDisplayDefault(&VtsEarlyConfigInfo) == TRUE) //Coverity Fix: 19359 (UNUSED VALUE)
    { /*setting changed */
        // --- save display configuration to NOR 
        Vs32ReturnCode = s32EarlyConfig_PddWriteDisplayConfig(&VtsEarlyConfigInfo.tsPddDisplay);
    }
}

/******************************************************************************
* FUNCTION: vEarlyConfig_PddConfigMoreDisplays
*
* DESCRIPTION: function to set the display configuration of two display
*                  --- read configuration data from PDD;
*                  --- if get valid data from PDD set configuration data
*                  --- if setting change, save diaplay configuration to NOR
*                  --- notify systemd that process early_config ready
*
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2016 01 06
*****************************************************************************/
static void  vEarlyConfig_PddConfigMoreDisplays(tU8 Pu8Number)
{
    tS32                               Vs32ReturnCode = -1;
    tsEarlyConfig_PddDisplaysMore      VtsDisplayConfigMore[Pu8Number];

    // init 
    memset(&VtsDisplayConfigMore[0], 0, sizeof(tsEarlyConfig_PddDisplaysMore)*Pu8Number);
    // --- read configuration data from PDD for pool PDD_DATASTREAM_EARLYCONFIGTWODISPLAYS*/
    if (Pu8Number == 2)
    {
#if defined(PDD_NOR_USER_DATASTREAM_NAME_EARLYCONFIGTWODISPLAYS) || defined(PDD_RAW_EMMC_USER_DATASTREAM_NAME_EARLYCONFIGTWODISPLAYS)
        Vs32ReturnCode = s32EarlyConfig_PddReadDisplayConfigs(&VtsDisplayConfigMore[0], Pu8Number, PDD_DATASTREAM_EARLYCONFIGTWODISPLAYS);
#else
        Vs32ReturnCode = -ENOENT;
        fprintf(stderr, "early_config_err:%d pool 'EarlyConfigTwoDisplays' isn't defined\n", Vs32ReturnCode);
#endif
    }

    if (0 > Vs32ReturnCode)
    {/* no configuration information available / unconfigured device */
        tU8 Vu8Inc = 0;
        if ((NULL != g_cpDefaultTimingLVDS) && (NULL != g_cpDefaultClockEdgeSelectLVDS))
        { /* default values exist ... use default values */
            for (Vu8Inc = 0; (sizeof(VtsDisplayConfigMore) / sizeof(VtsDisplayConfigMore[0])) > Vu8Inc; ++Vu8Inc)
            {
                // timing for LVDS1
                snprintf(VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution,
                    sizeof(VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution) - 1, "%s", g_cpDefaultTimingLVDS);
                VtsDisplayConfigMore[Vu8Inc].tResolution.bReadInfo = TRUE;
                M_DEBUG_STR_VALUE_STR("early_config:TrTimingLVDS%d: using default value '%s' \n", Vu8Inc + 1,
                    VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution);
                // clock edge select for LVDS1
                VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.s32ClockEdgeSelect = atoi((const tChar*)g_cpDefaultClockEdgeSelectLVDS);
                switch (VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.s32ClockEdgeSelect)
                {
                case 1:
                {
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.s32ClockEdgeSelect = 1;
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.cpClockEdgeSelect = "1\n";
                }break;
                case 0:
                {
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.s32ClockEdgeSelect = 0;
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.cpClockEdgeSelect = "0\n";
                }break;
                case -1:
                default:
                {
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.s32ClockEdgeSelect = -1;
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.cpClockEdgeSelect = "-1\n";
                }break;
                }
                VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.bReadInfo = TRUE;
                M_DEBUG_STR_2VALUES("early_config:TrClockEdgeSelectLVDS%d: using default '%d' \n", Vu8Inc + 1,
                    VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect.s32ClockEdgeSelect);
                // LVDS2 unused
                // commit using default values
                Vs32ReturnCode = 0;
            }
        }
        else
        { /* no default values specified ... abort */
            fprintf(stderr, "early_config:TrTimingLVDSx: no default value defined \n");
            fprintf(stderr, "early_config:TrClockEdgeSelectLVDSx: no default default defined \n");
        }
    }
    // --- if read valid data from PDD set configuration data
    if (Vs32ReturnCode >= 0)
    {
        tU8 Vu8Inc;
        tU8 Vu8NumberClockEdgeSelect = 0;
        tU8 _u8CountOfTimings = 0;

        /*for each display configuration*/
        tsEarlyConfig_PddDisplayResolution   *VtspResolution;
        tsEarlyConfig_PddDisplayClockEdgeSelect *VtspClockEdgeSelect;
        tsEarlyConfig_PddDisplaySSCG  *VtspSSCG;

        Vs32ReturnCode = EarlyConfig_FindPixelclockSSCGAndFwpatchPath();
        if (Vs32ReturnCode == 0)
        {
            for (Vu8Inc = 0; Vu8Inc < Pu8Number; Vu8Inc++)
            { /*print out read status*/

             /*-------TrResolution -----------*/
                VtspResolution = &VtsDisplayConfigMore[Vu8Inc].tResolution;

                /*------TrSpreadSpectrumClockGenerationLVDS1 ---*/
                VtspSSCG = &VtsDisplayConfigMore[Vu8Inc].tSSCG;

                /*-----  TrClockEdgeSelectLVDS ---*/
                VtspClockEdgeSelect = &VtsDisplayConfigMore[Vu8Inc].tClockEdgeSelect;

                /*
                   *Comments from Wohlrab Knut (CM/ESO2)
                   * The SSCG value must be set before the display timing TrTimingLVDS1/2 is activated! Otherwise it may cause a permanent display distortion.
                   */

                   /*-----  TrSpreadSpectrumClockGenerationLVDS ---*/
                if (VtspSSCG->bReadInfo == TRUE)
                {
                    M_DEBUG_STR_2VALUES("early_config:TrSpreadSpectrumClockGenerationLVDS%d: '%d' \n", Vu8Inc + 1, VtspSSCG->s32SpreadSpectrumClockGeneration);
                    /*set configuration of the display*/
                    Vs32ReturnCode = s32EarlyConfig_SetSpreadSpectrumClockGeneration(VtspSSCG->cpSpreadSpectrumClockGeneration, Vu8Inc);
                }
                else
                {
                    M_DEBUG_STR_VALUE("early_config:TrSpreadSpectrumClockGenerationLVDS%d: '-' \n", Vu8Inc + 1);
                }
                /*-----  TrTimingLVDS ---*/
                if ((VtspResolution->bReadInfo == TRUE) && (0 < strlen(VtspResolution->cResolution)))
                { // data could be read and resolution is not empty!
                    ++_u8CountOfTimings;
                    M_DEBUG_STR_VALUE_STR("early_config:TrTimingLVDS%d: '%s' \n", Vu8Inc + 1, VtspResolution->cResolution);
                    /*set configuration of the display*/
                    Vs32ReturnCode = s32EarlyConfig_SetDisplayTiming(VtspResolution->cResolution, TRUE, (tU8)(Vu8Inc + (tU8)1));
                    if (0 > Vs32ReturnCode)
                    {/* timing could not set */
                        memset(VtspResolution->cResolution, 0, sizeof(VtspResolution->cResolution));
                        /* read first timing */
                        Vs32ReturnCode = s32EarlyConfig_ReadDisplayString(VtspResolution->cResolution, EARLY_CONFIG_KIND_ACTION_GET_FIRST);
                        /* read actual timing ... if actual timing set then resolution will be overwritten */
                        Vs32ReturnCode = s32EarlyConfig_ReadDisplayString(VtspResolution->cResolution, EARLY_CONFIG_KIND_ACTION_GET_ACTUAL);
                        /* set actual or first timing */
                        if (0 < Vs32ReturnCode)
                        {/* actual or first timing available */
                            Vs32ReturnCode = s32EarlyConfig_SetDisplayTiming(VtspResolution->cResolution, TRUE, (tU8)(Vu8Inc + (tU8)1));
                            if (0 > Vs32ReturnCode)
                            {/* actual or first timing could not set */
                                fprintf(stderr, "early_config_err:TrTimingLVDS%d: '%s' (fail save) \n", Vu8Inc + 1, VtspResolution->cResolution);
                            }
                            else
                            {
                                M_DEBUG_STR_VALUE_STR("early_config:TrTimingLVDS%d: '%s' saved \n", Vu8Inc + 1, VtspResolution->cResolution);
                            }
                        }
                        else
                        {/* actual or first timing unavailable */
                            fprintf(stderr, "early_config_err:TrTimingLVDS%d: '%s' (fail save) \n", Vu8Inc + 1, "---");
                        }
                    }
                }
                else
                    fprintf(stderr, "early_config:TrTimingLVDS%d: '-' \n", Vu8Inc + 1);

                //Updating fwpatch with the display type
                if ((VtspResolution->bReadInfo == TRUE) && (0 < strlen(VtspResolution->cResolution)))
                {
                    size_t size = 0;
                    tChar *DiplayType = NULL;
                    tChar *DisplayDeserializerPath = NULL;
                    FILE  *fpDisplayDeserializerPath = NULL;
                    FILE  *fpFwpatchSysFsPath = NULL;
                    tChar *Databuffer = NULL;

                    DiplayType = (tChar *)malloc(strlen(VtspResolution->cResolution) + strlen(".raw") + 1);
                    if (NULL != DiplayType)
                    {
                        memset(DiplayType, 0, (strlen(VtspResolution->cResolution) + strlen(".raw") + 1));
                        strncpy(DiplayType, VtspResolution->cResolution, strlen(VtspResolution->cResolution )+1);
                        strncat(DiplayType, ".raw",strlen(".raw")+1);

                        DisplayDeserializerPath = malloc(strlen(DISPLAY_DESERIALIZER_FW_PATH) + strlen((const tChar *)DiplayType) + 1);
                        if (NULL != DisplayDeserializerPath)
                        {
                            memset(DisplayDeserializerPath, 0, (strlen(DISPLAY_DESERIALIZER_FW_PATH) + strlen((const tChar *)DiplayType) + 1));
                            strncpy(DisplayDeserializerPath, DISPLAY_DESERIALIZER_FW_PATH, strlen(DISPLAY_DESERIALIZER_FW_PATH)+1);
                            strncat(DisplayDeserializerPath, (const tChar *)DiplayType, strlen((const tChar *)DiplayType)+1);

                            fpDisplayDeserializerPath = fopen(DisplayDeserializerPath, "rb");
                            if (NULL == fpDisplayDeserializerPath)
                            {
                                fprintf(stderr, "early_config_err: open failed for display deserializer %s with errno %d \n", DisplayDeserializerPath, errno);
                            }
                            else
                            {
                                if (NULL != DiplayConfigLVDS[Vu8Inc].Fwpatchpath)
                                {
                                    fpFwpatchSysFsPath = fopen((const tChar *)DiplayConfigLVDS[Vu8Inc].Fwpatchpath, "wb");
                                    if (NULL != fpFwpatchSysFsPath)
                                    {
                                        size = strlen(DiplayType) + strlen("display-patches/");
                                        Databuffer = (tChar *)malloc(size + 1);
                                        if ( NULL != Databuffer )
                                        {
                                            memset(Databuffer, 0, (size + 1));
                                            strncpy(Databuffer, "display-patches/", strlen("display-patches/")+1);
                                            strncat(Databuffer, (const tChar *)DiplayType, strlen((const tChar *)DiplayType)+1);
    
                                            if (size != fwrite(Databuffer, 1, size, fpFwpatchSysFsPath))
                                            {
                                                fprintf(stderr, "early_config_err: write failed in %s  with errno %d \n", DiplayConfigLVDS[Vu8Inc].Fwpatchpath, errno);
                                                Vs32ReturnCode = -1;
                                            }                                            
                                            free(Databuffer);
                                        }
                                        fclose(fpFwpatchSysFsPath);
                                    }
                                    else
                                    {
                                        fprintf(stderr, "early_config_err: Fwpatch file not found with errno %d", errno);
                                        Vs32ReturnCode = -1;
                                    }
                                }
                                fclose(fpDisplayDeserializerPath);
                            }
                            free(DisplayDeserializerPath);
                        }
                        free(DiplayType);
                    }
                }
                /*-----  TrClockEdgeSelectLVDS ---*/
                if (VtspClockEdgeSelect->bReadInfo == TRUE)
                {
                    M_DEBUG_STR_2VALUES("early_config:TrClockEdgeSelectLVDS%d: '%d'\n", Vu8Inc + 1, VtspClockEdgeSelect->s32ClockEdgeSelect);
                    /*set configuration of the display*/
                    if ((VtspClockEdgeSelect->s32ClockEdgeSelect == 0) || (VtspClockEdgeSelect->s32ClockEdgeSelect == 1))
                    {
                        Vu8NumberClockEdgeSelect++;
                        if (Vu8NumberClockEdgeSelect > 1)
                        {/*at moment only one clock edge select could be set!!! */
                            tU8 Vu8Temp = (tU8)(Vu8Inc + (tU8)1);
                            M_DEBUG_STR_VALUE("early_config_err:TrClockEdgeSelectLVDS%d could not be set; configuration error: only one parameter 'TrClockEdgeSelectLVDS' could be set at moment\n", Vu8Temp);
                            vEarlyConfig_SetErrorEntry(EARLY_CONFIG_EM_ERROR_CONFIGURATION_WRONG, &Vu8Temp, sizeof(tU8));
                        }
                        else
                        {
                            Vs32ReturnCode = s32EarlyConfig_SetClockEgdeSelect(VtspClockEdgeSelect->cpClockEdgeSelect, Vu8Inc);
                        }
                    }
                }
                else
                {
                    fprintf(stderr, "early_config:TrClockEdgeSelectLVDS%d: '-' \n", Vu8Inc + 1);
                }
            }
            // check for empty display timings only
            if (0 == _u8CountOfTimings)
            { // no display activated ... activate both with timing "emptyTiming"
                vEarlyConfig_SetErrorEntry(EARLY_CONFIG_EM_ERROR_WRITE_DISPLAY_EMPTYSTRING, NULL, 0);
                for (Vu8Inc = 0; Vu8Inc < Pu8Number; Vu8Inc++)
                {
                    memset(VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution, 0, sizeof(VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution));
                    /* read first timing */
                    Vs32ReturnCode = s32EarlyConfig_ReadDisplayString(VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution, EARLY_CONFIG_KIND_ACTION_GET_FIRST);
                    /* set first timing */
                    Vs32ReturnCode = s32EarlyConfig_SetDisplayTiming(VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution, TRUE, (tU8)(Vu8Inc + (tU8)1));
                    if (0 > Vs32ReturnCode)
                    {/* first timing could not set */
                        fprintf(stderr, "early_config_err:TrTimingLVDS%d: '%s' (fail save) \n", Vu8Inc + 1, VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution);
                    }
                    else
                    {
                        fprintf(stderr, "early_config:TrTimingLVDS%d: '%s' saved \n", Vu8Inc + 1, VtsDisplayConfigMore[Vu8Inc].tResolution.cResolution);
                    }
                }
            }
        }
        else
        {
            fprintf(stderr, "early_config_err:%d Pixelclock Path not found\n", Vs32ReturnCode);
        }
        for (Vu8Inc = 0; Vu8Inc < 2; Vu8Inc++)
        {
            if (DiplayConfigLVDS[Vu8Inc].Pixelpath != NULL)
            {
                free(DiplayConfigLVDS[Vu8Inc].Pixelpath);
            }
            if (DiplayConfigLVDS[Vu8Inc].SSCGpath != NULL)
            {
                free(DiplayConfigLVDS[Vu8Inc].SSCGpath);
            }
            if (DiplayConfigLVDS[Vu8Inc].Fwpatchpath != NULL)
            {
                free(DiplayConfigLVDS[Vu8Inc].Fwpatchpath);
            }
        }
    }
    else
    {/*print out no valid configuration read*/
        fprintf(stderr, "early_config_err:%d no valid configuration read\n", Vs32ReturnCode);
    }
}
/******************************************************************************
* FUNCTION: tS32 s32EarlyConfig_PddSetConfigDisplay()

* DESCRIPTION: set the display configuration
*
* PARAMETERS:
*         PpsEarlyConfigInfo: info structure of the process
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2014 06 24
*****************************************************************************/
static tS32 s32EarlyConfig_PddSetConfigDisplay(tsEarlyConfig_Info *PpsEarlyConfigInfo)
{
    tS32 Vs32ReturnCode = 0;
    if (PpsEarlyConfigInfo->tsPddDisplay.tResolution.bReadInfo == TRUE)
    { // set display configuration
#ifdef EARLY_CONFIG_SET_DISPLAY_CONFIG_TO_FILE
        Vs32ReturnCode = s32EarlyConfig_SetDisplayTiming(PpsEarlyConfigInfo->tsPddDisplay.tResolution.cResolution, FALSE, 0);
#else
        PpsEarlyConfigInfo->tsDRMAccess.eKindAction = EARLY_CONFIG_KIND_ACTION_SET;
        Vs32ReturnCode = s32EarlyConfig_DrmResolution(PpsEarlyConfigInfo);
#endif
    }
    if ((PpsEarlyConfigInfo->tsPddDisplay.tBCMode.bReadInfo == TRUE) && (PpsEarlyConfigInfo->tsPddDisplay.tLFMode.bReadInfo == TRUE))
    {
        s32EarlyConfig_SetModes(PpsEarlyConfigInfo->tsPddDisplay.tBCMode.cpBCMode, PpsEarlyConfigInfo->tsPddDisplay.tLFMode.cpLFMode);
    }
    return(Vs32ReturnCode);
}
/******************************************************************************
* FUNCTION: tS32 s32EarlyConfig_PddSetConfigDisplayDefault()
*
* DESCRIPTION: set the default display configuration
*
* PARAMETERS:
*         PpsEarlyConfigInfo: info structure of the process
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2014 06 24
*****************************************************************************/
static tBool bEarlyConfig_PddSetConfigDisplayDefault(tsEarlyConfig_Info *PpsEarlyConfigInfo)
{
    tBool VbChange = FALSE;
    //check if read valid parameter from PDD_UserEarly => if saved, saved the default value into PDD_UserEarly
    if ((PpsEarlyConfigInfo->tsPddDisplay.tResolution.bReadInfo == FALSE)
        || (PpsEarlyConfigInfo->tsPddDisplay.tBCMode.bReadInfo == FALSE)
        || (PpsEarlyConfigInfo->tsPddDisplay.tLFMode.bReadInfo == FALSE))
    {/*get and set display configuration*/
        VbChange = TRUE;
#ifndef EARLY_CONFIG_SET_DISPLAY_CONFIG_TO_FILE    
        PpsEarlyConfigInfo->tsDRMAccess.eKindAction = EARLY_CONFIG_KIND_ACTION_GET_SET;
        if (s32EarlyConfig_DrmResolution(PpsEarlyConfigInfo) < 0)
        {
            VbChange = FALSE;
        }
#else
        if (s32EarlyConfig_GetSetDisplay(&PpsEarlyConfigInfo->tsPddDisplay) < 0)
        {
            VbChange = FALSE;
        }
#endif    
        // search for 'displayGM' in the string
        if (strncmp(PpsEarlyConfigInfo->tsPddDisplay.tResolution.cResolution, "displayGM", strlen("displayGM")) == 0)
        { //set default for GM BCMode/LFMode
            PpsEarlyConfigInfo->tsPddDisplay.tBCMode.s32BCMode = EARLY_CONFIG_DISPLAY_DEVICE_BC_MODE_DEFAULT_GM;
            PpsEarlyConfigInfo->tsPddDisplay.tBCMode.cpBCMode = EARLY_CONFIG_DISPLAY_DEVICE_BC_MODE_DEFAULT_STR_GM;
            PpsEarlyConfigInfo->tsPddDisplay.tLFMode.s32LFMode = EARLY_CONFIG_DISPLAY_DEVICE_LF_MODE_DEFAULT_GM;
            PpsEarlyConfigInfo->tsPddDisplay.tLFMode.cpLFMode = EARLY_CONFIG_DISPLAY_DEVICE_LF_MODE_DEFAULT_STR_GM;
        }
        else
        {
            PpsEarlyConfigInfo->tsPddDisplay.tBCMode.s32BCMode = EARLY_CONFIG_DISPLAY_DEVICE_BC_MODE_DEFAULT;
            PpsEarlyConfigInfo->tsPddDisplay.tBCMode.cpBCMode = EARLY_CONFIG_DISPLAY_DEVICE_BC_MODE_DEFAULT_STR;
            PpsEarlyConfigInfo->tsPddDisplay.tLFMode.s32LFMode = EARLY_CONFIG_DISPLAY_DEVICE_LF_MODE_DEFAULT;
            PpsEarlyConfigInfo->tsPddDisplay.tLFMode.cpLFMode = EARLY_CONFIG_DISPLAY_DEVICE_LF_MODE_DEFAULT_STR;
        }
        //set BC mode set LF mode
        if (s32EarlyConfig_SetModes(PpsEarlyConfigInfo->tsPddDisplay.tBCMode.cpBCMode, PpsEarlyConfigInfo->tsPddDisplay.tLFMode.cpLFMode) < 0)
        {
            VbChange = FALSE;
        }
    }
    return(VbChange);
}
/******************************************************************************
* FUNCTION: tS32 s32EarlyConfig_PddConfigLoadRtcTouchDriver()
*
* DESCRIPTION: This function loads a linux driver for single touch and Rtc
*
* PARAMETERS: PcDriver: string of the kind of driver, which should be loaded
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2015 27 07
*****************************************************************************/
static tS32  s32EarlyConfig_PddConfigLoadRtcTouchDriver(tChar* PcDriver)
{
    tChar VcDriverName[EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT];
    tChar VcConfigFileName[EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT];
    tChar VcI2cAddress[EARLY_CONFIG_I2CADDRESS_STRING_LENGHT];

    memset(VcConfigFileName, 0, EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT);
    memset(VcDriverName, 0, EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT);
    memset(VcI2cAddress, 0, EARLY_CONFIG_I2CADDRESS_STRING_LENGHT);

    tS32 Vs32ReturnCode = s32EarlyConfig_PddReadDriverName(PcDriver, VcDriverName, VcConfigFileName, VcI2cAddress, 0); //The last parameter is used for Touch2
    if (Vs32ReturnCode == 0)
    {
        M_DEBUG_STR_STR("early_config: read driver name '%s' \n", VcDriverName);
        //check if name of configuration file read 
        if (VcConfigFileName[0] == 0)
        {
            Vs32ReturnCode = s32EarlyConfig_KmodSetDriverName(VcDriverName); //This is the case for loading RTC Linux driver
        }
        else
        {
            M_DEBUG_STR_STR("early_config: read configuration file name '%s' \n", VcConfigFileName);
            s32EarlyConfig_KmodStartDriverWithScript(VcDriverName, VcConfigFileName, VcI2cAddress);  // This is the case for loading touch linux driver
        }
    }
    else
    {
        fprintf(stderr, "early_config_err: %d, no configuration found for '%s' driver\n", Vs32ReturnCode, PcDriver);
        if (strncmp("Touch", PcDriver, strlen(PcDriver)) == 0)
        {
            //if Touch set default
            strncpy(VcDriverName, EARLY_CONFIG_DEFAULT_TOUCH_DRIVER_NAME, strlen(EARLY_CONFIG_DEFAULT_TOUCH_DRIVER_NAME));
            strncpy(VcConfigFileName, EARLY_CONFIG_DEFAULT_TOUCH_CONFIG_NAME, strlen(EARLY_CONFIG_DEFAULT_TOUCH_CONFIG_NAME));
            Vs32ReturnCode = 0;

            M_DEBUG_STR_STR("early_config: set default driver '%s' \n", VcDriverName);
            M_DEBUG_STR_STR("early_config: set default configuration file name '%s' \n", VcConfigFileName);
            s32EarlyConfig_KmodStartDriverWithScript(VcDriverName, VcConfigFileName, VcI2cAddress);
        }
        else
        { /*error memory entry*/
            vEarlyConfig_SetErrorEntry(EARLY_CONFIG_EM_ERROR_LOAD_DRIVER, (tU8*)VcDriverName, (tU32)(strlen(VcDriverName) + 1));  //Case:If the Touch driver name is not found, then there is no need to load the default values
        }
    }
    return(Vs32ReturnCode);
}
/******************************************************************************
* FUNCTION: tS32 s32EarlyConfig_PddSetCSC()
*
* DESCRIPTION: This function reads the CSC pool from the PDD
*              and sets this configuration
*
* PARAMETERS: void
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Andrea Bueter 2015 27 07
*****************************************************************************/
static tS32  s32EarlyConfig_PddSetCSC(void)
{
    tS32                Vs32ReturnCode = 0;
#ifdef EARYL_CONFIG_USE_CSC_LIB
    tsEarlyConfig_CSC   VtsEarlyConfigCSC;

    //init
    vEarlyConfig_CSCInit(&VtsEarlyConfigCSC);
#ifdef EARLY_CONFIG_DEBUG
    //print out configuration
    fprintf(stderr, "early_config: default configuration: Hue=%d, Contrast=%d, Saturation=%d, Brightness=%d, HueOffset=%d, SaturationOffset=%d, BrightnessOffset=%d\n",
        VtsEarlyConfigCSC.tsCscProp.hue, VtsEarlyConfigCSC.tsCscProp.saturation, VtsEarlyConfigCSC.tsCscProp.brightness, VtsEarlyConfigCSC.tsCscProp.contrast,
        VtsEarlyConfigCSC.tsCscProp.hue_off, VtsEarlyConfigCSC.tsCscProp.saturation_off, VtsEarlyConfigCSC.tsCscProp.brightness_off);
#endif
    //read configuration
    Vs32ReturnCode = s32EarlyConfig_PddReadCSC(&VtsEarlyConfigCSC);
    if (Vs32ReturnCode == 0)
    {
#ifdef EARLY_CONFIG_DEBUG
        //print out configuration
        fprintf(stderr, "early_config: read current profile: %d\n", VtsEarlyConfigCSC.tU8Profile);
        fprintf(stderr, "early_config: read configuration: Hue=%d, Contrast=%d, Saturation=%d, Brightness=%d, HueOffset=%d, SaturationOffset=%d, BrightnessOffset=%d\n",
            VtsEarlyConfigCSC.tsCscProp.hue, VtsEarlyConfigCSC.tsCscProp.contrast, VtsEarlyConfigCSC.tsCscProp.saturation, VtsEarlyConfigCSC.tsCscProp.brightness,
            VtsEarlyConfigCSC.tsCscProp.hue_off, VtsEarlyConfigCSC.tsCscProp.saturation_off, VtsEarlyConfigCSC.tsCscProp.brightness_off);
        fprintf(stderr, "early_config: read gamma factor: %.2f\n", VtsEarlyConfigCSC.tF64GammaValue);
#endif
        //set configuration if profile valid
        if ((VtsEarlyConfigCSC.tU8Profile != 0) && (VtsEarlyConfigCSC.tU8Profile != 0xff))
        {
            Vs32ReturnCode = s32EarlyConfig_CSCAndGammaSet(&VtsEarlyConfigCSC);
#ifdef EARLY_CONFIG_DEBUG
            if (Vs32ReturnCode == 0)
            {
                fprintf(stderr, "early_config: success: set CSC and gamma \n");
            }
#endif
        }
        else
        {
            fprintf(stderr, "early_config_err: %d no valid profile found \n", VtsEarlyConfigCSC.tU8Profile);
        }
    }
#endif
    return(Vs32ReturnCode);
}
/******************************************************************************
* FUNCTION: tS32 vEarlyConfig_PrintHelp()
*
* DESCRIPTION: helper function for the process
*
* HISTORY:Created by Andrea Bueter 2015 27 07
*****************************************************************************/
static void  vEarlyConfig_PrintHelp(void)
{
    fprintf(stderr, "\n");
    fprintf(stderr, "========================early_config ===========================================\n");
    fprintf(stderr, " 1) without parameter: set display configuration (Datapool: EarlyConfigDisplay) \n");
    fprintf(stderr, " 2) with parameter: \n");
    fprintf(stderr, "       -h print out options\n");
    fprintf(stderr, "       -t[default_timing_LVDS1] default display timing LVDS1 and LVDS2 (Will be used if Datapool EarlyConfigDisplay not set!)\n");
    fprintf(stderr, "       -s[default_clock_edge_select {-1|0|1}] default display clock edge select LVDS1 and LVDS2 (Will be used if Datapool EarlyConfigDisplay not set!)\n");
    fprintf(stderr, "       -d1 set display configuration (Datapool: EarlyConfigDisplay)\n");
    fprintf(stderr, "       -d2 set display configuration for two displays (Datapool: EarlyConfigTwoDisplays)\n");
    fprintf(stderr, "       -l[typedriver] load driver (e.g. -lRtc) \n");
    fprintf(stderr, "           [typedriver]='Rtc' (Datapool: EarlyConfigRTCDriver) \n");
    fprintf(stderr, "           [typedriver]='Touch' (Datapool: EarlyConfigTouchDriver) \n");
    fprintf(stderr, "           [typedriver]='Touch2' (Datapool: EarlyConfigTwoTouchDriver) \n");
    fprintf(stderr, "       -c set Color Space Conversion  (Datapool: EarlyConfigCSC)\n");
    fprintf(stderr, "================================================================================\n");
}

/******************************************************************************
* FUNCTION:  EarlyConfig_FindPixelclockSSCGAndFwpatchPath()
*
* DESCRIPTION: This function used to find the Pixel, SSCG and fwpatch path for LVDS1/2
*
*
* PARAMETERS: void
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Selvan Krishnan 10.04.2017
*****************************************************************************/
static int EarlyConfig_FindPixelclockSSCGAndFwpatchPath(void)
{
    FILE   *fpDisplayPixelClockEdgePath = NULL;
    tChar   *line = NULL;
    tU32   len = 0;
    tS32   s32Read;
    tU8    u8DisplayNo;
    tS32   Vs32ReturnCode = -1;
    tU8    Count = 0;
    tBool bNewPathFound=FALSE;


    memset(DiplayConfigLVDS, 0, sizeof(tsDISPLAY_CONFIG) * 2);//Fixed Coverity CID 117133: Out-of-bounds access (OVERRUN)

    for (u8DisplayNo = 0; u8DisplayNo < 2; u8DisplayNo++)
    {
        DiplayConfigLVDS[u8DisplayNo].Pixelpath = (tChar *)malloc(200);
        if (NULL != DiplayConfigLVDS[u8DisplayNo].Pixelpath)
        {
            memset((void*)DiplayConfigLVDS[u8DisplayNo].Pixelpath, 0, 200);
        }
        else
        {
            fprintf(stderr, "early_config_err: Memory Allocation failed \n");
            break;
        }

        DiplayConfigLVDS[u8DisplayNo].SSCGpath = (tChar *)malloc(200);
        if (NULL != DiplayConfigLVDS[u8DisplayNo].SSCGpath)
        {
            memset((void*)DiplayConfigLVDS[u8DisplayNo].SSCGpath, 0, 200);
        }
        else
        {
            fprintf(stderr, "early_config_err: Memory Allocation failed \n");
            break;
        }

        DiplayConfigLVDS[u8DisplayNo].Fwpatchpath = (tChar *)malloc(200);
        if (NULL != DiplayConfigLVDS[u8DisplayNo].Fwpatchpath)
        {
            memset((void*)DiplayConfigLVDS[u8DisplayNo].Fwpatchpath, 0, 200);
        }
        else
        {
            fprintf(stderr, "early_config_err: Memory Allocation failed \n");
            break;
        }

        if (u8DisplayNo == 1)
        {
            Vs32ReturnCode = 0; //This will be set only for the sucessful memory allocation.
        }
    }

    if (Vs32ReturnCode == 0)
    {
        u8DisplayNo = 0; //Reset the Display count

        fpDisplayPixelClockEdgePath = popen(DISPLAY_PIXEL_CLOCK_EDGE_SYSFS_PATH_NEW, "r");

        if (NULL == fpDisplayPixelClockEdgePath)
        {
            fprintf(stderr, "early_config_err: open failed for New PixelClockEdgePath \n");
            Vs32ReturnCode = -1;
        }
        else
        {
            while (((s32Read = (tS32)getline(&line, (size_t *)&len, fpDisplayPixelClockEdgePath)) != -1) && (u8DisplayNo <= (tU8)1))
            {
                tChar*  VcpStr = NULL;
                VcpStr = strchr(line, '\n');
                if (VcpStr != NULL)
                {
                    *VcpStr = '\0';
                }
                strncat((tChar *)DiplayConfigLVDS[u8DisplayNo].Pixelpath, line, 200);
                bNewPathFound=TRUE;                
                M_DEBUG_STR_VALUE_STR("early_config: PixelcloclEdgePath for the LVDS%d is '%s' \n", u8DisplayNo, (tChar *)DiplayConfigLVDS[u8DisplayNo].Pixelpath);
                u8DisplayNo++;
            }
            pclose(fpDisplayPixelClockEdgePath);   //Coverity fix:117105:ALLOC_FREE_MISMATCH

            if (line)
                free(line);
            line = NULL;
            
        }

		
        if ( !bNewPathFound)
        {
            fpDisplayPixelClockEdgePath = popen(DISPLAY_PIXEL_CLOCK_EDGE_SYSFS_PATH, "r");

            if (NULL == fpDisplayPixelClockEdgePath)
            {
                fprintf(stderr, "early_config_err: open failed for old PixelClockEdgePath \n");
                Vs32ReturnCode = -1;
            }
            else
            {
                while (((s32Read = (tS32)getline(&line, (size_t *)&len, fpDisplayPixelClockEdgePath)) != -1) && (u8DisplayNo <= (tU8)1))
                {
                    tChar*  VcpStr = NULL;
                    VcpStr = strchr(line, '\n');
                    if (VcpStr != NULL)
                    {
                        *VcpStr = '\0';
                    }
                    strncat((tChar *)DiplayConfigLVDS[u8DisplayNo].Pixelpath, line, 200);
                    M_DEBUG_STR_VALUE_STR("early_config: PixelcloclEdgePath for the LVDS%d is '%s' \n", u8DisplayNo, (tChar *)DiplayConfigLVDS[u8DisplayNo].Pixelpath);
                    u8DisplayNo++;
                }
                pclose(fpDisplayPixelClockEdgePath);   //Coverity fix:117105:ALLOC_FREE_MISMATCH

                if (line)
                    free(line);
                line = NULL;
            }
        }                    

        if (Vs32ReturnCode == 0)
        {
            for (Count = 0; Count < u8DisplayNo; Count++)
            {
               if(bNewPathFound)
               {
                  strncpy(DiplayConfigLVDS[Count].SSCGpath, (const tChar *)DiplayConfigLVDS[Count].Pixelpath, (strlen((const tChar *)DiplayConfigLVDS[Count].Pixelpath) - strlen("remote_pixel_clock_edge")));
                  strncat(DiplayConfigLVDS[Count].SSCGpath, "remote_sscg", strlen("remote_sscg")+1);
               
                  strncpy(DiplayConfigLVDS[Count].Fwpatchpath, (const tChar *)DiplayConfigLVDS[Count].Pixelpath, (strlen((const tChar *)DiplayConfigLVDS[Count].Pixelpath) - strlen("remote_pixel_clock_edge")));
                  strncat(DiplayConfigLVDS[Count].Fwpatchpath, "remote_fwpatch", strlen("remote_fwpatch")+1);
               }
               else
               {
                  strncpy(DiplayConfigLVDS[Count].SSCGpath, (const tChar *)DiplayConfigLVDS[Count].Pixelpath, (strlen((const tChar *)DiplayConfigLVDS[Count].Pixelpath) - strlen("pixel_clock_edge")));
                  strncat(DiplayConfigLVDS[Count].SSCGpath, "sscg", strlen("sscg")+1);
               
                  strncpy(DiplayConfigLVDS[Count].Fwpatchpath, (const tChar *)DiplayConfigLVDS[Count].Pixelpath, (strlen((const tChar *)DiplayConfigLVDS[Count].Pixelpath) - strlen("pixel_clock_edge")));
                  strncat(DiplayConfigLVDS[Count].Fwpatchpath, "fwpatch", strlen("fwpatch")+1);
               }
            }
        } 
    }
    return Vs32ReturnCode;
}

/******************************************************************************
* FUNCTION:  s32EarlyConfig_PddConfigLoadDriver()
*
* DESCRIPTION: Find out whether to load the linux driver for single or double touch controller
*
* PARAMETERS: Touch type
*
* RETURNS: success(0)
*
* HISTORY:Created by Selvan Krishnan 10.04.2017
*****************************************************************************/
static tS32 s32EarlyConfig_PddConfigLoadDriver(tChar* PcStr)
{
    tS32 Vs32ReturnCode = 0;

    if (strncmp("Touch2", PcStr, strlen("Touch2")) == 0)
    {
        Vs32ReturnCode = s32EarlyConfig_PddConfigMoreTouch(PcStr);
    }
    else
    {
        Vs32ReturnCode = s32EarlyConfig_PddConfigLoadRtcTouchDriver(PcStr); //  Both Touch and Rtc
    }
    return Vs32ReturnCode;
}

/******************************************************************************
* FUNCTION:  s32EarlyConfig_PddConfigMoreTouch()
*
* DESCRIPTION: This function used to load the linux driver for the double touch controller
*
*
* PARAMETERS: Touch type
*
* RETURNS: success(0) or error code(<0)
*
* HISTORY:Created by Selvan Krishnan 10.04.2017
*****************************************************************************/
static tS32 s32EarlyConfig_PddConfigMoreTouch(tChar* PcDriver)
{
    tS32 Vs32ReturnCode = 0;

#if defined (PDD_NOR_USER_DATASTREAM_NAME_EARLYCONFIGTWOTOUCHDRIVER) || defined (PDD_RAW_EMMC_USER_DATASTREAM_NAME_EARLYCONFIGTWOTOUCHDRIVER)
    tChar VcDriverName[EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT];
    tChar VcConfigFileName[EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT];
    tChar VcI2cAddress[EARLY_CONFIG_I2CADDRESS_STRING_LENGHT];
    tU8 count;
    for (count = 0; count < MAX_TOUCH; count++)
    {
        memset(VcConfigFileName, 0, EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT);
        memset(VcDriverName, 0, EARLY_CONFIG_DRIVER_NAME_STRING_LENGHT);
        memset(VcI2cAddress, 0, EARLY_CONFIG_I2CADDRESS_STRING_LENGHT);

        Vs32ReturnCode = s32EarlyConfig_PddReadDriverName(PcDriver, VcDriverName, VcConfigFileName, VcI2cAddress, count);
        if (Vs32ReturnCode == 0)
        {
            s32EarlyConfig_KmodStartDriverWithScript(VcDriverName, VcConfigFileName, VcI2cAddress);
        }
        else
        {
            vEarlyConfig_SetErrorEntry(EARLY_CONFIG_EM_ERROR_LOAD_DRIVER, PcDriver, strlen(PcDriver) + 1);
        }
    }
#else
    Vs32ReturnCode = -ENOENT;
    (void)PcDriver; // intentionally unused
    fprintf(stderr, "early_config_err:%d pool 'EarlyConfigTwoTouchDriver' isn't defined\n", Vs32ReturnCode);
#endif
    return(Vs32ReturnCode);
}
